package net.sf.aguacate.swagger.configuration.spi.impl;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Map;

import javax.script.Compilable;
import javax.script.CompiledScript;
import javax.script.Invocable;
import javax.script.ScriptEngine;
import javax.script.ScriptEngineManager;
import javax.script.ScriptException;
import javax.script.SimpleBindings;

import org.apache.commons.io.IOUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;

import net.sf.aguacate.swagger.configuration.spi.SwaggerConfiguration;
import net.sf.aguacate.util.json.JsonCodec;
import net.sf.aguacate.util.resource.ResourceLocator;

public class SwaggerConfigurationImpl implements SwaggerConfiguration {

	private static final Logger LOGGER = LogManager.getFormatterLogger(SwaggerConfigurationImpl.class);

	private static final ObjectMapper YAML_MAPPER = new ObjectMapper(new YAMLFactory());

	private static final String JS_FILENAME = "convert.js";

	private final Invocable invocable;

	private final CompiledScript compiled;

	private final ResourceLocator locator;

	private final JsonCodec codec;

	private final ResourceLocator aguacateLocator;

	public SwaggerConfigurationImpl(ResourceLocator aguacateLocator, ResourceLocator locator, JsonCodec codec) {
		this.aguacateLocator = aguacateLocator;
		this.locator = locator;
		this.codec = codec;
		ScriptEngine engine = new ScriptEngineManager().getEngineByExtension("js");
		if (engine instanceof Compilable && engine instanceof Invocable) {
			invocable = (Invocable) engine;
			try {
				InputStream inputStream = SwaggerConfigurationImpl.class.getResourceAsStream(JS_FILENAME);
				if (inputStream == null) {
					throw new IllegalArgumentException(JS_FILENAME);
				} else {
					try {
						compiled = ((Compilable) engine)
								.compile(new InputStreamReader(inputStream, StandardCharsets.UTF_8));
					} finally {
						try {
							inputStream.close();
						} catch (IOException e) {
							LOGGER.warn("On resource close", e);
						}
					}
				}
			} catch (ScriptException e) {
				throw new IllegalStateException(e);
			}
		} else {
			throw new IllegalStateException("No Compilable | Invocable instance");
		}
	}

	@Override
	public String getConfiguration(String name) {
		try {
			String swagger = loadSwagger(name);
			if (swagger == null) {
				return null;
			} else {
				Object[] parameters = parameters(swagger);
				LOGGER.trace("before script");
				String result = (String) invocable.invokeMethod(compiled.eval(new SimpleBindings()), "convert", swagger,
						parameters);
				LOGGER.trace("after script");
				return YAML_MAPPER.writeValueAsString(codec.decode(result));
			}
		} catch (IOException | NoSuchMethodException | ScriptException e) {
			throw new IllegalStateException(e);
		}
	}

	String[] parameters(String swagger) throws IOException {
		Map<String, Object> conf = codec.decode(swagger);
		@SuppressWarnings("unchecked")
		List<String> entities = (List<String>) conf.get("entities");
		String[] params = new String[entities.size()];
		int index = 0;
		for (String name : entities) {
			String string = loadConfiguration(name);
			if (string == null) {
				throw new IllegalStateException(name);
			} else {
				params[index] = string;
				index += 1;
			}
		}
		return params;
	}

	String toString(String filename) throws IOException {
		InputStream inputStream = locator.open(filename);
		if (inputStream == null) {
			throw new IllegalStateException("not found ".concat(filename));
		} else {
			return toString(inputStream);
		}
	}

	String loadSwagger(String name) throws IOException {
		String filename = name.concat(".json");
		LOGGER.debug(filename);
		return load0(locator, filename);
	}

	String loadConfiguration(String name) throws IOException {
		String filename = name.concat(".json");
		LOGGER.debug(filename);
		return load0(aguacateLocator, filename);
	}

	String load0(ResourceLocator localLocator, String filename) throws IOException {
		InputStream inputStream = localLocator.open(filename);
		if (inputStream == null) {
			return null;
		} else {
			return toString(inputStream);
		}
	}

	String toString(InputStream inputStream) throws IOException {
		try {
			return IOUtils.toString(new InputStreamReader(inputStream, StandardCharsets.UTF_8));
		} finally {
			try {
				inputStream.close();
			} catch (IOException e) {
				LOGGER.warn("On resource close", e);
			}
		}
	}

}
