package net.sf.aguacate.scheduler.job.impl;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import net.sf.aguacate.effort.EffortSchedulerCoupling;
import net.sf.aguacate.scheduler.job.Job;
import net.sf.aguacate.scheduler.job.JobLoader;
import net.sf.aguacate.scheduler.job.JobSchedule;
import net.sf.aguacate.scheduler.job.JobScheduleCron;
import net.sf.aguacate.util.codec.bridge.CodecCoupling;
import net.sf.aguacate.util.filesystem.EventHandler;
import net.sf.aguacate.util.filesystem.FileSystemObserver;
import net.sf.aguacate.util.resource.ResourceLocator;
import net.sf.aguacate.util.resource.impl.ResourceLocatorClassImpl;
import net.sf.aguacate.util.resource.impl.ResourceLocatorFileImpl;

public class JobLoaderImpl implements JobLoader, EventHandler {

	private static final Logger LOGGER = LogManager.getLogger(JobLoaderImpl.class);

	private static final Logger LOGGER2 = LogManager.getLogger("aguacate.file.job");

	private static final String ENVIRONMENT = "DIRECTORY_JOB";

	private static final File DIRECTORY;

	private static final Map<String, Object> SPECS;

	private static final String SUFFIX = ".json";

	private static final int SUFFIX_LENGTH = 5;

	private final ResourceLocator locator;

	private Map<String, Job> cache;

	static {
		assert SUFFIX_LENGTH == SUFFIX.length();
		String temp = System.getProperty(ENVIRONMENT);
		if (temp == null || temp.isEmpty()) {
			temp = System.getenv(ENVIRONMENT);
			if (temp == null || temp.isEmpty()) {
				LOGGER.info("No " + ENVIRONMENT + " defined, using default");
				DIRECTORY = null;
			} else {
				LOGGER.info("using " + ENVIRONMENT + " (env): {}", temp);
				DIRECTORY = new File(temp);
			}
		} else {
			LOGGER.info("using " + ENVIRONMENT + " (prop): {}", temp);
			DIRECTORY = new File(temp);
		}
		Map<String, Object> specs = new HashMap<>();
		specs.put("kind", "job");
		specs.put("format", "1.0");
		SPECS = specs;
	}

	public JobLoaderImpl() {
		cache = Collections.emptyMap();
		assert cache != null && cache.isEmpty();
		if (DIRECTORY == null) {
			locator = new ResourceLocatorClassImpl(Job.class);
		} else {
			locator = new ResourceLocatorFileImpl(DIRECTORY);
			FileSystemObserver.watch(DIRECTORY.toPath(), this);
		}
	}

	@Override
	public void load() {
		String[] names = locator.list(SUFFIX);
		HashMap<String, Job> temp = new HashMap<>();
		for (String filename : names) {
			String name = removeSufix(filename);
			Job job = load0(filename);
			temp.put(name, job);
			List<JobSchedule> schedules = job.getSchedules();
			if (schedules != null && !schedules.isEmpty()) {
				int size = schedules.size();
				for (int i = 0; i < size; i++) {
					JobSchedule schedule = schedules.get(i);
					switch (schedule.getType()) {
					case CRON:
						EffortSchedulerCoupling.scheduleUnixCronExpression(name,
								((JobScheduleCron) schedule).getCronExpression());
						break;
					case AT:
						throw new IllegalStateException("Not implemented");
					default:
						throw new IllegalArgumentException(schedule.getType().name());
					}
				}
			}
		}
		LOGGER.trace("jobs: {}", temp);
		cache = temp;
	}

	@Override
	public Job get(String name) {
		return cache.get(name);
	}

	@SuppressWarnings("unchecked")
	Job load0(String filename) {
		LOGGER.debug(filename);
		LOGGER2.info("Loading script: {}", filename);
		try {
			InputStream inputStream = locator.open(filename);
			if (inputStream == null) {
				return null;
			} else {
				try {
					Map<String, Object> source = checkSpecs(CodecCoupling.jsonCodecBridge()
							.decodeMap(new InputStreamReader(inputStream, StandardCharsets.UTF_8)));
					List<Map<String, Object>> scheds = (List<Map<String, Object>>) source.get("schedules");
					List<JobSchedule> schedules;
					if (scheds == null || scheds.isEmpty()) {
						throw new IllegalArgumentException("No schedules");
					} else {
						int size = scheds.size();
						schedules = new ArrayList<>(size);
						for (int i = 0; i < size; i++) {
							Map<String, Object> current = scheds.get(i);
							JobSchedule jobSchedule;
							String type = (String) current.get("type");
							switch (type) {
							case "cron":
								jobSchedule = new JobScheduleCron((String) current.get("expression"));
								break;
							default:
								throw new IllegalArgumentException(type);
							}
							schedules.add(jobSchedule);
						}
					}
					return new Job((String) source.get("definition"), (Map<String, Object>) source.get("context"),
							schedules);
				} finally {
					try {
						inputStream.close();
					} catch (IOException e) {
						LOGGER.warn("on close inputstream", e);
					}
				}
			}
		} catch (IOException e) {
			throw new IllegalStateException(e);
		}
	}

	Map<String, Object> checkSpecs(Map<String, Object> data) {
		@SuppressWarnings("unchecked")
		Map<String, Object> specs = (Map<String, Object>) data.get("specs");
		if (specs == null) {
			throw new IllegalArgumentException("no specs");
		} else {
			if (SPECS.equals(specs)) {
				return data;
			} else {
				throw new IllegalArgumentException(
						"unsupported specs: ".concat(CodecCoupling.jsonCodecBridge().encode(specs)));
			}
		}
	}

	@Override
	public void onDelete(Path directory, Path deleted) {
		String file = deleted.toString();
		if (file.endsWith(SUFFIX)) {
			LOGGER2.warn("Removing job: {}", deleted);
			String name = removeSufix(file);
			synchronized (this) {
				if (cache.containsKey(name)) {
					Map<String, Job> temp = new HashMap<>(cache);
					temp.remove(name);
					cache = temp;
				}
			}
			EffortSchedulerCoupling.removeSchedules(name);
		} else {
			LOGGER.debug("ignore deleted file: {}", deleted);
		}
	}

	@Override
	public void onUpdate(Path directory, Path updated) {
		String file = updated.toString();
		if (file.endsWith(SUFFIX)) {
			LOGGER2.info("Change detected on job: {}", updated);
			String name = removeSufix(file);
			EffortSchedulerCoupling.removeSchedules(name);
			synchronized (this) {
				Map<String, Job> temp = new HashMap<>(cache);
				Job job = load0(file);
				temp.put(name, job);
				List<JobSchedule> schedules = job.getSchedules();
				if (schedules != null && !schedules.isEmpty()) {
					int size = schedules.size();
					for (int i = 0; i < size; i++) {
						JobSchedule schedule = schedules.get(i);
						switch (schedule.getType()) {
						case CRON:
							EffortSchedulerCoupling.scheduleUnixCronExpression(name,
									((JobScheduleCron) schedule).getCronExpression());
							break;
						case AT:
							throw new IllegalStateException("Not implemented");
						default:
							throw new IllegalArgumentException(schedule.getType().name());
						}
					}
				}
				cache = temp;
			}
		} else {
			LOGGER2.warn("Ignored : {}", updated);
			LOGGER.debug("ignore updated file: {}", updated);
		}
	}

	String removeSufix(String name) {
		return name.substring(0, name.length() - SUFFIX_LENGTH);
	}

}
