package net.sf.aguacate.definition.parser.impl;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import net.sf.aguacate.definition.Definition;
import net.sf.aguacate.definition.compiler.DefinitionCompilerCoupling;
import net.sf.aguacate.definition.parser.DefinitionParser;
import net.sf.aguacate.util.codec.bridge.CodecCoupling;
import net.sf.aguacate.util.filesystem.EventHandler;
import net.sf.aguacate.util.filesystem.FileSystemObserver;
import net.sf.aguacate.util.resource.ResourceLocator;
import net.sf.aguacate.util.resource.impl.ResourceLocatorClassImpl;
import net.sf.aguacate.util.resource.impl.ResourceLocatorFileImpl;

public class DefinitionParserDefault implements DefinitionParser, EventHandler {

	private static final Logger LOGGER = LogManager.getLogger(DefinitionParserDefault.class);

	private static final Logger LOGGER2 = LogManager.getLogger("aguacate.file.definition");

	private static final String ENVIRONMENT = "DIRECTORY_DEFINITION";

	private static final File DIRECTORY;

	private static final Map<String, Object> SPECS;

	private static final String SUFFIX = ".json";

	private static final int SUFFIX_LENGTH = 5;

	private final ResourceLocator locator;

	private Map<String, Definition> cache;

	static {
		assert SUFFIX_LENGTH == SUFFIX.length();
		String temp = System.getProperty(ENVIRONMENT);
		if (temp == null || temp.isEmpty()) {
			temp = System.getenv(ENVIRONMENT);
			if (temp == null || temp.isEmpty()) {
				LOGGER.info("No " + ENVIRONMENT + " defined, using default");
				DIRECTORY = null;
			} else {
				LOGGER.info("using " + ENVIRONMENT + " (env): {}", temp);
				DIRECTORY = new File(temp);
			}
		} else {
			LOGGER.info("using " + ENVIRONMENT + " (prop): {}", temp);
			DIRECTORY = new File(temp);
		}
		Map<String, Object> spec = new HashMap<>();
		spec.put("kind", "definition");
		spec.put("format", "1.0");
		SPECS = spec;
	}

	public DefinitionParserDefault() {
		cache = Collections.emptyMap();
		assert cache != null && cache.isEmpty();
		if (DIRECTORY == null) {
			locator = new ResourceLocatorClassImpl(Definition.class);
		} else {
			locator = new ResourceLocatorFileImpl(DIRECTORY);
			FileSystemObserver.watch(DIRECTORY.toPath(), this);
		}
	}

	@Override
	public Definition load(String name) {
		Definition definition = cache.get(name);
		if (definition == null) {
			synchronized (this) {
				definition = cache.get(name);
				if (definition == null) {
					Map<String, Definition> temp = new HashMap<>(cache);
					definition = load0(name.concat(SUFFIX));
					temp.put(name, definition);
					cache = temp;
				}
			}
		}
		return definition;
	}

	Definition load0(String filename) {
		LOGGER.debug(filename);
		LOGGER2.info("Loading script: {}", filename);
		try {
			InputStream inputStream = locator.open(filename);
			if (inputStream == null) {
				return null;
			} else {
				try {
					return DefinitionCompilerCoupling.compile(checkSpecs(CodecCoupling.jsonCodecBridge()
							.decodeMap(new InputStreamReader(inputStream, StandardCharsets.UTF_8))));
				} finally {
					try {
						inputStream.close();
					} catch (IOException e) {
						LOGGER.warn("on close inputstream", e);
					}
				}
			}
		} catch (IOException e) {
			throw new IllegalStateException(e);
		}
	}

	Map<String, Object> checkSpecs(Map<String, Object> data) {
		@SuppressWarnings("unchecked")
		Map<String, Object> specs = (Map<String, Object>) data.get("specs");
		if (specs == null) {
			throw new IllegalArgumentException("no specs");
		} else {
			if (SPECS.equals(specs)) {
				return data;
			} else {
				throw new IllegalArgumentException(
						"unsupported specs: ".concat(CodecCoupling.jsonCodecBridge().encode(specs)));
			}
		}
	}

	@Override
	public void onDelete(Path directory, Path deleted) {
		String file = deleted.toString();
		if (file.endsWith(SUFFIX)) {
			LOGGER2.warn("Removing script definition: {}", deleted);
			String name = removeSufix(file);
			synchronized (this) {
				if (cache.containsKey(name)) {
					Map<String, Definition> temp = new HashMap<>(cache);
					temp.remove(name);
					cache = temp;
				}
			}
		} else {
			LOGGER.debug("ignore deleted file: {}", deleted);
		}
	}

	@Override
	public void onUpdate(Path directory, Path updated) {
		String file = updated.toString();
		if (file.endsWith(SUFFIX)) {
			LOGGER2.info("Change detected on definition: {}", updated);
			String name = removeSufix(file);
			synchronized (this) {
				if (cache.containsKey(name)) {
					Map<String, Definition> temp = new HashMap<>(cache);
					temp.put(name, load0(file));
					cache = temp;
				}
			}
		} else {
			LOGGER2.warn("Ignored : {}", updated);
			LOGGER.debug("ignore updated file: {}", updated);
		}
	}

	String removeSufix(String name) {
		return name.substring(0, name.length() - SUFFIX_LENGTH);
	}

}
