package net.sf.aguacate.definition.compiler.impl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import net.sf.aguacate.context.spi.sql.impl.SentenceSqlStaticCountNotZero;
import net.sf.aguacate.context.spi.sql.impl.SentenceSqlStaticCountZero;
import net.sf.aguacate.context.spi.sql.impl.SentenceSqlStaticSelectSingle;
import net.sf.aguacate.function.Function;
import net.sf.aguacate.function.spi.impl.FunctionArrayIterator;
import net.sf.aguacate.function.spi.impl.FunctionBase64Decode;
import net.sf.aguacate.function.spi.impl.FunctionBase64Encode;
import net.sf.aguacate.function.spi.impl.FunctionConditional;
import net.sf.aguacate.function.spi.impl.FunctionConnectorCall;
import net.sf.aguacate.function.spi.impl.FunctionCopy;
import net.sf.aguacate.function.spi.impl.FunctionDivide;
import net.sf.aguacate.function.spi.impl.FunctionEquals;
import net.sf.aguacate.function.spi.impl.FunctionGreaterEquals;
import net.sf.aguacate.function.spi.impl.FunctionGreaterThan;
import net.sf.aguacate.function.spi.impl.FunctionJsonDecode;
import net.sf.aguacate.function.spi.impl.FunctionJsonEncode;
import net.sf.aguacate.function.spi.impl.FunctionLessEquals;
import net.sf.aguacate.function.spi.impl.FunctionLessThan;
import net.sf.aguacate.function.spi.impl.FunctionLiteralFloat;
import net.sf.aguacate.function.spi.impl.FunctionLiteralInteger;
import net.sf.aguacate.function.spi.impl.FunctionLiteralString;
import net.sf.aguacate.function.spi.impl.FunctionMultiply;
import net.sf.aguacate.function.spi.impl.FunctionNotEquals;
import net.sf.aguacate.function.spi.impl.FunctionRename;
import net.sf.aguacate.function.spi.impl.FunctionScript;
import net.sf.aguacate.function.spi.impl.FunctionSha256;
import net.sf.aguacate.function.spi.impl.FunctionStringSubstitutor;
import net.sf.aguacate.function.spi.impl.FunctionSubstract;
import net.sf.aguacate.function.spi.impl.FunctionSum;
import net.sf.aguacate.function.spi.impl.FunctionZero;
import net.sf.aguacate.util.parameter.Prm;
import net.sf.aguacate.util.type.Str;

public class ParserValidator {

	@SuppressWarnings("unchecked")
	public List<Function> parse(List<Map<String, Object>> validations) {
		if (validations == null) {
			return Collections.emptyList();
		} else {
			List<Function> functions = new ArrayList<>();
			int size = validations.size();
			for (int i = 0; i < size; i++) {
				Map<String, Object> validation = validations.get(i);
				String name = (String) validation.get("name");
				String message = (String) validation.get("message");
				String type = (String) validation.get("type");
				List<?> parameters = (List<?>) validation.get("parameters");
				List<String> methods = (List<String>) validation.get("methods");
				functions.add(toFunction(validation, methods, name, message, type, parameters));
			}
			return functions;
		}
	}

	@SuppressWarnings("unchecked")
	Function toFunction(Map<String, Object> validation, Collection<String> methods, String name, String message,
			String type, List<?> parameters) {
		Function function;
		switch (type) {
		case Function.LT:
			if (parameters.size() == 2) {
				function = new FunctionLessThan<>(methods, name, message, Prm.toParameter(parameters.get(0)),
						Prm.toParameter(parameters.get(1)));
			} else {
				throw new IllegalArgumentException(
						"The number or parameters for function type '" + type + "'(" + name + ") must be 2");
			}
			break;
		case Function.LE:
			if (parameters.size() == 2) {
				function = new FunctionLessEquals<>(methods, name, message, Prm.toParameter(parameters.get(0)),
						Prm.toParameter(parameters.get(1)));
			} else {
				throw new IllegalArgumentException(
						"The number or parameters for function type '" + type + "'(" + name + ") must be 2");
			}
			break;
		case Function.GT:
			if (parameters.size() == 2) {
				function = new FunctionGreaterThan<>(methods, name, message, Prm.toParameter(parameters.get(0)),
						Prm.toParameter(parameters.get(1)));
			} else {
				throw new IllegalArgumentException(
						"The number or parameters for function type '" + type + "'(" + name + ") must be 2");
			}
			break;
		case Function.GE:
			if (parameters.size() == 2) {
				function = new FunctionGreaterEquals<>(methods, name, message, Prm.toParameter(parameters.get(0)),
						Prm.toParameter(parameters.get(1)));
			} else {
				throw new IllegalArgumentException(
						"The number or parameters for function type '" + type + "'(" + name + ") must be 2");
			}
			break;
		case Function.EQ:
			if (parameters.size() == 2) {
				function = new FunctionEquals(methods, name, message, Prm.toParameter(parameters.get(0)),
						Prm.toParameter(parameters.get(1)));
			} else {
				throw new IllegalArgumentException(
						"The number or parameters for function type '" + type + "'(" + name + ") must be 2");
			}
			break;
		case Function.NE:
			if (parameters.size() == 2) {
				function = new FunctionNotEquals(methods, name, message, Prm.toParameter(parameters.get(0)),
						Prm.toParameter(parameters.get(1)));
			} else {
				throw new IllegalArgumentException(
						"The number or parameters for function type '" + type + "'(" + name + ") must be 2");
			}
			break;
		case Function.ZERO:
			if (parameters.size() == 1) {
				function = new FunctionZero(methods, name, message, Prm.toParameter(parameters.get(0)));
			} else {
				throw new IllegalArgumentException(
						"The number or parameters for function type '" + type + "'(" + name + ") must be 1");
			}
			break;
		case Function.NOT_ZERO:
			if (parameters.size() == 1) {
				function = new FunctionZero(methods, name, message, Prm.toParameter(parameters.get(0)));
			} else {
				throw new IllegalArgumentException(
						"The number or parameters for function type '" + type + "'(" + name + ") must be 1");
			}
			break;
		case Function.REN:
			if (parameters.size() == 1) {
				function = new FunctionRename(methods, name, message, Prm.toParameter(parameters.get(0)),
						Str.toList(validation.get("outputContext")), (String) validation.get("outputName"));
			} else {
				throw new IllegalArgumentException(
						"The number or parameters for function type '" + type + "'(" + name + ") must be 1");
			}
			break;
		case Function.COPY:
			if (parameters.size() == 1) {
				function = new FunctionCopy(methods, name, message, Prm.toParameter(parameters.get(0)),
						Str.toList(validation.get("outputContext")), (String) validation.get("outputName"));
			} else {
				throw new IllegalArgumentException(
						"The number or parameters for function type '" + type + "'(" + name + ") must be 1");
			}
			break;
		case Function.SCRIPT:
			function = new FunctionScript(methods, name, message, Prm.toList(parameters),
					Str.toList(validation.get("outputContext")), (String) validation.get("outputName"),
					(String) validation.get("scriptName"), (String) validation.get("functionName"));
			break;
		case Function.STRUCTURE_ARRAY_ITERATOR: {
			List<Function> subvalidations = parse((List<Map<String, Object>>) validation.get("validations"));
			function = new FunctionArrayIterator(Collections.emptyList(), name, message,
					Prm.toParameter(parameters.get(0)), Str.toList(validation.get("outputContext")),
					(String) validation.get("outputName"), subvalidations);
			break;
		}
		case Function.CONDITIONAL: {
			List<Function> subvalidations = parse((List<Map<String, Object>>) validation.get("validations"));
			function = new FunctionConditional(Collections.emptyList(), name, message,
					Prm.toParameter(parameters.get(0)), subvalidations, (String) validation.get("test"));
			break;
		}
		case Function.LITERAL_STRING:
			function = new FunctionLiteralString(methods, name, message, Str.toList(validation.get("outputContext")),
					(String) validation.get("outputName"), validation.get("value"));
			break;
		case Function.LITERAL_INTEGER:
			function = new FunctionLiteralInteger(methods, name, message, Str.toList(validation.get("outputContext")),
					(String) validation.get("outputName"), validation.get("value"));
			break;
		case Function.LITERAL_FLOAT:
			function = new FunctionLiteralFloat(methods, name, message, Str.toList(validation.get("outputContext")),
					(String) validation.get("outputName"), validation.get("value"));
			break;
		case Function.SQL_SELECT_SINGLE: {
			function = new SentenceSqlStaticSelectSingle(name, message, (String) validation.get("sql"), methods,
					Prm.toList(parameters), Prm.toList(Collections.emptyList()),
					Str.toList(validation.get("outputContext")), (String) validation.get("outputName"));
			break;
		}
		case Function.SQL_COUNT_NOT_ZERO:
			function = new SentenceSqlStaticCountNotZero(methods, name, message, (String) validation.get("sql"),
					Prm.toList(parameters));
			break;
		case Function.SQL_COUNT_ZERO:
			function = new SentenceSqlStaticCountZero(methods, name, message, (String) validation.get("sql"),
					Prm.toList(parameters));
			break;
		case Function.BASE64_DECODE:
			function = new FunctionBase64Decode(methods, name, message, Prm.toParameter(parameters.get(0)),
					Str.toList(validation.get("outputContext")), (String) validation.get("outputName"));
			break;
		case Function.BASE64_ENCODE:
			function = new FunctionBase64Encode(methods, name, message, Prm.toParameter(parameters.get(0)),
					Str.toList(validation.get("outputContext")), (String) validation.get("outputName"));
			break;
		case Function.SHA256:
			function = new FunctionSha256(methods, name, message, Prm.toParameter(parameters.get(0)),
					Str.toList(validation.get("outputContext")), (String) validation.get("outputName"));
			break;
		case Function.JSON_ENCODE:
			function = new FunctionJsonEncode(methods, name, message, Prm.toParameter(parameters.get(0)),
					Str.toList(validation.get("outputContext")), (String) validation.get("outputName"));
			break;
		case Function.JSON_DECODE:
			function = new FunctionJsonDecode(methods, name, message, Prm.toParameter(parameters.get(0)),
					Str.toList(validation.get("outputContext")), (String) validation.get("outputName"));
			break;
		case Function.STRING_SUBSTITUTOR:
			function = new FunctionStringSubstitutor(methods, name, message, Prm.toParameter(parameters.get(0)),
					Prm.toList(parameters, 1), Str.toList(validation.get("outputContext")),
					(String) validation.get("outputName"));
			break;
		case Function.SUM:
			function = new FunctionSum(methods, name, message, Prm.toParameter(parameters.get(0)),
					Prm.toParameter(parameters.get(1)), Str.toList(validation.get("outputContext")),
					(String) validation.get("outputName"));
			break;
		case Function.SUBSTRACT:
			function = new FunctionSubstract(methods, name, message, Prm.toParameter(parameters.get(0)),
					Prm.toParameter(parameters.get(1)), Str.toList(validation.get("outputContext")),
					(String) validation.get("outputName"));
			break;
		case Function.MULTIPLY:
			function = new FunctionMultiply(methods, name, message, Prm.toParameter(parameters.get(0)),
					Prm.toParameter(parameters.get(1)), Str.toList(validation.get("outputContext")),
					(String) validation.get("outputName"));
			break;
		case Function.DIVIDE:
			function = new FunctionDivide(methods, name, message, Prm.toParameter(parameters.get(0)),
					Prm.toParameter(parameters.get(1)), Str.toList(validation.get("outputContext")),
					(String) validation.get("outputName"));
			break;
		case Function.CONNECTOR_CALL:
			function = new FunctionConnectorCall(methods, name, (String) validation.get("connectorName"),
					Prm.toList(parameters), Str.toList(validation.get("outputContext")),
					(String) validation.get("outputName"));
			break;
		default:
			throw new IllegalArgumentException(type);
		}
		return function;
	}

}
