package net.sf.aguacate.definition.compiler.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.time.FastDateFormat;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import net.sf.aguacate.field.Field;
import net.sf.aguacate.field.FieldArray;
import net.sf.aguacate.field.FieldBoolean;
import net.sf.aguacate.field.FieldDate;
import net.sf.aguacate.field.FieldDynamicDate;
import net.sf.aguacate.field.FieldFloat;
import net.sf.aguacate.field.FieldInteger;
import net.sf.aguacate.field.FieldString;
import net.sf.aguacate.field.FieldStructure;
import net.sf.aguacate.field.FieldStructureArray;
import net.sf.aguacate.field.FieldTimeWithZone;
import net.sf.aguacate.util.type.Bool;

/**
 * <pre>
 * {
 *   "field1": {
 *     "type": "...",
 *     ...
 *   }
 * }
 * </pre>
 */
public class ParserField {

	private static final Logger LOGGER = LogManager.getLogger(ParserField.class);

	private static final FastDateFormat FMT_TIME;

	private static final FastDateFormat FMT_DATE;

	private static final FastDateFormat FMT_DYN_DATE;

	private static final FastDateFormat FMT_DATETIME;

	static {
		FMT_TIME = FastDateFormat.getInstance("HH:mmZZ");

		final String date = "yyyy-MM-dd";
		FMT_DATE = FastDateFormat.getInstance(date);

		final String dynamic = date + "ZZ";
		FMT_DYN_DATE = FastDateFormat.getInstance(dynamic);

		final String datetime = date + "'T'HH:mm:ss.SSSZZ";
		FMT_DATETIME = FastDateFormat.getInstance(datetime);
	}

	@SuppressWarnings("unchecked")
	List<Field> parse(Map<String, Object> source) {
		List<Field> fields = new ArrayList<>(source.size());
		for (Map.Entry<String, Object> entry : source.entrySet()) {
			String name = entry.getKey();
			Map<String, Object> meta2 = (Map<String, Object>) entry.getValue();
			fields.add(toField(name, meta2));
		}
		return fields;
	}

	Field toField(String name, Map<String, Object> source) {
		String string = (String) source.get("type");
		boolean optional = Bool.valueOf(source.get("optional"));
		switch (string) {
		case Field.TIME: {
			return new FieldTimeWithZone(name, Field.TIME, optional, FMT_TIME, (String) source.get("minval"),
					(String) source.get("maxval"));
		}
		case Field.DYNAMIC_DATE:
			return new FieldDynamicDate(name, optional, FMT_DYN_DATE, (String) source.get("minval"),
					(String) source.get("maxval"));
		case Field.DATE: {
			return new FieldDate(name, optional, FMT_DATE, (String) source.get("minval"),
					(String) source.get("maxval"));
		}
		case Field.DATETIME: {
			return new FieldTimeWithZone(name, Field.DATETIME, optional, FMT_DATETIME, (String) source.get("minval"),
					(String) source.get("maxval"));
		}
		case Field.FLOAT: {
			return new FieldFloat(name, optional, (String) source.get("minval"), (String) source.get("maxval"));
		}
		case Field.INTEGER: {
			return new FieldInteger(name, optional, (String) source.get("minval"), (String) source.get("maxval"));
		}
		case Field.STRING: {
			int minLenth = Integer.parseInt((String) source.get("minlen"));
			int maxLength = Integer.parseInt((String) source.get("maxlen"));
			String regex = (String) source.get("regex");
			LOGGER.trace("{} regex: {}", name, regex);
			return new FieldString(name, optional, minLenth, maxLength, regex);
		}
		case Field.BOOLEAN: {
			return new FieldBoolean(name, optional);
		}
		case Field.STRUCTURE_ARRAY: {
			@SuppressWarnings("unchecked")
			Map<String, Object> sub = (Map<String, Object>) source.get("data");
			return new FieldStructureArray(name, optional, parse(sub));
		}
		case Field.STRUCTURE: {
			@SuppressWarnings("unchecked")
			Map<String, Object> sub = (Map<String, Object>) source.get("data");
			return new FieldStructure(name, optional, parse(sub));
		}
		case Field.ARRAY: {
			@SuppressWarnings("unchecked")
			Map<String, Object> sub = (Map<String, Object>) source.get("data");
			return new FieldArray(name, optional, (String) sub.get("name"), toField("inner", sub),
					Bool.valueOf(source.get("unique")));
		}
		default:
			throw new IllegalArgumentException(string);
		}
	}

}
