package net.sf.aguacate.connector.spi;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.Map;

import org.apache.http.client.utils.URIBuilder;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import net.sf.aguacate.connector.Connector;
import net.sf.aguacate.util.codec.bridge.CodecBridge;

public abstract class ConnectorHttp implements Connector {

	private static final Logger LOGGER = LogManager.getLogger(ConnectorHttp.class);

	private static final Logger LOGGER2 = LogManager.getLogger("aguacate.net.http");

	private static final ConnectorHttpParameterComparator PARAMETER_COMPARATOR;

	private static final ConnectorHttpParameterComparatorAlt PARAMETER_COMPARATOR_ALT;

	private final boolean acceptsBody;

	private final String base;

	private final String expectedMimeType;

	private final Map<String, String> headers;

	private final ConnectorHttpParameter[] httpParameters;

	static {
		PARAMETER_COMPARATOR = new ConnectorHttpParameterComparator();
		PARAMETER_COMPARATOR_ALT = new ConnectorHttpParameterComparatorAlt();
	}

	public ConnectorHttp(boolean acceptsBody, String base, String expectedMimeType, Map<String, String> headers,
			Collection<ConnectorHttpParameter> parameters) {
		if (base.startsWith("http://") || base.startsWith("https://")) {
			this.acceptsBody = acceptsBody;
			this.base = base;
			this.expectedMimeType = expectedMimeType;
			this.headers = headers;
			this.httpParameters = parameters.toArray(new ConnectorHttpParameter[parameters.size()]);
			Arrays.sort(this.httpParameters, PARAMETER_COMPARATOR);
		} else {
			throw new IllegalArgumentException(base);
		}
	}

	@Override
	public Object execute(Map<String, Object> parameters) {
		try {
			LOGGER.trace(parameters);
			CodecBridge codec = getCodecBridge(expectedMimeType);
			URIBuilder builder = new URIBuilder(base);
			Map<String, Object> body;
			if (acceptsBody) {
				body = new LinkedHashMap<>();
			} else {
				body = null;
			}
			int length = httpParameters.length;
			boolean[] found = new boolean[length];
			for (Map.Entry<String, Object> parameter : parameters.entrySet()) {
				String name = parameter.getKey();
				int ndx = Arrays.binarySearch(httpParameters, name, PARAMETER_COMPARATOR_ALT);
				if (ndx >= 0) {
					found[ndx] = true;
					ConnectorHttpParameter httpParameter = httpParameters[ndx];
					Object value = parameter.getValue();
					if (httpParameter.isInQuery()) {
						if (value == null) {
							builder.addParameter(name, "");
						} else {
							builder.addParameter(name, String.valueOf(value));
						}
					}
					if (httpParameter.isInBody()) {
						body.put(name, value);
					}
				} else {
					throw new IllegalArgumentException(name);
				}
			}
			for (int i = 0; i < length; i++) {
				if (!found[i]) {
					String name = httpParameters[i].getName();
					LOGGER.warn("Missing configuration for parameter: {} ({} & {})", name, builder, body);
					throw new IllegalStateException(name);
				}
			}
			URI uri = builder.build();
			LOGGER2.info("build: {}", uri);
			return execute0(uri, headers, codec);
		} catch (IOException | URISyntaxException e) {
			throw new IllegalStateException(e);
		}
	}

	protected abstract CodecBridge getCodecBridge(String mimteType);

	protected abstract Object execute0(URI uri, Map<String, String> headers, CodecBridge codec) throws IOException;

}
