package net.sf.aguacate.cloud.notification.spi;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;
import java.util.ServiceLoader;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import net.sf.aguacate.cloud.notification.NotificationBridge;
import net.sf.aguacate.cloud.notification.NotificationBridgeBuilder;
import net.sf.aguacate.cloud.notification.NotificationBridgeLoader;
import net.sf.aguacate.util.codec.bridge.CodecCoupling;
import net.sf.aguacate.util.filesystem.EventHandler;
import net.sf.aguacate.util.filesystem.FileSystemObserver;
import net.sf.aguacate.util.resource.ResourceLocator;
import net.sf.aguacate.util.resource.impl.ResourceLocatorClassImpl;
import net.sf.aguacate.util.resource.impl.ResourceLocatorFileImpl;

/**
 * <pre>
 * {
 *   &quot;kind&quot;: &quot;notification&quot;,
 *   &quot;impl&quot;: &quot;local&quot;,
 *   &quot;format&quot;: &quot;1.0&quot;,
 *   &quot;local&quot;: {
 *     &quot;path&quot;: &quot;/dir/&quot;
 *   } 
 * }
 * </pre>
 *
 * <pre>
 * {
 *   &quot;kind&quot;: &quot;notification&quot;,
 *   &quot;impl&quot;: &quot;aws-sns&quot;,
 *   &quot;format&quot;: &quot;1.0&quot;,
 *   &quot;local&quot;: {
 *     &quot;bucket&quot;: &quot;storage&quot;,
 *     &quot;region&quot;: &quot;us-east-1&quot;
 *   } 
 * }
 * </pre>
 */
public class NotificationBridgeLoaderSpi implements NotificationBridgeLoader, EventHandler {

	private static final Logger LOGGER = LogManager.getLogger(NotificationBridgeLoaderSpi.class);

	private static final Logger LOGGER2 = LogManager.getLogger("aguacate.file.storage");

	private static final String ENVIRONMENT = "STORAGE_DATABASE";

	private static final String SUFFIX = ".json";

	private static final int SUFFIX_LENGTH = 5;

	private static final File DIRECTORY;

	private static final Map<String, Object> SPECS;

	private static final Map<String, NotificationBridgeBuilder> builders;

	private Map<String, NotificationBridge> bridges;

	private final ResourceLocator locator;

	static {
		assert SUFFIX_LENGTH == SUFFIX.length();
		String temp = System.getProperty(ENVIRONMENT);
		if (temp == null || temp.isEmpty()) {
			temp = System.getenv(ENVIRONMENT);
			if (temp == null || temp.isEmpty()) {
				LOGGER.info("No " + ENVIRONMENT + " defined, using default");
				DIRECTORY = null;
			} else {
				LOGGER.info("using " + ENVIRONMENT + " (env): {}", temp);
				DIRECTORY = new File(temp);
			}
		} else {
			LOGGER.info("using " + ENVIRONMENT + " (prop): {}", temp);
			DIRECTORY = new File(temp);
		}
		Map<String, Object> spec = new HashMap<>();
		spec.put("kind", "storage");
//		spec.put("impl", "*");
		spec.put("format", "1.0");
		SPECS = spec;

		builders = new HashMap<>();
		for (NotificationBridgeBuilder current : ServiceLoader.load(NotificationBridgeBuilder.class)) {
			builders.put(current.getProvider(), current);
		}
	}

	public NotificationBridgeLoaderSpi() {
		if (DIRECTORY == null) {
			locator = new ResourceLocatorClassImpl(NotificationBridge.class);
		} else {
			locator = new ResourceLocatorFileImpl(DIRECTORY);
			FileSystemObserver.watch(DIRECTORY.toPath(), this);
		}
		bridges = new HashMap<>();
	}

	@Override
	public NotificationBridge getNotificationBridge(String name) {
		NotificationBridge bridge = bridges.get(name);
		if (bridge == null) {
			synchronized (this) {
				bridge = bridges.get(name);
				if (bridge == null) {
					bridge = load0(name.concat(SUFFIX));
					Map<String, NotificationBridge> temp = new HashMap<>(bridges);
					temp.put(name, bridge);
					bridges = temp;
				}
			}
		}
		return bridge;
	}

	@SuppressWarnings("unchecked")
	public NotificationBridge load0(String file) {
		LOGGER.debug("trying to load {}", file);
		LOGGER2.info("Loading database: {}", file);
		try {
			Map<String, Object> configuration = checkSpecs(readConfiguration(file));
			return builders.get(((Map<String, Object>) configuration.get("specs")).get("impl")).build(configuration);
		} catch (IOException e) {
			LOGGER.error("on opening resource", e);
			throw new IllegalStateException(e);
		}
	}

	Map<String, Object> checkSpecs(Map<String, Object> data) {
		@SuppressWarnings("unchecked")
		Map<String, Object> specs = (Map<String, Object>) data.get("specs");
		if (specs == null) {
			throw new IllegalArgumentException("no specs");
		} else {
			for (String key : SPECS.keySet()) {
				if (!SPECS.get(key).equals(specs.get(key))) {
					throw new IllegalArgumentException(
							"unsupported specs: ".concat(CodecCoupling.jsonCodecBridge().encode(specs)));
				}
			}
			return data;
		}
	}

	Map<String, Object> readConfiguration(String file) throws IOException {
		InputStream inputStream = locator.open(file);
		if (inputStream == null) {
			LOGGER.warn("no configuration for {}", file);
			throw new UnsupportedOperationException();
		} else {
			try {
				return CodecCoupling.jsonCodecBridge()
						.decodeMap(new InputStreamReader(inputStream, StandardCharsets.UTF_8));
			} catch (IOException e) {
				throw new IllegalStateException(e);
			} finally {
				try {
					inputStream.close();
				} catch (IOException e) {
					LOGGER.error("on closing resource", e);
				}
			}
		}
	}

	@Override
	public void onDelete(Path directory, Path deleted) {
		String file = deleted.toString();
		if (file.endsWith(SUFFIX)) {
			LOGGER2.warn("Removing database definition: {}", deleted);
			String name = removeSufix(file);
			synchronized (this) {
				if (bridges.containsKey(name)) {
					Map<String, NotificationBridge> temp = new HashMap<>(bridges);
					temp.remove(name);
					bridges = temp;
					LOGGER.info("deleted: {} & {}", directory, deleted);
				} else {
					LOGGER.info("Not loaded: {} & {}", directory, deleted);
				}
			}
		} else {
			LOGGER.debug("ignore deleted file: {} & {}", directory, deleted);
		}
	}

	@Override
	public void onUpdate(Path directory, Path updated) {
		String file = updated.toString();
		if (file.endsWith(SUFFIX)) {
			LOGGER2.info("Change detected on database: {}", updated);
			String name = removeSufix(file);
			synchronized (this) {
				if (bridges.containsKey(name)) {
					Map<String, NotificationBridge> temp = new HashMap<>(bridges);
					temp.put(name, load0(file));
					bridges = temp;
					LOGGER.info("loaded: {} & {}", directory, updated);
				} else {
					LOGGER.info("not loaded until required: {} & {}", directory, updated);
				}
			}
		} else {
			LOGGER2.warn("Ignored : {}", updated);
			LOGGER.debug("ignore updated file: {} & {}", directory, updated);
		}
	}

	String removeSufix(String name) {
		return name.substring(0, name.length() - SUFFIX_LENGTH);
	}

}
