package net.sf.aguacate.util.environment.spi;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import net.sf.aguacate.util.codec.bridge.CodecCoupling;
import net.sf.aguacate.util.environment.EnvironmentBridge;
import net.sf.aguacate.util.environment.EnvironmentCoupling;
import net.sf.aguacate.util.filesystem.EventHandler;
import net.sf.aguacate.util.filesystem.FileSystemObserver;
import net.sf.aguacate.util.resource.ResourceLocator;
import net.sf.aguacate.util.resource.impl.ResourceLocatorClassImpl;
import net.sf.aguacate.util.resource.impl.ResourceLocatorFileImpl;
import net.sf.aguacate.util.type.Flt;
import net.sf.aguacate.util.type.Int;

/**
 * <pre>
 * {
 *    "specs": {
 *       "kind": "environment",
 *       "format": "1.0"
 *    },
 *    "variables": [
 *       {
 *          "name": "var1",
 *          "type": "INTEGER",
 *          "value": "1"
 *       }
 *    ]
 * }
 * </pre>
 */
public class EnvironmentBridgeSpi implements EnvironmentBridge, EventHandler {

	private static final Logger LOGGER = LogManager.getLogger(EnvironmentBridgeSpi.class);

	private static final Logger LOGGER2 = LogManager.getLogger("aguacate.file.environment");

	private static final String ENVIRONMENT = "DIRECTORY_ENVIRONMENT";

	private static final String SUFFIX = ".json";

	private static final int SUFFIX_LENGTH = 5;

	private static final File DIRECTORY;

	private static final Map<String, Object> SPECS;

	private final ResourceLocator locator;

	private Map<String, Map<String, Object>> environments;

	static {
		assert SUFFIX_LENGTH == SUFFIX.length();
		String temp = System.getProperty(ENVIRONMENT);
		if (temp == null || temp.isEmpty()) {
			temp = System.getenv(ENVIRONMENT);
			if (temp == null || temp.isEmpty()) {
				LOGGER.info("No " + ENVIRONMENT + " defined, using default");
				DIRECTORY = null;
			} else {
				LOGGER.info("using " + ENVIRONMENT + " (env): {}", temp);
				DIRECTORY = new File(temp);
			}
		} else {
			LOGGER.info("using " + ENVIRONMENT + " (prop): {}", temp);
			DIRECTORY = new File(temp);
		}
		Map<String, Object> spec = new HashMap<>();
		spec.put("kind", "environment");
		spec.put("format", "1.0");
		SPECS = spec;
	}

	public EnvironmentBridgeSpi() {
		environments = Collections.emptyMap();
		assert environments != null && environments.isEmpty();
		if (DIRECTORY == null) {
			locator = new ResourceLocatorClassImpl(EnvironmentCoupling.class);
		} else {
			locator = new ResourceLocatorFileImpl(DIRECTORY);
			FileSystemObserver.watch(DIRECTORY.toPath(), this);
		}
	}

	@Override
	public Map<String, Object> get(String name) {
		return getOrLoad(name);
	}

	Map<String, Object> getOrLoad(String name) {
		Map<String, Object> instance = environments.get(name);
		if (instance == null) {
			synchronized (this) {
				instance = environments.get(name);
				if (instance == null) {
					Map<String, Map<String, Object>> temp = new HashMap<>(environments);
					instance = load0(name.concat(SUFFIX));
					temp.put(name, instance);
					environments = temp;
				}
			}
		}
		return instance;
	}

	@SuppressWarnings("unchecked")
	Map<String, Object> load0(String filename) {
		LOGGER.debug(filename);
		LOGGER2.info("Loading script: {}", filename);
		try {
			InputStream inputStream = locator.open(filename);
			if (inputStream == null) {
				throw new IllegalArgumentException(filename);
			} else {
				try {
					return parse((List<Map<String, Object>>) checkSpecs(CodecCoupling.jsonCodecBridge()
							.decodeMap(new InputStreamReader(inputStream, StandardCharsets.UTF_8))).get("variables"));
				} finally {
					try {
						inputStream.close();
					} catch (IOException e) {
						LOGGER.warn("On closing resource", e);
					}
				}
			}
		} catch (IOException e) {
			throw new IllegalStateException(filename, e);
		}
	}

	Map<String, Object> checkSpecs(Map<String, Object> data) {
		@SuppressWarnings("unchecked")
		Map<String, Object> specs = (Map<String, Object>) data.get("specs");
		if (specs == null) {
			throw new IllegalArgumentException("no specs");
		} else {
			if (SPECS.equals(specs)) {
				return data;
			} else {
				throw new IllegalArgumentException(
						"unsupported specs: ".concat(CodecCoupling.jsonCodecBridge().encode(specs)));
			}
		}
	}

	@Override
	public void onDelete(Path directory, Path deleted) {
		String file = deleted.toString();
		if (file.endsWith(SUFFIX)) {
			LOGGER2.warn("Removing script definition: {}", deleted);
			String name = removeSufix(file);
			synchronized (this) {
				if (environments.containsKey(name)) {
					Map<String, Map<String, Object>> temp = new HashMap<>(environments);
					temp.remove(name);
					environments = temp;
				}
			}
		} else {
			LOGGER.debug("ignore deleted file: {}", deleted);
		}
	}

	@Override
	public void onUpdate(Path directory, Path updated) {
		String file = updated.toString();
		if (file.endsWith(SUFFIX)) {
			LOGGER2.info("Change detected on script: {}", updated);
			String name = removeSufix(file);
			synchronized (this) {
				if (environments.containsKey(name)) {
					Map<String, Map<String, Object>> temp = new HashMap<>(environments);
					temp.put(name, load0(file));
					environments = temp;
				}
			}
		} else {
			LOGGER2.warn("Ignored : {}", updated);
			LOGGER.debug("ignore updated file: {}", updated);
		}
	}

	String removeSufix(String name) {
		return name.substring(0, name.length() - SUFFIX_LENGTH);
	}

	Map<String, Object> parse(List<Map<String, Object>> data) {
		Map<String, Object> map = new LinkedHashMap<>();
		for (Map<String, Object> current : data) {
			String type = (String) current.get("type");
			Object val = current.get("value");
			Object value;
			switch (type) {
			case "STRING":
				value = val;
				break;
			case "INTEGER":
				value = Int.parse(val);
				break;
			case "FLOAT":
				value = Flt.parse(val);
				break;
			default:
				throw new IllegalArgumentException(type);
			}
			map.put((String) current.get("name"), value);
		}
		return map;
	}

}
