package net.sf.aguacate.definition;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.Map;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import net.sf.aguacate.field.Field;
import net.sf.aguacate.function.Function;
import net.sf.aguacate.function.FunctionContext;
import net.sf.aguacate.function.FunctionEvalResult;
import net.sf.aguacate.util.config.database.DatabaseCoupling;
import net.sf.aguacate.util.type.Fld;
import net.sf.aguacate.util.type.Fun;
import net.sf.aguacate.validator.ValidationConversionResult;

public class Definition {

	private static final Logger LOGGER = LogManager.getLogger(Definition.class);

	private final String datasource;

	private final Field[] fields;

	private final Function[] validators;

	private final Function[] processors;

	public Definition(String datasource, Collection<Field> fields, Collection<Function> validators,
			Collection<Function> processors) {
		this(datasource, Fld.toArray(fields), Fun.toArray(validators), Fun.toArray(processors));
	}

	public Definition(String datasource, Field[] fields, Function[] validators, Function[] processors) {
		this.datasource = datasource;
		this.fields = fields;
		this.validators = validators;
		this.processors = processors;
	}

	public void run(Map<String, Object> input, ExecutionListener listener) {
		Map<String, Object> context = new LinkedHashMap<>();
		int i;
		int length = fields.length;
		for (i = 0; i < length; i++) {
			Field field = fields[i];
			String name = field.getName();
			Object value = input.get(name);
			if (value == null) {
				if (field.isOptional()) {
					LOGGER.trace("Ignore {}: null & optional", name);
				} else {
					LOGGER.warn("Failure {}: null & not optional", name);
					listener.missingParameter(name);
					break;
				}
			} else {
				ValidationConversionResult vAndC = field.validateAndConvert(value);
				if (vAndC.isSuccess()) {
					Object val = vAndC.getValue();
					LOGGER.debug("Successful validation & conversion of {}", name);
					context.put(name, val);
				} else {
					listener.validationConversion(name, vAndC.getMessage());
					LOGGER.warn("Failure validating or converting:: {}", name);
					break;
				}
			}
		}
		if (i == length) {
			FunctionContext functionContext = new FunctionContextDefinition(
					DatabaseCoupling.getDatabaseBridge(datasource));
			try {
				FunctionEvalResult result = execute(validators, functionContext, context);
				if (result == null || result.isSuccess()) {
					LOGGER.info("validations: OK");
					result = execute(processors, functionContext, context);
					if (result == null || result.isSuccess()) {
						LOGGER.info("processors: OK");
						listener.success();
					} else {
						LOGGER.warn("processors: FAIL");
						functionContext.rollback();
						listener.error(result.getMessage());
					}
				} else {
					LOGGER.warn("validations: FAIL");
					listener.error(result.getMessage());
				}
			} catch (RuntimeException e) {
				LOGGER.warn("validations: EXCEPTION", e);
				functionContext.rollback();
				listener.exception(e.getMessage());
			} finally {
				try {
					functionContext.close();
				} catch (IOException e) {
					LOGGER.warn("On close resource", e);
				}
			}
		}
	}

	FunctionEvalResult execute(Function[] functions, FunctionContext functionContext, Map<String, Object> context) {
		for (Function function : functions) {
			if (LOGGER.isTraceEnabled()) {
				LOGGER.trace("Execution of: {}", function.getName());
			}
			FunctionEvalResult result = function.evaluate(functionContext, context);
			if (result.isSuccess()) {
				if (LOGGER.isInfoEnabled()) {
					LOGGER.info("Success execution of: {}", function.getName());
				}
				saveValue(function, result.getData(), context);
			} else {
				if (LOGGER.isWarnEnabled()) {
					LOGGER.warn("UnSuccess execution of: {}", function.getName());
				}
				return result;
			}
		}
		return null;
	}

	@SuppressWarnings("unchecked")
	void saveValue(Function function, Object value, Map<String, Object> context) {
		String outputName = function.getOutputName();
		if (outputName == null) {
			if (LOGGER.isWarnEnabled()) {
				LOGGER.warn("Unsaved value for function: {}", function.getName());
			}
		} else {
			String[] outputContext = function.getOutputContext();
			if (outputContext == null || outputContext.length == 0) {
				if (LOGGER.isDebugEnabled()) {
					LOGGER.debug("Save {}: {}", function.getName(), outputName);
				}
				context.put(outputName, value);
			} else {
				if (LOGGER.isTraceEnabled()) {
					LOGGER.trace("Save {}: {} & {}", function.getName(), outputName, Arrays.asList(outputContext));
				}
				Map<String, Object> ctx = context;
				for (String name : outputContext) {
					ctx = (Map<String, Object>) ctx.get(name);
				}
				ctx.put(outputName, value);
			}
		}
	}

}
