package net.sf.aguacate.context.impl;

import java.sql.SQLException;
import java.util.Collection;
import java.util.Map;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import net.sf.aguacate.context.ContextProcessor;
import net.sf.aguacate.field.format.FieldFormat;
import net.sf.aguacate.function.Function;
import net.sf.aguacate.function.FunctionContext;
import net.sf.aguacate.function.FunctionEvalResult;

public class ContextProcessorSql implements ContextProcessor {

	private static final Logger LOGGER = LogManager.getLogger(ContextProcessorSql.class);

	private final Function[] sentences;

	public ContextProcessorSql(Collection<Function> sentences) {
		this.sentences = sentences.toArray(new Function[sentences.size()]);
	}

	@Override
	public Object process(FunctionContext functionContext, String method, Map<String, Object> context,
			Map<String, FieldFormat> outputFields) {
		try {
			return process(method, context, functionContext);
		} catch (RuntimeException | SQLException e) {
			LOGGER.warn("Rollback on exception");
			functionContext.rollback();
			throw new IllegalStateException(e);
		}
	}

	Object process(String method, Map<String, Object> context, FunctionContext functionContext) throws SQLException {
		FunctionEvalResult last = null;
		for (Function sentence : sentences) {
			String name = sentence.getName();
			if (sentence.validFor(method)) {
				LOGGER.trace("trying to execute {} with: {}", name, context);
				FunctionEvalResult sentenceResult = sentence.evaluate(functionContext, context);
				if (sentenceResult.isSuccess()) {
					LOGGER.debug("succesful execution of {}", name);
					String outputName = sentence.getOutputName();
					if (outputName != null) {
						String[] outputContext = sentence.getOutputContext();
						if (outputContext != null && outputContext.length > 0) {
							Map<String, Object> ctx = context;
							for (String ctxName : outputContext) {
								@SuppressWarnings("unchecked")
								Map<String, Object> temp = (Map<String, Object>) ctx.get(ctxName);
								// TODO: check if temp is null
								ctx = temp;
							}
							LOGGER.trace("working ctx: {}", ctx);
							ctx.put(outputName, sentenceResult.getData());
						} else {
							context.put(outputName, sentenceResult.getData());
						}
					}
					LOGGER.trace("context after {}: {}", name, context);
					last = sentenceResult;
				} else {
					LOGGER.warn("unsuccesful execution of {}. Rollback transaction", name);
					functionContext.rollback();
					return null;
				}
			} else {
				LOGGER.trace("avoid execution: {}", name);
			}
		}
		if (last == null) {
			return null;
		} else {
			return last.getData();
		}
	}

}
