package net.sf.aguacate.connector.spi;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import net.sf.aguacate.connector.Connector;
import net.sf.aguacate.connector.ConnectorBridge;
import net.sf.aguacate.util.codec.bridge.CodecCoupling;
import net.sf.aguacate.util.filesystem.EventHandler;
import net.sf.aguacate.util.filesystem.FileSystemObserver;
import net.sf.aguacate.util.resource.ResourceLocator;
import net.sf.aguacate.util.resource.impl.ResourceLocatorClassImpl;
import net.sf.aguacate.util.resource.impl.ResourceLocatorFileImpl;

public class ConnectorBridgeSpi implements ConnectorBridge, EventHandler {

	private static final Logger LOGGER = LogManager.getLogger(ConnectorBridgeSpi.class);

	private static final Logger LOGGER2 = LogManager.getLogger("aguacate.file.connector");

	private static final String SUFFIX = ".json";

	private static final int SUFFIX_LENGTH = 5;

	private static final String ENVIRONMENT = "DIRECTORY_CONNECTOR";

	private static final File DIRECTORY;

	private static final Map<String, Object> SPECS;

	private final ResourceLocator locator;

	private Map<String, Connector> instances;

	private final ConnectorBuilder builder;

	static {
		assert SUFFIX_LENGTH == SUFFIX.length();
		String temp = System.getProperty(ENVIRONMENT);
		if (temp == null || temp.isEmpty()) {
			temp = System.getenv(ENVIRONMENT);
			if (temp == null || temp.isEmpty()) {
				LOGGER.info("No " + ENVIRONMENT + " defined, using default");
				DIRECTORY = null;
			} else {
				LOGGER.info("using " + ENVIRONMENT + " (env): {}", temp);
				DIRECTORY = new File(temp);
			}
		} else {
			LOGGER.info("using " + ENVIRONMENT + " (prop): {}", temp);
			DIRECTORY = new File(temp);
		}
		Map<String, Object> spec = new HashMap<>();
		spec.put("kind", "connector");
		spec.put("format", "1.0");
		SPECS = spec;
	}

	public ConnectorBridgeSpi() {
		this(new ConnectorBuilderImpl());
	}

	public ConnectorBridgeSpi(ConnectorBuilder builder) {
		if (DIRECTORY == null) {
			this.locator = new ResourceLocatorClassImpl(Connector.class);
		} else {
			this.locator = new ResourceLocatorFileImpl(DIRECTORY);
			FileSystemObserver.watch(DIRECTORY.toPath(), this);
		}
		this.builder = builder;
		this.instances = Collections.emptyMap();
	}

	@Override
	public Connector load(String name) {
		return get(name);
	}

	Connector get(String name) {
		Connector connector = instances.get(name);
		if (connector == null) {
			synchronized (this) {
				connector = instances.get(name);
				if (connector == null) {
					String resource = name.concat(SUFFIX);
					connector = load0(resource);
					if (connector != null) {
						LOGGER.trace("original: {}", instances);
						Map<String, Connector> temp = new HashMap<>(instances);
						temp.put(name, connector);
						LOGGER.trace("new: {}", temp);
						instances = temp;
					}
				}
			}
		}
		return connector;
	}

	Connector load0(String file) {
		LOGGER2.info("Loading configuration: {}", file);
		try {
			Connector connector;
			InputStream inputStream = locator.open(file);
			if (inputStream == null) {
				LOGGER.warn("Not Found: {}", file);
				connector = null;
			} else {
				Map<String, Object> data;
				try {
					data = checkSpecs(CodecCoupling.jsonCodecBridge()
							.decodeMap(new InputStreamReader(inputStream, StandardCharsets.UTF_8)));
				} finally {
					try {
						inputStream.close();
					} catch (IOException e) {
						LOGGER.warn("when closing a resource", e);
					}
				}
				LOGGER.trace("decoded: {}", data);
				connector = builder.build(data);
			}
			return connector;
		} catch (IOException e) {
			throw new IllegalStateException(e);
		}
	}

	Map<String, Object> checkSpecs(Map<String, Object> data) {
		@SuppressWarnings("unchecked")
		Map<String, Object> specs = (Map<String, Object>) data.get("specs");
		if (specs == null) {
			throw new IllegalArgumentException("no specs");
		} else {
			if (SPECS.equals(specs)) {
				return data;
			} else {
				throw new IllegalArgumentException(
						"unsupported specs: ".concat(CodecCoupling.jsonCodecBridge().encode(specs)));
			}
		}
	}

	@Override
	public void onDelete(Path directory, Path deleted) {
		String file = deleted.toString();
		if (file.endsWith(SUFFIX)) {
			LOGGER2.warn("Removing configuration definition: {}", deleted);
			String name = removeSufix(file);
			synchronized (this) {
				if (instances.containsKey(name)) {
					Map<String, Connector> temp = new HashMap<>(instances);
					temp.remove(name);
					instances = temp;
				}
			}
		} else {
			LOGGER.debug("ignore deleted file: {}", deleted);
		}
	}

	@Override
	public void onUpdate(Path directory, Path updated) {
		String file = updated.toString();
		if (file.endsWith(SUFFIX)) {
			LOGGER2.info("Change detected on configuration: {}", updated);
			String name = removeSufix(file);
			synchronized (this) {
				if (instances.containsKey(name)) {
					Map<String, Connector> temp = new HashMap<>(instances);
					temp.put(name, load0(file));
					instances = temp;
				}
			}
		} else {
			LOGGER2.warn("Ignored : {}", updated);
			LOGGER.debug("ignore updated file: {}", updated);
		}
	}

	String removeSufix(String name) {
		return name.substring(0, name.length() - SUFFIX_LENGTH);
	}

}
