package net.sf.aguacate.util.http.bridge.spi;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.net.HttpURLConnection;
import java.net.URI;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Map;

import org.apache.http.HttpEntity;
import org.apache.http.HttpMessage;
import org.apache.http.StatusLine;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.entity.ContentType;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import net.sf.aguacate.util.codec.bridge.CodecBridge;
import net.sf.aguacate.util.http.bridge.HttpBridge;

public class HttpBridgeHttpClient implements HttpBridge {

	private static final Logger LOGGER = LogManager.getLogger(HttpBridgeHttpClient.class);

	private static final CloseableHttpClient CLIENT;

	static {
		CLIENT = HttpClients.createDefault();
	}

	@Override
	public Object readMapFromGet(URI uri, CodecBridge codec) throws IOException {
		return readMapFromMethod(new HttpGet(uri), codec);
	}

	@Override
	public Object readMapFromGet(URI uri, Map<String, String> headers, CodecBridge codec) throws IOException {
		return readMapFromMethod(addHeaders(new HttpGet(uri), headers), codec);
	}

	@Override
	public Object readListFromGet(URI uri, CodecBridge codec) throws IOException {
		return readListFromGet(new HttpGet(uri), codec);
	}

	@Override
	public Object readListFromGet(URI uri, Map<String, String> headers, CodecBridge codec) throws IOException {
		return readListFromGet(addHeaders(new HttpGet(uri), headers), codec);
	}

	Object readMapFromMethod(HttpUriRequest request, CodecBridge codec) throws IOException, ClientProtocolException {
		CloseableHttpResponse response = CLIENT.execute(request);
		try {
			StatusLine statusLine = response.getStatusLine();
			if (statusLine == null) {
				throw new IllegalStateException("No status line");
			} else {
				int statusCode = statusLine.getStatusCode();
				if (LOGGER.isInfoEnabled()) {
					LOGGER.info("{}: {}({})", request.getURI(), statusCode, statusLine.getReasonPhrase());
				}
				if (HttpURLConnection.HTTP_OK == statusCode) {
					HttpEntity entity = response.getEntity();
					ContentType contentType = ContentType.get(entity);
					if (contentType == null) {
						throw new IllegalStateException("No Content-Type line");
					} else {
						if (codec.support(contentType.getMimeType())) {
							Reader reader = reader(entity, contentType, StandardCharsets.UTF_8);
							try {
								return codec.decodeMap(reader);
							} finally {
								try {
									reader.close();
								} catch (IOException e) {
									LOGGER.warn("On closing inputStream from HttpResponse", e);
								}
							}
						} else {
							throw new IllegalArgumentException(contentType.getMimeType());
						}
					}
				} else {
					return Integer.valueOf(statusCode);
				}
			}
		} finally {
			try {
				response.close();
			} catch (IOException e) {
				LOGGER.warn("On closing HttpResponse", e);
			}
		}
	}

	Object readListFromGet(HttpUriRequest request, CodecBridge codec) throws IOException, ClientProtocolException {
		CloseableHttpResponse response = CLIENT.execute(request);
		try {
			StatusLine statusLine = response.getStatusLine();
			if (statusLine == null) {
				throw new IllegalStateException("No status line");
			} else {
				int statusCode = statusLine.getStatusCode();
				if (LOGGER.isInfoEnabled()) {
					LOGGER.info("{}: {}({})", request.getURI(), statusCode, statusLine.getReasonPhrase());
				}
				if (HttpURLConnection.HTTP_OK == statusCode) {
					HttpEntity entity = response.getEntity();
					ContentType contentType = ContentType.get(entity);
					if (contentType == null) {
						throw new IllegalStateException("No Content-Type line");
					} else {
						if (codec.support(contentType.getMimeType())) {
							Reader reader = reader(entity, contentType, StandardCharsets.UTF_8);
							try {
								return codec.decodeList(reader);
							} finally {
								try {
									reader.close();
								} catch (IOException e) {
									LOGGER.warn("On closing inputStream from HttpResponse", e);
								}
							}
						} else {
							throw new IllegalArgumentException(contentType.getMimeType());
						}
					}
				} else {
					return Integer.valueOf(statusCode);
				}
			}
		} finally {
			try {
				response.close();
			} catch (IOException e) {
				LOGGER.warn("On closing HttpResponse", e);
			}
		}
	}

	Reader reader(HttpEntity entity, ContentType contentType, Charset defaultCharset) throws IOException {
		InputStream inputStream = entity.getContent();
		if (inputStream == null) {
			throw new IllegalStateException("No content");
		} else {
			Charset temp = contentType.getCharset();
			LOGGER.debug("charset from server: {}", temp);
			Charset charSet;
			if (temp == null) {
				charSet = defaultCharset;
			} else {
				charSet = temp;
			}
			return new InputStreamReader(inputStream, charSet);
		}
	}

	<T extends HttpMessage> T addHeaders(T message, Map<String, String> headers) {
		for (Map.Entry<String, String> entry : headers.entrySet()) {
			message.addHeader(entry.getKey(), entry.getValue());
		}
		return message;
	}

}
