package net.sf.aguacate.util.config.database.spi;

import java.math.BigInteger;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Map;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import net.sf.aguacate.util.parameter.Parameter;

public abstract class AbstractDatabaseInterfaceWithId extends AbstractDatabaseInterface {

	private static final Logger LOGGER = LogManager.getLogger(AbstractDatabaseInterfaceWithId.class);

	@Override
	public BigInteger executeSqlInsertWithId(Connection connection, String sql, Map<String, Object> context,
			Parameter[] required, Parameter[] optional, String[] keys) throws SQLException {
		PreparedStatement statement = prepareStatement(connection, sql, keys);
		try {
			int position = 1;
			for (Parameter parameter : required) {
				if (context.containsKey(parameter.getName())) {
					statement.setObject(position, parameter.calculateContext(context).get(parameter.getName()));
					position += 1;
				}
			}
			for (Parameter parameter : optional) {
				if (context.containsKey(parameter.getName())) {
					statement.setObject(position, parameter.calculateContext(context).get(parameter.getName()));
					position += 1;
				}
			}
			LOGGER.debug("execution: {}", sql);
			int count = statement.executeUpdate();
			if (count == 1) {
				ResultSet generatedKeys = statement.getGeneratedKeys();
				if (generatedKeys.next()) {
					String key = generatedKeys.getString(1);
					LOGGER.debug("result({}): {}", key, sql);
					return new BigInteger(key);
				} else {
					LOGGER.warn("No primary key");
					throw new IllegalStateException("No primary key");
				}
			} else {
				if (LOGGER.isWarnEnabled()) {
					LOGGER.warn("Multiple rows: {}", count);
				}
				throw new IllegalStateException("Multiple rows");
			}
		} finally {
			try {
				statement.close();
			} catch (SQLException e) {
				LOGGER.error("On close statement", e);
			}
		}
	}

	@Override
	public BigInteger executeInsertWithId(Connection connection, String table, Map<String, Object> context,
			Parameter[] required, Parameter[] optional, String[] keys) throws SQLException {
		String sql = buildInsertWithIdSql(connection.getMetaData().getIdentifierQuoteString(), table, context, required,
				optional);
		LOGGER.debug("prepare: {}", sql);
		return executeSqlInsertWithId(connection, sql, context, required, optional, keys);
	}

	public String buildInsertWithIdSql(String quote, String table, Map<String, Object> context, Parameter[] required,
			Parameter[] optional) {
		return buildInsertSql0(quote, table, context, required, optional).toString();
	}

	abstract PreparedStatement prepareStatement(Connection connection, String sql, String[] keys) throws SQLException;

}
