package net.sf.aguacate.util.filesystem.spi;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchEvent.Kind;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.text.MessageFormat;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import net.sf.aguacate.util.filesystem.EventHandler;

public class WatchdogRunner implements Runnable {

	private static final Logger LOGGER = LogManager.getLogger(WatchdogRunner.class);

	private final Path directory;

	private final EventHandler handler;

	private boolean running;

	private Thread thread;

	private static final WatchEvent.Kind<?>[] EVENTS = new WatchEvent.Kind<?>[] { StandardWatchEventKinds.ENTRY_MODIFY,
			StandardWatchEventKinds.ENTRY_DELETE };

	public WatchdogRunner(Path directory, EventHandler handler) {
		if (!Files.isDirectory(directory)) {
			throw new IllegalArgumentException(directory.toString());
		}
		this.directory = directory;
		this.handler = handler;
		running = true;
	}

	@Override
	public void run() {
		thread = Thread.currentThread();
		assert thread != null;
		try {
			WatchService watcher = directory.getFileSystem().newWatchService();
			directory.register(watcher, EVENTS);
			if (LOGGER.isInfoEnabled()) {
				LOGGER.info("Alive {} for {}: {}", Thread.currentThread().getName(), directory, running);
			}
			assert running;
			while (running) {
				try {
					LOGGER.info("Waiting for {} ...", directory);
					WatchKey key = watcher.take();
					LOGGER.debug("Events for {}", directory);
					List<WatchEvent<?>> events = key.pollEvents();
					int size = events.size();
					for (int i = 0; i < size; i++) {
						LOGGER.debug("Change on: {}", directory);
						WatchEvent<?> event = events.get(i);
						Kind<?> kind = event.kind();
						if (StandardWatchEventKinds.ENTRY_MODIFY.equals(kind)) {
							@SuppressWarnings("unchecked")
							Path child = ((WatchEvent<Path>) event).context();
							LOGGER.trace("updated: {} ({})", child, directory);
							try {
								handler.onUpdate(directory, child);
							} catch (RuntimeException e) {
								LOGGER.error(MessageFormat.format("On update event for {0} ({1})", child, directory),
										e);
							}
						} else {
							if (StandardWatchEventKinds.ENTRY_DELETE.equals(kind)) {
								@SuppressWarnings("unchecked")
								Path child = ((WatchEvent<Path>) event).context();
								LOGGER.trace("deleted: {} ({})", child, directory);
								try {
									handler.onDelete(directory, child);
								} catch (RuntimeException e) {
									LOGGER.error(
											MessageFormat.format("On delete event for {0} ({1})", child, directory), e);
								}
							}
						}
					}
					key.reset();
				} catch (InterruptedException e) {
					LOGGER.error("Forced Wakeup");
				}
			}
		} catch (IOException e) {
			LOGGER.error("On check directory", e);
		}
	}

	void shutdown() {
		if (LOGGER.isWarnEnabled()) {
			LOGGER.warn("stopping ... {}", Thread.currentThread().getName());
		}
		running = false;
		thread.interrupt();
	}

}
