package net.sf.aguacate.util.config.database.spi;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import net.sf.aguacate.util.codec.bridge.CodecCoupling;
import net.sf.aguacate.util.config.database.DatabaseBridge;
import net.sf.aguacate.util.filesystem.EventHandler;
import net.sf.aguacate.util.filesystem.FileSystemObserver;
import net.sf.aguacate.util.resource.ResourceLocator;
import net.sf.aguacate.util.resource.impl.ResourceLocatorClassImpl;
import net.sf.aguacate.util.resource.impl.ResourceLocatorFileImpl;

public class DatabaseBridgeLoaderSpi implements DatabaseBridgeLoader, EventHandler {

	private static final Logger LOGGER = LogManager.getLogger(DatabaseBridgeLoaderSpi.class);

	private static final String ENVIRONMENT = "DIRECTORY_DATABASE";

	private static final String SUFFIX = ".json";

	private static final int SUFFIX_LENGTH = 5;

	private static final File DIRECTORY;

	private static final Map<String, Object> SPECS;

	private Map<String, DatabaseBridge> bridges;

	private final DatabaseBridgeBuilder builder;

	private final ResourceLocator locator;

	static {
		assert SUFFIX_LENGTH == SUFFIX.length();
		String temp = System.getProperty(ENVIRONMENT);
		if (temp == null || temp.isEmpty()) {
			temp = System.getenv(ENVIRONMENT);
			if (temp == null || temp.isEmpty()) {
				LOGGER.info("No " + ENVIRONMENT + " defined, using default");
				DIRECTORY = null;
			} else {
				LOGGER.info("using " + ENVIRONMENT + " (env): {}", temp);
				DIRECTORY = new File(temp);
			}
		} else {
			LOGGER.info("using " + ENVIRONMENT + " (prop): {}", temp);
			DIRECTORY = new File(temp);
		}
		Map<String, Object> spec = new HashMap<>();
		spec.put("kind", "database");
		spec.put("impl", "hikari");
		spec.put("format", "1.0");
		SPECS = spec;
	}

	public DatabaseBridgeLoaderSpi() {
		if (DIRECTORY == null) {
			locator = new ResourceLocatorClassImpl(DatabaseBridge.class);
		} else {
			locator = new ResourceLocatorFileImpl(DIRECTORY);
			FileSystemObserver.watch(DIRECTORY.toPath(), this);
		}
		builder = new DatabaseBridgeBuilderHikari();
		bridges = Collections.emptyMap();
		assert bridges != null && bridges.isEmpty();
	}

	@Override
	public DatabaseBridge get(String name) {
		DatabaseBridge bridge = bridges.get(name);
		if (bridge == null) {
			synchronized (this) {
				bridge = bridges.get(name);
				if (bridge == null) {
					bridge = load0(name.concat(SUFFIX));
					Map<String, DatabaseBridge> temp = new HashMap<>(bridges);
					temp.put(name, bridge);
					bridges = temp;
				}
			}
		}
		return bridge;
	}

	public DatabaseBridge load0(String file) {
		LOGGER.debug("trying to load {}", file);
		try {
			return builder.build(checkSpecs(readConfiguration(file)));
		} catch (IOException e) {
			LOGGER.error("on opening resource", e);
			throw new IllegalStateException(e);
		}
	}

	Map<String, Object> checkSpecs(Map<String, Object> data) {
		@SuppressWarnings("unchecked")
		Map<String, Object> specs = (Map<String, Object>) data.get("specs");
		if (specs == null) {
			throw new IllegalArgumentException("no specs");
		} else {
			if (SPECS.equals(specs)) {
				return data;
			} else {
				throw new IllegalArgumentException(
						"unsupported specs: ".concat(CodecCoupling.jsonCodecBridge().encode(specs)));
			}
		}
	}

	Map<String, Object> readConfiguration(String file) throws IOException {
		InputStream inputStream = locator.open(file);
		if (inputStream == null) {
			LOGGER.warn("no configuration for {}", file);
			throw new UnsupportedOperationException();
		} else {
			try {
				return CodecCoupling.jsonCodecBridge()
						.decodeMap(new InputStreamReader(inputStream, StandardCharsets.UTF_8));
			} catch (IOException e) {
				throw new IllegalStateException(e);
			} finally {
				try {
					inputStream.close();
				} catch (IOException e) {
					LOGGER.error("on closing resource", e);
				}
			}
		}
	}

	@Override
	public void onDelete(Path directory, Path deleted) {
		String file = deleted.toString();
		if (file.endsWith(SUFFIX)) {
			String name = removeSufix(file);
			synchronized (this) {
				if (bridges.containsKey(name)) {
					Map<String, DatabaseBridge> temp = new HashMap<>(bridges);
					temp.remove(name);
					bridges = temp;
					LOGGER.info("deleted: {} & {}", directory, deleted);
				} else {
					LOGGER.info("Not loaded: {} & {}", directory, deleted);
				}
			}
		} else {
			LOGGER.debug("ignore deleted file: {} & {}", directory, deleted);
		}
	}

	@Override
	public void onUpdate(Path directory, Path updated) {
		String file = updated.toString();
		if (file.endsWith(SUFFIX)) {
			String name = removeSufix(file);
			synchronized (this) {
				if (bridges.containsKey(name)) {
					Map<String, DatabaseBridge> temp = new HashMap<>(bridges);
					temp.put(name, load0(file));
					bridges = temp;
					LOGGER.info("loaded: {} & {}", directory, updated);
				} else {
					LOGGER.info("not loaded until required: {} & {}", directory, updated);
				}
			}
		} else {
			LOGGER.debug("ignore updated file: {} & {}", directory, updated);
		}
	}

	String removeSufix(String name) {
		return name.substring(0, name.length() - SUFFIX_LENGTH);
	}

}
