package net.sf.aguacate.util.config.database.spi;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import net.sf.aguacate.util.codec.bridge.CodecCoupling;
import net.sf.aguacate.util.config.database.DatabaseBridge;
import net.sf.aguacate.util.filesystem.EventHandler;
import net.sf.aguacate.util.filesystem.FileSystemObserver;
import net.sf.aguacate.util.resource.ResourceLocator;
import net.sf.aguacate.util.resource.impl.ResourceLocatorClassImpl;
import net.sf.aguacate.util.resource.impl.ResourceLocatorFileImpl;

public class DatabaseBridgeLoaderSpi implements DatabaseBridgeLoader, EventHandler {

	private static final Logger LOGGER = LogManager.getLogger(DatabaseBridgeLoaderSpi.class);

	private static final String ENVIRONMENT = "DIRECTORY_DATABASE";

	private static final String SUFFIX = ".json";

	private static final int SUFFIX_LENGTH = 5;

	private static final File DIRECTORY;

	private Map<String, DatabaseBridge> bridges;

	private final DatabaseBridgeBuilder builder;

	private final ResourceLocator locator;

	static {
		assert SUFFIX_LENGTH == SUFFIX.length();
		String temp = System.getProperty(ENVIRONMENT);
		if (temp == null || temp.isEmpty()) {
			temp = System.getenv(ENVIRONMENT);
			if (temp == null || temp.isEmpty()) {
				LOGGER.info("No " + ENVIRONMENT + " defined, using default");
				DIRECTORY = null;
			} else {
				LOGGER.info("using " + ENVIRONMENT + " (env): {}", temp);
				DIRECTORY = new File(temp);
			}
		} else {
			LOGGER.info("using " + ENVIRONMENT + " (prop): {}", temp);
			DIRECTORY = new File(temp);
		}
	}

	public DatabaseBridgeLoaderSpi() {
		String temp = System.getProperty(ENVIRONMENT);
		if (temp == null) {
			locator = new ResourceLocatorClassImpl(DatabaseBridgeLoader.class);
		} else {
			locator = new ResourceLocatorFileImpl(DIRECTORY);
			FileSystemObserver.watch(DIRECTORY.toPath(), this);
		}
		builder = new DatabaseBridgeBuilderHikari();
		bridges = Collections.emptyMap();
		assert bridges != null && bridges.isEmpty();
	}

	@Override
	public DatabaseBridge get(String name) {
		DatabaseBridge bridge = bridges.get(name);
		if (bridge == null) {
			synchronized (this) {
				bridge = bridges.get(name);
				if (bridge == null) {
					bridge = load0(name.concat(SUFFIX));
					Map<String, DatabaseBridge> temp = new HashMap<>(bridges);
					temp.put(name, bridge);
					bridges = temp;
				}
			}
		}
		return bridge;
	}

	public DatabaseBridge load0(String file) {
		LOGGER.debug("trying to load {}", file);
		try {
			return builder.build(readConfiguration(file));
		} catch (IOException e) {
			LOGGER.error("on opening resource", e);
			throw new IllegalStateException(e);
		}
	}

	Map<String, Object> readConfiguration(String file) throws IOException {
		InputStream inputStream = locator.open(file);
		if (inputStream == null) {
			LOGGER.warn("no configuration for {}", file);
			throw new UnsupportedOperationException();
		} else {
			try {
				return CodecCoupling.jsonCodecBridge()
						.decodeMap(new InputStreamReader(inputStream, StandardCharsets.UTF_8));
			} catch (IOException e) {
				throw new IllegalStateException(e);
			} finally {
				try {
					inputStream.close();
				} catch (IOException e) {
					LOGGER.error("on closing resource", e);
				}
			}
		}
	}

	@Override
	public void onDelete(Path directory, Path deleted) {
		String file = deleted.toString();
		if (file.endsWith(SUFFIX)) {
			String name = removeSufix(file);
			synchronized (this) {
				if (bridges.containsKey(name)) {
					Map<String, DatabaseBridge> temp = new HashMap<>(bridges);
					temp.remove(name);
					bridges = temp;
				}
			}
		} else {
			LOGGER.debug("ignore deleted file: {}", deleted);
		}
	}

	@Override
	public void onUpdate(Path directory, Path updated) {
		String file = updated.toString();
		if (file.endsWith(SUFFIX)) {
			String name = removeSufix(file);
			synchronized (this) {
				if (bridges.containsKey(name)) {
					Map<String, DatabaseBridge> temp = new HashMap<>(bridges);
					temp.put(name, load0(file));
					bridges = temp;
				}
			}
		} else {
			LOGGER.debug("ignore updated file: {}", updated);
		}
	}

	String removeSufix(String name) {
		return name.substring(0, name.length() - SUFFIX_LENGTH);
	}

}
