package net.sf.aguacate.util.config.database.spi;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import net.sf.aguacate.util.config.database.DatabaseInterface;

public abstract class AbstractDatabaseInterface implements DatabaseInterface {

	private static final Logger LOGGER = LogManager.getLogger(AbstractDatabaseInterface.class);

	private static final Logger MESSAGE_LOGGER = LogManager.getLogger("aguacate.sentence");

	private static final String STR_AND = " AND ";

	private static final int STR_AND_LENGTH = 5;

	static {
		assert STR_AND_LENGTH == STR_AND.length();
	}

	@Override
	public Map<String, Object> executeSqlSelectSingleRow(String name, String message, Connection connection, String sql,
			Map<String, Object> context, String[] required, String[] optional) throws SQLException {
		PreparedStatement statement = connection.prepareStatement(sql);
		try {
			Object[] arguments = setSelectParameters(context, required, statement);
			String msg = MessageFormat.format(message, arguments);
			ResultSet resultSet = statement.executeQuery();
			try {
				if (resultSet.next()) {
					Map<String, Object> map = extract(optional, resultSet);
					if (resultSet.next()) {
						LOGGER.warn("Multiple results");
					}
					logSuccess(name, msg);
					return map;
				} else {
					logFailure(name, msg);
					return null;
				}
			} finally {
				try {
					resultSet.close();
				} catch (SQLException e) {
					LOGGER.error("On close resultSet", e);
				}
			}
		} finally {
			try {
				statement.close();
			} catch (SQLException e) {
				LOGGER.error("On close statement", e);
			}
		}
	}

	@Override
	public List<Map<String, Object>> executeSqlSelectMultipleRow(String name, String message, Connection connection,
			String sql, Map<String, Object> context, String[] required, String[] optional) throws SQLException {
		PreparedStatement statement = connection.prepareStatement(sql);
		try {
			Object[] arguments = setSelectParameters(context, required, statement);
			String msg = MessageFormat.format(message, arguments);
			ResultSet resultSet = statement.executeQuery();
			try {
				if (resultSet.next()) {
					Map<String, Object> first = extract(optional, resultSet);
					if (resultSet.next()) {
						List<Map<String, Object>> list = new ArrayList<>();
						list.add(first);
						do {
							list.add(extract(optional, resultSet));
						} while (resultSet.next());
						logSuccess(name, msg);
						return list;
					} else {
						logSuccess(name, msg);
						return Collections.singletonList(first);
					}
				} else {
					logFailure(name, msg);
					return null;
				}
			} finally {
				try {
					resultSet.close();
				} catch (SQLException e) {
					LOGGER.error("On close resultSet", e);
				}
			}
		} finally {
			try {
				statement.close();
			} catch (SQLException e) {
				LOGGER.error("On close statement", e);
			}
		}
	}

	@Override
	public Object executeSqlSelectValue(String name, String message, Connection connection, String sql,
			Map<String, Object> context, String[] required, String[] optional) throws SQLException {
		PreparedStatement statement = connection.prepareStatement(sql);
		try {
			Object[] arguments = setSelectParameters(context, required, statement);
			String msg = MessageFormat.format(message, arguments);
			ResultSet resultSet = statement.executeQuery();
			try {
				if (resultSet.next()) {
					Object value = resultSet.getObject(1);
					if (resultSet.next()) {
						LOGGER.warn("Multiple rows");
					}
					logSuccess(name, msg);
					return value;
				} else {
					logFailure(name, msg);
					return null;
				}
			} finally {
				try {
					resultSet.close();
				} catch (SQLException e) {
					LOGGER.error("On close resultSet", e);
				}
			}
		} finally {
			try {
				statement.close();
			} catch (SQLException e) {
				LOGGER.error("On close statement", e);
			}
		}
	}

	@Override
	public int executeSqlUpdate(Connection connection, String sql, Map<String, Object> context, String[] required,
			String[] optional) throws SQLException {
		PreparedStatement statement = connection.prepareStatement(sql);
		try {
			int position = 1;
			for (String parameter : optional) {
				if (context.containsKey(parameter)) {
					statement.setObject(position, context.get(parameter));
					position += 1;
				}
			}
			for (String parameter : required) {
				if (context.containsKey(parameter)) {
					statement.setObject(position, context.get(parameter));
					position += 1;
				}
			}
			return statement.executeUpdate();
		} finally {
			try {
				statement.close();
			} catch (SQLException e) {
				LOGGER.error("On close statement", e);
			}
		}
	}

	@Override
	public int executeSqlInsert(Connection connection, String sql, Map<String, Object> context, String[] required,
			String[] optional) throws SQLException {
		PreparedStatement statement = connection.prepareStatement(sql);
		try {
			int position = 1;
			for (String parameter : required) {
				if (context.containsKey(parameter)) {
					statement.setObject(position, context.get(parameter));
					position += 1;
				}
			}
			for (String parameter : optional) {
				if (context.containsKey(parameter)) {
					statement.setObject(position, context.get(parameter));
					position += 1;
				}
			}
			return statement.executeUpdate();
		} finally {
			try {
				statement.close();
			} catch (SQLException e) {
				LOGGER.error("On close statement", e);
			}
		}
	}

	@Override
	public Map<String, Object> executeSelectSingleRow(String name, String message, Connection connection, String table,
			Map<String, Object> context, String[] required, String[] optional) throws SQLException {
		String sql = buildSelectSql(connection.getMetaData().getIdentifierQuoteString(), table, context, required,
				optional);
		LOGGER.debug(sql);
		return executeSqlSelectSingleRow(name, message, connection, sql, context, required, optional);
	}

	@Override
	public List<Map<String, Object>> executeSelectMultipleRow(String name, String message, Connection connection,
			String table, Map<String, Object> context, String[] required, String[] optional) throws SQLException {
		String sql = buildSelectSql(connection.getMetaData().getIdentifierQuoteString(), table, context, required,
				optional);
		LOGGER.debug(sql);
		return executeSqlSelectMultipleRow(name, message, connection, sql, context, required, optional);
	}

	public Object executeSelectValue(String name, String message, Connection connection, String table,
			Map<String, Object> context, String[] required, String[] optional) throws SQLException {
		String sql = buildSelectSql(connection.getMetaData().getIdentifierQuoteString(), table, context, required,
				optional);
		LOGGER.debug(sql);
		return executeSqlSelectValue(name, message, connection, sql, context, required, optional);
	}

	@Override
	public int executeUpdate(Connection connection, String table, Map<String, Object> context, String[] required,
			String[] optional) throws SQLException {
		String sql = buildUpdateSql(connection.getMetaData().getIdentifierQuoteString(), table, context, required,
				optional);
		LOGGER.debug(sql);
		return executeSqlUpdate(connection, sql, context, required, optional);
	}

	@Override
	public int executeInsert(Connection connection, String table, Map<String, Object> context, String[] required,
			String[] optional) throws SQLException {
		String sql = buildInsertSql(connection.getMetaData().getIdentifierQuoteString(), table, context, required,
				optional);
		LOGGER.debug(sql);
		return executeSqlInsert(connection, sql, context, required, optional);
	}

	public String buildInsertSql(String quote, String table, Map<String, Object> context, String[] required,
			String[] optional) {
		return buildInsertSql0(quote, table, context, required, optional).toString();
	}

	public abstract String buildInsertWithIdSql(String quote, String table, Map<String, Object> context,
			String[] required, String[] optional);

	public StringBuilder buildInsertSql0(String quote, String table, Map<String, Object> context, String[] required,
			String[] optional) {
		StringBuilder principal = new StringBuilder();
		StringBuilder auxiliar = new StringBuilder();
		principal.append("INSERT INTO ").append(quote).append(table).append(quote).append('(');
		auxiliar.append(") VALUES(");

		for (String parameter : required) {
			if (context.containsKey(parameter)) {
				principal.append(quote).append(parameter).append(quote).append(',');
				auxiliar.append('?').append(',');
			} else {
				return null;
			}
		}

		for (String parameter : optional) {
			if (context.containsKey(parameter)) {
				principal.append(quote).append(parameter).append(quote).append(',');
				auxiliar.append('?').append(',');
			}
		}

		int index = principal.length() - 1;
		if (principal.charAt(index) == ',') {
			principal.setLength(index);
			auxiliar.setLength(auxiliar.length() - 1);
		} else {
			// Empty sentence?
			return null;
		}

		return principal.append(auxiliar).append(')');
	}

	public String buildUpdateSql(String quote, String table, Map<String, Object> context, String[] required,
			String[] optional) {
		StringBuilder principal = new StringBuilder();
		StringBuilder auxiliar = new StringBuilder();
		principal.append("UPDATE ").append(quote).append(table).append(quote).append(" SET ");
		auxiliar.append(" WHERE ");

		for (String parameter : required) {
			if (context.containsKey(parameter)) {
				auxiliar.append(quote).append(parameter).append(quote).append("=? AND ");
			} else {
				LOGGER.error("Missing parameter: {}", parameter);
				throw new IllegalArgumentException("Missing required parameter: ".concat(parameter));
			}
		}

		if (auxiliar.toString().endsWith(STR_AND)) {
			auxiliar.setLength(auxiliar.length() - STR_AND_LENGTH);
		} else {
			LOGGER.warn("No conditions in UPDATE");
			throw new IllegalStateException("No required parameters cofigured");
		}

		for (String parameter : optional) {
			if (context.containsKey(parameter)) {
				principal.append(quote).append(parameter).append(quote).append("=?,");
			}
		}

		int position = principal.length() - 1;
		if (principal.charAt(position) == ',') {
			principal.setLength(position);
		} else {
			LOGGER.warn("No data ^{}^ & ^{}^; required: {}, optional: {}", principal, auxiliar);
			return null;
		}

		return principal.append(auxiliar).toString();
	}

	public String buildSelectSql(String quote, String table, Map<String, Object> context, String[] required,
			String[] optional) {
		StringBuilder builder = new StringBuilder("SELECT ");
		for (String parameter : optional) {
			builder.append(quote).append(parameter).append(quote).append(',');
		}
		int position = builder.length() - 1;
		if (builder.charAt(position) == ',') {
			builder.setLength(position);
		} else {
			LOGGER.warn("No parameters");
			return null;
		}

		builder.append(" FROM ").append(quote).append(table).append(quote);

		if (required.length == 0) {
			return builder.toString();
		} else {
			builder.append(" WHERE ");

			for (String parameter : required) {
				if (context.containsKey(parameter)) {
					builder.append(quote).append(parameter).append(quote).append("=?").append(STR_AND);
				} else {
					LOGGER.warn("Invalid parameter: {}", parameter);
					return null;
				}
			}

			position = builder.length() - STR_AND_LENGTH;
			int count = 0;
			for (count = 0; count < STR_AND_LENGTH; count++) {
				if (STR_AND.charAt(count) != builder.charAt(position + count)) {
					break;
				}
			}

			if (count == STR_AND_LENGTH) {
				return builder.substring(0, position);
			} else {
				LOGGER.warn("Size mismatch");
				return null;
			}
		}
	}

	Object[] setSelectParameters(Map<String, Object> context, String[] required, PreparedStatement statement)
			throws SQLException {
		int length = required.length;
		Object[] data = new Object[length];
		for (int i = 0; i < length; i++) {
			Object value = context.get(required[i]);
			statement.setObject(i + 1, value);
			data[i] = value;
		}
		return data;
	}

	Map<String, Object> extract(String[] optional, ResultSet resultSet) throws SQLException {
		Map<String, Object> map = new HashMap<>();
		for (String field : optional) {
			map.put(field, resultSet.getObject(field));
		}
		return map;
	}

	void logFailure(String name, String msg) {
		MESSAGE_LOGGER.warn("{}({}): failure", msg, name);
	}

	void logSuccess(String name, String msg) {
		MESSAGE_LOGGER.info("{}({}): success", msg, name);
	}

}
