package net.sf.aguacate.util.config.database.spi;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.Map;
import java.util.Properties;

import javax.sql.DataSource;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.HikariDataSource;

import net.sf.aguacate.util.config.database.DatabaseBridge;
import net.sf.aguacate.util.config.database.DatabaseConfigurationLoader;
import net.sf.aguacate.util.resource.ResourceLocator;
import net.sf.aguacate.util.resource.impl.ResourceLocatorClassImpl;
import net.sf.aguacate.util.resource.impl.ResourceLocatorFileImpl;

public class DatabaseBridgeLoaderSpi implements DatabaseConfigurationLoader {

	private static final Logger LOGGER = LogManager.getLogger(DatabaseBridgeLoaderSpi.class);

	private static final String DIRECTORY = "DIRECTORY_DATABASE";

	private static final ObjectMapper MAPPER;

	private final ResourceLocator locator;

	static {
		MAPPER = new ObjectMapper();
	}

	public DatabaseBridgeLoaderSpi() {
		String temp = System.getProperty(DIRECTORY);
		if (temp == null || temp.isEmpty()) {
			temp = System.getenv(DIRECTORY);
			if (temp == null || temp.isEmpty()) {
				LOGGER.info("No " + DIRECTORY + " defined, using default");
				locator = new ResourceLocatorClassImpl(DatabaseBridgeLoaderSpi.class);
			} else {
				LOGGER.info("using " + DIRECTORY + " (env): {}", temp);
				locator = new ResourceLocatorFileImpl(temp);
			}
		} else {
			LOGGER.info("using " + DIRECTORY + " (prop): {}", temp);
			locator = new ResourceLocatorFileImpl(temp);
		}
	}

	@Override
	public DatabaseBridge get(String name) {
		String file = name.concat(".json");
		LOGGER.debug("trying to load {}", file);
		try {
			Map<String, Object> configuration = readConfiguration(file);
			String vendor = (String) configuration.get("vendor");
			if ("mysql".equalsIgnoreCase(vendor)) {
				return new DatabaseBridge(new DatabaseInterfaceMysql(), datasource(name, configuration));
			} else {
				if ("hsqldb".equalsIgnoreCase(vendor)) {
					return new DatabaseBridge(new DatabaseInterfaceHsqldb(), datasource(name, configuration));
				} else {
					throw new IllegalArgumentException(vendor);
				}
			}
		} catch (IOException e) {
			LOGGER.error("on opening resource", e);
			throw new IllegalStateException(e);
		}
	}

	@SuppressWarnings("unchecked")
	DataSource datasource(String name, Map<String, Object> configuration) {
		Properties properties = new Properties();
		properties.putAll((Map<String, String>) configuration.get("properties"));
		HikariDataSource ds = new HikariDataSource(new HikariConfig(properties));
		LOGGER.info("Succesful load of {}", name);
		return ds;
	}

	@SuppressWarnings("unchecked")
	Map<String, Object> readConfiguration(String file) throws IOException {
		InputStream inputStream = locator.open(file);
		if (inputStream == null) {
			LOGGER.warn("no configuration for {}", file);
			throw new UnsupportedOperationException();
		} else {
			try {
				return (Map<String, Object>) MAPPER
						.readValue(new InputStreamReader(inputStream, StandardCharsets.UTF_8), Map.class);
			} catch (IOException e) {
				throw new IllegalStateException(e);
			} finally {
				try {
					inputStream.close();
				} catch (IOException e) {
					LOGGER.error("on closing resource", e);
				}
			}
		}
	}

}
