package net.sf.aguacate.util.config.datasource.spi;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.Properties;

import javax.sql.DataSource;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.HikariDataSource;

import net.sf.aguacate.util.config.datasource.DataSourceLoader;
import net.sf.aguacate.util.resource.ResourceLocator;
import net.sf.aguacate.util.resource.impl.ResourceLocatorClassImpl;
import net.sf.aguacate.util.resource.impl.ResourceLocatorFileImpl;

public class DataSourceLoaderImpl implements DataSourceLoader {

	private static final String DIRECTORY_DATASOURCE = "DIRECTORY_DATASOURCE";

	private static final Logger LOGGER = LogManager.getLogger(DataSourceLoaderImpl.class);

	private final ResourceLocator locator;

	public DataSourceLoaderImpl() {
		String temp = System.getProperty(DIRECTORY_DATASOURCE);
		if (temp == null || temp.isEmpty()) {
			temp = System.getenv(DIRECTORY_DATASOURCE);
			if (temp == null || temp.isEmpty()) {
				LOGGER.info("No " + DIRECTORY_DATASOURCE + " defined, using default");
				locator = new ResourceLocatorClassImpl(DataSourceLoaderImpl.class);
			} else {
				LOGGER.info("using " + DIRECTORY_DATASOURCE + " (env): {}", temp);
				locator = new ResourceLocatorFileImpl(temp);
			}
		} else {
			LOGGER.info("using " + DIRECTORY_DATASOURCE + " (prop): {}", temp);
			locator = new ResourceLocatorFileImpl(temp);
		}
	}

	@Override
	public DataSource getDataSource(String name) {
		String file = name.concat(".properties");
		LOGGER.debug("trying to load {}", file);
		try {
			Properties properties = toProperties(file);
			HikariDataSource ds = new HikariDataSource(new HikariConfig(properties));
			LOGGER.info("Succesful load of {}", name);
			return ds;
		} catch (IOException e) {
			LOGGER.error("on opening resource", e);
			throw new IllegalStateException(e);
		}
	}

	Properties toProperties(String file) throws IOException {
		InputStream inputStream = locator.open(file);
		if (inputStream == null) {
			LOGGER.warn("no configuration for {}", file);
			throw new UnsupportedOperationException();
		} else {
			try {
				Properties properties = new Properties();
				InputStreamReader reader = new InputStreamReader(inputStream, StandardCharsets.UTF_8);
				properties.load(reader);
				LOGGER.trace("readed properties: {}", properties);
				return properties;
			} catch (IOException e) {
				throw new IllegalStateException(e);
			} finally {
				try {
					inputStream.close();
				} catch (IOException e) {
					LOGGER.error("on closing resource", e);
				}
			}
		}
	}

}
