package net.secodo.jcircuitbreaker.breakstrategy.impl;

import net.secodo.jcircuitbreaker.breaker.execution.ExecutedTask;
import net.secodo.jcircuitbreaker.breaker.execution.ExecutionContext;
import net.secodo.jcircuitbreaker.breakstrategy.BreakStrategy;
import net.secodo.jcircuitbreaker.util.TimeUtil;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;


public class TooLongCurrentAverageExecutionTimeStrategy implements BreakStrategy {

  /**
   * Protected access to allow extending classes to dynamically change it,
   * e.g. via MBean
   */
  protected float maxAllowedExecutionTimeMillis;

  /**
   * Protected access to allow extending classes to dynamically change it,
   * e.g. via MBean
   */
  protected float percentageOfMaxTimesToSkip;

  private final TimeUtil timeUtil;

  public TooLongCurrentAverageExecutionTimeStrategy(long maxAllowedAverageExecutionTimeMillis,
                                                    int percentageOfLongestTimesToSkip) {
    this.maxAllowedExecutionTimeMillis = maxAllowedAverageExecutionTimeMillis;
    timeUtil = new TimeUtil();

    this.percentageOfMaxTimesToSkip = ((percentageOfLongestTimesToSkip >= 0) && (percentageOfLongestTimesToSkip <= 100))
      ? (0.01f * (float) percentageOfLongestTimesToSkip) : 0;
  }

  public TooLongCurrentAverageExecutionTimeStrategy(long maxAllowedExecutionTimeMilis) {
    this(maxAllowedExecutionTimeMilis, 0);
  }

  @Override
  public boolean shouldBreak(ExecutionContext executionContext) {
    final Collection<ExecutedTask> executionsInProgress = executionContext.getExecutionsInProgress();

    if (executionsInProgress.isEmpty()) {
      return false;
    }

    final List<Long> startTimestamps = executionsInProgress.stream()
      .map(ExecutedTask::getExecutionStaredTimestamp)
      .collect(Collectors.toList());

    Collections.sort(startTimestamps);

    final List<Long> interestingTimestamps = startTimestamps.subList(0,
      (int) ((1f - percentageOfMaxTimesToSkip) * (float) startTimestamps.size()));

    if (interestingTimestamps.isEmpty()) {
      return false;
    }


    final long currentTime = timeUtil.getCurrentTimeMilis();

    final Long sumOfInterestingTimestamps = interestingTimestamps.stream().mapToLong(t -> currentTime - t).sum();


    final float currentAverageExecutionTime = (float) sumOfInterestingTimestamps / (float) interestingTimestamps.size();


    return currentAverageExecutionTime > maxAllowedExecutionTimeMillis;
  }
}
