package net.secodo.jcircuitbreaker.breaker;

import net.secodo.jcircuitbreaker.CircuitBreakerException;
import net.secodo.jcircuitbreaker.breaker.execution.ExecutionContext;
import net.secodo.jcircuitbreaker.breakhandler.BreakHandler;
import net.secodo.jcircuitbreaker.breakstrategy.BreakStrategy;
import java.util.concurrent.Callable;


public interface CircuitBreaker {
  /**
   * Executed the given {@link Callable} only when given {@link BreakStrategy} allows the execution at the time of call.
   * If execution is not allowed {@link BreakHandler} is used to handle this situation and return fallback value.
   *
   * @param task the {@link Callable} to be executed - typically this is java method
   * @param breakStrategy decides whether to execute the task or prevent execution and executed breakHandler instead
   * @param breakHandler handles the situation when the strategy does not allow method execution
   * @param <R> the return value that the breaker returns (if breaker wants to return any value)
   *
   * @return the return value of the executed task or the fallback value returned by breakHandler in case of execution of method did not take place
   * @throws CircuitBreakerException if the method execution resulted in exception or breakHandler resulted in exception
   */
  <R> R execute(Callable<R> task, BreakStrategy breakStrategy, BreakHandler<R> breakHandler)
         throws CircuitBreakerException;


  /**
   * Executed the given {@link Callable} only when given {@link BreakStrategy} allows the execution at the time of call.
   * If execution is not allowed {@link BreakHandler} is used to handle this situation and return fallback value.
   *
   * Allows to pass some custom data to the strategy which will be available via {@link ExecutionContext#getUserData()}
   *
   * @param task the {@link Callable} to be executed - typically this is java method
   * @param breakStrategy decides whether to execute the task or prevent execution and executed breakHandler instead
   * @param breakHandler handles the situation when the strategy does not allow method execution
   * @param userData custom user data which will be available to strategies via {@link ExecutionContext#getUserData()}
   * @param <R> the return value that the breaker returns (if breaker wants to return any value)
   * @param <U> the type of user data passed to execution context
   *
   * @return the return value of the executed task or the fallback value returned by breakHandler in case of execution of method did not take place
   * @throws CircuitBreakerException if the method execution resulted in exception or breakHandler resulted in exception
   */
  <R, U> R execute(Callable<R> task, BreakStrategy breakStrategy, BreakHandler<R> breakHandler, U userData)
            throws CircuitBreakerException;

}
