/*
 * Copyright © 2022 Pl4yingNight (pl4yingnight@gmail.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.runeduniverse.lib.rogm.pattern;

import static net.runeduniverse.lib.utils.common.StringUtils.isBlank;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Map;
import lombok.Getter;
import net.runeduniverse.lib.rogm.annotations.Direction;
import net.runeduniverse.lib.rogm.annotations.PreDelete;
import net.runeduniverse.lib.rogm.annotations.PreSave;
import net.runeduniverse.lib.rogm.annotations.RelationshipEntity;
import net.runeduniverse.lib.rogm.annotations.StartNode;
import net.runeduniverse.lib.rogm.annotations.TargetNode;
import net.runeduniverse.lib.rogm.pipeline.chain.data.SaveContainer;
import net.runeduniverse.lib.rogm.querying.IDataContainer;
import net.runeduniverse.lib.rogm.querying.IFilter;
import net.runeduniverse.lib.rogm.querying.IQueryBuilder;
import net.runeduniverse.lib.rogm.querying.QueryBuilder.NodeQueryBuilder;
import net.runeduniverse.lib.rogm.querying.QueryBuilder.RelationQueryBuilder;

//deprecation = internal use
@SuppressWarnings("deprecation")
public class RelationPattern extends APattern<RelationQueryBuilder> implements IRelationPattern<RelationQueryBuilder> {

	@Getter
	private String label = null;
	private Direction direction = null;
	private FieldPattern startField = null;
	private FieldPattern targetField = null;
	// start eq target
	@Getter
	private boolean stEqTr = false;
	private boolean readonlyStart = false;
	private boolean readonlyTarget = false;

	public RelationPattern(Archive archive, String pkg, ClassLoader loader, Class<?> type) {
		super(archive, pkg, loader, type);

		RelationshipEntity typeAnno = this.type.getAnnotation(RelationshipEntity.class);
		this.direction = typeAnno.direction();
		this.label = typeAnno.label();
	}

	@Override
	public void validate() throws Exception {
		super.validate();
		this.startField = super.getField(StartNode.class);

		if (this.startField == null)
			throw new Exception("Relation<" + type + "> is missing the @StartNode");
		if (Collection.class.isAssignableFrom(this.startField.getType()))
			throw new Exception("@StartNode of Relation<" + type + "> must not be a Collection");

		this.targetField = super.getField(TargetNode.class);
		if (this.targetField == null)
			throw new Exception("Relation<" + type + "> is missing the @TargetNode");
		if (Collection.class.isAssignableFrom(this.targetField.getType()))
			throw new Exception("@TargetNode of Relation<" + type + "> must not be a Collection");

		StartNode startAnno = this.startField.getAnno(StartNode.class);
		TargetNode targetAnno = this.targetField.getAnno(TargetNode.class);

		if (this.startField.getField() == this.targetField.getField())
			this.stEqTr = true;

		this.readonlyStart = startAnno.readonly();
		this.readonlyTarget = targetAnno.readonly();
	}

	public PatternType getPatternType() {
		return PatternType.RELATION;
	}

	@Override
	public Collection<String> getLabels() {
		return Arrays.asList(this.label);
	}

	public RelationQueryBuilder search(boolean lazy) {
		return completeSearch(this.archive.getQueryBuilder()
				.relation()
				.setAutoGenerated(true)
				.whereDirection(this.direction)
				.setLazy(lazy));
	}

	public RelationQueryBuilder search(Serializable id, boolean lazy) throws Exception {
		return completeSearch(this.archive.getQueryBuilder()
				.relation()
				.setAutoGenerated(true)
				.whereDirection(this.direction)
				.whereId(id)
				.setLazy(lazy));
	}

	public RelationQueryBuilder completeSearch(RelationQueryBuilder relationBuilder) {
		boolean lazy = relationBuilder.isLazy();
		if (!isBlank(this.label))
			relationBuilder.getLabels()
					.add(label);

		if (this.stEqTr) {
			NodeQueryBuilder nodeBuilder = this._getNode(this.startField.getType(), relationBuilder, lazy);
			return relationBuilder.setStart(nodeBuilder)
					.setTarget(nodeBuilder);
		}

		return relationBuilder.setStart(this._getNode(this.startField.getType(), relationBuilder, lazy))
				.setTarget(this._getNode(this.targetField.getType(), relationBuilder, lazy))
				.storePattern(this)
				.setReturned(true);
	}

	public RelationQueryBuilder createFilter(NodeQueryBuilder caller, Direction direction) {
		RelationQueryBuilder relationBuilder = this.archive.getQueryBuilder()
				.relation()
				.setAutoGenerated(true)
				.whereDirection(this.direction)
				.storePattern(this);
		if (!isBlank(this.label))
			relationBuilder.getLabels()
					.add(this.label);

		if (this.stEqTr)
			return relationBuilder.setStart(caller)
					.setTarget(caller);

		if ((this.direction == Direction.OUTGOING && direction == Direction.INCOMING)
				|| (this.direction == Direction.INCOMING && direction == Direction.OUTGOING))
			relationBuilder.setTarget(caller)
					.setStart(this._getNode(this.startField.getType(), relationBuilder, true));
		else
			relationBuilder.setStart(caller)
					.setTarget(this._getNode(this.targetField.getType(), relationBuilder, true));

		return relationBuilder;
	}

	@Override
	public SaveContainer save(Object entity, Integer depth) throws Exception {
		return new SaveContainer(
				includedData -> (IDataContainer) save(entity, null, direction, includedData, depth).getResult());
	}

	@Override
	public IDeleteContainer delete(final Serializable id, Object entity) throws Exception {
		this.archive.callMethod(entity.getClass(), PreDelete.class, entity);
		return new DeleteContainer(this, entity, id, null, this.archive.getQueryBuilder()
				.relation()
				.setAutoGenerated(true)
				.whereDirection(Direction.BIDIRECTIONAL)
				.whereId(id)
				.setReturned(true)
				.getResult());
	}

	public RelationQueryBuilder save(Object entity, NodeQueryBuilder caller, Direction direction,
			Map<Object, IQueryBuilder<?, ?, ? extends IFilter>> includedData, Integer depth) throws Exception {

		if (entity == null || this.startField.getValue(entity) == null || this.targetField.getValue(entity) == null)
			return null;

		if (includedData.containsKey(entity))
			return (RelationQueryBuilder) includedData.get(entity);

		this.callMethod(PreSave.class, entity);

		RelationQueryBuilder relationBuilder = this.archive.getQueryBuilder()
				.relation()
				.setAutoGenerated(true)
				.whereDirection(this.direction)
				.storeData(entity)
				.setReturned(true);

		if (this.isIdSet(entity))
			// update (id)
			relationBuilder.asUpdate();
		else
			// create (!id)
			relationBuilder.asWrite();

		includedData.put(entity, relationBuilder);

		if (!isBlank(this.label))
			relationBuilder.getLabels()
					.add(this.label);

		if (caller == null) {
			// Relation gets called first
			caller = _getDataNode(this.startField, entity, includedData, relationBuilder, depth);
			relationBuilder.setStart(caller);

			if (this.stEqTr)
				relationBuilder.setTarget(caller);
			else
				relationBuilder.setTarget(_getDataNode(this.targetField, entity, includedData, relationBuilder, depth));

			return _savecheck(relationBuilder);
		}

		if (this.stEqTr) {
			relationBuilder.setStart(caller)
					.setTarget(caller);
			return _savecheck(relationBuilder);
		}

		if ((this.direction == Direction.OUTGOING && direction == Direction.INCOMING)
				|| (this.direction == Direction.INCOMING && direction == Direction.OUTGOING)) {
			relationBuilder.setStart(_getDataNode(this.startField, entity, includedData, relationBuilder,
					this.readonlyStart ? -1 : depth));
			relationBuilder.setTarget(caller);
		} else {
			relationBuilder.setStart(caller);
			relationBuilder.setTarget(_getDataNode(this.targetField, entity, includedData, relationBuilder,
					this.readonlyTarget ? -1 : depth));
		}
		return _savecheck(relationBuilder);
	}

	private RelationQueryBuilder _savecheck(RelationQueryBuilder relationBuilder) {
		if (relationBuilder.getStart() == null || relationBuilder.getTarget() == null)
			return null;
		return relationBuilder;
	}

	public void setStart(Object entity, Object value) {
		try {
			this.startField.setValue(entity, value);
		} catch (IllegalArgumentException e) {
			e.printStackTrace();
		}
	}

	public void setTarget(Object entity, Object value) {
		try {
			this.targetField.setValue(entity, value);
		} catch (IllegalArgumentException e) {
			e.printStackTrace();
		}
	}

	private NodeQueryBuilder _getNode(Class<?> type, RelationQueryBuilder relation, boolean lazy) {
		INodePattern<?> node = this.archive.getPattern(type, NodePattern.class);
		if (node == null)
			return null;
		return node.search(relation, lazy);
	}

	private NodeQueryBuilder _getDataNode(FieldPattern field, Object entity,
			Map<Object, IQueryBuilder<?, ?, ? extends IFilter>> includedData, RelationQueryBuilder relation, Integer depth)
			throws Exception {
		INodePattern<?> node = this.archive.getPattern(field.getType(), NodePattern.class);
		if (node == null)
			throw new Exception("NodePattern for Field<" + field.toString() + "> undefined!");
		NodeQueryBuilder nodeBuilder = node.save(field.getValue(entity), includedData, depth);
		nodeBuilder.addRelation(relation);
		return nodeBuilder;
	}

}
