package net.rsprot.protocol.game.outgoing.camera

import net.rsprot.protocol.ServerProtCategory
import net.rsprot.protocol.game.outgoing.GameServerProtCategory
import net.rsprot.protocol.message.OutgoingGameMessage

/**
 * Cam shake packet is used to make the camera shake around.
 * It is worth noting that multiple different types of shakes
 * can be executed simultaneously, making the camera more and
 * more volatile as a result.
 *
 * The properties of this class are in the exact order as
 * the client reads them, which is consistent across revisions!
 *
 * Camera movements table:
 * ```
 * | Id |   Type  |    Observed Movement   |
 * |----|:-------:|:----------------------:|
 * | 0  |  X-axis |     Left and right     |
 * | 1  |  Y-axis |       Up and down      |
 * | 2  |  Z-axis | Forwards and backwards |
 * | 3  | Y-angle | Panning left and right |
 * | 4  | X-angle |   Panning up and down  |
 * ```
 *
 * @property axis the type of the shake (see table above)
 * @property random the amount of randomness involved.
 * The client will generate a random double from 0.0 to 1.0
 * and multiply it with the [random] as part of the shaking.
 * This property is called 'shakeIntensity' in the event inspector.
 * @property amplitude the amount of randomness generated by the
 * sine. Unlike [random], this is multiplied against the
 * [rate].
 * This property is called 'movementIntensity' in the event inspector.
 * @property rate the sine frequency.
 * This property is called 'speed' in the event inspector.
 */
public class CamShake private constructor(
    private val _axis: UByte,
    private val _random: UByte,
    private val _amplitude: UByte,
    private val _rate: UByte,
) : OutgoingGameMessage {
    public constructor(
        axis: Int,
        random: Int,
        amplitude: Int,
        rate: Int,
    ) : this(
        axis.toUByte(),
        random.toUByte(),
        amplitude.toUByte(),
        rate.toUByte(),
    )

    public val axis: Int
        get() = _axis.toInt()
    public val random: Int
        get() = _random.toInt()
    public val amplitude: Int
        get() = _amplitude.toInt()
    public val rate: Int
        get() = _rate.toInt()
    override val category: ServerProtCategory
        get() = GameServerProtCategory.LOW_PRIORITY_PROT

    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (javaClass != other?.javaClass) return false

        other as CamShake

        if (_axis != other._axis) return false
        if (_random != other._random) return false
        if (_amplitude != other._amplitude) return false
        if (_rate != other._rate) return false

        return true
    }

    override fun hashCode(): Int {
        var result = _axis.hashCode()
        result = 31 * result + _random.hashCode()
        result = 31 * result + _amplitude.hashCode()
        result = 31 * result + _rate.hashCode()
        return result
    }

    override fun toString(): String =
        "CamShake(" +
            "axis=$axis, " +
            "random=$random, " +
            "amplitude=$amplitude, " +
            "rate=$rate" +
            ")"
}
