package net.rsprot.protocol.game.outgoing.camera

import net.rsprot.protocol.ServerProtCategory
import net.rsprot.protocol.game.outgoing.GameServerProtCategory
import net.rsprot.protocol.message.OutgoingGameMessage

/**
 * Cam shake packet is used to make the camera shake around.
 * It is worth noting that multiple different types of shakes
 * can be executed simultaneously, making the camera more and
 * more volatile as a result.
 *
 * The properties of this class are in the exact order as
 * the client reads them, which is consistent across revisions!
 *
 * Camera movements table:
 * ```
 * | Id |   Type  |    Observed Movement   |
 * |----|:-------:|:----------------------:|
 * | 0  |  X-axis |     Left and right     |
 * | 1  |  Y-axis |       Up and down      |
 * | 2  |  Z-axis | Forwards and backwards |
 * | 3  | Y-angle | Panning left and right |
 * | 4  | X-angle |   Panning up and down  |
 * ```
 *
 * @property type the type of the shake (see table above)
 * @property randomAmount the amount of randomness involved.
 * The client will generate a random double from 0.0 to 1.0
 * and multiply it with the [randomAmount] as part of the shaking.
 * This property is called 'shakeIntensity' in the event inspector.
 * @property sineAmount the amount of randomness generated by the
 * sine. Unlike [randomAmount], this is multiplied against the
 * [sineFrequency].
 * This property is called 'movementIntensity' in the event inspector.
 * @property sineFrequency the sine frequency.
 * This property is called 'speed' in the event inspector.
 */
public class CamShake private constructor(
    private val _type: UByte,
    private val _randomAmount: UByte,
    private val _sineAmount: UByte,
    private val _sineFrequency: UByte,
) : OutgoingGameMessage {
    public constructor(
        type: Int,
        randomAmount: Int,
        sineAmount: Int,
        sineFrequency: Int,
    ) : this(
        type.toUByte(),
        randomAmount.toUByte(),
        sineAmount.toUByte(),
        sineFrequency.toUByte(),
    )

    public val type: Int
        get() = _type.toInt()
    public val randomAmount: Int
        get() = _randomAmount.toInt()
    public val sineAmount: Int
        get() = _sineAmount.toInt()
    public val sineFrequency: Int
        get() = _sineFrequency.toInt()
    override val category: ServerProtCategory
        get() = GameServerProtCategory.LOW_PRIORITY_PROT

    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (javaClass != other?.javaClass) return false

        other as CamShake

        if (_type != other._type) return false
        if (_randomAmount != other._randomAmount) return false
        if (_sineAmount != other._sineAmount) return false
        if (_sineFrequency != other._sineFrequency) return false

        return true
    }

    override fun hashCode(): Int {
        var result = _type.hashCode()
        result = 31 * result + _randomAmount.hashCode()
        result = 31 * result + _sineAmount.hashCode()
        result = 31 * result + _sineFrequency.hashCode()
        return result
    }

    override fun toString(): String {
        return "CamShake(" +
            "type=$type, " +
            "randomAmount=$randomAmount, " +
            "sineAmount=$sineAmount, " +
            "sineFrequency=$sineFrequency" +
            ")"
    }
}
