/*
   Copyright 2017 Remko Popma

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
 */
package org.rossonet.ext.picocli;

import static java.util.Locale.ENGLISH;
import static org.rossonet.ext.picocli.CommandLine.Help.Column.Overflow.SPAN;
import static org.rossonet.ext.picocli.CommandLine.Help.Column.Overflow.TRUNCATE;
import static org.rossonet.ext.picocli.CommandLine.Help.Column.Overflow.WRAP;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.Closeable;
import java.io.Console;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.io.StreamTokenizer;
import java.io.StringWriter;
import java.lang.annotation.Annotation;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;
import java.lang.reflect.AccessibleObject;
import java.lang.reflect.AnnotatedElement;
import java.lang.reflect.Array;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.GenericArrayType;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Proxy;
import java.lang.reflect.Type;
import java.lang.reflect.WildcardType;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.net.InetAddress;
import java.net.MalformedURLException;
import java.net.NetworkInterface;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.ByteOrder;
import java.nio.charset.Charset;
import java.text.BreakIterator;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.AbstractMap;
import java.util.AbstractSet;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Currency;
import java.util.Date;
import java.util.EnumSet;
import java.util.Enumeration;
import java.util.Formatter;
import java.util.HashMap;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.IllegalFormatException;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.NoSuchElementException;
import java.util.Properties;
import java.util.Queue;
import java.util.ResourceBundle;
import java.util.Set;
import java.util.SortedMap;
import java.util.SortedSet;
import java.util.Stack;
import java.util.TimeZone;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.rossonet.ext.picocli.CommandLine.Help.Ansi.IStyle;
import org.rossonet.ext.picocli.CommandLine.Help.Ansi.Style;
import org.rossonet.ext.picocli.CommandLine.Help.Ansi.Text;
import org.rossonet.ext.picocli.CommandLine.Model.ArgGroupSpec;
import org.rossonet.ext.picocli.CommandLine.Model.ArgSpec;
import org.rossonet.ext.picocli.CommandLine.Model.CaseAwareLinkedMap;
import org.rossonet.ext.picocli.CommandLine.Model.CommandSpec;
import org.rossonet.ext.picocli.CommandLine.Model.IAnnotatedElement;
import org.rossonet.ext.picocli.CommandLine.Model.IOrdered;
import org.rossonet.ext.picocli.CommandLine.Model.ITypeInfo;
import org.rossonet.ext.picocli.CommandLine.Model.Messages;
import org.rossonet.ext.picocli.CommandLine.Model.OptionSpec;
import org.rossonet.ext.picocli.CommandLine.Model.ParserSpec;
import org.rossonet.ext.picocli.CommandLine.Model.PositionalParamSpec;
import org.rossonet.ext.picocli.CommandLine.Model.TypedMember;
import org.rossonet.ext.picocli.CommandLine.Model.UnmatchedArgsBinding;
import org.rossonet.ext.picocli.CommandLine.Model.UsageMessageSpec;
import org.rossonet.ext.picocli.CommandLine.ParseResult.GroupMatchContainer;

/**
 * <p>
 * CommandLine interpreter that uses reflection to initialize an annotated user
 * object with values obtained from the command line arguments.
 * </p>
 * <p>
 * The full user manual is hosted at
 * <a href="https://picocli.info/">https://picocli.info</a>.
 * </p>
 * <h2>Example</h2>
 * <p id="checksum_example">
 * An example that implements {@code Callable} and uses the
 * {@link #execute(String...) CommandLine.execute} convenience API to run in a
 * single line of code:
 * </p>
 * 
 * <pre>
 * &#064;Command(name = "checksum", mixinStandardHelpOptions = true, version = "checksum 4.0", description = "Prints the checksum (SHA-1 by default) of a file to STDOUT.")
 * class CheckSum implements Callable&lt;Integer&gt; {
 *
 * 	&#064;Parameters(index = "0", description = "The file whose checksum to calculate.")
 * 	private File file;
 *
 * 	&#064;Option(names = { "-a", "--algorithm" }, description = "MD5, SHA-1, SHA-256, ...")
 * 	private String algorithm = "SHA-1";
 *
 * 	&#064;Override
 * 	public Integer call() throws Exception { // your business logic goes here...
 * 		byte[] fileContents = Files.readAllBytes(file.toPath());
 * 		byte[] digest = MessageDigest.getInstance(algorithm).digest(fileContents);
 * 		System.out.printf("%0" + (digest.length * 2) + "x%n", new BigInteger(1, digest));
 * 		return 0;
 * 	}
 *
 * 	// CheckSum implements Callable, so parsing, error handling and handling user
 * 	// requests for usage help or version help can be done with one line of code.
 * 	public static void main(String[] args) {
 * 		int exitCode = new CommandLine(new CheckSum()).execute(args);
 * 		System.exit(exitCode);
 * 	}
 * }
 * </pre>
 * <p>
 * Another example where the application calls {@code parseArgs} and takes
 * responsibility for error handling and checking whether the user requested
 * help:
 * </p>
 * 
 * <pre>
 * import static picocli.CommandLine.*;
 *
 * &#064;Command(mixinStandardHelpOptions = true, version = "v3.0.0", header = "Encrypt FILE(s), or standard input, to standard output or to the output file.")
 * public class Encrypt {
 *
 * 	&#064;Parameters(description = "Any number of input files")
 * 	private List&lt;File&gt; files = new ArrayList&lt;File&gt;();
 *
 * 	&#064;Option(names = { "-o", "--out" }, description = "Output file (default: print to console)")
 * 	private File outputFile;
 *
 * 	&#064;Option(names = { "-v",
 * 			"--verbose" }, description = "Verbose mode. Helpful for troubleshooting. Multiple -v options increase the verbosity.")
 * 	private boolean[] verbose;
 * }
 * </pre>
 * <p>
 * Use {@code CommandLine} to initialize a user object as follows:
 * </p>
 * 
 * <pre>
 * public static void main(String... args) {
 * 	Encrypt encrypt = new Encrypt();
 * 	try {
 * 		ParseResult parseResult = new CommandLine(encrypt).parseArgs(args);
 * 		if (!CommandLine.printHelpIfRequested(parseResult)) {
 * 			runProgram(encrypt);
 * 		}
 * 	} catch (ParameterException ex) { // command line arguments could not be parsed
 * 		System.err.println(ex.getMessage());
 * 		ex.getCommandLine().usage(System.err);
 * 	}
 * }
 * </pre>
 * <p>
 * Invoke the above program with some command line arguments. The below are all
 * equivalent:
 * </p>
 * 
 * <pre>
 * --verbose --out=outfile in1 in2
 * --verbose --out outfile in1 in2
 * -v --out=outfile in1 in2
 * -v -o outfile in1 in2
 * -v -o=outfile in1 in2
 * -vo outfile in1 in2
 * -vo=outfile in1 in2
 * -v -ooutfile in1 in2
 * -vooutfile in1 in2
 * </pre>
 * 
 * <h2>Classes and Interfaces for Defining a CommandSpec Model</h2>
 * <p>
 * <img src="doc-files/class-diagram-definition.png" alt="Classes and Interfaces
 * for Defining a CommandSpec Model">
 * </p>
 * <h2>Classes Related to Parsing Command Line Arguments</h2>
 * <p>
 * <img src="doc-files/class-diagram-parsing.png" alt="Classes Related to
 * Parsing Command Line Arguments">
 * </p>
 */
@SuppressWarnings("unused")
public class CommandLine {

	static class AbbreviationMatcher {
		static class MatchResult<V> {
			private final String fullName;
			private final V value;

			MatchResult(String fullName, V value) {
				this.fullName = fullName;
				this.value = value;
			}

			@Override
			public boolean equals(Object o) {
				if (!(o instanceof MatchResult)) {
					return false;
				}
				final MatchResult<?> p = (MatchResult<?>) o;
				return fullName.equals(p.fullName) && (hasValue() ? value.equals(p.value) : value == p.value);
			}

			String getFullName() {
				return fullName;
			}

			V getValue() {
				return value;
			}

			@Override
			public int hashCode() {
				return fullName.hashCode() ^ (value == null ? 0 : value.hashCode());
			}

			boolean hasValue() {
				return value != null;
			}
		}

		private static <T> boolean isAllCandidatesSame(Collection<T> candidates) {
			if (candidates.isEmpty()) {
				return true;
			}
			final Iterator<T> iterator = candidates.iterator();
			final T first = iterator.next();
			while (iterator.hasNext()) {
				if (iterator.next() != first) { // check reference equality as aliases point to the same object
					return false;
				}
			}
			return true;
		}

		private static boolean isNonAlphabetic(String str) {
			for (int i = 0, codepoint; i < str.length(); i += Character.charCount(codepoint)) {
				codepoint = str.codePointAt(i);
				if (Character.isLetterOrDigit(codepoint)) {
					return false;
				}
			}
			return true;
		}

		private static String makeCanonical(String str) {
			if ("-".equals(str)) {
				return "";
			}
			if (str.startsWith("-") && str.length() > 1) {
				final String uppercase = String.valueOf(Character.toChars(Character.toUpperCase(str.codePointAt(1))));
				return uppercase + str.substring(1 + uppercase.length());
			}
			return str;
		}

		/**
		 * Returns the non-abbreviated name if found, otherwise returns the specified
		 * original abbreviation name.
		 */
		public static <T> MatchResult<T> match(Map<String, T> map, String abbreviation, boolean caseInsensitive,
				CommandLine source) {
			MatchResult<T> result = new MatchResult<T>(abbreviation, map.get(abbreviation));
			if (result.hasValue() || map.isEmpty()) {
				return result;
			}
			final List<String> abbreviatedKeyChunks = splitIntoChunks(abbreviation, caseInsensitive);
			final Map<String, T> candidates = new LinkedHashMap<String, T>();
			for (final Map.Entry<String, T> entry : map.entrySet()) {
				final List<String> keyChunks = splitIntoChunks(entry.getKey(), caseInsensitive);
				if (matchKeyChunks(abbreviatedKeyChunks, keyChunks, caseInsensitive)) {
					if (!result.hasValue()) {
						result = new MatchResult<T>(entry.getKey(), entry.getValue());
					}
					candidates.put(entry.getKey(), entry.getValue());
				}
			}
			if (!isAllCandidatesSame(candidates.values())) {
				final String str = candidates.keySet().toString();
				throw new ParameterException(source, "Error: '" + abbreviation + "' is not unique: it matches '"
						+ str.substring(1, str.length() - 1).replace(", ", "', '") + "'");
			}
			return result; // return the original with null as value if no match found
		}

		private static boolean matchKeyChunks(List<String> abbreviatedKeyChunks, List<String> keyChunks,
				boolean caseInsensitive) {
			if (abbreviatedKeyChunks.size() > keyChunks.size()) {
				return false;
			}
			int matchCount = 0;
			if (isNonAlphabetic(keyChunks.get(0))) { // non-alphabetic prefix must be exactly the same
				if (!keyChunks.get(0).equals(abbreviatedKeyChunks.get(0))) {
					return false;
				}
				matchCount++;
			}
			if (!startsWith(keyChunks.get(matchCount), abbreviatedKeyChunks.get(matchCount), caseInsensitive)) { // first
																													// alphabetic
																													// chunk
																													// must
																													// match
				return false;
			}
			matchCount++;
			for (int i = matchCount, lastMatchChunk = matchCount; i < abbreviatedKeyChunks.size(); i++, matchCount++) {
				boolean found = false;
				for (int j = lastMatchChunk; j < keyChunks.size(); j++) {
					if ((found = startsWith(keyChunks.get(j), abbreviatedKeyChunks.get(i), caseInsensitive))) {
						lastMatchChunk = j + 1;
						break;
					}
				}
				if (!found) { // not a candidate
					break;
				}
			}
			return matchCount == abbreviatedKeyChunks.size();
		}

		public static List<String> splitIntoChunks(String command, boolean caseInsensitive) {
			final List<String> result = new ArrayList<String>();
			int start = 0, codepoint;
			final StringBuilder nonAlphabeticPrefix = new StringBuilder();
			while (start < command.length()) {
				codepoint = command.codePointAt(start);
				if (Character.isLetterOrDigit(codepoint)) {
					break;
				}
				nonAlphabeticPrefix.appendCodePoint(codepoint);
				start += Character.charCount(codepoint);
			}
			if (nonAlphabeticPrefix.length() > 0) {
				result.add(nonAlphabeticPrefix.toString());
//                if (command.codePointBefore(start) == '-') {
//                    start--; // hint makeCanonical() to canonicalize the first chunk
//                }
			}
			for (int i = start; i < command.length(); i += Character.charCount(codepoint)) {
				codepoint = command.codePointAt(i);
				if ((!caseInsensitive && Character.isUpperCase(codepoint)) || '-' == codepoint) {
					final String chunk = makeCanonical(command.substring(start, i));
					if (chunk.length() > 0) {
						result.add(chunk);
					}
					start = i;
				}
			}
			if (start < command.length()) {
				final String chunk = makeCanonical(command.substring(start));
				if (chunk.length() > 0) {
					result.add(chunk);
				}
			}
			return result;
		}

		private static boolean startsWith(String str, String prefix, boolean caseInsensitive) {
			if (prefix.length() > str.length()) {
				return false;
			}
			final String strPrefix = str.substring(0, prefix.length());
			return caseInsensitive ? strPrefix.equalsIgnoreCase(prefix) : strPrefix.equals(prefix);
		}
	}

	/**
	 * Abstract superclass for {@link IParseResultHandler2} and
	 * {@link IExceptionHandler2} implementations.
	 * <p>
	 * Note that {@code AbstractHandler} is a generic type. This, along with the
	 * abstract {@code self} method, allows method chaining to work properly in
	 * subclasses, without the need for casts. An example subclass can look like
	 * this:
	 * </p>
	 * 
	 * <pre>{@code
	 * class MyResultHandler extends AbstractHandler<MyReturnType, MyResultHandler> implements IParseResultHandler2<MyReturnType> {
	 *
	 *     public MyReturnType handleParseResult(ParseResult parseResult) { ... }
	 *
	 *     protected MyResultHandler self() { return this; }
	 * }
	 * }</pre>
	 * 
	 * @param <R> the return type of this handler
	 * @param <T> The type of the handler subclass; for fluent API method chaining
	 * @deprecated see {@link #execute(String...)}
	 * @since 3.0
	 */
	@Deprecated
	public static abstract class AbstractHandler<R, T extends AbstractHandler<R, T>> {
		private Help.ColorScheme colorScheme = Help.defaultColorScheme(Help.Ansi.AUTO);
		private Integer exitCode;
		private PrintStream out = System.out;
		private PrintStream err = System.err;

		/**
		 * Indicates that the handler should call {@link System#exit(int)} after
		 * processing completes and sets the exit code to use as the termination status.
		 * 
		 * @deprecated use {@link CommandLine#execute(String...)} instead, and call
		 *             {@code System.exit()} in the application.
		 */
		@Deprecated
		public T andExit(int exitCode) {
			this.exitCode = exitCode;
			return self();
		}

		/**
		 * Returns the ANSI style to use. Defaults to {@code Help.Ansi.AUTO}, unless
		 * {@link #useAnsi(CommandLine.Help.Ansi)} was called with a different setting.
		 * 
		 * @deprecated use {@link #colorScheme()} instead
		 */
		@Deprecated
		public Help.Ansi ansi() {
			return colorScheme.ansi();
		}

		/**
		 * Returns the ColorScheme to use. Defaults to
		 * {@code Help#defaultColorScheme(Help.Ansi.AUTO)}.
		 * 
		 * @since 4.0
		 */
		public Help.ColorScheme colorScheme() {
			return colorScheme;
		}

		/**
		 * Returns the stream to print diagnostic messages to. Defaults to
		 * {@code System.err}, unless {@link #useErr(PrintStream)} was called with a
		 * different stream.
		 * <p>
		 * {@code IExceptionHandler2} implementations should use this stream to print
		 * error messages (which may include a usage help message) when an unexpected
		 * error occurs.
		 * </p>
		 */
		public PrintStream err() {
			return err;
		}

		/** Calls {@code System.exit(int)} with the specified exit code. */
		protected void exit(int exitCode) {
			System.exit(exitCode);
		}

		/**
		 * Returns the exit code to use as the termination status, or {@code null} (the
		 * default) if the handler should not call {@link System#exit(int)} after
		 * processing completes.
		 * 
		 * @see #andExit(int)
		 */
		public Integer exitCode() {
			return exitCode;
		}

		/**
		 * Returns {@code true} if an exit code was set with {@link #andExit(int)}, or
		 * {@code false} (the default) if the handler should not call
		 * {@link System#exit(int)} after processing completes.
		 */
		public boolean hasExitCode() {
			return exitCode != null;
		}

		/**
		 * Returns the stream to print command output to. Defaults to
		 * {@code System.out}, unless {@link #useOut(PrintStream)} was called with a
		 * different stream.
		 * <p>
		 * {@code IParseResultHandler2} implementations should use this stream. By
		 * <a href=
		 * "http://www.gnu.org/prep/standards/html_node/_002d_002dhelp.html">convention</a>,
		 * when the user requests help with a {@code --help} or similar option, the
		 * usage help message is printed to the standard output stream so that it can be
		 * easily searched and paged.
		 * </p>
		 */
		public PrintStream out() {
			return out;
		}

		/**
		 * Convenience method for subclasses that returns the specified result object if
		 * no exit code was set, or otherwise, if an exit code {@linkplain #andExit(int)
		 * was set}, calls {@code System.exit} with the configured exit code to
		 * terminate the currently running Java virtual machine.
		 */
		protected R returnResultOrExit(R result) {
			if (hasExitCode()) {
				exit(exitCode());
			}
			return result;
		}

		/**
		 * Returns {@code this} to allow method chaining when calling the setters for a
		 * fluent API.
		 */
		protected abstract T self();

		/**
		 * Convenience method for subclasses that throws the specified
		 * ExecutionException if no exit code was set, or otherwise, if an exit code
		 * {@linkplain #andExit(int) was set}, prints the stacktrace of the specified
		 * exception to the diagnostic error stream and calls {@code System.exit} with
		 * the configured exit code to terminate the currently running Java virtual
		 * machine.
		 */
		protected R throwOrExit(ExecutionException ex) {
			if (hasExitCode()) {
				ex.printStackTrace(this.err());
				exit(exitCode());
				return null;
			}
			throw ex;
		}

		/**
		 * Sets the ANSI style to use and resets the color scheme to the default.
		 * 
		 * @deprecated use {@link CommandLine#setColorScheme(Help.ColorScheme)} and
		 *             {@link CommandLine#execute(String...)} instead
		 * @see #ansi()
		 */
		@Deprecated
		public T useAnsi(Help.Ansi ansi) {
			this.colorScheme = Help.defaultColorScheme(Assert.notNull(ansi, "ansi"));
			return self();
		}

		/**
		 * Sets the stream to print diagnostic messages to.
		 * 
		 * @deprecated use {@link CommandLine#setErr(PrintWriter)} and
		 *             {@link CommandLine#execute(String...)} instead
		 */
		@Deprecated
		public T useErr(PrintStream err) {
			this.err = Assert.notNull(err, "err");
			return self();
		}

		/**
		 * Sets the stream to print command output to.
		 * 
		 * @deprecated use {@link CommandLine#setOut(PrintWriter)} and
		 *             {@link CommandLine#execute(String...)} instead
		 */
		@Deprecated
		public T useOut(PrintStream out) {
			this.out = Assert.notNull(out, "out");
			return self();
		}
	}

	/**
	 * Command line parse result handler that returns a value. This handler prints
	 * help if requested, and otherwise calls
	 * {@link #handle(CommandLine.ParseResult)} with the parse result. Facilitates
	 * implementation of the {@link IParseResultHandler2} interface.
	 * <p>
	 * Note that {@code AbstractParseResultHandler} is a generic type. This, along
	 * with the abstract {@code self} method, allows method chaining to work
	 * properly in subclasses, without the need for casts. An example subclass can
	 * look like this:
	 * </p>
	 * 
	 * <pre>{@code
	 * class MyResultHandler extends AbstractParseResultHandler<MyReturnType> {
	 *
	 *     protected MyReturnType handle(ParseResult parseResult) throws ExecutionException { ... }
	 *
	 *     protected MyResultHandler self() { return this; }
	 * }
	 * }</pre>
	 * 
	 * @deprecated see {@link #execute(String...)}, {@link #getExecutionStrategy()},
	 *             {@link #getParameterExceptionHandler()},
	 *             {@link #getExecutionExceptionHandler()}
	 * @since 3.0
	 */
	@Deprecated
	public abstract static class AbstractParseResultHandler<R> extends AbstractHandler<R, AbstractParseResultHandler<R>>
			implements IParseResultHandler2<R>, IExecutionStrategy {
		@Override
		public int execute(ParseResult parseResult) throws ExecutionException {
			final Integer helpExitCode = executeHelpRequest(parseResult);
			if (helpExitCode != null) {
				return helpExitCode;
			}

			final Tracer t = CommandLine.tracer();
			t.debug("%s: handling ParseResult...", getClass().getSimpleName());
			final R executionResult = handle(parseResult);
			final List<IExitCodeGenerator> exitCodeGenerators = extractExitCodeGenerators(parseResult);
			t.debug("%s: ParseResult has %s exit code generators", getClass().getSimpleName(),
					exitCodeGenerators.size());
			return resolveExitCode(parseResult.commandSpec().exitCodeOnSuccess(), executionResult, exitCodeGenerators);
		}

		protected List<IExitCodeGenerator> extractExitCodeGenerators(ParseResult parseResult) {
			return Collections.emptyList();
		}

		/**
		 * Processes the specified {@code ParseResult} and returns the result as a list
		 * of objects. Implementations are responsible for catching any exceptions
		 * thrown in the {@code handle} method, and rethrowing an
		 * {@code ExecutionException} that details the problem and captures the
		 * offending {@code CommandLine} object.
		 *
		 * @param parseResult the {@code ParseResult} that resulted from successfully
		 *                    parsing the command line arguments
		 * @return the result of processing parse results
		 * @throws ExecutionException if a problem occurred while processing the parse
		 *                            results; client code can use
		 *                            {@link ExecutionException#getCommandLine()} to get
		 *                            the command or subcommand where processing failed
		 */
		protected abstract R handle(ParseResult parseResult) throws ExecutionException;

		/**
		 * Prints help if requested, and otherwise calls
		 * {@link #handle(CommandLine.ParseResult)}. Finally, either a list of result
		 * objects is returned, or the JVM is terminated if an exit code
		 * {@linkplain #andExit(int) was set}.
		 *
		 * @param parseResult the {@code ParseResult} that resulted from successfully
		 *                    parsing the command line arguments
		 * @return the result of {@link #handle(CommandLine.ParseResult) processing
		 *         parse results}
		 * @throws ParameterException if the {@link HelpCommand HelpCommand} was invoked
		 *                            for an unknown subcommand. Any
		 *                            {@code ParameterExceptions} thrown from this
		 *                            method are treated as if this exception was thrown
		 *                            during parsing and passed to the
		 *                            {@link IExceptionHandler2}
		 * @throws ExecutionException if a problem occurred while processing the parse
		 *                            results; client code can use
		 *                            {@link ExecutionException#getCommandLine()} to get
		 *                            the command or subcommand where processing failed
		 */
		@Override
		public R handleParseResult(ParseResult parseResult) throws ExecutionException {
			if (printHelpIfRequested(parseResult.asCommandLineList(), out(), err(), colorScheme())) {
				return returnResultOrExit(null);
			}
			return returnResultOrExit(handle(parseResult));
		}

		// Use the highest value (or if all values are negative, use the lowest value).
		private int resolveExitCode(int exitCodeOnSuccess, R executionResult,
				List<IExitCodeGenerator> exitCodeGenerators) {
			int result = 0;
			for (final IExitCodeGenerator generator : exitCodeGenerators) {
				try {
					final int exitCode = generator.getExitCode();
					if ((exitCode > 0 && exitCode > result) || (exitCode < result && result <= 0)) {
						result = exitCode;
					}
				} catch (final Exception ex) {
					result = (result == 0) ? 1 : result;
					ex.printStackTrace();
				}
			}
			final Tracer t = CommandLine.tracer();
			t.debug("resolveExitCode: exit code generators resulted in exit code=%d", result);
			if (executionResult instanceof List) {
				final List<?> resultList = (List<?>) executionResult;
				for (final Object obj : resultList) {
					if (obj instanceof Integer) {
						final int exitCode = (Integer) obj;
						if ((exitCode > 0 && exitCode > result) || (exitCode < result && result <= 0)) {
							result = exitCode;
						}
					}
				}
			}
			t.debug("resolveExitCode: execution results resulted in exit code=%d", result);
			t.debug("resolveExitCode: returning exit code=%d", result == 0 ? exitCodeOnSuccess : result);
			return result == 0 ? exitCodeOnSuccess : result;
		}
	}

	/**
	 * A {@code Command} may define one or more {@code ArgGroups}: a group of
	 * options, positional parameters or a mixture of the two. Groups can be used
	 * to:
	 * <ul>
	 * <li>define <b>mutually exclusive</b> arguments. By default, options and
	 * positional parameters in a group are mutually exclusive. This can be
	 * controlled with the {@link #exclusive() exclusive} attribute. Picocli will
	 * throw a {@link MutuallyExclusiveArgsException} if the command line contains
	 * multiple arguments that are mutually exclusive.</li>
	 * <li>define a set of arguments that <b>must co-occur</b>. Set
	 * {@link #exclusive() exclusive = false} to define a group of options and
	 * positional parameters that must always be specified together. Picocli will
	 * throw a {@link MissingParameterException MissingParameterException} if not
	 * all the options and positional parameters in a co-occurring group are
	 * specified together.</li>
	 * <li>create an <b>option section</b> in the usage help message. To be shown in
	 * the usage help message, a group needs to have a {@link #heading() heading}
	 * (which may come from a {@linkplain #headingKey() resource bundle}). Groups
	 * without a heading are only used for validation. Set {@link #validate()
	 * validate = false} for groups whose purpose is only to customize the usage
	 * help message.</li>
	 * <li>define <b>composite repeating argument groups</b>. Groups may contain
	 * other groups to create composite groups.</li>
	 * </ul>
	 * <p>
	 * Groups may be optional ({@code multiplicity = "0..1"}), required
	 * ({@code multiplicity = "1"}), or repeating groups
	 * ({@code multiplicity = "0..*"} or {@code multiplicity = "1..*"}). For a group
	 * of mutually exclusive arguments, making the group required means that one of
	 * the arguments in the group must appear on the command line, or a
	 * {@link MissingParameterException MissingParameterException} is thrown. For a
	 * group of co-occurring arguments, all arguments in the group must appear on
	 * the command line.
	 * </p>
	 * <p>
	 * Groups can be composed for validation purposes:
	 * </p>
	 * <ul>
	 * <li>When the parent group is mutually exclusive, only one of the subgroups
	 * may be present.</li>
	 * <li>When the parent group is a co-occurring group, all subgroups must be
	 * present.</li>
	 * <li>When the parent group is required, at least one subgroup must be
	 * present.</li>
	 * </ul>
	 * <p>
	 * Below is an example of an {@code ArgGroup} defining a set of dependent
	 * options that must occur together. All options are required <em>within the
	 * group</em>, while the group itself is optional:
	 * </p>
	 * 
	 * <pre>
	 * public class DependentOptions {
	 * 	&#064;ArgGroup(exclusive = false, multiplicity = "0..1")
	 * 	Dependent group;
	 *
	 * 	static class Dependent {
	 * 		&#064;Option(names = "-a", required = true)
	 * 		int a;
	 * 		&#064;Option(names = "-b", required = true)
	 * 		int b;
	 * 		&#064;Option(names = "-c", required = true)
	 * 		int c;
	 * 	}
	 * }
	 * </pre>
	 * 
	 * @see ArgGroupSpec
	 * @since 4.0
	 */
	@Retention(RetentionPolicy.RUNTIME)
	@Target({ ElementType.FIELD, ElementType.METHOD, ElementType.PARAMETER })
	public @interface ArgGroup {
		/**
		 * Determines whether this is a mutually exclusive group; {@code true} by
		 * default. If {@code false}, this is a co-occurring group. Ignored if
		 * {@link #validate()} is {@code false}.
		 */
		boolean exclusive() default true;

		/**
		 * The heading of this group, used when generating the usage documentation. When
		 * neither a {@link #heading() heading} nor a {@link #headingKey() headingKey}
		 * are specified, this group is used for validation only and does not change the
		 * usage help message.
		 */
		String heading() default "__no_heading__";

		/**
		 * ResourceBundle key for this group's usage help message section heading. When
		 * neither a {@link #heading() heading} nor a {@link #headingKey() headingKey}
		 * are specified, this group is used for validation only and does not change the
		 * usage help message.
		 */
		String headingKey() default "__no_heading_key__";

		/**
		 * Determines how often this group can be specified on the command line;
		 * {@code "0..1"} (optional) by default. For a group of mutually exclusive
		 * arguments, making the group required {@code multiplicity = "1"} means that
		 * one of the arguments in the group must appear on the command line, or a
		 * MissingParameterException is thrown. For a group of co-occurring arguments,
		 * making the group required means that all arguments in the group must appear
		 * on the command line. Ignored if {@link #validate()} is {@code false}.
		 */
		String multiplicity() default "0..1";

		/**
		 * Determines the position in the options list in the usage help message at
		 * which this group should be shown. Groups with a lower number are shown before
		 * groups with a higher number. This attribute is only honored for groups that
		 * have a {@link #heading() heading} (or a {@link #headingKey() headingKey} with
		 * a non-{@code null} resource bundle value).
		 */
		int order() default -1;

		/**
		 * Determines whether picocli should validate the rules of this group
		 * ({@code true} by default). For a mutually exclusive group validation means
		 * verifying that no more than one elements of the group is specified on the
		 * command line; for a co-ocurring group validation means verifying that all
		 * elements of the group are specified on the command line. Set
		 * {@link #validate() validate = false} for groups whose purpose is only to
		 * customize the usage help message.
		 * 
		 * @see #multiplicity()
		 * @see #heading()
		 */
		boolean validate() default true;
	}

	/**
	 * Utility class providing some defensive coding convenience methods.
	 */
	private static final class Assert {
		static void assertTrue(boolean condition, IHelpSectionRenderer producer) {
			if (!condition) {
				throw new IllegalStateException(producer.render(null));
			}
		}

		static void assertTrue(boolean condition, String message) {
			if (!condition) {
				throw new IllegalStateException(message);
			}
		}

		static boolean equals(Object obj1, Object obj2) {
			return obj1 == null ? obj2 == null : obj1.equals(obj2);
		}

		static int hashCode(boolean bool) {
			return bool ? 1 : 0;
		}

		static int hashCode(Object obj) {
			return obj == null ? 0 : obj.hashCode();
		}

		/**
		 * Throws a NullPointerException if the specified object is null.
		 * 
		 * @param object      the object to verify
		 * @param description error message
		 * @param <T>         type of the object to check
		 * @return the verified object
		 */
		static <T> T notNull(T object, String description) {
			if (object == null) {
				throw new NullPointerException(description);
			}
			return object;
		}

		private Assert() {
		} // private constructor: never instantiate
	}

	static class AutoHelpMixin {
		private static final String KEY = "mixinStandardHelpOptions";

		@Option(names = { "${picocli.help.name.0:--h}",
				"${picocli.help.name.1:---help}" }, usageHelp = true, descriptionKey = "mixinStandardHelpOptions.help", description = "Show this help message and exit.")
		private boolean helpRequested;

		@Option(names = { "${picocli.version.name.0:--V}",
				"${picocli.version.name.1:---version}" }, versionHelp = true, descriptionKey = "mixinStandardHelpOptions.version", description = "Print version information and exit.")
		private boolean versionRequested;
	}

	/**
	 * Inner class to group the built-in {@link ITypeConverter} implementations.
	 */
	private static class BuiltIn {
		static class BigDecimalConverter implements ITypeConverter<BigDecimal> {
			@Override
			public BigDecimal convert(String value) {
				return new BigDecimal(value);
			}
		}

		static class BigIntegerConverter implements ITypeConverter<BigInteger> {
			@Override
			public BigInteger convert(String value) {
				return new BigInteger(value);
			}
		}

		/**
		 * Converts {@code "true"} or {@code "false"} to a {@code Boolean}. Other values
		 * result in a ParameterException.
		 */
		static class BooleanConverter implements ITypeConverter<Boolean> {
			@Override
			public Boolean convert(String value) {
				if ("true".equalsIgnoreCase(value) || "false".equalsIgnoreCase(value)) {
					return Boolean.parseBoolean(value);
				} else {
					throw new TypeConversionException("'" + value + "' is not a boolean");
				}
			}
		}

		/**
		 * Converts text to a {@code Byte} by delegating to
		 * {@link Byte#valueOf(String)}.
		 */
		static class ByteConverter implements ITypeConverter<Byte> {
			@Override
			public Byte convert(String value) {
				try {
					return Byte.valueOf(value);
				} catch (final Exception ex) {
					throw fail(value, Byte.TYPE);
				}
			}
		}

		static class ByteOrderConverter implements ITypeConverter<ByteOrder> {
			@Override
			public ByteOrder convert(String s) throws Exception {
				if (s.equalsIgnoreCase(ByteOrder.BIG_ENDIAN.toString())) {
					return ByteOrder.BIG_ENDIAN;
				}
				if (s.equalsIgnoreCase(ByteOrder.LITTLE_ENDIAN.toString())) {
					return ByteOrder.LITTLE_ENDIAN;
				}
				throw new TypeConversionException("'" + s + "' is not a valid ByteOrder");
			}
		}

		static class CharacterConverter implements ITypeConverter<Character> {
			@Override
			public Character convert(String value) {
				if (value.length() > 1) {
					throw new TypeConversionException("'" + value + "' is not a single character");
				}
				return value.charAt(0);
			}
		}

		static class CharArrayConverter implements ITypeConverter<char[]> {
			@Override
			public char[] convert(String value) {
				return value.toCharArray();
			}
		}

		static class CharSequenceConverter implements ITypeConverter<CharSequence> {
			@Override
			public String convert(String value) {
				return value;
			}
		}

		static class CharsetConverter implements ITypeConverter<Charset> {
			@Override
			public Charset convert(String s) {
				return Charset.forName(s);
			}
		}

		static class ClassConverter implements ITypeConverter<Class<?>> {
			@Override
			public Class<?> convert(String s) throws Exception {
				return Class.forName(s);
			}
		}

		static class CurrencyConverter implements ITypeConverter<Currency> {
			@Override
			public Currency convert(String s) throws Exception {
				return Currency.getInstance(s);
			}
		}

		static class DoubleConverter implements ITypeConverter<Double> {
			@Override
			public Double convert(String value) {
				try {
					return Double.valueOf(value);
				} catch (final Exception ex) {
					throw fail(value, Double.TYPE);
				}
			}
		}

		static class FileConverter implements ITypeConverter<File> {
			@Override
			public File convert(String value) {
				return new File(value);
			}
		}

		static class FloatConverter implements ITypeConverter<Float> {
			@Override
			public Float convert(String value) {
				try {
					return Float.valueOf(value);
				} catch (final Exception ex) {
					throw fail(value, Float.TYPE);
				}
			}
		}

		/**
		 * Converts text to a {@code InetAddress} by delegating to
		 * {@link InetAddress#getByName(String)}.
		 */
		static class InetAddressConverter implements ITypeConverter<InetAddress> {
			@Override
			public InetAddress convert(String s) throws Exception {
				return InetAddress.getByName(s);
			}
		}

		/**
		 * Converts text to an {@code Integer} by delegating to
		 * {@link Integer#valueOf(String)}.
		 */
		static class IntegerConverter implements ITypeConverter<Integer> {
			@Override
			public Integer convert(String value) {
				try {
					return Integer.valueOf(value);
				} catch (final Exception ex) {
					throw fail(value, Integer.TYPE, "'%s' is not an %s");
				}
			}
		}

		/**
		 * Converts text in {@code yyyy-mm-dd} format to a {@code java.util.Date}.
		 * ParameterException on failure.
		 */
		static class ISO8601DateConverter implements ITypeConverter<Date> {
			@Override
			public Date convert(String value) {
				try {
					return new SimpleDateFormat("yyyy-MM-dd").parse(value);
				} catch (final ParseException e) {
					throw new TypeConversionException("'" + value + "' is not a yyyy-MM-dd date");
				}
			}
		}

		/**
		 * Converts text in any of the following formats to a {@code java.sql.Time}:
		 * {@code HH:mm}, {@code HH:mm:ss}, {@code HH:mm:ss.SSS}, {@code HH:mm:ss,SSS}.
		 * Other formats result in a ParameterException.
		 */
		static class ISO8601TimeConverter implements ITypeConverter<Object> {
			// Implementation note: use reflection so that picocli only requires the
			// java.base module in Java 9.
			private final Constructor<?> constructor;

			ISO8601TimeConverter(Constructor<?> constructor) {
				this.constructor = Assert.notNull(constructor, "time class constructor");
			}

			@Override
			public Object convert(String value) {
				try {
					if (value.length() <= 5) {
						return createTime(new SimpleDateFormat("HH:mm").parse(value).getTime());
					} else if (value.length() <= 8) {
						return createTime(new SimpleDateFormat("HH:mm:ss").parse(value).getTime());
					} else if (value.length() <= 12) {
						try {
							return createTime(new SimpleDateFormat("HH:mm:ss.SSS").parse(value).getTime());
						} catch (final ParseException e2) {
							return createTime(new SimpleDateFormat("HH:mm:ss,SSS").parse(value).getTime());
						}
					}
				} catch (final ParseException ignored) {
					// ignored because we throw a ParameterException below
				}
				throw new TypeConversionException("'" + value + "' is not a HH:mm[:ss[.SSS]] time");
			}

			private Object createTime(long epochMillis) {
				try {
					return constructor.newInstance(epochMillis);
				} catch (final Exception e) {
					throw new TypeConversionException("Unable to create new java.sql.Time with long value "
							+ epochMillis + ": " + e.getMessage());
				}
			}
		}

		/**
		 * Converts text to a {@code Long} by delegating to
		 * {@link Long#valueOf(String)}.
		 */
		static class LongConverter implements ITypeConverter<Long> {
			@Override
			public Long convert(String value) {
				try {
					return Long.valueOf(value);
				} catch (final Exception ex) {
					throw fail(value, Long.TYPE);
				}
			}
		}

		static class NetworkInterfaceConverter implements ITypeConverter<NetworkInterface> {
			@Override
			public NetworkInterface convert(String s) throws Exception {
				try {
					final InetAddress addr = new InetAddressConverter().convert(s);
					return NetworkInterface.getByInetAddress(addr);
				} catch (final Exception ex) {
					try {
						return NetworkInterface.getByName(s);
					} catch (final Exception ex2) {
						throw new TypeConversionException("'" + s + "' is not an InetAddress or NetworkInterface name");
					}
				}
			}
		}

		static class PatternConverter implements ITypeConverter<Pattern> {
			@Override
			public Pattern convert(String s) {
				return Pattern.compile(s);
			}
		}

		static class ReflectionConverter implements ITypeConverter<Object> {
			private final Method method;
			private final Class<?>[] paramTypes;

			public ReflectionConverter(Method method, Class<?>... paramTypes) {
				this.method = Assert.notNull(method, "method");
				this.paramTypes = Assert.notNull(paramTypes, "paramTypes");
			}

			@Override
			public Object convert(String s) {
				try {
					if (paramTypes.length > 1) {
						return method.invoke(null, s, new String[0]);
					} else {
						return method.invoke(null, s);
					}
				} catch (final InvocationTargetException e) {
					throw new TypeConversionException(String.format("cannot convert '%s' to %s (%s)", s,
							method.getReturnType(), e.getTargetException()));
				} catch (final Exception e) {
					throw new TypeConversionException(
							String.format("Internal error converting '%s' to %s (%s)", s, method.getReturnType(), e));
				}
			}
		}

		/**
		 * Converts text to a {@code Short} by delegating to
		 * {@link Short#valueOf(String)}.
		 */
		static class ShortConverter implements ITypeConverter<Short> {
			@Override
			public Short convert(String value) {
				try {
					return Short.valueOf(value);
				} catch (final Exception ex) {
					throw fail(value, Short.TYPE);
				}
			}
		}

		static class StringBuilderConverter implements ITypeConverter<StringBuilder> {
			@Override
			public StringBuilder convert(String value) {
				return new StringBuilder(value);
			}
		}

		static class StringConverter implements ITypeConverter<String> {
			@Override
			public String convert(String value) {
				return value;
			}
		}

		static class TimeZoneConverter implements ITypeConverter<TimeZone> {
			@Override
			public TimeZone convert(String s) throws Exception {
				return TimeZone.getTimeZone(s);
			}
		}

		static class URIConverter implements ITypeConverter<URI> {
			@Override
			public URI convert(String value) throws URISyntaxException {
				return new URI(value);
			}
		}

		static class URLConverter implements ITypeConverter<URL> {
			@Override
			public URL convert(String value) throws MalformedURLException {
				return new URL(value);
			}
		}

		static class UUIDConverter implements ITypeConverter<UUID> {
			@Override
			public UUID convert(String s) throws Exception {
				return UUID.fromString(s);
			}
		}

		static Set<String> traced = new HashSet<String>();

		static boolean excluded(String fqcn) {
			final Tracer tracer = CommandLine.tracer();
			final String[] excludes = System.getProperty("picocli.converters.excludes", "").split(",");
			for (final String regex : excludes) {
				if (fqcn.matches(regex)) {
					tracer.debug("BuiltIn type converter for %s is not loaded: (picocli.converters.excludes=%s)", fqcn,
							System.getProperty("picocli.converters.excludes"));
					return true;
				}
			}
			return false;
		}

		private static TypeConversionException fail(String value, Class<?> c) {
			return fail(value, c, "'%s' is not a %s");
		}

		private static TypeConversionException fail(String value, Class<?> c, String template) {
			return new TypeConversionException(String.format(template, value, c.getSimpleName()));
		}

		static void handle(Exception e, String fqcn) {
			final Tracer tracer = CommandLine.tracer();
			if (!traced.contains(fqcn)) {
				tracer.debug("Could not register converter for %s: %s", fqcn, e.toString());
			}
			traced.add(fqcn);
		}

		private BuiltIn() {
		} // private constructor: never instantiate
	}

	/**
	 * Extends StringWriter to use ColorScheme. Allows separating exception messages
	 * from stack traces by intercepting write method.
	 */
	static class ColoredStackTraceWriter extends StringWriter {
		Help.ColorScheme colorScheme;

		public ColoredStackTraceWriter(Help.ColorScheme colorScheme) {
			this.colorScheme = colorScheme;
		}

		@Override
		public void write(String str, int off, int len) {
			final List<IStyle> styles = str.startsWith("\t") ? colorScheme.stackTraceStyles()
					: colorScheme.errorStyles();
			super.write(colorScheme.apply(str.substring(off, len), styles).toString());
		}
	}

	/**
	 * <p>
	 * Annotate your class with {@code @Command} when you want more control over the
	 * format of the generated help message. From 3.6, methods can also be annotated
	 * with {@code @Command}, where the method parameters define the command options
	 * and positional parameters.
	 * </p>
	 * 
	 * <pre>
	 * &#064;Command(name = "Encrypt", mixinStandardHelpOptions = true, description = "Encrypt FILE(s), or standard input, to standard output or to the output file.", version = "Encrypt version 1.0", footer = "Copyright (c) 2017", exitCodeListHeading = "Exit Codes:%n", exitCodeList = {
	 * 		" 0:Successful program execution.",
	 * 		"64:Invalid input: an unknown option or invalid parameter was specified.",
	 * 		"70:Execution exception: an exception occurred while executing the business logic." })
	 * public class Encrypt {
	 * 	&#064;Parameters(paramLabel = "FILE", description = "Any number of input files")
	 * 	private List&lt;File&gt; files = new ArrayList&lt;File&gt;();
	 *
	 * 	&#064;Option(names = { "-o", "--out" }, description = "Output file (default: print to console)")
	 * 	private File outputFile;
	 *
	 * 	&#064;Option(names = { "-v",
	 * 			"--verbose" }, description = "Verbose mode. Helpful for troubleshooting. Multiple -v options increase the verbosity.")
	 * 	private boolean[] verbose;
	 * }
	 * </pre>
	 * <p>
	 * The structure of a help message looks like this:
	 * </p>
	 * <ul>
	 * <li>[header]</li>
	 * <li>[synopsis]: {@code Usage: <commandName> [OPTIONS] [FILE...]}</li>
	 * <li>[description]</li>
	 * <li>[parameter list]: {@code      [FILE...]   Any number of input files}</li>
	 * <li>[option list]:
	 * {@code   -h, --help   prints this help message and exits}</li>
	 * <li>[exit code list]</li>
	 * <li>[footer]</li>
	 * </ul>
	 */
	@Retention(RetentionPolicy.RUNTIME)
	@Target({ ElementType.TYPE, ElementType.LOCAL_VARIABLE, ElementType.FIELD, ElementType.PACKAGE,
			ElementType.METHOD })
	public @interface Command {
		/**
		 * Specify {@code true} to generate an abbreviated synopsis like
		 * {@code "<main> [OPTIONS] [PARAMETERS...] [COMMAND]"}. By default, a detailed
		 * synopsis with individual option names and parameters is generated.
		 * 
		 * @return whether the synopsis should be abbreviated
		 * @see Help#abbreviatedSynopsis()
		 * @see Help#detailedSynopsis(int, Comparator, boolean)
		 */
		boolean abbreviateSynopsis() default false;

		/**
		 * Specify whether methods annotated with {@code @Command} should be registered
		 * as subcommands of their enclosing {@code @Command} class. The default is
		 * {@code true}. For example:
		 * 
		 * <pre>
		 * &#064;Command
		 * public class Git {
		 *     &#064;Command
		 *     void status() { ... }
		 * }
		 *
		 * CommandLine git = new CommandLine(new Git());
		 * </pre>
		 * 
		 * is equivalent to this:
		 * 
		 * <pre>
		 * // don't add command methods as subcommands automatically
		 * &#064;Command(addMethodSubcommands = false)
		 * public class Git {
		 *     &#064;Command
		 *     void status() { ... }
		 * }
		 *
		 * // add command methods as subcommands programmatically
		 * CommandLine git = new CommandLine(new Git());
		 * CommandLine status = new CommandLine(CommandLine.getCommandMethods(Git.class, "status").get(0));
		 * git.addSubcommand("status", status);
		 * </pre>
		 * 
		 * @return whether methods annotated with {@code @Command} should be registered
		 *         as subcommands
		 * @see CommandLine#addSubcommand(String, Object)
		 * @see CommandLine#getCommandMethods(Class, String)
		 * @see CommandSpec#addMethodSubcommands()
		 * @since 3.6.0
		 */
		boolean addMethodSubcommands() default true;

		/**
		 * Alternative command names by which this subcommand is recognized on the
		 * command line.
		 * 
		 * @return one or more alternative command names
		 * @since 3.1
		 */
		String[] aliases() default {};

		/**
		 * Set the heading preceding the subcommands list. The default heading is
		 * {@code "Commands:%n"} (with a line break at the end).
		 * <p>
		 * May contain embedded {@linkplain java.util.Formatter format specifiers} like
		 * {@code %n} line separators. Literal percent {@code '%'} characters must be
		 * escaped with another {@code %}.
		 * </p>
		 * 
		 * @return the heading preceding the subcommands list
		 * @see Help#commandListHeading(Object...)
		 */
		String commandListHeading() default "Commands:%n";

		/**
		 * Specify one or more custom synopsis lines to display instead of an
		 * auto-generated synopsis. Each element of the array is rendered on a separate
		 * line.
		 * <p>
		 * May contain embedded {@linkplain java.util.Formatter format specifiers} like
		 * {@code %n} line separators. Literal percent {@code '%'} characters must be
		 * escaped with another {@code %}.
		 * </p>
		 * 
		 * @return custom synopsis text to replace the auto-generated synopsis
		 * @see Help#customSynopsis(Object...)
		 */
		String[] customSynopsis() default {};

		/**
		 * Class that can provide default values dynamically at runtime. An
		 * implementation may return default value obtained from a configuration file
		 * like a properties file or some other source.
		 * <p>
		 * Applications may be interested in the {@link PropertiesDefaultProvider}
		 * built-in default provider that allows end users to maintain their own default
		 * values for options and positional parameters, which may override the defaults
		 * that are hard-coded in the application.
		 * </p>
		 * 
		 * @return a Class that can provide default values dynamically at runtime
		 * @since 3.6
		 */
		Class<? extends IDefaultValueProvider> defaultValueProvider() default NoDefaultProvider.class;

		/**
		 * Optional text to display between the synopsis line(s) and the list of
		 * options. Each element of the array is rendered on a separate line.
		 * <p>
		 * May contain embedded {@linkplain java.util.Formatter format specifiers} like
		 * {@code %n} line separators. Literal percent {@code '%'} characters must be
		 * escaped with another {@code %}.
		 * </p>
		 * 
		 * @return description of this command
		 * @see Help#description(Object...)
		 */
		String[] description() default {};

		/**
		 * Set the heading preceding the description section.
		 * <p>
		 * May contain embedded {@linkplain java.util.Formatter format specifiers} like
		 * {@code %n} line separators. Literal percent {@code '%'} characters must be
		 * escaped with another {@code %}.
		 * </p>
		 * 
		 * @return the heading preceding the description section
		 * @see Help#descriptionHeading(Object...)
		 */
		String descriptionHeading() default "";

		/**
		 * Set the values to be displayed in the exit codes section as a list of
		 * {@code "key:value"} pairs: keys are exit codes, values are descriptions.
		 * Descriptions may contain {@code "%n"} line separators.
		 * <p>
		 * For example:
		 * </p>
		 * 
		 * <pre>
		 * &#064;Command(exitCodeListHeading = "Exit Codes:%n",
		 *          exitCodeList = { " 0:Successful program execution.",
		 *                           "64:Invalid input: an unknown option or invalid parameter was specified.",
		 *                           "70:Execution exception: an exception occurred while executing the business logic."})
		 * </pre>
		 * 
		 * @since 4.0
		 */
		String[] exitCodeList() default {};

		/**
		 * Set the heading preceding the exit codes section, may contain {@code "%n"}
		 * line separators. {@code ""} (empty string) by default.
		 * 
		 * @see Help#exitCodeListHeading(Object...)
		 * @since 4.0
		 */
		String exitCodeListHeading() default "";

		/**
		 * Exit code signifying that an exception occurred when invoking the Runnable,
		 * Callable or Method user object of a command.
		 * {@value org.rossonet.ext.picocli.CommandLine.ExitCode#SOFTWARE} by default.
		 * 
		 * @see #execute(String...)
		 * @since 4.0
		 */
		int exitCodeOnExecutionException() default ExitCode.SOFTWARE;

		/**
		 * Exit code for command line usage error.
		 * {@value org.rossonet.ext.picocli.CommandLine.ExitCode#USAGE} by default.
		 * 
		 * @see #execute(String...)
		 * @since 4.0
		 */
		int exitCodeOnInvalidInput() default ExitCode.USAGE;

		/**
		 * Exit code for successful termination.
		 * {@value org.rossonet.ext.picocli.CommandLine.ExitCode#OK} by default.
		 * 
		 * @see #execute(String...)
		 * @since 4.0
		 */
		int exitCodeOnSuccess() default ExitCode.OK;

		/**
		 * Exit code for successful termination after printing usage help on user
		 * request. {@value org.rossonet.ext.picocli.CommandLine.ExitCode#OK} by
		 * default.
		 * 
		 * @see #execute(String...)
		 * @since 4.0
		 */
		int exitCodeOnUsageHelp() default ExitCode.OK;

		/**
		 * Exit code for successful termination after printing version help on user
		 * request. {@value org.rossonet.ext.picocli.CommandLine.ExitCode#OK} by
		 * default.
		 * 
		 * @see #execute(String...)
		 * @since 4.0
		 */
		int exitCodeOnVersionHelp() default ExitCode.OK;

		/**
		 * Optional text to display after the list of options. Each element of the array
		 * is rendered on a separate line.
		 * <p>
		 * May contain embedded {@linkplain java.util.Formatter format specifiers} like
		 * {@code %n} line separators. Literal percent {@code '%'} characters must be
		 * escaped with another {@code %}.
		 * </p>
		 * 
		 * @return text to display after the list of options
		 * @see Help#footer(Object...)
		 */
		String[] footer() default {};

		/**
		 * Set the heading preceding the footer section.
		 * <p>
		 * May contain embedded {@linkplain java.util.Formatter format specifiers} like
		 * {@code %n} line separators. Literal percent {@code '%'} characters must be
		 * escaped with another {@code %}.
		 * </p>
		 * 
		 * @return the heading preceding the footer section
		 * @see Help#footerHeading(Object...)
		 */
		String footerHeading() default "";

		/**
		 * Optional summary description of the command, shown before the synopsis. Each
		 * element of the array is rendered on a separate line.
		 * <p>
		 * May contain embedded {@linkplain java.util.Formatter format specifiers} like
		 * {@code %n} line separators. Literal percent {@code '%'} characters must be
		 * escaped with another {@code %}.
		 * </p>
		 * 
		 * @return summary description of the command
		 * @see UsageMessageSpec#header()
		 * @see Help#header(Object...)
		 */
		String[] header() default {};

		/**
		 * Set the heading preceding the header section.
		 * <p>
		 * May contain embedded {@linkplain java.util.Formatter format specifiers} like
		 * {@code %n} line separators. Literal percent {@code '%'} characters must be
		 * escaped with another {@code %}.
		 * </p>
		 * 
		 * @return the heading preceding the header section
		 * @see UsageMessageSpec#headerHeading()
		 * @see Help#headerHeading(Object...)
		 */
		String headerHeading() default "";

		/**
		 * Set this attribute to {@code true} if this subcommand is a help command, and
		 * required options and positional parameters of the parent command should not
		 * be validated. If a subcommand marked as {@code helpCommand} is specified on
		 * the command line, picocli will not validate the parent arguments (so no
		 * "missing required option" errors) and the
		 * {@link CommandLine#printHelpIfRequested(List, PrintStream, PrintStream, Help.Ansi)}
		 * method will return {@code true}.
		 * 
		 * @return {@code true} if this subcommand is a help command and picocli should
		 *         not check for missing required options and positional parameters on
		 *         the parent command
		 * @since 3.0
		 */
		boolean helpCommand() default false;

		/**
		 * Set {@code hidden=true} if this command should not be included in the list of
		 * commands in the usage help of the parent command.
		 * 
		 * @return whether this command should be excluded from the usage message
		 * @since 3.0
		 */
		boolean hidden() default false;

		/**
		 * Adds the standard {@code -h} and {@code --help}
		 * {@linkplain Option#usageHelp() usageHelp} options and {@code -V} and
		 * {@code --version} {@linkplain Option#versionHelp() versionHelp} options to
		 * the options of this command.
		 * <p>
		 * Note that if no {@link #version()} or {@link #versionProvider()} is
		 * specified, the {@code --version} option will not print anything.
		 * </p>
		 * <p>
		 * For {@linkplain #resourceBundle() internationalization}: the help option has
		 * {@code descriptionKey = "mixinStandardHelpOptions.help"}, and the version
		 * option has {@code descriptionKey = "mixinStandardHelpOptions.version"}.
		 * </p>
		 * 
		 * @return whether the auto-help mixin should be added to this command
		 * @since 3.0
		 */
		boolean mixinStandardHelpOptions() default false;

		/**
		 * Returns the model transformer for this command.
		 * 
		 * @since 4.6
		 */
		Class<? extends IModelTransformer> modelTransformer() default NoOpModelTransformer.class;

		/**
		 * Program name to show in the synopsis. If omitted, {@code "<main class>"} is
		 * used. For {@linkplain #subcommands() declaratively added} subcommands, this
		 * attribute is also used by the parser to recognize subcommands in the command
		 * line arguments.
		 * 
		 * @return the program name to show in the synopsis
		 * @see CommandSpec#name()
		 * @see Help#commandName()
		 */
		String name() default CommandSpec.DEFAULT_COMMAND_NAME;

		/**
		 * Set the heading preceding the options list.
		 * <p>
		 * May contain embedded {@linkplain java.util.Formatter format specifiers} like
		 * {@code %n} line separators. Literal percent {@code '%'} characters must be
		 * escaped with another {@code %}.
		 * </p>
		 * 
		 * @return the heading preceding the options list
		 * @see Help#optionListHeading(Object...)
		 */
		String optionListHeading() default "";

		/**
		 * Set the heading preceding the parameters list.
		 * <p>
		 * May contain embedded {@linkplain java.util.Formatter format specifiers} like
		 * {@code %n} line separators. Literal percent {@code '%'} characters must be
		 * escaped with another {@code %}.
		 * </p>
		 * 
		 * @return the heading preceding the parameters list
		 * @see Help#parameterListHeading(Object...)
		 */
		String parameterListHeading() default "";

		/**
		 * Returns the preprocessor for this command.
		 * 
		 * @see IParameterPreprocessor
		 * @since 4.6
		 */
		Class<? extends IParameterPreprocessor> preprocessor() default NoOpParameterPreprocessor.class;

		/**
		 * Prefix required options with this character in the options list. The default
		 * is no marker: the synopsis indicates which options and parameters are
		 * required.
		 * 
		 * @return the character to show in the options list to mark required options
		 */
		char requiredOptionMarker() default ' ';

		/**
		 * Set the base name of the ResourceBundle to find option and positional
		 * parameters descriptions, as well as usage help message sections and section
		 * headings.
		 * <p>
		 * See {@link Messages} for more details and an example.
		 * </p>
		 * 
		 * @return the base name of the ResourceBundle for usage help strings
		 * @see ArgSpec#messages()
		 * @see UsageMessageSpec#messages()
		 * @see CommandSpec#resourceBundle()
		 * @see CommandLine#setResourceBundle(ResourceBundle)
		 * @since 3.6
		 */
		String resourceBundle() default "";

		/**
		 * Returns whether subcommands inherit their attributes from this parent
		 * command.
		 * 
		 * @since 4.6
		 */
		ScopeType scope() default ScopeType.LOCAL;

		/**
		 * String that separates options from option parameters. Default is {@code "="}.
		 * Spaces are also accepted.
		 * 
		 * @return the string that separates options from option parameters, used both
		 *         when parsing and when generating usage help
		 * @see CommandLine#setSeparator(String)
		 */
		String separator() default "=";

		/**
		 * Specify {@code true} to show a {@code [@<filename>...]} entry in the synopsis
		 * and parameter list of the usage help message. (The entry is not shown if
		 * {@linkplain CommandLine#isExpandAtFiles() expanding parameter files} is
		 * disabled.)
		 * 
		 * @since 4.2
		 */
		boolean showAtFileInUsageHelp() default false;

		/**
		 * Specify {@code true} to show default values in the description column of the
		 * options list (except for boolean options). False by default.
		 * <p>
		 * Note that picocli 3.2 allows {@linkplain Option#description() embedding
		 * default values} anywhere in the option or positional parameter description
		 * that ignores this setting.
		 * </p>
		 * 
		 * @return whether the default values for options and parameters should be shown
		 *         in the description column
		 */
		boolean showDefaultValues() default false;

		/**
		 * Specify {@code true} to show a {@code [--]} "End of options" entry in the
		 * synopsis and option list of the usage help message.
		 * 
		 * @since 4.3
		 */
		boolean showEndOfOptionsDelimiterInUsageHelp() default false;

		/**
		 * Specify {@code false} to show Options in declaration order in the option list
		 * of the usage help message (or to sort options by their
		 * {@linkplain Option#order() order index} if set). Note that picocli cannot
		 * reliably detect declaration order in commands that have both
		 * {@code @Option}-annotated methods and {@code @Option}-annotated fields. The
		 * default ({@code true}) is to sort alphabetically.
		 * 
		 * @return whether options should be shown in alphabetic order.
		 */
		boolean sortOptions() default true;

		/**
		 * Specify {@code false} to show options in declaration order in the synopsis of
		 * the usage help message (or to sort options by their
		 * {@linkplain Option#order() order index} if set). Note that picocli cannot
		 * reliably detect declaration order in commands that have both
		 * {@code @Option}-annotated methods and {@code @Option}-annotated fields. The
		 * default ({@code true}) is to sort alphabetically.
		 * 
		 * @return whether options in the synopsis should be shown in alphabetic order.
		 * @since 4.7.6-SNAPSHOT
		 */
		boolean sortSynopsis() default true;

		/**
		 * A list of classes to instantiate and register as subcommands. When
		 * registering subcommands declaratively like this, you don't need to call the
		 * {@link CommandLine#addSubcommand(String, Object)} method. For example, this:
		 * 
		 * <pre>
		 * &#064;Command(subcommands = {
		 *         GitStatus.class,
		 *         GitCommit.class,
		 *         GitBranch.class })
		 * public class Git { ... }
		 *
		 * CommandLine commandLine = new CommandLine(new Git());
		 * </pre>
		 * 
		 * is equivalent to this:
		 * 
		 * <pre>
		 * // alternative: programmatically add subcommands.
		 * // NOTE: in this case there should be no `subcommands` attribute on the @Command annotation.
		 * &#064;Command public class Git { ... }
		 *
		 * CommandLine commandLine = new CommandLine(new Git())
		 *         .addSubcommand("status",   new GitStatus())
		 *         .addSubcommand("commit",   new GitCommit())
		 *         .addSubcommand("branch",   new GitBranch());
		 * </pre>
		 * 
		 * Applications may be interested in the following built-in commands in picocli
		 * that can be used as subcommands:
		 * <ul>
		 * <li>{@link HelpCommand} - a {@code help} subcommand that prints help on the
		 * following or preceding command</li>
		 * <li>{@link AutoComplete.GenerateCompletion} - a {@code generate-completion}
		 * subcommand that prints a Bash/ZSH completion script for its parent command,
		 * so that clients can install autocompletion in one line by running
		 * {@code source <(parent-command generate-completion)} in the shell</li>
		 * </ul>
		 * 
		 * @return the declaratively registered subcommands of this command, or an empty
		 *         array if none
		 * @see CommandLine#addSubcommand(String, Object)
		 * @see HelpCommand
		 * @since 0.9.8
		 */
		Class<?>[] subcommands() default {};

		/**
		 * Returns whether the subcommands of this command are repeatable, that is,
		 * whether such subcommands can occur multiple times and may be followed by
		 * sibling commands instead of only by child commands of the subcommand.
		 * 
		 * @since 4.2
		 */
		boolean subcommandsRepeatable() default false;

		/**
		 * Set the heading preceding the synopsis text. The default heading is
		 * {@code "Usage: "} (without a line break between the heading and the synopsis
		 * text).
		 * <p>
		 * May contain embedded {@linkplain java.util.Formatter format specifiers} like
		 * {@code %n} line separators. Literal percent {@code '%'} characters must be
		 * escaped with another {@code %}.
		 * </p>
		 * 
		 * @return the heading preceding the synopsis text
		 * @see Help#synopsisHeading(Object...)
		 */
		String synopsisHeading() default "Usage: ";

		/**
		 * Specify the String to show in the synopsis for the subcommands of this
		 * command. The default is {@code "[COMMAND]"}. Ignored if this command has no
		 * {@linkplain #subcommands() subcommands}.
		 * 
		 * @since 4.0
		 */
		String synopsisSubcommandLabel() default "[COMMAND]";

		/**
		 * If {@code true}, picocli will attempt to detect the terminal width and adjust
		 * the usage help message accordingly. End users may enable this by setting
		 * system property {@code "picocli.usage.width"} to {@code AUTO}, and may
		 * disable this by setting this system property to a
		 * {@linkplain UsageMessageSpec#width() numeric value}. This feature requires
		 * Java 7 or greater. The default is {@code false}
		 * 
		 * @see UsageMessageSpec#autoWidth()
		 * @since 4.0
		 */
		boolean usageHelpAutoWidth() default false;

		/**
		 * Set the {@link UsageMessageSpec#width(int) usage help message width}. The
		 * default is 80.
		 * 
		 * @see UsageMessageSpec#width()
		 * @since 3.7
		 */
		int usageHelpWidth() default 80;

		/**
		 * Version information for this command, to print to the console when the user
		 * specifies an {@linkplain Option#versionHelp() option} to request version
		 * help. Each element of the array is rendered on a separate line.
		 * <p>
		 * May contain embedded {@linkplain java.util.Formatter format specifiers} like
		 * {@code %n} line separators. Literal percent {@code '%'} characters must be
		 * escaped with another {@code %}.
		 * </p>
		 * <p>
		 * This is not part of the usage help message.
		 * </p>
		 *
		 * @return a string or an array of strings with version information about this
		 *         command (each string in the array is displayed on a separate line).
		 * @since 0.9.8
		 * @see CommandLine#printVersionHelp(PrintStream)
		 */
		String[] version() default {};

		/**
		 * Class that can provide version information dynamically at runtime. An
		 * implementation may return version information obtained from the JAR manifest,
		 * a properties file or some other source.
		 * 
		 * @return a Class that can provide version information dynamically at runtime
		 * @since 2.2
		 */
		Class<? extends IVersionProvider> versionProvider() default NoVersionProvider.class;
	}

	/**
	 * Uses cosine similarity to find matches from a candidate set for a specified
	 * input. Based on code from
	 * http://www.nearinfinity.com/blogs/seth_schroeder/groovy_cosine_similarity_in_grails.html
	 *
	 * @author Burt Beckwith
	 */
	private static class CosineSimilarity {
		private static Map<String, Integer> countNgramFrequency(String sequence, int degree) {
			final Map<String, Integer> m = new HashMap<String, Integer>();
			for (int i = 0; i + degree <= sequence.length(); i++) {
				final String gram = sequence.substring(i, i + degree);
				m.put(gram, 1 + (m.containsKey(gram) ? m.get(gram) : 0));
			}
			return m;
		}

		private static double dotProduct(Map<String, Integer> m1, Map<String, Integer> m2) {
			double result = 0;
			for (final String key : m1.keySet()) {
				result += m1.get(key) * (m2.containsKey(key) ? m2.get(key) : 0);
			}
			return result;
		}

		static List<String> mostSimilar(String pattern, Iterable<String> candidates) {
			return mostSimilar(pattern, candidates, 0);
		}

		static List<String> mostSimilar(String pattern, Iterable<String> candidates, double threshold) {
			pattern = pattern.toLowerCase();
			final SortedMap<Double, String> sorted = new TreeMap<Double, String>();
			for (final String candidate : candidates) {
				final double score = similarity(pattern, candidate.toLowerCase(), 2);
				if (score > threshold) {
					sorted.put(score, candidate);
				}
			}
			return reverseList(new ArrayList<String>(sorted.values()));
		}

		private static double similarity(String sequence1, String sequence2, int degree) {
			final Map<String, Integer> m1 = countNgramFrequency(sequence1, degree);
			final Map<String, Integer> m2 = countNgramFrequency(sequence2, degree);
			return dotProduct(m1, m2) / Math.sqrt(dotProduct(m1, m1) * dotProduct(m2, m2));
		}
	}

	/**
	 * Default exception handler that handles invalid user input by printing the
	 * exception message, followed by the usage message for the command or
	 * subcommand whose input was invalid.
	 * <p>
	 * {@code ParameterExceptions} (invalid user input) is handled like this:
	 * </p>
	 * 
	 * <pre>
	 * err().println(paramException.getMessage());
	 * paramException.getCommandLine().usage(err(), ansi());
	 * if (hasExitCode())
	 * 	System.exit(exitCode());
	 * else
	 * 	return returnValue;
	 * </pre>
	 * <p>
	 * {@code ExecutionExceptions} that occurred while executing the
	 * {@code Runnable} or {@code Callable} command are simply rethrown and not
	 * handled.
	 * </p>
	 * 
	 * @deprecated see {@link #execute(String...)},
	 *             {@link #getParameterExceptionHandler()} and
	 *             {@link #getExecutionExceptionHandler()}
	 * @since 2.0
	 */
	@Deprecated
	public static class DefaultExceptionHandler<R> extends AbstractHandler<R, DefaultExceptionHandler<R>>
			implements IExceptionHandler, IExceptionHandler2<R> {
		static void internalHandleParseException(ParameterException ex, PrintWriter writer,
				Help.ColorScheme colorScheme) {
			writer.println(colorScheme.errorText(ex.getMessage()));
			if (!UnmatchedArgumentException.printSuggestions(ex, writer)) {
				ex.getCommandLine().usage(writer, colorScheme);
			}
			final Tracer tracer = CommandLine.tracer();
			if (tracer.isDebug()) { // #956 show error details if DEBUG is enabled
				ex.printStackTrace(tracer.stream);
			}
		}

		@Override
		public List<Object> handleException(ParameterException ex, PrintStream out, Help.Ansi ansi, String... args) {
			internalHandleParseException(ex, newPrintWriter(out, getStdoutEncoding()), Help.defaultColorScheme(ansi));
			return Collections.emptyList();
		}

		/**
		 * This implementation always simply rethrows the specified exception.
		 * 
		 * @param ex          the ExecutionException describing the problem that
		 *                    occurred while executing the {@code Runnable} or
		 *                    {@code Callable} command
		 * @param parseResult the result of parsing the command line arguments
		 * @return nothing: this method always rethrows the specified exception
		 * @throws ExecutionException always rethrows the specified exception
		 * @since 3.0
		 */
		@Override
		public R handleExecutionException(ExecutionException ex, ParseResult parseResult) {
			return throwOrExit(ex);
		}

		/**
		 * Prints the message of the specified exception, followed by the usage message
		 * for the command or subcommand whose input was invalid, to the stream returned
		 * by {@link #err()}.
		 * 
		 * @param ex   the ParameterException describing the problem that occurred while
		 *             parsing the command line arguments, and the CommandLine
		 *             representing the command or subcommand whose input was invalid
		 * @param args the command line arguments that could not be parsed
		 * @return the empty list
		 * @since 3.0
		 */
		@Override
		public R handleParseException(ParameterException ex, String[] args) {
			internalHandleParseException(ex, newPrintWriter(err(), getStderrEncoding()), colorScheme());
			return returnResultOrExit(null);
		}

		@Override
		protected DefaultExceptionHandler<R> self() {
			return this;
		}
	}

	private static class DefaultFactory implements IFactory {
		static Class<?> GROOVY_CLOSURE_CLASS = loadClosureClass();

		static <T> T create(IFactory factory, Class<T> cls) {
			try {
				return factory.create(cls);
			} catch (final NoSuchMethodException ex) {
				throw new InitializationException(
						"Cannot instantiate " + cls.getName() + ": the class has no constructor", ex);
			} catch (final InitializationException ex) {
				throw ex;
			} catch (final Exception ex) {
				throw new InitializationException("Could not instantiate " + cls + ": " + ex, ex);
			}
		}

		static Iterable<String> createCompletionCandidates(IFactory factory, Class<? extends Iterable<String>> cls) {
			return create(factory, cls);
		}

		private static ITypeConverter<?>[] createConverter(IFactory factory,
				Class<? extends ITypeConverter<?>>[] classes) {
			final ITypeConverter<?>[] result = new ITypeConverter<?>[classes.length];
			for (int i = 0; i < classes.length; i++) {
				result[i] = create(factory, classes[i]);
			}
			return result;
		}

		static IDefaultValueProvider createDefaultValueProvider(IFactory factory,
				Class<? extends IDefaultValueProvider> cls) {
			return create(factory, cls);
		}

		static IParameterConsumer createParameterConsumer(IFactory factory, Class<? extends IParameterConsumer> cls) {
			return create(factory, cls);
		}

		static IVersionProvider createVersionProvider(IFactory factory, Class<? extends IVersionProvider> cls) {
			return create(factory, cls);
		}

		private static Class<?> loadClosureClass() {
			if (Boolean.getBoolean("picocli.disable.closures")) {
				tracer().info("DefaultFactory: groovy Closures in annotations are disabled and will not be loaded");
				return null;
			}
			try {
				return Class.forName("groovy.lang.Closure");
			} catch (final Exception ignored) {
				return null;
			}
		}

		@Override
		@SuppressWarnings("unchecked")
		public <T> T create(Class<T> cls) throws Exception {
			if (GROOVY_CLOSURE_CLASS != null && GROOVY_CLOSURE_CLASS.isAssignableFrom(cls)) {
				final Callable<?> callable = Callable.class
						.cast(cls.getConstructor(Object.class, Object.class).newInstance(null, null));
				try {
					return (T) callable.call();
				} catch (final Exception ex) {
					throw new InitializationException("Error in Groovy closure: " + ex);
				}
			}
			if (cls.isInterface()) {
				if (Collection.class.isAssignableFrom(cls)) {
					if (List.class.isAssignableFrom(cls)) {
						return cls.cast(new ArrayList<Object>());
					} else if (SortedSet.class.isAssignableFrom(cls)) {
						return cls.cast(new TreeSet<Object>());
					} else if (Set.class.isAssignableFrom(cls)) {
						return cls.cast(new LinkedHashSet<Object>());
					} else if (Queue.class.isAssignableFrom(cls)) {
						return cls.cast(new LinkedList<Object>()); // ArrayDeque is only available since 1.6
					} else {
						return cls.cast(new ArrayList<Object>());
					}
				}
				if (SortedMap.class.isAssignableFrom(cls)) {
					return cls.cast(new TreeMap<Object, Object>());
				}
				if (Map.class.isAssignableFrom(cls)) {
					return cls.cast(new LinkedHashMap<Object, Object>());
				}
			}
			try {
				@SuppressWarnings("deprecation") // Class.newInstance is deprecated in Java 9
				final T result = cls.newInstance();
				return result;
			} catch (final Exception ex) {
				// TODO log the error at debug level
				final Constructor<T> constructor = cls.getDeclaredConstructor();
				try {
					return constructor.newInstance();
				} catch (final IllegalAccessException iaex) {
					constructor.setAccessible(true);
					return constructor.newInstance();
				}
			}
		}
	}

	private static class DefaultHelpFactory implements IHelpFactory {
		@Override
		public Help create(CommandSpec commandSpec, Help.ColorScheme colorScheme) {
			return new Help(commandSpec, colorScheme);
		}
	}

	/**
	 * Exception indicating that multiple named elements have incorrectly used the
	 * same name.
	 * 
	 * @since 4.0
	 */
	public static class DuplicateNameException extends InitializationException {
		private static final long serialVersionUID = -4126747467955626054L;

		public DuplicateNameException(String msg) {
			super(msg);
		}
	}

	/**
	 * Exception indicating that multiple fields have been annotated with the same
	 * Option name.
	 */
	public static class DuplicateOptionAnnotationsException extends DuplicateNameException {
		private static final long serialVersionUID = -3355128012575075641L;

		private static DuplicateOptionAnnotationsException create(String name, ArgSpec argSpec1, ArgSpec argSpec2) {
			return new DuplicateOptionAnnotationsException("Option name '" + name + "' is used by both "
					+ argSpec1.toString() + " and " + argSpec2.toString());
		}

		public DuplicateOptionAnnotationsException(String msg) {
			super(msg);
		}
	}

	/**
	 * Exception indicating a problem while invoking a command or subcommand. Keeps
	 * a reference to the {@code CommandLine} object where the cause exception
	 * occurred, so that client code can tailor their handling for the specific
	 * command (print the command's usage help message, for example).
	 * 
	 * @since 2.0
	 */
	public static class ExecutionException extends PicocliException {
		private static final long serialVersionUID = 7764539594267007998L;
		private final CommandLine commandLine;

		public ExecutionException(CommandLine commandLine, String msg) {
			super(msg);
			this.commandLine = Assert.notNull(commandLine, "commandLine");
		}

		public ExecutionException(CommandLine commandLine, String msg, Throwable t) {
			super(msg, t);
			this.commandLine = Assert.notNull(commandLine, "commandLine");
		}

		/**
		 * Returns the {@code CommandLine} object for the (sub)command that could not be
		 * invoked.
		 * 
		 * @return the {@code CommandLine} object for the (sub)command where invocation
		 *         failed.
		 */
		public CommandLine getCommandLine() {
			return commandLine;
		}
	}

	/**
	 * Defines some exit codes used by picocli as default return values from the
	 * {@link #execute(String...) execute} and
	 * {@link #executeHelpRequest(ParseResult) executeHelpRequest} methods.
	 * <p>
	 * Commands can override these defaults with annotations (e.g.
	 * {@code @Command(exitCodeOnInvalidInput = 64, exitCodeOnExecutionException = 70)}
	 * or programmatically (e.g. {@link CommandSpec#exitCodeOnInvalidInput(int)}).
	 * </p>
	 * <p>
	 * Additionally, there are several mechanisms for commands to return custom exit
	 * codes. See the javadoc of the {@link #execute(String...) execute} method for
	 * details.
	 * </p>
	 * <h3>Standard Exit Codes</h3>
	 * <p>
	 * There are a few conventions, but there is <a href=
	 * "https://stackoverflow.com/questions/1101957/are-there-any-standard-exit-status-codes-in-linux/40484670#40484670">no
	 * standard</a>. The specific set of codes returned is unique to the program
	 * that sets it. Typically an exit code of zero indicates success, any non-zero
	 * exit code indicates failure. For reference, here are a few conventions:
	 * </p>
	 * <ul>
	 * <li><a href="https://en.wikipedia.org/wiki/Exit_status">Wikipedia page on
	 * Exit Status</a></li>
	 * <li><a href=
	 * "http://www.faqs.org/docs/abs/HTML/exitcodes.html#EXITCODESREF">Bash exit
	 * codes</a></li>
	 * <li><a href=
	 * "https://www.freebsd.org/cgi/man.cgi?query=sysexits&amp;sektion=3">FreeBSD
	 * exit codes</a></li>
	 * <li><a href="http://www.hiteksoftware.com/knowledge/articles/049.htm">Windows
	 * exit codes</a></li>
	 * </ul>
	 * <h3>Valid Ranges</h3>
	 * <p>
	 * Note that *nix shells may restrict exit codes to the 0-255 range, DOS seems
	 * to allow larger numbers. See this <a href=
	 * "https://stackoverflow.com/questions/179565/exitcodes-bigger-than-255-possible">StackOverflow
	 * question</a>.
	 * </p>
	 * 
	 * @since 4.0
	 */
	public static final class ExitCode {
		/**
		 * Return value from the {@link #execute(String...) execute} and
		 * {@link #executeHelpRequest(ParseResult) executeHelpRequest} methods
		 * signifying successful termination.
		 * <p>
		 * The value of this constant is {@value}.
		 * </p>
		 */
		public static final int OK = 0;
		/**
		 * Return value from the {@link #execute(String...) execute} method signifying
		 * internal software error: an exception occurred when invoking the Runnable,
		 * Callable or Method user object of a command.
		 * <p>
		 * The value of this constant is {@value}.
		 * </p>
		 */
		public static final int SOFTWARE = 1;
		/**
		 * Return value from the {@link #execute(String...) execute} method signifying
		 * command line usage error: user input for the command was incorrect, e.g., the
		 * wrong number of arguments, a bad flag, a bad syntax in a parameter, or
		 * whatever.
		 * <p>
		 * The value of this constant is {@value}.
		 * </p>
		 */
		public static final int USAGE = 2;

		private ExitCode() {
		} // don't instantiate
	}

	/**
	 * A collection of methods and inner classes that provide fine-grained control
	 * over the contents and layout of the usage help message to display to end
	 * users when help is requested or invalid input values were specified.
	 * <h2>Class Diagram of the CommandLine.Help API</h2>
	 * <p>
	 * <img src="doc-files/class-diagram-help-api.png" alt="Class Diagram of the
	 * CommandLine.Help API">
	 * </p>
	 * <h2>Layered API</h2>
	 * <p>
	 * The {@link Command} annotation and the {@link UsageMessageSpec} programmatic
	 * API equivalent provide the easiest way to configure the usage help message.
	 * See the
	 * <a href="https://remkop.github.io/picocli/index.html#_usage_help">Manual</a>
	 * for details.
	 * </p>
	 * <p>
	 * This Help class provides high-level functions to create sections of the usage
	 * help message and headings for these sections. Instead of calling the
	 * {@link CommandLine#usage(PrintStream, CommandLine.Help.ColorScheme)} method,
	 * application authors may want to create a custom usage help message by
	 * reorganizing sections in a different order and/or adding custom sections.
	 * </p>
	 * <p>
	 * Finally, the Help class contains inner classes and interfaces that can be
	 * used to create custom help messages.
	 * </p>
	 * <h3>IOptionRenderer and IParameterRenderer</h3>
	 * <p>
	 * Renders a field annotated with {@link Option} or {@link Parameters} to an
	 * array of {@link Text} values. By default, these values are
	 * </p>
	 * <ul>
	 * <li>mandatory marker character (if the option/parameter is
	 * {@link Option#required() required})</li>
	 * <li>short option name (empty for parameters)</li>
	 * <li>comma or empty (empty for parameters)</li>
	 * <li>long option names (the parameter {@link IParamLabelRenderer label} for
	 * parameters)</li>
	 * <li>description</li>
	 * </ul>
	 * <p>
	 * Other components rely on this ordering.
	 * </p>
	 * <h3>Layout</h3>
	 * <p>
	 * Delegates to the renderers to create {@link Text} values for the annotated
	 * fields, and uses a {@link TextTable} to display these values in tabular
	 * format. Layout is responsible for deciding which values to display where in
	 * the table. By default, Layout shows one option or parameter per table row.
	 * </p>
	 * <h3>TextTable</h3>
	 * <p>
	 * Responsible for spacing out {@link Text} values according to the
	 * {@link Column} definitions the table was created with. Columns have a width,
	 * indentation, and an overflow policy that decides what to do if a value is
	 * longer than the column's width.
	 * </p>
	 * <h3>Text</h3>
	 * <p>
	 * Encapsulates rich text with styles and colors in a way that other components
	 * like {@link TextTable} are unaware of the embedded ANSI escape codes.
	 * </p>
	 */
	public static class Help {

		/**
		 * Provides methods and inner classes to support using ANSI escape codes in
		 * usage help messages.
		 */
		public enum Ansi {
			/**
			 * Only emit ANSI escape codes if the platform supports it and system property
			 * {@code "picocli.ansi"} is not set to any value other than {@code "true"}
			 * (case insensitive).
			 */
			AUTO,
			/** Forced ON: always emit ANSI escape code regardless of the platform. */
			ON,
			/** Forced OFF: never emit ANSI escape code regardless of the platform. */
			OFF;

			/** Defines the interface for an ANSI escape sequence. */
			public interface IStyle {

				/** The Control Sequence Introducer (CSI) escape sequence {@value}. */
				String CSI = "\u001B[";

				/**
				 * Returns the ANSI escape code for turning this style off.
				 * 
				 * @return the ANSI escape code for turning this style off
				 */
				String off();

				/**
				 * Returns the ANSI escape code for turning this style on.
				 * 
				 * @return the ANSI escape code for turning this style on
				 */
				String on();
			}

			/**
			 * Defines a palette map of 216 colors: 6 * 6 * 6 cube (216 colors): 16 + 36 * r
			 * + 6 * g + b (0 &lt;= r, g, b &lt;= 5).
			 */
			static class Palette256Color implements IStyle {
				private final int fgbg;
				private final int color;

				Palette256Color(boolean foreground, String color) {
					this.fgbg = foreground ? 38 : 48;
					final String[] rgb = color.split(";");
					if (rgb.length == 3) {
						this.color = 16 + 36 * Integer.decode(rgb[0]) + 6 * Integer.decode(rgb[1])
								+ Integer.decode(rgb[2]);
					} else {
						this.color = Integer.decode(color);
					}
				}

				@Override
				public boolean equals(Object obj) {
					if (obj == this) {
						return true;
					}
					if (!(obj instanceof Palette256Color)) {
						return false;
					}
					final Palette256Color other = (Palette256Color) obj;
					return other.fgbg == this.fgbg && other.color == this.color;
				}

				@Override
				public int hashCode() {
					return (17 + fgbg) * 37 + color;
				}

				@Override
				public String off() {
					return CSI + (fgbg + 1) + "m";
				}

				@Override
				public String on() {
					return String.format(CSI + "%d;5;%dm", fgbg, color);
				}
			}

			/**
			 * A set of pre-defined ANSI escape code styles and colors, and a set of
			 * convenience methods for parsing text with embedded markup style names, as
			 * well as convenience methods for converting styles to strings with embedded
			 * escape codes.
			 */
			public enum Style implements IStyle {
				reset(0, 0), bold(1, 21), faint(2, 22), italic(3, 23), underline(4, 24), blink(5, 25), reverse(7, 27),
				fg_black(30, 39), fg_red(31, 39), fg_green(32, 39), fg_yellow(33, 39), fg_blue(34, 39),
				fg_magenta(35, 39), fg_cyan(36, 39), fg_white(37, 39), bg_black(40, 49), bg_red(41, 49),
				bg_green(42, 49), bg_yellow(43, 49), bg_blue(44, 49), bg_magenta(45, 49), bg_cyan(46, 49),
				bg_white(47, 49),;

				/**
				 * Parses the specified style markup and returns the associated style. The
				 * markup may be one of the Style enum value names, or it may be one of the
				 * Style enum value names when {@code "bg_"} is prepended, or it may be one of
				 * the indexed colors in the 256 color palette.
				 * 
				 * @param str the case-insensitive style markup to convert, e.g. {@code "blue"}
				 *            or {@code "bg_blue"}, or {@code "46"} (indexed color) or
				 *            {@code "0;5;0"} (RGB components of an indexed color)
				 * @return the IStyle for the specified converter
				 */
				public static IStyle bg(String str) {
					try {
						return Style.valueOf(str.toLowerCase(ENGLISH));
					} catch (final Exception ignored) {
					}
					try {
						return Style.valueOf("bg_" + str.toLowerCase(ENGLISH));
					} catch (final Exception ignored) {
					}
					return new Palette256Color(false, str);
				}

				/**
				 * Parses the specified style markup and returns the associated style. The
				 * markup may be one of the Style enum value names, or it may be one of the
				 * Style enum value names when {@code "fg_"} is prepended, or it may be one of
				 * the indexed colors in the 256 color palette.
				 * 
				 * @param str the case-insensitive style markup to convert, e.g. {@code "blue"}
				 *            or {@code "fg_blue"}, or {@code "46"} (indexed color) or
				 *            {@code "0;5;0"} (RGB components of an indexed color)
				 * @return the IStyle for the specified converter
				 */
				public static IStyle fg(String str) {
					try {
						return Style.valueOf(str.toLowerCase(ENGLISH));
					} catch (final Exception ignored) {
					}
					try {
						return Style.valueOf("fg_" + str.toLowerCase(ENGLISH));
					} catch (final Exception ignored) {
					}
					return new Palette256Color(true, str);
				}

				/**
				 * Returns the concatenated ANSI escape codes for turning all specified styles
				 * off.
				 * 
				 * @param styles the styles to generate ANSI escape codes for
				 * @return the concatenated ANSI escape codes for turning all specified styles
				 *         off
				 */
				public static String off(IStyle... styles) {
					final StringBuilder result = new StringBuilder();
					for (final IStyle style : styles) {
						result.append(style.off());
					}
					return result.toString();
				}

				/**
				 * Returns the concatenated ANSI escape codes for turning all specified styles
				 * on.
				 * 
				 * @param styles the styles to generate ANSI escape codes for
				 * @return the concatenated ANSI escape codes for turning all specified styles
				 *         on
				 */
				public static String on(IStyle... styles) {
					final StringBuilder result = new StringBuilder();
					for (final IStyle style : styles) {
						result.append(style.on());
					}
					return result.toString();
				}

				/**
				 * Parses the specified comma-separated sequence of style descriptors and
				 * returns the associated styles. For each markup, strings starting with
				 * {@code "bg("} are delegated to {@link #bg(String)}, others are delegated to
				 * {@link #fg(String)}.
				 * 
				 * @param commaSeparatedCodes one or more descriptors, e.g.
				 *                            {@code "bg(blue),underline,red"}
				 * @return an array with all styles for the specified descriptors
				 */
				public static IStyle[] parse(String commaSeparatedCodes) {
					final String[] codes = commaSeparatedCodes.split(",");
					final IStyle[] styles = new IStyle[codes.length];
					for (int i = 0; i < codes.length; ++i) {
						if (codes[i].toLowerCase(ENGLISH).startsWith("fg(")) {
							final int end = codes[i].indexOf(')');
							styles[i] = Style.fg(codes[i].substring(3, end < 0 ? codes[i].length() : end));
						} else if (codes[i].toLowerCase(ENGLISH).startsWith("bg(")) {
							final int end = codes[i].indexOf(')');
							styles[i] = Style.bg(codes[i].substring(3, end < 0 ? codes[i].length() : end));
						} else {
							styles[i] = Style.fg(codes[i]);
						}
					}
					return styles;
				}

				private final int startCode;
				private final int endCode;

				Style(int startCode, int endCode) {
					this.startCode = startCode;
					this.endCode = endCode;
				}

				@Override
				public String off() {
					return CSI + endCode + "m";
				}

				@Override
				public String on() {
					return CSI + startCode + "m";
				}
			}

			private static class StyledSection {
				int startIndex, length;
				String startStyles, endStyles;

				StyledSection(int start, int len, String style1, String style2) {
					startIndex = start;
					length = len;
					startStyles = style1;
					endStyles = style2;
				}

				StyledSection withStartIndex(int newStart) {
					return new StyledSection(newStart, length, startStyles, endStyles);
				}
			}

			/**
			 * Encapsulates rich text with styles and colors. Text objects may be
			 * constructed with Strings containing markup like
			 * {@code @|bg(red),white,underline some text|@}, and this class converts the
			 * markup to ANSI escape codes.
			 * <p>
			 * Internally keeps both an enriched and a plain text representation to allow
			 * layout components to calculate text width while remaining unaware of the
			 * embedded ANSI escape codes.
			 * </p>
			 */
			public class Text implements Cloneable {
				private final int maxLength;
				private int from;
				private int length;
				private StringBuilder plain = new StringBuilder();
				private List<StyledSection> sections = new ArrayList<StyledSection>();
				private ColorScheme colorScheme;

				/**
				 * Constructs a Text with the specified max length (for use in a TextTable
				 * Column).
				 * 
				 * @param maxLength max length of this text
				 */
				public Text(int maxLength) {
					this(maxLength, Help.defaultColorScheme(Ansi.this));
				}

				/**
				 * Constructs a Text with the specified max length (for use in a TextTable
				 * Column).
				 * 
				 * @param maxLength   max length of this text
				 * @param colorScheme the colorScheme to use
				 * @since 4.2
				 */
				public Text(int maxLength, ColorScheme colorScheme) {
					this.maxLength = maxLength;
					this.colorScheme = colorScheme;
				}

				/**
				 * Constructs a Text with the specified String, which may contain markup like
				 * {@code @|bg(red),white,underline some text|@}.
				 * 
				 * @param input the string with markup to parse
				 */
				public Text(String input) {
					this(input, Help.defaultColorScheme(Ansi.this));
				}

				/**
				 * Constructs a Text with the specified String (which may contain markup), and
				 * ColorScheme.
				 * 
				 * @param input       the string with markup like
				 *                    {@code @|bg(red),white,underline some text|@} to parse
				 * @param colorScheme the ColorScheme to use to map markup to replacement
				 *                    strings
				 * @since 4.2
				 */
				public Text(String input, ColorScheme colorScheme) {
					this.colorScheme = colorScheme;
					maxLength = -1;
					plain.setLength(0);
					int i = 0;

					while (true) {
						int j = input.indexOf("@|", i);
						if (j == -1) {
							if (i == 0) {
								plain.append(input);
								length = plain.length();
								return;
							}
							plain.append(input.substring(i));
							length = plain.length();
							return;
						}
						plain.append(input, i, j);
						final int k = input.indexOf("|@", j);
						if (k == -1) {
							plain.append(input);
							length = plain.length();
							return;
						}

						j += 2;
						final String spec = input.substring(j, k);
						final String[] items = spec.split(" ", 2);
						if (items.length == 1) {
							plain.append(input);
							length = plain.length();
							return;
						}

						final IStyle[] styles = colorScheme.parse(items[0]);
						addStyledSection(plain.length(), items[1].length(), Style.on(styles),
								Style.off(reverseArray(styles)) + colorScheme.resetStyle().off());
						plain.append(items[1]);
						i = k + 2;
					}
				}

				/**
				 * Copy constructor.
				 * 
				 * @since 3.9
				 */
				public Text(Text other) {
					this.maxLength = other.maxLength;
					this.from = other.from;
					this.length = other.length;
					this.plain = new StringBuilder(other.plain);
					this.sections = new ArrayList<StyledSection>(other.sections);
					this.colorScheme = other.colorScheme;
				}

				private void addStyledSection(int start, int length, String startStyle, String endStyle) {
					sections.add(new StyledSection(start, length, startStyle, endStyle));
				}

				/** @deprecated use {@link #concat(String)} instead */
				@Deprecated
				public Text append(String string) {
					return concat(string);
				}

				/** @deprecated use {@link #concat(CommandLine.Help.Ansi.Text)} instead */
				@Deprecated
				public Text append(Text text) {
					return concat(text);
				}

				@Override
				public Object clone() {
					return new Text(this);
				}

				/**
				 * Returns a copy of this {@code Text} instance with the specified text
				 * concatenated to the end. Does not modify this instance!
				 * 
				 * @param string the text to concatenate to the end of this Text
				 * @return a new Text instance
				 * @since 3.0
				 */
				public Text concat(String string) {
					return concat(new Text(string, colorScheme));
				}

				/**
				 * Returns a copy of this {@code Text} instance with the specified text
				 * concatenated to the end. Does not modify this instance!
				 * 
				 * @param other the text to concatenate to the end of this Text
				 * @return a new Text instance
				 * @since 3.0
				 */
				public Text concat(Text other) {
					final Text result = (Text) clone();
					result.plain = new StringBuilder(plain.toString().substring(from, from + length));
					result.from = 0;
					result.sections = new ArrayList<StyledSection>();
					for (final StyledSection section : sections) {
						result.sections.add(section.withStartIndex(section.startIndex - from));
					}
					result.plain.append(other.plain.toString(), other.from, other.from + other.length);
					for (final StyledSection section : other.sections) {
						final int index = result.length + section.startIndex - other.from;
						result.sections.add(section.withStartIndex(index));
					}
					result.length = result.plain.length();
					return result;
				}

				@Override
				public boolean equals(Object obj) {
					return toString().equals(String.valueOf(obj));
				}

				private StyledSection findSectionContaining(int index) {
					for (final StyledSection section : sections) {
						if (index >= section.startIndex && index < section.startIndex + section.length) {
							return section;
						}
					}
					return null;
				}

				/**
				 * Returns the number of columns this Text will occupy on the console, adjusted
				 * for wide CJK characters.
				 * 
				 * @return the number of columns this Text will occupy on the console, adjusted
				 *         for wide CJK characters
				 * @since 4.0
				 */
				public int getCJKAdjustedLength() {
					return getCJKAdjustedLength(from, length);
				}

				/**
				 * Returns the number of columns that the specified portion of this Text will
				 * occupy on the console, adjusted for wide CJK characters.
				 * 
				 * @param fromPosition the position to start counting
				 * @param charCount    the number of characters in this Text to consider
				 * @return the number of columns that the specified portion of this Text will
				 *         occupy on the console, adjusted for wide CJK characters
				 * @since 4.0
				 */
				public int getCJKAdjustedLength(int fromPosition, int charCount) {
					final String lengthOf = plain.substring(from, from + charCount);

					int result = 0;
					int i = 0;
					while (i < lengthOf.length()) {
						int codePoint;
						final char c1 = lengthOf.charAt(i++);
						if (!Character.isHighSurrogate(c1) || i >= length) {
							codePoint = c1;
						} else {
							final char c2 = lengthOf.charAt(i);
							if (Character.isLowSurrogate(c2)) {
								i++;
								codePoint = Character.toCodePoint(c1, c2);
							} else {
								codePoint = c1;
							}
						}
						result += UsageMessageSpec.isCodePointCJK(codePoint) ? 2 : 1;
					}

					return result;
				}

				/**
				 * Copies the specified substring of this Text into the specified destination,
				 * preserving the markup.
				 * 
				 * @param from        start of the substring
				 * @param length      length of the substring
				 * @param destination destination Text to modify
				 * @param offset      indentation (padding)
				 */
				public void getStyledChars(int from, int length, Text destination, int offset) {
					if (destination.length < offset) {
						for (int i = destination.length; i < offset; i++) {
							destination.plain.append(' ');
						}
						destination.length = offset;
					}
					for (final StyledSection section : sections) {
						if ((section.startIndex - from) + section.length >= 0) {
							destination.sections
									.add(section.withStartIndex(section.startIndex - from + destination.length));
						}
					}
					destination.plain.append(plain.toString(), from, from + length);
					destination.length = destination.plain.length();
				}

				@Override
				public int hashCode() {
					return toString().hashCode();
				}

				/**
				 * Returns the plain text without any formatting.
				 * 
				 * @return the plain text without any formatting
				 */
				public String plainString() {
					return plain.toString().substring(from, from + length);
				}

				public Text[] splitLines() {
					final List<Text> result = new ArrayList<Text>();
					int start = 0, end = 0;
					for (int i = 0; i < plain.length(); i++, end = i) {
						final char c = plain.charAt(i);
						boolean eol = c == '\n';
						if (c == '\r' && i + 1 < plain.length() && plain.charAt(i + 1) == '\n') {
							eol = true;
							i++;
						} // \r\n
						eol |= c == '\r';
						if (eol) {
							result.add(this.substring(start, end));
							start = i + 1;
						}
					}
					// add remainder (may be empty string)
					result.add(this.substring(start, plain.length()));
					return result.toArray(new Text[result.size()]);
				}

				/**
				 * Returns a new {@code Text} instance that is a substring of this Text. Does
				 * not modify this instance!
				 * 
				 * @param start index in the plain text where to start the substring
				 * @return a new Text instance that is a substring of this Text
				 */
				public Text substring(int start) {
					return substring(start, length);
				}

				/**
				 * Returns a new {@code Text} instance that is a substring of this Text. Does
				 * not modify this instance!
				 * 
				 * @param start index in the plain text where to start the substring
				 * @param end   index in the plain text where to end the substring
				 * @return a new Text instance that is a substring of this Text
				 */
				public Text substring(int start, int end) {
					final Text result = (Text) clone();
					result.from = from + start;
					result.length = end - start;
					result.sections.clear();
					for (final StyledSection section : this.sections) {
						if (section.startIndex >= result.from + result.length) {
							continue;
						}
						if (section.startIndex + section.length <= result.from) {
							continue;
						}
						result.sections.add(section);
					}
					return result;
				}

				/**
				 * Returns a String representation of the text with ANSI escape codes embedded,
				 * unless ANSI is {@linkplain Ansi#enabled()} not enabled}, in which case the
				 * plain text is returned.
				 * 
				 * @return a String representation of the text with ANSI escape codes embedded
				 *         (if enabled)
				 */
				@Override
				public String toString() {
					if (!Ansi.this.enabled()) {
						return plain.toString().substring(from, from + length);
					}
					if (length == 0) {
						return "";
					}
					final StringBuilder sb = new StringBuilder(plain.length() + 20 * sections.size());
					StyledSection current = null;
					final int end = Math.min(from + length, plain.length());
					for (int i = from; i < end; i++) {
						final StyledSection section = findSectionContaining(i);
						if (section != current) {
							if (current != null) {
								sb.append(current.endStyles);
							}
							if (section != null) {
								sb.append(section.startStyles);
							}
							current = section;
						}
						sb.append(plain.charAt(i));
					}
					if (current != null) {
						sb.append(current.endStyles);
					}
					return sb.toString();
				}
			}

			static Text EMPTY_TEXT = OFF.new Text(0);
			static Boolean tty;
			/**
			 * Caches the result of method {@link #isJansiConsoleInstalled()} so it doesn't
			 * repeatedly call Class#forName, which can cause performance issues.
			 */
			static Boolean jansiInstalled;

			static boolean ansiPossible() {
				if (forceDisabled()) {
					return false;
				}
				if (forceEnabled()) {
					return true;
				}
				if (isWindows() && isJansiConsoleInstalled()) {
					return true;
				} // #630 JVM crash loading jansi.AnsiConsole on Linux
				if (hintDisabled()) {
					return false;
				}
				if (!isTTY() && !isPseudoTTY()) {
					return false;
				}
				return hintEnabled() || !isWindows() || isXterm() || isCygwin() || hasOsType();
			}

			/**
			 * Returns {@code false} if system property
			 * {@code org.fusesource.jansi.Ansi.disable} is set to {@code "true"}
			 * (case-insensitive); otherwise, returns {@code false} if the Jansi library is
			 * in the classpath but has been disabled (either via system property
			 * {@code org.fusesource.jansi.Ansi.disable} or via a Jansi API call);
			 * otherwise, returns {@code true} if the Jansi library is in the classpath and
			 * has been installed.
			 */
			static boolean calcIsJansiConsoleInstalled() {
				try {
					// first check if JANSI was explicitly disabled _without loading any JANSI
					// classes_:
					// see https://github.com/remkop/picocli/issues/1106
					if (Boolean.getBoolean("org.fusesource.jansi.Ansi.disable")) {
						return false;
					}
					// the Ansi class internally also checks system property
					// "org.fusesource.jansi.Ansi.disable"
					// but may also have been set with Ansi.setEnabled or a custom detector
					final Class<?> ansi = Class.forName("org.fusesource.jansi.Ansi");
					final Boolean enabled = (Boolean) ansi.getDeclaredMethod("isEnabled").invoke(null);
					if (!enabled) {
						return false;
					}
					// loading this class will load the native library
					// org.fusesource.jansi.internal.CLibrary
					final Class<?> ansiConsole = Class.forName("org.fusesource.jansi.AnsiConsole");
					final Field out = ansiConsole.getField("out");
					return out.get(null) == System.out;
				} catch (final Exception reflectionFailed) {
					return false;
				}
			}

			/**
			 * http://stackoverflow.com/questions/1403772/how-can-i-check-if-a-java-programs-input-output-streams-are-connected-to-a-term
			 */
			static boolean calcTTY() {
				try {
					final Object console = System.class.getDeclaredMethod("console").invoke(null);
					if (console == null) {
						return false;
					}
					try { // [#2083][#2084] Java 22 update
						final Method isTerminal = Class.forName("java.io.Console").getDeclaredMethod("isTerminal");
						return (Boolean) isTerminal.invoke(console);
					} catch (final NoSuchMethodException e) {
						return true;
					}
				} catch (final Throwable reflectionFailed) {
					return true;
				}
			}

			/** https://no-color.org/ */
			static final boolean forceDisabled() {
				return System.getenv("NO_COLOR") != null;
			}

			/** Jan Niklas Hasse's https://bixense.com/clicolors/ proposal */
			static final boolean forceEnabled() {
				return System.getenv("CLICOLOR_FORCE") != null && !"0".equals(System.getenv("CLICOLOR_FORCE"));
			}

			// null on Windows unless on Cygwin or MSYS
			static final boolean hasOsType() {
				return System.getenv("OSTYPE") != null;
			}

			// see Jan Niklas Hasse's https://bixense.com/clicolors/ proposal
			// https://conemu.github.io/en/AnsiEscapeCodes.html#Environment_variable
			static final boolean hintDisabled() {
				return "0".equals(System.getenv("CLICOLOR")) || "OFF".equals(System.getenv("ConEmuANSI"));
			}

			/**
			 * https://github.com/adoxa/ansicon/blob/master/readme.txt, Jan Niklas Hasse's
			 * https://bixense.com/clicolors/ proposal,
			 * https://conemu.github.io/en/AnsiEscapeCodes.html#Environment_variable
			 */
			static final boolean hintEnabled() {
				return System.getenv("ANSICON") != null || "1".equals(System.getenv("CLICOLOR"))
						|| "ON".equals(System.getenv("ConEmuANSI"));
			}

			static final boolean isCygwin() {
				return System.getenv("TERM") != null && System.getenv("TERM").toLowerCase(ENGLISH).contains("cygwin");
			}

			/**
			 * The first time this method is called, it invokes the
			 * {@link #calcIsJansiConsoleInstalled()} method, caches its result and returns
			 * this result; subsequently it returns the cached result.
			 */
			static boolean isJansiConsoleInstalled() {
				if (jansiInstalled == null) {
					jansiInstalled = calcIsJansiConsoleInstalled();
				}
				return jansiInstalled;
			}

			static final boolean isMac() {
				return System.getProperty("os.name").toLowerCase().contains("mac");
			}

			/** Cygwin and MSYS use pseudo-tty and console is always null... */
			static boolean isPseudoTTY() {
				return isWindows() && (isXterm() || isCygwin() || hasOsType());
			}

			static boolean isTTY() {
				if (tty == null) {
					tty = calcTTY();
				}
				return tty;
			}

			static final boolean isWindows() {
				return System.getProperty("os.name").toLowerCase().contains("win");
			}

			static final boolean isXterm() {
				return System.getenv("TERM") != null && System.getenv("TERM").startsWith("xterm");
			}

			/**
			 * Returns Ansi.ON if the specified {@code enabled} flag is true, Ansi.OFF
			 * otherwise.
			 * 
			 * @since 3.4
			 */
			public static Ansi valueOf(boolean enabled) {
				return enabled ? ON : OFF;
			}

			/** @deprecated use {@link ColorScheme#apply(String, List)} instead */
			@Deprecated
			public Text apply(String plainText, List<IStyle> styles) {
				return CommandLine.Help.defaultColorScheme(this).apply(plainText, styles);
			}

			/**
			 * Returns {@code true} if ANSI escape codes should be emitted, {@code false}
			 * otherwise.
			 * 
			 * @return ON: {@code true}, OFF: {@code false}, AUTO: if system property
			 *         {@code "picocli.ansi"} has value {@code "tty"} (case-insensitive),
			 *         then return {@code true} if either {@code System.console() != null}
			 *         or picocli guesses the application is running in a pseudo-terminal
			 *         pty on a Linux emulator in Windows. If system property
			 *         {@code "picocli.ansi"} has value {@code "true"} (case-sensitive) then
			 *         return {@code true}. Otherwise use picocli's <a href=
			 *         "https://picocli.info/#_heuristics_for_enabling_ansi">Heuristics for
			 *         Enabling ANSI</a> to determine whether the platform supports ANSI
			 *         escape codes.
			 */
			public boolean enabled() {
				if (this == ON) {
					return true;
				}
				if (this == OFF) {
					return false;
				}
				final String ansi = System.getProperty("picocli.ansi");
				final boolean auto = ansi == null || "AUTO".equalsIgnoreCase(ansi);
				final boolean tty = "TTY".equalsIgnoreCase(ansi) && (isTTY() || isPseudoTTY());
				return auto ? ansiPossible() : tty || Boolean.getBoolean("picocli.ansi");
			}

			/**
			 * Returns a String where any markup like
			 * {@code @|bg(red),white,underline some text|@} is converted to ANSI escape
			 * codes if this Ansi is ON, or suppressed if this Ansi is OFF.
			 * <p>
			 * Equivalent to {@code this.new Text(stringWithMarkup).toString()}.
			 * 
			 * @see ColorScheme#string(String)
			 * @since 3.4
			 */
			public String string(String stringWithMarkup) {
				return this.new Text(stringWithMarkup).toString();
			}

			/**
			 * Returns a new Text object for this Ansi mode, encapsulating the specified
			 * string which may contain markup like
			 * {@code @|bg(red),white,underline some text|@}.
			 * <p>
			 * Calling {@code toString()} on the returned Text will either include ANSI
			 * escape codes (if this Ansi mode is ON), or suppress ANSI escape codes (if
			 * this Ansi mode is OFF).
			 * <p>
			 * Equivalent to {@code this.new Text(stringWithMarkup)}.
			 * 
			 * @see ColorScheme#text(String)
			 * @since 3.4
			 */
			public Text text(String stringWithMarkup) {
				return this.new Text(stringWithMarkup);
			}
		}

		/**
		 * All usage help message are generated with a color scheme that assigns certain
		 * styles and colors to common parts of a usage message: the command name,
		 * options, positional parameters and option parameters. Users may customize
		 * these styles by creating Help with a custom color scheme.
		 * <p>
		 * Note that these options and styles may not be rendered if ANSI escape codes
		 * are not {@linkplain Ansi#enabled() enabled}.
		 * </p>
		 * <p>
		 * From 4.0, instances of this class are immutable.
		 * </p>
		 * 
		 * @see Builder
		 * @see Help#defaultColorScheme(Ansi)
		 */
		public static class ColorScheme {
			/**
			 * Builder class to create {@code ColorScheme} instances.
			 * 
			 * @since 4.0
			 */
			public static class Builder {
				private final List<IStyle> commandStyles = new ArrayList<IStyle>();
				private final List<IStyle> optionStyles = new ArrayList<IStyle>();
				private final List<IStyle> parameterStyles = new ArrayList<IStyle>();
				private final List<IStyle> optionParamStyles = new ArrayList<IStyle>();
				private final List<IStyle> errorStyles = new ArrayList<IStyle>();
				private final List<IStyle> stackTraceStyles = new ArrayList<IStyle>();
				private Ansi ansi = Ansi.AUTO;
				private Map<String, IStyle> markupMap;

				/** Constructs an empty color scheme builder with Ansi.AUTO. */
				public Builder() {
				}

				/** Constructs an empty color scheme builder with the specified Ansi value. */
				public Builder(Ansi ansi) {
					this.ansi = Assert.notNull(ansi, "ansi");
				}

				/**
				 * Constructs a color scheme builder with all attributes copied from the
				 * specified color scheme.
				 */
				public Builder(ColorScheme existing) {
					Assert.notNull(existing, "colorScheme");
					this.ansi = Assert.notNull(existing.ansi(), "ansi");
					this.commandStyles.addAll(existing.commandStyles());
					this.optionStyles.addAll(existing.optionStyles());
					this.parameterStyles.addAll(existing.parameterStyles());
					this.optionParamStyles.addAll(existing.optionParamStyles());
					this.errorStyles.addAll(existing.errorStyles());
					this.stackTraceStyles.addAll(existing.stackTraceStyles());
					if (existing.markupMap != null) {
						this.markupMap = new HashMap<String, IStyle>(existing.markupMap);
					}
				}

				private ColorScheme.Builder addAll(List<IStyle> styles, IStyle... add) {
					styles.addAll(Arrays.asList(add));
					return this;
				}

				/** Returns the {@code Ansi} setting of this color scheme builder. */
				public Ansi ansi() {
					return ansi;
				}

				/** Set the {@code Ansi} setting of this color scheme builder. */
				public ColorScheme.Builder ansi(Ansi ansi) {
					this.ansi = Assert.notNull(ansi, "ansi");
					return this;
				}

				/**
				 * Replaces colors and styles in this scheme builder with ones specified in
				 * system properties, and returns this builder. Supported property names:
				 * <ul>
				 * <li>{@code picocli.color.commands}</li>
				 * <li>{@code picocli.color.options}</li>
				 * <li>{@code picocli.color.parameters}</li>
				 * <li>{@code picocli.color.optionParams}</li>
				 * <li>{@code picocli.color.errors}</li>
				 * <li>{@code picocli.color.stackTraces}</li>
				 * </ul>
				 * <p>
				 * Property values can be anything that {@link Help.Ansi.Style#parse(String)}
				 * can handle.
				 * </p>
				 * 
				 * @return this ColorScheme builder
				 */
				public ColorScheme.Builder applySystemProperties() {
					replace(commandStyles, System.getProperty("picocli.color.commands"));
					replace(optionStyles, System.getProperty("picocli.color.options"));
					replace(parameterStyles, System.getProperty("picocli.color.parameters"));
					replace(optionParamStyles, System.getProperty("picocli.color.optionParams"));
					replace(errorStyles, System.getProperty("picocli.color.errors"));
					replace(stackTraceStyles, System.getProperty("picocli.color.stackTraces"));
					return this;
				}

				/**
				 * Creates and returns a new {@code ColorScheme} with the values configured on
				 * this builder.
				 */
				public ColorScheme build() {
					return new ColorScheme(this);
				}

				/**
				 * Adds the specified styles to the registered styles for commands in this color
				 * scheme builder and returns this builder.
				 * 
				 * @param styles the styles to add to the registered styles for commands in this
				 *               color scheme builder
				 * @return this color scheme builder to enable method chaining for a more fluent
				 *         API
				 */
				public ColorScheme.Builder commands(IStyle... styles) {
					return addAll(commandStyles, styles);
				}

				/** Returns the registered styles for commands in this color scheme builder. */
				public List<IStyle> commandStyles() {
					return commandStyles;
				}

				/**
				 * Returns the custom mapping from markup names (the names of the {@link Style}
				 * enum constants, like bold, italic, fg_blue, bg_green, etc) to {@link IStyle}
				 * objects in this color scheme. By default this returns {@code null}, unless a
				 * custom map was configured.
				 * 
				 * @since 4.2
				 */
				public Map<String, IStyle> customMarkupMap() {
					return markupMap;
				}

				/**
				 * Sets the custom mapping from markup names (the names of the {@link Style}
				 * enum constants, like bold, italic, fg_blue, bg_green, etc) to {@link IStyle}
				 * objects in this color scheme.
				 * 
				 * @return this color scheme builder to enable method chaining for a more fluent
				 *         API
				 * @since 4.2
				 */
				public ColorScheme.Builder customMarkupMap(Map<String, IStyle> newValue) {
					markupMap = newValue;
					return this;
				}

				/**
				 * Adds the specified styles to the registered styles for errors in this color
				 * scheme builder and returns this builder.
				 * 
				 * @param styles the styles to add to the registered styles for errors in this
				 *               color scheme builder
				 * @return this color scheme builder to enable method chaining for a more fluent
				 *         API
				 * @since 4.3
				 */
				public ColorScheme.Builder errors(IStyle... styles) {
					return addAll(errorStyles, styles);
				}

				/**
				 * Returns the registered styles for errors in this color scheme builder.
				 * 
				 * @since 4.3
				 */
				public List<IStyle> errorStyles() {
					return errorStyles;
				}

				/**
				 * Adds the specified styles to the registered styles for option parameters in
				 * this color scheme builder and returns this builder.
				 * 
				 * @param styles the styles to add to the registered styles for option
				 *               parameters in this color scheme builder
				 * @return this color scheme builder to enable method chaining for a more fluent
				 *         API
				 */
				public ColorScheme.Builder optionParams(IStyle... styles) {
					return addAll(optionParamStyles, styles);
				}

				/**
				 * Returns the registered styles for option parameters in this color scheme
				 * builder.
				 */
				public List<IStyle> optionParamStyles() {
					return optionParamStyles;
				}

				/**
				 * Adds the specified styles to the registered styles for options in this color
				 * scheme and returns this color scheme.
				 * 
				 * @param styles the styles to add to registered the styles for options in this
				 *               color scheme builder
				 * @return this color scheme builder to enable method chaining for a more fluent
				 *         API
				 */
				public ColorScheme.Builder options(IStyle... styles) {
					return addAll(optionStyles, styles);
				}

				/** Returns the registered styles for options in this color scheme builder. */
				public List<IStyle> optionStyles() {
					return optionStyles;
				}

				/**
				 * Adds the specified styles to the registered styles for positional parameters
				 * in this color scheme builder and returns this builder.
				 * 
				 * @param styles the styles to add to registered the styles for parameters in
				 *               this color scheme builder
				 * @return this color scheme builder to enable method chaining for a more fluent
				 *         API
				 */
				public ColorScheme.Builder parameters(IStyle... styles) {
					return addAll(parameterStyles, styles);
				}

				/**
				 * Returns the registered styles for positional parameters in this color scheme
				 * builder.
				 */
				public List<IStyle> parameterStyles() {
					return parameterStyles;
				}

				private void replace(List<IStyle> styles, String property) {
					if (property != null) {
						styles.clear();
						addAll(styles, Style.parse(property));
					}
				}

				/**
				 * Adds the specified styles to the registered styles for stack traces in this
				 * color scheme builder and returns this builder.
				 * 
				 * @param styles the styles to add to the registered styles for stack traces in
				 *               this color scheme builder
				 * @return this color scheme builder to enable method chaining for a more fluent
				 *         API
				 * @since 4.3
				 */
				public ColorScheme.Builder stackTraces(IStyle... styles) {
					return addAll(stackTraceStyles, styles);
				}

				/**
				 * Returns the registered styles for stack traces in this color scheme builder.
				 * 
				 * @since 4.3
				 */
				public List<IStyle> stackTraceStyles() {
					return stackTraceStyles;
				}
			}

			private static final IStyle EMPTY_STYLE = new IStyle() {
				@Override
				public String off() {
					return "";
				}

				@Override
				public String on() {
					return "";
				}
			};
			private final List<IStyle> commandStyles;
			private final List<IStyle> optionStyles;
			private final List<IStyle> parameterStyles;
			private final List<IStyle> optionParamStyles;
			private final List<IStyle> errorStyles;
			private final List<IStyle> stackTraceStyles;
			private final Ansi ansi;

			private final Map<String, IStyle> markupMap;

			/**
			 * Constructs a new empty ColorScheme with the specified Ansi enabled mode.
			 * 
			 * @see Help#defaultColorScheme(Ansi)
			 * @param builder contains the color scheme attributes to use
			 */
			ColorScheme(ColorScheme.Builder builder) {
				Assert.notNull(builder, "builder");
				this.ansi = Assert.notNull(builder.ansi(), "ansi");
				commandStyles = Collections.unmodifiableList(new ArrayList<IStyle>(builder.commandStyles()));
				optionStyles = Collections.unmodifiableList(new ArrayList<IStyle>(builder.optionStyles()));
				parameterStyles = Collections.unmodifiableList(new ArrayList<IStyle>(builder.parameterStyles()));
				optionParamStyles = Collections.unmodifiableList(new ArrayList<IStyle>(builder.optionParamStyles()));
				errorStyles = Collections.unmodifiableList(new ArrayList<IStyle>(builder.errorStyles()));
				stackTraceStyles = Collections.unmodifiableList(new ArrayList<IStyle>(builder.stackTraceStyles()));
				markupMap = builder.markupMap == null ? null
						: Collections.unmodifiableMap(new HashMap<String, IStyle>(builder.markupMap));
			}

			/** Returns the {@code Ansi} setting of this color scheme. */
			public Ansi ansi() {
				return ansi;
			}

			/**
			 * Returns a new Text object where all the specified styles are applied to the
			 * full length of the specified plain text.
			 * 
			 * @param plainText the string to apply all styles to. Must not contain markup!
			 * @param styles    the styles to apply to the full plain text
			 * @return a new Text object
			 * @since 4.2
			 */
			public Text apply(String plainText, List<IStyle> styles) {
				final Text result = ansi().new Text(plainText.length());
				result.colorScheme = this;
				if (plainText.length() == 0) {
					return result;
				}
				final IStyle[] all = styles.toArray(new IStyle[styles.size()]);
				result.sections.add(new Ansi.StyledSection(0, plainText.length(), Style.on(all),
						Style.off(reverseArray(all)) + resetStyle().off()));
				result.plain.append(plainText);
				result.length = result.plain.length();
				return result;
			}

			/**
			 * Returns the registered styles for commands in this color scheme.
			 * 
			 * @since 4.0
			 */
			public List<IStyle> commandStyles() {
				return commandStyles;
			}

			/**
			 * Returns a Text with all command styles applied to the specified command
			 * string.
			 * 
			 * @param command the command string to apply the registered command styles to
			 * @return a Text with all command styles applied to the specified command
			 *         string
			 */
			public Ansi.Text commandText(String command) {
				return apply(command, commandStyles);
			}

			/**
			 * Returns the custom mapping from markup names (the names of the {@link Style}
			 * enum constants, like bold, italic, fg_blue, bg_green, etc) to {@link IStyle}
			 * objects in this color scheme. By default this returns an empty map, unless a
			 * custom map was configured.
			 * 
			 * @since 4.2
			 */
			public Map<String, IStyle> customMarkupMap() {
				return markupMap == null ? Collections.<String, IStyle>emptyMap() : markupMap;
			}

			@Override
			public boolean equals(Object obj) {
				if (this == obj) {
					return true;
				}
				if (!(obj instanceof ColorScheme)) {
					return false;
				}
				final ColorScheme other = (ColorScheme) obj;
				return ansi.equals(other.ansi) && commandStyles.equals(other.commandStyles)
						&& optionStyles.equals(other.optionStyles) && parameterStyles.equals(other.parameterStyles)
						&& optionParamStyles.equals(other.optionParamStyles) && errorStyles.equals(other.errorStyles)
						&& stackTraceStyles.equals(other.stackTraceStyles) && markupMap == null
								? other.markupMap == null
								: markupMap.equals(other.markupMap);
			}

			/**
			 * Returns the registered styles for errors in this color scheme.
			 * 
			 * @since 4.3
			 */
			public List<IStyle> errorStyles() {
				return errorStyles;
			}

			/**
			 * Returns a Text with all error styles applied to the specified error string.
			 * 
			 * @param error the error string to apply the registered error styles to
			 * @return a Text with all error styles applied to the specified error string
			 * @since 4.3
			 */
			public Ansi.Text errorText(String error) {
				return apply(error, errorStyles);
			}

			@Override
			public int hashCode() {
				int result = 17;
				result = result * 37 + ansi.hashCode();
				result = result * 37 + commandStyles.hashCode();
				result = result * 37 + optionStyles.hashCode();
				result = result * 37 + parameterStyles.hashCode();
				result = result * 37 + optionParamStyles.hashCode();
				result = result * 37 + errorStyles.hashCode();
				result = result * 37 + stackTraceStyles.hashCode();
				result = result * 37 + (markupMap == null ? 0 : markupMap.hashCode());
				return result;
			}

			/**
			 * Returns the registered styles for option parameters in this color scheme.
			 * 
			 * @since 4.0
			 */
			public List<IStyle> optionParamStyles() {
				return optionParamStyles;
			}

			/**
			 * Returns a Text with all optionParam styles applied to the specified
			 * optionParam string.
			 * 
			 * @param optionParam the option parameter string to apply the registered option
			 *                    parameter styles to
			 * @return a Text with all option parameter styles applied to the specified
			 *         option parameter string
			 */
			public Ansi.Text optionParamText(String optionParam) {
				return apply(optionParam, optionParamStyles);
			}

			/**
			 * Returns the registered styles for options in this color scheme.
			 * 
			 * @since 4.0
			 */
			public List<IStyle> optionStyles() {
				return optionStyles;
			}

			/**
			 * Returns a Text with all option styles applied to the specified option string.
			 * 
			 * @param option the option string to apply the registered option styles to
			 * @return a Text with all option styles applied to the specified option string
			 */
			public Ansi.Text optionText(String option) {
				return apply(option, optionStyles);
			}

			/**
			 * Returns the registered styles for positional parameters in this color scheme.
			 * 
			 * @since 4.0
			 */
			public List<IStyle> parameterStyles() {
				return parameterStyles;
			}

			/**
			 * Returns a Text with all parameter styles applied to the specified parameter
			 * string.
			 * 
			 * @param parameter the parameter string to apply the registered parameter
			 *                  styles to
			 * @return a Text with all parameter styles applied to the specified parameter
			 *         string
			 */
			public Ansi.Text parameterText(String parameter) {
				return apply(parameter, parameterStyles);
			}

			/**
			 * Converts the specified markup styles to an array of {@link IStyle} objects.
			 * If no {@linkplain #customMarkupMap() custom markup mapping} is specified,
			 * this method delegates to {@link Style#parse(String)}, otherwise it returns
			 * the styles found in the custom mapping for the specified markup styles.
			 * 
			 * @param commaSeparatedCodes a string with a comma-separated list of markup
			 *                            styles (for example,
			 *                            {@code "bold,underline,bg_red"}
			 * @since 4.2
			 */
			public IStyle[] parse(String commaSeparatedCodes) {
				if (markupMap == null) {
					return Style.parse(commaSeparatedCodes);
				}
				final String[] codes = commaSeparatedCodes.split(",");
				final List<IStyle> styles = new ArrayList<IStyle>();
				for (String code : codes) {
					code = code.toLowerCase(ENGLISH).replace("(", "_").replace(")", "");
					final IStyle found = (markupMap.containsKey(code)) ? markupMap.get(code)
							: markupMap.get("fg_" + code);
					if (found != null) {
						styles.add(found);
					}
				}
				return styles.toArray(new IStyle[0]);
			}

			/**
			 * Returns the style that "resets" the style state to neutral.
			 * 
			 * @return {@link Style#reset} if no {@linkplain #customMarkupMap()} is defined,
			 *         otherwise either the style registered with the "reset" name or an
			 *         empty {@code IStyle} if no such style is registered.
			 * @since 4.2
			 */
			public IStyle resetStyle() {
				if (markupMap == null) {
					return Style.reset;
				}
				return markupMap.containsKey("reset") ? markupMap.get("reset") : EMPTY_STYLE;
			}

			/**
			 * Returns a String with the {@link #errorStyles() error styles} applied to the
			 * stack trace lines showing the throwable class name and error message
			 * (including "Caused by:..." lines), and the {@link #stackTraceStyles() stack
			 * trace styles} applied to the remaining stack trace of lines the specified
			 * Throwable.
			 * 
			 * @param t the Throwable whose stack trace string to apply the error and stack
			 *          trace styles to
			 * @return a String with error and stack trace styles applied to the stack trace
			 *         of the specified Throwable
			 * @since 4.5
			 */
			public String richStackTraceString(Throwable t) {
				return throwableToColorString(t, this);
			}

			/**
			 * Returns the registered styles for stack traces in this color scheme.
			 * 
			 * @since 4.3
			 */
			public List<IStyle> stackTraceStyles() {
				return stackTraceStyles;
			}

			/**
			 * Returns a Text with all stackTrace styles applied to all lines in the
			 * specified stackTrace string.
			 * 
			 * @param stackTrace the stack trace string to apply the registered stack trace
			 *                   styles to
			 * @return a Text with all stack trace styles applied to the specified stack
			 *         trace string
			 * @since 4.3
			 */
			public Ansi.Text stackTraceText(String stackTrace) {
				return apply(stackTrace, stackTraceStyles);
			}

			/**
			 * Returns a Text with all stackTrace styles applied to all lines in the stack
			 * trace of the specified Throwable.
			 * 
			 * @param t the Throwable whose stack trace string to apply the registered stack
			 *          trace styles to
			 * @return a Text with all stack trace styles applied to the stack trace of the
			 *         specified Throwable
			 * @since 4.5
			 */
			public Ansi.Text stackTraceText(Throwable t) {
				final StringWriter sw = new StringWriter();
				t.printStackTrace(new PrintWriter(sw, true));
				return stackTraceText(sw.toString());
			}

			/**
			 * Returns a String where any markup like
			 * {@code @|bg(red),white,underline some text|@} is
			 * {@linkplain ColorScheme#parse(String) converted} to the styles defined in
			 * this ColorScheme (if its Ansi mode is ON), or to the plain text without the
			 * markup (if this ColorScheme's Ansi mode is OFF).
			 * <p>
			 * Equivalent to
			 * {@code this.ansi().new Text(stringWithMarkup, this).toString()}.
			 * 
			 * @see Ansi#string(String)
			 * @since 4.2
			 */
			public String string(String stringWithMarkup) {
				return ansi().new Text(stringWithMarkup, this).toString();
			}

			/**
			 * Returns a new Text object for this ColorScheme, encapsulating the specified
			 * string which may contain markup like
			 * {@code @|bg(red),white,underline some text|@}.
			 * <p>
			 * Calling {@code toString()} on the returned Text will
			 * {@linkplain ColorScheme#parse(String) convert} the markup to the styles
			 * defined in this ColorScheme (if its Ansi mode is ON), or to the plain text
			 * without the markup (if this ColorScheme's Ansi mode is OFF).
			 * <p>
			 * Equivalent to {@code this.ansi().new Text(stringWithMarkup, this)}.
			 * 
			 * @see Ansi#text(String)
			 * @since 4.2
			 */
			public Text text(String stringWithMarkup) {
				return ansi().new Text(stringWithMarkup, this);
			}

			@Override
			public String toString() {
				return "ColorScheme[ansi=" + ansi + ", commands=" + commandStyles + ", optionStyles=" + optionStyles
						+ ", parameterStyles=" + parameterStyles + ", optionParamStyles=" + optionParamStyles
						+ ", errorStyles=" + errorStyles + ", stackTraceStyles=" + stackTraceStyles
						+ ", customMarkupMap=" + markupMap + "]";
			}
		}

		/**
		 * Columns define the width, indent (leading number of spaces in a column before
		 * the value) and {@linkplain Overflow Overflow} policy of a column in a
		 * {@linkplain TextTable TextTable}.
		 */
		public static class Column {

			/**
			 * Policy for handling text that is longer than the column width: span multiple
			 * columns, wrap to the next row, or simply truncate the portion that doesn't
			 * fit.
			 */
			public enum Overflow {
				TRUNCATE, SPAN, WRAP
			}

			/** Column width in characters */
			public final int width;

			/**
			 * Indent (number of empty spaces at the start of the column preceding the text
			 * value)
			 */
			public int indent;

			/** Policy that determines how to handle values larger than the column width. */
			public final Overflow overflow;

			public Column(int width, int indent, Overflow overflow) {
				this.width = width;
				this.indent = indent;
				this.overflow = Assert.notNull(overflow, "overflow");
			}

			@Override
			public boolean equals(Object obj) {
				return obj instanceof Column && ((Column) obj).width == width && ((Column) obj).indent == indent
						&& ((Column) obj).overflow == overflow;
			}

			@Override
			public int hashCode() {
				return 17 * width + 37 * indent + 37 * overflow.hashCode();
			}

			@Override
			public String toString() {
				return String.format("Column[width=%d, indent=%d, overflow=%s]", width, indent, overflow);
			}
		}

		/**
		 * The DefaultOptionRenderer converts {@link OptionSpec Options} to five columns
		 * of text to match the default {@linkplain TextTable TextTable} column layout.
		 * The first row of values looks like this:
		 * <ol>
		 * <li>the required option marker (if the option is required)</li>
		 * <li>2-character short option name (or empty string if no short option
		 * exists)</li>
		 * <li>comma separator (only if both short option and long option exist, empty
		 * string otherwise)</li>
		 * <li>comma-separated string with long option name(s)</li>
		 * <li>first element of the {@link OptionSpec#description()} array</li>
		 * </ol>
		 * <p>
		 * Following this, there will be one row for each of the remaining elements of
		 * the {@link OptionSpec#description()} array, and these rows look like
		 * {@code {"", "", "", option.description()[i]}}.
		 * </p>
		 */
		static class DefaultOptionRenderer implements IOptionRenderer {
			private final String requiredMarker;
			private final boolean showDefaultValues;
			private String sep;

			public DefaultOptionRenderer(boolean showDefaultValues, String requiredMarker) {
				this.showDefaultValues = showDefaultValues;
				this.requiredMarker = Assert.notNull(requiredMarker, "requiredMarker");
			}

			private Text createLongOptionText(OptionSpec option, IParamLabelRenderer renderer, ColorScheme scheme,
					String longOption) {
				Text paramLabelText = renderer.renderParameterLabel(option, scheme.ansi(), scheme.optionParamStyles);

				// if no long option, fill in the space between the short option name and the
				// param label value
				if (paramLabelText.length > 0 && longOption.length() == 0) {
					sep = renderer.separator();
					// #181 paramLabelText may be =LABEL or [=LABEL...]
					final int sepStart = paramLabelText.plainString().indexOf(sep);
					final Text prefix = paramLabelText.substring(0, sepStart);
					paramLabelText = prefix.concat(paramLabelText.substring(sepStart + sep.length()));
				}
				Text longOptionText = scheme.optionText(longOption);
				longOptionText = longOptionText.concat(paramLabelText);
				return longOptionText;
			}

			@Override
			public Text[][] render(OptionSpec option, IParamLabelRenderer paramLabelRenderer, ColorScheme scheme) {
				final String[] names = ShortestFirst.sort(option.names());
				final int shortOptionCount = names[0].length() == 2 ? 1 : 0;
				String shortOption = shortOptionCount > 0 ? names[0] : "";
				sep = shortOptionCount > 0 && names.length > 1 ? "," : "";

				if (option.negatable()) {
					final INegatableOptionTransformer transformer = option.commandSpec.negatableOptionTransformer();
					if (shortOptionCount > 0) {
						shortOption = transformer.makeSynopsis(shortOption, option.commandSpec);
					}
					for (int i = 0; i < names.length; i++) {
						names[i] = transformer.makeSynopsis(names[i], option.commandSpec);
					}
				}

				final String longOption = join(names, shortOptionCount, names.length - shortOptionCount, ", ");
				final Text longOptionText = createLongOptionText(option, paramLabelRenderer, scheme, longOption);

				final String requiredOption = !option.originallyRequired() && option.required() ? requiredMarker : "";
				return renderDescriptionLines(option, scheme, requiredOption, shortOption, longOptionText);
			}

			private Text[][] renderDescriptionLines(OptionSpec option, ColorScheme scheme, String requiredOption,
					String shortOption, Text longOptionText) {
				final Text EMPTY = Ansi.EMPTY_TEXT;
				final boolean[] showDefault = { option.internalShowDefaultValue(showDefaultValues) };
				final List<Text[]> result = new ArrayList<Text[]>();
				final String[] description = option.description();
				final Text[] descriptionFirstLines = createDescriptionFirstLines(scheme, option, description,
						showDefault);
				result.add(new Text[] { scheme.optionText(requiredOption), scheme.optionText(shortOption),
						scheme.ansi().new Text(sep, scheme), longOptionText, descriptionFirstLines[0] });
				for (int i = 1; i < descriptionFirstLines.length; i++) {
					result.add(new Text[] { EMPTY, EMPTY, EMPTY, EMPTY, descriptionFirstLines[i] });
				}
				for (int i = 1; i < description.length; i++) {
					final Text[] descriptionNextLines = scheme.ansi().new Text(description[i], scheme).splitLines();
					for (final Text line : descriptionNextLines) {
						result.add(new Text[] { EMPTY, EMPTY, EMPTY, EMPTY, line });
					}
				}
				if (showDefault[0]) {
					addTrailingDefaultLine(result, option, scheme);
				}
				return result.toArray(new Text[result.size()][]);
			}
		}

		/**
		 * The DefaultParameterRenderer converts {@linkplain PositionalParamSpec
		 * positional parameters} to five columns of text to match the default
		 * {@linkplain TextTable TextTable} column layout. The first row of values looks
		 * like this:
		 * <ol>
		 * <li>the required option marker (if the parameter's arity is to have at least
		 * one value)</li>
		 * <li>empty string</li>
		 * <li>empty string</li>
		 * <li>parameter(s) label as rendered by the {@link IParamLabelRenderer}</li>
		 * <li>first element of the {@link PositionalParamSpec#description()} array</li>
		 * </ol>
		 * <p>
		 * Following this, there will be one row for each of the remaining elements of
		 * the {@link PositionalParamSpec#description()} array, and these rows look like
		 * {@code {"", "", "", param.description()[i]}}.
		 * </p>
		 */
		static class DefaultParameterRenderer implements IParameterRenderer {
			private final String requiredMarker;
			private final boolean showDefaultValues;

			public DefaultParameterRenderer(boolean showDefaultValues, String requiredMarker) {
				this.showDefaultValues = showDefaultValues;
				this.requiredMarker = Assert.notNull(requiredMarker, "requiredMarker");
			}

			@Override
			public Text[][] render(PositionalParamSpec param, IParamLabelRenderer paramLabelRenderer,
					ColorScheme scheme) {
				final Text label = paramLabelRenderer.renderParameterLabel(param, scheme.ansi(),
						scheme.parameterStyles);
				final Text requiredParameter = scheme.parameterText(param.arity().min > 0 ? requiredMarker : "");

				final Text EMPTY = Ansi.EMPTY_TEXT;
				final boolean[] showDefault = { param.internalShowDefaultValue(showDefaultValues) };
				final List<Text[]> result = new ArrayList<Text[]>();
				final String[] description = param.description();
				final Text[] descriptionFirstLines = createDescriptionFirstLines(scheme, param, description,
						showDefault);
				result.add(new Text[] { requiredParameter, EMPTY, EMPTY, label, descriptionFirstLines[0] });
				for (int i = 1; i < descriptionFirstLines.length; i++) {
					result.add(new Text[] { EMPTY, EMPTY, EMPTY, EMPTY, descriptionFirstLines[i] });
				}
				for (int i = 1; i < description.length; i++) {
					final Text[] descriptionNextLines = scheme.ansi().new Text(description[i], scheme).splitLines();
					for (final Text line : descriptionNextLines) {
						result.add(new Text[] { EMPTY, EMPTY, EMPTY, EMPTY, line });
					}
				}
				if (showDefault[0]) {
					addTrailingDefaultLine(result, param, scheme);
				}
				return result.toArray(new Text[result.size()][]);
			}
		}

		/**
		 * DefaultParamLabelRenderer separates option parameters from their
		 * {@linkplain OptionSpec option names} with a
		 * {@linkplain CommandLine.Model.ParserSpec#separator() separator} string, and,
		 * unless {@link ArgSpec#hideParamSyntax()} is true, surrounds optional values
		 * with {@code '['} and {@code ']'} characters and uses ellipses ("...") to
		 * indicate that any number of values is allowed for options or parameters with
		 * variable arity.
		 */
		static class DefaultParamLabelRenderer implements IParamLabelRenderer {
			private final CommandSpec commandSpec;

			/**
			 * Constructs a new DefaultParamLabelRenderer with the specified separator
			 * string.
			 */
			public DefaultParamLabelRenderer(CommandSpec commandSpec) {
				this.commandSpec = Assert.notNull(commandSpec, "commandSpec");
			}

			@Override
			public Text renderParameterLabel(ArgSpec argSpec, Ansi ansi, List<IStyle> styles) {
				final Range capacity = argSpec.isOption() ? argSpec.arity()
						: ((PositionalParamSpec) argSpec).capacity();
				final ColorScheme colorScheme = commandSpec.commandLine() == null ? Help.defaultColorScheme(ansi)
						: commandSpec.commandLine().getColorScheme();
				if (capacity.max == 0) {
					return ansi.new Text("", colorScheme);
				}
				if (argSpec.hideParamSyntax()) {
					return colorScheme.apply((argSpec.isOption() ? separator() : "") + argSpec.paramLabel(), styles);
				}

				String split = argSpec.splitRegex();
				if (!empty(split)) { // #1044 only check splitSynopsisLabel if we have a split regex
					split = empty(argSpec.splitRegexSynopsisLabel()) ? split : argSpec.splitRegexSynopsisLabel();
				}
				String mandatorySep = empty(split) ? " " : split;
				String optionalSep = empty(split) ? " [" : "[" + split;

				final boolean unlimitedSplit = !empty(split) && !commandSpec.parser().limitSplit();
				final boolean limitedSplit = !empty(split) && commandSpec.parser().limitSplit();
				final Text paramName = colorScheme.apply(argSpec.paramLabel(), styles);
				Text repeating = paramName;
				int paramCount = 1;
				if (unlimitedSplit) {
					repeating = paramName.concat("[" + split).concat(paramName).concat("...]");
					paramCount++;
					mandatorySep = " ";
					optionalSep = " [";
				}
				Text result = repeating;

				int done = 1;
				for (; done < capacity.min; done++) {
					result = result.concat(mandatorySep).concat(repeating); // " PARAM" or ",PARAM"
					paramCount += paramCount;
				}
				if (!capacity.isVariable) {
					for (int i = done; i < capacity.max; i++) {
						result = result.concat(optionalSep).concat(paramName); // " [PARAM" or "[,PARAM"
						paramCount++;
					}
					for (int i = done; i < capacity.max; i++) {
						result = result.concat("]");
					}
				}
				// show an extra trailing "[,PARAM]" if split and either max=* or splitting is
				// not restricted to max
				final boolean effectivelyVariable = capacity.isVariable || (limitedSplit && paramCount == 1);
				if (limitedSplit && effectivelyVariable && paramCount == 1) {
					result = result.concat(optionalSep).concat(repeating).concat("]"); // PARAM[,PARAM]...
				}
				if (effectivelyVariable) {
					if (!argSpec.arity().isVariable && argSpec.arity().min > 1) {
						result = ansi.new Text("(", colorScheme).concat(result).concat(")"); // repeating group
					}
					result = result.concat("..."); // PARAM...
				}
				final String optionSeparator = argSpec.isOption() ? separator() : "";
				if (capacity.min == 0) { // optional
					final String sep2 = empty(optionSeparator.trim()) ? optionSeparator + "[" : "[" + optionSeparator;
					result = ansi.new Text(sep2, colorScheme).concat(result).concat("]");
				} else {
					result = ansi.new Text(optionSeparator, colorScheme).concat(result);
				}
				return result;
			}

			@Override
			public String separator() {
				return commandSpec.parser().separator();
			}
		}

		/**
		 * When customizing online help for {@link OptionSpec Option} details, a custom
		 * {@code IOptionRenderer} can be used to create textual representation of an
		 * Option in a tabular format: one or more rows, each containing one or more
		 * columns. The {@link Layout Layout} is responsible for placing these text
		 * values in the {@link TextTable TextTable}.
		 */
		public interface IOptionRenderer {
			/**
			 * Returns a text representation of the specified option and its parameter(s) if
			 * any.
			 * 
			 * @param option                 the command line option to show online usage
			 *                               help for
			 * @param parameterLabelRenderer responsible for rendering option parameters to
			 *                               text
			 * @param scheme                 color scheme for applying ansi color styles to
			 *                               options and option parameters
			 * @return a 2-dimensional array of text values: one or more rows, each
			 *         containing one or more columns
			 * @since 3.0
			 */
			Text[][] render(OptionSpec option, IParamLabelRenderer parameterLabelRenderer, ColorScheme scheme);
		}

		/**
		 * When customizing online help for {@linkplain PositionalParamSpec positional
		 * parameters} details, a custom {@code IParameterRenderer} can be used to
		 * create textual representation of a Parameters field in a tabular format: one
		 * or more rows, each containing one or more columns. The {@link Layout Layout}
		 * is responsible for placing these text values in the {@link TextTable
		 * TextTable}.
		 */
		public interface IParameterRenderer {
			/**
			 * Returns a text representation of the specified positional parameter.
			 * 
			 * @param param                  the positional parameter to show online usage
			 *                               help for
			 * @param parameterLabelRenderer responsible for rendering parameter labels to
			 *                               text
			 * @param scheme                 color scheme for applying ansi color styles to
			 *                               positional parameters
			 * @return a 2-dimensional array of text values: one or more rows, each
			 *         containing one or more columns
			 * @since 3.0
			 */
			Text[][] render(PositionalParamSpec param, IParamLabelRenderer parameterLabelRenderer, ColorScheme scheme);
		}

		/**
		 * When customizing online usage help for an option parameter or a positional
		 * parameter, a custom {@code IParamLabelRenderer} can be used to render the
		 * parameter name or label to a String.
		 */
		public interface IParamLabelRenderer {

			/**
			 * Returns a text rendering of the option parameter or positional parameter;
			 * returns an empty string {@code ""} if the option is a boolean and does not
			 * take a parameter.
			 * 
			 * @param argSpec the named or positional parameter with a parameter label
			 * @param ansi    determines whether ANSI escape codes should be emitted or not
			 * @param styles  the styles to apply to the parameter label
			 * @return a text rendering of the Option parameter or positional parameter
			 * @since 3.0
			 */
			Text renderParameterLabel(ArgSpec argSpec, Ansi ansi, List<IStyle> styles);

			/**
			 * Returns the separator between option name and param label.
			 * 
			 * @return the separator between option name and param label
			 */
			String separator();
		}

		/**
		 * Use a Layout to format usage help text for options and parameters in tabular
		 * format.
		 * <p>
		 * Delegates to the renderers to create {@link Text} values for the annotated
		 * fields, and uses a {@link TextTable} to display these values in tabular
		 * format. Layout is responsible for deciding which values to display where in
		 * the table. By default, Layout shows one option or parameter per table row.
		 * </p>
		 * <p>
		 * Customize by overriding the
		 * {@link #layout(CommandLine.Model.ArgSpec, CommandLine.Help.Ansi.Text[][])}
		 * method.
		 * </p>
		 * 
		 * @see IOptionRenderer rendering options to text
		 * @see IParameterRenderer rendering parameters to text
		 * @see TextTable showing values in a tabular format
		 */
		public static class Layout {
			protected final ColorScheme colorScheme;
			protected final TextTable table;
			protected IOptionRenderer optionRenderer;
			protected IParameterRenderer parameterRenderer;

			/**
			 * Constructs a Layout with the specified color scheme, a new default TextTable,
			 * the {@linkplain Help#createDefaultOptionRenderer() default option renderer},
			 * and the {@linkplain Help#createDefaultParameterRenderer() default parameter
			 * renderer}.
			 * 
			 * @param colorScheme the color scheme to use for common, auto-generated parts
			 *                    of the usage help message
			 */
			public Layout(ColorScheme colorScheme, int tableWidth) {
				this(colorScheme, TextTable.forDefaultColumns(colorScheme,
						UsageMessageSpec.DEFAULT_USAGE_LONG_OPTIONS_WIDTH + 4, tableWidth));
			}

			/**
			 * Constructs a Layout with the specified color scheme, the specified TextTable,
			 * the {@linkplain Help#createDefaultOptionRenderer() default option renderer},
			 * and the {@linkplain Help#createDefaultParameterRenderer() default parameter
			 * renderer}.
			 * 
			 * @param colorScheme the color scheme to use for common, auto-generated parts
			 *                    of the usage help message
			 * @param textTable   the TextTable to lay out parts of the usage help message
			 *                    in tabular format
			 */
			public Layout(ColorScheme colorScheme, TextTable textTable) {
				this(colorScheme, textTable, new DefaultOptionRenderer(false, " "),
						new DefaultParameterRenderer(false, " "));
			}

			/**
			 * Constructs a Layout with the specified color scheme, the specified TextTable,
			 * the specified option renderer and the specified parameter renderer.
			 * 
			 * @param colorScheme       the color scheme to use for common, auto-generated
			 *                          parts of the usage help message
			 * @param optionRenderer    the object responsible for rendering Options to Text
			 * @param parameterRenderer the object responsible for rendering Parameters to
			 *                          Text
			 * @param textTable         the TextTable to lay out parts of the usage help
			 *                          message in tabular format
			 */
			public Layout(ColorScheme colorScheme, TextTable textTable, IOptionRenderer optionRenderer,
					IParameterRenderer parameterRenderer) {
				this.colorScheme = Assert.notNull(colorScheme, "colorScheme");
				this.table = Assert.notNull(textTable, "textTable");
				this.optionRenderer = Assert.notNull(optionRenderer, "optionRenderer");
				this.parameterRenderer = Assert.notNull(parameterRenderer, "parameterRenderer");
			}

			/**
			 * Calls
			 * {@link #addOption(CommandLine.Model.OptionSpec, CommandLine.Help.IParamLabelRenderer)}
			 * for all Options in the specified list.
			 * 
			 * @param options            options to add usage descriptions for; it is the
			 *                           responsibility of the caller to exclude options
			 *                           that should not be shown
			 * @param paramLabelRenderer object that knows how to render option parameters
			 * @since 4.4
			 */
			public void addAllOptions(List<OptionSpec> options, IParamLabelRenderer paramLabelRenderer) {
				for (final OptionSpec option : options) {
					addOption(option, paramLabelRenderer);
				}
			}

			/**
			 * Calls
			 * {@link #addPositionalParameter(CommandLine.Model.PositionalParamSpec, CommandLine.Help.IParamLabelRenderer)}
			 * for all positional parameters in the specified list.
			 * 
			 * @param params             positional parameters to add usage descriptions
			 *                           for; it is the responsibility of the caller to
			 *                           exclude positional parameters that should not be
			 *                           shown
			 * @param paramLabelRenderer knows how to render option parameters
			 * @since 4.4
			 */
			public void addAllPositionalParameters(List<PositionalParamSpec> params,
					IParamLabelRenderer paramLabelRenderer) {
				for (final PositionalParamSpec param : params) {
					addPositionalParameter(param, paramLabelRenderer);
				}
			}

			/**
			 * Delegates to the {@link #optionRenderer option renderer} of this layout to
			 * obtain text values for the specified {@link OptionSpec}, and then calls the
			 * {@link #layout(CommandLine.Model.ArgSpec, CommandLine.Help.Ansi.Text[][])}
			 * method to write these text values into the correct cells in the TextTable.
			 * 
			 * @param option             the option argument
			 * @param paramLabelRenderer knows how to render option parameters
			 * @since 3.0
			 */
			public void addOption(OptionSpec option, IParamLabelRenderer paramLabelRenderer) {
				final Text[][] values = optionRenderer.render(option, paramLabelRenderer, colorScheme);
				layout(option, values);
			}

			/**
			 * Calls
			 * {@link #addOption(CommandLine.Model.OptionSpec, CommandLine.Help.IParamLabelRenderer)}
			 * for all non-hidden Options in the list.
			 * 
			 * @param options            options to add usage descriptions for
			 * @param paramLabelRenderer object that knows how to render option parameters
			 * @since 3.0
			 */
			public void addOptions(List<OptionSpec> options, IParamLabelRenderer paramLabelRenderer) {
				for (final OptionSpec option : options) {
					if (!option.hidden()) {
						addOption(option, paramLabelRenderer);
					}
				}
			}

			/**
			 * Delegates to the {@link #parameterRenderer parameter renderer} of this layout
			 * to obtain text values for the specified {@linkplain PositionalParamSpec
			 * positional parameter}, and then calls
			 * {@link #layout(CommandLine.Model.ArgSpec, CommandLine.Help.Ansi.Text[][])} to
			 * write these text values into the correct cells in the TextTable.
			 * 
			 * @param param              the positional parameter
			 * @param paramLabelRenderer knows how to render option parameters
			 * @since 3.0
			 */
			public void addPositionalParameter(PositionalParamSpec param, IParamLabelRenderer paramLabelRenderer) {
				final Text[][] values = parameterRenderer.render(param, paramLabelRenderer, colorScheme);
				layout(param, values);
			}

			/**
			 * Calls
			 * {@link #addPositionalParameter(CommandLine.Model.PositionalParamSpec, CommandLine.Help.IParamLabelRenderer)}
			 * for all non-hidden Parameters in the list.
			 * 
			 * @param params             positional parameters to add usage descriptions for
			 * @param paramLabelRenderer knows how to render option parameters
			 * @since 3.0
			 */
			public void addPositionalParameters(List<PositionalParamSpec> params,
					IParamLabelRenderer paramLabelRenderer) {
				for (final PositionalParamSpec param : params) {
					if (!param.hidden()) {
						addPositionalParameter(param, paramLabelRenderer);
					}
				}
			}

			/**
			 * Returns the ColorScheme used to create Text objects in this layout.
			 * 
			 * @since 4.6
			 */
			public ColorScheme colorScheme() {
				return colorScheme;
			}

			/**
			 * Copies the specified text values into the correct cells in the
			 * {@link TextTable}. This implementation delegates to
			 * {@link TextTable#addRowValues(CommandLine.Help.Ansi.Text...)} for each row of
			 * values.
			 * <p>
			 * Subclasses may override.
			 * </p>
			 * 
			 * @param argSpec    the Option or Parameters
			 * @param cellValues the text values representing the Option/Parameters, to be
			 *                   displayed in tabular form
			 * @since 3.0
			 */
			public void layout(ArgSpec argSpec, Text[][] cellValues) {
				for (final Text[] oneRow : cellValues) {
					table.addRowValues(oneRow);
				}
			}

			/**
			 * Returns the IOptionRenderer used to render options to Text before adding this
			 * text to the TextTable in this layout.
			 * 
			 * @since 4.6
			 */
			public IOptionRenderer optionRenderer() {
				return optionRenderer;
			}

			/**
			 * Returns the IParameterRenderer used to render positional params to Text
			 * before adding this text to the TextTable in this layout.
			 * 
			 * @since 4.6
			 */
			public IParameterRenderer parameterRenderer() {
				return parameterRenderer;
			}

			/**
			 * Returns the TextTable used in this layout.
			 * 
			 * @since 4.6
			 */
			public TextTable textTable() {
				return table;
			}

			/**
			 * Returns the section of the usage help message accumulated in the TextTable
			 * owned by this layout.
			 */
			@Override
			public String toString() {
				return table.toString();
			}
		}

		/**
		 * The MinimalOptionRenderer converts {@link OptionSpec Options} to a single row
		 * with two columns of text: an option name and a description. If multiple names
		 * or description lines exist, the first value is used.
		 */
		static class MinimalOptionRenderer implements IOptionRenderer {
			@Override
			public Text[][] render(OptionSpec option, IParamLabelRenderer parameterLabelRenderer, ColorScheme scheme) {
				Text optionText = option.negatable()
						? scheme.optionText(option.commandSpec.negatableOptionTransformer()
								.makeSynopsis(option.names()[0], option.commandSpec))
						: scheme.optionText(option.names()[0]);
				final Text paramLabelText = parameterLabelRenderer.renderParameterLabel(option, scheme.ansi(),
						scheme.optionParamStyles);
				optionText = optionText.concat(paramLabelText);
				return new Text[][] { { optionText, scheme.ansi().new Text(
						option.description().length == 0 ? "" : option.description()[0], scheme) } };
			}
		}

		/**
		 * The MinimalParameterRenderer converts {@linkplain PositionalParamSpec
		 * positional parameters} to a single row with two columns of text: the
		 * parameters label and a description. If multiple description lines exist, the
		 * first value is used.
		 */
		static class MinimalParameterRenderer implements IParameterRenderer {
			@Override
			public Text[][] render(PositionalParamSpec param, IParamLabelRenderer parameterLabelRenderer,
					ColorScheme scheme) {
				return new Text[][] {
						{ parameterLabelRenderer.renderParameterLabel(param, scheme.ansi(), scheme.parameterStyles),
								scheme.ansi().new Text(param.description().length == 0 ? "" : param.description()[0],
										scheme) } };
			}
		}

		/** Sorts short strings before longer strings. */
		static class ShortestFirst implements Comparator<String> {
			/** Sorts the specified array of Strings longest-first and returns it. */
			public static String[] longestFirst(String[] names) {
				Arrays.sort(names, Collections.reverseOrder(new ShortestFirst()));
				return names;
			}

			/** Sorts the specified array of Strings shortest-first and returns it. */
			public static String[] sort(String[] names) {
				Arrays.sort(names, new ShortestFirst());
				return names;
			}

			@Override
			public int compare(String o1, String o2) {
				return o1.length() - o2.length();
			}
		}

		/**
		 * Sorts {@code OptionSpec} instances by their max arity first, then their min
		 * arity, then delegates to super class.
		 */
		static class SortByOptionArityAndNameAlphabetically extends SortByShortestOptionNameAlphabetically {
			@Override
			public int compare(OptionSpec o1, OptionSpec o2) {
				final Range arity1 = o1.arity();
				final Range arity2 = o2.arity();
				int result = arity1.max - arity2.max;
				if (result == 0) {
					result = arity1.min - arity2.min;
				}
				if (result == 0) { // arity is same
					if (o1.isMultiValue() && !o2.isMultiValue()) {
						result = 1;
					} // f1 > f2
					if (!o1.isMultiValue() && o2.isMultiValue()) {
						result = -1;
					} // f1 < f2
				}
				return result == 0 ? super.compare(o1, o2) : result;
			}
		}

		static class SortByOrder<T extends IOrdered> implements Comparator<T> {
			@Override
			public int compare(T o1, T o2) {
				return Integer.signum(o1.order() - o2.order());
			}
		}

		/**
		 * Sorts {@code OptionSpec} instances by their name in case-insensitive
		 * alphabetic order. If an option has multiple names, the shortest name is used
		 * for the sorting. Help options follow non-help options.
		 */
		static class SortByShortestOptionNameAlphabetically implements Comparator<OptionSpec> {
			@Override
			public int compare(OptionSpec o1, OptionSpec o2) {
				if (o1 == null) {
					return 1;
				} else if (o2 == null) {
					return -1;
				} // options before params
				final String[] names1 = ShortestFirst.sort(o1.names());
				final String[] names2 = ShortestFirst.sort(o2.names());
				final String s1 = CommandSpec.stripPrefix(names1[0]);
				final String s2 = CommandSpec.stripPrefix(names2[0]);
				int result = s1.toUpperCase().compareTo(s2.toUpperCase()); // case insensitive sort
				result = result == 0 ? -s1.compareTo(s2) : result; // lower case before upper case
				return o1.help() == o2.help() ? result : o2.help() ? -1 : 1; // help options come last
			}
		}

		/**
		 * <p>
		 * Responsible for spacing out {@link Text} values according to the
		 * {@link Column} definitions the table was created with. Columns have a width,
		 * indentation, and an overflow policy that decides what to do if a value is
		 * longer than the column's width.
		 * </p>
		 */
		public static class TextTable {
			/**
			 * Helper class to index positions in a {@code Help.TextTable}.
			 * 
			 * @since 2.0
			 */
			public static class Cell {
				/** Table column index (zero based). */
				public final int column;
				/** Table row index (zero based). */
				public final int row;

				/**
				 * Constructs a new Cell with the specified coordinates in the table.
				 * 
				 * @param column the zero-based table column
				 * @param row    the zero-based table row
				 */
				public Cell(int column, int row) {
					this.column = column;
					this.row = row;
				}
			}

			static class Count {
				int charCount;
				int columnCount;
			}

			private static final int OPTION_SEPARATOR_COLUMN = 2;

			private static final int LONG_OPTION_COLUMN = 3;

			/**
			 * Constructs a {@code TextTable} with the specified columns.
			 * 
			 * @param ansi    whether to emit ANSI escape codes or not
			 * @param columns columns to construct this TextTable with
			 * @deprecated use
			 *             {@link #forColumns(CommandLine.Help.ColorScheme, CommandLine.Help.Column...)}
			 *             instead
			 */
			@Deprecated
			public static TextTable forColumns(Ansi ansi, Column... columns) {
				return new TextTable(ansi, columns);
			}

			/**
			 * Constructs a {@code TextTable} with the specified columns.
			 * 
			 * @param colorScheme the styles and ANSI mode to use for embedded markup
			 * @param columns     columns to construct this TextTable with
			 * @since 4.2
			 */
			public static TextTable forColumns(ColorScheme colorScheme, Column... columns) {
				return new TextTable(colorScheme, columns);
			}

			/**
			 * Constructs a new TextTable with columns with the specified width, all
			 * SPANning multiple columns on overflow except the last column which WRAPS to
			 * the next row.
			 * 
			 * @param ansi         whether to emit ANSI escape codes or not
			 * @param columnWidths the width of each table column (all columns have zero
			 *                     indent)
			 * @deprecated use
			 *             {@link #forColumns(CommandLine.Help.ColorScheme, CommandLine.Help.Column...)}
			 *             instead
			 */
			@Deprecated
			public static TextTable forColumnWidths(Ansi ansi, int... columnWidths) {
				return forColumnWidths(Help.defaultColorScheme(ansi), columnWidths);
			}

			/**
			 * Constructs a new TextTable with columns with the specified width, all
			 * SPANning multiple columns on overflow except the last column which WRAPS to
			 * the next row.
			 * 
			 * @param colorScheme  the styles and ANSI mode to use for embedded markup
			 * @param columnWidths the width of each table column (all columns have zero
			 *                     indent)
			 * @since 4.2
			 */
			public static TextTable forColumnWidths(ColorScheme colorScheme, int... columnWidths) {
				final Column[] columns = new Column[columnWidths.length];
				for (int i = 0; i < columnWidths.length; i++) {
					columns[i] = new Column(columnWidths[i], 0, i == columnWidths.length - 1 ? WRAP : SPAN);
				}
				return new TextTable(colorScheme, columns);
			}

			/**
			 * Constructs a TextTable with five columns as follows:
			 * <ol>
			 * <li>required option/parameter marker (width: 2, indent: 0, TRUNCATE on
			 * overflow)</li>
			 * <li>short option name (width: 2, indent: 0, TRUNCATE on overflow)</li>
			 * <li>comma separator (width: 1, indent: 0, TRUNCATE on overflow)</li>
			 * <li>long option name(s) (width: 24, indent: 1, SPAN multiple columns on
			 * overflow)</li>
			 * <li>description line(s) (width: 51, indent: 1, WRAP to next row on
			 * overflow)</li>
			 * </ol>
			 * 
			 * @param ansi           whether to emit ANSI escape codes or not
			 * @param usageHelpWidth the total width of the columns combined
			 * @deprecated use {@link #forDefaultColumns(CommandLine.Help.Ansi, int, int)}
			 *             instead
			 */
			@Deprecated
			public static TextTable forDefaultColumns(Ansi ansi, int usageHelpWidth) {
				// TODO split out the 1 (for long column indent) and 3 (should be description
				// indent)
				return forDefaultColumns(Help.defaultColorScheme(ansi),
						UsageMessageSpec.DEFAULT_USAGE_LONG_OPTIONS_WIDTH + 4, usageHelpWidth);
			}

			/**
			 * Constructs a TextTable with five columns as follows:
			 * <ol>
			 * <li>required option/parameter marker (width: 2, indent: 0, TRUNCATE on
			 * overflow)</li>
			 * <li>short option name (width: 2, indent: 0, TRUNCATE on overflow)</li>
			 * <li>comma separator (width: 1, indent: 0, TRUNCATE on overflow)</li>
			 * <li>long option name(s) (width: 24, indent: 1, SPAN multiple columns on
			 * overflow)</li>
			 * <li>description line(s) (width: 51, indent: 1, WRAP to next row on
			 * overflow)</li>
			 * </ol>
			 * 
			 * @param ansi                   whether to emit ANSI escape codes or not
			 * @param longOptionsColumnWidth the width of the long options column
			 * @param usageHelpWidth         the total width of the columns combined
			 * @deprecated use
			 *             {@link #forDefaultColumns(CommandLine.Help.ColorScheme, int, int)}
			 *             instead
			 */
			@Deprecated
			public static TextTable forDefaultColumns(Ansi ansi, int longOptionsColumnWidth, int usageHelpWidth) {
				return forDefaultColumns(Help.defaultColorScheme(ansi), longOptionsColumnWidth, usageHelpWidth);
			}

			/**
			 * Constructs a TextTable with five columns as follows:
			 * <ol>
			 * <li>required option/parameter marker (width: 2, indent: 0, TRUNCATE on
			 * overflow)</li>
			 * <li>short option name (width: 2, indent: 0, TRUNCATE on overflow)</li>
			 * <li>comma separator (width: 1, indent: 0, TRUNCATE on overflow)</li>
			 * <li>long option name(s) (width: 24, indent: 1, SPAN multiple columns on
			 * overflow)</li>
			 * <li>description line(s) (width: 51, indent: 1, WRAP to next row on
			 * overflow)</li>
			 * </ol>
			 * 
			 * @param colorScheme            the styles and ANSI mode to use for embedded
			 *                               markup
			 * @param longOptionsColumnWidth the width of the long options column
			 * @param usageHelpWidth         the total width of the columns combined
			 * @since 4.2
			 */
			public static TextTable forDefaultColumns(ColorScheme colorScheme, int longOptionsColumnWidth,
					int usageHelpWidth) {
				// "* -c, --create Creates a ...."
				final int descriptionWidth = usageHelpWidth - 5 - longOptionsColumnWidth;
				return forColumns(colorScheme, new Column(2, 0, TRUNCATE), // "*"
						new Column(2, 0, SPAN), // "-c"
						new Column(1, 0, TRUNCATE), // ","
						new Column(longOptionsColumnWidth, 1, SPAN), // " --create"
						new Column(descriptionWidth, 1, WRAP)); // " Creates a ..."
			}

			/** The column definitions of this table. */
			private final Column[] columns;

			/**
			 * The {@code char[]} slots of the {@code TextTable} to copy text values into.
			 */
			protected final List<Text> columnValues = new ArrayList<Text>();
			/** By default, indent wrapped lines by 2 spaces. */
			public int indentWrappedLines = 2;
			private final ColorScheme colorScheme;
			private final int tableWidth;
			private boolean adjustLineBreaksForWideCJKCharacters = true;

			/**
			 * @deprecated use
			 *             {@link org.rossonet.ext.picocli.CommandLine.Help.TextTable#TextTable(org.rossonet.ext.picocli.CommandLine.Help.ColorScheme, org.rossonet.ext.picocli.CommandLine.Help.Column[])}
			 *             instead
			 */
			@Deprecated
			protected TextTable(Ansi ansi, Column[] columns) {
				this(Help.defaultColorScheme(ansi), columns);
			}

			protected TextTable(ColorScheme colorScheme, Column[] columns) {
				this.colorScheme = Assert.notNull(colorScheme, "ansi");
				this.columns = Assert.notNull(columns, "columns").clone();
				if (columns.length == 0) {
					throw new IllegalArgumentException("At least one column is required");
				}
				int totalWidth = 0;
				for (final Column col : columns) {
					totalWidth += col.width;
				}
				tableWidth = totalWidth;
			}

			/**
			 * Adds the required {@code char[]} slots for a new row to the
			 * {@link #columnValues} field.
			 */
			public void addEmptyRow() {
				for (final Column column : columns) {
					columnValues.add(colorScheme.ansi().new Text(column.width, colorScheme));
				}
			}

			/**
			 * Delegates to {@link #addRowValues(CommandLine.Help.Ansi.Text...)}, after
			 * ensuring that multi-line values are layed out in the correct row and column.
			 * 
			 * @param values the text values to display in each column of the current row
			 */
			public void addRowValues(String... values) {
				final int numColumns = values.length;
				final Text[][] cells = new Text[numColumns][]; // an array of columns
				int maxRows = 0;
				for (int col = 0; col < numColumns; col++) {
					cells[col] = values[col] == null ? new Text[] { Ansi.EMPTY_TEXT }
							: colorScheme.text(values[col]).splitLines();
					maxRows = Math.max(maxRows, cells[col].length);
				}
				final Text[] rowValues = new Text[numColumns];
				for (int row = 0; row < maxRows; row++) {
					Arrays.fill(rowValues, Ansi.EMPTY_TEXT);
					for (int col = 0; col < numColumns; col++) {
						if (row < cells[col].length) {
							rowValues[col] = cells[col][row];
						}
					}
					addRowValues(rowValues);
				}
			}

			/**
			 * Adds a new {@linkplain TextTable#addEmptyRow() empty row}, then calls
			 * {@link TextTable#putValue(int, int, CommandLine.Help.Ansi.Text) putValue} for
			 * each of the specified values, adding more empty rows if the return value
			 * indicates that the value spanned multiple columns or was wrapped to multiple
			 * rows.
			 * 
			 * @param values the values to write into a new row in this TextTable
			 * @throws IllegalArgumentException if the number of values exceeds the number
			 *                                  of Columns in this table
			 */
			public void addRowValues(Text... values) {
				if (values.length > columns.length) {
					throw new IllegalArgumentException(
							values.length + " values don't fit in " + columns.length + " columns");
				}
				addEmptyRow();
				final int oldIndent = unindent(values);
				for (int col = 0; col < values.length; col++) {
					final int row = rowCount() - 1;// write to last row: previous value may have wrapped to next row
					final Cell cell = putValue(row, col, values[col]);

					// add row if a value spanned/wrapped and there are still remaining values
					if ((cell.row != row || cell.column != col) && col != values.length - 1) {
						addEmptyRow();
					}
				}
				reindent(oldIndent);
			}

			/**
			 * Returns the {@code Text} slot at the specified row and column to write a text
			 * value into.
			 * 
			 * @param row the row of the cell whose Text to return
			 * @param col the column of the cell whose Text to return
			 * @return the Text object at the specified row and column
			 * @deprecated use {@link #textAt(int, int)} instead
			 */
			@Deprecated
			public Text cellAt(int row, int col) {
				return textAt(row, col);
			}

			/** The column definitions of this table. */
			public Column[] columns() {
				return columns.clone();
			}

			private int copy(BreakIterator line, Text text, Text columnValue, int offset) {
				// Deceive the BreakIterator to ensure no line breaks after '-' character
				line.setText(text.plainString().replace("-", "\u00ff"));
				final Count count = new Count();
				for (int start = line.first(),
						end = line.next(); end != BreakIterator.DONE; start = end, end = line.next()) {
					final Text word = text.substring(start, end); // .replace("\u00ff", "-"); // not needed
					if (columnValue.maxLength >= offset + count.columnCount + length(word)) {
						copy(word, columnValue, offset + count.charCount, count);
					} else {
						break;
					}
				}
				if (count.charCount == 0 && length(text) + offset > columnValue.maxLength) {
					// The value is a single word that is too big to be written to the column. Write
					// as much as we can.
					copy(text, columnValue, offset, count);
				}
				return count.charCount;
			}

			private int copy(Text value, Text destination, int offset) {
				final Count count = new Count();
				copy(value, destination, offset, count);
				return count.charCount;
			}

			private void copy(Text value, Text destination, int offset, Count count) {
				final int length = Math.min(value.length, destination.maxLength - offset);
				value.getStyledChars(value.from, length, destination, offset);
				count.columnCount += length(value, value.from, length);
				count.charCount += length;
			}

			/**
			 * @see UsageMessageSpec#adjustLineBreaksForWideCJKCharacters()
			 * @since 4.0
			 */
			public boolean isAdjustLineBreaksForWideCJKCharacters() {
				return adjustLineBreaksForWideCJKCharacters;
			}

			private int length(Text str) {
				return str.getCJKAdjustedLength();
			}

			private int length(Text str, int from, int length) {
				if (!adjustLineBreaksForWideCJKCharacters) {
					return length - from;
				}
				return str.getCJKAdjustedLength(from, length);
			}

			/**
			 * Writes the specified value into the cell at the specified row and column and
			 * returns the last row and column written to. Depending on the Column's
			 * {@link Column#overflow Overflow} policy, the value may span multiple columns
			 * or wrap to multiple rows when larger than the column width.
			 * 
			 * @param row   the target row in the table
			 * @param col   the target column in the table to write to
			 * @param value the value to write
			 * @return a Cell indicating the position in the table that was last written to
			 *         (since 2.0)
			 * @throws IllegalArgumentException if the specified row exceeds the table's
			 *                                  {@linkplain TextTable#rowCount() row count}
			 * @since 2.0 (previous versions returned a {@code java.awt.Point} object)
			 */
			public Cell putValue(int row, int col, Text value) {
				if (row > rowCount() - 1) {
					throw new IllegalArgumentException("Cannot write to row " + row + ": rowCount=" + rowCount());
				}
				if (value == null || value.plain.length() == 0) {
					return new Cell(col, row);
				}
				final Column column = columns[col];
				int indent = column.indent;
				switch (column.overflow) {
				case TRUNCATE:
					copy(value, textAt(row, col), indent);
					return new Cell(col, row);
				case SPAN:
					final int startColumn = col;
					do {
						final boolean lastColumn = col == columns.length - 1;
						final int charsWritten = lastColumn
								? copy(BreakIterator.getLineInstance(), value, textAt(row, col), indent)
								: copy(value, textAt(row, col), indent);
						value = value.substring(charsWritten);
						indent = 0;
						if (value.length > 0) { // value did not fit in column
							++col; // write remainder of value in next column
						}
						if (value.length > 0 && col >= columns.length) { // we filled up all columns on this row
							addEmptyRow();
							row++;
							col = startColumn;
							indent = column.indent + indentWrappedLines;
						}
					} while (value.length > 0);
					return new Cell(col, row);
				case WRAP:
					final BreakIterator lineBreakIterator = BreakIterator.getLineInstance();
					do {
						final int charsWritten = copy(lineBreakIterator, value, textAt(row, col), indent);
						value = value.substring(charsWritten);
						indent = column.indent + indentWrappedLines;
						if (value.length > 0) { // value did not fit in column
							++row; // write remainder of value in next row
							addEmptyRow();
						}
					} while (value.length > 0);
					return new Cell(col, row);
				}
				throw new IllegalStateException(column.overflow.toString());
			}

			private void reindent(int oldIndent) {
				if (columns.length <= LONG_OPTION_COLUMN) {
					return;
				}
				columns[LONG_OPTION_COLUMN].indent = oldIndent;
			}

			/**
			 * Returns the current number of rows of this {@code TextTable}.
			 * 
			 * @return the current number of rows in this TextTable
			 */
			public int rowCount() {
				return columnValues.size() / columns.length;
			}

			/**
			 * @see UsageMessageSpec#adjustLineBreaksForWideCJKCharacters(boolean)
			 * @since 4.0
			 */
			public TextTable setAdjustLineBreaksForWideCJKCharacters(boolean adjustLineBreaksForWideCJKCharacters) {
				this.adjustLineBreaksForWideCJKCharacters = adjustLineBreaksForWideCJKCharacters;
				return this;
			}

			/**
			 * Returns the {@code Text} slot at the specified row and column to write a text
			 * value into.
			 * 
			 * @param row the row of the cell whose Text to return
			 * @param col the column of the cell whose Text to return
			 * @return the Text object at the specified row and column
			 * @since 2.0
			 */
			public Text textAt(int row, int col) {
				return columnValues.get(col + (row * columns.length));
			}

			@Override
			public String toString() {
				return toString(new StringBuilder()).toString();
			}

			/**
			 * Copies the text representation that we built up from the options into the
			 * specified StringBuilder.
			 * 
			 * @param text the StringBuilder to write into
			 * @return the specified StringBuilder object (to allow method chaining and a
			 *         more fluid API)
			 */
			public StringBuilder toString(StringBuilder text) {
				final int columnCount = this.columns.length;
				final StringBuilder row = new StringBuilder(tableWidth);
				for (int i = 0; i < columnValues.size(); i++) {
					final Text column = columnValues.get(i);
					row.append(column.toString());
					row.append(new String(spaces(columns[i % columnCount].width - column.length)));
					if (i % columnCount == columnCount - 1) {
						int lastChar = row.length() - 1;
						while (lastChar >= 0 && row.charAt(lastChar) == ' ') {
							lastChar--;
						} // rtrim
						row.setLength(lastChar + 1);
						text.append(row.toString()).append(System.getProperty("line.separator"));
						row.setLength(0);
					}
				}
				return text;
			}

			private int unindent(Text[] values) {
				if (columns.length <= LONG_OPTION_COLUMN) {
					return 0;
				}
				final int oldIndent = columns[LONG_OPTION_COLUMN].indent;
				if ("=".equals(values[OPTION_SEPARATOR_COLUMN].toString())) {
					columns[LONG_OPTION_COLUMN].indent = 0;
				}
				return oldIndent;
			}
		}

		/** Controls the visibility of certain aspects of the usage help message. */
		public enum Visibility {
			ALWAYS, NEVER, ON_DEMAND
		}

		/**
		 * Constant String holding the default program name, value defined in
		 * {@link CommandSpec#DEFAULT_COMMAND_NAME}.
		 */
		protected static final String DEFAULT_COMMAND_NAME = CommandSpec.DEFAULT_COMMAND_NAME;

		/**
		 * Constant String holding the default string that separates options from option
		 * parameters, value defined in {@link ParserSpec#DEFAULT_SEPARATOR}.
		 */
		protected static final String DEFAULT_SEPARATOR = ParserSpec.DEFAULT_SEPARATOR;

		private static void addTrailingDefaultLine(List<Text[]> result, ArgSpec arg, ColorScheme scheme) {
			final Text EMPTY = Ansi.EMPTY_TEXT;
			result.add(new Text[] { EMPTY, EMPTY, EMPTY, EMPTY,
					scheme.ansi().new Text("  Default: " + arg.defaultValueString(true), scheme) });
		}

		static Text concatOptionText(String prefix, Text text, Help.ColorScheme colorScheme, OptionSpec option,
				Help.IParamLabelRenderer parameterLabelRenderer) {
			if (!option.hidden()) {
				String nameString = option.shortestName();
				if (option.negatable) {
					final INegatableOptionTransformer trans = option.commandSpec == null
							? RegexTransformer.createDefault()
							: option.commandSpec.negatableOptionTransformer();
					nameString = trans.makeSynopsis(option.shortestName(), option.commandSpec);
				}
				final Text name = colorScheme.optionText(nameString);
				final Text param = parameterLabelRenderer.renderParameterLabel(option, colorScheme.ansi(),
						colorScheme.optionParamStyles);
				text = text.concat(prefix);

				if (option.required()) { // e.g., -x=VAL
					text = text.concat(name).concat(param).concat("");
					if (option.isMultiValue()) { // e.g., -x=VAL [-x=VAL]...
						text = text.concat(" [").concat(name).concat(param).concat("]...");
					}
				} else {
					text = text.concat("[").concat(name).concat(param).concat("]");
					if (option.isMultiValue()) { // add ellipsis to show option is repeatable
						text = text.concat("...");
					}
				}
			}
			return text;
		}

		static Text concatPositionalText(String prefix, Text text, Help.ColorScheme colorScheme,
				PositionalParamSpec positionalParam, IParamLabelRenderer parameterLabelRenderer) {
			if (!positionalParam.hidden()) {
				final Text label = parameterLabelRenderer.renderParameterLabel(positionalParam, colorScheme.ansi(),
						colorScheme.parameterStyles);
				text = text.concat(prefix).concat(label);
			}
			return text;
		}

		private static int countTrailingSpaces(String str) {
			if (str == null) {
				return 0;
			}
			int trailingSpaces = 0;
			for (int i = str.length() - 1; i >= 0 && str.charAt(i) == ' '; i--) {
				trailingSpaces++;
			}
			return trailingSpaces;
		}

		private static Text[] createDescriptionFirstLines(ColorScheme scheme, ArgSpec arg, String[] description,
				boolean[] showDefault) {
			Text[] result = scheme.ansi().new Text(str(description, 0), scheme).splitLines();
			if (result.length == 0 || (result.length == 1 && result[0].plain.length() == 0)) {
				if (showDefault[0]) {
					result = new Text[] {
							scheme.ansi().new Text("  Default: " + arg.defaultValueString(true), scheme) };
					showDefault[0] = false; // don't show the default value twice
				} else {
					result = new Text[] { Ansi.EMPTY_TEXT };
				}
			}
			return result;
		}

		/**
		 * Returns a new minimal OptionRenderer which converts {@link OptionSpec
		 * Options} to a single row with two columns of text: an option name and a
		 * description. If multiple names or descriptions exist, the first value is
		 * used.
		 * 
		 * @return a new minimal OptionRenderer
		 */
		public static IOptionRenderer createMinimalOptionRenderer() {
			return new MinimalOptionRenderer();
		}

		/**
		 * Returns a new minimal ParameterRenderer which converts
		 * {@linkplain PositionalParamSpec positional parameters} to a single row with
		 * two columns of text: an option name and a description. If multiple
		 * descriptions exist, the first value is used.
		 * 
		 * @return a new minimal ParameterRenderer
		 */
		public static IParameterRenderer createMinimalParameterRenderer() {
			return new MinimalParameterRenderer();
		}

		/**
		 * Returns a value renderer that returns the {@code paramLabel} if defined or
		 * the field name otherwise.
		 * 
		 * @return a new minimal ParamLabelRenderer
		 */
		public static IParamLabelRenderer createMinimalParamLabelRenderer() {
			return new IParamLabelRenderer() {
				@Override
				public Text renderParameterLabel(ArgSpec argSpec, Ansi ansi, List<IStyle> styles) {
					return argSpec.command() != null && argSpec.command().commandLine() != null
							? argSpec.command().commandLine().getColorScheme().apply(argSpec.paramLabel(), styles)
							: ansi.apply(argSpec.paramLabel(), styles);
				}

				@Override
				public String separator() {
					return "";
				}
			};
		}

		/**
		 * Sorts {@link OptionSpec options} by their option {@linkplain IOrdered#order()
		 * order}, lowest first, highest last.
		 * 
		 * @return a comparator that sorts OptionSpecs by their order
		 * @since 3.9
		 */
		static Comparator<OptionSpec> createOrderComparator() {
			return new SortByOrder<OptionSpec>();
		}

		private static Comparator<OptionSpec> createOrderComparatorIfNecessary(List<OptionSpec> options) {
			for (final OptionSpec option : options) {
				if (option.order() != OptionSpec.DEFAULT_ORDER) {
					return createOrderComparator();
				}
			}
			return null;
		}

		/**
		 * Sorts {@link OptionSpec OptionSpecs} by their option {@linkplain Range#max
		 * max arity} first, by {@linkplain Range#min min arity} next, and by
		 * {@linkplain #createShortOptionNameComparator() option name} last.
		 * 
		 * @return a comparator that sorts OptionSpecs by arity first, then their option
		 *         name
		 */
		public static Comparator<OptionSpec> createShortOptionArityAndNameComparator() {
			return new SortByOptionArityAndNameAlphabetically();
		}

		/**
		 * Sorts {@link OptionSpec OptionSpecs} by their option name in case-insensitive
		 * alphabetic order. If an option has multiple names, the shortest name is used
		 * for the sorting. Help options follow non-help options.
		 * 
		 * @return a comparator that sorts OptionSpecs by their option name in
		 *         case-insensitive alphabetic order
		 */
		public static Comparator<OptionSpec> createShortOptionNameComparator() {
			return new SortByShortestOptionNameAlphabetically();
		}

		/**
		 * Creates and returns a new {@link ColorScheme} initialized with picocli
		 * default values: commands are bold, options and parameters use a yellow
		 * foreground, and option parameters use italic.
		 * 
		 * @param ansi whether the usage help message should contain ANSI escape codes
		 *             or not
		 * @return a new default color scheme
		 */
		public static ColorScheme defaultColorScheme(Ansi ansi) {
			return new ColorScheme.Builder(ansi).commands(Style.bold).options(Style.fg_yellow)
					.parameters(Style.fg_yellow).optionParams(Style.italic).errors(Style.fg_red, Style.bold)
					.stackTraces(Style.italic).build();
		}

		private static String heading(Ansi ansi, int usageWidth, boolean adjustCJK, String values, Object... params) {
			final StringBuilder sb = join(ansi, usageWidth, adjustCJK, new String[] { values }, new StringBuilder(),
					params);
			return trimLineSeparator(sb.toString()) + new String(spaces(countTrailingSpaces(values)));
		}

		/**
		 * Formats each of the specified values and appends it to the specified
		 * StringBuilder.
		 * 
		 * @param ansi           whether the result should contain ANSI escape codes or
		 *                       not
		 * @param usageHelpWidth the width of the usage help message
		 * @param adjustCJK      true if wide Chinese, Japanese and Korean characters
		 *                       should be counted as double the size of other
		 *                       characters for line-breaking purposes
		 * @param values         the values to format and append to the StringBuilder
		 * @param sb             the StringBuilder to collect the formatted strings
		 * @param params         the parameters to pass to the format method when
		 *                       formatting each value
		 * @return the specified StringBuilder
		 * @since 4.0
		 */
		public static StringBuilder join(Ansi ansi, int usageHelpWidth, boolean adjustCJK, String[] values,
				StringBuilder sb, Object... params) {
			if (values != null) {
				final TextTable table = TextTable.forColumnWidths(ansi, usageHelpWidth);
				table.setAdjustLineBreaksForWideCJKCharacters(adjustCJK);
				table.indentWrappedLines = 0;
				for (final String summaryLine : values) {
					table.addRowValues(format(summaryLine, params));
				}
				table.toString(sb);
			}
			return sb;
		}

		/**
		 * @deprecated Use
		 *             {@link #join(Ansi, int, boolean, String[], StringBuilder, Object...)}
		 *             instead
		 */
		@Deprecated
		public static StringBuilder join(Ansi ansi, int usageHelpWidth, String[] values, StringBuilder sb,
				Object... params) {
			return join(ansi, usageHelpWidth, UsageMessageSpec.DEFAULT_ADJUST_CJK, values, sb, params);
		}

		private static String join(String[] names, int offset, int length, String separator) {
			if (names == null) {
				return "";
			}
			final StringBuilder result = new StringBuilder();
			for (int i = offset; i < offset + length; i++) {
				result.append((i > offset) ? separator : "").append(names[i]);
			}
			return result.toString();
		}

		private static int maxLength(Collection<?> any) {
			int result = 0;
			for (final Object value : any) {
				result = Math.max(result, String.valueOf(value).length());
			}
			return result;
		}

		private static void optionSectionGroups(List<ArgGroupSpec> groups, List<ArgGroupSpec> result) {
			for (final ArgGroupSpec group : groups) {
				optionSectionGroups(group.subgroups(), result);
				if (group.heading() != null) {
					result.add(group);
				}
			}
		}

		/**
		 * Sorts short strings before longer strings.
		 * 
		 * @return a comparators that sorts short strings before longer strings
		 */
		public static Comparator<String> shortestFirst() {
			return new ShortestFirst();
		}

		private static char[] spaces(int length) {
			final char[] result = new char[length];
			Arrays.fill(result, ' ');
			return result;
		}

		private static String stringOf(char chr, int length) {
			final char[] buff = new char[length];
			Arrays.fill(buff, chr);
			return new String(buff);
		}

		static String trimLineSeparator(String result) {
			return result.endsWith(System.getProperty("line.separator"))
					? result.substring(0, result.length() - System.getProperty("line.separator").length())
					: result;
		}

		public final PositionalParamSpec AT_FILE_POSITIONAL_PARAM = PositionalParamSpec.builder()
				.paramLabel("${picocli.atfile.label:-@<filename>}")
				.description("${picocli.atfile.description:-One or more argument files containing options.}")
				.arity("0..*").descriptionKey("picocli.atfile").build();

		public final OptionSpec END_OF_OPTIONS_OPTION = createEndOfOptionsOption(
				ParserSpec.DEFAULT_END_OF_OPTIONS_DELIMITER);

		private final CommandSpec commandSpec;
		private final ColorScheme colorScheme;
		private final Map<String, Help> allCommands = new LinkedHashMap<String, Help>();

		private final Map<String, Help> visibleCommands = new LinkedHashMap<String, Help>();

		private List<String> aliases;
		private final IParamLabelRenderer parameterLabelRenderer;

		/**
		 * Constructs a new {@code Help} instance with the specified color scheme,
		 * initialized from annotations on the specified class and superclasses.
		 * 
		 * @param commandSpec the command model to create usage help for
		 * @param colorScheme the color scheme to use
		 */
		public Help(CommandSpec commandSpec, ColorScheme colorScheme) {
			this.commandSpec = Assert.notNull(commandSpec, "commandSpec");
			this.aliases = new ArrayList<String>(Arrays.asList(commandSpec.aliases()));
			this.aliases.add(0, commandSpec.name());
			this.colorScheme = new ColorScheme.Builder(colorScheme).applySystemProperties().build();
			this.parameterLabelRenderer = new DefaultParamLabelRenderer(commandSpec); // uses help separator

			this.registerSubcommands(commandSpec.subcommands());
			AT_FILE_POSITIONAL_PARAM.commandSpec = commandSpec; // for interpolation
		}

		/**
		 * Constructs a new {@code Help} instance with a default color scheme,
		 * initialized from annotations on the specified class and superclasses.
		 * 
		 * @param command the annotated object to create usage help for
		 */
		public Help(Object command) {
			this(command, Ansi.AUTO);
		}

		/**
		 * Constructs a new {@code Help} instance with a default color scheme,
		 * initialized from annotations on the specified class and superclasses.
		 * 
		 * @param command the annotated object to create usage help for
		 * @param ansi    whether to emit ANSI escape codes or not
		 */
		public Help(Object command, Ansi ansi) {
			this(command, defaultColorScheme(ansi));
		}

		/**
		 * Constructs a new {@code Help} instance with the specified color scheme,
		 * initialized from annotations on the specified class and superclasses.
		 * 
		 * @param command     the annotated object to create usage help for
		 * @param colorScheme the color scheme to use
		 * @deprecated use
		 *             {@link org.rossonet.ext.picocli.CommandLine.Help#Help(org.rossonet.ext.picocli.CommandLine.Model.CommandSpec, org.rossonet.ext.picocli.CommandLine.Help.ColorScheme)}
		 */
		@Deprecated
		public Help(Object command, ColorScheme colorScheme) {
			this(CommandSpec.forAnnotatedObject(command, new DefaultFactory()), colorScheme);
		}

		/**
		 * Generates a generic synopsis like
		 * {@code <command name> [OPTIONS] [PARAM1 [PARAM2]...]}, omitting parts that
		 * don't apply to the command (e.g., does not show [OPTIONS] if the command has
		 * no options).
		 * 
		 * @return a generic synopsis
		 */
		public String abbreviatedSynopsis() {
			final StringBuilder sb = new StringBuilder();
			if (!commandSpec.optionsMap().isEmpty()) { // only show if annotated object actually has options
				sb.append(" [OPTIONS]");
			}
			// sb.append(" [--] "); // implied
			for (final PositionalParamSpec positionalParam : commandSpec.positionalParameters()) {
				if (!positionalParam.hidden()) {
					sb.append(' ').append(parameterLabelRenderer().renderParameterLabel(positionalParam, ansi(),
							colorScheme.parameterStyles));
				}
			}

			// only show if object has subcommands
			if (!commandSpec.subcommands().isEmpty()) {
				sb.append(" ").append(commandSpec.usageMessage().synopsisSubcommandLabel());
			}

			return colorScheme.commandText(commandSpec.qualifiedName()).toString() + (sb.toString())
					+ System.getProperty("line.separator");
		}

		/**
		 * Registers all specified subcommands with this Help.
		 * 
		 * @param subcommands the subcommands of this command
		 * @return this Help instance (for method chaining)
		 * @see #subcommands()
		 * @see #allSubcommands()
		 */
		public Help addAllSubcommands(Map<String, CommandLine> subcommands) {
			if (subcommands != null) {
				registerSubcommands(subcommands);
			}
			return this;
		}

		/**
		 * Registers the specified subcommand as one of the visible commands in this
		 * Help. This method does not check whether the specified command is hidden or
		 * not.
		 * 
		 * @param commandName the name of the subcommand to display in the usage message
		 * @param command     the {@code CommandSpec} or {@code @Command} annotated
		 *                    object to get more information from
		 * @return this Help instance (for method chaining)
		 * @see #subcommands()
		 * @deprecated use {@link #addAllSubcommands(Map)} instead
		 */
		@Deprecated
		public Help addSubcommand(String commandName, Object command) {
			final Help sub = getHelpFactory().create(
					CommandSpec.forAnnotatedObject(command, commandSpec.commandLine().factory),
					defaultColorScheme(Ansi.AUTO));
			visibleCommands.put(commandName, sub);
			allCommands.put(commandName, sub);
			return this;
		}

		private boolean adjustCJK() {
			return commandSpec.usageMessage().adjustLineBreaksForWideCJKCharacters();
		}

		/**
		 * Returns the list of aliases for the command in this Help.
		 * 
		 * @since 3.9
		 */
		protected List<String> aliases() {
			return Collections.unmodifiableList(aliases);
		}

		/**
		 * Returns the map of all subcommand {@code Help} instances (including hidden
		 * commands) for this command Help.
		 * 
		 * @since 4.4
		 * @see #subcommands()
		 */
		public Map<String, Help> allSubcommands() {
			return Collections.unmodifiableMap(allCommands);
		}

		/**
		 * Returns whether ANSI escape codes are enabled or not.
		 * 
		 * @return whether ANSI escape codes are enabled or not
		 */
		public Ansi ansi() {
			return colorScheme.ansi;
		}

		/**
		 * Returns the section of the usage help message that lists the @-file and its
		 * description.
		 * 
		 * @return the section of the usage help message that lists the @-file and its
		 *         description
		 * @since 4.2
		 */
		public String atFileParameterList() {
			if (hasAtFileParameter()) {
				AT_FILE_POSITIONAL_PARAM.messages(commandSpec.usageMessage().messages());
				final Layout layout = createDefaultLayout();
				layout.addPositionalParameter(AT_FILE_POSITIONAL_PARAM, parameterLabelRenderer());
				return layout.toString();
			}
			return "";
		}

		/**
		 * Returns the width of the long options column in the usage help message.
		 * 
		 * @param options      the options shown in the usage help message
		 * @param positionals  the positional parameters shown in the usage help message
		 * @param aColorScheme the colorscheme used in the layout to create
		 *                     {@linkplain Text} values
		 * @return the width of the long options column in the layout
		 * @since 4.6
		 */
		public int calcLongOptionColumnWidth(List<OptionSpec> options, List<PositionalParamSpec> positionals,
				ColorScheme aColorScheme) {
			int max = 0;
			final IOptionRenderer optionRenderer = createDefaultOptionRenderer();
			final boolean cjk = commandSpec.usageMessage().adjustLineBreaksForWideCJKCharacters();
			final int longOptionsColWidth = commandSpec.usageMessage().longOptionsMaxWidth() + 1; // add 1 space for
																									// indentation
			for (final OptionSpec option : options) {
				if (option.hidden()) {
					continue;
				}
				final Text[][] values = optionRenderer.render(option, parameterLabelRenderer(), aColorScheme);
				final int len = cjk ? values[0][3].getCJKAdjustedLength() : values[0][3].length;
				if (len < longOptionsColWidth) {
					max = Math.max(max, len);
				}
			}
			final List<PositionalParamSpec> positionalsWithAtFile = new ArrayList<PositionalParamSpec>(positionals); // iterate
																														// in
																														// declaration
																														// order
			if (hasAtFileParameter()) {
				positionalsWithAtFile.add(0, AT_FILE_POSITIONAL_PARAM);
				AT_FILE_POSITIONAL_PARAM.messages(commandSpec.usageMessage().messages());
			}
			// IParameterRenderer paramRenderer = new DefaultParameterRenderer(false, " ");
			for (final PositionalParamSpec positional : positionalsWithAtFile) {
				if (positional.hidden()) {
					continue;
				}
				// Text[][] values = paramRenderer.render(positional, parameterLabelRenderer(),
				// colorScheme); // values[0][3]; //
				final Text label = parameterLabelRenderer().renderParameterLabel(positional, aColorScheme.ansi(),
						aColorScheme.parameterStyles);
				final int len = cjk ? label.getCJKAdjustedLength() : label.length;
				if (len < longOptionsColWidth) {
					max = Math.max(max, len);
				}
			}

			return max + 3;
		}

		/**
		 * Returns the {@code ColorScheme} model that this Help was constructed with.
		 * 
		 * @since 3.0
		 */
		public ColorScheme colorScheme() {
			return colorScheme;
		}

		/**
		 * Returns a 2-column list with the command names and first line of their header
		 * or (if absent) description of the commands returned by
		 * {@link #subcommands()}.
		 * 
		 * @return a usage help section describing the added commands
		 * @see #commandList(Map)
		 */
		public String commandList() {
			return commandList(subcommands());
		}

		/**
		 * Returns a 2-column list with the command names and first line of their header
		 * or (if absent) description of the specified command map.
		 * 
		 * @return a usage help section describing the added commands
		 * @see #subcommands()
		 * @see #allSubcommands()
		 * @since 4.4
		 */
		public String commandList(Map<String, Help> subcommands) {
			if (subcommands.isEmpty()) {
				return "";
			}
			final int maxCommandLength = width() / 2;
			final int commandLength = Math.min(maxLength(subcommands.keySet()), maxCommandLength);
			final Help.TextTable textTable = Help.TextTable.forColumns(colorScheme().ansi(),
					new Help.Column(commandLength + 2, 2, Help.Column.Overflow.SPAN),
					new Help.Column(width() - (commandLength + 2), 2, Help.Column.Overflow.WRAP));
			textTable.setAdjustLineBreaksForWideCJKCharacters(adjustCJK());

			for (final Map.Entry<String, Help> entry : subcommands.entrySet()) {
				final Help help = entry.getValue();
				final UsageMessageSpec usage = help.commandSpec().usageMessage();
				final String header = !empty(usage.header()) ? usage.header()[0]
						: (!empty(usage.description()) ? usage.description()[0] : "");
				final Text[] lines = colorScheme().text(format(header)).splitLines();
				for (int i = 0; i < lines.length; i++) {
					textTable.addRowValues(i == 0 ? help.commandNamesText(", ") : Ansi.EMPTY_TEXT, lines[i]);
				}
			}
			return textTable.toString();
		}

		/**
		 * Returns the text displayed before the command list; an empty string if there
		 * are no commands, otherwise the result of
		 * {@code String.format(commandListHeading, params)}.
		 * 
		 * @param params the parameters to use to format the command list heading
		 * @return the formatted command list heading
		 */
		public String commandListHeading(Object... params) {
			return visibleCommands.isEmpty() ? ""
					: createHeading(commandSpec.usageMessage().commandListHeading(), params);
		}

		String commandName() {
			return commandSpec.name();
		}

		/**
		 * Returns a {@code Text} object containing the command name and all aliases,
		 * separated with the specified separator. Command names will use the
		 * {@link ColorScheme#commandText(String) command style} for the color scheme of
		 * this Help.
		 * 
		 * @since 3.9
		 */
		public Text commandNamesText(String separator) {
			Text result = colorScheme().commandText(aliases().get(0));
			for (int i = 1; i < aliases().size(); i++) {
				result = result.concat(separator).concat(colorScheme().commandText(aliases().get(i)));
			}
			return result;
		}

		/**
		 * Returns the {@code CommandSpec} model that this Help was constructed with.
		 * 
		 * @since 3.9
		 */
		public CommandSpec commandSpec() {
			return commandSpec;
		}

		/**
		 * Returns a {@code Layout} instance configured with the user preferences
		 * captured in this Help instance.
		 * 
		 * @return a Layout
		 */
		public Layout createDefaultLayout() {
			return createDefaultLayout(options(), positionalParameters(), colorScheme());
		}

		/**
		 * Returns a {@code Layout} instance configured with the user preferences
		 * captured in this Help instance.
		 * 
		 * @param options      used to calculate the long options column width in the
		 *                     layout
		 * @param positionals  used to calculate the long options column width in the
		 *                     layout
		 * @param aColorScheme used in the layout to create {@linkplain Text} values
		 * @return a Layout with the default columns
		 * @since 4.4
		 */
		public Layout createDefaultLayout(List<OptionSpec> options, List<PositionalParamSpec> positionals,
				ColorScheme aColorScheme) {
			return createLayout(calcLongOptionColumnWidth(options, positionals, aColorScheme), aColorScheme);
		}

		/**
		 * Returns a new default OptionRenderer which converts {@link OptionSpec
		 * Options} to five columns of text to match the default {@linkplain TextTable
		 * TextTable} column layout. The first row of values looks like this:
		 * <ol>
		 * <li>the required option marker</li>
		 * <li>2-character short option name (or empty string if no short option
		 * exists)</li>
		 * <li>comma separator (only if both short option and long option exist, empty
		 * string otherwise)</li>
		 * <li>comma-separated string with long option name(s)</li>
		 * <li>first element of the {@link OptionSpec#description()} array</li>
		 * </ol>
		 * <p>
		 * Following this, there will be one row for each of the remaining elements of
		 * the {@link OptionSpec#description()} array, and these rows look like
		 * {@code {"", "", "", "", option.description()[i]}}.
		 * </p>
		 * <p>
		 * If configured, this option renderer adds an additional row to display the
		 * default field value.
		 * </p>
		 * 
		 * @return a new default OptionRenderer
		 */
		public IOptionRenderer createDefaultOptionRenderer() {
			return new DefaultOptionRenderer(commandSpec.usageMessage.showDefaultValues(),
					"" + commandSpec.usageMessage().requiredOptionMarker());
		}

		/**
		 * Returns a comparator for sorting options, or {@code null}, depending on the
		 * settings for this command.
		 * 
		 * @return if {@linkplain Command#sortOptions() sortOptions} is selected, return
		 *         a comparator for sorting options based on their short name.
		 *         Otherwise, if any of the options has a non-default value for their
		 *         {@linkplain Option#order() order} attribute, then return a comparator
		 *         for sorting options based on the order attribute. Otherwise, return
		 *         {@code null} to indicate that options should not be sorted.
		 * @since 4.4
		 */
		public Comparator<OptionSpec> createDefaultOptionSort() {
			return commandSpec.usageMessage().sortOptions() ? createShortOptionNameComparator()
					: createOrderComparatorIfNecessary(commandSpec.options());
		}

		/**
		 * Returns a new default ParameterRenderer which converts
		 * {@linkplain PositionalParamSpec positional parameters} to four columns of
		 * text to match the default {@linkplain TextTable TextTable} column layout. The
		 * first row of values looks like this:
		 * <ol>
		 * <li>empty string</li>
		 * <li>empty string</li>
		 * <li>parameter(s) label as rendered by the {@link IParamLabelRenderer}</li>
		 * <li>first element of the {@link PositionalParamSpec#description()} array</li>
		 * </ol>
		 * <p>
		 * Following this, there will be one row for each of the remaining elements of
		 * the {@link PositionalParamSpec#description()} array, and these rows look like
		 * {@code {"", "", "", param.description()[i]}}.
		 * </p>
		 * <p>
		 * If configured, this parameter renderer adds an additional row to display the
		 * default field value.
		 * </p>
		 * 
		 * @return a new default ParameterRenderer
		 */
		public IParameterRenderer createDefaultParameterRenderer() {
			return new DefaultParameterRenderer(commandSpec.usageMessage.showDefaultValues(),
					"" + commandSpec.usageMessage().requiredOptionMarker());
		}

		/**
		 * Returns a new default param label renderer that separates option parameters
		 * from their option name with the specified separator string, and, unless
		 * {@link ArgSpec#hideParamSyntax()} is true, surrounds optional parameters with
		 * {@code '['} and {@code ']'} characters and uses ellipses ("...") to indicate
		 * that any number of a parameter are allowed.
		 * 
		 * @return a new default ParamLabelRenderer
		 */
		public IParamLabelRenderer createDefaultParamLabelRenderer() {
			return new DefaultParamLabelRenderer(commandSpec);
		}

		/**
		 * Returns a Text object containing a partial detailed synopsis showing only the
		 * subcommands, starting with a {@code " "} space. Follows the unix convention
		 * of showing optional elements in square brackets ({@code [ ]}).
		 * 
		 * @return this implementation returns " " +
		 *         {@link UsageMessageSpec#synopsisSubcommandLabel()} if this command
		 *         has subcommands, an empty Text otherwise.
		 * @since 3.9
		 */
		protected Text createDetailedSynopsisCommandText() {
			final Text commandText = ansi().new Text(0);
			if (!commandSpec.subcommands().isEmpty()) {
				return commandText.concat(" ").concat(commandSpec.usageMessage().synopsisSubcommandLabel());
			}
			return commandText;
		}

		/**
		 * Returns a Text object containing a partial detailed synopsis showing only the
		 * end of options delimiter (if enabled), starting with a {@code " "} space.
		 * Follows the unix convention of showing optional options and parameters in
		 * square brackets ({@code [ ]}).
		 * 
		 * @return the formatted end of options delimiter, starting with a {@code " "}
		 *         space, or an empty Text if the end of options delimiter should not be
		 *         shown
		 * @since 4.3
		 */
		protected Text createDetailedSynopsisEndOfOptionsText() {
			if (!commandSpec.usageMessage.showEndOfOptionsDelimiterInUsageHelp()) {
				return ansi().new Text(0);
			}
			return ansi().new Text(0).concat(" [")
					.concat(colorScheme.optionText(commandSpec.parser().endOfOptionsDelimiter())).concat("]");
		}

		/**
		 * Returns a Text object containing a partial detailed synopsis showing only the
		 * options and positional parameters in the specified
		 * {@linkplain ArgGroup#validate() validating} {@linkplain ArgGroup groups},
		 * starting with a {@code " "} space.
		 * 
		 * @param outparam_groupArgs all options and positional parameters in the groups
		 *                           this method generates a synopsis for; these options
		 *                           and positional parameters should be excluded from
		 *                           appearing elsewhere in the synopsis
		 * @return the formatted groups synopsis elements, starting with a {@code " "}
		 *         space, or an empty Text if this command has no validating groups
		 * @since 4.0
		 */
		protected Text createDetailedSynopsisGroupsText(Set<ArgSpec> outparam_groupArgs) {
			Text groupText = ansi().new Text(0);
			for (final ArgGroupSpec group : commandSpec().argGroups()) {
				if (group.validate()) { // non-validating groups are not shown in the synopsis
					groupText = groupText.concat(" ").concat(group.synopsisText(colorScheme(), outparam_groupArgs));
				}
			}
			return groupText;
		}

		/**
		 * Returns a Text object containing a partial detailed synopsis showing only the
		 * options, starting with a {@code " "} space. Follows the unix convention of
		 * showing optional options and parameters in square brackets ({@code [ ]}).
		 * 
		 * @param done                  the list of options and positional parameters
		 *                              for which a synopsis was already generated.
		 *                              Options in this set should be excluded.
		 * @param optionSort            comparator to sort options or {@code null} if
		 *                              options should not be sorted
		 * @param clusterBooleanOptions {@code true} if boolean short options should be
		 *                              clustered into a single string
		 * @return the formatted options, starting with a {@code " "} space, or an empty
		 *         Text if this command has no named options
		 * @since 3.9
		 */
		protected Text createDetailedSynopsisOptionsText(Collection<ArgSpec> done, Comparator<OptionSpec> optionSort,
				boolean clusterBooleanOptions) {
			return createDetailedSynopsisOptionsText(done, commandSpec.options(), optionSort, clusterBooleanOptions);
		}

		/**
		 * Returns a Text object containing a partial detailed synopsis showing only the
		 * specified options, starting with a {@code " "} space. Follows the unix
		 * convention of showing optional options and parameters in square brackets
		 * ({@code [ ]}).
		 * 
		 * @param done                  the list of options and positional parameters
		 *                              for which a synopsis was already generated.
		 *                              Options in this set should be excluded.
		 * @param optionList            the list of options to include in the synopsis
		 * @param optionSort            comparator to sort options or {@code null} if
		 *                              options should not be sorted
		 * @param clusterBooleanOptions {@code true} if boolean short options should be
		 *                              clustered into a single string
		 * @return the formatted options, starting with a {@code " "} space, or an empty
		 *         Text if this command has no named options
		 * @since 4.4
		 */
		protected Text createDetailedSynopsisOptionsText(Collection<ArgSpec> done, List<OptionSpec> optionList,
				Comparator<OptionSpec> optionSort, boolean clusterBooleanOptions) {
			Text optionText = ansi().new Text(0);
			final List<OptionSpec> options = new ArrayList<OptionSpec>(optionList); // iterate in declaration order
			if (optionSort != null) {
				Collections.sort(options, optionSort);// iterate in specified sort order
			}
			options.removeAll(done);
			if (clusterBooleanOptions) { // cluster all short boolean options into a single string
				final List<OptionSpec> booleanOptions = new ArrayList<OptionSpec>();
				final StringBuilder clusteredRequired = new StringBuilder("-");
				final StringBuilder clusteredOptional = new StringBuilder("-");
				for (final OptionSpec option : options) {
					if (option.hidden()) {
						continue;
					}
					final boolean isFlagOption = option.typeInfo().isBoolean();
					if (isFlagOption && option.arity().max <= 0) { // #612 consider arity: boolean options may require a
																	// parameter
						final String shortestName = option.shortestName();
						if (shortestName.length() == 2 && shortestName.startsWith("-")) { // POSIX short option
							// we don't want to show negatable options as clustered options in the synopsis
							if (!option.negatable() || shortestName.equals(
									commandSpec.negatableOptionTransformer().makeSynopsis(shortestName, commandSpec))) {
								booleanOptions.add(option);
								if (option.required()) {
									clusteredRequired.append(shortestName.substring(1));
								} else {
									clusteredOptional.append(shortestName.substring(1));
								}
							}
						}
					}
				}
				options.removeAll(booleanOptions);
				if (clusteredRequired.length() > 1) { // initial length was 1
					optionText = optionText.concat(" ").concat(colorScheme.optionText(clusteredRequired.toString()));
				}
				if (clusteredOptional.length() > 1) { // initial length was 1
					optionText = optionText.concat(" [").concat(colorScheme.optionText(clusteredOptional.toString()))
							.concat("]");
				}
			}
			for (final OptionSpec option : options) {
				optionText = concatOptionText(" ", optionText, colorScheme, option, parameterLabelRenderer());
			}
			return optionText;
		}

		/**
		 * Returns a Text object containing a partial detailed synopsis showing only the
		 * positional parameters, starting with a {@code " "} space. Follows the unix
		 * convention of showing optional options and parameters in square brackets
		 * ({@code [ ]}).
		 * 
		 * @param done the list of options and positional parameters for which a
		 *             synopsis was already generated. Positional parameters in this set
		 *             should be excluded.
		 * @return the formatted positional parameters, starting with a {@code " "}
		 *         space, or an empty Text if this command has no positional parameters
		 * @since 3.9
		 */
		protected Text createDetailedSynopsisPositionalsText(Collection<ArgSpec> done) {
			Text positionalParamText = ansi().new Text(0);
			final List<PositionalParamSpec> positionals = new ArrayList<PositionalParamSpec>(
					commandSpec.positionalParameters()); // iterate in declaration order
			if (hasAtFileParameter()) {
				positionals.add(0, AT_FILE_POSITIONAL_PARAM);
				AT_FILE_POSITIONAL_PARAM.messages(commandSpec.usageMessage().messages());
			}
			positionals.removeAll(done);
			for (final PositionalParamSpec positionalParam : positionals) {
				positionalParamText = concatPositionalText(" ", positionalParamText, colorScheme, positionalParam,
						parameterLabelRenderer());
			}
			return positionalParamText;
		}

		private OptionSpec createEndOfOptionsOption(String name) {
			return OptionSpec.builder(name).description(
					"${picocli.endofoptions.description:-This option can be used to separate command-line options from the list of positional parameters.}")
					.arity("0").descriptionKey("picocli.endofoptions").build();
		}

		/**
		 * Returns a String that can be used as a help section heading. Embedded
		 * {@code %n} format specifiers will be converted to platform-specific line
		 * breaks. Long lines will be wrapped on word boundaries to ensure they do not
		 * exceed the {@linkplain UsageMessageSpec#width() usage message width}.
		 * Embedded {@code @|style[,style] ...|@} markup will be converted to
		 * {@linkplain Ansi.Text Ansi} escape codes when {@linkplain Ansi#enabled() Ansi
		 * is enabled}, and stripped out otherwise.
		 * 
		 * @param text   a printf-style {@linkplain Formatter format string} that may
		 *               one or more embedded format specifiers
		 * @param params optional parameters to use when formatting the specified text
		 *               string
		 * @return a help section heading String
		 * @since 4.1
		 */
		public String createHeading(String text, Object... params) {
			return heading(ansi(), width(), adjustCJK(), text, params);
		}

		private Layout createLayout(int longOptionsColumnWidth, ColorScheme aColorScheme) {
			final TextTable tt = TextTable.forDefaultColumns(aColorScheme, longOptionsColumnWidth, width());
			tt.setAdjustLineBreaksForWideCJKCharacters(
					commandSpec.usageMessage().adjustLineBreaksForWideCJKCharacters());
			return new Layout(aColorScheme, tt, createDefaultOptionRenderer(), createDefaultParameterRenderer());
		}

		/**
		 * Returns a 2-column {@code TextTable} containing data from the specified map:
		 * the keys are put in the left column and the map values are in the right
		 * column.
		 * <p>
		 * The width of the left column is the width of the longest key, plus 3 for
		 * spacing between the columns.
		 * </p>
		 * <p>
		 * All map entries are converted to Strings and any embedded {@code %n} format
		 * specifiers are converted to platform-specific line breaks. Long lines are
		 * wrapped on word boundaries to ensure they do not exceed the column width.
		 * </p>
		 * <p>
		 * Embedded {@code @|style[,style] ...|@} markup will be converted to
		 * {@linkplain Ansi.Text Ansi} escape codes when {@linkplain Ansi#enabled() Ansi
		 * is enabled}, and stripped out otherwise.
		 * </p>
		 * 
		 * @param map the map to convert to a {@code TextTable}
		 * @return a 2-column {@code TextTable} containing data from the specified map
		 * @since 4.1
		 */
		public TextTable createTextTable(Map<?, ?> map) {
			if (map == null || map.isEmpty()) {
				return TextTable.forColumnWidths(colorScheme, 10, width() - 10);
			}
			final int spacing = 3;
			final int indent = 2;
			final int keyLength = Math.min(width() - spacing - 1, maxLength(map.keySet()));
			final Help.TextTable textTable = Help.TextTable.forColumns(ansi(),
					new Help.Column(keyLength + spacing, indent, Help.Column.Overflow.SPAN),
					new Help.Column(width() - (keyLength + spacing), indent, Help.Column.Overflow.WRAP));
			textTable.setAdjustLineBreaksForWideCJKCharacters(adjustCJK());

			for (final Map.Entry<?, ?> entry : map.entrySet()) {
				textTable.addRowValues(format(String.valueOf(entry.getKey())),
						format(String.valueOf(entry.getValue())));
			}
			return textTable;
		}

		/**
		 * Returns command custom synopsis as a string. A custom synopsis can be zero or
		 * more lines, and can be specified declaratively with the
		 * {@link Command#customSynopsis()} annotation attribute or programmatically by
		 * setting the Help instance's {@link Help#customSynopsis} field.
		 * 
		 * @param params Arguments referenced by the format specifiers in the synopsis
		 *               strings
		 * @return the custom synopsis lines combined into a single String (which may be
		 *         empty)
		 */
		public String customSynopsis(Object... params) {
			return join(ansi(), width(), adjustCJK(), commandSpec.usageMessage().customSynopsis(), new StringBuilder(),
					params).toString();
		}

		/**
		 * Returns command description text as a string. Description text can be zero or
		 * more lines, and can be specified declaratively with the
		 * {@link Command#description()} annotation attribute or programmatically by
		 * setting the Help instance's {@link Help#description} field.
		 * 
		 * @param params Arguments referenced by the format specifiers in the
		 *               description strings
		 * @return the description lines combined into a single String (which may be
		 *         empty)
		 */
		public String description(Object... params) {
			return join(ansi(), width(), adjustCJK(), commandSpec.usageMessage().description(), new StringBuilder(),
					params).toString();
		}

		/**
		 * Returns the text displayed before the description text; an empty string if
		 * there is no description, otherwise the result of
		 * {@code String.format(descriptionHeading, params)}.
		 * 
		 * @param params the parameters to use to format the description heading
		 * @return the formatted description heading
		 */
		public String descriptionHeading(Object... params) {
			return empty(commandSpec.usageMessage().descriptionHeading()) ? ""
					: createHeading(commandSpec.usageMessage().descriptionHeading(), params);
		}

		/**
		 * Generates a detailed synopsis message showing all options and parameters.
		 * Follows the unix convention of showing optional options and parameters in
		 * square brackets ({@code [ ]}).
		 * 
		 * @param optionSort            comparator to sort options or {@code null} if
		 *                              options should not be sorted
		 * @param clusterBooleanOptions {@code true} if boolean short options should be
		 *                              clustered into a single string
		 * @return a detailed synopsis
		 * @deprecated use {@link #detailedSynopsis(int, Comparator, boolean)} instead.
		 */
		@Deprecated
		public String detailedSynopsis(Comparator<OptionSpec> optionSort, boolean clusterBooleanOptions) {
			return detailedSynopsis(0, optionSort, clusterBooleanOptions);
		}

		/**
		 * Generates a detailed synopsis message showing all options and parameters.
		 * Follows the unix convention of showing optional options and parameters in
		 * square brackets ({@code [ ]}).
		 * 
		 * @param synopsisHeadingLength the length of the synopsis heading that will be
		 *                              displayed on the same line
		 * @param optionSort            comparator to sort options or {@code null} if
		 *                              options should not be sorted
		 * @param clusterBooleanOptions {@code true} if boolean short options should be
		 *                              clustered into a single string
		 * @return a detailed synopsis
		 * @since 3.0
		 */
		public String detailedSynopsis(int synopsisHeadingLength, Comparator<OptionSpec> optionSort,
				boolean clusterBooleanOptions) {
			final Set<ArgSpec> argsInGroups = new HashSet<ArgSpec>();
			final Text groupsText = createDetailedSynopsisGroupsText(argsInGroups);
			final Text optionText = createDetailedSynopsisOptionsText(argsInGroups, optionSort, clusterBooleanOptions);
			final Text endOfOptionsText = createDetailedSynopsisEndOfOptionsText();
			final Text positionalParamText = createDetailedSynopsisPositionalsText(argsInGroups);
			final Text commandText = createDetailedSynopsisCommandText();

			return makeSynopsisFromParts(synopsisHeadingLength, optionText, groupsText, endOfOptionsText,
					positionalParamText, commandText);
		}

		/**
		 * Returns the section of the usage help message that lists the {@code --} End
		 * of Options delimiter and its description.
		 * 
		 * @return the section of the usage help message that lists the {@code --} End
		 *         of Options delimiter and its description.
		 * @since 4.3
		 */
		public String endOfOptionsList() {
			if (!commandSpec.usageMessage.showEndOfOptionsDelimiterInUsageHelp()) {
				return "";
			}
			final OptionSpec endOfOptionsOption = ParserSpec.DEFAULT_END_OF_OPTIONS_DELIMITER
					.equals(commandSpec.parser().endOfOptionsDelimiter()) ? END_OF_OPTIONS_OPTION
							: createEndOfOptionsOption(commandSpec.parser().endOfOptionsDelimiter());
			endOfOptionsOption.commandSpec = this.commandSpec; // needed for interpolation
			try {
				endOfOptionsOption.messages(commandSpec.usageMessage().messages());
				final Layout layout = createDefaultLayout();
				layout.addOption(endOfOptionsOption, parameterLabelRenderer());
				return layout.toString();
			} finally {
				endOfOptionsOption.commandSpec = null;
			}
		}

		private List<OptionSpec> excludeHiddenAndGroupOptions(List<OptionSpec> all) {
			final List<OptionSpec> result = new ArrayList<OptionSpec>(all);
			for (final ArgGroupSpec group : optionSectionGroups()) {
				result.removeAll(group.allOptionsNested());
			}
			for (final Iterator<OptionSpec> iter = result.iterator(); iter.hasNext();) {
				if (iter.next().hidden()) {
					iter.remove();
				}
			}
			return result;
		}

		private List<PositionalParamSpec> excludeHiddenAndGroupParams(List<PositionalParamSpec> all) {
			final List<PositionalParamSpec> result = new ArrayList<PositionalParamSpec>(all);
			for (final ArgGroupSpec group : optionSectionGroups()) {
				result.removeAll(group.allPositionalParametersNested());
			}
			for (final Iterator<PositionalParamSpec> iter = result.iterator(); iter.hasNext();) {
				if (iter.next().hidden()) {
					iter.remove();
				}
			}
			return result;
		}

		/**
		 * Returns a 2-column list with exit codes and their description. Descriptions
		 * containing {@code "%n"} line separators are broken up into multiple lines.
		 * 
		 * @return a usage help section describing the exit codes
		 * @since 4.0
		 */
		public String exitCodeList() {
			return createTextTable(commandSpec.usageMessage().exitCodeList()).toString();
		}

		/**
		 * Returns the text displayed before the exit code list text; the result of
		 * {@code String.format(exitCodeHeading, params)}.
		 * 
		 * @param params the parameters to use to format the exit code heading
		 * @return the formatted heading of the exit code section of the usage help
		 *         message
		 * @since 4.0
		 */
		public String exitCodeListHeading(Object... params) {
			return createHeading(commandSpec.usageMessage().exitCodeListHeading(), params);
		}

		/**
		 * Returns command footer text as a string. Footer text can be zero or more
		 * lines, and can be specified declaratively with the {@link Command#footer()}
		 * annotation attribute or programmatically by setting the Help instance's
		 * {@link Help#footer} field.
		 * 
		 * @param params Arguments referenced by the format specifiers in the footer
		 *               strings
		 * @return the footer lines combined into a single String (which may be empty)
		 */
		public String footer(Object... params) {
			return join(ansi(), width(), adjustCJK(), commandSpec.usageMessage().footer(), new StringBuilder(), params)
					.toString();
		}

		/**
		 * Returns the text displayed before the footer text; the result of
		 * {@code String.format(footerHeading, params)}.
		 * 
		 * @param params the parameters to use to format the footer heading
		 * @return the formatted footer heading
		 */
		public String footerHeading(Object... params) {
			return createHeading(commandSpec.usageMessage().footerHeading(), params);
		}

		/**
		 * Returns the full usage synopsis of this command. This is equivalent to:
		 * {@code this.synopsisHeading() + this.synopsis(this.synopsisHeadingLength())}
		 * 
		 * @since 4.1
		 */
		public String fullSynopsis() {
			return this.synopsisHeading() + this.synopsis(this.synopsisHeadingLength());
		}

		/**
		 * Returns the {@code IHelpFactory} that this Help was constructed with.
		 * 
		 * @since 3.9
		 */
		private IHelpFactory getHelpFactory() {
			return commandSpec.usageMessage().helpFactory();
		}

		/**
		 * Returns true if the usage help should show the at file parameter in the
		 * parameter list, otherwise false.
		 * 
		 * @since 4.3
		 */
		public boolean hasAtFileParameter() {
			return commandSpec.parser.expandAtFiles() && commandSpec.usageMessage.showAtFileInUsageHelp();
		}

		/**
		 * Returns the command header text as a string. Header text can be zero or more
		 * lines, and can be specified declaratively with the {@link Command#header()}
		 * annotation attribute or programmatically by setting the Help instance's
		 * {@link Help#header} field.
		 * 
		 * @param params Arguments referenced by the format specifiers in the header
		 *               strings
		 * @return the header lines combined into a single String (which may be empty)
		 */
		public String header(Object... params) {
			return join(ansi(), width(), adjustCJK(), commandSpec.usageMessage().header(), new StringBuilder(), params)
					.toString();
		}

		/**
		 * Returns the text displayed before the header text; the result of
		 * {@code String.format(headerHeading, params)}.
		 * 
		 * @param params the parameters to use to format the header heading
		 * @return the formatted header heading
		 */
		public String headerHeading(Object... params) {
			return createHeading(commandSpec.usageMessage().headerHeading(), params);
		}

		/**
		 * Returns the detailed synopsis text by inserting the command name before the
		 * specified text with options and positional parameters details.
		 * 
		 * @param synopsisHeadingLength                   length of the synopsis heading
		 *                                                string to be displayed on the
		 *                                                same line as the first
		 *                                                synopsis line. For example, if
		 *                                                the synopsis heading is
		 *                                                {@code "Usage: "}, this value
		 *                                                is 7.
		 * @param optionsAndPositionalsAndCommandsDetails formatted string with options,
		 *                                                positional parameters and
		 *                                                subcommands. Follows the unix
		 *                                                convention of showing optional
		 *                                                options and parameters in
		 *                                                square brackets ({@code [ ]}).
		 * @return the detailed synopsis text, in multiple lines if the length exceeds
		 *         the usage width
		 */
		protected String insertSynopsisCommandName(int synopsisHeadingLength,
				Text optionsAndPositionalsAndCommandsDetails) {
			if (synopsisHeadingLength < 0) {
				throw new IllegalArgumentException(
						"synopsisHeadingLength must be a positive number but was " + synopsisHeadingLength);
			}

			// Fix for #142: first line of synopsis overshoots width
			final String commandName = commandSpec.qualifiedName();

			// Fix for #739: infinite loop if firstColumnLength >= width (so 2nd column
			// width becomes zero or negative)
			int indent = synopsisHeadingLength + commandName.length() + 1; // +1 for space after command name
			if (indent > commandSpec.usageMessage().synopsisAutoIndentThreshold() * width()) {
				indent = commandSpec.usageMessage().synopsisIndent() < 0 ? synopsisHeadingLength
						: commandSpec.usageMessage().synopsisIndent();
				indent = Math.min(indent, (int) (UsageMessageSpec.MAX_SYNOPSIS_AUTO_INDENT_THRESHOLD * width()));
			}
			final TextTable textTable = TextTable.forColumnWidths(colorScheme, width());
			textTable.setAdjustLineBreaksForWideCJKCharacters(
					commandSpec.usageMessage().adjustLineBreaksForWideCJKCharacters());
			textTable.indentWrappedLines = indent;

			// right-adjust the command name by length of synopsis heading
			final Text PADDING = Ansi.OFF.new Text(stringOf('X', synopsisHeadingLength),
					optionsAndPositionalsAndCommandsDetails.colorScheme);
			textTable.addRowValues(PADDING.concat(colorScheme.commandText(commandName))
					.concat(optionsAndPositionalsAndCommandsDetails));
			return textTable.toString().substring(synopsisHeadingLength); // cut off leading synopsis heading spaces
		}

		/**
		 * Concatenates the command name and the specified synopsis parts and returns a
		 * fully rendered synopsis String.
		 * 
		 * @param synopsisHeadingLength length of the synopsis heading string to be
		 *                              displayed on the same line as the first synopsis
		 *                              line. For example, if the synopsis heading is
		 *                              {@code "Usage: "}, this value is 7.
		 * @param optionText            the Ansi.Text object with the rendered options
		 *                              list (excluding the argument groups)
		 * @param groupsText            the Ansi.Text object showing the rendered
		 *                              argument groups
		 * @param endOfOptionsText      the Ansi.Text object containing the end of
		 *                              options delimiter (if enabled)
		 * @param positionalParamText   the Ansi.Text object showing the rendered
		 *                              positional parameters
		 * @param commandText           the Ansi.Text object showing the subcommands
		 *                              part of the synopsis
		 * @return a fully rendered synopsis String
		 * @since 4.4
		 */
		protected String makeSynopsisFromParts(int synopsisHeadingLength, Text optionText, Text groupsText,
				Text endOfOptionsText, Text positionalParamText, Text commandText) {
			boolean positionalsOnly = true;
			for (final ArgGroupSpec group : commandSpec().argGroups()) {
				if (group.validate()) { // non-validating groups are not shown in the synopsis
					positionalsOnly &= group.allOptionsNested().isEmpty();
				}
			}
			Text text;
			if (positionalsOnly) { // show end-of-options delimiter before the (all-positional params) groups
				text = optionText.concat(endOfOptionsText).concat(groupsText).concat(positionalParamText)
						.concat(commandText);
			} else {
				text = optionText.concat(groupsText).concat(endOfOptionsText).concat(positionalParamText)
						.concat(commandText);
			}
			return insertSynopsisCommandName(synopsisHeadingLength, text);
		}

		/**
		 * <p>
		 * Returns a description of {@linkplain #options() all options} in this command,
		 * including any argument groups.
		 * </p>
		 * <p>
		 * This implementation {@linkplain #createShortOptionNameComparator() sorts
		 * options alphabetically}, and shows only the {@linkplain Option#hidden()
		 * non-hidden} options in a {@linkplain TextTable tabular format} using the
		 * {@linkplain #createDefaultOptionRenderer() default renderer} and
		 * {@linkplain Layout default layout}.
		 * </p>
		 * 
		 * @return the fully formatted option list, including any argument groups
		 * @see #optionListExcludingGroups(List)
		 * @see #optionListGroupSections()
		 */
		public String optionList() {
			return optionList(createDefaultLayout(), createDefaultOptionSort(), parameterLabelRenderer());
		}

		/**
		 * Sorts all {@code Options} with the specified {@code comparator} (if the
		 * comparator is non-{@code null}), then
		 * {@linkplain Layout#addOption(CommandLine.Model.OptionSpec, CommandLine.Help.IParamLabelRenderer)
		 * adds} all non-hidden options to the specified TextTable and returns the
		 * result of TextTable.toString().
		 * 
		 * @param layout             the layout responsible for rendering the option
		 *                           list
		 * @param valueLabelRenderer used for options with a parameter
		 * @return the fully formatted option list, including any argument groups
		 * @see #optionListExcludingGroups(List, Layout, Comparator,
		 *      IParamLabelRenderer)
		 * @see #optionListGroupSections()
		 * @since 3.0
		 */
		public String optionList(Layout layout, Comparator<OptionSpec> optionSort,
				IParamLabelRenderer valueLabelRenderer) {
			final List<OptionSpec> visibleOptionsNotInGroups = excludeHiddenAndGroupOptions(options());
			return optionListExcludingGroups(visibleOptionsNotInGroups, layout, optionSort, valueLabelRenderer)
					+ optionListGroupSections();
		}

		/**
		 * <p>
		 * Returns a description of the specified list of options.
		 * </p>
		 * <p>
		 * This implementation {@linkplain #createShortOptionNameComparator() sorts
		 * options alphabetically}, and shows only the specified options in a
		 * {@linkplain TextTable tabular format} using the
		 * {@linkplain #createDefaultOptionRenderer() default renderer} and
		 * {@linkplain #createDefaultLayout(List, List, ColorScheme)} default layout}.
		 * </p>
		 * <p>
		 * Argument groups are not rendered by this method.
		 * </p>
		 * 
		 * @param options the options to display in the returned rendered section of the
		 *                usage help message
		 * @return the fully formatted portion of the option list for the specified
		 *         options only (argument groups are not included)
		 * @see #optionListExcludingGroups(List, Layout, Comparator,
		 *      IParamLabelRenderer)
		 * @since 4.4
		 */
		public String optionListExcludingGroups(List<OptionSpec> options) {
			return optionListExcludingGroups(options, createDefaultLayout(), createDefaultOptionSort(),
					parameterLabelRenderer());
		}

		/**
		 * Sorts all {@code Options} with the specified {@code comparator} (if the
		 * comparator is non-{@code null}), then
		 * {@linkplain Layout#addOption(CommandLine.Model.OptionSpec, CommandLine.Help.IParamLabelRenderer)
		 * adds} the specified options to the specified TextTable and returns the result
		 * of TextTable.toString(). Argument groups are not rendered by this method.
		 * 
		 * @param optionList         the options to show (this may be a subset of the
		 *                           options in this command); it is the responsibility
		 *                           of the caller to remove options that should not be
		 *                           displayed
		 * @param layout             the layout responsible for rendering the option
		 *                           list
		 * @param valueLabelRenderer used for options with a parameter
		 * @return the fully formatted portion of the option list for the specified
		 *         options only (argument groups are not included)
		 * @since 4.4
		 */
		public String optionListExcludingGroups(List<OptionSpec> optionList, Layout layout,
				Comparator<OptionSpec> optionSort, IParamLabelRenderer valueLabelRenderer) {
			final List<OptionSpec> options = new ArrayList<OptionSpec>(optionList); // options are stored in order of
																					// declaration
			if (optionSort != null) {
				Collections.sort(options, optionSort); // default: sort options ABC
			}

			layout.addAllOptions(options, valueLabelRenderer);
			return layout.toString();
		}

		/**
		 * Returns a rendered section of the usage help message that contains the
		 * argument groups that have a non-{@code null} {@linkplain ArgGroup#heading()
		 * heading}. This is usually shown below the "normal" options of the command
		 * (that are not in an argument group).
		 * 
		 * @return the fully formatted portion of the option list showing the argument
		 *         groups
		 * @since 4.4
		 * @see #optionList()
		 * @see #optionListExcludingGroups(List)
		 * @see #optionSectionGroups()
		 */
		public String optionListGroupSections() {
			return optionListGroupSections(optionSectionGroups(), createDefaultOptionSort(), parameterLabelRenderer());
		}

		// @since 4.4
		private String optionListGroupSections(List<ArgGroupSpec> groupList, Comparator<OptionSpec> optionSort,
				IParamLabelRenderer paramLabelRenderer) {
			final Set<ArgSpec> done = new HashSet<ArgSpec>();
			final List<ArgGroupSpec> groups = new ArrayList<ArgGroupSpec>(groupList);
			Collections.sort(groups, new SortByOrder<ArgGroupSpec>());

			final StringBuilder sb = new StringBuilder();
			for (final ArgGroupSpec group : groups) {
				final List<OptionSpec> groupOptions = new ArrayList<OptionSpec>(group.allOptionsNested());
				if (optionSort != null) {
					Collections.sort(groupOptions, optionSort);
				}
				groupOptions.removeAll(done);
				done.addAll(groupOptions);

				final List<PositionalParamSpec> groupPositionals = new ArrayList<PositionalParamSpec>(
						group.allPositionalParametersNested());
				groupPositionals.removeAll(done);
				done.addAll(groupPositionals);

				final Layout groupLayout = createDefaultLayout(); // create new empty layout based on the length of all
																	// options/positionals in this command
				groupLayout.addPositionalParameters(groupPositionals, paramLabelRenderer);
				groupLayout.addOptions(groupOptions, paramLabelRenderer);

				sb.append(createHeading(group.heading()));
				sb.append(groupLayout);
			}
			return sb.toString();
		}

		/**
		 * Returns the text displayed before the option list; an empty string if there
		 * are no options, otherwise the result of
		 * {@code String.format(optionListHeading, params)}.
		 * 
		 * @param params the parameters to use to format the option list heading
		 * @return the formatted option list heading
		 */
		public String optionListHeading(Object... params) {
			boolean hasVisibleOption = false;
			for (final OptionSpec option : commandSpec.options()) {
				hasVisibleOption |= !option.hidden();
			}
			if (commandSpec.usageMessage().showEndOfOptionsDelimiterInUsageHelp() || hasVisibleOption) {
				return createHeading(commandSpec.usageMessage().optionListHeading(), params);
			}
			return "";
		}

		List<OptionSpec> options() {
			return commandSpec.options();
		}

		/**
		 * Returns the list of {@code ArgGroupSpec} instances in this command that have
		 * a non-{@code null} heading, most deeply nested argument groups first.
		 * 
		 * @see #optionListGroupSections()
		 * @since 4.4
		 */
		public List<ArgGroupSpec> optionSectionGroups() {
			final List<ArgGroupSpec> result = new ArrayList<ArgGroupSpec>();
			optionSectionGroups(commandSpec.argGroups(), result);
			return result;
		}

		/**
		 * Option and positional parameter value label renderer used for the synopsis
		 * line(s) and the option list. By default initialized to the result of
		 * {@link #createDefaultParamLabelRenderer()}, which takes a snapshot of the
		 * {@link ParserSpec#separator()} at construction time. If the separator is
		 * modified after Help construction, you may need to re-initialize this field by
		 * calling {@link #createDefaultParamLabelRenderer()} again.
		 */
		public IParamLabelRenderer parameterLabelRenderer() {
			return parameterLabelRenderer;
		}

		/**
		 * Returns the rendered positional parameters section of the usage help message
		 * for {@linkplain #positionalParameters() all positional parameters} in this
		 * command.
		 * 
		 * @return the section of the usage help message that lists the parameters
		 * @see #parameterList(List)
		 */
		public String parameterList() {
			return parameterList(excludeHiddenAndGroupParams(positionalParameters()));
		}

		/**
		 * Returns the rendered section of the usage help message that lists all
		 * positional parameters in this command with their descriptions.
		 * 
		 * @param layout             the layout to use
		 * @param paramLabelRenderer for rendering parameter names
		 * @return the section of the usage help message that lists the parameters
		 */
		public String parameterList(Layout layout, IParamLabelRenderer paramLabelRenderer) {
			return parameterList(excludeHiddenAndGroupParams(positionalParameters()), layout, paramLabelRenderer);
		}

		/**
		 * Returns the rendered positional parameters section of the usage help message
		 * for the specified positional parameters.
		 * 
		 * @param positionalParams the positional parameters to display in the returned
		 *                         rendered section of the usage help message; the
		 *                         caller is responsible for removing parameters that
		 *                         should not be displayed
		 * @return the section of the usage help message that lists the parameters
		 * @see #parameterList(List, Layout, IParamLabelRenderer)
		 * @since 4.4
		 */
		public String parameterList(List<PositionalParamSpec> positionalParams) {
			return parameterList(positionalParams, createDefaultLayout(), parameterLabelRenderer());
		}

		/**
		 * Returns the rendered section of the usage help message that lists the
		 * specified parameters with their descriptions.
		 * 
		 * @param positionalParams   the positional parameters to display in the
		 *                           returned rendered section of the usage help
		 *                           message; the caller is responsible for removing
		 *                           parameters that should not be displayed
		 * @param layout             the layout to use
		 * @param paramLabelRenderer for rendering parameter names
		 * @return the section of the usage help message that lists the parameters
		 * @since 4.4
		 */
		public String parameterList(List<PositionalParamSpec> positionalParams, Layout layout,
				IParamLabelRenderer paramLabelRenderer) {
			layout.addAllPositionalParameters(positionalParams, paramLabelRenderer);
			return layout.toString();
		}

		/**
		 * Returns the text displayed before the positional parameter list; an empty
		 * string if there are no positional parameters, otherwise the result of
		 * {@code String.format(parameterListHeading, params)}.
		 * 
		 * @param params the parameters to use to format the parameter list heading
		 * @return the formatted parameter list heading
		 */
		public String parameterListHeading(Object... params) {
			if (hasAtFileParameter() || !commandSpec.positionalParameters().isEmpty()) {
				return createHeading(commandSpec.usageMessage().parameterListHeading(), params);
			}
			return "";
		}

		List<PositionalParamSpec> positionalParameters() {
			return commandSpec.positionalParameters();
		}

		private void registerSubcommands(Map<String, CommandLine> subcommands) {
			// first collect aliases
			final Map<CommandLine, List<String>> done = new IdentityHashMap<CommandLine, List<String>>();
			for (final CommandLine cmd : subcommands.values()) {
				if (!done.containsKey(cmd)) {
					done.put(cmd, new ArrayList<String>(Arrays.asList(cmd.commandSpec.aliases())));
				}
			}
			// then loop over all names that the command was registered with and add this
			// name to the front of the list (if it isn't already in the list)
			for (final Map.Entry<String, CommandLine> entry : subcommands.entrySet()) {
				final List<String> aliases = done.get(entry.getValue());
				if (!aliases.contains(entry.getKey())) {
					aliases.add(0, entry.getKey());
				}
			}
			// The aliases list for each command now has at least one entry, with the main
			// name at the front.
			// Now we loop over the commands in the order that they were registered on their
			// parent command.
			for (final Map.Entry<String, CommandLine> entry : subcommands.entrySet()) {
				// not registering hidden commands is easier than suppressing display in
				// Help.commandList():
				// if all subcommands are hidden, help should not show command list header
				final CommandLine commandLine = entry.getValue();
				final List<String> commandNames = done.remove(commandLine);
				if (commandNames == null) {
					continue;
				} // we already processed this command by another alias
				final String key = commandNames.toString().substring(1, commandNames.toString().length() - 1);
				final Help sub = getHelpFactory().create(commandLine.commandSpec, colorScheme)
						.withCommandNames(commandNames);
				allCommands.put(key, sub);
				if (!sub.commandSpec().usageMessage().hidden()) {
					visibleCommands.put(key, sub);
				}
			}
		}

		/**
		 * Returns the map of non-hidden subcommand {@code Help} instances for this
		 * command Help.
		 * 
		 * @since 3.9
		 * @see #allSubcommands()
		 */
		public Map<String, Help> subcommands() {
			return Collections.unmodifiableMap(visibleCommands);
		}

		/**
		 * Returns a synopsis for the command without reserving space for the synopsis
		 * heading.
		 * 
		 * @return a synopsis
		 * @see #abbreviatedSynopsis()
		 * @see #detailedSynopsis(Comparator, boolean)
		 * @deprecated use {@link #synopsis(int)} instead
		 */
		@Deprecated
		public String synopsis() {
			return synopsis(0);
		}

		/**
		 * Returns a synopsis for the command, reserving the specified space for the
		 * synopsis heading.
		 * 
		 * @param synopsisHeadingLength the length of the synopsis heading that will be
		 *                              displayed on the same line
		 * @return a synopsis
		 * @see #abbreviatedSynopsis()
		 * @see #detailedSynopsis(Comparator, boolean)
		 * @see #synopsisHeading
		 */
		public String synopsis(int synopsisHeadingLength) {
			if (!empty(commandSpec.usageMessage().customSynopsis())) {
				return customSynopsis();
			}
			final Comparator<OptionSpec> sortStrategy = commandSpec.usageMessage().sortSynopsis()
					? createShortOptionArityAndNameComparator() // alphabetic sort
					: createOrderComparatorIfNecessary(commandSpec.options()); // explicit sort
			final boolean clusterBooleanOptions = commandSpec.parser().posixClusteredShortOptionsAllowed();
			return commandSpec.usageMessage().abbreviateSynopsis() ? abbreviatedSynopsis()
					: detailedSynopsis(synopsisHeadingLength, sortStrategy, clusterBooleanOptions);
		}

		/**
		 * Returns the text displayed before the synopsis text; the result of
		 * {@code String.format(synopsisHeading, params)}.
		 * 
		 * @param params the parameters to use to format the synopsis heading
		 * @return the formatted synopsis heading
		 */
		public String synopsisHeading(Object... params) {
			return createHeading(commandSpec.usageMessage().synopsisHeading(), params);
		}

		/**
		 * Returns the number of characters the synopsis heading will take on the same
		 * line as the synopsis.
		 * 
		 * @return the number of characters the synopsis heading will take on the same
		 *         line as the synopsis.
		 * @see #detailedSynopsis(int, Comparator, boolean)
		 */
		public int synopsisHeadingLength() {
			final String[] lines = Ansi.OFF.new Text(commandSpec.usageMessage().synopsisHeading()).toString()
					.split("\\r?\\n|\\r|%n", -1);
			return lines[lines.length - 1].length();
		}

		private int width() {
			return commandSpec.usageMessage().width();
		}

		Help withCommandNames(List<String> aliases) {
			this.aliases = aliases;
			return this;
		}
	}

	/**
	 * Help command that can be installed as a subcommand on all application
	 * commands. When invoked with a subcommand argument, it prints usage help for
	 * the specified subcommand. For example:
	 * 
	 * <pre>
	 *
	 * // print help for subcommand
	 * command help subcommand
	 * </pre>
	 * <p>
	 * When invoked without additional parameters, it prints usage help for the
	 * parent command. For example:
	 * </p>
	 * 
	 * <pre>
	 *
	 * // print help for command
	 * command help
	 * </pre>
	 * 
	 * For {@linkplain Messages internationalization}: this command has a
	 * {@code --help} option with {@code descriptionKey = "helpCommand.help"}, and a
	 * {@code COMMAND} positional parameter with
	 * {@code descriptionKey = "helpCommand.command"}.
	 * 
	 * @since 3.0
	 */
	@Command(name = "help", header = "Display help information about the specified command.", synopsisHeading = "%nUsage: ", helpCommand = true, description = {
			"%nWhen no COMMAND is given, the usage help for the main command is displayed.",
			"If a COMMAND is specified, the help for that command is shown.%n" })
	public static final class HelpCommand implements IHelpCommandInitializable, IHelpCommandInitializable2, Runnable {

		@Option(names = { "-h",
				"--help" }, usageHelp = true, descriptionKey = "helpCommand.help", description = "Show usage help for the help command and exit.")
		private boolean helpRequested;

		@Parameters(paramLabel = "COMMAND", arity = "0..1", descriptionKey = "helpCommand.command", description = "The COMMAND to display the usage help message for.")
		private String commands;

		private CommandLine self;
		private PrintStream out;
		private PrintStream err;
		private PrintWriter outWriter;
		private PrintWriter errWriter;
		private Help.Ansi ansi; // for backwards compatibility with pre-4.0
		private Help.ColorScheme colorScheme;

		/** {@inheritDoc} */
		@Override
		@Deprecated
		public void init(CommandLine helpCommandLine, Help.Ansi ansi, PrintStream out, PrintStream err) {
			this.self = Assert.notNull(helpCommandLine, "helpCommandLine");
			this.ansi = Assert.notNull(ansi, "ansi");
			this.out = Assert.notNull(out, "out");
			this.err = Assert.notNull(err, "err");
		}

		/** {@inheritDoc} */
		@Override
		public void init(CommandLine helpCommandLine, Help.ColorScheme colorScheme, PrintWriter out, PrintWriter err) {
			this.self = Assert.notNull(helpCommandLine, "helpCommandLine");
			this.colorScheme = Assert.notNull(colorScheme, "colorScheme");
			this.outWriter = Assert.notNull(out, "outWriter");
			this.errWriter = Assert.notNull(err, "errWriter");
		}

		/**
		 * Invokes {@link #usage(PrintStream, Help.ColorScheme) usage} for the specified
		 * command, or for the parent command.
		 */
		@Override
		public void run() {
			final CommandLine parent = self == null ? null : self.getParent();
			if (parent == null) {
				return;
			}
			final Help.ColorScheme colors = colorScheme != null ? colorScheme : Help.defaultColorScheme(ansi);
			if (commands != null) {
				final Map<String, CommandLine> parentSubcommands = parent.getCommandSpec().subcommands();
				CommandLine subcommand = parentSubcommands.get(commands);
				if (subcommand == null && parent.isAbbreviatedSubcommandsAllowed()) {
					subcommand = AbbreviationMatcher
							.match(parentSubcommands, commands, parent.isSubcommandsCaseInsensitive(), self).getValue();
				}
				if (subcommand != null) {
					if (outWriter != null) {
						subcommand.usage(outWriter, colors);
					} else {
						subcommand.usage(out, colors); // for compatibility with pre-4.0 clients
					}
				} else {
					throw new ParameterException(parent, "Unknown subcommand '" + commands + "'.", null, commands);
				}
			} else {
				if (outWriter != null) {
					parent.usage(outWriter, colors);
				} else {
					parent.usage(out, colors); // for compatibility with pre-4.0 clients
				}
			}
		}
	}

	/**
	 * Provides default value for a command. Commands may configure a provider with
	 * the {@link Command#defaultValueProvider()} annotation attribute.
	 * 
	 * @since 3.6
	 */
	public interface IDefaultValueProvider {

		/**
		 * Returns the default value for an option or positional parameter or
		 * {@code null}. The returned value is converted to the type of the
		 * option/positional parameter via the same type converter used when populating
		 * this option/positional parameter from a command line argument.
		 * 
		 * @param argSpec the option or positional parameter, never {@code null}
		 * @return the default value for the option or positional parameter, or
		 *         {@code null} if this provider has no default value for the specified
		 *         option or positional parameter
		 * @throws Exception when there was a problem obtaining the default value
		 */
		String defaultValue(ArgSpec argSpec) throws Exception;
	}

	/**
	 * Represents a function that can handle a {@code ParameterException} that
	 * occurred while {@linkplain #parse(String...) parsing} the command line
	 * arguments. This is a <a href=
	 * "https://docs.oracle.com/javase/8/docs/api/java/util/function/package-summary.html">functional
	 * interface</a> whose functional method is
	 * {@link #handleException(CommandLine.ParameterException, PrintStream, CommandLine.Help.Ansi, String...)}.
	 * <p>
	 * Implementations of this function can be passed to the
	 * {@link #parseWithHandlers(IParseResultHandler, PrintStream, Help.Ansi, IExceptionHandler, String...)
	 * CommandLine::parseWithHandlers} methods to handle situations when the command
	 * line could not be parsed.
	 * </p>
	 * 
	 * @deprecated see {@link #execute(String...)},
	 *             {@link IParameterExceptionHandler} and
	 *             {@link IExecutionExceptionHandler}
	 * @since 2.0
	 */
	@Deprecated
	public interface IExceptionHandler {
		/**
		 * Handles a {@code ParameterException} that occurred while
		 * {@linkplain #parse(String...) parsing} the command line arguments and
		 * optionally returns a list of results.
		 * 
		 * @param ex   the ParameterException describing the problem that occurred while
		 *             parsing the command line arguments, and the CommandLine
		 *             representing the command or subcommand whose input was invalid
		 * @param out  the {@code PrintStream} to print help to if requested
		 * @param ansi for printing help messages using ANSI styles and colors
		 * @param args the command line arguments that could not be parsed
		 * @return a list of results, or an empty list if there are no results
		 */
		List<Object> handleException(ParameterException ex, PrintStream out, Help.Ansi ansi, String... args);
	}

	/**
	 * Classes implementing this interface know how to handle
	 * {@code ParameterExceptions} (usually from invalid user input) and
	 * {@code ExecutionExceptions} that occurred while executing the
	 * {@code Runnable} or {@code Callable} command.
	 * <p>
	 * Implementations of this interface can be passed to the
	 * {@link #parseWithHandlers(IParseResultHandler2, IExceptionHandler2, String...)
	 * CommandLine::parseWithHandlers} method.
	 * </p>
	 * <p>
	 * This interface replaces the {@link IParseResultHandler} interface.
	 * </p>
	 * 
	 * @param <R> the return type of this handler
	 * @see DefaultExceptionHandler
	 * @deprecated see {@link #execute(String...)},
	 *             {@link IParameterExceptionHandler} and
	 *             {@link IExecutionExceptionHandler}
	 * @since 3.0
	 */
	@Deprecated
	public interface IExceptionHandler2<R> {
		/**
		 * Handles a {@code ExecutionException} that occurred while executing the
		 * {@code Runnable} or {@code Callable} command and optionally returns a list of
		 * results.
		 * 
		 * @param ex          the ExecutionException describing the problem that
		 *                    occurred while executing the {@code Runnable} or
		 *                    {@code Callable} command, and the CommandLine representing
		 *                    the command or subcommand that was being executed
		 * @param parseResult the result of parsing the command line arguments
		 * @return an object resulting from handling the exception
		 */
		R handleExecutionException(ExecutionException ex, ParseResult parseResult);

		/**
		 * Handles a {@code ParameterException} that occurred while
		 * {@linkplain #parseArgs(String...) parsing} the command line arguments and
		 * optionally returns a list of results.
		 * 
		 * @param ex   the ParameterException describing the problem that occurred while
		 *             parsing the command line arguments, and the CommandLine
		 *             representing the command or subcommand whose input was invalid
		 * @param args the command line arguments that could not be parsed
		 * @return an object resulting from handling the exception
		 */
		R handleParseException(ParameterException ex, String[] args);
	}

	/**
	 * Classes implementing this interface know how to handle Exceptions that
	 * occurred while executing the {@code Runnable}, {@code Callable} or
	 * {@code Method} user object of the command.
	 * <p>
	 * <b>Implementation Requirements:</b>
	 * </p>
	 * <p>
	 * Implementors that need to print messages to the console should use the
	 * {@linkplain #getOut() output} and {@linkplain #getErr() error} PrintWriters,
	 * and the {@linkplain #getColorScheme() color scheme} from the CommandLine
	 * object obtained from the exception.
	 * </p>
	 * <p>
	 * <b>API Note:</b>
	 * </p>
	 * <p>
	 * This interface supersedes {@link IExceptionHandler2}.
	 * </p>
	 * <p>
	 * Example usage:
	 * </p>
	 * 
	 * <pre>
	 * IExecutionExceptionHandler errorHandler = new IExecutionExceptionHandler() {
	 * 	public int handleExecutionException(Exception ex, CommandLine commandLine, ParseResult parseResult) {
	 * 		// ex.printStackTrace(); // no stack trace
	 * 		commandLine.getErr().println(ex.getMessage());
	 * 		commandLine.usage(commandLine.getErr());
	 * 		return commandLine.getCommandSpec().exitCodeOnExecutionException();
	 * 	}
	 * };
	 * int exitCode = new CommandLine(new App()).setExecutionExceptionHandler(errorHandler).execute(args);
	 * </pre>
	 * 
	 * @see CommandLine#setExecutionExceptionHandler(IExecutionExceptionHandler)
	 * @since 4.0
	 */
	public interface IExecutionExceptionHandler {
		/**
		 * Handles an {@code Exception} that occurred while executing the
		 * {@code Runnable} or {@code Callable} command and returns an exit code
		 * suitable for returning from {@link #execute(String...)}.
		 * 
		 * @param ex          the Exception thrown by the {@code Runnable},
		 *                    {@code Callable} or {@code Method} user object of the
		 *                    command
		 * @param commandLine the CommandLine representing the command or subcommand
		 *                    where the exception occurred
		 * @param parseResult the result of parsing the command line arguments
		 * @return an exit code
		 */
		int handleExecutionException(Exception ex, CommandLine commandLine, ParseResult parseResult) throws Exception;
	}

	/**
	 * Implementations are responsible for "executing" the user input and returning
	 * an exit code. The {@link #execute(String...)} method delegates to a
	 * {@linkplain #setExecutionStrategy(IExecutionStrategy) configured} execution
	 * strategy.
	 * <p>
	 * <b>Implementation Requirements:</b>
	 * </p>
	 * <p>
	 * Implementers responsibilities are:
	 * </p>
	 * <ul>
	 * <li>From the {@code ParseResult}, select which {@code CommandSpec} should be
	 * executed. This is especially important for commands that have
	 * subcommands.</li>
	 * <li>"Execute" the selected {@code CommandSpec}. Often this means invoking a
	 * method on the spec's {@linkplain CommandSpec#userObject() user object}.</li>
	 * <li>Call {@link CommandLine#setExecutionResult(Object) setExecutionResult} to
	 * make the return value of that method invocation available to the
	 * application</li>
	 * <li>Return an exit code. Common sources of exit values are the invoked
	 * method's return value, or the user object if it implements
	 * {@link IExitCodeGenerator}.</li>
	 * </ul>
	 * <p>
	 * Implementors that need to print messages to the console should use the
	 * {@linkplain #getOut() output} and {@linkplain #getErr() error} PrintWriters,
	 * and the {@linkplain #getColorScheme() color scheme} from the CommandLine
	 * object obtained from ParseResult's CommandSpec.
	 * </p>
	 * <p>
	 * <b>API Note:</b>
	 * </p>
	 * <p>
	 * This interface supersedes {@link IParseResultHandler2}.
	 * </p>
	 * 
	 * @since 4.0
	 */
	public interface IExecutionStrategy {
		/**
		 * "Executes" the user input and returns an exit code. Execution often means
		 * invoking a method on the selected CommandSpec's
		 * {@linkplain CommandSpec#userObject() user object}, and making the return
		 * value of that invocation available via
		 * {@link CommandLine#setExecutionResult(Object) setExecutionResult}.
		 * 
		 * @param parseResult the parse result from which to select one or more
		 *                    {@code CommandSpec} instances to execute.
		 * @return an exit code
		 * @throws ParameterException if the invoked method on the CommandSpec's user
		 *                            object threw a ParameterException to signify
		 *                            invalid user input.
		 * @throws ExecutionException if any problem occurred while executing the
		 *                            command. Any exceptions (other than
		 *                            ParameterException) should be wrapped in a
		 *                            ExecutionException and not thrown as is.
		 */
		int execute(ParseResult parseResult) throws ExecutionException, ParameterException;
	}

	/**
	 * Interface that provides the appropriate exit code that will be returned from
	 * the {@link #execute(String...) execute} method for an exception that occurred
	 * during parsing or while invoking the command's Runnable, Callable, or Method.
	 * <p>
	 * Example usage:
	 * </p>
	 * 
	 * <pre>
	 * &#064;Command
	 * class FailingCommand implements Callable&lt;Void&gt; {
	 *     public Void call() throws IOException {
	 *         throw new IOException("error");
	 *     }
	 * }
	 * IExitCodeExceptionMapper mapper = new IExitCodeExceptionMapper() {
	 *     public int getExitCode(Throwable t) {
	 *         if (t instanceof IOException &amp;&amp; "error".equals(t.getMessage())) {
	 *             return 123;
	 *         }
	 *         return 987;
	 *     }
	 * }
	 *
	 * CommandLine cmd = new CommandLine(new FailingCommand());
	 * cmd.setExitCodeExceptionMapper(mapper);
	 * int exitCode = cmd.execute(args);
	 * assert exitCode == 123;
	 * System.exit(exitCode);
	 * </pre>
	 * 
	 * @see #setExitCodeExceptionMapper(IExitCodeExceptionMapper)
	 * @since 4.0
	 */
	public interface IExitCodeExceptionMapper {
		/**
		 * Returns the exit code that should be returned from the
		 * {@link #execute(String...) execute} method.
		 * 
		 * @param exception the exception that occurred during parsing or while invoking
		 *                  the command's Runnable, Callable, or Method.
		 * @return the exit code
		 */
		int getExitCode(Throwable exception);
	}

	/**
	 * {@code @Command}-annotated classes can implement this interface to specify an
	 * exit code that will be returned from the {@link #execute(String...) execute}
	 * method when the command is successfully invoked.
	 *
	 * <p>
	 * Example usage:
	 * </p>
	 * 
	 * <pre>
	 * &#064;Command
	 * class MyCommand implements Runnable, IExitCodeGenerator {
	 * 	public void run() {
	 * 		System.out.println("Hello");
	 * 	}
	 * 
	 * 	public int getExitCode() {
	 * 		return 123;
	 * 	}
	 * }
	 * CommandLine cmd = new CommandLine(new MyCommand());
	 * int exitCode = cmd.execute(args);
	 * assert exitCode == 123;
	 * System.exit(exitCode);
	 * </pre>
	 * 
	 * @since 4.0
	 */
	public interface IExitCodeGenerator {
		/**
		 * Returns the exit code that should be returned from the
		 * {@link #execute(String...) execute} method.
		 * 
		 * @return the exit code
		 */
		int getExitCode();
	}

	/**
	 * Factory for instantiating classes that are registered declaratively with
	 * annotation attributes, like {@link Command#subcommands()},
	 * {@link Option#converter()}, {@link Parameters#converter()} and
	 * {@link Command#versionProvider()}. The factory is also used to instantiate
	 * the {@code Collection} or {@code Map} implementation class for multi-value
	 * options and positional parameters with an abstract type, like
	 * {@code List<String>}.
	 * <p>
	 * You may provide a custom implementation of this interface. For example, a
	 * custom factory implementation could delegate to a dependency injection
	 * container that provides the requested instance.
	 * </p>
	 * <p>
	 * <b><em>Custom factory implementations should always fall back to the
	 * {@linkplain #defaultFactory() default factory} if instantiation
	 * failed.</em></b> For example:
	 * </p>
	 * 
	 * <pre>
	 * class MyFactory implements IFactory {
	 * 	private final ApplicationContext applicationContext = getAppContext();
	 *
	 * 	public &lt;T&gt; T create(Class&lt;T&gt; cls) throws Exception {
	 * 		try {
	 * 			applicationContext.getBean(cls);
	 * 		} catch (Exception ex) {
	 * 			CommandLine.defaultFactory().create(cls);
	 * 		}
	 * 	}
	 * }
	 * </pre>
	 * <p>
	 * Tip: custom factory implementations that have resources that need to be
	 * closed when done should consider implementing {@code java.lang.AutoCloseable}
	 * or {@code java.io.Closeable}. This allows applications to use the following
	 * idiom for configuring picocli before running their application:
	 * </p>
	 * 
	 * <pre>
	 * public static void main(String[] args) {
	 * 	int exitCode = 0;
	 * 	try (MyFactory factory = createMyFactory()) {
	 * 		exitCode = new CommandLine(MyClass.class, factory).setXxx(x) // configure the picocli parser...
	 * 				.execute(args);
	 * 	}
	 * 	System.exit(exitCode);
	 * }
	 * </pre>
	 * 
	 * @see org.rossonet.ext.picocli.CommandLine#CommandLine(Object, IFactory)
	 * @see #call(Class, IFactory, PrintStream, PrintStream, Help.Ansi, String...)
	 * @see #run(Class, IFactory, PrintStream, PrintStream, Help.Ansi, String...)
	 * @see #defaultFactory()
	 * @since 2.2
	 */
	public interface IFactory {
		/**
		 * Returns an instance of the specified class.
		 * 
		 * @param cls the class of the object to return
		 * @param <K> the type of the object to return
		 * @return the instance
		 * @throws Exception an exception detailing what went wrong when creating or
		 *                   obtaining the instance
		 */
		<K> K create(Class<K> cls) throws Exception;
	}

	/**
	 * Help commands that provide usage help for other commands can implement this
	 * interface to be initialized with the information they need.
	 * <p>
	 * The {@link #printHelpIfRequested(List, PrintStream, PrintStream, Help.Ansi)
	 * CommandLine::printHelpIfRequested} method calls the
	 * {@link #init(CommandLine, org.rossonet.ext.picocli.CommandLine.Help.Ansi, PrintStream, PrintStream)
	 * init} method on commands marked as {@link Command#helpCommand() helpCommand}
	 * before the help command's {@code run} or {@code call} method is called.
	 * </p>
	 * <p>
	 * <b>Implementation note:</b>
	 * </p>
	 * <p>
	 * If an error occurs in the {@code run} or {@code call} method while processing
	 * the help request, it is recommended custom Help commands throw a
	 * {@link ParameterException ParameterException} with a reference to the parent
	 * command. The {@link DefaultExceptionHandler DefaultExceptionHandler} will
	 * print the error message and the usage for the parent command, and will
	 * terminate with the exit code of the exception handler if one was set.
	 * </p>
	 * 
	 * @deprecated use {@link IHelpCommandInitializable2} instead
	 * @since 3.0
	 */
	@Deprecated
	public interface IHelpCommandInitializable {
		/**
		 * Initializes this object with the information needed to implement a help
		 * command that provides usage help for other commands.
		 * 
		 * @param helpCommandLine the {@code CommandLine} object associated with this
		 *                        help command. Implementors can use this to walk the
		 *                        command hierarchy and get access to the help command's
		 *                        parent and sibling commands.
		 * @param ansi            whether to use Ansi colors or not
		 * @param out             the stream to print the usage help message to
		 * @param err             the error stream to print any diagnostic messages to,
		 *                        in addition to the output from the exception handler
		 */
		@Deprecated
		void init(CommandLine helpCommandLine, Help.Ansi ansi, PrintStream out, PrintStream err);
	}

	/**
	 * Help commands that provide usage help for other commands can implement this
	 * interface to be initialized with the information they need.
	 * <p>
	 * The {@link #executeHelpRequest(List) CommandLine::printHelpIfRequested}
	 * method calls the
	 * {@link #init(CommandLine, org.rossonet.ext.picocli.CommandLine.Help.ColorScheme, PrintWriter, PrintWriter)
	 * init} method on commands marked as {@link Command#helpCommand() helpCommand}
	 * before the help command's {@code run} or {@code call} method is called.
	 * </p>
	 * <p>
	 * <b>Implementation note:</b>
	 * </p>
	 * <p>
	 * If an error occurs in the {@code run} or {@code call} method while processing
	 * the help request, it is recommended custom Help commands throw a
	 * {@link ParameterException ParameterException} with a reference to the parent
	 * command. The {@link DefaultExceptionHandler default ParameterException
	 * handler} will print the error message and the usage for the parent command.
	 * </p>
	 * 
	 * @since 4.0
	 */
	public interface IHelpCommandInitializable2 {
		/**
		 * Initializes this object with the information needed to implement a help
		 * command that provides usage help for other commands.
		 * 
		 * @param helpCommandLine the {@code CommandLine} object associated with this
		 *                        help command. Implementors can use this to walk the
		 *                        command hierarchy and get access to the help command's
		 *                        parent and sibling commands.
		 * @param colorScheme     the color scheme to use when printing help, including
		 *                        whether to use Ansi colors or not
		 * @param outWriter       the output writer to print the usage help message to
		 * @param errWriter       the error writer to print any diagnostic messages to,
		 *                        in addition to the output from the exception handler
		 */
		void init(CommandLine helpCommandLine, Help.ColorScheme colorScheme, PrintWriter outWriter,
				PrintWriter errWriter);
	}

	/**
	 * Creates the {@link Help} instance used to render the usage help message.
	 * 
	 * @since 3.9
	 */
	public interface IHelpFactory {
		/**
		 * Returns a {@code Help} instance to assist in rendering the usage help message
		 * 
		 * @param commandSpec the command to create usage help for
		 * @param colorScheme the color scheme to use when rendering usage help
		 * @return a {@code Help} instance
		 */
		Help create(CommandSpec commandSpec, Help.ColorScheme colorScheme);
	}

	/**
	 * Renders a section of the usage help message. The usage help message can be
	 * customized: use the {@link #setHelpSectionKeys(List)} and
	 * {@link #setHelpSectionMap(Map)} to change the order of sections, delete
	 * standard sections, add custom sections or replace the renderer of a standard
	 * sections with a custom one.
	 * <p>
	 * This gives complete freedom on how a usage help message section is rendered,
	 * but it also means that the section renderer is responsible for all aspects of
	 * rendering the section, including layout and emitting ANSI escape codes. The
	 * {@link Help.TextTable} and {@link Help.Ansi.Text} classes, and the
	 * {@link CommandLine.Help.Ansi#string(String)} and
	 * {@link CommandLine.Help.Ansi#text(String)} methods may be useful.
	 * </p>
	 * 
	 * @see UsageMessageSpec
	 * @since 3.9
	 */
	public interface IHelpSectionRenderer {
		/**
		 * Renders a section of the usage help, like header heading, header, synopsis
		 * heading, synopsis, description heading, description, etc.
		 * 
		 * @param help the {@code Help} instance for which to render a section
		 * @return the text for this section; may contain {@linkplain Help.Ansi ANSI}
		 *         escape codes
		 * @since 3.9
		 */
		String render(Help help);
	}

	/**
	 * Provides a way to modify how the command model is built. This is useful for
	 * applications that need to modify the model dynamically depending on the
	 * runtime environment.
	 * <p>
	 * Commands may configure a model transformer using the
	 * {@link Command#modelTransformer()} annotation attribute, or via the
	 * {@link CommandSpec#modelTransformer(IModelTransformer)} programmatic API.
	 * <p>
	 * Model transformers are invoked only once, after the full command hierarchy is
	 * constructed.
	 * 
	 * @since 4.6
	 */
	public interface IModelTransformer {
		/**
		 * Given an original CommandSpec, return the object that should be used instead.
		 * Implementors may modify the specified CommandSpec and return it, or create a
		 * full or partial copy of the specified CommandSpec, and return that, or even
		 * return a completely new CommandSpec.
		 * <p>
		 * Implementors are free to add or remove options, positional parameters,
		 * subcommands or modify the command in any other way.
		 * </p>
		 * <p>
		 * This method is called once, after the full command hierarchy is constructed,
		 * and before any command line arguments are parsed.
		 * </p>
		 * 
		 * @return the CommandSpec to use instead of the specified one
		 */
		CommandSpec transform(CommandSpec commandSpec);
	}

	/**
	 * Determines the option name transformation of {@linkplain Option#negatable()
	 * negatable} boolean options. Making an option negatable has two aspects:
	 * <ul>
	 * <li>the negative form recognized by the parser while parsing the command
	 * line</li>
	 * <li>the documentation string showing both the positive and the negative form
	 * in the usage help message</li>
	 * </ul>
	 * <p>
	 * Additionally, this transformer controls which names of a negatable option are
	 * actually negatable: for example, by default short options like {@code -v} do
	 * not have a negative form, even if the same option's long form,
	 * {@code --verbose}, may have a negative form, {@code --no-verbose}.
	 * </p>
	 * 
	 * @see RegexTransformer
	 * @since 4.0
	 */
	public interface INegatableOptionTransformer {
		/**
		 * Returns the negative form of the specified option name for the parser to
		 * recognize when parsing command line arguments.
		 * 
		 * @param optionName the option name to create a negative form for, for example
		 *                   {@code --force}
		 * @param cmd        the command that the option is part of
		 * @return the negative form of the specified option name, for example
		 *         {@code --no-force}
		 */
		String makeNegative(String optionName, CommandSpec cmd);

		/**
		 * Returns the documentation string to show in the synopsis and usage help
		 * message for the specified option. The returned value should be concise and
		 * clearly suggest that both the positive and the negative form are valid option
		 * names
		 * 
		 * @param optionName the option name to create a documentation string for, for
		 *                   example {@code --force}, or {@code -XX:+<option>}
		 * @param cmd        the command that the option is part of
		 * @return the documentation string for the negatable option, for example
		 *         {@code --[no-]force}, or {@code -XX:(+|-)<option>}
		 */
		String makeSynopsis(String optionName, CommandSpec cmd);
	}

	/**
	 * Exception indicating a problem during {@code CommandLine} initialization.
	 * 
	 * @since 2.0
	 */
	public static class InitializationException extends PicocliException {
		private static final long serialVersionUID = 8423014001666638895L;

		public InitializationException(String msg) {
			super(msg);
		}

		public InitializationException(String msg, Exception ex) {
			super(msg, ex);
		}
	}

	/**
	 * Helper class responsible for processing command line arguments.
	 */
	private class Interpreter {
		/** Value displayed in trace logs for options with echo=false. */
		private static final String MASKED_VALUE = "*****(masked)"; // see #2087
		private final Map<Class<?>, ITypeConverter<?>> converterRegistry = new HashMap<Class<?>, ITypeConverter<?>>();
		private boolean isHelpRequested;
		private int position;
		private int interactiveCount;
		private boolean endOfOptions;
		private ParseResult.Builder parseResultBuilder;

		Interpreter() {
			registerBuiltInConverters();
		}

		private void addOrExpand(String arg, List<String> arguments, Set<String> visited) {
			if (config().expandAtFiles() && !arg.equals("@") && arg.startsWith("@")) {
				arg = arg.substring(1);
				final Tracer tracer = tracer();
				if (arg.startsWith("@")) {
					if (tracer.isInfo()) {
						tracer.info("Not expanding @-escaped argument %s (trimmed leading '@' char)", arg);
					}
				} else {
					if (tracer.isInfo()) {
						tracer.info("Expanding argument file @%s", arg);
					}
					expandArgumentFile(arg, arguments, visited);
					return;
				}
			}
			arguments.add(arg);
		}

		private void addPostponedRequiredArgs(Collection<ArgSpec> inheritedRequired, List<ArgSpec> required) {
			for (final ArgSpec postponed : inheritedRequired) {
				if (postponed.isOption()) {
					final OptionSpec inherited = commandSpec.findOption(((OptionSpec) postponed).longestName());
					Assert.notNull(inherited, "inherited option " + postponed);
					required.add(inherited);
				} else {
					final PositionalParamSpec positional = (PositionalParamSpec) postponed;
					for (final PositionalParamSpec existing : commandSpec.positionalParameters()) {
						if (existing.inherited() && existing.index().equals(positional.index())
								&& existing.arity().equals(positional.arity())
								&& existing.typeInfo().equals(positional.typeInfo())
								&& Assert.equals(existing.paramLabel(), positional.paramLabel())
								&& Assert.equals(existing.hideParamSyntax(), positional.hideParamSyntax())
								&& Assert.equals(existing.required(), positional.required())
								&& Assert.equals(existing.splitRegex(), positional.splitRegex())
								&& Arrays.equals(existing.description(), positional.description())
								&& Assert.equals(existing.descriptionKey(), positional.descriptionKey())
								&& Assert.equals(existing.parameterConsumer(), positional.parameterConsumer())) {
							required.add(existing);
						}
					}
				}
			}
		}

		private void addToInitialized(ArgSpec argSpec, Set<ArgSpec> initialized) {
			initialized.add(argSpec);
			final ArgSpec rootArgSpec = argSpec.root();
			if (rootArgSpec != null) {
				initialized.add(rootArgSpec);
			}
		}

		private int addUserInputToList(ArgSpec argSpec, List<Object> result, int consumed, String argDescription) {
			final char[] input = readUserInput(argSpec);
			final String inputString = new String(input);
			if (tracer().isInfo()) {
				final String value = argSpec.echo() ? inputString + " (interactive value)"
						: "*** (masked interactive value)";
				tracer().info("Adding %s to %s for %s on %s", value, argSpec.toString(), argDescription,
						argSpec.scopeString());
			}
			final String maskedValue = getMaskedValue(argSpec, inputString);
			parseResultBuilder.addStringValue(argSpec, maskedValue);
			parseResultBuilder.addOriginalStringValue(argSpec, maskedValue);
			if (!char[].class.equals(argSpec.auxiliaryTypes()[0]) && !char[].class.equals(argSpec.type())) {
				final Object value = tryConvert(argSpec, consumed,
						getTypeConverter(argSpec.auxiliaryTypes(), argSpec, 0), new String(input), 0);
				result.add(value);
			} else {
				result.add(input);
			}
			consumed++;
			return consumed;
		}

		private boolean applyDefault(IDefaultValueProvider defaultValueProvider, ArgSpec arg) throws Exception {

			// Default value provider return value is only used if provider exists and if
			// value
			// is not null otherwise the original default or initial value are used
			final String fromProvider = defaultValueProvider == null ? null : defaultValueProvider.defaultValue(arg);
			String defaultValue = fromProvider == null ? arg.defaultValue() : arg.interpolate(fromProvider);
			final String provider = defaultValueProvider == null ? "" : (" from " + defaultValueProvider.toString());

			final String displayDefaultVal = arg.echo() ? defaultValue : MASKED_VALUE;
			final Tracer tracer = CommandLine.tracer();
			if (defaultValue != null && !ArgSpec.NULL_VALUE.equals(defaultValue)) {
				if (tracer.isDebug()) {
					tracer.debug("Applying defaultValue (%s)%s to %s on %s", displayDefaultVal, provider, arg,
							arg.scopeString());
				}
				final Range arity = arg.arity().min(Math.max(1, arg.arity().min));
				applyOption(arg, false, LookBehind.SEPARATE, false, arity, stack(defaultValue), new HashSet<ArgSpec>(),
						arg.toString);
				arg.valueIsDefaultValue = true;
			} else {
				if (arg.typeInfo().isOptional()) {
					if (tracer.isDebug()) {
						tracer.debug("Applying Optional.empty() to %s on %s", arg, arg.scopeString());
					}
					arg.setValue(getOptionalEmpty());
					arg.valueIsDefaultValue = true;
				} else if (ArgSpec.UNSPECIFIED.equals(arg.originalDefaultValue)) {
					tracer.debug("defaultValue not defined for %s", arg);
					return false;
				} else {
					if (ArgSpec.NULL_VALUE.equals(arg.originalDefaultValue)) {
						defaultValue = null;
						if (tracer.isDebug()) {
							tracer.debug("Applying defaultValue (%s)%s to %s on %s", displayDefaultVal, provider, arg,
									arg.scopeString());
						}
						arg.setValue(defaultValue);
						arg.valueIsDefaultValue = true;
						return true;
					}
					tracer.debug("defaultValue not defined for %s", arg);
				}
			}
			return defaultValue != null;
		}

		private void applyDefaultValues(List<ArgSpec> required, Set<ArgSpec> initialized) throws Exception {
			parseResultBuilder.isInitializingDefaultValues = true;
			final Tracer tracer = CommandLine.tracer();
			tracer.debug("Applying default values for command '%s'", CommandLine.this.commandSpec.qualifiedName());
			for (final ArgSpec arg : commandSpec.args()) {
				if (arg.group() == null && !initialized.contains(arg)) {
					if (arg.inherited()) {
						tracer.debug("Not applying default value for inherited %s", optionDescription("", arg, -1));
					} else {
						if (applyDefault(commandSpec.defaultValueProvider(), arg)) {
							required.remove(arg);
						}
					}
				}
			}
			for (final ArgGroupSpec group : commandSpec.argGroups()) {
				applyGroupDefaults(commandSpec.defaultValueProvider(), group, required, initialized);
			}
			parseResultBuilder.isInitializingDefaultValues = false;
		}

		private void applyGroupDefaults(IDefaultValueProvider defaultValueProvider, ArgGroupSpec group,
				List<ArgSpec> required, Set<ArgSpec> initialized) throws Exception {
			final Tracer tracer = CommandLine.tracer();
			tracer.debug("Applying default values for group '%s'", group.synopsis());
			for (final ArgSpec arg : group.args()) {
				if (arg.scope().get() != null && !initialized.contains(arg)) {
					if (arg.inherited()) {
						tracer.debug("Not applying default value for inherited %s", optionDescription("", arg, -1));
					} else {
						if (applyDefault(defaultValueProvider, arg)) {
							required.remove(arg);
						}
					}
				}
			}
			for (final ArgGroupSpec sub : group.subgroups()) {
				applyGroupDefaults(defaultValueProvider, sub, required, initialized);
			}
		}

		private int applyOption(ArgSpec argSpec, boolean negated, LookBehind lookBehind, boolean alreadyUnquoted,
				Range arity, Stack<String> args, Set<ArgSpec> initialized, String argDescription) throws Exception {
			updateHelpRequested(argSpec);

			parseResultBuilder.beforeMatchingGroupElement(argSpec); // #1004 ensure groups are initialized before
																	// calling parameter consumer

			final int originalSize = args.size();
			final Map<String, Object> info = mapOf("separator", lookBehind.toString(commandSpec.parser().separator()),
					"negated", negated, "unquoted", alreadyUnquoted, "versionHelpRequested",
					parseResultBuilder.versionHelpRequested, "usageHelpRequested",
					parseResultBuilder.usageHelpRequested);
			final boolean done = argSpec.preprocessor().preprocess(args, commandSpec, argSpec, info);
			parseResultBuilder.versionHelpRequested = (Boolean) info.get("versionHelpRequested");
			parseResultBuilder.usageHelpRequested = (Boolean) info.get("usageHelpRequested");
			negated = (Boolean) info.get("negated");
			alreadyUnquoted = (Boolean) info.get("unquoted");
			lookBehind = LookBehind.parse(String.valueOf(info.get("separator")));
			if (done) {
				return args.size() - originalSize;
			}

			if (argSpec.parameterConsumer() != null) {
				argSpec.parameterConsumer().consumeParameters(args, argSpec, commandSpec);
				addToInitialized(argSpec, initialized);
				return args.size() - originalSize;
			}
			final boolean consumeOnlyOne = commandSpec.parser().aritySatisfiedByAttachedOptionParam()
					&& lookBehind.isAttached();
			Stack<String> workingStack = args;
			if (consumeOnlyOne) {
				workingStack = args.isEmpty() ? args : stack(args.pop());
			} else {
				if (!assertNoMissingParameters(argSpec, arity, args)) {
					return 0;
				} // #389 collectErrors parsing
			}

			int result;
			if (argSpec.typeInfo().isArray()) {
				result = applyValuesToArrayField(argSpec, negated, lookBehind, alreadyUnquoted, arity, workingStack,
						initialized, argDescription);
			} else if (argSpec.typeInfo().isCollection()) {
				result = applyValuesToCollectionField(argSpec, negated, lookBehind, alreadyUnquoted, arity,
						workingStack, initialized, argDescription);
			} else if (argSpec.typeInfo().isMap()) {
				result = applyValuesToMapField(argSpec, lookBehind, alreadyUnquoted, arity, workingStack, initialized,
						argDescription);
			} else {
				result = applyValueToSingleValuedField(argSpec, negated, lookBehind, alreadyUnquoted, arity,
						workingStack, initialized, argDescription);
			}
			if (workingStack != args && !workingStack.isEmpty()) {
				args.push(workingStack.pop());
				Assert.assertTrue(workingStack.isEmpty(),
						"Working stack should be empty but was " + new ArrayList<String>(workingStack));
			}
			return result;
		}

		private int applyValuesToArrayField(ArgSpec argSpec, boolean negated, LookBehind lookBehind,
				boolean alreadyUnquoted, Range arity, Stack<String> args, Set<ArgSpec> initialized,
				String argDescription) throws Exception {
			final Object existing = argSpec.getValue();
			final int length = existing == null ? 0 : Array.getLength(existing);
			final int pos = getPosition(argSpec);
			final List<Object> converted = consumeArguments(argSpec, negated, lookBehind, alreadyUnquoted,
					alreadyUnquoted, arity, args, argDescription);
			final List<Object> newValues = new ArrayList<Object>();
			if (initialized.contains(argSpec)) { // existing values are default values if initialized does NOT contain
													// argsSpec
				for (int i = 0; i < length; i++) {
					newValues.add(Array.get(existing, i)); // keep non-default values
				}
			}
			addToInitialized(argSpec, initialized);
			for (final Object obj : converted) {
				if (obj instanceof Collection<?>) {
					newValues.addAll((Collection<?>) obj);
				} else {
					newValues.add(obj);
				}
			}
			final Object array = Array.newInstance(argSpec.auxiliaryTypes()[0], newValues.size());
			for (int i = 0; i < newValues.size(); i++) {
				Array.set(array, i, newValues.get(i));
			}
			argSpec.setValue(array);
			parseResultBuilder.add(argSpec, pos);
			return converted.size(); // return how many args were consumed
		}

		private int applyValuesToCollectionField(ArgSpec argSpec, boolean negated, LookBehind lookBehind,
				boolean alreadyUnquoted, Range arity, Stack<String> args, Set<ArgSpec> initialized,
				String argDescription) throws Exception {
			Collection<Object> collection = argSpec.getValue();
			final int pos = getPosition(argSpec);
			final List<Object> converted = consumeArguments(argSpec, negated, lookBehind, alreadyUnquoted,
					alreadyUnquoted, arity, args, argDescription);
			if (collection == null || !initialized.contains(argSpec)) {
				tracer().debug("Initializing binding for %s on %s with empty %s", optionDescription("", argSpec, 0),
						argSpec.scopeString(), argSpec.type().getSimpleName());
				collection = createCollection(argSpec.type(), argSpec.auxiliaryTypes()); // collection type, element
																							// type
				argSpec.setValue(collection);
			}
			addToInitialized(argSpec, initialized);
			for (final Object element : converted) {
				if (element instanceof Collection<?>) {
					collection.addAll((Collection<?>) element);
				} else {
					collection.add(element);
				}
			}
			parseResultBuilder.add(argSpec, pos);
			argSpec.setValue(collection);
			return converted.size();
		}

		private int applyValuesToMapField(ArgSpec argSpec, LookBehind lookBehind, boolean alreadyUnquoted, Range arity,
				Stack<String> args, Set<ArgSpec> initialized, String argDescription) throws Exception {
			if (argSpec.auxiliaryTypes().length < 2) {
				throw new ParameterException(CommandLine.this,
						argSpec.toString() + " needs two types (one for the map key, one for the value) but only has "
								+ argSpec.auxiliaryTypes().length + " types configured.",
						argSpec, null);
			}
			Map<Object, Object> map = argSpec.getValue();
			final Tracer tracer = CommandLine.tracer();
			if (map == null || !initialized.contains(argSpec)) {
				tracer.debug("Initializing binding for %s on %s with empty %s", optionDescription("", argSpec, 0),
						argSpec.scopeString(), argSpec.type().getSimpleName());
				map = createMap(argSpec.type()); // map class
				argSpec.setValue(map);
			}
			addToInitialized(argSpec, initialized);
			final int originalSize = map.size();
			final int pos = getPosition(argSpec);
			consumeMapArguments(argSpec, lookBehind, alreadyUnquoted, arity, args, map, argDescription);
			parseResultBuilder.add(argSpec, pos);
			argSpec.setValue(map);
			return map.size() - originalSize;
		}

		private int applyValueToSingleValuedField(ArgSpec argSpec, boolean negated, LookBehind lookBehind,
				boolean alreadyUnquoted, Range derivedArity, Stack<String> args, Set<ArgSpec> initialized,
				String argDescription) throws Exception {
			final Tracer tracer = CommandLine.tracer();
			final boolean noMoreValues = args.isEmpty();
			String value = noMoreValues ? null : args.pop();
			final String quotedValue = value;
			if (commandSpec.parser().trimQuotes() && !alreadyUnquoted) {
				value = unquote(value);
			}
			final Range arity = argSpec.arity().isUnspecified ? derivedArity : argSpec.arity(); // #509
			if (arity.max == 0 && !arity.isUnspecified && lookBehind == LookBehind.ATTACHED_WITH_SEPARATOR) { // #509
				throw new MaxValuesExceededException(CommandLine.this,
						optionDescription("", argSpec, 0) + " should be specified without '" + value + "' parameter");
			}
			if (arity.min > 0) {
				args.push(quotedValue);
				final boolean discontinue = assertNoMissingMandatoryParameter(argSpec, args, 0, arity) // #1055
						|| isArgResemblesOptionThereforeDiscontinue(argSpec, args, 0, arity); // #1015 #639
				args.pop();
				if (discontinue) {
					return 0;
				}
			}
			int consumed = arity.min; // the number or args we need to consume

			String actualValue = value;
			char[] interactiveValue = null;
			final Class<?> cls = argSpec.auxiliaryTypes()[0]; // field may be interface/abstract type, use annotation to
																// get concrete type
			if (arity.min <= 0) { // value may be optional
				boolean optionalValueExists = true; // assume we will use the command line value
				consumed = 1;

				// special logic for booleans: BooleanConverter accepts only "true" or "false".
				if (cls == Boolean.class || cls == Boolean.TYPE) {

					// boolean option with arity = 0..1 or 0..*: value MAY be a param
					final boolean optionalWithBooleanValue = arity.max > 0
							&& ("true".equalsIgnoreCase(value) || "false".equalsIgnoreCase(value));
					if (!optionalWithBooleanValue && lookBehind != LookBehind.ATTACHED_WITH_SEPARATOR) { // if attached,
																											// try
																											// converting
																											// the value
																											// to
																											// boolean
																											// (and fail
																											// if
																											// invalid
																											// value)
						boolean defaultValue = booleanValue(argSpec, argSpec.calcDefaultValue(true)); // #712 flip the
																										// default value
						if (argSpec.isOption() && !empty(((OptionSpec) argSpec).fallbackValue())) {
							defaultValue = !booleanValue(argSpec, ((OptionSpec) argSpec).fallbackValue()); // #754 Allow
																											// boolean
																											// options
																											// to get
																											// value
																											// from
																											// fallback
																											// instead
																											// of
																											// defaultProvider
						}
						// don't process cmdline arg: it's okay to ignore value if not attached to
						// option
						Boolean oppositeValue = commandSpec.parser().toggleBooleanFlags() ? (Boolean) argSpec.getValue() // #147
																															// toggle
																															// existing
																															// boolean
																															// value
								: defaultValue; // #712 flip the default value
						if (oppositeValue == null) {
							oppositeValue = false;
						}
						actualValue = String.valueOf(!oppositeValue);
						if (argSpec.isOption() && ((OptionSpec) argSpec).negatable() && negated) {
							actualValue = String.valueOf(oppositeValue);
						}
						optionalValueExists = false;
						consumed = 0;
					}
				} else { // non-boolean option with optional value #325, #279
					final String fallbackValue = argSpec.isOption() ? ((OptionSpec) argSpec).fallbackValue() : "";
					// #828 #1125 use varargCanConsumeNextValue(argSpec, value) to detect if value
					// can be a parameter
					if (!varargCanConsumeNextValue(argSpec, value) // not a parameter
							|| value == null) { // stack is empty, option with arity=0..1 was the last arg
						actualValue = fallbackValue;
						optionalValueExists = false;
						consumed = 0;
					}
				}
				// if argSpec is interactive, we may need to read the password from the console:
				// - if arity = 0 : ALWAYS read from console
				// - if arity = 0..1: ONLY read from console if user specified a non-option
				// value
				if (argSpec.interactive() && (arity.max == 0 || !optionalValueExists)) {
					interactiveValue = readUserInput(argSpec);
					consumed = 0;
				}
			}
			if (consumed == 0) { // optional value was not specified on command line, we made something up
				if (value != null) {
					args.push(value); // we don't consume the command line value
				}
			} else { // value was non-optional or optional value was actually specified
				// process the command line value
				if (!lookBehind.isAttached()) {
					parseResultBuilder.nowProcessing(argSpec, value);
				} // update position for Completers
			}
			String initValueMessage = "Setting %s to '%3$s' (was '%2$s') for %4$s on %5$s";
			String overwriteValueMessage = "Overwriting %s value '%s' with '%s' for %s on %s";
			Object newValue = interactiveValue != null ? interactiveValue : actualValue;
			if (noMoreValues && actualValue == null && interactiveValue == null) {
				consumed = 0;
			} else {
				consumed = 1;
				if (interactiveValue != null) {
					initValueMessage = "Setting %s to %3$s (interactive value) for %4$s on %5$s";
					overwriteValueMessage = "Overwriting %s value with %3$s (interactive value) for %s on %5$s";
				}
				// #1642 Negatable options should negate explicit values
				if ((cls == Boolean.class || cls == Boolean.TYPE) && arity.min >= 1) {
					final Boolean boolValue = booleanValue(argSpec, value);
					// note: we ignore commandSpec.parser().toggleBooleanFlags() for explicit
					// non-optional params
					if (argSpec.isOption() && ((OptionSpec) argSpec).negatable() && negated) {
						actualValue = String.valueOf(!boolValue);
					} else {
						actualValue = String.valueOf(boolValue);
					}
				}
				if (!char[].class.equals(cls) && !char[].class.equals(argSpec.type())) {
					if (interactiveValue != null) {
						actualValue = new String(interactiveValue);
					}
					final ITypeConverter<?> converter = getTypeConverter(argSpec.auxiliaryTypes(), argSpec, 0);
					newValue = tryConvert(argSpec, -1, converter, actualValue, 0);
				} else { // type is char[], no type conversion needed
					if (interactiveValue == null) { // setting command line arg to char[] field
						newValue = actualValue.toCharArray();
					} else {
						actualValue = getMaskedValue(argSpec, new String(interactiveValue)); // mask interactive value
																								// if echo is false
						newValue = interactiveValue;
					}
				}
			}
			final Object oldValue = argSpec.getValue();
			String traceMessage = initValueMessage;
			if (argSpec.group() == null && initialized.contains(argSpec)) {
				if (!isOverwrittenOptionsAllowed()) {
					throw new OverwrittenOptionException(CommandLine.this, argSpec,
							optionDescription("", argSpec, 0) + " should be specified only once");
				}
				traceMessage = overwriteValueMessage;
			}
			addToInitialized(argSpec, initialized);

			if (argSpec.typeInfo().isOptional()) {
				newValue = getOptionalOfNullable(newValue);
			}
			final String displayVal = !argSpec.interactive() || argSpec.echo() ? String.valueOf(newValue)
					: MASKED_VALUE;
			if (tracer.isInfo()) {
				tracer.info(traceMessage, argSpec.toString(), String.valueOf(oldValue), displayVal, argDescription,
						argSpec.scopeString());
			}
			final int pos = getPosition(argSpec);
			argSpec.setValue(newValue);
			parseResultBuilder.addOriginalStringValue(argSpec, actualValue);// #279 track empty string value if no
																			// command line argument was consumed
			parseResultBuilder.addStringValue(argSpec, actualValue);
			parseResultBuilder.addTypedValues(argSpec, pos, newValue);
			parseResultBuilder.add(argSpec, pos);
			return consumed;
		}

		private boolean assertNoMissingMandatoryParameter(ArgSpec argSpec, Stack<String> args, int i, Range arity) {
			if (!varargCanConsumeNextValue(argSpec, args.peek())) {
				final String msg = createMissingParameterMessageFoundOtherOption(argSpec, args, i, arity);
				maybeThrow(new MissingParameterException(CommandLine.this, argSpec, msg));
				return true;
			}
			return false;
		}

		private boolean assertNoMissingParameters(ArgSpec argSpec, Range arity, Stack<String> args) {
			if (argSpec.interactive()) {
				return true;
			}
			int available = args.size();
			if (available > 0 && commandSpec.parser().splitFirst() && argSpec.splitRegex().length() > 0) {
				available += argSpec.splitValue(args.peek(), commandSpec.parser(), arity, 0).length - 1;
			}
			if (arity.min > available) {
				List<PositionalParamSpec> missingList = Collections.emptyList();
				final List<PositionalParamSpec> positionals = commandSpec.positionalParameters();
				if (argSpec.isPositional() && positionals.contains(argSpec)) {
					missingList = positionals.subList(positionals.indexOf(argSpec), positionals.size());
				}
				final String msg = createMissingParameterMessage(argSpec, arity, missingList, args, available);
				maybeThrow(new MissingParameterException(CommandLine.this, argSpec, msg));
				return false;
			}
			return true;
		}

		private boolean booleanValue(ArgSpec argSpec, Object value) {
			if (value == null) {
				return false;
			}
			if (isOptional(value.getClass())) {
				try {
					value = value.getClass().getMethod("orElse", Object.class).invoke(value, "null");
				} catch (final Exception e) {
					throw new TypeConversionException(
							"Could not convert '" + value + "' to an Optional<Boolean>: " + e.getMessage());
				}
			}
			final String stringValue = String.valueOf(value);
			if (empty(stringValue) || "null".equals(stringValue) || "Optional.empty".equals(value)) {
				return false;
			}
			final ITypeConverter<?> converter = getTypeConverter(new Class<?>[] { Boolean.class }, argSpec, 0);
			return (Boolean) tryConvert(argSpec, -1, converter, stringValue, 0);
		}

		private boolean canConsumeOneArgument(ArgSpec argSpec, LookBehind lookBehind, boolean alreadyUnquoted,
				Range arity, int consumed, String arg, String argDescription) {
			if (char[].class.equals(argSpec.auxiliaryTypes()[0]) || char[].class.equals(argSpec.type())) {
				return true;
			}
			final ITypeConverter<?> converter = getTypeConverter(argSpec.auxiliaryTypes(), argSpec, 0);
			try {
				final String[] values = unquoteAndSplit(argSpec, lookBehind, alreadyUnquoted, arity, consumed, arg);
//                if (!argSpec.acceptsValues(values.length, commandSpec.parser())) {
//                    tracer.debug("$s would split into %s values but %s cannot accept that many values.", arg, values.length, argDescription);
//                    return false;
//                }
				for (final String value : values) {
					tryConvert(argSpec, -1, converter, value, 0);
				}
				return true;
			} catch (final PicocliException ex) { // Note: we don't mask hidden (echo=false) options to facilitate
													// debugging
				tracer().debug("%s cannot be assigned to %s: type conversion fails: %s.", arg, argDescription,
						ex.getMessage());
				return false;
			}
		}

		private boolean canConsumeOneMapArgument(ArgSpec argSpec, LookBehind lookBehind, boolean alreadyUnquoted,
				Range arity, int consumed, String arg, Class<?>[] classes, ITypeConverter<?> keyConverter,
				ITypeConverter<?> valueConverter, String argDescription) {
			final String[] values = unquoteAndSplit(argSpec, lookBehind, alreadyUnquoted, arity, consumed, arg);
			try {
				for (final String value : values) {
					final String[] keyValue = splitKeyValue(argSpec, value);
					tryConvert(argSpec, -1, keyConverter, keyValue[0], 0);
					final String mapValue = keyValue.length == 1 ? argSpec.mapFallbackValue() : keyValue[1];
					tryConvert(argSpec, -1, valueConverter, mapValue, 1);
				}
				return true;
			} catch (final PicocliException ex) {
				tracer().debug("%s cannot be assigned to %s: type conversion fails: %s.", arg, argDescription,
						ex.getMessage());
				return false;
			}
		}

		private void clear() {
			getCommandSpec().userObject(); // #690 instantiate user object when cmd matched on the command line
			position = 0;
			endOfOptions = false;
			isHelpRequested = false;
			parseResultBuilder = ParseResult.builder(getCommandSpec());
			for (final OptionSpec option : getCommandSpec().options()) {
				clear(option);
			}
			for (final PositionalParamSpec positional : getCommandSpec().positionalParameters()) {
				clear(positional);
			}
			for (final ArgGroupSpec group : getCommandSpec().argGroups()) {
				clear(group);
			}
			for (final UnmatchedArgsBinding unmatched : getCommandSpec().unmatchedArgsBindings()) {
				unmatched.clear();
			}
		}

		private void clear(ArgGroupSpec group) {
			for (final ArgSpec arg : group.args()) {
				clear(arg);
			}
			for (final ArgGroupSpec sub : group.subgroups()) {
				clear(sub);
			}
		}

		private void clear(ArgSpec argSpec) {
			argSpec.resetStringValues();
			argSpec.resetOriginalStringValues();
			argSpec.typedValues.clear();
			argSpec.typedValueAtPosition.clear();
			if (argSpec.inherited()) { // inherited args are cleared only at their origin
				tracer().debug("Not applying initial value for inherited %s", optionDescription("", argSpec, -1));
			} else { // groups do their own initialization;
				if (argSpec.group() == null) {
					argSpec.applyInitialValue();
				}
			}
		}

		private ParserSpec config() {
			return commandSpec.parser();
		}

		private List<Object> consumeArguments(ArgSpec argSpec, boolean negated, LookBehind lookBehind,
				boolean alreadyUnquoted, boolean unquoted, Range arity, Stack<String> args, String argDescription)
				throws Exception {
			final List<Object> result = new ArrayList<Object>();

			// don't modify Interpreter.position: same position may be consumed by multiple
			// ArgSpec objects
			int currentPosition = getPosition(argSpec);

			// first do the arity.min mandatory parameters
			final int initialSize = argSpec.stringValues().size();
			int consumed = consumedCount(0, initialSize, argSpec);
			for (int i = 0; consumed < arity.min && !args.isEmpty(); i++) {
				final List<Object> typedValuesAtPosition = new ArrayList<Object>();
				parseResultBuilder.addTypedValues(argSpec, currentPosition++, typedValuesAtPosition);
				if (assertNoMissingMandatoryParameter(argSpec, args, i, arity)
						|| isArgResemblesOptionThereforeDiscontinue(argSpec, args, i, arity)) {
					break;
				}
				consumeOneArgument(argSpec, lookBehind, alreadyUnquoted, arity, consumed, args.pop(),
						typedValuesAtPosition, i, argDescription);
				result.addAll(typedValuesAtPosition);
				consumed = consumedCount(i + 1, initialSize, argSpec);
				lookBehind = LookBehind.SEPARATE;
				alreadyUnquoted = false;
			}
			if (argSpec.interactive() && argSpec.arity().max == 0) {
				consumed = addUserInputToList(argSpec, result, consumed, argDescription);
			}
			// now process the varargs if any
			final String fallback = argSpec.isOption()
					&& !OptionSpec.DEFAULT_FALLBACK_VALUE.equals(((OptionSpec) argSpec).fallbackValue())
							? ((OptionSpec) argSpec).fallbackValue()
							: null;
			final boolean hasFallback = fallback != null
					|| (argSpec.isOption() && Option.NULL_VALUE.equals(((OptionSpec) argSpec).originalFallbackValue));
			if (hasFallback && consumed == 0
					&& (args.isEmpty() || !varargCanConsumeNextValue(argSpec, args.peek())
							|| (!canConsumeOneArgument(argSpec, lookBehind, alreadyUnquoted, arity, consumed,
									args.peek(), argDescription)))) {
				args.push(fallback);
			}
			for (int i = consumed; consumed < arity.max && !args.isEmpty(); i++) {
				if (argSpec.interactive() && argSpec.arity().max == 1
						&& !varargCanConsumeNextValue(argSpec, args.peek())) {
					// if interactive and arity = 0..1, we consume from command line if possible (if
					// next arg not an option or subcommand)
					consumed = addUserInputToList(argSpec, result, consumed, argDescription);
				} else {
					if (!varargCanConsumeNextValue(argSpec, args.peek())) {
						break;
					}
					final List<Object> typedValuesAtPosition = new ArrayList<Object>();
					parseResultBuilder.addTypedValues(argSpec, currentPosition++, typedValuesAtPosition);
					if (!canConsumeOneArgument(argSpec, lookBehind, alreadyUnquoted, arity, consumed, args.peek(),
							argDescription)) {
						break; // leave empty list at argSpec.typedValueAtPosition[currentPosition] so we won't
								// try to consume that position again
					}
					if (isArgResemblesOptionThereforeDiscontinue(argSpec, args, i, arity)) {
						break;
					}
					consumeOneArgument(argSpec, lookBehind, alreadyUnquoted, arity, consumed, args.pop(),
							typedValuesAtPosition, i, argDescription);
					result.addAll(typedValuesAtPosition);
					consumed = consumedCount(i + 1, initialSize, argSpec);
					lookBehind = LookBehind.SEPARATE;
					alreadyUnquoted = false;
				}
			}
			if (result.isEmpty() && arity.min == 0 && arity.max <= 1 && isBoolean(argSpec.auxiliaryTypes())) {
				if (argSpec.isOption() && ((OptionSpec) argSpec).negatable()) {
					final Object defaultValue = argSpec.calcDefaultValue(true);
					boolean booleanDefault = false;
					if (defaultValue instanceof String) {
						booleanDefault = Boolean.parseBoolean(String.valueOf(defaultValue));
					}
					if (negated) {
						return Collections.singletonList((Object) booleanDefault);
					} else {
						return Collections.singletonList((Object) !booleanDefault);
					}
				} else {
					return Collections.singletonList((Object) Boolean.TRUE);
				}
			}
			return result;
		}

		private int consumedCount(int i, int initialSize, ArgSpec arg) {
			return commandSpec.parser().splitFirst() ? arg.stringValues().size() - initialSize : i;
		}

		private int consumedCountMap(int i, int initialSize, ArgSpec arg) {
			return commandSpec.parser().splitFirst() ? (arg.stringValues().size() - initialSize) / 2 : i;
		}

		private void consumeMapArguments(ArgSpec argSpec, LookBehind lookBehind, boolean alreadyUnquoted, Range arity,
				Stack<String> args, Map<Object, Object> result, String argDescription) throws Exception {

			// don't modify Interpreter.position: same position may be consumed by multiple
			// ArgSpec objects
			int currentPosition = getPosition(argSpec);

			final Class<?>[] classes = argSpec.auxiliaryTypes();
			final ITypeConverter<?> keyConverter = getTypeConverter(classes, argSpec, 0);
			final ITypeConverter<?> valueConverter = getTypeConverter(classes, argSpec, 1);

			// first do the arity.min mandatory parameters
			final int initialSize = argSpec.stringValues().size();
			int consumed = consumedCountMap(0, initialSize, argSpec);
			for (int i = 0; consumed < arity.min && !args.isEmpty(); i++) {
				final Map<Object, Object> typedValuesAtPosition = new LinkedHashMap<Object, Object>();
				parseResultBuilder.addTypedValues(argSpec, currentPosition++, typedValuesAtPosition);
				if (assertNoMissingMandatoryParameter(argSpec, args, i, arity)
						|| isArgResemblesOptionThereforeDiscontinue(argSpec, args, i, arity)) {
					break;
				}
				consumeOneMapArgument(argSpec, lookBehind, alreadyUnquoted, arity, consumed, args.pop(), classes,
						keyConverter, valueConverter, typedValuesAtPosition, i, argDescription);
				result.putAll(typedValuesAtPosition);
				consumed = consumedCountMap(i + 1, initialSize, argSpec);
				lookBehind = LookBehind.SEPARATE;
				alreadyUnquoted = false;
			}
			// now process the varargs if any
			final String fallback = argSpec.isOption()
					&& !OptionSpec.DEFAULT_FALLBACK_VALUE.equals(((OptionSpec) argSpec).fallbackValue())
							? ((OptionSpec) argSpec).fallbackValue()
							: null;
			if (fallback != null && consumed == 0
					&& (args.isEmpty() || !varargCanConsumeNextValue(argSpec, args.peek())
							|| !canConsumeOneMapArgument(argSpec, lookBehind, alreadyUnquoted, arity, consumed,
									args.peek(), classes, keyConverter, valueConverter, argDescription))) {
				args.push(fallback);
			}
			for (int i = consumed; consumed < arity.max && !args.isEmpty(); i++) {
				if (!varargCanConsumeNextValue(argSpec, args.peek())) {
					break;
				}

				final Map<Object, Object> typedValuesAtPosition = new LinkedHashMap<Object, Object>();
				parseResultBuilder.addTypedValues(argSpec, currentPosition++, typedValuesAtPosition);
				if (!canConsumeOneMapArgument(argSpec, lookBehind, alreadyUnquoted, arity, consumed, args.peek(),
						classes, keyConverter, valueConverter, argDescription)) {
					break; // leave empty map at argSpec.typedValueAtPosition[currentPosition] so we won't
							// try to consume that position again
				}
				if (isArgResemblesOptionThereforeDiscontinue(argSpec, args, i, arity)) {
					break;
				}
				consumeOneMapArgument(argSpec, lookBehind, alreadyUnquoted, arity, consumed, args.pop(), classes,
						keyConverter, valueConverter, typedValuesAtPosition, i, argDescription);
				result.putAll(typedValuesAtPosition);
				consumed = consumedCountMap(i + 1, initialSize, argSpec);
				lookBehind = LookBehind.SEPARATE;
				alreadyUnquoted = false;
			}
		}

		private int consumeOneArgument(ArgSpec argSpec, LookBehind lookBehind, boolean alreadyUnquoted, Range arity,
				int consumed, String arg, List<Object> result, int index, String argDescription) {
			if (!lookBehind.isAttached()) {
				parseResultBuilder.nowProcessing(argSpec, arg);
			}
			final String[] values = unquoteAndSplit(argSpec, lookBehind, alreadyUnquoted, arity, consumed, arg);
			final ITypeConverter<?> converter = getTypeConverter(argSpec.auxiliaryTypes(), argSpec, 0);
			for (final String value : values) {
				final Object stronglyTypedValue = tryConvert(argSpec, index, converter, value, 0);
				result.add(stronglyTypedValue);
				if (tracer().isInfo()) {
					final String displayVal = !argSpec.interactive() || argSpec.echo()
							? String.valueOf(stronglyTypedValue)
							: MASKED_VALUE;
					tracer().info("Adding [%s] to %s for %s on %s", displayVal, argSpec.toString(), argDescription,
							argSpec.scopeString());
				}
				parseResultBuilder.addStringValue(argSpec, value);
			}
			parseResultBuilder.addOriginalStringValue(argSpec, arg);
			return ++index;
		}

		private void consumeOneMapArgument(ArgSpec argSpec, LookBehind lookBehind, boolean alreadyUnquoted, Range arity,
				int consumed, String arg, Class<?>[] classes, ITypeConverter<?> keyConverter,
				ITypeConverter<?> valueConverter, Map<Object, Object> result, int index, String argDescription)
				throws Exception {
			if (!lookBehind.isAttached()) {
				parseResultBuilder.nowProcessing(argSpec, arg);
			}
			final String[] values = unquoteAndSplit(argSpec, lookBehind, alreadyUnquoted, arity, consumed, arg);
			final Tracer tracer = CommandLine.tracer();
			for (final String value : values) {
				final String[] keyValue = splitKeyValue(argSpec, value);
				final Object mapKey = tryConvert(argSpec, index, keyConverter, keyValue[0], 0);
				final String rawMapValue = keyValue.length == 1 ? argSpec.mapFallbackValue() : keyValue[1];
				final Object mapValue = tryConvert(argSpec, index, valueConverter, rawMapValue, 1);
				result.put(mapKey, mapValue);
				final String displayVal = !argSpec.interactive() || argSpec.echo() ? String.valueOf(mapValue)
						: MASKED_VALUE;
				if (tracer.isInfo()) {
					tracer.info("Putting [%s : %s] in %s<%s, %s> %s for %s on %s", String.valueOf(mapKey), displayVal,
							result.getClass().getSimpleName(), classes[0].getSimpleName(), classes[1].getSimpleName(),
							argSpec.toString(), argDescription, argSpec.scopeString());
				}
				parseResultBuilder.addStringValue(argSpec, keyValue[0]);
				parseResultBuilder.addStringValue(argSpec, rawMapValue);
			}
			parseResultBuilder.addOriginalStringValue(argSpec, arg);
		}

		@SuppressWarnings({ "unchecked", "rawtypes" })
		private Collection<Object> createCollection(Class<?> collectionClass, Class<?>[] elementType) throws Exception {
			if (EnumSet.class.isAssignableFrom(collectionClass) && Enum.class.isAssignableFrom(elementType[0])) {
				final Object enumSet = EnumSet.noneOf((Class<Enum>) elementType[0]);
				return (Collection<Object>) enumSet;
			}
			// custom Collection implementation class must have default constructor
			return (Collection<Object>) factory.create(collectionClass);
		}

		@SuppressWarnings("unchecked")
		private Map<Object, Object> createMap(Class<?> mapClass) throws Exception {
			return (Map<Object, Object>) factory.create(mapClass);
		}

		private String createMissingParameterMessageFoundOtherOption(ArgSpec argSpec, Stack<String> args, int i,
				Range arity) {
			final String desc = arity.min > 1 ? (i + 1) + " (of " + arity.min + " mandatory parameters) " : "";
			return "Expected parameter " + desc + "for " + optionDescription("", argSpec, -1) + " but found '"
					+ args.peek() + "'";
		}

		private String createUserInputDebugString(ArgSpec argSpec, char[] result, String name) {
			return argSpec.echo() ? String.format("User entered %s for %s.%n", new String(result), name)
					: String.format("User entered %d characters for %s.%n", result.length, name);
		}

		private boolean excl(String fqcn) {
			return BuiltIn.excluded(fqcn);
		}

		private void expandArgumentFile(String fileName, List<String> arguments, Set<String> visited) {
			final Tracer tracer = tracer();
			final File file = new File(fileName);
			if (!file.canRead()) {
				if (tracer.isInfo()) {
					tracer.info("File %s does not exist or cannot be read; treating argument literally", fileName);
				}
				arguments.add("@" + fileName);
			} else if (visited.contains(file.getAbsolutePath())) {
				if (tracer.isInfo()) {
					tracer.info("Already visited file %s; ignoring...", file.getAbsolutePath());
				}
			} else {
				expandValidArgumentFile(fileName, file, arguments, visited);
			}
		}

		private void expandValidArgumentFile(String fileName, File file, List<String> arguments, Set<String> visited) {
			final List<String> result = new ArrayList<String>();
			LineNumberReader reader = null;
			try {
				visited.add(file.getAbsolutePath());
				reader = new LineNumberReader(new FileReader(file));
				if (commandSpec.parser().useSimplifiedAtFiles()) {
					String token;
					while ((token = reader.readLine()) != null) {
						if (token.length() > 0
								&& !token.trim().startsWith(String.valueOf(commandSpec.parser().atFileCommentChar()))) {
							addOrExpand(token, result, visited);
						}
					}
				} else {
					final StreamTokenizer tok = new StreamTokenizer(reader);
					tok.resetSyntax();
					tok.wordChars(' ', 255);
					tok.whitespaceChars(0, ' ');
					tok.quoteChar('"');
					tok.quoteChar('\'');
					if (commandSpec.parser().atFileCommentChar() != null) {
						tok.commentChar(commandSpec.parser().atFileCommentChar());
					}
					while (tok.nextToken() != StreamTokenizer.TT_EOF) {
						addOrExpand(tok.sval, result, visited);
					}
				}
			} catch (final Exception ex) {
				throw new InitializationException("Could not read argument file @" + fileName, ex);
			} finally {
				close(reader);
			}
			final Tracer tracer = tracer();
			if (tracer.isInfo()) {
				tracer.info("Expanded file @%s to arguments %s", fileName, result);
			}
			arguments.addAll(result);
		}

		private ITypeConverter<?> getActualTypeConverter(final Class<?> type, ArgSpec argSpec) {
			// https://github.com/remkop/picocli/pull/648
			// consider adding ParserSpec.charArraysCanCaptureStrings() to allow
			// non-interactive options to capture multi-char values in a char[] array
			// Note that this will require special logic for char[] types in
			// CommandLine$Interpreter.applyValuesToArrayField;
			// TBD: what to do with multiple values? Append or overwrite?
			if (char[].class.equals(argSpec.type()) && argSpec.interactive()) {
				return converterRegistry.get(char[].class);
			}
			if (converterRegistry.containsKey(type)) {
				return converterRegistry.get(type);
			}
			if (type.isEnum()) {
				return getEnumTypeConverter(type);
			}
			throw new MissingTypeConverterException(CommandLine.this,
					"No TypeConverter registered for " + type.getName() + " of " + argSpec);
		}

		private ITypeConverter<Object> getEnumTypeConverter(final Class<?> type) {
			return new ITypeConverter<Object>() {
				@Override
				@SuppressWarnings({ "unchecked", "rawtypes" })
				public Object convert(String value) throws Exception {
					try {
						return Enum.valueOf((Class<Enum>) type, value);
					} catch (final IllegalArgumentException ex) {
						final boolean insensitive = commandSpec.parser().caseInsensitiveEnumValuesAllowed();
						for (final Enum<?> enumConstant : ((Class<Enum<?>>) type).getEnumConstants()) {
							final String str = enumConstant.toString();
							final String name = enumConstant.name();
							if (value.equals(str) || value.equals(name)
									|| insensitive && (value.equalsIgnoreCase(str) || value.equalsIgnoreCase(name))) {
								return enumConstant;
							}
						}
						final String sensitivity = insensitive ? "case-insensitive" : "case-sensitive";
						final Enum<?>[] constants = ((Class<Enum<?>>) type).getEnumConstants();
						final List<String> names = new ArrayList<String>();
						for (final Enum<?> constant : constants) {
							names.add(constant.name());
							if (!names.contains(constant.toString())) { // name() != toString()
								if (!(insensitive && constant.name().equalsIgnoreCase(constant.toString()))) {
									names.add(constant.toString());
								}
							}
						}
						throw new TypeConversionException(
								format("expected one of %s (%s) but was '%s'", names, sensitivity, value));
					}
				}
			};
		}

		private String getMaskedValue(ArgSpec argSpec, String input) {
			return argSpec.echo() ? input : "***";
		}

		int getPosition(ArgSpec arg) {
			if (arg.group() == null) {
				return position;
			}
			final GroupMatchContainer container = parseResultBuilder.groupMatchContainer
					.findLastMatchContainer(arg.group());
			return container == null ? 0 : container.lastMatch().position;
		}

		private ITypeConverter<?> getTypeConverter(Class<?>[] types, final ArgSpec argSpec, int index) {
			if (argSpec.converters().length > index
					&& !argSpec.converters()[index].getClass().equals(UseDefaultConverter.class)) {
				return argSpec.converters()[index];
			} // use custom converters if defined
			final Class<?> type = types[index];
			if (isOptional(type)) { // #1214 #1108
				if (types.length <= index + 1) {
					throw new PicocliException(
							"Cannot create converter for types " + Arrays.asList(types) + " for " + argSpec);
				}
				final ITypeConverter<?> converter = getActualTypeConverter(types[index + 1], argSpec);
				return new ITypeConverter<Object>() {
					@Override
					public Object convert(String value) throws Exception {
						return value == null ? getOptionalEmpty() : getOptionalOfNullable(converter.convert(value));
					}
				};
			}
			return getActualTypeConverter(type, argSpec);
		}

		private void handleUnmatchedArgument(Stack<String> args) throws Exception {
			if (!args.isEmpty()) {
				parseResultBuilder.addUnmatched(parseResultBuilder.totalArgCount() - args.size(), args.pop());
			}
			if (config().stopAtUnmatched()) {
				parseResultBuilder.addUnmatched(args);
			}
		}

		private boolean is(ArgSpec p, String attribute, boolean value) {
			if (value) {
				if (tracer().isInfo()) {
					tracer().info("%s has '%s' annotation: not validating required fields", p.toString(), attribute);
				}
			}
			return value;
		}

		private boolean isAnyHelpRequested() {
			return isHelpRequested || parseResultBuilder.versionHelpRequested || parseResultBuilder.usageHelpRequested;
		}

		private boolean isArgResemblesOptionThereforeDiscontinue(ArgSpec argSpec, Stack<String> args, int i,
				Range arity) throws Exception {
			boolean result = false;
			final String arg = args.peek();
			if (commandSpec.resemblesOption(arg)) {
				if (argSpec.isPositional()
						&& !(endOfOptions || commandSpec.parser().unmatchedOptionsArePositionalParams())) { // #1015
					handleUnmatchedArgument(args);
					result = true;
				}
				if (argSpec.isOption() && !commandSpec.parser().unmatchedOptionsAllowedAsOptionParameters()) { // #639
					final String msg = "Unknown option: '" + arg + "'; "
							+ createMissingParameterMessageFoundOtherOption(argSpec, args, i, arity);
					maybeThrow(new UnmatchedArgumentException(commandSpec.commandLine(), msg));
					result = true;
				}
				if (tracer().isDebug()) {
					tracer().debug(
							"Parser is configured to allow unmatched option '%s' as option or positional parameter.",
							arg);
				}
			}
			return result;
		}

		private boolean isCommand(String arg) {
			// [#828] Subcommands should not be parsed as option values for options with
			// optional parameters.
			if (commandSpec.subcommands().containsKey(arg)) {
				return true;
			}

			// #454 repeatable subcommands
			final CommandSpec parent = commandSpec.parent();
			return parent != null && parent.subcommandsRepeatable() && parent.subcommands().containsKey(arg);
		}

		private boolean isEndOfOptionsDelimiter(String arg) {
			return commandSpec.parser().endOfOptionsDelimiter().equals(arg);
		}

		/**
		 * Returns true if the specified arg is "--", a registered option, or
		 * potentially a clustered POSIX option. Called when parsing varargs parameters
		 * for a multi-value option. When an option is encountered, the remainder should
		 * not be interpreted as vararg elements.
		 * 
		 * @param arg the string to determine whether it is an option or not
		 * @return true if it is an option, false otherwise
		 * @see #isCommand(String)
		 * @see #isEndOfOptionsDelimiter(String)
		 */
		private boolean isOption(String arg) {
			if (arg == null) {
				return false;
			}
			if (isEndOfOptionsDelimiter(arg)) {
				return true;
			}

			// not just arg prefix: we may be in the middle of parsing -xrvfFILE
			if (commandSpec.optionsMap().containsKey(arg)) { // -v or -f or --file (not attached to param or other
																// option)
				return true;
			}
			if (commandSpec.negatedOptionsByNameMap.containsKey(arg)) { // negated option like --no-verbose
				return true;
			}
			final int separatorIndex = arg.indexOf(config().separator());
			if (separatorIndex > 0) { // -f=FILE or --file==FILE (attached to param via separator)
				if (commandSpec.optionsMap().containsKey(arg.substring(0, separatorIndex))) {
					return true;
				}
			}
			return (arg.length() > 2 && arg.startsWith("-")
					&& commandSpec.posixOptionsMap().containsKey(arg.charAt(1)));
		}

		private boolean isStandaloneOption(String arg) {
			return commandSpec.optionsMap().containsKey(arg) || commandSpec.negatedOptionsMap().containsKey(arg);
		}

		void maybeThrow(PicocliException ex) throws PicocliException {
			if (commandSpec.parser().collectErrors) {
				parseResultBuilder.addError(ex);
			} else {
				throw ex;
			}
		}

		private void parse(List<CommandLine> parsedCommands, Stack<String> argumentStack, String[] originalArgs,
				List<Object> nowProcessing, Collection<ArgSpec> inheritedRequired) {
			parse(parsedCommands, argumentStack, originalArgs, nowProcessing, inheritedRequired,
					new LinkedHashSet<ArgSpec>());
		}

		private void parse(List<CommandLine> parsedCommands, Stack<String> argumentStack, String[] originalArgs,
				List<Object> nowProcessing, Collection<ArgSpec> inheritedRequired, Set<ArgSpec> initialized) {
			final Tracer tracer = CommandLine.tracer();
			if (tracer.isDebug()) {
				tracer.debug(
						"Initializing %s: %d options, %d positional parameters, %d required, %d groups, %d subcommands.",
						commandSpec.toString(), new HashSet<ArgSpec>(commandSpec.optionsMap().values()).size(),
						commandSpec.positionalParameters().size(), commandSpec.requiredArgs().size(),
						commandSpec.argGroups().size(), commandSpec.subcommands().size());
			}
			clear(); // first reset any state in case this CommandLine instance is being reused
			parsedCommands.add(CommandLine.this);
			final List<ArgSpec> required = new ArrayList<ArgSpec>(commandSpec.requiredArgs());
			addPostponedRequiredArgs(inheritedRequired, required);
			Collections.sort(required, new PositionalParametersSorter());

			// TODO Callback to preprocessor here? //
			// https://github.com/remkop/picocli/issues/1217

			final boolean continueOnError = commandSpec.parser().collectErrors();

			final Map<String, Object> info = mapOf("versionHelpRequested", parseResultBuilder.versionHelpRequested,
					"usageHelpRequested", parseResultBuilder.usageHelpRequested);
			if (commandSpec.preprocessor().preprocess(argumentStack, commandSpec, null, info)) {
				parseResultBuilder.versionHelpRequested = (Boolean) info.get("versionHelpRequested");
				parseResultBuilder.usageHelpRequested = (Boolean) info.get("usageHelpRequested");
				return;
			}
			do {
				final int stackSize = argumentStack.size();
				try {
					processArguments(parsedCommands, argumentStack, required, initialized, originalArgs, nowProcessing);
					applyDefaultValues(required, initialized);
				} catch (final InitializationException ex) {
					maybeThrow(ex);
				} catch (final ParameterException ex) {
					maybeThrow(ex);
				} catch (final Exception ex) {
					final int offendingArgIndex = originalArgs.length - argumentStack.size() - 1;
					final String arg = offendingArgIndex >= 0 && offendingArgIndex < originalArgs.length
							? originalArgs[offendingArgIndex]
							: "?";
					maybeThrow(ParameterException.create(CommandLine.this, ex, arg, offendingArgIndex, originalArgs));
				}
				if (continueOnError && stackSize == argumentStack.size() && stackSize > 0) {
					parseResultBuilder.addUnmatched(parseResultBuilder.totalArgCount() - argumentStack.size(),
							argumentStack.pop());
				}
			} while (!argumentStack.isEmpty() && continueOnError);

			boolean anyHelpRequested = isAnyHelpRequested();
			CommandLine parsed = CommandLine.this;
			while (parsed.getParent() != null) {
				parsed = parsed.getParent();
				anyHelpRequested |= parsed.interpreter.isAnyHelpRequested();
			}
			if (!anyHelpRequested) {
				validateConstraints(argumentStack, required, initialized);
			}
		}

		/**
		 * Entry point into parsing command line arguments.
		 * 
		 * @param args the command line arguments
		 * @return a list with all commands and subcommands initialized by this method
		 * @throws ParameterException if the specified command line arguments are
		 *                            invalid
		 */
		List<CommandLine> parse(String... args) {
			Assert.notNull(args, "argument array");
			final Tracer tracer = tracer();
			if (tracer.isInfo()) {
				tracer.info("Picocli version: %s", versionString());
			}
			if (tracer.isInfo()) {
				tracer.info("Parsing %d command line args %s", args.length, Arrays.toString(args));
			}
			if (tracer.isDebug()) {
				tracer.debug("Parser configuration: optionsCaseInsensitive=%s, subcommandsCaseInsensitive=%s, %s",
						commandSpec.optionsCaseInsensitive(), commandSpec.subcommandsCaseInsensitive(), config());
			}
			if (tracer.isDebug()) {
				tracer.debug(
						"(ANSI is %s by default: systemproperty[picocli.ansi]=%s, isatty=%s, TERM=%s, OSTYPE=%s, isWindows=%s, JansiConsoleInstalled=%s, ANSICON=%s, ConEmuANSI=%s, NO_COLOR=%s, CLICOLOR=%s, CLICOLOR_FORCE=%s)",
						Help.Ansi.AUTO.enabled() ? "enabled" : "disabled", System.getProperty("picocli.ansi"),
						Help.Ansi.isTTY(), System.getenv("TERM"), System.getenv("OSTYPE"), Help.Ansi.isWindows(),
						Help.Ansi.isJansiConsoleInstalled(), System.getenv("ANSICON"), System.getenv("ConEmuANSI"),
						System.getenv("NO_COLOR"), System.getenv("CLICOLOR"), System.getenv("CLICOLOR_FORCE"));
			}
			final List<String> expanded = new ArrayList<String>();
			for (final String arg : args) {
				addOrExpand(arg, expanded, new LinkedHashSet<String>());
			}
			final Stack<String> arguments = new Stack<String>();
			arguments.addAll(reverseList(expanded));
			final List<CommandLine> result = new ArrayList<CommandLine>();
			parse(result, arguments, args, new ArrayList<Object>(), new HashSet<ArgSpec>());
			return result;
		}

		String positionDesc(ArgSpec arg) {
			final int pos = getPosition(arg);
			return (arg.group() == null) ? pos + " (command-local)"
					: pos + " (in group " + arg.group().synopsis() + ")";
		}

		private void processArguments(List<CommandLine> parsedCommands, Stack<String> args,
				Collection<ArgSpec> required, Set<ArgSpec> initialized, String[] originalArgs,
				List<Object> nowProcessing) throws Exception {
			// arg must be one of:
			// 1. the "--" double dash separating options from positional arguments
			// 1. a stand-alone flag, like "-v" or "--verbose": no value required, must map
			// to boolean or Boolean field
			// 2. a short option followed by an argument, like "-f file" or "-ffile": may
			// map to any type of field
			// 3. a long option followed by an argument, like "-file out.txt" or
			// "-file=out.txt"
			// 3. one or more remaining arguments without any associated options. Must be
			// the last in the list.
			// 4. a combination of stand-alone options, like "-vxr". Equivalent to "-v -x
			// -r", "-v true -x true -r true"
			// 5. a combination of stand-alone options and one option with an argument, like
			// "-vxrffile"

			if (parseResultBuilder.expandedArgList.isEmpty()) { // don't add args again if called from do-while in
																// parse()
				final List<String> expandedArgs = new ArrayList<String>(args);
				Collections.reverse(expandedArgs); // Need to reverse the stack to get args in specified order
				parseResultBuilder.expandedArgs(expandedArgs);
				parseResultBuilder.originalArgs(originalArgs);
				parseResultBuilder.nowProcessing = nowProcessing;
			}

			final String separator = config().separator();
			final Tracer tracer = CommandLine.tracer();
			while (!args.isEmpty()) {
				if (endOfOptions) {
					processRemainderAsPositionalParameters(required, initialized, args);
					return;
				}
				final String originalArg = args.pop();
				String arg = smartUnquoteIfEnabled(originalArg);
				final boolean actuallyUnquoted = !originalArg.equals(arg);
				if (tracer.isDebug()) {
					final int argIndex = originalArgs.length - (args.size() + 1);
					if (actuallyUnquoted) {
						tracer.debug("[%d] Processing argument '%s' (trimmed from '%s'). Remainder=%s", argIndex, arg,
								originalArg, reverse(copy(args)));
					} else {
						tracer.debug("[%d] Processing argument '%s'. Remainder=%s", argIndex, arg, reverse(copy(args)));
					}
				}

				// Double-dash separates options from positional arguments.
				// If found, then interpret the remaining args as positional parameters.
				if (commandSpec.parser.endOfOptionsDelimiter().equals(arg)) {
					tracer.info("Found end-of-options delimiter '%s'. Treating remainder as positional parameters.",
							commandSpec.parser.endOfOptionsDelimiter());
					endOfOptions = true;
					processRemainderAsPositionalParameters(required, initialized, args);
					return; // we are done
				}

				// if we find another command, we are done with the current command
				CommandLine subcommand = commandSpec.subcommands().get(arg);
				if (subcommand == null && commandSpec.parser().abbreviatedSubcommandsAllowed()) {
					subcommand = AbbreviationMatcher.match(commandSpec.subcommands(), arg,
							commandSpec.subcommandsCaseInsensitive(), CommandLine.this).getValue();
				}
				if (subcommand != null) {
					processSubcommand(subcommand, parseResultBuilder, parsedCommands, args, required, initialized,
							originalArgs, nowProcessing, separator, arg);
					return; // remainder done by the command
				}
				final CommandSpec parent = commandSpec.parent();
				if (parent != null && parent.subcommandsRepeatable()) {
					subcommand = parent.subcommands().get(arg);
					if (subcommand == null && parent.parser().abbreviatedSubcommandsAllowed()) {
						subcommand = AbbreviationMatcher
								.match(parent.subcommands(), arg, parent.subcommandsCaseInsensitive(), CommandLine.this)
								.getValue();
					}
					if (subcommand != null) {
						tracer.debug("'%s' is a repeatable subcommand of %s", arg,
								commandSpec.parent().qualifiedName()); // #454 repeatable subcommands
						Set<ArgSpec> inheritedInitialized = initialized;
						if (subcommand.interpreter.parseResultBuilder != null) {
							tracer.debug("Subcommand '%s' has been matched before. Making a copy...",
									subcommand.getCommandName());
							subcommand = subcommand.copy();
							subcommand.getCommandSpec().parent(commandSpec.parent()); // hook it up with its parent
							inheritedInitialized = new LinkedHashSet<ArgSpec>(inheritedInitialized);
						}
						processSubcommand(subcommand, getParent().interpreter.parseResultBuilder, parsedCommands, args,
								required, inheritedInitialized, originalArgs, nowProcessing, separator, arg);
						continue;
					}
				}

				// First try to interpret the argument as a single option (as opposed to a
				// compact group of options).
				// A single option may be without option parameters, like "-v" or "--verbose" (a
				// boolean value),
				// or an option may have one or more option parameters.
				// A parameter may be attached to the option.
				final LinkedHashMap<String, OptionSpec> aggregatedOptions = new LinkedHashMap<String, OptionSpec>();
				if (commandSpec.parser().abbreviatedOptionsAllowed()) {
					aggregatedOptions.putAll(commandSpec.optionsMap());
					aggregatedOptions.putAll(commandSpec.negatedOptionsMap());
					arg = AbbreviationMatcher
							.match(aggregatedOptions, arg, commandSpec.optionsCaseInsensitive(), CommandLine.this)
							.getFullName();
				}
				LookBehind lookBehind = LookBehind.SEPARATE;
				final int separatorIndex = arg.indexOf(separator);
				if (separatorIndex > 0) {
					String key = arg.substring(0, separatorIndex);
					key = AbbreviationMatcher
							.match(aggregatedOptions, key, commandSpec.optionsCaseInsensitive(), CommandLine.this)
							.getFullName(); // #1159, #1162
					// be greedy. Consume the whole arg as an option if possible.
					if (isStandaloneOption(key) && isStandaloneOption(arg)) {
						tracer.warn("Both '%s' and '%s' are valid option names in %s. Using '%s'...", arg, key,
								getCommandName(), arg);
					} else if (isStandaloneOption(key)) {
						lookBehind = LookBehind.ATTACHED_WITH_SEPARATOR;
						final String optionParam = arg.substring(separatorIndex + separator.length());
						args.push(optionParam);
						arg = key;
						if (tracer.isDebug()) {
							tracer.debug("Separated '%s' option from '%s' option parameter", key, optionParam);
						}
					} else {
						if (tracer.isDebug()) {
							tracer.debug("'%s' contains separator '%s' but '%s' is not a known option", arg, separator,
									key);
						}
					}
				} else {
					if (tracer.isDebug()) {
						tracer.debug("'%s' cannot be separated into <option>%s<option-parameter>", arg, separator);
					}
				}
				if (isStandaloneOption(arg)) {
					processStandaloneOption(required, initialized, arg, actuallyUnquoted, args, lookBehind);
				}
				// Compact (single-letter) options can be grouped with other options or with an
				// argument.
				// only single-letter options can be combined with other options or with an
				// argument
				else if (config().posixClusteredShortOptionsAllowed() && arg.length() > 2 && arg.startsWith("-")) {
					if (tracer.isDebug()) {
						tracer.debug("Trying to process '%s' as clustered short options", arg, args);
					}
					processClusteredShortOptions(required, initialized, arg, actuallyUnquoted, args);
				}
				// The argument could not be interpreted as an option: process it as a
				// positional argument
				else {
					args.push(arg);
					if (tracer.isDebug()) {
						tracer.debug(
								"Could not find option '%s', deciding whether to treat as unmatched option or positional parameter...",
								arg);
					}
					if (tracer.isDebug()) {
						tracer.debug("No option named '%s' found. Processing as positional parameter", arg);
					}
					processPositionalParameter(required, initialized, actuallyUnquoted, args);
				}
			}
		}

		private void processClusteredShortOptions(Collection<ArgSpec> required, Set<ArgSpec> initialized, String arg,
				boolean alreadyUnquoted, Stack<String> args) throws Exception {
			final String prefix = arg.substring(0, 1);
			String cluster = arg.substring(1);
			boolean paramAttachedToOption = true;
			boolean first = true;
			final Tracer tracer = CommandLine.tracer();
			do {
				if (cluster.length() > 0 && commandSpec.posixOptionsMap().containsKey(cluster.charAt(0))) {
					final ArgSpec argSpec = commandSpec.posixOptionsMap().get(cluster.charAt(0));
					Range arity = argSpec.arity();
					final String argDescription = "option " + prefix + cluster.charAt(0);
					if (tracer.isDebug()) {
						tracer.debug("Found option '%s%s' in %s: %s, arity=%s", prefix, cluster.charAt(0), arg, argSpec,
								arity);
					}
					required.remove(argSpec);
					cluster = cluster.substring(1);
					paramAttachedToOption = cluster.length() > 0;
					LookBehind lookBehind = paramAttachedToOption ? LookBehind.ATTACHED : LookBehind.SEPARATE;
					if (cluster.startsWith(config().separator())) {// attached with separator, like -f=FILE or -v=true
						lookBehind = LookBehind.ATTACHED_WITH_SEPARATOR;
						cluster = cluster.substring(config().separator().length());
						arity = arity.min(Math.max(1, arity.min)); // if key=value, minimum arity is at least 1
					}
					if (arity.min > 0 && !empty(cluster)) {
						if (tracer.isDebug()) {
							tracer.debug("Trying to process '%s' as option parameter", cluster);
						}
					}
					// arity may be >= 1, or
					// arity <= 0 && !cluster.startsWith(separator)
					// e.g., boolean @Option("-v", arity=0, varargs=true); arg "-rvTRUE", remainder
					// cluster="TRUE"
					if (!empty(cluster)) {
						args.push(cluster); // interpret remainder as option parameter (CAUTION: may be empty string!)
					}
					if (first) {
						parseResultBuilder.nowProcessing.add(argSpec);
						first = false;
					} else {
						parseResultBuilder.nowProcessing.set(parseResultBuilder.nowProcessing.size() - 1, argSpec); // replace
					}
					final int argCount = args.size();
					final int consumed = applyOption(argSpec, false, lookBehind, alreadyUnquoted, arity, args,
							initialized, argDescription);
					// if cluster was consumed as a parameter or if this field was the last in the
					// cluster we're done; otherwise continue do-while loop
					if (empty(cluster) || args.isEmpty() || args.size() < argCount) {
						return;
					}
					cluster = args.pop();
				} else { // cluster is empty || cluster.charAt(0) is not a short option key
					if (cluster.length() == 0) { // we finished parsing a group of short options like -rxv
						return; // return normally and parse the next arg
					}
					// We get here when the remainder of the cluster group is neither an option,
					// nor a parameter that the last option could consume.
					if (arg.endsWith(cluster)) {
						args.push(paramAttachedToOption ? prefix + cluster : cluster);
						if (args.peek().equals(arg)) { // #149 be consistent between unmatched short and long options
							if (tracer.isDebug()) {
								tracer.debug(
										"Could not match any short options in %s, deciding whether to treat as unmatched option or positional parameter...",
										arg);
							}
							processPositionalParameter(required, initialized, alreadyUnquoted, args);
							return;
						}
						// remainder was part of a clustered group that could not be completely parsed
						if (tracer.isDebug()) {
							tracer.debug("No option found for %s in %s", cluster, arg);
						}
						String tmp = args.pop();
						tmp = tmp + " (while processing option: '" + arg + "')";
						args.push(tmp);
						handleUnmatchedArgument(args);
					} else {
						args.push(cluster);
						if (tracer.isDebug()) {
							tracer.debug("%s is not an option parameter for %s", cluster, arg);
						}
						processPositionalParameter(required, initialized, alreadyUnquoted, args);
					}
					return;
				}
			} while (true);
		}

		private void processPositionalParameter(Collection<ArgSpec> required, Set<ArgSpec> initialized,
				boolean alreadyUnquoted, Stack<String> args) throws Exception {
			final Tracer tracer = CommandLine.tracer();
			final String arg = args.peek();
			if (!endOfOptions && commandSpec.resemblesOption(arg)) {
				if (!commandSpec.parser().unmatchedOptionsArePositionalParams()) {
					handleUnmatchedArgument(args); // #149
					return;
				}
				if (tracer.isDebug()) {
					tracer.debug("Parser is configured to treat all unmatched options as positional parameter", arg);
				}
			}
			final int argIndex = parseResultBuilder.originalArgList.size() - args.size();
			if (tracer.isDebug()) {
				tracer.debug(
						"[%d] Processing next arg as a positional parameter. Command-local position=%d. Remainder=%s",
						argIndex, position, reverse(copy(args)));
			}
			if (config().stopAtPositional()) {
				if (!endOfOptions && tracer.isDebug()) {
					tracer.debug(
							"Parser was configured with stopAtPositional=true, treating remaining arguments as positional parameters.");
				}
				endOfOptions = true;
			}
			final int originalInteractiveCount = this.interactiveCount;
			int consumedByGroup = 0;
			int argsConsumed = 0;
			int interactiveConsumed = 0;
			final int originalNowProcessingSize = parseResultBuilder.nowProcessing.size();
			final List<Runnable> bookKeeping = new ArrayList<Runnable>();
			for (int i = 0; i < commandSpec.positionalParameters().size(); i++) {
				final PositionalParamSpec positionalParam = commandSpec.positionalParameters().get(i);
				final Range indexRange = positionalParam.index();
				final int localPosition = getPosition(positionalParam);
				if (positionalParam.group() != null) { // does the positionalParam's index range contain the current
														// position in the currently matching group
					final GroupMatchContainer groupMatchContainer = parseResultBuilder.groupMatchContainer
							.findOrCreateMatchingGroup(positionalParam, commandSpec.commandLine());
					if (!groupMatchContainer.canMatchPositionalParam(positionalParam)) {
						continue;
					}
				} else {
					if (!indexRange.contains(localPosition)
							|| positionalParam.typedValueAtPosition.get(localPosition) != null) {
						continue;
					}
				}
				final Stack<String> argsCopy = copy(args);
				final Range arity = positionalParam.arity();
				if (tracer.isDebug()) {
					tracer.debug("Position %s is in index range %s. Trying to assign args to %s, arity=%s",
							positionDesc(positionalParam), indexRange.internalToString(), positionalParam, arity);
				}
				if (!assertNoMissingParameters(positionalParam, arity, argsCopy)) {
					break;
				} // #389 collectErrors parsing
				final int originalSize = argsCopy.size();
				final int actuallyConsumed = applyOption(positionalParam, false, LookBehind.SEPARATE, alreadyUnquoted,
						arity, argsCopy, initialized,
						"args[" + indexRange.internalToString() + "] at position " + localPosition);
				final int count = originalSize - argsCopy.size();
				if (count > 0 || actuallyConsumed > 0) {
					required.remove(positionalParam);
					interactiveConsumed = this.interactiveCount - originalInteractiveCount;
				}
				if (positionalParam.group() == null) { // don't update the command-level position for group args
					argsConsumed = Math.max(argsConsumed, count);
				} else {
					final int updatedPosition = localPosition + count;
					final GroupMatchContainer groupMatchContainer = parseResultBuilder.groupMatchContainer
							.findOrCreateMatchingGroup(positionalParam, commandSpec.commandLine());
					bookKeeping.add(new Runnable() {
						@Override
						public void run() {
							if (groupMatchContainer != null) {
								groupMatchContainer.lastMatch().position = updatedPosition;
								if (tracer.isDebug()) {
									tracer.debug("Updated group position to %s for group %s.",
											groupMatchContainer.lastMatch().position, groupMatchContainer);
								}
							}
						}
					});
					consumedByGroup = Math.max(consumedByGroup, count);
				}
				while (parseResultBuilder.nowProcessing.size() > originalNowProcessingSize + count) {
					parseResultBuilder.nowProcessing.remove(parseResultBuilder.nowProcessing.size() - 1);
				}
			}
			// remove processed args from the stack
			final int maxConsumed = Math.max(consumedByGroup, argsConsumed);
			for (int i = 0; i < maxConsumed; i++) {
				args.pop();
			}
			position += argsConsumed + interactiveConsumed;
			if (tracer.isDebug()) {
				tracer.debug(
						"Consumed %d arguments and %d interactive values, moving command-local position to index %d.",
						argsConsumed, interactiveConsumed, position);
			}
			for (final Runnable task : bookKeeping) {
				task.run();
			}
			if (consumedByGroup == 0 && argsConsumed == 0 && interactiveConsumed == 0 && !args.isEmpty()) {
				handleUnmatchedArgument(args); // this gives error msg: "Unmatched argument at index 1: 'val2'"
//                // instead, we want a "positional parameter at index 0..* (<str>) should be specified only once"
//                List<String> unmatched = new ArrayList<String>();
//                while (!args.isEmpty()) {unmatched.add(args.pop());}
//                // positional parameter at index 0..* (<str>) should be specified only once
//                String msg = null; // FIXME report which positional has been fully filled and cannot take more args
//                for (int i = 0; i < commandSpec.positionalParameters().size(); i++) {
//                    PositionalParamSpec positionalParam = commandSpec.positionalParameters().get(i);
//                    Range indexRange = positionalParam.index();
//                    if (indexRange.contains(position - 1) && positionalParam.typedValueAtPosition.get(position - 1) == null) {
//                        aaaaa;
//                    }
//                }
//                if (!isUnmatchedArgumentsAllowed()) {
//                    maybeThrow(new UnmatchedArgumentException(CommandLine.this, unmatched, msg));
//                }
			}
		}

		private void processRemainderAsPositionalParameters(Collection<ArgSpec> required, Set<ArgSpec> initialized,
				Stack<String> args) throws Exception {
			while (!args.empty()) {
				processPositionalParameter(required, initialized, false, args);
			}
		}

		private void processStandaloneOption(Collection<ArgSpec> required, Set<ArgSpec> initialized, String arg,
				boolean alreadyUnquoted, Stack<String> args, LookBehind lookBehind) throws Exception {
			ArgSpec argSpec = commandSpec.optionsMap().get(arg);
			final boolean negated = argSpec == null;
			if (negated) {
				argSpec = commandSpec.negatedOptionsMap().get(arg);
			}
			required.remove(argSpec);
			Range arity = argSpec.arity();
			if (lookBehind.isAttached()) {
				arity = arity.min(Math.max(1, arity.min)); // if key=value, minimum arity is at least 1
			}
			final Tracer tracer = CommandLine.tracer();
			if (tracer.isDebug()) {
				tracer.debug("Found option named '%s': %s, arity=%s", arg, argSpec, arity);
			}
			parseResultBuilder.nowProcessing.add(argSpec);
			applyOption(argSpec, negated, lookBehind, alreadyUnquoted, arity, args, initialized, "option " + arg);
		}

		private void processSubcommand(CommandLine subcommand, ParseResult.Builder builder,
				List<CommandLine> parsedCommands, Stack<String> args, Collection<ArgSpec> required,
				Set<ArgSpec> initialized, String[] originalArgs, List<Object> nowProcessing, String separator,
				String arg) {
			final Tracer tracer = CommandLine.tracer();
			if (tracer.isDebug()) {
				tracer.debug("Found subcommand '%s' (%s)", arg, subcommand.commandSpec.toString());
			}
			nowProcessing.add(subcommand.commandSpec);
			updateHelpRequested(subcommand.commandSpec);
			final List<ArgSpec> inheritedRequired = new ArrayList<ArgSpec>();
			if (tracer.isDebug()) {
				tracer.debug("Checking required args for parent %s...", subcommand.commandSpec.parent());
			}
			final Iterator<ArgSpec> requiredIter = required.iterator();
			while (requiredIter.hasNext()) {
				final ArgSpec requiredArg = requiredIter.next();
				if (requiredArg.scopeType() == ScopeType.INHERIT || requiredArg.inherited()) {
					if (tracer.isDebug()) {
						tracer.debug("Postponing validation for required %s: scopeType=%s, inherited=%s",
								optionDescription("", requiredArg, -1), requiredArg.scopeType(),
								requiredArg.inherited());
					}
					if (!inheritedRequired.contains(requiredArg)) {
						inheritedRequired.add(requiredArg);
					}
					requiredIter.remove();
				}
			}
			if (!isAnyHelpRequested() && !required.isEmpty()) { // ensure current command portion is valid
				throw MissingParameterException.create(CommandLine.this, required, separator);
			}
			final Set<ArgSpec> inheritedInitialized = new LinkedHashSet<ArgSpec>();
			subcommand.interpreter.parse(parsedCommands, args, originalArgs, nowProcessing, inheritedRequired,
					inheritedInitialized);
			initialized.addAll(inheritedInitialized);
			builder.subcommand(subcommand.interpreter.parseResultBuilder.build());
		}

		char[] readPassword(String prompt) {
			try {
				final Object console = System.class.getDeclaredMethod("console").invoke(null);
				final Method method = Class.forName("java.io.Console").getDeclaredMethod("readPassword", String.class,
						Object[].class);
				return (char[]) method.invoke(console, prompt, new Object[0]);
			} catch (final Exception e) {
				return readUserInputWithEchoing(prompt);
			}
		}

		char[] readUserInput(ArgSpec argSpec) {
			final String name = argSpec.isOption() ? ((OptionSpec) argSpec).longestName() : "position " + position;
			final String desc = str(argSpec.description(), 0);
			final String standardPrompt = empty(desc) ? String.format("Enter value for %s: ", name)
					: String.format("Enter value for %s (%s): ", name, desc);
			final String prompt = empty(argSpec.prompt()) ? standardPrompt : argSpec.prompt();
			try {
				final Tracer t = tracer();
				if (t.isDebug()) {
					t.debug("Reading value for %s from console...", name);
				}
				final char[] result = argSpec.echo() ? readUserInputWithEchoing(prompt) : readPassword(prompt);
				if (t.isDebug()) {
					t.debug(createUserInputDebugString(argSpec, result, name));
				}
				return result;
			} finally {
				interactiveCount++;
			}
		}

		char[] readUserInputWithEchoing(String prompt) {
			System.out.print(prompt);
			final InputStreamReader isr = new InputStreamReader(System.in);
			final BufferedReader in = new BufferedReader(isr);
			try {
				final String input = in.readLine();
				return input == null ? new char[0] : input.toCharArray();
			} catch (final IOException e) {
				throw new IllegalStateException(e);
			}
		}

		private void reg(Class<?> timeClass, BuiltIn.ISO8601TimeConverter converter) {
			converterRegistry.put(timeClass, converter);
		}

		private void reg(Class<?> cls, Method method, Class<?>... paramTypes) {
			converterRegistry.put(cls, new BuiltIn.ReflectionConverter(method, paramTypes));
		}

		private void registerBuiltInConverters() {
			converterRegistry.put(Object.class, new BuiltIn.StringConverter());
			converterRegistry.put(String.class, new BuiltIn.StringConverter());
			converterRegistry.put(StringBuilder.class, new BuiltIn.StringBuilderConverter());
			converterRegistry.put(char[].class, new BuiltIn.CharArrayConverter());
			converterRegistry.put(CharSequence.class, new BuiltIn.CharSequenceConverter());
			converterRegistry.put(Byte.class, new BuiltIn.ByteConverter());
			converterRegistry.put(Byte.TYPE, new BuiltIn.ByteConverter());
			converterRegistry.put(Boolean.class, new BuiltIn.BooleanConverter());
			converterRegistry.put(Boolean.TYPE, new BuiltIn.BooleanConverter());
			converterRegistry.put(Character.class, new BuiltIn.CharacterConverter());
			converterRegistry.put(Character.TYPE, new BuiltIn.CharacterConverter());
			converterRegistry.put(Short.class, new BuiltIn.ShortConverter());
			converterRegistry.put(Short.TYPE, new BuiltIn.ShortConverter());
			converterRegistry.put(Integer.class, new BuiltIn.IntegerConverter());
			converterRegistry.put(Integer.TYPE, new BuiltIn.IntegerConverter());
			converterRegistry.put(Long.class, new BuiltIn.LongConverter());
			converterRegistry.put(Long.TYPE, new BuiltIn.LongConverter());
			converterRegistry.put(Float.class, new BuiltIn.FloatConverter());
			converterRegistry.put(Float.TYPE, new BuiltIn.FloatConverter());
			converterRegistry.put(Double.class, new BuiltIn.DoubleConverter());
			converterRegistry.put(Double.TYPE, new BuiltIn.DoubleConverter());
			converterRegistry.put(File.class, new BuiltIn.FileConverter());
			converterRegistry.put(URI.class, new BuiltIn.URIConverter());
			converterRegistry.put(URL.class, new BuiltIn.URLConverter());
			converterRegistry.put(Date.class, new BuiltIn.ISO8601DateConverter());
			converterRegistry.put(BigDecimal.class, new BuiltIn.BigDecimalConverter());
			converterRegistry.put(BigInteger.class, new BuiltIn.BigIntegerConverter());
			converterRegistry.put(Charset.class, new BuiltIn.CharsetConverter());
			converterRegistry.put(InetAddress.class, new BuiltIn.InetAddressConverter());
			converterRegistry.put(Pattern.class, new BuiltIn.PatternConverter());
			converterRegistry.put(UUID.class, new BuiltIn.UUIDConverter());
			converterRegistry.put(Currency.class, new BuiltIn.CurrencyConverter());
			converterRegistry.put(TimeZone.class, new BuiltIn.TimeZoneConverter());
			converterRegistry.put(ByteOrder.class, new BuiltIn.ByteOrderConverter());
			converterRegistry.put(Class.class, new BuiltIn.ClassConverter());
			converterRegistry.put(NetworkInterface.class, new BuiltIn.NetworkInterfaceConverter());

			// #698 use direct calls to Class.forName() and Class.getDeclaredMethod() with
			// constant Strings
			// to allow GraalVM static analysis to resolve the target elements
			if (!excl("java.sql.Time")) {
				try {
					reg(Class.forName("java.sql.Time"), new BuiltIn.ISO8601TimeConverter(
							Class.forName("java.sql.Time").getDeclaredConstructor(long.class)));
				} catch (final Exception e) {
					BuiltIn.handle(e, "java.sql.Time");
				}
			}

			if (!excl("java.sql.Connection")) {
				try {
					reg(Class.forName("java.sql.Connection"),
							Class.forName("java.sql.DriverManager").getDeclaredMethod("getConnection", String.class),
							String.class);
				} catch (final Exception e) {
					BuiltIn.handle(e, "java.sql.Connection");
				}
			}
			if (!excl("java.sql.Driver")) {
				try {
					reg(Class.forName("java.sql.Driver"),
							Class.forName("java.sql.DriverManager").getDeclaredMethod("getDriver", String.class),
							String.class);
				} catch (final Exception e) {
					BuiltIn.handle(e, "java.sql.DriverManager");
				}
			}
			if (!excl("java.sql.Timestamp")) {
				try {
					reg(Class.forName("java.sql.Timestamp"),
							Class.forName("java.sql.Timestamp").getDeclaredMethod("valueOf", String.class),
							String.class);
				} catch (final Exception e) {
					BuiltIn.handle(e, "java.sql.Timestamp");
				}
			}

			if (!excl("java.time.Duration")) {
				try {
					reg(Class.forName("java.time.Duration"),
							Class.forName("java.time.Duration").getDeclaredMethod("parse", CharSequence.class),
							CharSequence.class);
				} catch (final Exception e) {
					BuiltIn.handle(e, "java.time.Duration");
				}
			}
			if (!excl("java.time.Instant")) {
				try {
					reg(Class.forName("java.time.Instant"),
							Class.forName("java.time.Instant").getDeclaredMethod("parse", CharSequence.class),
							CharSequence.class);
				} catch (final Exception e) {
					BuiltIn.handle(e, "java.time.Instant");
				}
			}
			if (!excl("java.time.LocalDate")) {
				try {
					reg(Class.forName("java.time.LocalDate"),
							Class.forName("java.time.LocalDate").getDeclaredMethod("parse", CharSequence.class),
							CharSequence.class);
				} catch (final Exception e) {
					BuiltIn.handle(e, "java.time.LocalDate");
				}
			}
			if (!excl("java.time.LocalDateTime")) {
				try {
					reg(Class.forName("java.time.LocalDateTime"),
							Class.forName("java.time.LocalDateTime").getDeclaredMethod("parse", CharSequence.class),
							CharSequence.class);
				} catch (final Exception e) {
					BuiltIn.handle(e, "java.time.LocalDateTime");
				}
			}
			if (!excl("java.time.LocalTime")) {
				try {
					reg(Class.forName("java.time.LocalTime"),
							Class.forName("java.time.LocalTime").getDeclaredMethod("parse", CharSequence.class),
							CharSequence.class);
				} catch (final Exception e) {
					BuiltIn.handle(e, "java.time.LocalTime");
				}
			}
			if (!excl("java.time.MonthDay")) {
				try {
					reg(Class.forName("java.time.MonthDay"),
							Class.forName("java.time.MonthDay").getDeclaredMethod("parse", CharSequence.class),
							CharSequence.class);
				} catch (final Exception e) {
					BuiltIn.handle(e, "java.time.MonthDay");
				}
			}
			if (!excl("java.time.OffsetDateTime")) {
				try {
					reg(Class.forName("java.time.OffsetDateTime"),
							Class.forName("java.time.OffsetDateTime").getDeclaredMethod("parse", CharSequence.class),
							CharSequence.class);
				} catch (final Exception e) {
					BuiltIn.handle(e, "java.time.OffsetDateTime");
				}
			}
			if (!excl("java.time.OffsetTime")) {
				try {
					reg(Class.forName("java.time.OffsetTime"),
							Class.forName("java.time.OffsetTime").getDeclaredMethod("parse", CharSequence.class),
							CharSequence.class);
				} catch (final Exception e) {
					BuiltIn.handle(e, "java.time.OffsetTime");
				}
			}
			if (!excl("java.time.Period")) {
				try {
					reg(Class.forName("java.time.Period"),
							Class.forName("java.time.Period").getDeclaredMethod("parse", CharSequence.class),
							CharSequence.class);
				} catch (final Exception e) {
					BuiltIn.handle(e, "java.time.Period");
				}
			}
			if (!excl("java.time.Year")) {
				try {
					reg(Class.forName("java.time.Year"),
							Class.forName("java.time.Year").getDeclaredMethod("parse", CharSequence.class),
							CharSequence.class);
				} catch (final Exception e) {
					BuiltIn.handle(e, "java.time.Year");
				}
			}
			if (!excl("java.time.YearMonth")) {
				try {
					reg(Class.forName("java.time.YearMonth"),
							Class.forName("java.time.YearMonth").getDeclaredMethod("parse", CharSequence.class),
							CharSequence.class);
				} catch (final Exception e) {
					BuiltIn.handle(e, "java.time.YearMonth");
				}
			}
			if (!excl("java.time.ZonedDateTime")) {
				try {
					reg(Class.forName("java.time.ZonedDateTime"),
							Class.forName("java.time.ZonedDateTime").getDeclaredMethod("parse", CharSequence.class),
							CharSequence.class);
				} catch (final Exception e) {
					BuiltIn.handle(e, "java.time.ZonedDateTime");
				}
			}
			if (!excl("java.time.ZoneId")) {
				try {
					reg(Class.forName("java.time.ZoneId"),
							Class.forName("java.time.ZoneId").getDeclaredMethod("of", String.class), String.class);
				} catch (final Exception e) {
					BuiltIn.handle(e, "java.time.ZoneId");
				}
			}
			if (!excl("java.time.ZoneOffset")) {
				try {
					reg(Class.forName("java.time.ZoneOffset"),
							Class.forName("java.time.ZoneOffset").getDeclaredMethod("of", String.class), String.class);
				} catch (final Exception e) {
					BuiltIn.handle(e, "java.time.ZoneOffset");
				}
			}

			if (!excl("java.nio.file.Path")) {
				try {
					reg(Class.forName("java.nio.file.Path"),
							Class.forName("java.nio.file.Paths").getDeclaredMethod("get", String.class, String[].class),
							String.class, String[].class);
				} catch (final Exception e) {
					BuiltIn.handle(e, "java.nio.file.Path");
				}
			}
		}

		private String[] splitKeyValue(ArgSpec argSpec, String value) {
			final String[] keyValue = ArgSpec.splitRespectingQuotedStrings(value, 2, config(), argSpec, "=");

			// #1214: support for -Dkey map options
			// validation is disabled if `mapFallbackValue` is specified
			if (keyValue.length < 2 && ArgSpec.UNSPECIFIED.equals(argSpec.mapFallbackValue())) {
				final String splitRegex = argSpec.splitRegex();
				if (splitRegex.length() == 0) {
					throw new ParameterException(CommandLine.this, "Value for option "
							+ optionDescription("", argSpec, 0) + " should be in KEY=VALUE format but was " + value,
							argSpec, value);
				} else {
					throw new ParameterException(
							CommandLine.this, "Value for option " + optionDescription("", argSpec, 0)
									+ " should be in KEY=VALUE[" + splitRegex + "KEY=VALUE]... format but was " + value,
							argSpec, value);
				}
			}
			return keyValue;
		}

		private Stack<String> stack(String value) {
			final Stack<String> result = new Stack<String>();
			result.push(value);
			return result;
		}

		private Object tryConvert(ArgSpec argSpec, int index, ITypeConverter<?> converter, String value, int typeIndex)
				throws ParameterException {
			try {
				return converter.convert(value);
			} catch (final TypeConversionException ex) {
				final String msg = String.format("Invalid value for %s: %s", optionDescription("", argSpec, index),
						ex.getMessage());
				throw new ParameterException(CommandLine.this, msg, ex, argSpec, value);
			} catch (final Exception other) {
				final String desc = optionDescription("", argSpec, index);
				String typeDescr = argSpec.auxiliaryTypes()[typeIndex].getSimpleName();
				if (isOptional(argSpec.auxiliaryTypes()[typeIndex])) {
					typeDescr += "<" + argSpec.auxiliaryTypes()[typeIndex + 1].getSimpleName() + ">";
				}
				final String msg = String.format("Invalid value for %s: cannot convert '%s' to %s (%s)", desc, value,
						typeDescr, other);
				throw new ParameterException(CommandLine.this, msg, other, argSpec, value);
			}
		}

		private String[] unquoteAndSplit(ArgSpec argSpec, LookBehind lookBehind, boolean alreadyUnquoted, Range arity,
				int consumed, String arg) {
			final String raw = lookBehind.isAttached() && alreadyUnquoted ? arg : smartUnquoteIfEnabled(arg); // if
																												// attached,
																												// we
																												// already
																												// trimmed
																												// quotes
																												// once
//            String raw = smartUnquoteIfEnabled(arg);
			return argSpec.splitValue(raw, commandSpec.parser(), arity, consumed);
		}

		private void updateHelpRequested(ArgSpec argSpec) {
			if (!parseResultBuilder.isInitializingDefaultValues && argSpec.isOption()) {
				final OptionSpec option = (OptionSpec) argSpec;
				isHelpRequested |= is(argSpec, "help", option.help());
				parseResultBuilder.versionHelpRequested |= is(argSpec, "versionHelp", option.versionHelp());
				parseResultBuilder.usageHelpRequested |= is(argSpec, "usageHelp", option.usageHelp());
			}
		}

		private void updateHelpRequested(CommandSpec command) {
			isHelpRequested |= command.helpCommand();
		}

		private void validateConstraints(Stack<String> argumentStack, List<ArgSpec> required, Set<ArgSpec> matched) {
			if (!required.isEmpty()) {
				for (final ArgSpec missing : required) {
					Assert.assertTrue(missing.group() == null,
							"Arguments in a group are not necessarily required for the command");
					if (missing.isOption()) {
						maybeThrow(MissingParameterException.create(CommandLine.this, required, config().separator()));
					} else {
						assertNoMissingParameters(missing, missing.arity(), argumentStack);
					}
				}
			}
			if (!parseResultBuilder.unmatched.isEmpty()) {
				final String[] unmatched = parseResultBuilder.unmatched.toArray(new String[0]);
				for (final UnmatchedArgsBinding unmatchedArgsBinding : getCommandSpec().unmatchedArgsBindings()) {
					unmatchedArgsBinding.addAll(unmatched.clone());
				}
				if (!isUnmatchedArgumentsAllowed()) {
					maybeThrow(new UnmatchedArgumentException(CommandLine.this,
							Collections.unmodifiableList(parseResultBuilder.unmatched)));
				}
				final Tracer tracer = CommandLine.tracer();
				if (tracer.isInfo()) {
					tracer.info("Unmatched arguments: %s", parseResultBuilder.unmatched);
				}
			}
			final ParseResult pr = parseResultBuilder.build();
			pr.validateGroups();
		}

		/**
		 * Returns whether the next argument can be assigned to a vararg
		 * option/positional parameter.
		 * <p>
		 * Usually, we stop if we encounter '--', a command, or another option. However,
		 * if end-of-options has been reached, positional parameters may consume all
		 * remaining arguments.
		 * </p>
		 */
		private boolean varargCanConsumeNextValue(ArgSpec argSpec, String nextValue) {
			if (endOfOptions && argSpec.isPositional()) {
				return true;
			}
			if (isEndOfOptionsDelimiter(nextValue)) {
				return false;
			} // never consume EOO delim, regardless of parser config
			final boolean isCommand = isCommand(nextValue);
			final boolean commandsAllowed = argSpec.isOption()
					&& commandSpec.parser().allowSubcommandsAsOptionParameters();
			final boolean optionsAllowed = argSpec.isOption() && commandSpec.parser().allowOptionsAsOptionParameters();
			return (!isCommand || commandsAllowed) && (!isOption(nextValue) || optionsAllowed);
		}
	}

	/**
	 * Options or positional parameters can be assigned a {@code IParameterConsumer}
	 * that implements custom logic to process the parameters for this option or
	 * this position. When an option or positional parameter with a custom
	 * {@code IParameterConsumer} is matched on the command line, picocli's internal
	 * parser is temporarily suspended, and this object becomes responsible for
	 * consuming and processing as many command line arguments as needed.
	 * <p>
	 * This may be useful when passing through parameters to another command.
	 * </p>
	 * <p>
	 * Example usage:
	 * </p>
	 * 
	 * <pre>
	 * &#064;Command(name = "find")
	 * class Find {
	 * 	&#064;Option(names = "-exec", parameterConsumer = Find.ExecParameterConsumer.class)
	 * 	List&lt;String&gt; list = new ArrayList&lt;String&gt;();
	 *
	 * 	static class ExecParameterConsumer implements IParameterConsumer {
	 * 		public void consumeParameters(Stack&lt;String&gt; args, ArgSpec argSpec, CommandSpec commandSpec) {
	 * 			List&lt;String&gt; list = argSpec.getValue();
	 * 			while (!args.isEmpty()) {
	 * 				String arg = args.pop();
	 * 				list.add(arg);
	 *
	 * 				// `find -exec` semantics: stop processing after a ';' or '+' argument
	 * 				if (";".equals(arg) || "+".equals(arg)) {
	 * 					break;
	 * 				}
	 * 			}
	 * 		}
	 * 	}
	 * }
	 * </pre>
	 * <p>
	 * If this interface does not meet your requirements, you may have a look at the
	 * more powerful and flexible {@link IParameterPreprocessor} interface
	 * introduced with picocli 4.6.
	 * </p>
	 * 
	 * @see Option#parameterConsumer()
	 * @see Parameters#parameterConsumer()
	 * @since 4.0
	 */
	public interface IParameterConsumer {
		/**
		 * Consumes as many of the specified command line arguments as needed by popping
		 * them off the specified Stack. Implementors are free to ignore the
		 * {@linkplain ArgSpec#arity() arity} of the option or positional parameter,
		 * they are free to consume arguments that would normally be matched as other
		 * options of the command, and they are free to consume arguments that would
		 * normally be matched as an end-of-options delimiter.
		 * <p>
		 * Implementors are responsible for saving the consumed values; if the user
		 * object of the option or positional parameter is a Collection or a Map, a
		 * common approach would be to obtain the current instance via the
		 * {@link ArgSpec#getValue()}, and add to this instance. If the user object is
		 * an array, the implementation would need to create a new array that contains
		 * the old values as well as the newly consumed values, and store this array in
		 * the user object via the {@link ArgSpec#setValue(Object)}.
		 * </p>
		 * <p>
		 * If the user input is invalid, implementations should throw a
		 * {@link ParameterException} with a message to display to the user.
		 * </p>
		 * <p>
		 * When this method returns, the picocli parser will process the remaining
		 * arguments on the Stack.
		 * </p>
		 * 
		 * @param args        the command line arguments
		 * @param argSpec     the option or positional parameter for which to consume
		 *                    command line arguments
		 * @param commandSpec the command that the option or positional parameter
		 *                    belongs to
		 * @throws ParameterException if the user input is invalid
		 */
		void consumeParameters(Stack<String> args, ArgSpec argSpec, CommandSpec commandSpec);
	}

	/**
	 * Classes implementing this interface know how to handle
	 * {@code ParameterExceptions} (usually from invalid user input).
	 * <p>
	 * <b>Implementation Requirements:</b>
	 * </p>
	 * <p>
	 * Implementors that need to print messages to the console should use the
	 * {@linkplain #getOut() output} and {@linkplain #getErr() error} PrintWriters,
	 * and the {@linkplain #getColorScheme() color scheme} from the CommandLine
	 * object obtained from the exception.
	 * </p>
	 * <p>
	 * <b>Implementation Note:</b>
	 * </p>
	 * <p>
	 * See {@link #getParameterExceptionHandler()} for a description of the default
	 * handler.
	 * </p>
	 * <p>
	 * <b>API Note:</b>
	 * </p>
	 * <p>
	 * This interface supersedes {@link IExceptionHandler2}.
	 * </p>
	 * 
	 * @see CommandLine#setParameterExceptionHandler(IParameterExceptionHandler)
	 * @since 4.0
	 */
	public interface IParameterExceptionHandler {
		/**
		 * Handles a {@code ParameterException} that occurred while
		 * {@linkplain #parseArgs(String...) parsing} the command line arguments and
		 * returns an exit code suitable for returning from {@link #execute(String...)}.
		 * 
		 * @param ex   the ParameterException describing the problem that occurred while
		 *             parsing the command line arguments, and the CommandLine
		 *             representing the command or subcommand whose input was invalid
		 * @param args the command line arguments that could not be parsed
		 * @return an exit code
		 */
		int handleParseException(ParameterException ex, String[] args) throws Exception;
	}

	/**
	 * Options, positional parameters and commands can be assigned a
	 * {@code IParameterPreprocessor} that implements custom logic to preprocess the
	 * parameters for this option, position or command. When an option, positional
	 * parameter or command with a custom {@code IParameterPreprocessor} is matched
	 * on the command line, picocli's internal parser is temporarily suspended, and
	 * this custom logic is invoked.
	 * <p>
	 * This custom logic may completely replace picocli's internal parsing for this
	 * option, positional parameter or command, or it may do some preprocessing
	 * before picocli's internal parsing is resumed for this option, positional
	 * parameter or command.
	 * </p>
	 * <p>
	 * The "preprocessing" can include modifying the stack of command line
	 * parameters, or modifying the model.
	 * </p>
	 * <p>
	 * This may be useful when disambiguating input for commands that have both a
	 * positional parameter and an option with an optional parameter.
	 * </p>
	 * <p>
	 * Example usage:
	 * </p>
	 * 
	 * <pre>
	 * &#064;Command(name = "edit")
	 * class Edit {
	 * 	&#064;Parameters(index = "0", description = "The file to edit.")
	 * 	File file;
	 *
	 * 	enum Editor {
	 * 		defaultEditor, eclipse, idea, netbeans
	 * 	}
	 *
	 * 	&#064;Option(names = "--open", arity = "0..1", preprocessor = Edit.MyPreprocessor.class, description = {
	 * 			"Optionally specify the editor to use; if omitted the default editor is used. ",
	 * 			"Example: edit --open=idea FILE opens IntelliJ IDEA (notice the '=' separator)",
	 * 			"         edit --open FILE opens the specified file in the default editor" })
	 * 	Editor editor = Editor.defaultEditor;
	 *
	 * 	static class MyPreprocessor implements IParameterPreprocessor {
	 * 		public boolean preprocess(Stack&lt;String&gt; args, CommandSpec commandSpec, ArgSpec argSpec,
	 * 				Map&lt;String, Object&gt; info) {
	 * 			// we need to decide whether the next arg is the file to edit or the name of
	 * 			// the editor to use...
	 * 			if (" ".equals(info.get("separator"))) { // parameter was not attached to option
	 * 				args.push(Editor.defaultEditor.name()); // act as if the user specified --open=defaultEditor
	 * 			}
	 * 			return false; // picocli's internal parsing is resumed for this option
	 * 		}
	 * 	}
	 * }
	 * </pre>
	 * 
	 * @see Option#preprocessor()
	 * @see Parameters#preprocessor()
	 * @since 4.6
	 */
	public interface IParameterPreprocessor {
		/**
		 * Called when either the command, option or positional parameter that has this
		 * preprocessor configured was recognized by the picocli parser.
		 * <p>
		 * Implementors are free to modify one or more of the specified command line
		 * arguments before they are processed by the picocli parser (or by the option's
		 * {@link IParameterConsumer parameter consumer}, if one is specified).
		 * </p>
		 * <p>
		 * Implementors may optionally <em>consume</em> one or more of the specified
		 * command line arguments: a return value of {@code true} signals that the
		 * preprocessor consumed the parameter: picocli should skip further processing
		 * of this option or positional parameter, and the preprocessor implementation
		 * takes responsibility for assigning the option or positional parameter a
		 * value. A return value of {@code false} means that picocli should process the
		 * stack as usual for this option or positional parameter, and picocli is
		 * responsible for assigning this option or positional parameter a value.
		 * </p>
		 * <p>
		 * For a command, returning {@code true} signals that the preprocessor takes
		 * responsibility for parsing all options and positional parameters for this
		 * command, and takes responsibility for validating constraints like whether all
		 * required options and positional parameters were specified. A return value of
		 * {@code false} means that picocli should process the stack as usual for this
		 * command, and picocli is responsible for validation. Command preprocessors can
		 * signal back to the picocli parser when they detect that the user requested
		 * version information or usage help by putting a value of {@code true} in the
		 * specified {@code info} map for keys {@code versionHelpRequested} or
		 * {@code usageHelpRequested}, respectively.
		 * </p>
		 * <p>
		 * If the user input is invalid, implementations should throw a
		 * {@link ParameterException} with a message to display to the user.
		 * </p>
		 * 
		 * @param args        the remaining command line arguments that follow the
		 *                    matched argument (the matched argument is not on the stack
		 *                    anymore)
		 * @param commandSpec the command or subcommand that was matched (if the
		 *                    specified {@code argSpec} is {@code null}), or the command
		 *                    that the matched option or positional parameter belongs to
		 * @param argSpec     the option or positional parameter for which to
		 *                    pre-process command line arguments (may be {@code null}
		 *                    when this method is called for a subcommand that was
		 *                    matched)
		 * @param info        a map containing additional information on the current
		 *                    parser state, including whether the option parameter was
		 *                    attached to the option name with a `=` separator, whether
		 *                    quotes have already been stripped off the option, etc.
		 *                    Implementations may modify this map to communicate back to
		 *                    the picocli parser. Supported values:
		 *                    <table>
		 *                    <caption>Supported values in the info Map</caption>
		 *                    <tr>
		 *                    <th>key</th>
		 *                    <th>valid values</th>
		 *                    <th>type</th>
		 *                    </tr>
		 *                    <tr>
		 *                    <td>separator</td>
		 *                    <td>'' (empty string): attached without separator, ' '
		 *                    (space): not attached, (any other string): option name was
		 *                    attached to option param with specified separator</td>
		 *                    <td>java.lang.String</td>
		 *                    </tr>
		 *                    <tr>
		 *                    <td>negated</td>
		 *                    <td>{@code true}: the option or positional parameter is a
		 *                    {@linkplain Option#negatable() negated} option/parameter,
		 *                    {@code false}: the option or positional parameter is not a
		 *                    negated option/parameter</td>
		 *                    <td>java.lang.Boolean</td>
		 *                    </tr>
		 *                    <tr>
		 *                    <td>unquoted</td>
		 *                    <td>{@code true}: quotes surrounding the value have
		 *                    already been stripped off, {@code false}: quotes
		 *                    surrounding the value have not yet been stripped off</td>
		 *                    <td>java.lang.Boolean</td>
		 *                    </tr>
		 *                    <tr>
		 *                    <td>versionHelpRequested</td>
		 *                    <td>{@code true}: version help was requested,
		 *                    {@code false}: version help was not requested</td>
		 *                    <td>java.lang.Boolean</td>
		 *                    </tr>
		 *                    <tr>
		 *                    <td>usageHelpRequested</td>
		 *                    <td>{@code true}: usage help was requested, {@code false}:
		 *                    usage help was not requested</td>
		 *                    <td>java.lang.Boolean</td>
		 *                    </tr>
		 *                    </table>
		 * @return true if the preprocessor consumed the parameter and picocli should
		 *         skip further processing of the stack for this option or positional
		 *         parameter; false if picocli should continue processing the stack for
		 *         this option or positional parameter
		 * @throws ParameterException if the user input is invalid
		 */
		boolean preprocess(Stack<String> args, CommandSpec commandSpec, ArgSpec argSpec, Map<String, Object> info);
	}

	/**
	 * Represents a function that can process a List of {@code CommandLine} objects
	 * resulting from successfully {@linkplain #parse(String...) parsing} the
	 * command line arguments. This is a <a href=
	 * "https://docs.oracle.com/javase/8/docs/api/java/util/function/package-summary.html">functional
	 * interface</a> whose functional method is
	 * {@link #handleParseResult(List, PrintStream, CommandLine.Help.Ansi)}.
	 * <p>
	 * Implementations of this functions can be passed to the
	 * {@link #parseWithHandlers(IParseResultHandler, PrintStream, Help.Ansi, IExceptionHandler, String...)
	 * CommandLine::parseWithHandler} methods to take some next step after the
	 * command line was successfully parsed.
	 * </p>
	 * 
	 * @see RunFirst
	 * @see RunLast
	 * @see RunAll
	 * @deprecated Use {@link IExecutionStrategy} instead.
	 * @since 2.0
	 */
	@Deprecated
	public interface IParseResultHandler {
		/**
		 * Processes a List of {@code CommandLine} objects resulting from successfully
		 * {@linkplain #parse(String...) parsing} the command line arguments and
		 * optionally returns a list of results.
		 * 
		 * @param parsedCommands the {@code CommandLine} objects that resulted from
		 *                       successfully parsing the command line arguments
		 * @param out            the {@code PrintStream} to print help to if requested
		 * @param ansi           for printing help messages using ANSI styles and colors
		 * @return a list of results, or an empty list if there are no results
		 * @throws ParameterException if a help command was invoked for an unknown
		 *                            subcommand. Any {@code ParameterExceptions} thrown
		 *                            from this method are treated as if this exception
		 *                            was thrown during parsing and passed to the
		 *                            {@link IExceptionHandler}
		 * @throws ExecutionException if a problem occurred while processing the parse
		 *                            results; use
		 *                            {@link ExecutionException#getCommandLine()} to get
		 *                            the command or subcommand where processing failed
		 */
		List<Object> handleParseResult(List<CommandLine> parsedCommands, PrintStream out, Help.Ansi ansi)
				throws ExecutionException;
	}

	/**
	 * Represents a function that can process the {@code ParseResult} object
	 * resulting from successfully {@linkplain #parseArgs(String...) parsing} the
	 * command line arguments. This is a <a href=
	 * "https://docs.oracle.com/javase/8/docs/api/java/util/function/package-summary.html">functional
	 * interface</a> whose functional method is
	 * {@link IParseResultHandler2#handleParseResult(CommandLine.ParseResult)}.
	 * <p>
	 * Implementations of this function can be passed to the
	 * {@link #parseWithHandlers(IParseResultHandler2, IExceptionHandler2, String...)
	 * CommandLine::parseWithHandlers} methods to take some next step after the
	 * command line was successfully parsed.
	 * </p>
	 * <p>
	 * This interface replaces the {@link IParseResultHandler} interface; it takes
	 * the parse result as a {@code ParseResult} object instead of a List of
	 * {@code CommandLine} objects, and it has the freedom to select the
	 * {@link Help.Ansi} style to use and what {@code PrintStreams} to print to.
	 * </p>
	 * 
	 * @param <R> the return type of this handler
	 * @see RunFirst
	 * @see RunLast
	 * @see RunAll
	 * @deprecated use {@link IExecutionStrategy} instead, see
	 *             {@link #execute(String...)}
	 * @since 3.0
	 */
	@Deprecated
	public interface IParseResultHandler2<R> {
		/**
		 * Processes the {@code ParseResult} object resulting from successfully
		 * {@linkplain CommandLine#parseArgs(String...) parsing} the command line
		 * arguments and returns a return value.
		 * 
		 * @param parseResult the {@code ParseResult} that resulted from successfully
		 *                    parsing the command line arguments
		 * @throws ParameterException if a help command was invoked for an unknown
		 *                            subcommand. Any {@code ParameterExceptions} thrown
		 *                            from this method are treated as if this exception
		 *                            was thrown during parsing and passed to the
		 *                            {@link IExceptionHandler2}
		 * @throws ExecutionException if a problem occurred while processing the parse
		 *                            results; use
		 *                            {@link ExecutionException#getCommandLine()} to get
		 *                            the command or subcommand where processing failed
		 */
		R handleParseResult(ParseResult parseResult) throws ExecutionException;
	}

	/**
	 * <p>
	 * When parsing command line arguments and initializing fields annotated with
	 * {@link Option @Option} or {@link Parameters @Parameters}, String values can
	 * be converted to any type for which a {@code ITypeConverter} is registered.
	 * </p>
	 * <p>
	 * This interface defines the contract for classes that know how to convert a
	 * String into some domain object. Custom converters can be registered with the
	 * {@link #registerConverter(Class, ITypeConverter)} method.
	 * </p>
	 * <p>
	 * Java 8 lambdas make it easy to register custom type converters:
	 * </p>
	 * 
	 * <pre>
	 * commandLine.registerConverter(java.nio.file.Path.class, s -&gt; java.nio.file.Paths.get(s));
	 * commandLine.registerConverter(java.time.Duration.class, s -&gt; java.time.Duration.parse(s));
	 * </pre>
	 * <p>
	 * Built-in type converters are pre-registered for the following java 1.5 types:
	 * </p>
	 * <ul>
	 * <li>all primitive types</li>
	 * <li>all primitive wrapper types: Boolean, Byte, Character, Double, Float,
	 * Integer, Long, Short</li>
	 * <li>any enum</li>
	 * <li>java.io.File</li>
	 * <li>java.math.BigDecimal</li>
	 * <li>java.math.BigInteger</li>
	 * <li>java.net.InetAddress</li>
	 * <li>java.net.URI</li>
	 * <li>java.net.URL</li>
	 * <li>java.nio.charset.Charset</li>
	 * <li>java.sql.Time</li>
	 * <li>java.util.Date</li>
	 * <li>java.util.UUID</li>
	 * <li>java.util.regex.Pattern</li>
	 * <li>StringBuilder</li>
	 * <li>CharSequence</li>
	 * <li>String</li>
	 * </ul>
	 * 
	 * @param <K> the type of the object that is the result of the conversion
	 */
	public interface ITypeConverter<K> {
		/**
		 * Converts the specified command line argument value to some domain object.
		 * 
		 * @param value the command line argument String value
		 * @return the resulting domain object
		 * @throws Exception               an exception detailing what went wrong during
		 *                                 the conversion. Any exception thrown from
		 *                                 this method will be caught and shown to the
		 *                                 end user. An example error message shown to
		 *                                 the end user could look something like this:
		 *                                 {@code Invalid value for option '--some-option': cannot convert 'xxxinvalidinput' to SomeType (java.lang.IllegalArgumentException: Invalid format: must be 'x:y:z' but was 'xxxinvalidinput')}
		 * @throws TypeConversionException throw this exception to have more control
		 *                                 over the error message that is shown to the
		 *                                 end user when type conversion fails. An
		 *                                 example message shown to the user could look
		 *                                 like this:
		 *                                 {@code Invalid value for option '--some-option': Invalid format: must be 'x:y:z' but was 'xxxinvalidinput'}
		 */
		K convert(String value) throws Exception;
	}

	/**
	 * Provides version information for a command. Commands may configure a provider
	 * with the {@link Command#versionProvider()} annotation attribute.
	 * 
	 * @since 2.2
	 */
	public interface IVersionProvider {
		/**
		 * Returns version information for a command.
		 * 
		 * @return version information (each string in the array is displayed on a
		 *         separate line)
		 * @throws Exception an exception detailing what went wrong when obtaining
		 *                   version information
		 */
		String[] getVersion() throws Exception;
	}

	private enum LookBehind {
		SEPARATE, ATTACHED, ATTACHED_WITH_SEPARATOR;

		static LookBehind parse(String separator) {
			if ("".equals(separator)) {
				return LookBehind.ATTACHED;
			}
			if (" ".equals(separator)) {
				return LookBehind.SEPARATE;
			}
			return ATTACHED_WITH_SEPARATOR;
		}

		public boolean isAttached() {
			return this != LookBehind.SEPARATE;
		}

		String toString(String separator) {
			return this == LookBehind.ATTACHED ? "" : this == LookBehind.SEPARATE ? " " : separator;
		}
	}

	/**
	 * Exception indicating that more values were specified for an option or
	 * parameter than its {@link Option#arity() arity} allows.
	 */
	public static class MaxValuesExceededException extends ParameterException {
		private static final long serialVersionUID = 6536145439570100641L;

		public MaxValuesExceededException(CommandLine commandLine, String msg) {
			super(commandLine, msg);
		}
	}

	/**
	 * Exception indicating that a required parameter was not specified.
	 */
	public static class MissingParameterException extends ParameterException {
		private static final long serialVersionUID = 5075678535706338753L;

		private static MissingParameterException create(CommandLine cmd, Collection<ArgSpec> missing,
				String separator) {
			final String missingArgs = ArgSpec.describe(missing, ", ", separator, "'", "'");
			final String types = ArgSpec.describeTypes(missing);
			return new MissingParameterException(cmd, missing, "Missing required " + types + ": " + missingArgs);
		}

		private final List<ArgSpec> missing;

		public MissingParameterException(CommandLine commandLine, ArgSpec missing, String msg) {
			this(commandLine, Collections.singletonList(missing), msg);
		}

		public MissingParameterException(CommandLine commandLine, Collection<ArgSpec> missing, String msg) {
			super(commandLine, msg);
			this.missing = Collections.unmodifiableList(new ArrayList<ArgSpec>(missing));
		}

		public List<ArgSpec> getMissing() {
			return missing;
		}
	}

	/**
	 * Exception indicating that an annotated field had a type for which no
	 * {@link ITypeConverter} was
	 * {@linkplain #registerConverter(Class, ITypeConverter) registered}.
	 */
	public static class MissingTypeConverterException extends ParameterException {
		private static final long serialVersionUID = -6050931703233083760L;

		public MissingTypeConverterException(CommandLine commandLine, String msg) {
			super(commandLine, msg);
		}
	}

	/**
	 * <p>
	 * Fields annotated with {@code @Mixin} are "expanded" into the current command:
	 * {@link Option @Option} and {@link Parameters @Parameters} in the mixin class
	 * are added to the options and positional parameters of this command. A
	 * {@link DuplicateOptionAnnotationsException} is thrown if any of the options
	 * in the mixin has the same name as an option in this command.
	 * </p>
	 * <p>
	 * The {@code Mixin} annotation provides a way to reuse common options and
	 * parameters without subclassing. For example:
	 * </p>
	 * 
	 * <pre>
	 * &#064;Command(name = "HelloWorld")
	 * class HelloWorld implements Runnable {
	 *
	 * 	// adds the --help and --version options to this command
	 * 	&#064;Mixin
	 * 	private HelpOptions options = new HelpOptions();
	 *
	 * 	&#064;Option(names = { "-u", "--userName" }, required = true, description = "The user name")
	 * 	String userName;
	 *
	 * 	public void run() {
	 * 		System.out.println("Hello, " + userName);
	 * 	}
	 * }
	 *
	 * // Common reusable help options.
	 * class HelpOptions {
	 *
	 * 	&#064;Option(names = { "-h", "--help" }, usageHelp = true, description = "Display this help and exit")
	 * 	private boolean help;
	 *
	 * 	&#064;Option(names = { "-V", "--version" }, versionHelp = true, description = "Display version info and exit")
	 * 	private boolean versionHelp;
	 * }
	 * </pre>
	 * 
	 * @since 3.0
	 */
	@Retention(RetentionPolicy.RUNTIME)
	@Target({ ElementType.FIELD, ElementType.PARAMETER })
	public @interface Mixin {
		/**
		 * Optionally specify a name that the mixin object can be retrieved with from
		 * the {@code CommandSpec}. If not specified the name of the annotated field is
		 * used.
		 * 
		 * @return a String to register the mixin object with, or an empty String if the
		 *         name of the annotated field should be used
		 */
		String name() default "";
	}

	/**
	 * This class provides a namespace for classes and interfaces that model
	 * concepts and attributes of command line interfaces in picocli.
	 * 
	 * @since 3.0
	 */
	public static final class Model {
		/**
		 * The {@code ArgGroupSpec} class models a {@link ArgGroup group} of arguments
		 * (options, positional parameters or a mixture of the two).
		 * 
		 * @see ArgGroup
		 * @since 4.0
		 */
		public static class ArgGroupSpec implements IOrdered {
			/**
			 * Builder responsible for creating valid {@code ArgGroupSpec} objects.
			 * 
			 * @since 4.0
			 */
			public static class Builder {
				private IGetter getter;
				private ISetter setter;
				private IScope scope;
				private ITypeInfo typeInfo;
				private String heading;
				private String headingKey;
				private boolean exclusive = true;
				private Range multiplicity = Range.valueOf("0..1");
				private boolean validate = true;
				private int order = DEFAULT_ORDER;
				private final List<ArgSpec> args = new ArrayList<ArgSpec>();
				private final List<ArgGroupSpec> subgroups = new ArrayList<ArgGroupSpec>();
				private final List<IAnnotatedElement> specElements = new ArrayList<IAnnotatedElement>();

				// for topological sorting; private only
				private Boolean topologicalSortDone;
				private final List<Builder> compositesReferencingMe = new ArrayList<Builder>();

				Builder() {
				}

				Builder(IAnnotatedElement source) {
					typeInfo = source.getTypeInfo();
					getter = source.getter();
					setter = source.setter();
					scope = source.scope();
				}

				/**
				 * Adds the specified argument to the list of options and positional parameters
				 * that depend on this group.
				 */
				public Builder addArg(ArgSpec arg) {
					args.add(arg);
					return this;
				}

				/**
				 * Adds the specified {@code {@literal @}Spec} annotated program element to the
				 * list of spec elements for this group.
				 * 
				 * @since 4.6
				 */
				public Builder addSpecElement(IAnnotatedElement element) {
					specElements.add(element);
					return this;
				}

				/**
				 * Adds the specified group to the list of subgroups that this group is composed
				 * of.
				 */
				public Builder addSubgroup(ArgGroupSpec group) {
					subgroups.add(group);
					return this;
				}

				/**
				 * Returns the list of options and positional parameters that depend on this
				 * group.
				 */
				public List<ArgSpec> args() {
					return args;
				}

				/** Returns a valid {@code ArgGroupSpec} instance. */
				public ArgGroupSpec build() {
					return new ArgGroupSpec(this);
				}

				/**
				 * Returns whether this is a mutually exclusive group; {@code true} by default.
				 * If {@code false}, this is a co-occurring group. Ignored if
				 * {@link #validate()} is {@code false}.
				 * 
				 * @see ArgGroup#exclusive()
				 */
				public boolean exclusive() {
					return exclusive;
				}

				/**
				 * Sets whether this is a mutually exclusive group; {@code true} by default. If
				 * {@code false}, this is a co-occurring group. Ignored if {@link #validate()}
				 * is {@code false}.
				 * 
				 * @see ArgGroup#exclusive()
				 */
				public Builder exclusive(boolean newValue) {
					exclusive = newValue;
					return this;
				}

				/**
				 * Returns the {@link IGetter} that is responsible for supplying the value of
				 * the annotated program element associated with this group.
				 */
				public IGetter getter() {
					return getter;
				}

				/**
				 * Sets the {@link IGetter} that is responsible for getting the value of the
				 * annotated program element associated with this group, and returns this
				 * builder.
				 */
				public Builder getter(IGetter getter) {
					this.getter = getter;
					return this;
				}

				/**
				 * Returns the heading of this group, used when generating the usage
				 * documentation.
				 * 
				 * @see ArgGroup#heading()
				 */
				public String heading() {
					return heading;
				}

				/**
				 * Sets the heading of this group (may be {@code null}), used when generating
				 * the usage documentation.
				 * 
				 * @see ArgGroup#heading()
				 */
				public Builder heading(String newValue) {
					this.heading = newValue;
					return this;
				}

				/**
				 * Returns the heading key of this group, used to get the heading from a
				 * resource bundle.
				 * 
				 * @see ArgGroup#headingKey()
				 */
				public String headingKey() {
					return headingKey;
				}

				/**
				 * Sets the heading key of this group, used to get the heading from a resource
				 * bundle.
				 * 
				 * @see ArgGroup#headingKey()
				 */
				public Builder headingKey(String newValue) {
					this.headingKey = newValue;
					return this;
				}

				/**
				 * Returns the multiplicity of this group: how many occurrences it may have on
				 * the command line; {@code "0..1"} (optional) by default. A group can be made
				 * required by specifying a multiplicity of {@code "1"}. For a group of mutually
				 * exclusive arguments, being required means that one of the arguments in the
				 * group must appear on the command line, or a MissingParameterException is
				 * thrown. For a group of co-occurring arguments, being required means that all
				 * arguments in the group must appear on the command line. Ignored if
				 * {@link #validate()} is {@code false}.
				 * 
				 * @see ArgGroup#multiplicity()
				 */
				public Range multiplicity() {
					return multiplicity;
				}

				/**
				 * Sets the multiplicity of this group: how many occurrences it may have on the
				 * command line; {@code "0..1"} (optional) by default. A group can be made
				 * required by specifying a multiplicity of {@code "1"}. For a group of mutually
				 * exclusive arguments, being required means that one of the arguments in the
				 * group must appear on the command line, or a MissingParameterException is
				 * thrown. For a group of co-occurring arguments, being required means that all
				 * arguments in the group must appear on the command line. Ignored if
				 * {@link #validate()} is {@code false}.
				 * 
				 * @see ArgGroup#multiplicity()
				 */
				public Builder multiplicity(Range newValue) {
					multiplicity = newValue;
					return this;
				}

				/**
				 * Sets the multiplicity of this group: how many occurrences it may have on the
				 * command line; {@code "0..1"} (optional) by default. A group can be made
				 * required by specifying a multiplicity of {@code "1"}. For a group of mutually
				 * exclusive arguments, being required means that one of the arguments in the
				 * group must appear on the command line, or a MissingParameterException is
				 * thrown. For a group of co-occurring arguments, being required means that all
				 * arguments in the group must appear on the command line. Ignored if
				 * {@link #validate()} is {@code false}.
				 * 
				 * @see ArgGroup#multiplicity()
				 */
				public Builder multiplicity(String newValue) {
					return multiplicity(Range.valueOf(newValue));
				}

				/**
				 * Returns the position in the options list in the usage help message at which
				 * this group should be shown. Groups with a lower number are shown before
				 * groups with a higher number. This attribute is only honored for groups that
				 * have a {@link #heading() heading} (or a {@link #headingKey() headingKey} with
				 * a non-{@code null} resource bundle value).
				 */
				public int order() {
					return order;
				}

				/**
				 * Sets the position in the options list in the usage help message at which this
				 * group should be shown, and returns this builder.
				 */
				public Builder order(int order) {
					this.order = order;
					return this;
				}

				/**
				 * Returns the {@link IScope} that determines where the setter sets the value
				 * (or the getter gets the value) of the annotated program element associated
				 * with this group.
				 */
				public IScope scope() {
					return scope;
				}

				/**
				 * Sets the {@link IScope} that targets where the setter sets the value of the
				 * annotated program element associated with this group, and returns this
				 * builder.
				 */
				public Builder scope(IScope scope) {
					this.scope = scope;
					return this;
				}

				/**
				 * Returns the {@link ISetter} that is responsible for modifying the value of
				 * the annotated program element associated with this group.
				 */
				public ISetter setter() {
					return setter;
				}

				/**
				 * Sets the {@link ISetter} that is responsible for modifying the value of the
				 * annotated program element associated with this group, and returns this
				 * builder.
				 */
				public Builder setter(ISetter setter) {
					this.setter = setter;
					return this;
				}

				/**
				 * Returns the list of program elements annotated with {@code {@literal @}Spec}
				 * configured for this group.
				 * 
				 * @since 4.6
				 */
				public List<IAnnotatedElement> specElements() {
					return specElements;
				}

				/** Returns the list of subgroups that this group is composed of. */
				public List<ArgGroupSpec> subgroups() {
					return subgroups;
				}

				/**
				 * Returns the type info for the annotated program element associated with this
				 * group.
				 * 
				 * @return type information that does not require {@code Class} objects and be
				 *         constructed both at runtime and compile time
				 */
				public ITypeInfo typeInfo() {
					return typeInfo;
				}

				/**
				 * Sets the type info for the annotated program element associated with this
				 * group, and returns this builder.
				 * 
				 * @param newValue type information that does not require {@code Class} objects
				 *                 and be constructed both at runtime and compile time
				 */
				public Builder typeInfo(ITypeInfo newValue) {
					this.typeInfo = newValue;
					return this;
				}

				/**
				 * Updates this builder from the specified annotation values.
				 * 
				 * @param group annotation values
				 * @return this builder for method chaining
				 */
				public Builder updateArgGroupAttributes(ArgGroup group) {
					return this.heading(group.heading()).headingKey(group.headingKey()).exclusive(group.exclusive())
							.multiplicity(group.multiplicity()).validate(group.validate()).order(group.order());
				}

				/**
				 * Returns whether picocli should validate the rules of this group: for a
				 * mutually exclusive group this means that no more than one arguments in the
				 * group is specified on the command line; for a co-ocurring group this means
				 * that all arguments in the group are specified on the command line.
				 * {@code true} by default.
				 * 
				 * @see ArgGroup#validate()
				 */
				public boolean validate() {
					return validate;
				}

				/**
				 * Sets whether picocli should validate the rules of this group: for a mutually
				 * exclusive group this means that no more than one arguments in the group is
				 * specified on the command line; for a co-ocurring group this means that all
				 * arguments in the group are specified on the command line. {@code true} by
				 * default.
				 * 
				 * @see ArgGroup#validate()
				 */
				public Builder validate(boolean newValue) {
					validate = newValue;
					return this;
				}
			}

			static final int DEFAULT_ORDER = -1;
			private static final String NO_HEADING = "__no_heading__";
			private static final String NO_HEADING_KEY = "__no_heading_key__";

			/**
			 * Returns a new {@link Builder}.
			 * 
			 * @return a new ArgGroupSpec.Builder instance
			 */
			public static Builder builder() {
				return new Builder();
			}

			/**
			 * Returns a new {@link Builder} associated with the specified annotated
			 * element.
			 * 
			 * @param annotatedElement the annotated element containing {@code @Option} and
			 *                         {@code @Parameters}
			 * @return a new ArgGroupSpec.Builder instance
			 */
			public static Builder builder(IAnnotatedElement annotatedElement) {
				return new Builder(Assert.notNull(annotatedElement, "annotatedElement"));
			}

			private static String quote(String s) {
				return s == null ? "null" : "'" + s + "'";
			}

			private final String heading;
			private final String headingKey;
			private final boolean exclusive;
			private final Range multiplicity;
			private final boolean validate;
			private final int order;
			private final IGetter getter;
			private final ISetter setter;
			private final IScope scope;
			private final ITypeInfo typeInfo;
			private final List<ArgGroupSpec> subgroups;
			private final Set<ArgSpec> args;

			private Messages messages;

			private ArgGroupSpec parentGroup;

			private String id = "1";

			private final List<IAnnotatedElement> specElements;

			ArgGroupSpec(ArgGroupSpec.Builder builder) {
				heading = NO_HEADING.equals(builder.heading) ? null : builder.heading;
				headingKey = NO_HEADING_KEY.equals(builder.headingKey) ? null : builder.headingKey;
				exclusive = builder.exclusive && builder.validate; // non-validating groups cannot be exclusive:
																	// https://github.com/remkop/picocli/issues/810
				multiplicity = builder.multiplicity;
				if (multiplicity.max() <= 0) {
					throw new InitializationException(
							"ArgGroup must have multiplicity that allows at least one occurrence, but had multiplicity="
									+ multiplicity);
				}

				validate = builder.validate;
				order = builder.order;
				typeInfo = builder.typeInfo;
				getter = builder.getter;
				setter = builder.setter;
				scope = builder.scope;

				specElements = Collections.unmodifiableList(new ArrayList<IAnnotatedElement>(builder.specElements()));
				args = Collections.unmodifiableSet(new LinkedHashSet<ArgSpec>(builder.args()));
				subgroups = Collections.unmodifiableList(new ArrayList<ArgGroupSpec>(builder.subgroups()));
				if (args.isEmpty() && subgroups.isEmpty()) {
					throw new InitializationException(
							"ArgGroup has no options or positional parameters, and no subgroups: "
									+ (getter == null ? setter : getter) + " in " + scope);
				}

				int i = 1;
				for (final ArgGroupSpec sub : subgroups) {
					sub.parentGroup = this;
					sub.id = id + "." + i++;
				}
				for (final ArgSpec arg : args) {
					arg.group = this;
				}

				if (!validate && builder.exclusive) {
					CommandLine.tracer().info("Setting exclusive=%s because %s is a non-validating group.", exclusive,
							synopsisUnit());
				}
				if (exclusive) {
					String modifiedArgs = "";
					String sep = "";
					for (final ArgSpec arg : args) {
						if (!arg.required()) {
							modifiedArgs += sep + (arg.isOption() ? ((OptionSpec) arg).longestName()
									: (arg.paramLabel() + "[" + ((PositionalParamSpec) arg).index() + "]"));
							sep = ",";
							// Keep initial required as originallyRequired for Issue#1380
							// https://github.com/remkop/picocli/issues/1380
							arg.originallyRequired = true;
							arg.required = true;
						}
					}
					if (modifiedArgs.length() > 0) {
						CommandLine.tracer().info("Made %s required in the group because %s is an exclusive group.",
								modifiedArgs, synopsisUnit());
					}
				}
			}

			private List<OptionSpec> addGroupOptionsToListRecursively(List<OptionSpec> result) {
				result.addAll(options());
				for (final ArgGroupSpec subGroup : subgroups()) {
					subGroup.addGroupOptionsToListRecursively(result);
				}
				return result;
			}

			private List<PositionalParamSpec> addGroupPositionalsToListRecursively(List<PositionalParamSpec> result) {
				result.addAll(positionalParameters());
				for (final ArgGroupSpec subGroup : subgroups()) {
					subGroup.addGroupPositionalsToListRecursively(result);
				}
				return result;
			}

			/**
			 * Returns all options configured for this group and all subgroups.
			 * 
			 * @return an immutable list of all options in this group and its subgroups.
			 * @since 4.4
			 */
			public List<OptionSpec> allOptionsNested() {
				return addGroupOptionsToListRecursively(new ArrayList<OptionSpec>());
			}

			/**
			 * Returns all positional parameters configured for this group and all
			 * subgroups.
			 * 
			 * @return an immutable list of all positional parameters in this group and its
			 *         subgroups.
			 * @since 4.4
			 */
			public List<PositionalParamSpec> allPositionalParametersNested() {
				return addGroupPositionalsToListRecursively(new ArrayList<PositionalParamSpec>());
			}

			int argCount() {
				int result = args.size();
				for (final ArgGroupSpec sub : subgroups()) {
					result += sub.argCount();
				}
				return result;
			}

			/**
			 * Returns the options and positional parameters in this group; may be empty but
			 * not {@code null}.
			 */
			public Set<ArgSpec> args() {
				return args;
			}

			private Text concatOptionText(String prefix, Text text, Help.ColorScheme colorScheme, OptionSpec option) {
				return Help.concatOptionText(prefix, text, colorScheme, option,
						createLabelRenderer(option.commandSpec));
			}

			private Text concatPositionalText(String prefix, Text text, Help.ColorScheme colorScheme,
					PositionalParamSpec positionalParam) {
				return Help.concatPositionalText(prefix, text, colorScheme, positionalParam,
						createLabelRenderer(positionalParam.commandSpec));
			}

			public Help.IParamLabelRenderer createLabelRenderer(CommandSpec commandSpec) {
				return new Help.DefaultParamLabelRenderer(commandSpec == null ? CommandSpec.create() : commandSpec);
			}

			@Override
			public boolean equals(Object obj) {
				if (obj == this) {
					return true;
				}
				if (!(obj instanceof ArgGroupSpec)) {
					return false;
				}
				final ArgGroupSpec other = (ArgGroupSpec) obj;
				return exclusive == other.exclusive && Assert.equals(multiplicity, other.multiplicity)
						&& validate == other.validate && order == other.order && Assert.equals(heading, other.heading)
						&& Assert.equals(headingKey, other.headingKey) && Assert.equals(subgroups, other.subgroups)
						&& Assert.equals(args, other.args);
			}

			/**
			 * Returns whether this is a mutually exclusive group; {@code true} by default.
			 * If {@code false}, this is a co-occurring group. Ignored if
			 * {@link #validate()} is {@code false}.
			 * 
			 * @see ArgGroup#exclusive()
			 */
			public boolean exclusive() {
				return exclusive;
			}

			/**
			 * Returns the {@link IGetter} that is responsible for supplying the value of
			 * the annotated program element associated with this group.
			 */
			public IGetter getter() {
				return getter;
			}

			@Override
			public int hashCode() {
				int result = 17;
				result += 37 * result + Assert.hashCode(exclusive);
				result += 37 * result + Assert.hashCode(multiplicity);
				result += 37 * result + Assert.hashCode(validate);
				result += 37 * result + order;
				result += 37 * result + Assert.hashCode(heading);
				result += 37 * result + Assert.hashCode(headingKey);
				result += 37 * result + Assert.hashCode(subgroups);
				result += 37 * result + Assert.hashCode(args);
				return result;
			}

			/**
			 * Returns the heading of this group (may be {@code null}), used when generating
			 * the usage documentation.
			 * 
			 * @see ArgGroup#heading()
			 */
			public String heading() {
				if (messages() == null) {
					return heading;
				}
				final String newValue = messages().getString(headingKey(), null);
				if (newValue != null) {
					return newValue;
				}
				return heading;
			}

			/**
			 * Returns the heading key of this group (may be {@code null}), used to get the
			 * heading from a resource bundle.
			 * 
			 * @see ArgGroup#headingKey()
			 */
			public String headingKey() {
				return headingKey;
			}

			String id() {
				return id;
			}

			void initUserObject(CommandLine commandLine) {
				if (commandLine == null) {
					CommandLine.tracer().debug("Could not create user object for %s with null CommandLine%n.", this);
					return;
				}
				try {
					tryInitUserObject(commandLine);
				} catch (final PicocliException ex) {
					throw ex;
				} catch (final Exception ex) {
					throw new InitializationException("Could not create user object for " + this, ex);
				}
			}

			/**
			 * Returns {@code true} if this group is a subgroup (or a nested sub-subgroup,
			 * to any level of depth) of the specified group, {@code false} otherwise.
			 * 
			 * @param group the group to check if it contains this group
			 * @return {@code true} if this group is a subgroup or a nested sub-subgroup of
			 *         the specified group
			 */
			public boolean isSubgroupOf(ArgGroupSpec group) {
				for (final ArgGroupSpec sub : group.subgroups) {
					if (this == sub) {
						return true;
					}
					if (isSubgroupOf(sub)) {
						return true;
					}
				}
				return false;
			}

			int localPositionalParamCount() {
				int result = 0;
				for (final ArgSpec arg : args) {
					if (arg.isPositional()) {
						result += ((PositionalParamSpec) arg).capacity().max();
					}
				}
				return result;
			}

			/**
			 * Returns the Messages for this argument group specification, or {@code null}.
			 */
			public Messages messages() {
				return messages;
			}

			/**
			 * Sets the Messages for this ArgGroupSpec, and returns this ArgGroupSpec.
			 * 
			 * @param msgs the new Messages value, may be {@code null}
			 * @see Command#resourceBundle()
			 * @see #headingKey()
			 */
			public ArgGroupSpec messages(Messages msgs) {
				messages = msgs;
				for (final ArgGroupSpec sub : subgroups()) {
					sub.messages(msgs);
				}
				return this;
			}

			/**
			 * Returns the multiplicity of this group: how many occurrences it may have on
			 * the command line; {@code "0..1"} (optional) by default. A group can be made
			 * required by specifying a multiplicity of {@code "1"}. For a group of mutually
			 * exclusive arguments, being required means that one of the arguments in the
			 * group must appear on the command line, or a MissingParameterException is
			 * thrown. For a group of co-occurring arguments, being required means that all
			 * arguments in the group must appear on the command line. Ignored if
			 * {@link #validate()} is {@code false}.
			 * 
			 * @see ArgGroup#multiplicity()
			 */
			public Range multiplicity() {
				return multiplicity;
			}

			/**
			 * Returns the list of options configured for this group.
			 * 
			 * @return an immutable list of options in this group.
			 */
			public List<OptionSpec> options() {
				final List<OptionSpec> result = new ArrayList<OptionSpec>();
				for (final ArgSpec arg : args()) {
					if (arg instanceof OptionSpec) {
						result.add((OptionSpec) arg);
					}
				}
				return Collections.unmodifiableList(result);
			}

			/**
			 * Returns the position in the options list in the usage help message at which
			 * this group should be shown. Groups with a lower number are shown before
			 * groups with a higher number. This attribute is only honored for groups that
			 * have a {@link #heading() heading} (or a {@link #headingKey() headingKey} with
			 * a non-{@code null} resource bundle value).
			 */
			@Override
			public int order() {
				return this.order;
			}

			/**
			 * Returns the parent group that this group is part of, or {@code null} if this
			 * group is not part of a composite.
			 */
			public ArgGroupSpec parentGroup() {
				return parentGroup;
			}

			/**
			 * Returns the list of positional parameters configured for this group.
			 * 
			 * @return an immutable list of positional parameters in this group.
			 */
			public List<PositionalParamSpec> positionalParameters() {
				final List<PositionalParamSpec> result = new ArrayList<PositionalParamSpec>();
				for (final ArgSpec arg : args()) {
					if (arg instanceof PositionalParamSpec) {
						result.add((PositionalParamSpec) arg);
					}
				}
				return Collections.unmodifiableList(result);
			}

			private Text rawSynopsisUnitText(Help.ColorScheme colorScheme, Set<ArgSpec> outparam_groupArgs) {
				final String infix = exclusive() ? " | " : " ";
				Text synopsis = colorScheme.ansi().new Text(0);

				// if not sorted alphabetically then SortByOrder
				final boolean sortExplicitly = !args.isEmpty() && args.iterator().next().command() != null
						&& !args.iterator().next().command().usageMessage().sortSynopsis();
				if (sortExplicitly) {
					final List<IOrdered> sortableComponents = new ArrayList<IOrdered>();
					final List<PositionalParamSpec> remainder = new ArrayList<PositionalParamSpec>();
					for (final ArgSpec arg : args()) {
						if (arg instanceof OptionSpec) {
							sortableComponents.add((IOrdered) arg);
						} else {
							remainder.add((PositionalParamSpec) arg);
						}
					}
					sortableComponents.addAll(subgroups());
					Collections.sort(sortableComponents, new Help.SortByOrder<IOrdered>());
					for (final IOrdered ordered : sortableComponents) {
						final String prefix = synopsis.length > 0 ? infix : "";
						if (ordered instanceof OptionSpec) {
							synopsis = concatOptionText(prefix, synopsis, colorScheme, (OptionSpec) ordered);
							outparam_groupArgs.add((OptionSpec) ordered);
						} else {
							synopsis = synopsis
									.concat(((ArgGroupSpec) ordered).synopsisText(colorScheme, outparam_groupArgs));
						}
					}
					for (final PositionalParamSpec positional : remainder) {
						final String prefix = synopsis.length > 0 ? infix : "";
						synopsis = concatPositionalText(prefix, synopsis, colorScheme, positional);
						outparam_groupArgs.add(positional);
					}
				} else {
					for (final ArgSpec arg : args()) {
						final String prefix = synopsis.length > 0 ? infix : "";
						if (arg instanceof OptionSpec) {
							synopsis = concatOptionText(prefix, synopsis, colorScheme, (OptionSpec) arg);
						} else {
							synopsis = concatPositionalText(prefix, synopsis, colorScheme, (PositionalParamSpec) arg);
						}
						outparam_groupArgs.add(arg);
					}
					for (final ArgGroupSpec subgroup : subgroups()) {
						if (synopsis.length > 0) {
							synopsis = synopsis.concat(infix);
						}
						synopsis = synopsis.concat(subgroup.synopsisText(colorScheme, outparam_groupArgs));
					}
				}
				return synopsis;
			}

			/**
			 * Returns the required options and positional parameters in this group; may be
			 * empty but not {@code null}.
			 */
			public Set<ArgSpec> requiredArgs() {
				final Set<ArgSpec> result = new LinkedHashSet<ArgSpec>(args);
				for (final Iterator<ArgSpec> iter = result.iterator(); iter.hasNext();) {
					if (!iter.next().required()) {
						iter.remove();
					}
				}
				return Collections.unmodifiableSet(result);
			}

			/**
			 * Returns the {@link IScope} that determines where the setter sets the value
			 * (or the getter gets the value) of the annotated program element associated
			 * with this group.
			 */
			public IScope scope() {
				return scope;
			}

			/**
			 * Returns the {@link ISetter} that is responsible for modifying the value of
			 * the annotated program element associated with this group.
			 */
			public ISetter setter() {
				return setter;
			}

			void setUserObject(Object userObject, IFactory factory) throws Exception {
				if (typeInfo().isCollection()) {
					Collection<Object> c = getter().get();
					if (c == null) {
						@SuppressWarnings("unchecked")
						final Collection<Object> c2 = (Collection<Object>) DefaultFactory.create(factory,
								typeInfo.getType());
						setter().set(c = c2);
					}
					(c).add(userObject);
				} else if (typeInfo().isArray()) {
					final Object old = getter().get();
					final int oldSize = old == null ? 0 : Array.getLength(old);
					final Object array = Array.newInstance(typeInfo().getAuxiliaryTypes()[0], oldSize + 1);
					for (int i = 0; i < oldSize; i++) {
						Array.set(array, i, Array.get(old, i));
					}
					Array.set(array, oldSize, userObject);
					setter().set(array);
				} else {
					setter().set(userObject);
				}
			}

			/**
			 * Returns the list of program elements annotated with {@code {@literal @}Spec}
			 * configured for this group.
			 * 
			 * @since 4.6
			 */
			public List<IAnnotatedElement> specElements() {
				return specElements;
			}

			/**
			 * Return the subgroups that this group is composed of; may be empty but not
			 * {@code null}.
			 * 
			 * @return immutable list of subgroups that this group is composed of.
			 */
			public List<ArgGroupSpec> subgroups() {
				return subgroups;
			}

			/** Returns the synopsis of this group. */
			public String synopsis() {
				return synopsisText(new Help.ColorScheme.Builder(Help.Ansi.OFF).build(), new HashSet<ArgSpec>())
						.toString();
			}

			/**
			 * Returns the synopsis of this group.
			 * 
			 * @param colorScheme        the color scheme to use for options and positional
			 *                           parameters in this group and subgroups
			 * @param outparam_groupArgs all options and positional parameters in the groups
			 *                           this method generates a synopsis for; these options
			 *                           and positional parameters should be excluded from
			 *                           appearing elsewhere in the synopsis
			 * @return the synopsis Text
			 */
			public Text synopsisText(Help.ColorScheme colorScheme, Set<ArgSpec> outparam_groupArgs) {
				final Text synopsis = rawSynopsisUnitText(colorScheme, outparam_groupArgs);
				Text result = synopsisUnitText(colorScheme, synopsis);
				int i = 1;
				for (; i < multiplicity.min(); i++) {
					result = result.concat(" (").concat(synopsis).concat(")");
				}
				if (multiplicity().isVariable()) {
					result = result.concat("...");
				} else {
					for (; i < multiplicity.max(); i++) {
						result = result.concat(" [").concat(synopsis).concat("]");
					}
				}
				return result;
			}

			String synopsisUnit() {
				final Help.ColorScheme colorScheme = new Help.ColorScheme.Builder(Help.Ansi.OFF).build();
				return synopsisUnitText(colorScheme, rawSynopsisUnitText(colorScheme, new HashSet<ArgSpec>()))
						.toString();
			}

			private Text synopsisUnitText(Help.ColorScheme colorScheme, Text synopsis) {
				final String requiredOpen = args().size() > 1 || !subgroups().isEmpty() ? "(" : "";
				final String requiredClose = args().size() > 1 || !subgroups().isEmpty() ? ")" : "";
				final String prefix = multiplicity().min() > 0 ? requiredOpen : "[";
				final String postfix = multiplicity().min() > 0 ? requiredClose : "]";
				return colorScheme.text(prefix).concat(synopsis).concat(postfix);
			}

			@Override
			public String toString() {
				return "ArgGroup[exclusive=" + exclusive + ", multiplicity=" + multiplicity + ", validate=" + validate
						+ ", order=" + order + ", args=[" + ArgSpec.describe(args()) + "], headingKey="
						+ quote(headingKey) + ", heading=" + quote(heading) + ", subgroups=" + subgroups + "]";
			}

			void tryInitUserObject(CommandLine commandLine) throws Exception {
				final Tracer tracer = tracer();
				if (typeInfo() != null) {
					tracer.debug("Creating new user object of type %s for group %s", typeInfo().getAuxiliaryTypes()[0],
							synopsis());
					final Object userObject = DefaultFactory.create(commandLine.factory,
							typeInfo().getAuxiliaryTypes()[0]);
					tracer.debug("Created %s, invoking setter %s with scope %s", userObject, setter(), scope());
					setUserObject(userObject, commandLine.factory);
					for (final ArgSpec arg : args()) {
						tracer.debug(
								"Initializing %s in group %s: setting scope to user object %s and initializing initial and default values",
								ArgSpec.describe(arg, "="), synopsis(), userObject);
						arg.scope().set(userObject); // flip the actual user object for the arg (and all other args in
														// this group; they share the same IScope instance)
						commandLine.interpreter.parseResultBuilder.isInitializingDefaultValues = true;
						arg.applyInitialValue();
						commandLine.interpreter.applyDefault(commandLine.getCommandSpec().defaultValueProvider(), arg);
						commandLine.interpreter.parseResultBuilder.isInitializingDefaultValues = false;
					}
					for (final ArgGroupSpec subgroup : subgroups()) {
						tracer.debug("Setting scope for subgroup %s with setter=%s in group %s to user object %s",
								subgroup.synopsis(), subgroup.setter(), synopsis(), userObject);
						subgroup.scope().set(userObject); // flip the actual user object for the arg (and all other args
															// in this group; they share the same IScope instance)
					}
					for (final IAnnotatedElement specElement : specElements()) {
						tracer.debug("Setting @Spec with setter=%s in user object %s to %s", specElement.setter(),
								userObject, commandLine.getCommandSpec());
						specElement.scope().set(userObject);
						specElement.setter().set(commandLine.getCommandSpec());
					}
				} else {
					tracer.debug(
							"No type information available for group %s: cannot create new user object. Scope for arg setters is not changed.",
							synopsis());
				}
				tracer.debug("Initialization complete for group %s", synopsis());
			}

			/**
			 * Returns the type info for the annotated program element associated with this
			 * group.
			 * 
			 * @return type information that does not require {@code Class} objects and be
			 *         constructed both at runtime and compile time
			 */
			public ITypeInfo typeInfo() {
				return typeInfo;
			}

			Object userObject() {
				try {
					return getter.get();
				} catch (final Exception ex) {
					return ex.toString();
				}
			}

			Object userObjectOr(Object defaultValue) {
				try {
					return getter.get();
				} catch (final Exception ex) {
					return defaultValue;
				}
			}

			/**
			 * Returns whether picocli should validate the rules of this group: for a
			 * mutually exclusive group this means that no more than one arguments in the
			 * group is specified on the command line; for a co-ocurring group this means
			 * that all arguments in the group are specified on the command line.
			 * {@code true} by default.
			 * 
			 * @see ArgGroup#validate()
			 */
			public boolean validate() {
				return validate;
			}

			private ParseResult.GroupValidationResult validate(CommandLine commandLine, int presentCount,
					boolean haveMissing, boolean someButNotAllSpecified, String exclusiveElements,
					String requiredElements, String missingElements) {
				if (exclusive()) {
					if (presentCount > 1) {
						return new ParseResult.GroupValidationResult(
								ParseResult.GroupValidationResult.Type.FAILURE_PRESENT,
								new MutuallyExclusiveArgsException(commandLine,
										"Error: " + exclusiveElements + " are mutually exclusive (specify only one)"));
					}
					// check that exactly one member was matched
					if (multiplicity().min > 0 && haveMissing) {
						return new ParseResult.GroupValidationResult(
								ParseResult.GroupValidationResult.Type.FAILURE_ABSENT,
								new MissingParameterException(commandLine, args(),
										"Error: Missing required argument (specify one of these): "
												+ requiredElements));
					}
				} else { // co-occurring group
					if (someButNotAllSpecified) {
						return new ParseResult.GroupValidationResult(
								ParseResult.GroupValidationResult.Type.FAILURE_PARTIAL,
								new MissingParameterException(commandLine, args(),
										"Error: Missing required argument(s): " + missingElements));
					}
					if ((multiplicity().min > 0 && haveMissing)) {
						return new ParseResult.GroupValidationResult(
								ParseResult.GroupValidationResult.Type.FAILURE_ABSENT,
								new MissingParameterException(commandLine, args(),
										"Error: Missing required argument(s): " + missingElements));
					}
				}
				return presentCount > 0 ? ParseResult.GroupValidationResult.SUCCESS_PRESENT
						: ParseResult.GroupValidationResult.SUCCESS_ABSENT;
			}

			ParseResult.GroupValidationResult validateArgs(CommandLine commandLine, Collection<ArgSpec> matchedArgs) {
				final Set<ArgSpec> intersection = new LinkedHashSet<ArgSpec>(args());
				final Set<ArgSpec> missing = new LinkedHashSet<ArgSpec>(requiredArgs());
				final Set<ArgSpec> found = new LinkedHashSet<ArgSpec>(matchedArgs);
				missing.removeAll(matchedArgs);
				intersection.retainAll(found);
				final int presentCount = intersection.size();
				final boolean haveMissing = !missing.isEmpty() && !exclusive();
				final boolean someButNotAllSpecified = haveMissing && !intersection.isEmpty();
				final String exclusiveElements = ArgSpec.describe(intersection);
				final String requiredElements = ArgSpec.describe(requiredArgs());
				final String missingElements = ArgSpec.describe(missing);

				return validate(commandLine, presentCount, haveMissing, someButNotAllSpecified, exclusiveElements,
						requiredElements, missingElements);
			}
		}

		/**
		 * Models the shared attributes of {@link OptionSpec} and
		 * {@link PositionalParamSpec}.
		 * 
		 * @since 3.0
		 */
		public abstract static class ArgSpec {
			abstract static class Builder<T extends Builder<T>> {
				private static String inferLabel(String label, IAnnotatedElement annotatedElement) {
					if (!empty(label)) {
						return label.trim();
					}
					String name = annotatedElement.getName();
					if (annotatedElement.getTypeInfo().isMap()) { // #195 better param labels for map fields
						final List<ITypeInfo> aux = annotatedElement.getTypeInfo().getAuxiliaryTypeInfos();
						if (aux.size() < 2 || aux.get(0) == null || aux.get(1) == null) {
							name = "String=String";
						} else {
							name = aux.get(0).getClassSimpleName() + "=" + aux.get(1).getClassSimpleName();
						}
					} else {
						final int dollar = name.indexOf('$');
						if (dollar >= 0 && Method.class.equals(annotatedElement.userObject().getClass())) {
							// #1984 Kotlin internal members have mangled names with "$" followed by random
							// postfix
							name = name.substring(0, dollar);
						}
					}
					return "<" + name + ">";
				}

				private Object userObject;
				private Range arity;
				private String[] description;
				private String descriptionKey;
				private boolean required;
				private boolean originallyRequired;
				private boolean interactive;
				private boolean echo;
				private String prompt;
				private String paramLabel;
				private boolean hideParamSyntax;
				private String splitRegex;
				private String splitRegexSynopsisLabel;
				private boolean hidden;
				private ArgSpec root;
				private boolean inherited;
				private Class<?> type;
				private Class<?>[] auxiliaryTypes;
				private ITypeInfo typeInfo;
				private ITypeConverter<?>[] converters;
				private String defaultValue;
				private Object initialValue;
				private boolean hasInitialValue = true;
				private InitialValueState initialValueState = InitialValueState.UNAVAILABLE;
				private Help.Visibility showDefaultValue;
				private Iterable<String> completionCandidates;
				private IParameterConsumer parameterConsumer;
				private IParameterPreprocessor preprocessor;
				private String toString;
				private IGetter getter = new ObjectBinding();
				private ISetter setter = (ISetter) getter;
				private IScope scope = new ObjectScope(null);
				private ScopeType scopeType = ScopeType.LOCAL;
				private IAnnotatedElement annotatedElement;
				private String mapFallbackValue = UNSPECIFIED;
				private String originalDefaultValue = UNSPECIFIED;

				private String originalMapFallbackValue = UNSPECIFIED;

				Builder() {
				}

				Builder(ArgSpec original) {
					userObject = original.userObject;
					arity = original.arity;
					description = original.description;
					descriptionKey = original.descriptionKey;
					required = original.required;
					originallyRequired = original.originallyRequired;
					interactive = original.interactive;
					echo = original.echo;
					prompt = original.prompt;
					paramLabel = original.paramLabel;
					hideParamSyntax = original.hideParamSyntax;
					splitRegex = original.splitRegex;
					splitRegexSynopsisLabel = original.splitRegexSynopsisLabel;
					hidden = original.hidden;
					inherited = original.inherited;
					root = original.root;
					setTypeInfo(original.typeInfo);
					converters = original.converters;
					defaultValue = original.defaultValue;
					annotatedElement = original.annotatedElement;
					initialValue = original.initialValue;
					initialValueState = original.initialValueState;
					hasInitialValue = original.hasInitialValue;
					showDefaultValue = original.showDefaultValue;
					completionCandidates = original.completionCandidates;
					parameterConsumer = original.parameterConsumer;
					preprocessor = original.preprocessor;
					toString = original.toString;
					getter = original.getter;
					setter = original.setter;
					scope = original.scope;
					scopeType = original.scopeType;
					mapFallbackValue = original.mapFallbackValue;
					originalDefaultValue = original.originalDefaultValue;
					originalMapFallbackValue = original.originalMapFallbackValue;
				}

				Builder(IAnnotatedElement annotatedElement) {
					this.annotatedElement = annotatedElement;
					userObject = annotatedElement.userObject();
					setTypeInfo(annotatedElement.getTypeInfo());
					toString = annotatedElement.getToString();
					getter = annotatedElement.getter();
					setter = annotatedElement.setter();
					scope = annotatedElement.scope();
					hasInitialValue = annotatedElement.hasInitialValue();
					if (annotatedElement instanceof IExtensible) {
						initialValueState = ((IExtensible) annotatedElement).getExtension(InitialValueState.class);
					}
				}

				Builder(Option option, IAnnotatedElement annotatedElement, IFactory factory) {
					this(annotatedElement);
					arity = Range.optionArity(annotatedElement);
					required = option.required();

					paramLabel = inferLabel(option.paramLabel(), annotatedElement);

					hideParamSyntax = option.hideParamSyntax();
					interactive = option.interactive();
					echo = option.echo();
					prompt = option.prompt();
					description = option.description();
					descriptionKey = option.descriptionKey();
					splitRegex = option.split();
					splitRegexSynopsisLabel = option.splitSynopsisLabel();
					hidden = option.hidden();
					defaultValue = NULL_VALUE.equals(option.defaultValue()) ? null : option.defaultValue();
					mapFallbackValue = NULL_VALUE.equals(option.mapFallbackValue()) ? null : option.mapFallbackValue();
					originalDefaultValue = option.defaultValue();
					originalMapFallbackValue = option.mapFallbackValue();
					showDefaultValue = option.showDefaultValue();
					scopeType = option.scope();
					inherited = false;
					if (factory != null) {
						converters = DefaultFactory.createConverter(factory, option.converter());
						if (!NoCompletionCandidates.class.equals(option.completionCandidates())) {
							completionCandidates = DefaultFactory.createCompletionCandidates(factory,
									option.completionCandidates());
						}
						if (!NullParameterConsumer.class.equals(option.parameterConsumer())) {
							parameterConsumer = DefaultFactory.createParameterConsumer(factory,
									option.parameterConsumer());
						}
						if (!NoOpParameterPreprocessor.class.equals(option.preprocessor())) {
							preprocessor = DefaultFactory.create(factory, option.preprocessor());
						}
					}
				}

				Builder(Parameters parameters, IAnnotatedElement annotatedElement, IFactory factory) {
					this(annotatedElement);
					arity = Range.parameterArity(annotatedElement);
					required = arity.min > 0; // but arity may still be unresolved...

					// method parameters may be positional parameters without @Parameters annotation
					if (parameters == null) {
						paramLabel = inferLabel(null, annotatedElement);
					} else {
						paramLabel = inferLabel(parameters.paramLabel(), annotatedElement);

						hideParamSyntax = parameters.hideParamSyntax();
						interactive = parameters.interactive();
						echo = parameters.echo();
						prompt = parameters.prompt();
						description = parameters.description();
						descriptionKey = parameters.descriptionKey();
						splitRegex = parameters.split();
						splitRegexSynopsisLabel = parameters.splitSynopsisLabel();
						hidden = parameters.hidden();
						defaultValue = NULL_VALUE.equals(parameters.defaultValue()) ? null : parameters.defaultValue();
						mapFallbackValue = NULL_VALUE.equals(parameters.mapFallbackValue()) ? null
								: parameters.mapFallbackValue();
						originalDefaultValue = parameters.defaultValue();
						originalMapFallbackValue = parameters.mapFallbackValue();
						showDefaultValue = parameters.showDefaultValue();
						scopeType = parameters.scope();
						inherited = false;
						if (factory != null) { // annotation processors will pass a null factory
							converters = DefaultFactory.createConverter(factory, parameters.converter());
							if (!NoCompletionCandidates.class.equals(parameters.completionCandidates())) {
								completionCandidates = DefaultFactory.createCompletionCandidates(factory,
										parameters.completionCandidates());
							}
							if (!NullParameterConsumer.class.equals(parameters.parameterConsumer())) {
								parameterConsumer = DefaultFactory.createParameterConsumer(factory,
										parameters.parameterConsumer());
							}
							if (!NoOpParameterPreprocessor.class.equals(parameters.preprocessor())) {
								preprocessor = DefaultFactory.create(factory, parameters.preprocessor());
							}
						}
					}
				}

				/**
				 * Returns how many arguments this option or positional parameter requires.
				 * 
				 * @see Option#arity()
				 */
				public Range arity() {
					return arity;
				}

				/**
				 * Sets how many arguments this option or positional parameter requires, and
				 * returns this builder.
				 */
				public T arity(Range arity) {
					this.arity = Assert.notNull(arity, "arity");
					return self();
				}

				/**
				 * Sets how many arguments this option or positional parameter requires, and
				 * returns this builder.
				 */
				public T arity(String range) {
					return arity(Range.valueOf(range));
				}

				/**
				 * Returns auxiliary type information used when the {@link #type()} is a generic
				 * container like {@code Collection}, {@code Map}, {@code Optional} or an
				 * abstract class.
				 * 
				 * @see Option#type()
				 */
				public Class<?>[] auxiliaryTypes() {
					return auxiliaryTypes;
				}

				/**
				 * Sets auxiliary type information, and returns this builder.
				 * 
				 * @param types the element type(s) when the {@link #type()} is a generic type
				 *              like {@code Collection}, {@code Map} or {@code Optional}; or the
				 *              concrete type when the {@link #type()} is an abstract class.
				 */
				public T auxiliaryTypes(Class<?>... types) {
					this.auxiliaryTypes = Assert.notNull(types, "types").clone();
					return self();
				}

				public abstract ArgSpec build();

				/**
				 * Returns the completion candidates for this option or positional parameter, or
				 * {@code null}.
				 * 
				 * @since 3.2
				 */
				public Iterable<String> completionCandidates() {
					return completionCandidates;
				}

				/**
				 * Sets the completion candidates for this option or positional parameter, and
				 * returns this builder.
				 * 
				 * @since 3.2
				 */
				public T completionCandidates(Iterable<String> completionCandidates) {
					this.completionCandidates = completionCandidates;
					return self();
				}

				/**
				 * Returns one or more {@link CommandLine.ITypeConverter type converters} to use
				 * to convert the command line argument into a strongly typed value (or
				 * key-value pair for map fields). This is useful when a particular option or
				 * positional parameter should use a custom conversion that is different from
				 * the normal conversion for the arg spec's type.
				 * 
				 * @see Option#converter()
				 */
				public ITypeConverter<?>[] converters() {
					return converters;
				}

				/**
				 * Sets option/positional param-specific converter (or converters for Maps), and
				 * returns this builder.
				 */
				public T converters(ITypeConverter<?>... cs) {
					this.converters = Assert.notNull(cs, "type converters").clone();
					return self();
				}

				/**
				 * Returns the default value of this option or positional parameter, before
				 * splitting and type conversion. A value of {@code null} means this option or
				 * positional parameter does not have a default.
				 */
				public String defaultValue() {
					return defaultValue;
				}

				/**
				 * Sets the default value of this option or positional parameter to the
				 * specified value, and returns this builder. Before parsing the command line,
				 * the result of {@linkplain #splitRegex() splitting} and
				 * {@linkplain #converters() type converting} this default value is applied to
				 * the option or positional parameter. A value of {@code null} or
				 * {@code "__no_default_value__"} means no default.
				 */
				public T defaultValue(String defaultValue) {
					this.defaultValue = defaultValue;
					return self();
				}

				/**
				 * Returns the description of this option, used when generating the usage
				 * documentation.
				 * 
				 * @see Option#description()
				 */
				public String[] description() {
					return description;
				}

				/**
				 * Sets the description of this option, used when generating the usage
				 * documentation, and returns this builder.
				 * 
				 * @see Option#description()
				 */
				public T description(String... description) {
					this.description = Assert.notNull(description, "description").clone();
					return self();
				}

				/**
				 * Returns the description key of this arg spec, used to get the description
				 * from a resource bundle.
				 * 
				 * @see Option#descriptionKey()
				 * @see Parameters#descriptionKey()
				 * @since 3.6
				 */
				public String descriptionKey() {
					return descriptionKey;
				}

				/**
				 * Sets the description key that is used to look up the description in a
				 * resource bundle, and returns this builder.
				 * 
				 * @see Option#descriptionKey()
				 * @see Parameters#descriptionKey()
				 * @since 3.6
				 */
				public T descriptionKey(String descriptionKey) {
					this.descriptionKey = descriptionKey;
					return self();
				}

				/**
				 * Returns whether the user input is echoed to the console or not for an
				 * interactive option or positional parameter when asking for user input.
				 * 
				 * @see Option#echo()
				 * @see Parameters#echo()
				 * @since 4.6
				 */
				public boolean echo() {
					return echo;
				}

				/**
				 * Sets whether the user input is echoed to the console or not for an
				 * interactive option or positional parameter.
				 */
				public T echo(boolean echo) {
					this.echo = echo;
					return self();
				}

				/**
				 * Returns the {@link IGetter} that is responsible for supplying the value of
				 * this argument.
				 */
				public IGetter getter() {
					return getter;
				}

				/**
				 * Sets the {@link IGetter} that is responsible for getting the value of this
				 * argument, and returns this builder.
				 */
				public T getter(IGetter getter) {
					this.getter = getter;
					return self();
				}

				/**
				 * Determines whether the option or positional parameter will be reset to the
				 * {@link #initialValue()} before parsing new input.
				 */
				public boolean hasInitialValue() {
					return hasInitialValue;
				}

				/**
				 * Determines whether the option or positional parameter will be reset to the
				 * {@link #initialValue()} before parsing new input.
				 */
				public T hasInitialValue(boolean hasInitialValue) {
					this.hasInitialValue = hasInitialValue;
					return self();
				}

				/**
				 * Returns whether this option should be excluded from the usage message.
				 * 
				 * @see Option#hidden()
				 */
				public boolean hidden() {
					return hidden;
				}

				/**
				 * Sets whether this option should be excluded from the usage message, and
				 * returns this builder.
				 */
				public T hidden(boolean hidden) {
					this.hidden = hidden;
					return self();
				}

				/**
				 * Returns whether usage syntax decorations around the {@linkplain #paramLabel()
				 * paramLabel} should be suppressed. The default is {@code false}: by default,
				 * the paramLabel is surrounded with {@code '['} and {@code ']'} characters if
				 * the value is optional and followed by ellipses ("...") when multiple values
				 * can be specified.
				 * 
				 * @since 3.6.0
				 */
				public boolean hideParamSyntax() {
					return hideParamSyntax;
				}

				/**
				 * Sets whether usage syntax decorations around the {@linkplain #paramLabel()
				 * paramLabel} should be suppressed. The default is {@code false}: by default,
				 * the paramLabel is surrounded with {@code '['} and {@code ']'} characters if
				 * the value is optional and followed by ellipses ("...") when multiple values
				 * can be specified.
				 * 
				 * @since 3.6.0
				 */
				public T hideParamSyntax(boolean hideParamSyntax) {
					this.hideParamSyntax = hideParamSyntax;
					return self();
				}

				/**
				 * Returns whether this option is inherited from a parent command.
				 * 
				 * @see Option#scope()
				 * @since 4.3.0
				 */
				public boolean inherited() {
					return inherited;
				}

				/**
				 * Sets whether this option is inherited from a parent command, and returns this
				 * builder.
				 * 
				 * @since 4.3.0
				 */
				public T inherited(boolean inherited) {
					this.inherited = inherited;
					return self();
				}

				/**
				 * Returns the initial value this option or positional parameter. If
				 * {@link #hasInitialValue()} is true, the option will be reset to the initial
				 * value before parsing (regardless of whether a default value exists), to clear
				 * values that would otherwise remain from parsing previous input.
				 */
				public Object initialValue() {
					return initialValue;
				}

				/**
				 * Sets the initial value of this option or positional parameter to the
				 * specified value, and returns this builder. If {@link #hasInitialValue()} is
				 * true, the option will be reset to the initial value before parsing
				 * (regardless of whether a default value exists), to clear values that would
				 * otherwise remain from parsing previous input.
				 */
				// TODO test
				// ModelCommandSpecTest.testDontClearScalarOptionOldValueBeforeParseIfInitialValueFalse
				// and Groovy CliBuilder before setting state to CACHED
				// testDontClearListOptionOldValueBeforeParseIfInitialValueFalse and
				// testDontClearArrayOptionOldValueBeforeParse
				public T initialValue(Object initialValue) {
					this.initialValue = initialValue;
					/* initialValueState = InitialValueState.CACHED; */ return self();
				}

				/**
				 * Returns whether this option prompts the user to enter a value on the command
				 * line.
				 * 
				 * @see Option#interactive()
				 */
				public boolean interactive() {
					return interactive;
				}

				/**
				 * Sets whether this option prompts the user to enter a value on the command
				 * line, and returns this builder.
				 */
				public T interactive(boolean interactive) {
					this.interactive = interactive;
					return self();
				}

				/**
				 * Returns the fallback value for this Map option or positional parameter: the
				 * value that is put into the Map when only the key is specified for the option
				 * or positional parameter, like {@code -Dkey} instead of {@code -Dkey=value}.
				 * <p>
				 * If no {@code mapFallbackValue} is set, key-only Map parameters like
				 * {@code -Dkey} are considered invalid user input and cause a
				 * {@link ParameterException} to be thrown.
				 * </p>
				 * <p>
				 * By default, this method returns a special "__unspecified__" value indicating
				 * that no {@code mapFallbackValue} was set.
				 * </p>
				 * 
				 * @see Option#mapFallbackValue()
				 * @see Parameters#mapFallbackValue()
				 * @since 4.6
				 */
				public String mapFallbackValue() {
					return mapFallbackValue;
				}

				/**
				 * Sets the fallback value for this Map option or positional parameter: the
				 * value that is put into the Map when only the key is specified for the option
				 * or positional parameter, like {@code -Dkey} instead of {@code -Dkey=value}.
				 * <p>
				 * If no {@code mapFallbackValue} is set, key-only Map parameters like
				 * {@code -Dkey} are considered invalid user input and cause a
				 * {@link ParameterException} to be thrown.
				 * </p>
				 * 
				 * @see Option#mapFallbackValue()
				 * @see Parameters#mapFallbackValue()
				 * @since 4.6
				 */
				@SuppressWarnings("rawtypes")
				public Builder mapFallbackValue(String fallbackValue) {
					this.mapFallbackValue = fallbackValue;
					return self();
				}

				/**
				 * Returns the custom parameter handler for this option or positional parameter,
				 * or {@code null}.
				 * 
				 * @since 4.0
				 */
				public IParameterConsumer parameterConsumer() {
					return parameterConsumer;
				}

				/**
				 * Sets the parameterConsumer for this option or positional parameter, and
				 * returns this builder.
				 * 
				 * @since 4.0
				 */
				public T parameterConsumer(IParameterConsumer parameterConsumer) {
					this.parameterConsumer = parameterConsumer;
					return self();
				}

				/**
				 * Returns the name of the option or positional parameter used in the usage help
				 * message.
				 * 
				 * @see Option#paramLabel()
				 * @see Parameters#paramLabel()
				 */
				public String paramLabel() {
					return paramLabel;
				}

				/**
				 * Sets the name of the option or positional parameter used in the usage help
				 * message, and returns this builder.
				 */
				public T paramLabel(String paramLabel) {
					this.paramLabel = Assert.notNull(paramLabel, "paramLabel");
					return self();
				}

				/**
				 * Returns the custom {@code IParameterPreprocessor} to either replace or
				 * complement picocli's parsing logic for the parameter(s) of this option or
				 * position, or {@code null}.
				 * 
				 * @since 4.6
				 */
				public IParameterPreprocessor preprocessor() {
					return preprocessor;
				}

				/**
				 * Sets the custom {@code IParameterPreprocessor} for this option or position,
				 * and returns this builder.
				 * 
				 * @since 4.6
				 */
				public T preprocessor(IParameterPreprocessor preprocessor) {
					this.preprocessor = preprocessor;
					return self();
				}

				/**
				 * Returns the text displayed to the end user for an interactive option or
				 * positional parameter when asking for user input.
				 * 
				 * @see Option#prompt()
				 * @see Parameters#prompt()
				 * @since 4.6
				 */
				public String prompt() {
					return prompt;
				}

				/**
				 * Sets the text displayed to the end user for an interactive option or
				 * positional parameter when asking for user input.
				 */
				public T prompt(String prompt) {
					this.prompt = prompt;
					return self();
				}

				/**
				 * Returns whether this is a required option or positional parameter.
				 * 
				 * @see Option#required()
				 */
				public boolean required() {
					return required;
				}

				/**
				 * Sets whether this is a required option or positional parameter, and returns
				 * this builder.
				 */
				public T required(boolean required) {
					this.required = required;
					return self();
				}

				/**
				 * Returns the root option or positional parameter (on the parent command), if
				 * this option or positional parameter was inherited; or {@code null} if it was
				 * not.
				 * 
				 * @see Option#scope()
				 * @since 4.6.0
				 */
				public ArgSpec root() {
					return root;
				}

				/**
				 * Sets the root object for this inherited option, and returns this builder.
				 * 
				 * @since 4.6.0
				 */
				public T root(ArgSpec root) {
					this.root = root;
					return self();
				}

				/**
				 * Returns the binding {@link IScope} that determines the instance of the
				 * enclosing element where the setter sets the value (or the getter gets the
				 * value) of this argument.
				 */
				public IScope scope() {
					return scope;
				}

				/**
				 * Sets the binding {@link IScope} that targets where the setter sets the value,
				 * and returns this builder.
				 */
				public T scope(IScope scope) {
					this.scope = scope;
					return self();
				}

				/**
				 * Returns the scope of this argument.
				 * 
				 * @return whether this argument applies to all descendent subcommands of the
				 *         command where it is defined
				 * @since 4.3
				 */
				public ScopeType scopeType() {
					return scopeType;
				}

				/**
				 * Sets the scope of where this argument applies: only this command, or also all
				 * sub (and sub-sub) commands, and returns this builder.
				 * 
				 * @since 4.3
				 */
				public T scopeType(ScopeType scopeType) {
					this.scopeType = scopeType;
					return self();
				}

				protected abstract T self(); // subclasses must override to return "this"

				/**
				 * Returns the {@link ISetter} that is responsible for modifying the value of
				 * this argument.
				 */
				public ISetter setter() {
					return setter;
				}

				/**
				 * Sets the {@link ISetter} that is responsible for modifying the value of this
				 * argument, and returns this builder.
				 */
				public T setter(ISetter setter) {
					this.setter = setter;
					return self();
				}

				private void setTypeInfo(ITypeInfo newValue) {
					this.typeInfo = newValue;
					if (typeInfo != null) {
						type = typeInfo.getType();
						auxiliaryTypes = typeInfo.getAuxiliaryTypes();
					}
				}

				/**
				 * Returns whether this option or positional parameter's default value should be
				 * shown in the usage help.
				 */
				public Help.Visibility showDefaultValue() {
					return showDefaultValue;
				}

				/**
				 * Sets whether this option or positional parameter's default value should be
				 * shown in the usage help, and returns this builder.
				 */
				public T showDefaultValue(Help.Visibility visibility) {
					showDefaultValue = Assert.notNull(visibility, "visibility");
					return self();
				}

				/**
				 * Returns a regular expression to split option parameter values or {@code ""}
				 * if the value should not be split.
				 * 
				 * @see Option#split()
				 */
				public String splitRegex() {
					return splitRegex;
				}

				/**
				 * Sets a regular expression to split option parameter values or {@code ""} if
				 * the value should not be split, and returns this builder.
				 */
				public T splitRegex(String splitRegex) {
					this.splitRegex = Assert.notNull(splitRegex, "splitRegex");
					return self();
				}

				/**
				 * Returns a regular expression to split option parameter for usage information.
				 * 
				 * @see Option#splitSynopsisLabel()
				 * @since 4.3
				 */
				public String splitRegexSynopsisLabel() {
					return splitRegexSynopsisLabel;
				}

				/**
				 * Sets a regular expression to split option parameter for usage information.
				 */
				public T splitRegexSynopsisLabel(String splitRegexSynopsisLabel) {
					this.splitRegexSynopsisLabel = Assert.notNull(splitRegexSynopsisLabel, "splitRegexSynopsisLabel");
					return self();
				}

				@Override
				public String toString() {
					return toString;
				}

				/**
				 * Returns the type to convert the option or positional parameter to before
				 * {@linkplain #setValue(Object) setting} the value. This may be a container
				 * type like {@code List}, {@code Map}, or {@code Optional}, in which case the
				 * type or types of the elements are returned by {@link #auxiliaryTypes()}.
				 */
				public Class<?> type() {
					return type;
				}

				/**
				 * Sets the type to convert the option or positional parameter to before
				 * {@linkplain #setValue(Object) setting} the value, and returns this builder.
				 * 
				 * @param propertyType the type of this option or parameter. For multi-value
				 *                     options and positional parameters this can be an array,
				 *                     or a (sub-type of) Collection or Map.
				 */
				public T type(Class<?> propertyType) {
					this.type = Assert.notNull(propertyType, "type");
					return self();
				}

				/**
				 * Returns the type info for this option or positional parameter.
				 * 
				 * @return type information that does not require {@code Class} objects and be
				 *         constructed both at runtime and compile time
				 * @since 4.0
				 */
				public ITypeInfo typeInfo() {
					return typeInfo;
				}

				/**
				 * Sets the type info for this option or positional parameter, and returns this
				 * builder.
				 * 
				 * @param typeInfo type information that does not require {@code Class} objects
				 *                 and be constructed both at runtime and compile time
				 * @since 4.0
				 */
				public T typeInfo(ITypeInfo typeInfo) {
					setTypeInfo(Assert.notNull(typeInfo, "typeInfo"));
					return self();
				}

				/**
				 * Returns the user object associated with this option or positional parameters.
				 * 
				 * @return may return the annotated program element, or some other useful object
				 * @since 4.0
				 */
				public Object userObject() {
					return userObject;
				}

				/**
				 * Sets the user object associated with this option or positional parameters,
				 * and returns this builder.
				 * 
				 * @param userObject may be the annotated program element, or some other useful
				 *                   object
				 * @since 4.0
				 */
				public T userObject(Object userObject) {
					this.userObject = Assert.notNull(userObject, "userObject");
					return self();
				}

				/**
				 * Sets the string representation of this option or positional parameter to the
				 * specified value, and returns this builder.
				 */
				public T withToString(String toString) {
					this.toString = toString;
					return self();
				}
			}

			/**
			 * Special value that can be used to designate {@code null}.
			 * 
			 * @see Option#defaultValue()
			 * @see Option#fallbackValue()
			 * @see Option#mapFallbackValue()
			 * @see Parameters#defaultValue()
			 * @see Parameters#mapFallbackValue()
			 * @since 4.6
			 */
			static final String NULL_VALUE = "_NULL_";
			static final String DESCRIPTION_VARIABLE_DEFAULT_VALUE = "${DEFAULT-VALUE}";
			static final String DESCRIPTION_VARIABLE_FALLBACK_VALUE = "${FALLBACK-VALUE}";
			static final String DESCRIPTION_VARIABLE_MAP_FALLBACK_VALUE = "${MAP-FALLBACK-VALUE}";
			static final String DESCRIPTION_VARIABLE_COMPLETION_CANDIDATES = "${COMPLETION-CANDIDATES}";
			private static final String NO_DEFAULT_VALUE = "__no_default_value__";

			private static final String UNSPECIFIED = "__unspecified__";

			/**
			 * Returns a description of the option or positional arg, e.g. {@code -a=<a>}
			 * 
			 * @param separator separator between arg and arg parameter label, usually '='
			 */
			private static String describe(ArgSpec argSpec, String separator) {
				return describe(argSpec, separator, argSpec.paramLabel());
			}

			/**
			 * Returns a description of the option or positional arg
			 * 
			 * @param separator separator between arg and arg parameter value, usually '='
			 * @param value     the value to append after the separator
			 */
			private static String describe(ArgSpec argSpec, String separator, String value) {
				final String prefix = (argSpec.isOption()) ? ((OptionSpec) argSpec).longestName()
						: "params[" + ((PositionalParamSpec) argSpec).index() + "]";
				return argSpec.arity().min > 0 ? prefix + separator + value : prefix;
			}

			private static String describe(Collection<ArgSpec> args) {
				return describe(args, ", ", "=", "", "");
			}

			private static String describe(Collection<ArgSpec> args, String separator, String optionParamSeparator,
					String openingQuote, String closingQuote) {
				final StringBuilder sb = new StringBuilder();
				for (final ArgSpec arg : args) {
					if (sb.length() > 0) {
						sb.append(separator);
					}
					if (arg.isPositional()) {
						sb.append(openingQuote).append(arg.paramLabel()).append(closingQuote);
					} else {
						sb.append(openingQuote).append(((OptionSpec) arg).longestName());
						if (arg.arity().min() > 0) {
							sb.append(optionParamSeparator).append(arg.paramLabel());
						}
						sb.append(closingQuote);
					}
				}
				return sb.toString();
			}

			private static String describeTypes(Collection<ArgSpec> args) {
				if (args.isEmpty()) {
					return "";
				}
				int optionCount = 0;
				int paramCount = 0;
				for (final ArgSpec arg : args) {
					if (arg.isOption()) {
						optionCount++;
					} else {
						paramCount++;
					}
				}
				if (optionCount == 0) {
					return paramCount == 1 ? "parameter" : "parameters";
				}
				if (paramCount == 0) {
					return optionCount == 1 ? "option" : "options";
				}
				return "options and parameters";
			}

			private static String restoreQuotedValues(String part, Queue<String> quotedValues, ParserSpec parser) {
				final StringBuilder result = new StringBuilder();
				boolean escaping = false, inQuote = false, skip = false;
				for (int ch, i = 0; i < part.length(); i += Character.charCount(ch)) {
					ch = part.codePointAt(i);
					switch (ch) {
					case '\\':
						escaping = !escaping;
						break;
					case '\"':
						if (!escaping) {
							inQuote = !inQuote;
							if (!inQuote) {
								result.append(quotedValues.remove());
							}
//                                skip = parser.trimQuotes();
						}
						break;
					default:
						escaping = false;
						break;
					}
					if (!skip) {
						result.appendCodePoint(ch);
					}
					skip = false;
				}
				return parser.trimQuotes() ? smartUnquote(result.toString()) : result.toString();
			}

			// @since 3.7
			private static String[] splitRespectingQuotedStrings(String value, int limit, ParserSpec parser,
					ArgSpec argSpec, String splitRegex) {
				final Queue<String> quotedValues = new LinkedList<String>();
				final StringBuilder splittable = new StringBuilder();
				final StringBuilder temp = new StringBuilder();
				StringBuilder current = splittable;
				boolean escaping = false, inQuote = false;
				for (int ch, i = 0; i < value.length(); i += Character.charCount(ch)) {
					ch = value.codePointAt(i);
					switch (ch) {
					case '\\':
						escaping = !escaping;
						break;
					case '\"':
						if (!escaping) {
							inQuote = !inQuote;
							current = inQuote ? temp : splittable;
							if (inQuote) {
								splittable.appendCodePoint(ch);
								continue;
							} else {
								quotedValues.add(temp.toString());
								temp.setLength(0);
							}
						}
						escaping = false;
						break;
					default:
						escaping = false;
						break;
					}
					current.appendCodePoint(ch);
				}
				if (temp.length() > 0) {
					CommandLine.tracer().warn("Unbalanced quotes in [%s] for %s (value=%s)", temp, argSpec, value);
					quotedValues.add(temp.toString());
					temp.setLength(0);
				}
				// if value was 'a,b,"c,d",e' -> splittable now contains 'a,b,"",e', and
				// quotedValues contains ['c,d']
				final String[] result = splittable.toString().split(splitRegex, limit);
				for (int i = 0; i < result.length; i++) {
					result[i] = restoreQuotedValues(result[i], quotedValues, parser);
				}
				if (!quotedValues.isEmpty()) {
					CommandLine.tracer().warn(
							"Unable to respect quotes while splitting value %s for %s (unprocessed remainder: %s)",
							value, argSpec, quotedValues);
					return value.split(splitRegex, limit);
				}
				return result;
			}

			private final boolean inherited;
			private final ArgSpec root;
			// help-related fields
			private final boolean hidden;
			private final String paramLabel;

			private final boolean hideParamSyntax;
			private final String[] description;
			private final String descriptionKey;
			private final Help.Visibility showDefaultValue;
			private Messages messages;
			CommandSpec commandSpec;
			private ArgGroupSpec group;
			private final Object userObject;
			// parser fields
			private boolean required;
			private boolean originallyRequired;
			private final boolean interactive;
			private final boolean echo;
			private final String prompt;
			private final String splitRegex;
			private final String splitRegexSynopsisLabel;
			protected final ITypeInfo typeInfo;
			private final ITypeConverter<?>[] converters;
			private final Iterable<String> completionCandidates;
			private final IParameterConsumer parameterConsumer;
			private final IParameterPreprocessor preprocessor;
			private final String mapFallbackValue;
			private final String defaultValue;
			private final String originalDefaultValue;
			private final String originalMapFallbackValue;
			private Object initialValue;
			private final boolean hasInitialValue;
			private InitialValueState initialValueState;
			protected boolean valueIsDefaultValue;
			protected final IAnnotatedElement annotatedElement;
			private final IGetter getter;
			private final ISetter setter;

			private final IScope scope;

			private final ScopeType scopeType;

			private Range arity;

			private List<String> stringValues = new ArrayList<String>();
			private List<String> originalStringValues = new ArrayList<String>();
			protected String toString;
			private final List<Object> typedValues = new ArrayList<Object>();

			Map<Integer, Object> typedValueAtPosition = new TreeMap<Integer, Object>();

			/** Constructs a new {@code ArgSpec}. */
			private <T extends Builder<T>> ArgSpec(Builder<T> builder) {
				userObject = builder.userObject;
				description = builder.description == null ? new String[0] : builder.description;
				descriptionKey = builder.descriptionKey;
				splitRegex = builder.splitRegex == null ? "" : builder.splitRegex;
				splitRegexSynopsisLabel = builder.splitRegexSynopsisLabel == null ? ""
						: builder.splitRegexSynopsisLabel;
				paramLabel = empty(builder.paramLabel) ? "PARAM" : builder.paramLabel;
				hideParamSyntax = builder.hideParamSyntax;
				converters = builder.converters == null ? new ITypeConverter<?>[0] : builder.converters;
				parameterConsumer = builder.parameterConsumer;
				preprocessor = builder.preprocessor != null ? builder.preprocessor : new NoOpParameterPreprocessor();
				showDefaultValue = builder.showDefaultValue == null ? Help.Visibility.ON_DEMAND
						: builder.showDefaultValue;
				hidden = builder.hidden;
				inherited = builder.inherited;
				root = builder.root == null && ScopeType.INHERIT.equals(builder.scopeType) ? this : builder.root;
				interactive = builder.interactive;
				echo = builder.echo;
				prompt = builder.prompt;
				initialValue = builder.initialValue;
				hasInitialValue = builder.hasInitialValue;
				initialValueState = builder.initialValueState;
				annotatedElement = builder.annotatedElement;
				defaultValue = NO_DEFAULT_VALUE.equals(builder.defaultValue) ? null : builder.defaultValue;
				required = builder.required;
				originallyRequired = builder.originallyRequired;
				toString = builder.toString;
				getter = builder.getter;
				setter = builder.setter;
				scope = builder.scope;
				scopeType = builder.scopeType;
				mapFallbackValue = builder.mapFallbackValue;
				originalDefaultValue = builder.originalDefaultValue;
				originalMapFallbackValue = builder.originalMapFallbackValue;

				Range tempArity = builder.arity;
				if (tempArity == null) {
					if (interactive) {
						tempArity = Range.valueOf("0");
					} else if (isOption()) {
						tempArity = (builder.type == null || isBoolean(builder.type)) ? Range.valueOf("0")
								: Range.valueOf("1");
					} else {
						tempArity = Range.valueOf("1");
					}
					tempArity = tempArity.unspecified(true);
				}
				arity = tempArity;

				if (builder.typeInfo == null) {
					this.typeInfo = RuntimeTypeInfo.create(builder.type, builder.auxiliaryTypes,
							Collections.<String>emptyList(), arity, (isOption() ? boolean.class : String.class),
							interactive);
				} else {
					this.typeInfo = builder.typeInfo;
				}

				if (builder.completionCandidates == null && typeInfo.isEnum()) {
					final List<String> list = new ArrayList<String>();
					for (final Object c : typeInfo.getEnumConstantNames()) {
						list.add(c.toString());
					}
					completionCandidates = Collections.unmodifiableList(list);
				} else {
					completionCandidates = builder.completionCandidates;
				}
				if (interactive && !arity.isValidForInteractiveArgs()) {
					throw new InitializationException(
							"Interactive options and positional parameters are only supported for arity=0 and arity=0..1; not for arity="
									+ arity);
				}
				// https://github.com/remkop/picocli/issues/745
				if (!empty(splitRegex) && !typeInfo.isMultiValue()
						&& System.getProperty("picocli.ignore.invalid.split") == null) {
					throw new InitializationException(
							"Only multi-value options and positional parameters should have a split regex (this check can be disabled by setting system property 'picocli.ignore.invalid.split')");
				}
			}

			void applyInitialValue() {
				final Tracer tracer = CommandLine.tracer();
				if (hasInitialValue()) {
					try {
						setter().set(initialValue());
						tracer.debug("Set initial value for %s of type %s to %s.", this, type(),
								String.valueOf(initialValue()));
					} catch (final Exception ex) {
						tracer.warn("Could not set initial value for %s of type %s to %s: %s", this, type(),
								String.valueOf(initialValue()), ex);
					}
				} else {
					tracer.debug("Initial value not available for %s", this);
				}
			}

			/**
			 * Returns how many arguments this option or positional parameter requires.
			 * 
			 * @see Option#arity()
			 */
			public Range arity() {
				return arity;
			}

			/**
			 * Returns auxiliary type information used when the {@link #type()} is a generic
			 * type like {@code Collection}, {@code Map} or {@code Optional}; returns the
			 * concrete type when {@link #type()} is an abstract class, otherwise, returns
			 * the same as {@link #type()}.
			 * 
			 * @see Option#type()
			 */
			public Class<?>[] auxiliaryTypes() {
				return typeInfo.getAuxiliaryTypes();
			}

			private Object calcDefaultValue(boolean interpolate) {
				String result = defaultValueFromProvider();
				if (result == null) {
					result = interpolate ? this.defaultValue() : this.defaultValue;
				}
				return result == null ? initialValue() : result;
			}

			/**
			 * Returns the command this option or positional parameter belongs to.
			 * <p>
			 * Beware that it is possible to programmatically add an option or positional
			 * parameter to more than one command model. (This will not happen in models
			 * that are auto-generated from annotations). In that case this method will only
			 * return the one it was added to last.
			 * <p>
			 * If the option or positional parameter has not yet been attached to a command,
			 * {@code null} will be returned.
			 * 
			 * @since 4.1
			 */
			public CommandSpec command() {
				return commandSpec;
			}

			/**
			 * Returns the explicitly set completion candidates for this option or
			 * positional parameter, valid enum constant names, or {@code null} if this
			 * option or positional parameter does not have any completion candidates and
			 * its type is not an enum.
			 * 
			 * @return the completion candidates for this option or positional parameter,
			 *         valid enum constant names, or {@code null}
			 * @since 3.2
			 */
			public Iterable<String> completionCandidates() {
				return completionCandidates;
			}

			/**
			 * Returns one or more {@link CommandLine.ITypeConverter type converters} to use
			 * to convert the command line argument into a strongly typed value (or
			 * key-value pair for map fields). This is useful when a particular option or
			 * positional parameter should use a custom conversion that is different from
			 * the normal conversion for the arg spec's type.
			 * 
			 * @see Option#converter()
			 */
			public ITypeConverter<?>[] converters() {
				return converters.clone();
			}

			private String[] debug(String[] result, String msg, String value) {
				final Tracer t = CommandLine.tracer();
				if (t.isDebug()) {
					t.debug("%s with regex '%s' resulted in %s parts: %s", msg, splitRegex(), result.length,
							Arrays.asList(result));
				}
				return result;
			}

			/**
			 * Returns the default value to assign if this option or positional parameter
			 * was not specified on the command line, before splitting and type conversion.
			 * This method returns the programmatically set value; this may differ from the
			 * default value that is actually used: if this ArgSpec is part of a CommandSpec
			 * with a {@link IDefaultValueProvider}, picocli will first try to obtain the
			 * default value from the default value provider, and this method is only called
			 * if the default provider is {@code null} or returned a {@code null} value.
			 * 
			 * @return the programmatically set default value of this option/positional
			 *         parameter, returning {@code null} means this option or positional
			 *         parameter does not have a default
			 * @see CommandSpec#defaultValueProvider()
			 * @see OptionSpec#fallbackValue()
			 */
			public String defaultValue() {
				return interpolate(defaultValue);
			}

			private String defaultValueFromProvider() {
				if (commandSpec == null) {
					return null;
				} // still initializing...
				String fromProvider = null;
				IDefaultValueProvider defaultValueProvider = null;
				try {
					defaultValueProvider = commandSpec.defaultValueProvider();
					fromProvider = defaultValueProvider == null ? null : defaultValueProvider.defaultValue(this);
				} catch (final Exception ex) {
					CommandLine.tracer().info("Error getting default value for %s from %s: %s", this,
							defaultValueProvider, ex);
				}
				return fromProvider;
			}

			/**
			 * Returns the default value String for the purpose of displaying it in the
			 * description, without interpolating variables.
			 * 
			 * @see #defaultValueString(boolean)
			 */
			public String defaultValueString() {
				return defaultValueString(false);
			}

			/**
			 * Returns the default value String displayed in the description; interpolating
			 * variables if specified. If this ArgSpec is part of a CommandSpec with a
			 * {@link IDefaultValueProvider}, this method will first try to obtain the
			 * default value from the default value provider; if the provider is
			 * {@code null} or if it returns a {@code null} value, then next any value set
			 * to {@link ArgSpec#defaultValue()} is returned, and if this is also
			 * {@code null}, finally the {@linkplain ArgSpec#initialValue() initial value}
			 * is returned.
			 * 
			 * @param interpolateVariables whether to interpolate variables in the
			 *                             {@code defaultValue} attribute of this ArgSpec
			 * @see CommandSpec#defaultValueProvider()
			 * @see ArgSpec#defaultValue()
			 * @since 4.0
			 */
			public String defaultValueString(boolean interpolateVariables) {
				// implementation note: don't call this.defaultValue(), that will interpolate
				// variables too soon!
				final Object value = calcDefaultValue(interpolateVariables);
				if (value != null && value.getClass().isArray()) {
					final StringBuilder sb = new StringBuilder();
					for (int i = 0; i < Array.getLength(value); i++) {
						sb.append(i > 0 ? ", " : "").append(Array.get(value, i));
					}
					return sb.insert(0, "[").append("]").toString();
				}
				return String.valueOf(value);
			}

			/**
			 * Returns the description of this option or positional parameter, after all
			 * variables have been rendered, including the {@code ${DEFAULT-VALUE}} and
			 * {@code ${COMPLETION-CANDIDATES}} variables. Use
			 * {@link CommandSpec#interpolateVariables(Boolean)} to switch off variable
			 * expansion if needed.
			 * <p>
			 * If a resource bundle has been
			 * {@linkplain ArgSpec#messages(CommandLine.Model.Messages) set}, this method
			 * will first try to find a value in the resource bundle: If the resource bundle
			 * has no entry for the
			 * {@code fully qualified commandName + "." + descriptionKey} or for the
			 * unqualified {@code descriptionKey}, an attempt is made to find the option or
			 * positional parameter description using any of the
			 * {@linkplain #getAdditionalDescriptionKeys() additional description keys},
			 * first with the {@code fully qualified commandName + "."} prefix, then
			 * without.
			 * </p>
			 * 
			 * @see CommandSpec#qualifiedName(String)
			 * @see #getAdditionalDescriptionKeys()
			 * @see Parameters#description()
			 * @see Option#description()
			 */
			public String[] description() {
				String[] result = description.clone();
				if (messages() != null) { // localize if possible
					String[] newValue = messages().getStringArray(descriptionKey(), null);
					if (newValue == null) {
						for (final String name : getAdditionalDescriptionKeys()) {
							newValue = messages().getStringArray(name, null);
							if (newValue != null) {
								result = newValue;
								break;
							}
						}
					} else {
						result = newValue;
					}
				}
				if (commandSpec == null || commandSpec.interpolateVariables()) { // expand variables
					result = expandVariables(result);
				}
				return result;
			}

			/**
			 * Returns the description key of this arg spec, used to get the description
			 * from a resource bundle.
			 * 
			 * @see Option#descriptionKey()
			 * @see Parameters#descriptionKey()
			 * @since 3.6
			 */
			public String descriptionKey() {
				return interpolate(descriptionKey);
			}

			/**
			 * Returns whether the user input is echoed to the console or not for an
			 * interactive option or positional parameter when asking for user input.
			 * 
			 * @see Option#echo()
			 * @see Parameters#echo()
			 * @since 4.6
			 */
			public boolean echo() {
				return echo;
			}

			protected boolean equalsImpl(ArgSpec other) {
				return Assert.equals(this.defaultValue, other.defaultValue)
						&& Assert.equals(this.mapFallbackValue, other.mapFallbackValue)
						&& Assert.equals(this.arity, other.arity) && Assert.equals(this.hidden, other.hidden)
						&& Assert.equals(this.inherited, other.inherited)
						&& Assert.equals(this.paramLabel, other.paramLabel)
						&& Assert.equals(this.hideParamSyntax, other.hideParamSyntax)
						&& Assert.equals(this.required, other.required)
						&& Assert.equals(this.splitRegex, other.splitRegex)
						&& Assert.equals(this.splitRegexSynopsisLabel, other.splitRegexSynopsisLabel)
						&& Arrays.equals(this.description, other.description)
						&& Assert.equals(this.descriptionKey, other.descriptionKey)
						&& Assert.equals(this.parameterConsumer, other.parameterConsumer)
						&& Assert.equals(this.preprocessor, other.preprocessor) && this.typeInfo.equals(other.typeInfo)
						&& this.scopeType.equals(other.scopeType);
			}

			private String[] expandVariables(String[] desc) {
				if (desc.length == 0) {
					return desc;
				}
				final StringBuilder candidates = new StringBuilder();
				boolean isCompletionCandidatesUsed = false;
				for (final String s : desc) {
					if (s.contains(DESCRIPTION_VARIABLE_COMPLETION_CANDIDATES)) {
						isCompletionCandidatesUsed = true;
						break;
					}
				}
				if (isCompletionCandidatesUsed) {
					final Iterable<String> iter = completionCandidates();
					if (iter != null) {
						for (final String c : iter) {
							if (candidates.length() > 0) {
								candidates.append(", ");
							}
							candidates.append(c);
						}
					}
				}
				final String defaultValueString = defaultValueString(false); // interpolate later
				final String fallbackValueString = isOption() ? ((OptionSpec) this).fallbackValue : ""; // interpolate
																										// later
				final String mapFallbackValueString = String.valueOf(mapFallbackValue); // interpolate later
				final String[] result = new String[desc.length];
				for (int i = 0; i < desc.length; i++) {
					result[i] = format(desc[i]
							.replace(DESCRIPTION_VARIABLE_DEFAULT_VALUE, defaultValueString.replace("%", "%%"))
							.replace(DESCRIPTION_VARIABLE_FALLBACK_VALUE, fallbackValueString.replace("%", "%%"))
							.replace(DESCRIPTION_VARIABLE_MAP_FALLBACK_VALUE, mapFallbackValueString.replace("%", "%%"))
							.replace(DESCRIPTION_VARIABLE_COMPLETION_CANDIDATES, candidates.toString()));
				}
				return interpolate(result);
			}

			/**
			 * Subclasses should override to return a collection of additional description
			 * keys that may be used to find description text for this option or positional
			 * parameter in the resource bundle.
			 * 
			 * @see OptionSpec#getAdditionalDescriptionKeys()
			 * @see PositionalParamSpec#getAdditionalDescriptionKeys()
			 * @since 4.0
			 */
			protected abstract Collection<String> getAdditionalDescriptionKeys();

			/**
			 * Returns the {@link IGetter} that is responsible for supplying the value of
			 * this argument.
			 */
			public IGetter getter() {
				return getter;
			}

			/**
			 * Returns the current value of this argument. Delegates to the current
			 * {@link #getter()}.
			 */
			public <T> T getValue() throws PicocliException {
				if (!isValueGettable()) {
					return null;
				}
				try {
					return getter.<T>get();
				} catch (final PicocliException ex) {
					throw ex;
				} catch (final Exception ex) {
					throw new PicocliException("Could not get value for " + this + ": " + ex, ex);
				}
			}

			/**
			 * Returns the groups this option or positional parameter belongs to, or
			 * {@code null} if this option is not part of a group.
			 * 
			 * @since 4.0
			 */
			public ArgGroupSpec group() {
				return group;
			}

			protected int hashCodeImpl() {
				return 17 + 37 * Assert.hashCode(defaultValue) + 37 * Assert.hashCode(mapFallbackValue)
						+ 37 * Assert.hashCode(arity) + 37 * Assert.hashCode(hidden) + 37 * Assert.hashCode(inherited)
						+ 37 * Assert.hashCode(paramLabel) + 37 * Assert.hashCode(hideParamSyntax)
						+ 37 * Assert.hashCode(required) + 37 * Assert.hashCode(splitRegex)
						+ 37 * Assert.hashCode(splitRegexSynopsisLabel) + 37 * Arrays.hashCode(description)
						+ 37 * Assert.hashCode(descriptionKey) + 37 * Assert.hashCode(parameterConsumer)
						+ 37 * Assert.hashCode(preprocessor) + 37 * typeInfo.hashCode() + 37 * scopeType.hashCode();
			}

			/**
			 * Determines whether the option or positional parameter will be reset to the
			 * {@link #initialValue()} before parsing new input.
			 */
			public boolean hasInitialValue() {
				return hasInitialValue || initialValueState == InitialValueState.CACHED
						|| initialValueState == InitialValueState.POSTPONED;
			}

			/**
			 * Returns whether this option should be excluded from the usage message.
			 * 
			 * @see Option#hidden()
			 */
			public boolean hidden() {
				return hidden;
			}

			/**
			 * Returns whether usage syntax decorations around the {@linkplain #paramLabel()
			 * paramLabel} should be suppressed. The default is {@code false}: by default,
			 * the paramLabel is surrounded with {@code '['} and {@code ']'} characters if
			 * the value is optional and followed by ellipses ("...") when multiple values
			 * can be specified.
			 * 
			 * @since 3.6.0
			 */
			public boolean hideParamSyntax() {
				return hideParamSyntax;
			}

			/**
			 * Returns whether this option is inherited from a parent command.
			 * 
			 * @see Option#scope()
			 * @since 4.3.0
			 */
			public boolean inherited() {
				return inherited;
			}

			/**
			 * Returns the initial value of this option or positional parameter: the value
			 * that, if {@link #hasInitialValue()} is true, the option will be reset to
			 * before parsing (regardless of whether a default value exists), to clear
			 * values that would otherwise remain from parsing previous input.
			 */
			public Object initialValue() {
				// not not initialize if already CACHED, or UNAVAILABLE, or if
				// annotatedElement==null
				if (initialValueState == InitialValueState.POSTPONED && annotatedElement != null) {
					try {
						initialValue = annotatedElement.getter().get();
						initialValueState = InitialValueState.CACHED; // only if successfully initialized
					} catch (final Exception ex) {
					} // #1300 if error: keep initialValueState == POSTPONED
				}
				return initialValue;
			}

			/**
			 * Returns whether this option will prompt the user to enter a value on the
			 * command line.
			 * 
			 * @see Parameters#interactive()
			 * @see Option#interactive()
			 */
			public boolean interactive() {
				return interactive;
			}

			/**
			 * Returns whether the default for this option or positional parameter should be
			 * shown, potentially overriding the specified global setting.
			 * 
			 * @param usageHelpShowDefaults whether the command's UsageMessageSpec is
			 *                              configured to show default values.
			 */
			protected boolean internalShowDefaultValue(boolean usageHelpShowDefaults) {
				if (showDefaultValue() == Help.Visibility.ALWAYS) {
					return true;
				} // override global usage help setting
				if (showDefaultValue() == Help.Visibility.NEVER) {
					return false;
				} // override global usage help setting
				if (initialValue() == null && defaultValue() == null && defaultValueFromProvider() == null) {
					return false;
				} // no default value to show
				return usageHelpShowDefaults && !isBoolean(type());
			}

			String interpolate(String value) {
				return commandSpec == null ? value : commandSpec.interpolator.interpolate(value);
			}

			String[] interpolate(String[] values) {
				return commandSpec == null ? values : commandSpec.interpolator.interpolate(values);
			}

			/**
			 * Returns {@code true} if this argument's {@link #type()} is an array, a
			 * {@code Collection} or a {@code Map}, {@code false} otherwise.
			 */
			public boolean isMultiValue() {
				return typeInfo.isMultiValue();
			}

			/**
			 * Returns {@code true} if this argument is a named option, {@code false}
			 * otherwise.
			 */
			public abstract boolean isOption();

			/**
			 * Returns {@code true} if this argument is a positional parameter,
			 * {@code false} otherwise.
			 */
			public abstract boolean isPositional();

			/**
			 * Check whether the {@link #getValue()} method is able to get an actual value
			 * from the current {@link #getter()}.
			 * 
			 * @since 4.7
			 */
			public boolean isValueGettable() {
				if (getter instanceof IScoped) {
					final IScoped scoped = (IScoped) getter;
					final IScope scope = scoped.getScope();
					if (scope == null) {
						return false;
					}
					try {
						return scope.get() != null;
					} catch (final Exception e) {
						return false;
					}
				}
				return true;
			}

			/**
			 * Returns the fallback value for this Map option or positional parameter: the
			 * value that is put into the Map when only the key is specified for the option
			 * or positional parameter, like {@code -Dkey} instead of {@code -Dkey=value}.
			 * <p>
			 * If no {@code mapFallbackValue} is set, key-only Map parameters like
			 * {@code -Dkey} are considered invalid user input and cause a
			 * {@link ParameterException} to be thrown.
			 * </p>
			 * <p>
			 * By default, this method returns a special "__unspecified__" value indicating
			 * that no {@code mapFallbackValue} was set.
			 * </p>
			 * 
			 * @see Option#mapFallbackValue()
			 * @see Parameters#mapFallbackValue()
			 * @since 4.6
			 */
			public String mapFallbackValue() {
				final String result = interpolate(mapFallbackValue);
				return NULL_VALUE.equals(result) ? null : result;
			}

			/**
			 * Returns the Messages for this arg specification, or {@code null}.
			 * 
			 * @since 3.6
			 */
			public Messages messages() {
				return messages;
			}

			/**
			 * Sets the Messages for this ArgSpec, and returns this ArgSpec.
			 * 
			 * @param msgs the new Messages value, may be {@code null}
			 * @see Command#resourceBundle()
			 * @see OptionSpec#description()
			 * @see PositionalParamSpec#description()
			 * @since 3.6
			 */
			public ArgSpec messages(Messages msgs) {
				messages = msgs;
				return this;
			}

			/**
			 * Returns the original value of the option's required attribute, regardless of
			 * whether the option is used in an exclusive group or not.
			 * 
			 * @since 4.7.6-SNAPSHOT
			 * @see Option#required()
			 */
			public boolean originallyRequired() {
				return originallyRequired;
			}

			/**
			 * Returns the original command line arguments matched by this option or
			 * positional parameter spec.
			 * 
			 * @return the matched arguments as found on the command line: empty Strings for
			 *         options without value, the values have not been
			 *         {@linkplain #splitRegex() split}, and for map properties values may
			 *         look like {@code "key=value"}
			 */
			public List<String> originalStringValues() {
				return Collections.unmodifiableList(originalStringValues);
			}

			/**
			 * Returns a custom {@code IParameterConsumer} to temporarily suspend picocli's
			 * parsing logic and process one or more command line arguments in a custom
			 * manner, or {@code null}. An example of when this may be useful is when
			 * passing arguments through to another program.
			 * 
			 * @since 4.0
			 */
			public IParameterConsumer parameterConsumer() {
				return parameterConsumer;
			}

			/**
			 * Returns the name of the option or positional parameter used in the usage help
			 * message.
			 * 
			 * @see Option#paramLabel()
			 * @see Parameters#paramLabel()
			 */
			public String paramLabel() {
				return interpolate(paramLabel);
			}

			/**
			 * Returns a custom {@code IParameterPreprocessor} to either replace or
			 * complement picocli's parsing logic for the parameter(s) of this option or
			 * position.
			 * 
			 * @since 4.6
			 */
			public IParameterPreprocessor preprocessor() {
				return preprocessor;
			}

			/**
			 * Returns the text displayed to the end user for an interactive option or
			 * positional parameter when asking for user input.
			 * 
			 * @see Option#prompt()
			 * @see Parameters#prompt()
			 * @since 4.6
			 */
			public String prompt() {
				return prompt;
			}

			/** @deprecated Use {@link #description()} instead */
			@Deprecated
			public String[] renderedDescription() {
				return description();
			}

			/**
			 * Returns whether this is a required option or positional parameter without a
			 * default value. If this argument is part of a {@linkplain ArgGroup group},
			 * this method returns whether this argument is required <em>within the
			 * group</em> (so it is not necessarily a required argument for the command).
			 * 
			 * @see Option#required()
			 */
			public boolean required() {
				// #261 not required if it has a default; #676 default value may be a variable
				return required && defaultValue() == null && defaultValueFromProvider() == null;
			}

			/** Sets the {@code originalStringValues} to a new list instance. */
			protected void resetOriginalStringValues() {
				originalStringValues = new ArrayList<String>();
			}

			/** Sets the {@code stringValues} to a new list instance. */
			protected void resetStringValues() {
				stringValues = new ArrayList<String>();
			}

			/**
			 * Returns the root option or positional parameter (on the parent command), if
			 * this option or positional parameter was inherited; or {@code null} if it was
			 * not.
			 * 
			 * @see Option#scope()
			 * @since 4.6.0
			 */
			public ArgSpec root() {
				return root;
			}

			/**
			 * Returns the binding {@link IScope} that determines on which object to set the
			 * value (or from which object to get the value) of this argument.
			 */
			public IScope scope() {
				return scope;
			}

			private String scopeString() {
				try {
					final Object obj = scope.get();
					if (obj == null) {
						return "<no user object>";
					}
					return obj.getClass().getSimpleName() + "@" + Integer.toHexString(System.identityHashCode(obj));
				} catch (final Exception ex) {
					return "?: " + ex.toString();
				}
			}

			/**
			 * Returns the scope of this argument; it it local, or inherited (it applies to
			 * this command as well as all sub- and sub-subcommands).
			 * 
			 * @return whether this argument applies to all descendent subcommands of the
			 *         command where it is defined
			 * @since 4.3
			 */
			public ScopeType scopeType() {
				return scopeType;
			}

			/**
			 * Returns the {@link ISetter} that is responsible for modifying the value of
			 * this argument.
			 */
			public ISetter setter() {
				return setter;
			}

			/**
			 * Sets the value of this argument to the specified value and returns the
			 * previous value. Delegates to the current {@link #setter()}.
			 */
			public <T> T setValue(T newValue) throws PicocliException {
				try {
					return setter.set(newValue);
				} catch (final PicocliException ex) {
					throw ex;
				} catch (final Exception ex) {
					throw new PicocliException("Could not set value (" + newValue + ") for " + this + ": " + ex, ex);
				}
			}

			/**
			 * Sets the value of this argument to the specified value and returns the
			 * previous value. Delegates to the current {@link #setter()}.
			 * 
			 * @deprecated use {@link #setValue(Object)} instead. This was a design mistake.
			 * @since 3.5
			 */
			@Deprecated
			public <T> T setValue(T newValue, CommandLine commandLine) throws PicocliException {
				return setValue(newValue);
			}

			/**
			 * Returns whether this option or positional parameter's default value should be
			 * shown in the usage help.
			 */
			public Help.Visibility showDefaultValue() {
				return showDefaultValue;
			}

			/**
			 * Returns a regular expression to split option parameter values or {@code ""}
			 * if the value should not be split.
			 * 
			 * @see Option#split()
			 */
			public String splitRegex() {
				return interpolate(splitRegex);
			}

			/**
			 * Returns a regular expression to split option parameter for usage information.
			 * 
			 * @see Option#splitSynopsisLabel()
			 * @since 4.3
			 */
			public String splitRegexSynopsisLabel() {
				return interpolate(splitRegexSynopsisLabel);
			}

			String[] splitValue(String value, ParserSpec parser, Range arity, int consumed) {
				if (splitRegex().length() == 0) {
					return new String[] { value };
				}
				final int limit = parser.limitSplit() ? Math.max(arity.max - consumed, 0) : 0;
				if (parser.splitQuotedStrings()) {
					return debug(value.split(splitRegex(), limit), "Split (ignoring quotes)", value);
				}
				return debug(splitRespectingQuotedStrings(value, limit, parser, this, splitRegex()), "Split", value);
			}

			/**
			 * Returns the untyped command line arguments matched by this option or
			 * positional parameter spec.
			 * 
			 * @return the matched arguments after {@linkplain #splitRegex() splitting}, but
			 *         before type conversion. For map properties, {@code "key=value"}
			 *         values are split into the key and the value part.
			 */
			public List<String> stringValues() {
				return Collections.unmodifiableList(stringValues);
			}

			/** Returns a string respresentation of this option or positional parameter. */
			@Override
			public String toString() {
				return toString;
			}

			/**
			 * Returns the type to convert the option or positional parameter to before
			 * {@linkplain #setValue(Object) setting} the value. This may be a container
			 * type like {@code List}, {@code Map}, or {@code Optional}, in which case the
			 * type or types of the elements are returned by {@link #auxiliaryTypes()}.
			 */
			public Class<?> type() {
				return typeInfo.getType();
			}

			/**
			 * Returns the typed command line arguments matched by this option or positional
			 * parameter spec.
			 * 
			 * @return the matched arguments after {@linkplain #splitRegex() splitting} and
			 *         type conversion. For map properties, {@code "key=value"} values are
			 *         split into the key and the value part.
			 */
			public List<Object> typedValues() {
				return Collections.unmodifiableList(typedValues);
			}

			/**
			 * Returns the {@code ITypeInfo} that can be used both at compile time (by
			 * annotation processors) and at runtime.
			 * 
			 * @since 4.0
			 */
			public ITypeInfo typeInfo() {
				return typeInfo;
			}

			/**
			 * Returns the user object associated with this option or positional parameters.
			 * 
			 * @return may return the annotated program element, or some other useful object
			 * @since 4.0
			 */
			public Object userObject() {
				return userObject;
			}
		}

		/**
		 * This class provides a case-aware Linked HashMap. Supports both case-sensitive
		 * and case-insensitive modes.
		 * 
		 * @param <V> type of the value
		 */
		static class CaseAwareLinkedMap<K, V> extends AbstractMap<K, V> {
			class CaseAwareKeySet extends AbstractSet<K> {
				@Override
				public boolean contains(Object o) {
					return containsKey(o);
				}

				@Override
				public Iterator<K> iterator() {
					return targetMap.keySet().iterator();
				}

				@Override
				public int size() {
					return targetMap.keySet().size();
				}
			}

			static boolean isCaseConvertible(Class<?> clazz) {
				return clazz == String.class || clazz == Character.class;
			}

			private final LinkedHashMap<K, V> targetMap = new LinkedHashMap<K, V>();
			private final HashMap<K, K> keyMap = new HashMap<K, K>();
			private final Set<K> keySet = new CaseAwareKeySet();
			private boolean caseInsensitive = false;

			private final Locale locale;

			/**
			 * Constructs an empty {@code CaseAwareLinkedMap} instance with
			 * {@link java.util.Locale#ENGLISH}.
			 */
			public CaseAwareLinkedMap() {
				this(ENGLISH);
			}

			/**
			 * Constructs a {@code CaseAwareLinkedMap} instance with the same mappings,
			 * case-sensitivity and locale as the specified map.
			 * 
			 * @param map the map whose mappings, case-sensitivity and locale are to be
			 *            placed in this map
			 * @throws NullPointerException if the specified map is null
			 */
			public CaseAwareLinkedMap(CaseAwareLinkedMap<? extends K, ? extends V> map) {
				this.targetMap.putAll(map.targetMap);
				this.keyMap.putAll(map.keyMap);
				this.caseInsensitive = map.caseInsensitive;
				this.locale = map.locale;
			}

			/**
			 * Constructs an empty {@code CaseAwareLinkedMap} instance with the specified
			 * {@link java.util.Locale}.
			 * 
			 * @param locale the locale to convert character cases
			 */
			public CaseAwareLinkedMap(Locale locale) {
				this.locale = locale;
			}

			@Override
			public void clear() {
				targetMap.clear();
				keyMap.clear();
			}

			@Override
			public boolean containsKey(Object key) {
				if (key != null && caseInsensitive) {
					if (!isCaseConvertible(key.getClass())) {
						return false;
					}
					return keyMap.containsKey(toLowerCase(key));
				} else {
					return targetMap.containsKey(key);
				}
			}

			@Override
			public boolean containsValue(Object value) {
				return targetMap.containsValue(value);
			}

			@Override
			public Set<Entry<K, V>> entrySet() {
				return targetMap.entrySet();
			}

			@Override
			public V get(Object key) {
				if (key != null && isCaseConvertible(key.getClass()) && caseInsensitive) {
					final K caseSensitiveKey = keyMap.get(toLowerCase(key));
					if (caseSensitiveKey == null) {
						return null;
					}
					return targetMap.get(caseSensitiveKey);
				} else {
					return targetMap.get(key);
				}
			}

			/**
			 * Returns the case-sensitive key of the specified case-insensitive key if
			 * {@code isCaseSensitive()}. Otherwise, the specified case-insensitive key is
			 * returned.
			 */
			public K getCaseSensitiveKey(K caseInsensitiveKey) {
				if (caseInsensitiveKey != null && caseInsensitive) {
					return keyMap.get(toLowerCase(caseInsensitiveKey));
				} else {
					return caseInsensitiveKey;
				}
			}

			/** Returns the locale of the map. */
			public Locale getLocale() {
				return locale;
			}

			/** Returns the case-insensitivity of the map. */
			public boolean isCaseInsensitive() {
				return caseInsensitive;
			}

			@Override
			public Set<K> keySet() {
				return keySet;
			}

			@Override
			public V put(K key, V value) {
				if (key != null && caseInsensitive) {
					final K caseSensitiveKey = keyMap.put(toLowerCase(key), key);
					if (caseSensitiveKey != null) {
						final V removedValue = targetMap.remove(caseSensitiveKey);
						targetMap.put(key, value);
						return removedValue;
					}
				}
				return targetMap.put(key, value);
			}

			@Override
			public V remove(Object key) {
				if (key != null && caseInsensitive) {
					final K caseSensitiveKey = keyMap.remove(toLowerCase(key));
					if (caseSensitiveKey == null) {
						return null;
					}
					return targetMap.remove(caseSensitiveKey);
				} else {
					return targetMap.remove(key);
				}
			}

			/** Sets the case-insensitivity of the map. */
			public void setCaseInsensitive(boolean caseInsensitive) {
				if (!isCaseInsensitive() && caseInsensitive) {
					for (final K key : targetMap.keySet()) {
						final K duplicatedKey = keyMap.put(key != null ? toLowerCase(key) : null, key);
						if (duplicatedKey != null) {
							throw new DuplicateNameException("Duplicated keys: " + duplicatedKey + " and " + key);
						}
					}
				} else if (isCaseInsensitive()) {
					keyMap.clear();
				}
				this.caseInsensitive = caseInsensitive;
			}

			@Override
			public int size() {
				return targetMap.size();
			}

			@SuppressWarnings("unchecked")
			private K toLowerCase(Object caseSensitiveKey) {
				if (caseSensitiveKey.getClass() == String.class) {
					return (K) ((String) caseSensitiveKey).toLowerCase(locale);
				} else if (caseSensitiveKey.getClass() == Character.class) {
					return (K) (Character) Character.toLowerCase((Character) caseSensitiveKey);
				} else {
					throw new UnsupportedOperationException(
							"Unsupported case-conversion for key " + caseSensitiveKey.getClass());
				}
			}

			@Override
			public Collection<V> values() {
				return targetMap.values();
			}
		}

		private static class CommandReflection {
			@SuppressWarnings("unchecked") // warning: [unchecked] Possible heap pollution from parameterized vararg
											// type Class<? extends Annotation>
			private static void assertNoDuplicateAnnotations(TypedMember member,
					Class<? extends Annotation> myAnnotation, Class<? extends Annotation>... forbidden) {
				for (final Class<? extends Annotation> annotation : forbidden) {
					if (member.isAnnotationPresent(annotation)) {
						throw new DuplicateOptionAnnotationsException(
								"A member cannot have both @" + myAnnotation.getSimpleName() + " and @"
										+ annotation.getSimpleName() + " annotations, but '" + member + "' has both.");
					}
				}
			}

			private static ArgSpec buildArgForMember(IAnnotatedElement member, IFactory factory) {
				if (member.isOption()) {
					return OptionSpec.builder(member, factory).build();
				} else if (member.isParameter()) {
					return PositionalParamSpec.builder(member, factory).build();
				} else {
					return PositionalParamSpec.builder(member, factory).build();
				}
			}

			private static ArgGroupSpec buildArgGroupForMember(IAnnotatedElement member, IFactory factory,
					CommandSpec commandSpec) {
				try {
					return extractArgGroupSpec(member, factory, commandSpec, true);
				} catch (final InitializationException ex) {
					throw ex;
				} catch (final Exception ex) {
					throw new InitializationException(
							"Could not access or modify ArgGroup member " + member + ": " + ex, ex);
				}
			}

			private static CommandSpec buildMixinForMember(IAnnotatedElement member, IFactory factory) {
				try {
					Object userObject = member.getter().get();
					if (userObject == null) {
						userObject = factory.create(member.getTypeInfo().getType());
						member.setter().set(userObject);
					}
					final CommandSpec result = CommandSpec.forAnnotatedObject(userObject, factory);
					return result.withToString(member.getToString());
				} catch (final InitializationException ex) {
					throw ex;
				} catch (final Exception ex) {
					throw new InitializationException("Could not access or modify mixin member " + member + ": " + ex,
							ex);
				}
			}

			private static UnmatchedArgsBinding buildUnmatchedForMember(final IAnnotatedElement member) {
				final ITypeInfo info = member.getTypeInfo();
				if (!(info.getClassName().equals(String[].class.getName()) || (info.isCollection() && info
						.getActualGenericTypeArguments().equals(Collections.singletonList(String.class.getName()))))) {
					throw new InitializationException(
							"Invalid type for " + member + ": must be either String[] or List<String>");
				}
				if (info.getClassName().equals(String[].class.getName())) {
					return UnmatchedArgsBinding.forStringArrayConsumer(member.setter());
				} else {
					return UnmatchedArgsBinding.forStringCollectionSupplier(member.getter());
				}
			}

			static ArgGroupSpec extractArgGroupSpec(IAnnotatedElement member, IFactory factory, CommandSpec commandSpec,
					boolean annotationsAreMandatory) throws Exception {
				Object instance = null;
				try {
					instance = member.getter().get();
				} catch (final Exception ignored) {
				}
				Class<?> cls = instance == null ? member.getTypeInfo().getType() : instance.getClass();

				if (member.isMultiValue()) {
					cls = member.getTypeInfo().getAuxiliaryTypes()[0];
				}
				final IScope scope = new ObjectScope(instance);
				final ArgGroupSpec.Builder builder = ArgGroupSpec.builder(member);
				builder.updateArgGroupAttributes(member.getAnnotation(ArgGroup.class));
				if (member.isOption() || member.isParameter()) {
					if (member instanceof TypedMember) {
						validateArgSpecMember((TypedMember) member);
					}
					builder.addArg(buildArgForMember(member, factory));
				}

				final Stack<Class<?>> hierarchy = new Stack<Class<?>>();
				while (cls != null) {
					hierarchy.add(cls);
					cls = cls.getSuperclass();
				}
				boolean hasArgAnnotation = false;
				while (!hierarchy.isEmpty()) {
					cls = hierarchy.pop();
					hasArgAnnotation |= initFromAnnotatedMembers(scope, cls, commandSpec, builder, factory, null);
				}
				final ArgGroupSpec result = builder.build();
				if (annotationsAreMandatory) {
					validateArgGroupSpec(result, hasArgAnnotation, cls.getName());
				}
				return result;
			}

			static CommandSpec extractCommandSpec(Object command, IFactory factory, boolean annotationsAreMandatory) {
				Assert.notNull(command, "command user object");
				final Tracer t = CommandLine.tracer();
				if (command instanceof CommandSpec) {
					t.debug("extractCommandSpec returning existing CommandSpec instance %s", command);
					return (CommandSpec) command;
				}

				final CommandUserObject userObject = CommandUserObject.create(command, factory);
				t.debug("Creating CommandSpec for %s with factory %s", userObject, factory.getClass().getName());
				final CommandSpec result = CommandSpec.wrapWithoutInspection(userObject, factory);

				boolean hasCommandAnnotation = false;
				if (userObject.isMethod()) {
					final Method method = (Method) command;
					t.debug("Using method %s as command ", method);
					method.setAccessible(true);
					final Command cmd = method.getAnnotation(Command.class);
					result.updateCommandAttributes(cmd, factory);
					injectSpecIntoVersionProvider(result, cmd, factory);
					result.setAddMethodSubcommands(false); // method commands don't have method subcommands
					hasCommandAnnotation = true;
					initSubcommands(cmd, null, result, factory, Collections.<Class<?>>emptySet()); // after adding
																									// options
					result.mixinStandardHelpOptions(cmd.mixinStandardHelpOptions()); // do this last
					initFromMethodParameters(userObject, method, result, null, factory);
					// set command name to method name, unless @Command#name is set
					result.initName(((Method) command).getName());
				} else {
					Class<?> cls = userObject.getType();
					final Stack<Class<?>> hierarchy = new Stack<Class<?>>();
					while (cls != null) {
						hierarchy.add(cls);
						cls = cls.getSuperclass();
					}
					final Set<Class<?>> fullHierarchySet = new HashSet<Class<?>>(hierarchy);
					boolean mixinStandardHelpOptions = false;
					while (!hierarchy.isEmpty()) {
						cls = hierarchy.pop();
						final Command cmd = cls.getAnnotation(Command.class);
						if (cmd != null) {
							result.updateCommandAttributes(cmd, factory);
							injectSpecIntoVersionProvider(result, cmd, factory);
							hasCommandAnnotation = true;
							mixinStandardHelpOptions |= cmd.mixinStandardHelpOptions();
						}
						initSubcommands(cmd, cls, result, factory, fullHierarchySet); // after adding options
						initMethodSubcommands(cls, result, factory); // regardless of @Command annotation. NOTE: after
																		// adding options
						hasCommandAnnotation |= initFromAnnotatedMembers(userObject, cls, result, null, factory, null);
					}
					result.mixinStandardHelpOptions(mixinStandardHelpOptions); // #377 Standard help options should be
																				// added last
				}

				result.updateArgSpecMessages();

				if (annotationsAreMandatory) {
					validateCommandSpec(result, hasCommandAnnotation, userObject.toString());
				}
				result.validate();
				return result;
			}

			private static boolean initFromAnnotatedMembers(IScope scope, Class<?> cls, CommandSpec receiver,
					ArgGroupSpec.Builder groupBuilder, IFactory factory, Predicate<TypedMember> predicate) {
				boolean result = false;
				for (final Field field : cls.getDeclaredFields()) {
					result |= initFromAnnotatedTypedMembers(TypedMember.createIfAnnotated(field, scope), predicate,
							receiver, groupBuilder, factory);
				}
				for (final Method method : cls.getDeclaredMethods()) {
					result |= initFromAnnotatedTypedMembers(TypedMember.createIfAnnotated(method, scope, receiver),
							predicate, receiver, groupBuilder, factory);
				}
				return result;
			}

			@SuppressWarnings("unchecked")
			private static boolean initFromAnnotatedTypedMembers(TypedMember member, Predicate<TypedMember> predicate,
					CommandSpec commandSpec, ArgGroupSpec.Builder groupBuilder, IFactory factory) {
				if (member == null || (predicate != null && !predicate.test(member))) {
					return false;
				}
				boolean result = false;
				if (member.isMixin()) {
					assertNoDuplicateAnnotations(member, Mixin.class, Option.class, Parameters.class, Unmatched.class,
							Spec.class, ArgGroup.class);
					if (groupBuilder != null) {
						throw new InitializationException("@Mixins are not supported on @ArgGroups");
						// TODO groupBuilder.addMixin(member.getMixinName(), buildMixinForMember(member,
						// factory));
					} else {
						final CommandSpec mixin = buildMixinForMember(member, factory);
						commandSpec.addMixin(member.getMixinName(), mixin, member);
						for (final IAnnotatedElement specElement : mixin.specElements) {
							if (specElement.getAnnotation(Spec.class).value() == Spec.Target.MIXEE) {
								try {
									specElement.setter().set(commandSpec);
								} catch (final Exception ex) {
									throw new InitializationException("Could not inject MIXEE spec", ex);
								}
							}
						}
					}
					result = true;
				}
				if (member.isArgGroup()) {
					assertNoDuplicateAnnotations(member, ArgGroup.class, Spec.class, Parameters.class, Option.class,
							Unmatched.class, Mixin.class);
					if (groupBuilder != null) {
						groupBuilder.addSubgroup(buildArgGroupForMember(member, factory, commandSpec));
					} else {
						commandSpec.addArgGroup(buildArgGroupForMember(member, factory, commandSpec));
					}
					return true;
				}
				if (member.isUnmatched()) {
					assertNoDuplicateAnnotations(member, Unmatched.class, Mixin.class, Option.class, Parameters.class,
							Spec.class, ArgGroup.class);
					if (groupBuilder != null) {
						// we don't support @Unmatched on @ArgGroup class members...
						throw new InitializationException("@Unmatched are not supported on @ArgGroups");
					} else {
						commandSpec.addUnmatchedArgsBinding(buildUnmatchedForMember(member));
					}
				}
				if (member.isArgSpec()) {
					validateArgSpecMember(member);
					if (groupBuilder != null) {
						groupBuilder.addArg(buildArgForMember(member, factory));
					} else {
						commandSpec.add(buildArgForMember(member, factory));
					}
					result = true;
				}
				if (member.isSpec()) {
					validateInjectSpec(member);
					if (groupBuilder != null) {
						groupBuilder.addSpecElement(member);
						// for ArgGroups, injecting the command spec is postponed until the user object
						// is created
					} else {
						commandSpec.addSpecElement(member);
						if (member.getAnnotation(Spec.class).value() == Spec.Target.SELF) {
							try {
								member.setter().set(commandSpec);
							} catch (final Exception ex) {
								throw new InitializationException("Could not inject spec", ex);
							}
						}
					}
				}
				if (member.isParentCommand()) {
					commandSpec.addParentCommandElement(member);
				}
				return result;
			}

			private static boolean initFromMethodParameters(IScope scope, Method method, CommandSpec receiver,
					ArgGroupSpec.Builder groupBuilder, IFactory factory) {
				boolean result = false;
				int optionCount = 0;
				final TypedMember[] members = new TypedMember[method.getParameterTypes().length];
				for (int i = 0, count = members.length; i < count; i++) {
					final MethodParam param = new MethodParam(method, i);
					if (param.isAnnotationPresent(Option.class) || param.isAnnotationPresent(Mixin.class)
							|| param.isAnnotationPresent(ArgGroup.class)) {
						optionCount++;
					} else {

						param.position = i - optionCount;
					}
					members[i] = new TypedMember(param, scope);
					result |= initFromAnnotatedTypedMembers(members[i], null, receiver, groupBuilder, factory);
				}
				receiver.methodParams = members;
				return result;
			}

			private static void initMethodSubcommands(Class<?> cls, CommandSpec parent, IFactory factory) {
				if (parent.isAddMethodSubcommands() && cls != null) {
					for (final CommandLine sub : CommandSpec.createMethodSubcommands(cls, factory, false)) {
						parent.addSubcommand(sub.getCommandName(), sub);
						for (final CommandSpec mixin : sub.getCommandSpec().mixins().values()) {
							mixin.injectParentCommand(parent.userObject);
						}
					}
				}
			}

			private static void initSubcommands(Command cmd, Class<?> cls, CommandSpec parent, IFactory factory,
					Set<Class<?>> hierarchy) {
				if (cmd == null) {
					return;
				}
				for (final Class<?> sub : cmd.subcommands()) {
					if (sub.equals(cls)) {
						throw new InitializationException(
								cmd.name() + " (" + cls.getName() + ") cannot be a subcommand of itself");
					}
					if (hierarchy.contains(sub)) {
						throw new InitializationException(cmd.name() + " (" + cls.getName() + ") has a subcommand ("
								+ sub.getName() + ") that is a subclass of itself");
					}
					try {
						if (Help.class == sub) {
							throw new InitializationException(Help.class.getName()
									+ " is not a valid subcommand. Did you mean " + HelpCommand.class.getName() + "?");
						}
						final CommandLine subcommandLine = toCommandLine(sub, factory);
						parent.addSubcommand(subcommandName(sub), subcommandLine);
						subcommandLine.getCommandSpec().injectParentCommand(parent.userObject);
						for (final CommandSpec mixin : subcommandLine.getCommandSpec().mixins().values()) {
							mixin.injectParentCommand(parent.userObject);
						}
					} catch (final InitializationException ex) {
						throw ex;
					} catch (final Exception ex) {
						throw new InitializationException(
								"Could not instantiate and add subcommand " + sub.getName() + ": " + ex, ex);
					}
				}
			}

			private static void injectSpecIntoVersionProvider(CommandSpec result, Command cmd, IFactory factory) {
				if (result.versionProvider() == null) {
					return;
				}
				initFromAnnotatedMembers(new ObjectScope(result.versionProvider()), cmd.versionProvider(), result, null,
						factory, new Predicate<TypedMember>() {
							@Override
							public boolean test(TypedMember tm) {
								return tm.isSpec() && !(tm.isArgGroup() || tm.isUnmatched() || tm.isMixin()
										|| tm.isOption() || tm.isParameter() || tm.isParentCommand());
							}
						});
			}

			private static String subcommandName(Class<?> sub) {
				final Command subCommand = sub.getAnnotation(Command.class);
				if (subCommand == null || Help.DEFAULT_COMMAND_NAME.equals(subCommand.name())) {
					throw new InitializationException("Subcommand " + sub.getName()
							+ " is missing the mandatory @Command annotation with a 'name' attribute");
				}
				return subCommand.name();
			}

			private static void validateArgGroupSpec(ArgGroupSpec result, boolean hasArgAnnotation, String className) {
				if (!hasArgAnnotation && result.args().isEmpty()) {
					throw new InitializationException(
							className + " is not a group: it has no @Option or @Parameters annotations");
				}
			}

			@SuppressWarnings("unchecked")
			private static void validateArgSpecMember(TypedMember member) {
				if (!member.isArgSpec()) {
					throw new IllegalStateException(
							"Bug: validateArgSpecMember() should only be called with an @Option or @Parameters member");
				}
				if (member.isOption()) {
					assertNoDuplicateAnnotations(member, Option.class, Unmatched.class, Mixin.class, Parameters.class,
							Spec.class, ArgGroup.class);
				} else {
					assertNoDuplicateAnnotations(member, Parameters.class, Option.class, Unmatched.class, Mixin.class,
							Spec.class, ArgGroup.class);
				}
				if (!(member.accessible instanceof Field)) {
					return;
				}
				final Field field = (Field) member.accessible;
				if (Modifier.isFinal(field.getModifiers())
						&& (field.getType().isPrimitive() || String.class.isAssignableFrom(field.getType()))) {
					throw new InitializationException("Constant (final) primitive and String fields like " + field
							+ " cannot be used as " + (member.isOption() ? "an @Option" : "a @Parameter")
							+ ": compile-time constant inlining may hide new values written to it.");
				}
			}

			private static void validateCommandSpec(CommandSpec result, boolean hasCommandAnnotation,
					String commandClassName) {
				if (!hasCommandAnnotation && result.positionalParameters.isEmpty() && result.optionsByNameMap.isEmpty()
						&& result.unmatchedArgs.isEmpty()) {
					throw new InitializationException(commandClassName
							+ " is not a command: it has no @Command, @Option, @Parameters or @Unmatched annotations");
				}
			}

			@SuppressWarnings("unchecked")
			private static void validateInjectSpec(TypedMember member) {
				if (!member.isSpec()) {
					throw new IllegalStateException(
							"Bug: validateInjectSpec() should only be called with @Spec members");
				}
				assertNoDuplicateAnnotations(member, Spec.class, Parameters.class, Option.class, Unmatched.class,
						Mixin.class, ArgGroup.class);
				if (!CommandSpec.class.getName().equals(member.getTypeInfo().getClassName())) {
					throw new InitializationException(
							"@picocli.CommandLine.Spec annotation is only supported on fields of type "
									+ CommandSpec.class.getName());
				}
			}
		}

		/**
		 * The {@code CommandSpec} class models a command specification, including the
		 * options, positional parameters and subcommands supported by the command, as
		 * well as attributes for the version help message and the usage help message of
		 * the command.
		 * <p>
		 * Picocli views a command line application as a hierarchy of commands: there is
		 * a top-level command (usually the Java class with the {@code main} method)
		 * with optionally a set of command line options, positional parameters and
		 * subcommands. Subcommands themselves can have options, positional parameters
		 * and nested sub-subcommands to any level of depth.
		 * </p>
		 * <p>
		 * The object model has a corresponding hierarchy of {@code CommandSpec}
		 * objects, each with a set of {@link OptionSpec}, {@link PositionalParamSpec}
		 * and {@linkplain CommandLine subcommands} associated with it. This object
		 * model is used by the picocli command line interpreter and help message
		 * generator.
		 * </p>
		 * <p>
		 * Picocli can construct a {@code CommandSpec} automatically from classes with
		 * {@link Command @Command}, {@link Option @Option} and
		 * {@link Parameters @Parameters} annotations. Alternatively a
		 * {@code CommandSpec} can be constructed programmatically.
		 * </p>
		 * 
		 * @since 3.0
		 */
		public static class CommandSpec {
			/**
			 * Constant String holding the default program name: {@code "<main class>" }.
			 * 
			 * @since 4.0
			 */
			public static final String DEFAULT_COMMAND_NAME = "<main class>";

			/**
			 * Constant Boolean holding the default setting for whether this is a help
			 * command: <code>{@value}</code>.
			 */
			static final Boolean DEFAULT_IS_HELP_COMMAND = false;

			/**
			 * Constant Boolean holding the default setting for whether method commands
			 * should be added as subcommands: <code>{@value}</code>.
			 */
			static final Boolean DEFAULT_IS_ADD_METHOD_SUBCOMMANDS = true;

			/**
			 * Constant Boolean holding the default setting for whether variables should be
			 * interpolated in String values: <code>{@value}</code>.
			 */
			static final Boolean DEFAULT_INTERPOLATE_VARIABLES = true;

			static final Boolean DEFAULT_SUBCOMMANDS_REPEATABLE = false;

			/**
			 * Creates and returns a new {@code CommandSpec} without any associated user
			 * object.
			 */
			public static CommandSpec create() {
				return wrapWithoutInspection(null);
			}

			static List<CommandLine> createMethodSubcommands(Class<?> cls, IFactory factory, boolean includeInherited) {
				final List<CommandLine> result = new ArrayList<CommandLine>();
				for (final Method method : getCommandMethods(cls, null, includeInherited)) {
					result.add(new CommandLine(method, factory));
				}
				return result;
			}

			static OptionSpec findOption(char shortName, Iterable<OptionSpec> options) {
				for (final OptionSpec option : options) {
					for (final String name : option.names()) {
						if (name.length() == 2 && name.charAt(0) == '-' && name.charAt(1) == shortName) {
							return option;
						}
						if (name.length() == 1 && name.charAt(0) == shortName) {
							return option;
						}
					}
				}
				return null;
			}

			static OptionSpec findOption(String name, List<OptionSpec> options) {
				for (final OptionSpec option : options) {
					for (final String prefixed : option.names()) {
						if (prefixed.equals(name) || stripPrefix(prefixed).equals(name)) {
							return option;
						}
					}
				}
				return null;
			}

			/**
			 * Creates and returns a new {@code CommandSpec} initialized from the specified
			 * associated user object. The specified user object must have at least one
			 * {@link Command}, {@link Option} or {@link Parameters} annotation.
			 * 
			 * @param userObject the user object annotated with {@link Command},
			 *                   {@link Option} and/or {@link Parameters} annotations.
			 * @throws InitializationException if the specified object has no picocli
			 *                                 annotations or has invalid annotations
			 */
			public static CommandSpec forAnnotatedObject(Object userObject) {
				return forAnnotatedObject(userObject, new DefaultFactory());
			}

			/**
			 * Creates and returns a new {@code CommandSpec} initialized from the specified
			 * associated user object. The specified user object must have at least one
			 * {@link Command}, {@link Option} or {@link Parameters} annotation.
			 * 
			 * @param userObject the user object annotated with {@link Command},
			 *                   {@link Option} and/or {@link Parameters} annotations.
			 * @param factory    the factory used to create instances of
			 *                   {@linkplain Command#subcommands() subcommands},
			 *                   {@linkplain Option#converter() converters}, etc., that are
			 *                   registered declaratively with annotation attributes
			 * @throws InitializationException if the specified object has no picocli
			 *                                 annotations or has invalid annotations
			 */
			public static CommandSpec forAnnotatedObject(Object userObject, IFactory factory) {
				return CommandReflection.extractCommandSpec(userObject, factory, true);
			}

			/**
			 * Creates and returns a new {@code CommandSpec} initialized from the specified
			 * associated user object. If the specified user object has no {@link Command},
			 * {@link Option} or {@link Parameters} annotations, an empty
			 * {@code CommandSpec} is returned.
			 * 
			 * @param userObject the user object annotated with {@link Command},
			 *                   {@link Option} and/or {@link Parameters} annotations.
			 * @throws InitializationException if the specified object has invalid
			 *                                 annotations
			 */
			public static CommandSpec forAnnotatedObjectLenient(Object userObject) {
				return forAnnotatedObjectLenient(userObject, new DefaultFactory());
			}

			/**
			 * Creates and returns a new {@code CommandSpec} initialized from the specified
			 * associated user object. If the specified user object has no {@link Command},
			 * {@link Option} or {@link Parameters} annotations, an empty
			 * {@code CommandSpec} is returned.
			 * 
			 * @param userObject the user object annotated with {@link Command},
			 *                   {@link Option} and/or {@link Parameters} annotations.
			 * @param factory    the factory used to create instances of
			 *                   {@linkplain Command#subcommands() subcommands},
			 *                   {@linkplain Option#converter() converters}, etc., that are
			 *                   registered declaratively with annotation attributes
			 * @throws InitializationException if the specified object has invalid
			 *                                 annotations
			 */
			public static CommandSpec forAnnotatedObjectLenient(Object userObject, IFactory factory) {
				return CommandReflection.extractCommandSpec(userObject, factory, false);
			}

			private static <T extends Object> int remove(ArgSpec arg, Map<T, OptionSpec> map) {
				int result = 0;
				for (final Iterator<Map.Entry<T, OptionSpec>> iterator = map.entrySet().iterator(); iterator
						.hasNext();) {
					final Map.Entry<?, OptionSpec> entry = iterator.next();
					if (entry.getValue() == arg) {
						iterator.remove();
						result++;
					}
				}
				return result;
			}

			static String stripPrefix(String prefixed) {
				for (int i = 0; i < prefixed.length(); i++) {
					if (Character.isJavaIdentifierPart(prefixed.charAt(i))) {
						return prefixed.substring(i);
					}
				}
				return prefixed;
			}

			/**
			 * Creates and returns a new {@code CommandSpec} with the specified associated
			 * user object. The specified user object is <em>not</em> inspected for
			 * annotations.
			 * 
			 * @param userObject the associated user object. May be any object, may be
			 *                   {@code null}.
			 */
			public static CommandSpec wrapWithoutInspection(Object userObject) {
				return new CommandSpec(CommandUserObject.create(userObject, defaultFactory()));
			}

			/**
			 * Creates and returns a new {@code CommandSpec} with the specified associated
			 * user object. The specified user object is <em>not</em> inspected for
			 * annotations.
			 * 
			 * @param userObject the associated user object. May be any object, may be
			 *                   {@code null}.
			 * @param factory    the factory used to create instances of
			 *                   {@linkplain Command#subcommands() subcommands},
			 *                   {@linkplain Option#converter() converters}, etc., that are
			 *                   registered declaratively with annotation attributes
			 * @since 4.2
			 */
			public static CommandSpec wrapWithoutInspection(Object userObject, IFactory factory) {
				return new CommandSpec(CommandUserObject.create(userObject, factory));
			}

			private final CaseAwareLinkedMap<String, CommandLine> commands = new CaseAwareLinkedMap<String, CommandLine>();
			private final CaseAwareLinkedMap<String, OptionSpec> optionsByNameMap = new CaseAwareLinkedMap<String, OptionSpec>();
			private final CaseAwareLinkedMap<String, OptionSpec> negatedOptionsByNameMap = new CaseAwareLinkedMap<String, OptionSpec>();
			private final CaseAwareLinkedMap<Character, OptionSpec> posixOptionsByKeyMap = new CaseAwareLinkedMap<Character, OptionSpec>();
			private final Map<String, CommandSpec> mixins = new LinkedHashMap<String, CommandSpec>();
			private final Map<String, IAnnotatedElement> mixinAnnotatedElements = new LinkedHashMap<String, IAnnotatedElement>();

			private final List<ArgSpec> requiredArgs = new ArrayList<ArgSpec>();
			private final List<ArgSpec> args = new ArrayList<ArgSpec>();
			private final List<OptionSpec> options = new ArrayList<OptionSpec>();
			private final List<PositionalParamSpec> positionalParameters = new ArrayList<PositionalParamSpec>();
			private final List<UnmatchedArgsBinding> unmatchedArgs = new ArrayList<UnmatchedArgsBinding>();

			private final List<IAnnotatedElement> specElements = new ArrayList<IAnnotatedElement>();
			private final List<IAnnotatedElement> parentCommandElements = new ArrayList<IAnnotatedElement>();
			private final List<ArgGroupSpec> groups = new ArrayList<ArgGroupSpec>();
			private final ParserSpec parser = new ParserSpec();
			private final Interpolator interpolator = new Interpolator(this);
			private final UsageMessageSpec usageMessage = new UsageMessageSpec(interpolator);
			private TypedMember[] methodParams;
			private final CommandUserObject userObject;
			private CommandLine commandLine;
			private CommandSpec parent;
			private Boolean isAddMethodSubcommands;

			private Boolean interpolateVariables;
			private String name;
			private Set<String> aliases = new LinkedHashSet<String>();
			private Boolean isHelpCommand;
			private IVersionProvider versionProvider;

			private IDefaultValueProvider defaultValueProvider;
			private INegatableOptionTransformer negatableOptionTransformer = RegexTransformer.createDefault();

			private Boolean subcommandsRepeatable;

			private String[] version;

			private String toString;

			private boolean inherited = false;

			private ScopeType scopeType = null;

			private Integer exitCodeOnSuccess;

			private Integer exitCodeOnUsageHelp;

			private Integer exitCodeOnVersionHelp;

			private Integer exitCodeOnInvalidInput;

			private Integer exitCodeOnExecutionException;

			private IModelTransformer modelTransformer = null;

			private IParameterPreprocessor preprocessor = new NoOpParameterPreprocessor();

			private CommandSpec(CommandUserObject userObject) {
				this.userObject = userObject;
				this.userObject.commandSpec = this;
			}

			/**
			 * Adds the specified option spec or positional parameter spec to the list of
			 * configured arguments to expect.
			 * 
			 * @param arg the option spec or positional parameter spec to add
			 * @return this CommandSpec for method chaining
			 */
			public CommandSpec add(ArgSpec arg) {
				return arg.isOption() ? addOption((OptionSpec) arg) : addPositional((PositionalParamSpec) arg);
			}

			private void addAlias(String alias, String name, CommandLine subCommandLine, Tracer t) {
				final CommandSpec subSpec = subCommandLine.getCommandSpec();
				if (t.isDebug()) {
					t.debug("Adding alias '%s' for '%s'",
							(subSpec.parent() == null ? "" : subSpec.parent().qualifiedName() + " ") + alias,
							subSpec.qualifiedName());
				}
				final CommandLine previous = commands.put(interpolator.interpolate(alias), subCommandLine);
				if (previous != null && previous != subCommandLine) {
					throw new DuplicateNameException("Alias '" + alias + "' for subcommand '" + name
							+ "' is already used by another subcommand of '" + name() + "'");
				}
			}

			private CommandSpec addArg(ArgSpec arg) {
				args.add(arg);
				arg.messages(usageMessage().messages());
				arg.commandSpec = this;
				if (arg.arity().isUnresolved()) {
					arg.arity = Range.valueOf(interpolator.interpolate(arg.arity().originalValue));
					if (arg.isPositional()) { // #2060 fix bug with late-resolved arity variable
						arg.required = arg.arity.min > 0; // Builder may have set `required` flag before arity was
															// resolved
					}
				}
				// do this last: arg.required() needs to resolve variables in arg.defaultValue()
				if (arg.required() && arg.group() == null && !arg.inherited()) {
					requiredArgs.add(arg);
				}
				return this;
			}

			/**
			 * Adds the specified {@linkplain ArgGroupSpec argument group} to the groups in
			 * this command.
			 * 
			 * @param group the group spec to add
			 * @return this CommandSpec for method chaining
			 * @throws InitializationException if the specified group or one of its
			 *                                 {@linkplain ArgGroupSpec#parentGroup()
			 *                                 ancestors} has already been added
			 * @since 4.0
			 */
			public CommandSpec addArgGroup(ArgGroupSpec group) {
				return addArgGroup(group, new HashSet<OptionSpec>(), new HashSet<PositionalParamSpec>());
			}

			private CommandSpec addArgGroup(ArgGroupSpec group, Set<OptionSpec> groupOptions,
					Set<PositionalParamSpec> groupPositionals) {
				Assert.notNull(group, "group");
				if (group.parentGroup() != null) {
					throw new InitializationException(
							"Groups that are part of another group should not be added to a command. Add only the top-level group.");
				}
				check(group, flatten(groups, new HashSet<ArgGroupSpec>()));
				this.groups.add(group);
				addGroupArgsToCommand(group, new HashMap<String, ArgGroupSpec>(), groupOptions, groupPositionals);
				return this;
			}

			private void addGroupArgsToCommand(ArgGroupSpec group, Map<String, ArgGroupSpec> added,
					Set<OptionSpec> groupOptions, Set<PositionalParamSpec> groupPositionals) {
				final Map<String, OptionSpec> options = new HashMap<String, OptionSpec>();
				for (final ArgSpec arg : group.args()) {
					if (arg.isOption()) {
						final String[] names = interpolator.interpolate(((OptionSpec) arg).names());
						for (final String name : names) {
							final ArgGroupSpec other = added.get(name);
							if (other == null) {
								continue;
							}
							if (other == group) {
								throw DuplicateOptionAnnotationsException.create(name, arg, options.get(name));
							} else {
								throw new DuplicateNameException("An option cannot be in multiple groups but " + name
										+ " is in " + group.synopsisUnit() + " and " + added.get(name).synopsisUnit()
										+ ". Refactor to avoid this. For example, (-a | (-a -b)) can be rewritten as (-a [-b]), and (-a -b | -a -c) can be rewritten as (-a (-b | -c)).");
							}
						}
						for (final String name : names) {
							added.put(name, group);
						}
						for (final String name : names) {
							options.put(name, (OptionSpec) arg);
						}
						groupOptions.add((OptionSpec) arg);
					} else {
						groupPositionals.add((PositionalParamSpec) arg);
					}
					add(arg);
				}
				for (final ArgGroupSpec sub : group.subgroups()) {
					addGroupArgsToCommand(sub, added, groupOptions, groupPositionals);
				}
			}

			/**
			 * Reflects on the class of the {@linkplain #userObject() user object} and
			 * registers any command methods (class methods annotated with {@code @Command})
			 * as subcommands.
			 *
			 * @return this {@link CommandSpec} object for method chaining
			 * @see #addMethodSubcommands(CommandLine.IFactory)
			 * @see #addSubcommand(String, CommandLine)
			 * @since 3.6.0
			 */
			public CommandSpec addMethodSubcommands() {
				return addMethodSubcommands(new DefaultFactory());
			}

			/**
			 * Reflects on the class of the {@linkplain #userObject() user object} and
			 * registers any command methods (class methods annotated with {@code @Command})
			 * as subcommands.
			 * 
			 * @param factory the factory used to create instances of subcommands,
			 *                converters, etc., that are registered declaratively with
			 *                annotation attributes
			 * @return this {@link CommandSpec} object for method chaining
			 * @see #addSubcommand(String, CommandLine)
			 * @since 3.7.0
			 */
			public CommandSpec addMethodSubcommands(IFactory factory) {
				if (userObject.isMethod()) {
					throw new InitializationException(
							"Cannot discover subcommand methods of this Command Method: " + userObject);
				}
				for (final CommandLine sub : createMethodSubcommands(userObject.getType(), factory, true)) {
					addSubcommand(sub.getCommandName(), sub);
				}
				isAddMethodSubcommands = true;
				return this;
			}

			/**
			 * Adds the specified mixin {@code CommandSpec} object to the map of mixins for
			 * this command.
			 * 
			 * @param name  the name that can be used to later retrieve the mixin
			 * @param mixin the mixin whose options and positional parameters and other
			 *              attributes to add to this command
			 * @return this CommandSpec for method chaining
			 */
			public CommandSpec addMixin(String name, CommandSpec mixin) {
				mixins.put(interpolator.interpolate(name), mixin);

				initName(interpolator.interpolateCommandName(mixin.name()));
				initAliases(mixin.aliases()); // should we?
				// TODO
				// initCommandHierarchyWithResourceBundle(mixin.usageMessage().messages().resourceBundleBaseName(),
				// );
				initFrom(mixin);

				for (final Map.Entry<String, CommandLine> entry : mixin.subcommands().entrySet()) {
					addSubcommand(entry.getKey(), entry.getValue());
				}
				final Set<OptionSpec> options = new LinkedHashSet<OptionSpec>(mixin.options());
				final Set<PositionalParamSpec> positionals = new LinkedHashSet<PositionalParamSpec>(
						mixin.positionalParameters());
				for (final ArgGroupSpec argGroupSpec : mixin.argGroups()) {
					final Set<OptionSpec> groupOptions = new HashSet<OptionSpec>();
					final Set<PositionalParamSpec> groupPositionals = new HashSet<PositionalParamSpec>();
					addArgGroup(argGroupSpec, groupOptions, groupPositionals);
					options.removeAll(groupOptions);
					positionals.removeAll(groupPositionals);
				}
				for (final OptionSpec optionSpec : options) {
					addOption(optionSpec);
				}
				for (final PositionalParamSpec paramSpec : positionals) {
					addPositional(paramSpec);
				}
				return this;
			}

			/**
			 * Adds the specified mixin {@code CommandSpec} object to the map of mixins for
			 * this command.
			 * 
			 * @param name             the name that can be used to later retrieve the mixin
			 * @param mixin            the mixin whose options and positional parameters and
			 *                         other attributes to add to this command
			 * @param annotatedElement the `{@literal @}Mixin`-annotated program element
			 * @return this CommandSpec for method chaining
			 * @see #mixinAnnotatedElements()
			 * @since 4.1
			 */
			public CommandSpec addMixin(String name, CommandSpec mixin, IAnnotatedElement annotatedElement) {
				final CommandSpec result = addMixin(name, mixin);
				mixinAnnotatedElements.put(interpolator.interpolate(name), annotatedElement);
				return result;
			}

			/**
			 * Adds the specified option spec to the list of configured arguments to expect.
			 * The option's {@linkplain OptionSpec#description()} may now return Strings
			 * from this CommandSpec's {@linkplain UsageMessageSpec#messages() messages}.
			 * The option parameter's {@linkplain OptionSpec#defaultValueString()} may now
			 * return Strings from this CommandSpec's
			 * {@link CommandSpec#defaultValueProvider()} IDefaultValueProvider}.
			 * 
			 * @param option the option spec to add
			 * @return this CommandSpec for method chaining
			 * @throws DuplicateOptionAnnotationsException if any of the names of the
			 *                                             specified option is the same as
			 *                                             the name of another option
			 */
			public CommandSpec addOption(OptionSpec option) {
				final Tracer tracer = CommandLine.tracer();
				for (final String name : interpolator.interpolate(option.names())) { // cannot be null or empty
					final String existingName = optionsByNameMap.getCaseSensitiveKey(name);
					final OptionSpec existing = optionsByNameMap.put(name, option);
					if (existing != null) { /* was: && !existing.equals(option)) { */
						// since 4.0 ArgGroups: an option cannot be in multiple groups
						throw DuplicateOptionAnnotationsException.create(existingName, option, existing);
					}
					// #1022 checks if negated options exist with the same name
					final String existingNegatedName = negatedOptionsByNameMap.getCaseSensitiveKey(name);
					final OptionSpec existingNegated = negatedOptionsByNameMap.get(name);
					if (existingNegated != null && existingNegated != option) {
						throw DuplicateOptionAnnotationsException.create(existingNegatedName, option, existingNegated);
					}
					if (name.length() == 2 && name.startsWith("-")) {
						posixOptionsByKeyMap.put(name.charAt(1), option);
					}
				}
				options.add(option);
				addOptionNegative(option, tracer);
				if (option.scopeType() == ScopeType.INHERIT) {
					final Set<CommandLine> done = new HashSet<CommandLine>();
					for (final CommandLine sub : subcommands().values()) {
						if (!done.contains(sub)) {
							sub.getCommandSpec().addOption(OptionSpec.builder(option).inherited(true).build());
							done.add(sub);
						}
					}
				}
				return addArg(option);
			}

			private void addOptionNegative(OptionSpec option, Tracer tracer) {
				if (option.negatable()) {
					if (!option.typeInfo().isBoolean() && !option.typeInfo().isOptional()) { // #1108
						throw new InitializationException("Only boolean options can be negatable, but " + option
								+ " is of type " + option.typeInfo().getClassName());
					}
					for (final String name : interpolator.interpolate(option.names())) { // cannot be null or empty
						final String negatedName = negatableOptionTransformer().makeNegative(name, this);
						if (name.equals(negatedName)) {
							tracer.debug("Option %s is negatable, but has no negative form.", name);
						} else {
							tracer.debug("Option %s is negatable, registering negative name %s.", name, negatedName);
							String existingName = negatedOptionsByNameMap.getCaseSensitiveKey(negatedName);
							OptionSpec existing = negatedOptionsByNameMap.put(negatedName, option);
							if (existing == null) {
								existingName = optionsByNameMap.getCaseSensitiveKey(negatedName);
								existing = optionsByNameMap.get(negatedName);
							}
							if (existing != null) {
								throw DuplicateOptionAnnotationsException.create(existingName, option, existing);
							}
						}
					}
				}
			}

			/**
			 * Adds the specified {@code {@literal @}ParentCommand}-annotated program
			 * element to the list of elements for this command.
			 * 
			 * @return this CommandSpec for method chaining
			 * @since 4.0
			 */
			public CommandSpec addParentCommandElement(IAnnotatedElement spec) {
				parentCommandElements.add(spec);
				return this;
			}

			/**
			 * Adds the specified positional parameter spec to the list of configured
			 * arguments to expect. The positional parameter's
			 * {@linkplain PositionalParamSpec#description()} may now return Strings from
			 * this CommandSpec's {@linkplain UsageMessageSpec#messages() messages}. The
			 * positional parameter's {@linkplain PositionalParamSpec#defaultValueString()}
			 * may now return Strings from this CommandSpec's
			 * {@link CommandSpec#defaultValueProvider()} IDefaultValueProvider}.
			 * 
			 * @param positional the positional parameter spec to add
			 * @return this CommandSpec for method chaining
			 */
			public CommandSpec addPositional(PositionalParamSpec positional) {
				positionalParameters.add(positional);
				addArg(positional);
				if (positional.index().isUnresolved()) {
					positional.index = Range.valueOf(interpolator.interpolate(positional.index().originalValue));
					positional.initCapacity();
				}
				adjustRelativeIndices(positional);
				if (positional.scopeType() == ScopeType.INHERIT) {
					final Set<CommandLine> subCmds = new HashSet<CommandLine>(subcommands().values());// subcommands may
																										// be registered
																										// multiple
																										// times with
																										// different
																										// aliases
					for (final CommandLine sub : subCmds) {
						sub.getCommandSpec()
								.addPositional(PositionalParamSpec.builder(positional).inherited(true).build());
					}
				}
				return this;
			}

			/**
			 * Adds the specified {@code {@literal @}Spec}-annotated program element to the
			 * list of elements for this command.
			 * 
			 * @return this CommandSpec for method chaining
			 * @since 4.0
			 */
			public CommandSpec addSpecElement(IAnnotatedElement spec) {
				specElements.add(spec);
				return this;
			}

			/**
			 * Adds the specified subcommand with the specified name. If the specified
			 * subcommand does not have a ResourceBundle set, it is initialized to the
			 * ResourceBundle of this command spec.
			 * 
			 * @param name           subcommand name - the preferred subcommand name to
			 *                       register the subcommand under. If {@code null}, the
			 *                       {@linkplain CommandLine#getCommandName() name} of the
			 *                       specified subcommand is used; if this is also
			 *                       {@code null}, the first
			 *                       {@linkplain CommandSpec#aliases() alias} is used. When
			 *                       this String is encountered in the command line
			 *                       arguments, the subcommand is invoked.
			 * @param subCommandLine the subcommand to envoke when the name is encountered
			 *                       on the command line
			 * @return this {@code CommandSpec} object for method chaining
			 * @throws InitializationException if the specified name is {@code null}, and no
			 *                                 alternative name could be found, or if
			 *                                 another subcommand was already registered
			 *                                 under the same name, or if one of the aliases
			 *                                 of the specified subcommand was already used
			 *                                 by another subcommand.
			 */
			public CommandSpec addSubcommand(String name, CommandLine subCommandLine) {
				final CommandSpec subSpec = subCommandLine.getCommandSpec();
				final String actualName = validateSubcommandName(interpolator.interpolateCommandName(name), subSpec);
				final Tracer t = CommandLine.tracer();
				if (t.isDebug()) {
					t.debug("Adding subcommand '%s' to '%s'", actualName, this.qualifiedName());
				}
				final String previousName = commands.getCaseSensitiveKey(actualName);
				final CommandLine previous = commands.put(actualName, subCommandLine);
				if (previous != null && previous != subCommandLine) {
					throw new DuplicateNameException("Another subcommand named '" + previousName
							+ "' already exists for command '" + this.name() + "'");
				}
				if (subSpec.name == null) {
					subSpec.name(actualName);
				}
				subSpec.parent(this);
				for (final String alias : subSpec.aliases()) {
					addAlias(alias, actualName, subCommandLine, t);
				}
				subSpec.initCommandHierarchyWithResourceBundle(resourceBundleBaseName(), resourceBundle());
				if (scopeType() == ScopeType.INHERIT) {
					subSpec.inheritAttributesFrom(this);
				} else if (this.inherited()) {
					CommandSpec root = this.parent();
					while (root != null && root.scopeType() != ScopeType.INHERIT) {
						root = root.parent();
					}
					if (root == null) {
						throw new InitializationException("Cannot find scope=INHERIT root for " + this);
					}
					subSpec.inheritAttributesFrom(root);
				}

				for (final ArgSpec arg : args()) {
					if (arg.scopeType() == ScopeType.INHERIT) {
						subSpec.add(arg.isOption() ? OptionSpec.builder((OptionSpec) arg).inherited(true).build()
								: PositionalParamSpec.builder((PositionalParamSpec) arg).inherited(true).build());
					}
				}
				return this;
			}

			/**
			 * Adds the specified subcommand with the specified name. If the specified
			 * subcommand does not have a ResourceBundle set, it is initialized to the
			 * ResourceBundle of this command spec.
			 * 
			 * @param name       subcommand name - the preferred subcommand name to register
			 *                   the subcommand under. If {@code null}, the
			 *                   {@linkplain CommandSpec#name() name} of the specified
			 *                   subcommand is used; if this is also {@code null}, the first
			 *                   {@linkplain CommandSpec#aliases() alias} is used. When this
			 *                   String is encountered in the command line arguments, the
			 *                   subcommand is invoked.
			 * @param subcommand describes the subcommand to envoke when the name is
			 *                   encountered on the command line
			 * @return this {@code CommandSpec} object for method chaining
			 * @throws InitializationException if the specified name is {@code null}, and no
			 *                                 alternative name could be found, or if
			 *                                 another subcommand was already registered
			 *                                 under the same name, or if one of the aliases
			 *                                 of the specified subcommand was already used
			 *                                 by another subcommand.
			 */
			public CommandSpec addSubcommand(String name, CommandSpec subcommand) {
				return addSubcommand(name, new CommandLine(subcommand));
			}

			/**
			 * Adds the specified {@code UnmatchedArgsBinding} to the list of model objects
			 * to capture unmatched arguments for this command.
			 * 
			 * @param spec the unmatched arguments binding to capture unmatched arguments
			 * @return this CommandSpec for method chaining
			 */
			public CommandSpec addUnmatchedArgsBinding(UnmatchedArgsBinding spec) {
				unmatchedArgs.add(spec);
				parser().unmatchedArgumentsAllowed(true);
				return this;
			}

			/**
			 * Adjusts the index of the new positional param and all others with relative
			 * indices that were sorted after this positional.
			 * 
			 * @param newlyAdded the newly added positional parameter
			 */
			private void adjustRelativeIndices(PositionalParamSpec newlyAdded) {
				Collections.sort(positionalParameters, new PositionalParametersSorter());
				for (int i = positionalParameters.indexOf(newlyAdded); i < positionalParameters.size(); i++) {
					final PositionalParamSpec adjust = positionalParameters.get(i);
					final Range index = adjust.index();
					if (index.isRelative()) {
						// int min = i == 0 ? 0 : positionalParameters.get(i - 1).index().min();
						final int previousMax = i == 0 ? -1 : positionalParameters.get(i - 1).index().max();
						int max = i == 0 ? 0 : (previousMax == Integer.MAX_VALUE ? previousMax : previousMax + 1); // prevent
																													// overflow
						max = index.isRelativeToAnchor() ? Math.max(max, index.anchor()) : max; // never less than
																								// anchor
						adjust.index = new Range(max, max, index.isVariable(), index.isUnspecified,
								index.originalValue);
						adjust.initCapacity();
					}
				}
			}

			/**
			 * Returns the alias command names of this subcommand.
			 * 
			 * @since 3.1
			 */
			public String[] aliases() {
				return interpolator.interpolate(aliases.toArray(new String[0]));
			}

			/**
			 * Sets the alternative names by which this subcommand is recognized on the
			 * command line.
			 * 
			 * @return this CommandSpec for method chaining
			 * @since 3.1
			 */
			public CommandSpec aliases(String... aliases) {
				final Set<String> previousAliasSet = this.aliases;
				this.aliases = new LinkedHashSet<String>(Arrays.asList(aliases == null ? new String[0] : aliases));
				if (parent != null) {
					// remove & add aliases
					final Set<String> addedAliasSet = new LinkedHashSet<String>(this.aliases);
					addedAliasSet.removeAll(previousAliasSet);
					final Tracer t = CommandLine.tracer();
					for (final String alias : addedAliasSet) {
						parent.addAlias(alias, name, commandLine, t);
					}
					previousAliasSet.removeAll(this.aliases);
					for (final String alias : previousAliasSet) {
						parent.removeAlias(alias, commandLine, t);
					}
				}
				return this;
			}

			/**
			 * Returns the {@linkplain ArgGroupSpec argument groups} in this command.
			 * 
			 * @return an immutable list of groups of options and positional parameters in
			 *         this command
			 * @since 4.0
			 */
			public List<ArgGroupSpec> argGroups() {
				return Collections.unmodifiableList(groups);
			}

			/**
			 * Returns the list of all options and positional parameters configured for this
			 * command.
			 * 
			 * @return an immutable list of all options and positional parameters for this
			 *         command.
			 */
			public List<ArgSpec> args() {
				return Collections.unmodifiableList(args);
			}

			private void check(ArgGroupSpec group, Set<ArgGroupSpec> existing) {
				if (existing.contains(group)) {
					throw new InitializationException("The specified group " + group.synopsisUnit()
							+ " has already been added to the " + qualifiedName() + " command.");
				}
				for (final ArgGroupSpec sub : group.subgroups()) {
					check(sub, existing);
				}
			}

			/** Returns the CommandLine constructed with this {@code CommandSpec} model. */
			public CommandLine commandLine() {
				return commandLine;
			}

			/** Sets the CommandLine constructed with this {@code CommandSpec} model. */
			protected CommandSpec commandLine(CommandLine commandLine) {
				this.commandLine = commandLine;
				for (final CommandSpec mixedInSpec : mixins.values()) {
					mixedInSpec.commandLine(commandLine);
				}
				for (final CommandLine sub : commands.values()) {
					sub.getCommandSpec().parent(this);
				}
				return this;
			}

			Object[] commandMethodParamValues() {
				final Object[] values = new Object[methodParams.length];
				final CommandSpec autoHelpMixin = mixins.get(AutoHelpMixin.KEY);
				int argIndex = autoHelpMixin == null || autoHelpMixin.inherited() ? 0 : 2;
				for (int i = 0; i < methodParams.length; i++) {
					if (methodParams[i].isAnnotationPresent(Mixin.class)) {
						final String name = methodParams[i].getAnnotation(Mixin.class).name();
						final CommandSpec mixin = mixins.get(empty(name) ? methodParams[i].name : name);
						values[i] = mixin.userObject.getInstance();
						argIndex += mixin.args.size();
					} else if (methodParams[i].isAnnotationPresent(ArgGroup.class)) {
						values[i] = null;
						for (final ArgGroupSpec group : groups) {
							if (group.typeInfo.equals(methodParams[i].typeInfo)) {
								values[i] = group.userObjectOr(null);
								argIndex += group.argCount();
								break;
							}
						}
					} else {
						values[i] = args.get(argIndex++).getValue();
					}
				}
				return values;
			}

			private CommandSpec copy() {
				final Object obj = userObject.type == null ? userObject.instance : userObject.type;
				final CommandSpec result = obj == null ? CommandSpec.create()
						: CommandSpec.forAnnotatedObject(obj, userObject.factory);
				result.commandLine = commandLine;
				result.parent = parent;
				result.methodParams = methodParams;
				result.isAddMethodSubcommands = isAddMethodSubcommands;
				result.interpolateVariables = interpolateVariables;
				result.name = name;
				result.aliases = aliases;
				result.isHelpCommand = isHelpCommand;
				result.versionProvider = versionProvider;
				result.modelTransformer = modelTransformer;
				result.defaultValueProvider = defaultValueProvider;
				result.negatableOptionTransformer = negatableOptionTransformer;
				result.subcommandsRepeatable = subcommandsRepeatable;
				result.version = version;
				result.toString = toString;

				result.exitCodeOnSuccess = exitCodeOnSuccess;
				result.exitCodeOnUsageHelp = exitCodeOnUsageHelp;
				result.exitCodeOnVersionHelp = exitCodeOnVersionHelp;
				result.exitCodeOnInvalidInput = exitCodeOnInvalidInput;
				result.exitCodeOnExecutionException = exitCodeOnExecutionException;

				result.usageMessage.initFrom(usageMessage, this);
				result.parser(parser);
				result.inherited = inherited;
				result.scopeType = scopeType;

				// TODO if this CommandSpec was created/modified via the programmatic API,
				// we need to copy all attributes that are modifiable via the programmatic API
				// and point them to this CommandSpec instance.
//                result.commands.clear();                result.commands.putAll(this.commands);
//                result.optionsByNameMap.clear();        result.optionsByNameMap.putAll(this.optionsByNameMap);
//                result.negatedOptionsByNameMap.clear(); result.negatedOptionsByNameMap.putAll(this.negatedOptionsByNameMap);
//                result.posixOptionsByKeyMap.clear();    result.posixOptionsByKeyMap.putAll(this.posixOptionsByKeyMap);
//                result.mixins.clear();                  result.mixins.putAll(this.mixins);
//                result.mixinAnnotatedElements.clear();  result.mixinAnnotatedElements.putAll(this.mixinAnnotatedElements);
//                result.requiredArgs.clear();            result.requiredArgs.addAll(requiredArgs);
//                result.args.clear();                    result.args.addAll(args);
//                result.options.clear();                 result.options.addAll(options);
//                result.positionalParameters.clear();    result.positionalParameters.addAll(positionalParameters);
//                result.unmatchedArgs.clear();           result.unmatchedArgs.addAll(unmatchedArgs);
//                result.specElements.clear();            result.specElements.addAll(specElements);
//                result.parentCommandElements.clear();   result.parentCommandElements.addAll(parentCommandElements);
//                result.groups.clear();                  result.groups.addAll(groups);
				return result;
			}

			/**
			 * Returns the default value provider for this command.
			 * 
			 * @return the default value provider or {@code null}
			 * @since 3.6
			 */
			public IDefaultValueProvider defaultValueProvider() {
				return defaultValueProvider;
			}

			/**
			 * Sets default value provider for this command.
			 * 
			 * @param defaultValueProvider the default value provider to use, or
			 *                             {@code null}.
			 * @return this CommandSpec for method chaining
			 * @since 3.6
			 */
			public CommandSpec defaultValueProvider(IDefaultValueProvider defaultValueProvider) {
				this.defaultValueProvider = defaultValueProvider;
				return this;
			}

			/**
			 * Returns exit code signifying that an exception occurred when invoking the
			 * Runnable, Callable or Method user object of a command.
			 * {@value org.rossonet.ext.picocli.CommandLine.ExitCode#SOFTWARE} by default,
			 * may be set programmatically or via the
			 * {@link Command#exitCodeOnExecutionException() exitCodeOnExecutionException}
			 * annotation.
			 * 
			 * @see #execute(String...)
			 * @since 4.0
			 */
			public int exitCodeOnExecutionException() {
				return exitCodeOnExecutionException == null ? ExitCode.SOFTWARE : exitCodeOnExecutionException;
			}

			/**
			 * Sets exit code signifying that an exception occurred when invoking the
			 * Runnable, Callable or Method user object of a command.
			 * {@value org.rossonet.ext.picocli.CommandLine.ExitCode#SOFTWARE} by default.
			 * 
			 * @see #execute(String...)
			 * @since 4.0
			 */
			public CommandSpec exitCodeOnExecutionException(int newValue) {
				exitCodeOnExecutionException = newValue;
				return this;
			}

			/**
			 * Returns exit code for command line usage error.
			 * {@value org.rossonet.ext.picocli.CommandLine.ExitCode#USAGE} by default, may
			 * be set programmatically or via the {@link Command#exitCodeOnInvalidInput()
			 * exitCodeOnInvalidInput} annotation.
			 * 
			 * @see #execute(String...)
			 * @since 4.0
			 */
			public int exitCodeOnInvalidInput() {
				return exitCodeOnInvalidInput == null ? ExitCode.USAGE : exitCodeOnInvalidInput;
			}

			/**
			 * Sets exit code for command line usage error.
			 * {@value org.rossonet.ext.picocli.CommandLine.ExitCode#USAGE} by default.
			 * 
			 * @see #execute(String...)
			 * @since 4.0
			 */
			public CommandSpec exitCodeOnInvalidInput(int newValue) {
				exitCodeOnInvalidInput = newValue;
				return this;
			}

			/**
			 * Returns exit code for successful termination.
			 * {@value org.rossonet.ext.picocli.CommandLine.ExitCode#OK} by default, may be
			 * set programmatically or via the {@link Command#exitCodeOnSuccess()
			 * exitCodeOnSuccess} annotation.
			 * 
			 * @see #execute(String...)
			 * @since 4.0
			 */
			public int exitCodeOnSuccess() {
				return exitCodeOnSuccess == null ? ExitCode.OK : exitCodeOnSuccess;
			}

			/**
			 * Sets exit code for successful termination.
			 * {@value org.rossonet.ext.picocli.CommandLine.ExitCode#OK} by default.
			 * 
			 * @see #execute(String...)
			 * @since 4.0
			 */
			public CommandSpec exitCodeOnSuccess(int newValue) {
				exitCodeOnSuccess = newValue;
				return this;
			}

			/**
			 * Returns exit code for successful termination after printing usage help on
			 * user request. {@value org.rossonet.ext.picocli.CommandLine.ExitCode#OK} by
			 * default, may be set programmatically or via the
			 * {@link Command#exitCodeOnVersionHelp() exitCodeOnVersionHelp} annotation.
			 * 
			 * @see #execute(String...)
			 * @since 4.0
			 */
			public int exitCodeOnUsageHelp() {
				return exitCodeOnUsageHelp == null ? ExitCode.OK : exitCodeOnUsageHelp;
			}

			/**
			 * Sets exit code for successful termination after printing usage help on user
			 * request. {@value org.rossonet.ext.picocli.CommandLine.ExitCode#OK} by
			 * default.
			 * 
			 * @see #execute(String...)
			 * @since 4.0
			 */
			public CommandSpec exitCodeOnUsageHelp(int newValue) {
				exitCodeOnUsageHelp = newValue;
				return this;
			}

			/**
			 * Returns exit code for successful termination after printing version help on
			 * user request. {@value org.rossonet.ext.picocli.CommandLine.ExitCode#OK} by
			 * default, may be set programmatically or via the
			 * {@link Command#exitCodeOnUsageHelp() exitCodeOnUsageHelp} annotation.
			 * 
			 * @see #execute(String...)
			 * @since 4.0
			 */
			public int exitCodeOnVersionHelp() {
				return exitCodeOnVersionHelp == null ? ExitCode.OK : exitCodeOnVersionHelp;
			}

			/**
			 * Sets exit code for successful termination after printing version help on user
			 * request. {@value org.rossonet.ext.picocli.CommandLine.ExitCode#OK} by
			 * default.
			 * 
			 * @see #execute(String...)
			 * @since 4.0
			 */
			public CommandSpec exitCodeOnVersionHelp(int newValue) {
				exitCodeOnVersionHelp = newValue;
				return this;
			}

			/**
			 * Returns the option with the specified short name, or {@code null} if no
			 * option with that name is defined for this command.
			 */
			public OptionSpec findOption(char shortName) {
				return findOption(shortName, options());
			}

			/**
			 * Returns the option with the specified name, or {@code null} if no option with
			 * that name is defined for this command.
			 * 
			 * @param name used to search the options. May include option name prefix
			 *             characters or not.
			 */
			public OptionSpec findOption(String name) {
				return findOption(name, options());
			}

			List<String> findVisibleOptionNamesWithPrefix(String prefix) {
				final List<String> result = new ArrayList<String>();
				for (final OptionSpec option : options()) {
					for (final String name : option.names()) {
						if (!option.hidden() && stripPrefix(name).startsWith(prefix)) {
							result.add(name);
						}
					}
				}
				return result;
			}

			private Set<ArgGroupSpec> flatten(ArgGroupSpec group, Set<ArgGroupSpec> result) {
				result.add(group);
				for (final ArgGroupSpec sub : group.subgroups()) {
					flatten(sub, result);
				}
				return result;
			}

			private Set<ArgGroupSpec> flatten(Collection<ArgGroupSpec> groups, Set<ArgGroupSpec> result) {
				for (final ArgGroupSpec group : groups) {
					flatten(group, result);
				}
				return result;
			}

			/**
			 * Returns whether this subcommand is a help command, and required options and
			 * positional parameters of the parent command should not be validated.
			 * 
			 * @return {@code true} if this subcommand is a help command and picocli should
			 *         not check for missing required options and positional parameters on
			 *         the parent command
			 * @see Command#helpCommand()
			 */
			public boolean helpCommand() {
				return (isHelpCommand == null) ? DEFAULT_IS_HELP_COMMAND : isHelpCommand;
			}

			/**
			 * Sets whether this is a help command and required parameter checking should be
			 * suspended.
			 * 
			 * @return this CommandSpec for method chaining
			 * @see Command#helpCommand()
			 */
			public CommandSpec helpCommand(boolean newValue) {
				isHelpCommand = newValue;
				return this;
			}

			private void inheritAttributesFrom(CommandSpec root) {
				setInheritedDeep();
				initFrom(root);
				updatedSubcommandsToInheritFrom(root);
			}

			/**
			 * Returns whether this command is inherited from a parent command.
			 * 
			 * @see Command#scope()
			 * @since 4.6
			 */
			public boolean inherited() {
				return inherited;
			}

			void initAliases(String[] aliases) {
				if (aliases != null) {
					this.aliases.addAll(Arrays.asList(aliases));
				}
			}

			private void initCommandHierarchyWithResourceBundle(String bundleBaseName, ResourceBundle rb) {
				if (resourceBundle() == null && resourceBundleBaseName() == null) {
					setBundle(bundleBaseName, rb);
				}
				for (final CommandLine sub : commands.values()) { // percolate down the hierarchy
					sub.getCommandSpec().initCommandHierarchyWithResourceBundle(bundleBaseName, rb);
				}
			}

			void initDefaultValueProvider(Class<? extends IDefaultValueProvider> value, IFactory factory) {
				if (initializable(defaultValueProvider, value, NoDefaultProvider.class)) {
					defaultValueProvider = (DefaultFactory.createDefaultValueProvider(factory, value));
				}
			}

			void initDefaultValueProvider(IDefaultValueProvider value) {
				if (defaultValueProvider == null) {
					defaultValueProvider = value;
				}
			}

			void initExitCodeOnExecutionException(int exitCode) {
				if (initializable(exitCodeOnExecutionException, exitCode, ExitCode.SOFTWARE)) {
					exitCodeOnExecutionException = exitCode;
				}
			}

			void initExitCodeOnInvalidInput(int exitCode) {
				if (initializable(exitCodeOnInvalidInput, exitCode, ExitCode.USAGE)) {
					exitCodeOnInvalidInput = exitCode;
				}
			}

			void initExitCodeOnSuccess(int exitCode) {
				if (initializable(exitCodeOnSuccess, exitCode, ExitCode.OK)) {
					exitCodeOnSuccess = exitCode;
				}
			}

			void initExitCodeOnUsageHelp(int exitCode) {
				if (initializable(exitCodeOnUsageHelp, exitCode, ExitCode.OK)) {
					exitCodeOnUsageHelp = exitCode;
				}
			}

			void initExitCodeOnVersionHelp(int exitCode) {
				if (initializable(exitCodeOnVersionHelp, exitCode, ExitCode.OK)) {
					exitCodeOnVersionHelp = exitCode;
				}
			}

			private void initFrom(CommandSpec spec) {
				initExitCodeOnSuccess(spec.exitCodeOnSuccess());
				initExitCodeOnUsageHelp(spec.exitCodeOnUsageHelp());
				initExitCodeOnVersionHelp(spec.exitCodeOnVersionHelp());
				initExitCodeOnInvalidInput(spec.exitCodeOnInvalidInput());
				initExitCodeOnExecutionException(spec.exitCodeOnExecutionException());

				parser.initSeparator(spec.parser.separator());
				initVersion(spec.version());
				initHelpCommand(spec.helpCommand());
				initVersionProvider(spec.versionProvider());
				initDefaultValueProvider(spec.defaultValueProvider());
				initModelTransformer(spec.modelTransformer());
				initPreprocessor(spec.preprocessor());
				usageMessage.initFromMixin(spec.usageMessage, this);
				initSubcommandsRepeatable(spec.subcommandsRepeatable());
			}

			void initHelpCommand(boolean value) {
				if (initializable(isHelpCommand, value, DEFAULT_IS_HELP_COMMAND)) {
					isHelpCommand = value;
				}
			}

			void initModelTransformer(IModelTransformer value) {
				if (modelTransformer == null) {
					modelTransformer = value;
				}
			}

			void initName(String value) {
				if (initializable(name, value, DEFAULT_COMMAND_NAME)) {
					name = value;
				}
			}

			void initPreprocessor(IParameterPreprocessor value) {
				if (preprocessor instanceof NoOpParameterPreprocessor) {
					preprocessor = Assert.notNull(value, "preprocessor");
				}
			}

			void initSubcommandsRepeatable(boolean value) {
				if (initializable(subcommandsRepeatable, value, DEFAULT_SUBCOMMANDS_REPEATABLE)) {
					subcommandsRepeatable = value;
				}
			}

			void initVersion(String[] value) {
				if (initializable(version, value, UsageMessageSpec.DEFAULT_MULTI_LINE)) {
					version = value.clone();
				}
			}

			void initVersionProvider(IVersionProvider value) {
				if (versionProvider == null) {
					versionProvider = value;
				}
			}

			void injectParentCommand(CommandUserObject commandUserObject) {
				try {
					for (final IAnnotatedElement injectionTarget : parentCommandElements()) {
						injectionTarget.setter().set(commandUserObject.getInstance());
					}
				} catch (final Exception ex) {
					throw new InitializationException("Unable to initialize @ParentCommand field: " + ex, ex);
				}
			}

			/**
			 * Returns whether variables should be interpolated in String values. True by
			 * default.
			 * 
			 * @since 4.0
			 */
			public boolean interpolateVariables() {
				return (interpolateVariables == null) ? DEFAULT_INTERPOLATE_VARIABLES : interpolateVariables;
			}

			/**
			 * Sets whether variables should be interpolated in String values. True by
			 * default.
			 * 
			 * @since 4.0
			 */
			public CommandSpec interpolateVariables(Boolean interpolate) {
				interpolateVariables = interpolate;
				return this;
			}

			/**
			 * Returns whether method commands should be added as subcommands. True by
			 * default. Used by the annotation processor.
			 * 
			 * @since 4.0
			 */
			public boolean isAddMethodSubcommands() {
				return (isAddMethodSubcommands == null) ? DEFAULT_IS_ADD_METHOD_SUBCOMMANDS : isAddMethodSubcommands;
			}

			/**
			 * Returns a map of the mixin names to mixin {@code IAnnotatedElement} objects
			 * for this command.
			 * 
			 * @return an immutable map of `{@literal @}Mixin`-annotated elements added to
			 *         this command.
			 * @see #addMixin(String, CommandLine.Model.CommandSpec,
			 *      CommandLine.Model.IAnnotatedElement)
			 * @since 4.1
			 */
			public Map<String, IAnnotatedElement> mixinAnnotatedElements() {
				return Collections.unmodifiableMap(mixinAnnotatedElements);
			}

			/**
			 * Returns a map of the mixin names to mixin {@code CommandSpec} objects
			 * configured for this command.
			 * 
			 * @return an immutable map of mixins added to this command.
			 */
			public Map<String, CommandSpec> mixins() {
				return Collections.unmodifiableMap(mixins);
			}

			/**
			 * Returns {@code true} if the standard help options have been mixed in with
			 * this command, {@code false} otherwise.
			 */
			public boolean mixinStandardHelpOptions() {
				return mixins.containsKey(AutoHelpMixin.KEY);
			}

			/**
			 * Sets whether the standard help options should be mixed in with this command.
			 * 
			 * @return this CommandSpec for method chaining
			 * @see Command#mixinStandardHelpOptions()
			 */
			public CommandSpec mixinStandardHelpOptions(boolean newValue) {
				if (newValue) {
					final CommandSpec mixin = CommandSpec.forAnnotatedObject(new AutoHelpMixin(), new DefaultFactory());
					boolean overlap = false;
					for (final String key : mixin.optionsMap().keySet()) {
						if (optionsMap().containsKey(key)) {
							overlap = true;
							break;
						}
					}
					if (!overlap) { // #1316, 1319 avoid DuplicateOptionAnnotationsException
						mixin.inherited = this.inherited();
						addMixin(AutoHelpMixin.KEY, mixin);
					}
					// #1331 if inherit(ed) we also add to subcommands
					if (scopeType() == ScopeType.INHERIT || inherited()) {
						for (final CommandLine sub : new HashSet<CommandLine>(subcommands().values())) {
							sub.getCommandSpec().mixinStandardHelpOptions(newValue);
						}
					}
				} else {
					final CommandSpec helpMixin = mixins.remove(AutoHelpMixin.KEY);
					if (helpMixin != null) {
						options.removeAll(helpMixin.options);
						for (final OptionSpec option : helpMixin.options()) {
							for (final String name : interpolator.interpolate(option.names())) {
								optionsByNameMap.remove(name);
								if (name.length() == 2 && name.startsWith("-")) {
									posixOptionsByKeyMap.remove(name.charAt(1));
								}
							}
						}
					}
					// #1331 we don't remove StandardHelpOptions from subcommands, even if they
					// inherit from us
				}
				return this;
			}

			/**
			 * Returns the model transformer for this CommandSpec instance.
			 * 
			 * @since 4.6
			 */
			public IModelTransformer modelTransformer() {
				return modelTransformer;
			}

			/**
			 * Sets the model transformer for this CommandSpec instance.
			 * 
			 * @since 4.6
			 */
			public CommandSpec modelTransformer(IModelTransformer modelTransformer) {
				this.modelTransformer = modelTransformer;
				return this;
			}

			/**
			 * Returns name of this command. Used in the synopsis line of the help message.
			 * {@link #DEFAULT_COMMAND_NAME} by default, initialized from
			 * {@link Command#name()} if defined.
			 * 
			 * @see #qualifiedName()
			 */
			public String name() {
				return interpolator.interpolateCommandName((name == null) ? DEFAULT_COMMAND_NAME : name);
			}

			/**
			 * Sets the String to use as the program name in the synopsis line of the help
			 * message.
			 * 
			 * @return this CommandSpec for method chaining
			 */
			public CommandSpec name(String name) {
				this.name = name;
				return this;
			}

			/**
			 * Returns all names of this command, including {@link #name()} and
			 * {@link #aliases()}.
			 * 
			 * @since 3.9
			 */
			public Set<String> names() {
				final Set<String> result = new LinkedHashSet<String>();
				result.add(name());
				result.addAll(Arrays.asList(aliases()));
				return result;
			}

			/**
			 * Returns the {@code INegatableOptionTransformer} used to create the negative
			 * form of {@linkplain Option#negatable() negatable} options.
			 * 
			 * @see Option#negatable()
			 * @since 4.0
			 */
			public INegatableOptionTransformer negatableOptionTransformer() {
				return negatableOptionTransformer;
			}

			/**
			 * Sets the {@code INegatableOptionTransformer} used to create the negative form
			 * of {@linkplain Option#negatable() negatable} options. Note that
			 * {@link CommandSpec#optionsCaseInsensitive()} will also change the case
			 * sensitivity of {@linkplain Option#negatable() negatable} options: any custom
			 * {@link INegatableOptionTransformer} that was previously installed will be
			 * replaced by the case-insensitive version of the default transformer. To
			 * ensure your custom transformer is used, install it last, after changing case
			 * sensitivity.
			 * 
			 * @see Option#negatable()
			 * @since 4.0
			 */
			public CommandSpec negatableOptionTransformer(INegatableOptionTransformer newValue) {
				final Tracer tracer = CommandLine.tracer();
				tracer.debug("Replacing negatableOptionTransformer %s with %s", negatableOptionTransformer, newValue);
				negatableOptionTransformer = newValue;
				resetNegativeOptionNames();
				return this;
			}

			/**
			 * Returns a map of the negated option names to option spec objects configured
			 * for this command.
			 * 
			 * @return an immutable map of negatable options that this command recognizes.
			 * @since 4.0
			 */
			public Map<String, OptionSpec> negatedOptionsMap() {
				return Collections.unmodifiableMap(negatedOptionsByNameMap);
			}

			/**
			 * Returns the list of options configured for this command.
			 * 
			 * @return an immutable list of options that this command recognizes.
			 */
			public List<OptionSpec> options() {
				return Collections.unmodifiableList(options);
			}

			/**
			 * Returns whether the options are case-insensitive.
			 * 
			 * @since 4.3
			 */
			public boolean optionsCaseInsensitive() {
				return optionsByNameMap.isCaseInsensitive();
			}

			/**
			 * Sets the case-insensitivity of options. Note that changing case sensitivity
			 * will also change the case sensitivity of {@linkplain Option#negatable()
			 * negatable} options: any custom {@link INegatableOptionTransformer} that was
			 * previously installed will be replaced by the case-insensitive version of the
			 * default transformer. To ensure your custom transformer is used, install it
			 * last, after changing case sensitivity.
			 * 
			 * @since 4.3
			 */
			public CommandSpec optionsCaseInsensitive(boolean caseInsensitiveOptions) {
				if (optionsCaseInsensitive() == caseInsensitiveOptions) {
					return this;
				} // no change, no action
				CommandLine.tracer().debug("Changing optionsCaseInsensitive to %s", caseInsensitiveOptions);
				optionsByNameMap.setCaseInsensitive(caseInsensitiveOptions);
				negatedOptionsByNameMap.setCaseInsensitive(caseInsensitiveOptions);
				posixOptionsByKeyMap.setCaseInsensitive(caseInsensitiveOptions);
				final RegexTransformer transformer = caseInsensitiveOptions ? RegexTransformer.createCaseInsensitive()
						: RegexTransformer.createDefault();
				negatableOptionTransformer(transformer);
				return this;
			}

			/**
			 * Returns a map of the option names to option spec objects configured for this
			 * command.
			 * 
			 * @return an immutable map of options that this command recognizes.
			 */
			public Map<String, OptionSpec> optionsMap() {
				return Collections.unmodifiableMap(optionsByNameMap);
			}

			/**
			 * Returns the parent command of this subcommand, or {@code null} if this is a
			 * top-level command.
			 */
			public CommandSpec parent() {
				return parent;
			}

			/**
			 * Sets the parent command of this subcommand.
			 * 
			 * @return this CommandSpec for method chaining
			 */
			public CommandSpec parent(CommandSpec parent) {
				this.parent = parent;
				injectParentCommand(parent.userObject);
				return this;
			}

			/**
			 * Returns the list of program elements annotated with
			 * {@code {@literal @}ParentCommand} configured for this command.
			 * 
			 * @since 4.0
			 */
			public List<IAnnotatedElement> parentCommandElements() {
				return Collections.unmodifiableList(parentCommandElements);
			}

			/** Returns the parser specification for this command. */
			public ParserSpec parser() {
				return parser;
			}

			/**
			 * Initializes the parser specification for this command from the specified
			 * settings and returns this commandSpec.
			 */
			public CommandSpec parser(ParserSpec settings) {
				parser.initFrom(settings);
				return this;
			}

			/**
			 * Returns the list of positional parameters configured for this command.
			 * 
			 * @return an immutable list of positional parameters that this command
			 *         recognizes.
			 */
			public List<PositionalParamSpec> positionalParameters() {
				return Collections.unmodifiableList(positionalParameters);
			}

			/**
			 * Returns a map of the short (single character) option names to option spec
			 * objects configured for this command.
			 * 
			 * @return an immutable map of options that this command recognizes.
			 */
			public Map<Character, OptionSpec> posixOptionsMap() {
				return Collections.unmodifiableMap(posixOptionsByKeyMap);
			}

			/**
			 * Returns the preprocessor for this CommandSpec instance.
			 * 
			 * @since 4.6
			 */
			public IParameterPreprocessor preprocessor() {
				return preprocessor;
			}

			/**
			 * Sets the preprocessor for this CommandSpec instance.
			 * 
			 * @since 4.6
			 */
			public CommandSpec preprocessor(IParameterPreprocessor preprocessor) {
				this.preprocessor = Assert.notNull(preprocessor, "preprocessor");
				return this;
			}

			/**
			 * Returns the String to use as the program name in the synopsis line of the
			 * help message: this command's {@link #name() name}, preceded by the qualified
			 * name of the parent command, if any, separated by a space.
			 * 
			 * @return {@link #DEFAULT_COMMAND_NAME} by default, initialized from
			 *         {@link Command#name()} and the parent command if defined.
			 * @since 3.0.1
			 */
			public String qualifiedName() {
				return qualifiedName(" ");
			}

			/**
			 * Returns this command's fully qualified name, which is its {@link #name()
			 * name}, preceded by the qualified name of the parent command, if this command
			 * has a parent command.
			 * 
			 * @return {@link #DEFAULT_COMMAND_NAME} by default, initialized from
			 *         {@link Command#name()} and the parent command if any.
			 * @param separator the string to put between the names of the commands in the
			 *                  hierarchy
			 * @since 3.6
			 */
			public String qualifiedName(String separator) {
				String result = name();
				if (parent() != null) {
					result = parent().qualifiedName(separator) + separator + result;
				}
				return result;
			}

			/**
			 * (INCUBATING) Removes the specified option spec or positional parameter spec
			 * from the list of configured arguments to expect.
			 * 
			 * @param arg the option spec or positional parameter spec to remove
			 * @return this CommandSpec for method chaining
			 * @throws UnsupportedOperationException if the specified ArgSpec is part of a
			 *                                       {@link ArgGroupSpec}
			 * @throws NoSuchElementException        if the specified ArgSpec is not part of
			 *                                       this {@code CommandSpec}
			 * @since 4.0
			 */
			public CommandSpec remove(ArgSpec arg) {
				if (arg.group() != null) {
					throw new UnsupportedOperationException("Cannot remove ArgSpec that is part of an ArgGroup");
				}
				int removed = remove(arg, optionsByNameMap);
				removed += remove(arg, posixOptionsByKeyMap);
				removed += remove(arg, negatedOptionsByNameMap);
				requiredArgs.remove(arg);
				options.remove(arg);
				if (positionalParameters.remove(arg)) {
					removed++;
				}
				args.remove(arg);
				if (removed == 0) {
					throw new NoSuchElementException(String.valueOf(arg));
				}
				arg.commandSpec = null;
				arg.messages(null);
				return this;
			}

			private void removeAlias(String alias, CommandLine subCommandLine, Tracer t) {
				final CommandSpec subSpec = subCommandLine.getCommandSpec();
				if (t.isDebug()) {
					t.debug("Removing alias '%s' for '%s'",
							(subSpec.parent() == null ? "" : subSpec.parent().qualifiedName() + " ") + alias,
							subSpec.qualifiedName());
				}
				commands.remove(interpolator.interpolate(alias));
			}

			/**
			 * Removes the subcommand with the specified name or alias from this CommandSpec
			 * and returns the {@code CommandLine} instance that was associated with the
			 * specified name, or {@code null} of the specified name was not associated with
			 * a subcommand.
			 * 
			 * @param name name or alias of the subcommand to remove; may be
			 *             {@link #isAbbreviatedSubcommandsAllowed() abbreviated} or
			 *             {@link #isSubcommandsCaseInsensitive() case-insensitive}
			 * @return the removed {@code CommandLine} instance or {@code null}
			 * @since 4.6
			 */
			public CommandLine removeSubcommand(String name) {
				String actualName = name;
				if (parser().abbreviatedSubcommandsAllowed()) {
					actualName = AbbreviationMatcher.match(commands, name, subcommandsCaseInsensitive(), commandLine)
							.getFullName();
				}

				final Set<String> removedNames = new TreeSet<String>();
				final CommandLine result = commands.remove(actualName);
				if (result != null) {
					removedNames.add(actualName);
					commands.remove(result.getCommandName());
					removedNames.add(result.getCommandName());
					for (final String alias : result.getCommandSpec().aliases()) {
						commands.remove(alias);
						removedNames.add(alias);
					}
				}
				final Tracer t = CommandLine.tracer();
				if (t.isDebug()) {
					t.debug("Removed %d subcommand entries %s for key '%s' from '%s'", removedNames.size(),
							removedNames, name, this.qualifiedName());
				}
				return result;
			}

			/**
			 * Returns the list of required options and positional parameters configured for
			 * this command. This does not include options and positional parameters that
			 * are part of a {@linkplain ArgGroupSpec group}.
			 * 
			 * @return an immutable list of the required options and positional parameters
			 *         for this command.
			 */
			public List<ArgSpec> requiredArgs() {
				return Collections.unmodifiableList(requiredArgs);
			}

			boolean resemblesOption(String arg) {
				if (arg == null) {
					return false;
				}
				final Tracer tracer = tracer();
				if (arg.length() == 1) {
					if (tracer != null && tracer.isDebug()) {
						tracer.debug(
								"Single-character arguments that don't match known options are considered positional parameters: %s",
								arg);
					}
					return false;
				}
				try {
					Long.decode(arg);
					return false;
				} catch (final NumberFormatException nan) {
				} // negative numbers are not unknown options
				try {
					Double.parseDouble(arg);
					return false;
				} catch (final NumberFormatException nan) {
				} // negative numbers are not unknown options

				if (options().isEmpty()) {
					final boolean result = arg.startsWith("-");
					if (tracer != null && tracer.isDebug()) {
						tracer.debug("'%s' %s an option", arg, (result ? "resembles" : "doesn't resemble"));
					}
					return result;
				}
				int count = 0;
				for (final String optionName : optionsMap().keySet()) {
					for (int i = 0; i < arg.length(); i++) {
						if (optionName.length() > i && arg.charAt(i) == optionName.charAt(i)) {
							count++;
						} else {
							break;
						}
					}
				}
				final boolean result = count > 0 && count * 10 >= optionsMap().size() * 9; // at least one prefix char
																							// in common with 9 out of
																							// 10 options
				if (tracer != null && tracer.isDebug()) {
					tracer.debug("'%s' %s an option: %d matching prefix chars out of %d option names", arg,
							(result ? "resembles" : "doesn't resemble"), count, optionsMap().size());
				}
				return result;
			}

			private void resetNegativeOptionNames() {
				final Tracer tracer = CommandLine.tracer();
				tracer.debug("Clearing negatedOptionsByNameMap...");
				negatedOptionsByNameMap.clear();
				for (final OptionSpec option : options) {
					addOptionNegative(option, tracer);
				}
			}

			/**
			 * Returns the resource bundle for this command.
			 * 
			 * @return the resource bundle from the {@linkplain UsageMessageSpec#messages()}
			 * @since 3.6
			 */
			public ResourceBundle resourceBundle() {
				return Messages.resourceBundle(usageMessage.messages());
			}

			/**
			 * Initializes the resource bundle for this command: sets the
			 * {@link UsageMessageSpec#messages(CommandLine.Model.Messages)
			 * UsageMessageSpec.messages} to a {@link Messages Messages} object created from
			 * this command spec and the specified bundle, and then sets the
			 * {@link ArgSpec#messages(CommandLine.Model.Messages) ArgSpec.messages} of all
			 * options and positional parameters in this command to the same
			 * {@code Messages} instance. Subcommands are not modified.
			 * 
			 * @param bundle the ResourceBundle to set, may be {@code null}
			 * @return this commandSpec
			 * @see #addSubcommand(String, CommandLine)
			 * @since 3.6
			 */
			public CommandSpec resourceBundle(ResourceBundle bundle) {
				setBundle(Messages.extractName(bundle), bundle);
				return this;
			}

			/**
			 * Returns the resource bundle base name for this command.
			 * 
			 * @return the resource bundle base name from the
			 *         {@linkplain UsageMessageSpec#messages()}
			 * @since 4.0
			 */
			public String resourceBundleBaseName() {
				return Messages.resourceBundleBaseName(usageMessage.messages());
			}

			/**
			 * Initializes the resource bundle for this command: sets the
			 * {@link UsageMessageSpec#messages(CommandLine.Model.Messages)
			 * UsageMessageSpec.messages} to a {@link Messages Messages} object created from
			 * this command spec and the specified bundle, and then sets the
			 * {@link ArgSpec#messages(CommandLine.Model.Messages) ArgSpec.messages} of all
			 * options and positional parameters in this command to the same
			 * {@code Messages} instance. Subcommands are not modified.
			 * <p>
			 * This method is preferable to {@link #resourceBundle(ResourceBundle)} for
			 * pre-Java 8
			 * </p>
			 * 
			 * @param resourceBundleBaseName the base name of the ResourceBundle to set, may
			 *                               be {@code null}
			 * @return this commandSpec
			 * @see #addSubcommand(String, CommandLine)
			 * @since 4.0
			 */
			public CommandSpec resourceBundleBaseName(String resourceBundleBaseName) {
				final ResourceBundle bundle = empty(resourceBundleBaseName) ? null
						: ResourceBundle.getBundle(resourceBundleBaseName);
				setBundle(resourceBundleBaseName, bundle);
				return this;
			}

			/**
			 * Returns the root command: the top-level command of the hierarchy, never
			 * {@code null}.
			 * 
			 * @since 4.3
			 */
			public CommandSpec root() {
				CommandSpec root = this;
				while (root.parent != null) {
					root = root.parent;
				}
				return root;
			}

			/**
			 * Returns the scope of this argument; it it local, or inherited (it applies to
			 * this command as well as all sub- and sub-subcommands).
			 * 
			 * @return whether this argument applies to all descendent subcommands of the
			 *         command where it is defined
			 * @since 4.6
			 */
			public ScopeType scopeType() {
				return scopeType == null ? ScopeType.LOCAL : scopeType;
			}

			/**
			 * Sets the scope of where this argument applies: only this command, or also all
			 * sub (and sub-sub) commands, and returns this builder.
			 * 
			 * @since 4.6
			 */
			public CommandSpec scopeType(ScopeType scopeType) {
				if (this.scopeType == ScopeType.INHERIT && scopeType != ScopeType.INHERIT && !subcommands().isEmpty()) {
					throw new IllegalStateException(
							"Cannot un-inherit: subcommands have already been initialized with values from this command");
				}
				this.scopeType = scopeType;
				if (scopeType == ScopeType.INHERIT) {
					updatedSubcommandsToInheritFrom(this);
				}
				return this;
			}

			/**
			 * Sets whether method commands should be added as subcommands. True by default.
			 * Used by the annotation processor.
			 * 
			 * @since 4.0
			 */
			public CommandSpec setAddMethodSubcommands(Boolean addMethodSubcommands) {
				isAddMethodSubcommands = addMethodSubcommands;
				return this;
			}

			private void setBundle(String bundleBaseName, ResourceBundle bundle) {
				usageMessage().messages(new Messages(this, bundleBaseName, bundle));
				updateArgSpecMessages();
			}

			// Issue #1741: ensure all commands in the hierarchy downwards have `inherited =
			// true` set
			// before mixing in the standard help options
			private void setInheritedDeep() {
				inherited = true;
				for (final CommandLine sub : subcommands().values()) {
					sub.getCommandSpec().setInheritedDeep();
				}
			}

			/**
			 * Returns the list of program elements annotated with {@code {@literal @}Spec}
			 * configured for this command.
			 * 
			 * @since 4.0
			 */
			public List<IAnnotatedElement> specElements() {
				return Collections.unmodifiableList(specElements);
			}

			/** Returns a read-only view of the subcommand map. */
			public Map<String, CommandLine> subcommands() {
				return Collections.unmodifiableMap(commands);
			}

			/**
			 * Returns whether the subcommands are case-insensitive.
			 * 
			 * @since 4.3
			 */
			public boolean subcommandsCaseInsensitive() {
				return commands.isCaseInsensitive();
			}

			/**
			 * Sets the case-insensitivity of subcommands.
			 * 
			 * @since 4.3
			 */
			public CommandSpec subcommandsCaseInsensitive(boolean caseInsensitiveSubcommands) {
				if (subcommandsCaseInsensitive() == caseInsensitiveSubcommands) {
					return this;
				} // no change, no action
				CommandLine.tracer().debug("Changing subcommandsCaseInsensitive to %s", caseInsensitiveSubcommands);
				commands.setCaseInsensitive(caseInsensitiveSubcommands);
				return this;
			}

			/**
			 * Returns whether the subcommands of this command are repeatable, that is,
			 * whether such subcommands can occur multiple times and may be followed by
			 * sibling commands instead of just child commands.
			 * 
			 * @see Command#subcommandsRepeatable()
			 * @since 4.2
			 */
			public boolean subcommandsRepeatable() {
				return (subcommandsRepeatable == null) ? DEFAULT_SUBCOMMANDS_REPEATABLE : subcommandsRepeatable;
			}

			/**
			 * Sets whether the subcommands of this command are repeatable, that is, whether
			 * such subcommands can occur multiple times and may be followed by sibling
			 * commands instead of just child commands.
			 * 
			 * @see Command#subcommandsRepeatable()
			 * @since 4.2
			 */
			public CommandSpec subcommandsRepeatable(boolean subcommandsRepeatable) {
				this.subcommandsRepeatable = subcommandsRepeatable;
				return this;
			}

			/**
			 * Returns a string representation of this command, used in error messages and
			 * trace messages.
			 */
			@Override
			public String toString() {
				return toString == null ? "command '" + name + "' (user object: " + userObject + ")" : toString;
			}

			/**
			 * Returns the list of {@link UnmatchedArgsBinding UnmatchedArgumentsBindings}
			 * configured for this command; each {@code UnmatchedArgsBinding} captures the
			 * arguments that could not be matched to any options or positional parameters.
			 */
			public List<UnmatchedArgsBinding> unmatchedArgsBindings() {
				return Collections.unmodifiableList(unmatchedArgs);
			}

			void updateAddMethodSubcommands(boolean value) {
				if (isNonDefault(value, DEFAULT_IS_ADD_METHOD_SUBCOMMANDS)) {
					isAddMethodSubcommands = value;
				}
			}

			private void updateArgSpecMessages() {
				for (final OptionSpec opt : options()) {
					opt.messages(usageMessage().messages());
				}
				for (final PositionalParamSpec pos : positionalParameters()) {
					pos.messages(usageMessage().messages());
				}
				for (final ArgGroupSpec group : argGroups()) {
					group.messages(usageMessage().messages());
				}
			}

			/**
			 * Updates the following attributes from the specified {@code @Command}
			 * annotation: aliases, {@link ParserSpec#separator() parser separator}, command
			 * name, version, help command, version provider, default provider and
			 * {@link UsageMessageSpec usage message spec}.
			 * 
			 * @param cmd     the {@code @Command} annotation to get attribute values from
			 * @param factory factory used to instantiate classes
			 * @since 3.7
			 */
			public void updateCommandAttributes(Command cmd, IFactory factory) {
				parser().updateSeparator(interpolator.interpolate(cmd.separator()));

				updateExitCodeOnSuccess(cmd.exitCodeOnSuccess());
				updateExitCodeOnUsageHelp(cmd.exitCodeOnUsageHelp());
				updateExitCodeOnVersionHelp(cmd.exitCodeOnVersionHelp());
				updateExitCodeOnInvalidInput(cmd.exitCodeOnInvalidInput());
				updateExitCodeOnExecutionException(cmd.exitCodeOnExecutionException());

				aliases(cmd.aliases());
				updateName(cmd.name());
				updateVersion(cmd.version());
				updateHelpCommand(cmd.helpCommand());
				updateSubcommandsRepeatable(cmd.subcommandsRepeatable());
				updateAddMethodSubcommands(cmd.addMethodSubcommands());
				usageMessage().updateFromCommand(cmd, this, factory != null);

				updateScopeType(cmd.scope());

				if (factory != null) {
					updateModelTransformer(cmd.modelTransformer(), factory);
					updateVersionProvider(cmd.versionProvider(), factory);
					initDefaultValueProvider(cmd.defaultValueProvider(), factory);
					updatePreprocessor(cmd.preprocessor(), factory);
				}
			}

			private void updatedSubcommandsToInheritFrom(CommandSpec root) {
				if (root != this && root.mixinStandardHelpOptions()) { // #1331 only add, don't remove
					mixinStandardHelpOptions(true);
				}
				final Set<CommandLine> subcommands = new HashSet<CommandLine>(subcommands().values());
				for (final CommandLine sub : subcommands) {
					sub.getCommandSpec().inheritAttributesFrom(root);
				}
			}

			void updateExitCodeOnExecutionException(int exitCode) {
				if (isNonDefault(exitCode, ExitCode.SOFTWARE)) {
					exitCodeOnExecutionException = exitCode;
				}
			}

			void updateExitCodeOnInvalidInput(int exitCode) {
				if (isNonDefault(exitCode, ExitCode.USAGE)) {
					exitCodeOnInvalidInput = exitCode;
				}
			}

			void updateExitCodeOnSuccess(int exitCode) {
				if (isNonDefault(exitCode, ExitCode.OK)) {
					exitCodeOnSuccess = exitCode;
				}
			}

			void updateExitCodeOnUsageHelp(int exitCode) {
				if (isNonDefault(exitCode, ExitCode.OK)) {
					exitCodeOnUsageHelp = exitCode;
				}
			}

			void updateExitCodeOnVersionHelp(int exitCode) {
				if (isNonDefault(exitCode, ExitCode.OK)) {
					exitCodeOnVersionHelp = exitCode;
				}
			}

			void updateHelpCommand(boolean value) {
				if (isNonDefault(value, DEFAULT_IS_HELP_COMMAND)) {
					isHelpCommand = value;
				}
			}

			void updateModelTransformer(Class<? extends IModelTransformer> value, IFactory factory) {
				if (isNonDefault(value, NoOpModelTransformer.class)) {
					this.modelTransformer = DefaultFactory.create(factory, value);
				}
			}

			void updateName(String value) {
				if (isNonDefault(value, DEFAULT_COMMAND_NAME)) {
					name = value;
				}
			}

			void updatePreprocessor(Class<? extends IParameterPreprocessor> value, IFactory factory) {
				if (isNonDefault(value, NoOpParameterPreprocessor.class)) {
					this.preprocessor = DefaultFactory.create(factory, value);
				}
			}

			void updateScopeType(ScopeType scopeType) {
				if (this.scopeType == null) {
					scopeType(scopeType);
				}
			}

			void updateSubcommandsRepeatable(boolean value) {
				if (isNonDefault(value, DEFAULT_SUBCOMMANDS_REPEATABLE)) {
					subcommandsRepeatable = value;
				}
			}

			void updateVersion(String[] value) {
				if (isNonDefault(value, UsageMessageSpec.DEFAULT_MULTI_LINE)) {
					version = value.clone();
				}
			}

			void updateVersionProvider(Class<? extends IVersionProvider> value, IFactory factory) {
				if (isNonDefault(value, NoVersionProvider.class)) {
					versionProvider = (DefaultFactory.createVersionProvider(factory, value));
				}
			}

			/** Returns the usage help message specification for this command. */
			public UsageMessageSpec usageMessage() {
				return usageMessage;
			}

			/**
			 * Initializes the usageMessage specification for this command from the
			 * specified settings and returns this commandSpec.
			 */
			public CommandSpec usageMessage(UsageMessageSpec settings) {
				usageMessage.initFrom(settings, this);
				return this;
			}

			/**
			 * Returns the user object associated with this command.
			 * 
			 * @see CommandLine#getCommand()
			 */
			public Object userObject() {
				return userObject.getInstance();
			}

			/**
			 * Ensures all attributes of this {@code CommandSpec} have a valid value; throws
			 * an {@link InitializationException} if this cannot be achieved.
			 */
			void validate() {
				Collections.sort(positionalParameters, new PositionalParametersSorter());
				validatePositionalParameters(positionalParameters);
				final List<String> wrongUsageHelpAttr = new ArrayList<String>();
				final List<String> wrongVersionHelpAttr = new ArrayList<String>();
				final List<String> usageHelpAttr = new ArrayList<String>();
				final List<String> versionHelpAttr = new ArrayList<String>();
				for (final OptionSpec option : options()) {
					if (option.usageHelp()) {
						usageHelpAttr.add(option.longestName());
						if (!isBoolean(option.type())) {
							wrongUsageHelpAttr.add(option.longestName());
						}
					}
					if (option.versionHelp()) {
						versionHelpAttr.add(option.longestName());
						if (!isBoolean(option.type())) {
							wrongVersionHelpAttr.add(option.longestName());
						}
					}
				}
				final String wrongType = "Non-boolean options like %s should not be marked as '%s=true'. Usually a command has one %s boolean flag that triggers display of the %s. Alternatively, consider using @Command(mixinStandardHelpOptions = true) on your command instead.";
				final String multiple = "Multiple options %s are marked as '%s=true'. Usually a command has only one %s option that triggers display of the %s. Alternatively, consider using @Command(mixinStandardHelpOptions = true) on your command instead.";
				if (!wrongUsageHelpAttr.isEmpty()) {
					throw new InitializationException(
							String.format(wrongType, wrongUsageHelpAttr, "usageHelp", "--help", "usage help message"));
				}
				if (!wrongVersionHelpAttr.isEmpty()) {
					throw new InitializationException(String.format(wrongType, wrongVersionHelpAttr, "versionHelp",
							"--version", "version information"));
				}
				if (usageHelpAttr.size() > 1) {
					CommandLine.tracer().warn(multiple, usageHelpAttr, "usageHelp", "--help", "usage help message");
				}
				if (versionHelpAttr.size() > 1) {
					CommandLine.tracer().warn(multiple, versionHelpAttr, "versionHelp", "--version",
							"version information");
				}
			}

			private String validateSubcommandName(String name, CommandSpec subSpec) {
				if (name != null) {
					return name;
				}
				if (subSpec.name != null) { // NOTE: check subSpec.name field, not subSpec.name()!
					return subSpec.name;
				}
				if (!subSpec.aliases.isEmpty()) {
					final Iterator<String> iter = subSpec.aliases.iterator();
					final String result = iter.next();
					iter.remove();
					if (result != null) {
						return result;
					}
				}
				throw new InitializationException("Cannot add subcommand with null name to " + qualifiedName());
			}

			/**
			 * Returns version information for this command, to print to the console when
			 * the user specifies an {@linkplain OptionSpec#versionHelp() option} to request
			 * version help. This is not part of the usage help message.
			 * 
			 * @return the version strings generated by the {@link #versionProvider()
			 *         version provider} if one is set, otherwise the
			 *         {@linkplain #version(String...) version literals}
			 */
			public String[] version() {
				if (versionProvider != null) {
					try {
						return interpolator.interpolate(versionProvider.getVersion());
					} catch (final Exception ex) {
						final String msg = "Could not get version info from " + versionProvider + ": " + ex;
						throw new ExecutionException(this.commandLine, msg, ex);
					}
				}
				return interpolator.interpolate(version == null ? UsageMessageSpec.DEFAULT_MULTI_LINE : version);
			}

			/**
			 * Sets version information literals for this command, to print to the console
			 * when the user specifies an {@linkplain OptionSpec#versionHelp() option} to
			 * request version help. Only used if no {@link #versionProvider()
			 * versionProvider} is set.
			 * 
			 * @return this CommandSpec for method chaining
			 */
			public CommandSpec version(String... version) {
				this.version = version;
				return this;
			}

			/**
			 * Returns the version provider for this command, to generate the
			 * {@link #version()} strings.
			 * 
			 * @return the version provider or {@code null} if the version strings should be
			 *         returned from the {@linkplain #version(String...) version literals}.
			 */
			public IVersionProvider versionProvider() {
				return versionProvider;
			}

			/**
			 * Sets version provider for this command, to generate the {@link #version()}
			 * strings.
			 * 
			 * @param versionProvider the version provider to use to generate the version
			 *                        strings, or {@code null} if the
			 *                        {@linkplain #version(String...) version literals}
			 *                        should be used.
			 * @return this CommandSpec for method chaining
			 */
			public CommandSpec versionProvider(IVersionProvider versionProvider) {
				this.versionProvider = versionProvider;
				return this;
			}

			/**
			 * Sets the string representation of this command, used in error messages and
			 * trace messages.
			 * 
			 * @param newValue the string representation
			 * @return this CommandSpec for method chaining
			 */
			public CommandSpec withToString(String newValue) {
				this.toString = newValue;
				return this;
			}
		}

		static class CommandUserObject implements IScope {
			public static CommandUserObject create(Object userObject, IFactory factory) {
				if (userObject instanceof CommandUserObject) {
					return (CommandUserObject) userObject;
				}
				return new CommandUserObject(userObject, factory);
			}

			private final IFactory factory;
			private Object instance;
			private Class<?> type;

			private CommandSpec commandSpec; // initialized in CommandSpec constructor

			private CommandUserObject(Object objectOrClass, IFactory factory) {
				this.factory = Assert.notNull(factory, "factory");
				if (objectOrClass instanceof Class) {
					instance = null;
					type = (Class<?>) objectOrClass;
				} else {
					instance = objectOrClass;
					type = objectOrClass == null || objectOrClass instanceof Method // don't mix in options/positional
																					// params from outer class @Command
							? null
							: objectOrClass.getClass();
				}
			}

			public CommandUserObject copy() {
				// type is only null for @Command-annotated Methods
				return new CommandUserObject(type == null ? instance : type, factory);
			}

			@Override
			@SuppressWarnings("unchecked")
			public <T> T get() {
				return (T) getInstance();
			}

			public Object getInstance() {
				if (instance == null) {
					final Tracer t = CommandLine.tracer();
					if (type == null) {
						t.debug("Returning a null user object instance");
						return null;
					}
					try {
						t.debug("Getting a %s instance from factory %s", type.getName(), factory);
						instance = DefaultFactory.create(factory, type);
						type = instance.getClass(); // potentially change interface name to impl type name
						t.debug("Factory returned a %s instance (%s)", type.getName(),
								Integer.toHexString(System.identityHashCode(instance)));
					} catch (final InitializationException ex) {
						if (type.isInterface()) {
							t.debug("%s. Creating Proxy for interface %s", ex.getCause(), type.getName());
							instance = Proxy.newProxyInstance(type.getClassLoader(), new Class<?>[] { type },
									new PicocliInvocationHandler());
							t.debug("Created Proxy instance (%s)",
									Integer.toHexString(System.identityHashCode(instance)));
						} else {
							throw ex;
						}
					}
					if (commandSpec != null) { // now initialize initial values: this CUO _is_ the IScope for the
												// command's args
						for (final ArgSpec arg : commandSpec.args()) {
							if (arg.group() == null && !arg.hasInitialValue()) {
								arg.initialValue(); // initializes isInitialValueCached
							}
						}
					}
				}
				return instance;
			}

			public Class<?> getType() {
				return type;
			}

			public boolean isMethod() {
				return instance instanceof Method;
			}

			public boolean isProxyClass() {
				if (type == null || !type.isInterface()) {
					return false;
				}
				return Proxy.isProxyClass(getInstance().getClass());
			}

			@Override
			public <T> T set(T value) {
				throw new UnsupportedOperationException();
			}

			@Override
			public String toString() {
				if (type == null && instance == null) {
					return "null";
				}
				if (type == null) {
					return String.valueOf(instance);
				}
				if (instance == null) {
					return String.valueOf(type);
				}
				return type.getName() + "@" + Integer.toHexString(System.identityHashCode(instance));
			}
		}

		static class FieldBinding implements IGetter, ISetter, IScoped {
			private final IScope scope;
			private final Field field;

			FieldBinding(IScope scope, Field field) {
				this.scope = scope;
				this.field = field;
			}

			FieldBinding(Object scope, Field field) {
				this(ObjectScope.asScope(scope), field);
			}

			@Override
			public <T> T get() throws PicocliException {
				Object obj;
				try {
					obj = scope.get();
				} catch (final Exception ex) {
					throw new PicocliException("Could not get scope for field " + field, ex);
				}
				try {
					@SuppressWarnings("unchecked")
					final T result = (T) field.get(obj);
					return result;
				} catch (final Exception ex) {
					throw new PicocliException("Could not get value for field " + field, ex);
				}
			}

			@Override
			public IScope getScope() {
				return scope;
			}

			@Override
			public <T> T set(T value) throws PicocliException {
				Object obj;
				try {
					obj = scope.get();
				} catch (final Exception ex) {
					throw new PicocliException("Could not get scope for field " + field, ex);
				}
				try {
					@SuppressWarnings("unchecked")
					final T result = (T) field.get(obj);
					field.set(obj, value);
					return result;
				} catch (final Exception ex) {
					throw new PicocliException("Could not set value for field " + field + " to " + value, ex);
				}
			}

			@Override
			public String toString() {
				return String.format("%s(%s %s.%s)", getClass().getSimpleName(), field.getType().getName(),
						field.getDeclaringClass().getName(), field.getName());
			}
		}

		/**
		 * Internal interface to allow annotation processors to construct a command
		 * model at compile time.
		 * 
		 * @since 4.0
		 */
		public interface IAnnotatedElement {
			<T extends Annotation> T getAnnotation(Class<T> annotationClass);

			int getMethodParamPosition();

			String getMixinName();

			String getName();

			CommandLine.Model.IGetter getter();

			String getToString();

			ITypeInfo getTypeInfo();

			boolean hasInitialValue();

			boolean isAnnotationPresent(Class<? extends Annotation> annotationClass);

			boolean isArgGroup();

			boolean isArgSpec();

			boolean isInteractive();

			boolean isMethodParameter();

			boolean isMixin();

			boolean isMultiValue();

			boolean isOption();

			boolean isParameter();

			boolean isParentCommand();

			boolean isSpec();

			boolean isUnmatched();

			CommandLine.Model.IScope scope();

			CommandLine.Model.ISetter setter();

			Object userObject();
		}

		/**
		 * Interface to allow extending the capabilities of other interface without Java
		 * 8 default methods.
		 * <p>
		 * Example usage:
		 * </p>
		 * 
		 * <pre>
		 * // suppose we want to add a method `getInitialValueState` to `IAnnotatedElement`
		 * IAnnotatedElement element = getAnnotatedElement();
		 * if (element instanceof IExtensible) {
		 * 	InitialValueState state = ((IExtensible) element).getExtension(InitialValueState.class);
		 * 	if (state != null) {
		 * 		// ...
		 * 	}
		 * }
		 * </pre>
		 * 
		 * @since 4.3
		 */
		public interface IExtensible {
			/**
			 * Returns an instance of the specified class, or {@code null} if this extension
			 * is not supported.
			 * 
			 * @param cls class of the desired extension
			 */
			<T> T getExtension(Class<T> cls);
		}

		/**
		 * Customizable getter for obtaining the current value of an option or
		 * positional parameter. When an option or positional parameter is matched on
		 * the command line, its getter or setter is invoked to capture the value. For
		 * example, an option can be bound to a field or a method, and when the option
		 * is matched on the command line, the field's value is set or the method is
		 * invoked with the option parameter value.
		 * 
		 * @since 3.0
		 */
		public interface IGetter {
			/**
			 * Returns the current value of the binding. For multi-value options and
			 * positional parameters, this method returns an array, collection or map to add
			 * values to.
			 * 
			 * @throws PicocliException if a problem occurred while obtaining the current
			 *                          value
			 * @throws Exception        internally, picocli call sites will catch any
			 *                          exceptions thrown from here and rethrow them wrapped
			 *                          in a PicocliException
			 */
			<T> T get() throws Exception;
		}

		enum InitialValueState {
			CACHED, POSTPONED, UNAVAILABLE
		}

		static class Interpolator {
			interface ILookup {
				String get(String key);
			}

			private static String bundleValue(ResourceBundle rb, String key) {
				if (rb != null) {
					try {
						return rb.getString(key);
					} catch (final MissingResourceException ex) {
						return null;
					}
				}
				return null;
			}

			private final CommandSpec commandSpec;
			private final Map<String, ILookup> lookups = new LinkedHashMap<String, ILookup>();

			public Interpolator(final CommandSpec commandSpec) {
				this.commandSpec = commandSpec;
				lookups.put("sys:", new ILookup() {
					@Override
					public String get(String key) {
						return System.getProperty(key);
					}
				});
				lookups.put("env:", new ILookup() {
					@Override
					public String get(String key) {
						return System.getenv(key);
					}
				});
				lookups.put("bundle:", new ILookup() {
					@Override
					public String get(String key) {
						// commandSpec.usageMessage().messages().
						return bundleValue(commandSpec.resourceBundle(), key);
					}
				});
				lookups.put("", new ILookup() {
					@Override
					public String get(String key) {
						String result = "COMMAND-NAME".equals(key) ? commandSpec.name()
								: "COMMAND-FULL-NAME".equals(key) ? commandSpec.qualifiedName()
										: "PARENT-COMMAND-NAME".equals(key) && commandSpec.parent() != null
												? commandSpec.parent().name()
												: "PARENT-COMMAND-FULL-NAME".equals(key) && commandSpec.parent() != null
														? commandSpec.parent().qualifiedName()
														: "ROOT-COMMAND-NAME".equals(key) ? commandSpec.root().name()
																: null;
						if (result == null) {
							result = System.getProperty(key);
						}
						if (result == null) {
							result = System.getenv(key);
						}
						if (result == null) {
							result = bundleValue(commandSpec.resourceBundle(), key);
						}
						return result;
					}
				});
			}

			private int findClosingBrace(String text, int start) {
				int open = 1;
				boolean escaping = false;
				for (int ch, i = start; i < text.length(); i += Character.charCount(ch)) {
					ch = text.codePointAt(i);
					switch (ch) {
					case '\\':
						escaping = !escaping;
						break;
					case '}':
						if (!escaping) {
							open--;
						}
						if (open == 0) {
							return i;
						}
						escaping = false;
						break;
					case '{':
						if (!escaping) {
							open++;
						}
						escaping = false;
						break;
					default:
						escaping = false;
						break;
					}
				}
				return -1;
			}

			private int findOpeningDollar(String text, String prefix, int start) {
				int open = -1;
				boolean escaping = false;
				for (int ch, i = start; i < text.length(); i += Character.charCount(ch)) {
					ch = text.codePointAt(i);
					if (ch == '$') {
						open = escaping ? -1 : i;
						escaping = !escaping;
					} else {
						escaping = false;
					}
					if (open != -1 && ch != prefix.codePointAt(i - open)) {
						open = -1;
					}
					if (open != -1 && i - open == prefix.length() - 1) {
						return open;
					}
				}
				return -1;
			}

			public String interpolate(String original) {
				if (original == null || !commandSpec.interpolateVariables()) {
					return original;
				}
				return resolveLookups(original, new HashSet<String>(), new HashMap<String, String>());
			}

			public String[] interpolate(String[] values) {
				if (values == null || values.length == 0) {
					return values;
				}
				final String[] result = new String[values.length];
				for (int i = 0; i < result.length; i++) {
					result[i] = interpolate(values[i]);
				}
				return result;
			}

			public String interpolateCommandName(String original) {
				if (original == null || !commandSpec.interpolateVariables()) {
					return original;
				}
				return resolveLookups(original, new HashSet<String>(), new HashMap<String, String>());
			}

			private String resolveLookups(String text, Set<String> visited, Map<String, String> resolved) {
				if (text == null) {
					return null;
				}
				for (final String lookupKey : lookups.keySet()) {
					final ILookup lookup = lookups.get(lookupKey);
					final String prefix = "${" + lookupKey;
					int startPos = 0;
					while ((startPos = findOpeningDollar(text, prefix, startPos)) >= 0) {
						int endPos = findClosingBrace(text, startPos + prefix.length());
						if (endPos < 0) {
							endPos = text.length() - 1;
						}
						final String fullKey = text.substring(startPos + prefix.length(), endPos);
						String actualKey = fullKey;

						final int defaultStartPos = fullKey.indexOf(":-");
						if (defaultStartPos >= 0) {
							actualKey = fullKey.substring(0, defaultStartPos);
						}
						String value = resolved.containsKey(prefix + actualKey) ? resolved.get(prefix + actualKey)
								: lookup.get(actualKey);
						if (visited.contains(prefix + actualKey) && !resolved.containsKey(prefix + actualKey)) {
							throw new InitializationException(
									"Lookup '" + prefix + actualKey + "' has a circular reference.");
						}
						visited.add(prefix + actualKey);
						if (value == null && defaultStartPos >= 0) {
							final String defaultValue = fullKey.substring(defaultStartPos + 2);
							value = resolveLookups(defaultValue, visited, resolved);
						}
						resolved.put(prefix + actualKey, value);
						if (value == null && startPos == 0 && endPos == text.length() - 1) {
							return null; // #676 x="${var}" should resolve to x=null if not found (not x="null")
						}

						// interpolate
						text = text.substring(0, startPos) + value + text.substring(endPos + 1);
						startPos += value == null ? "null".length() : value.length();
					}
				}
				return text.replace("$$", "$");
			}
		}

		/**
		 * Interface for sorting {@link OptionSpec options} and {@link ArgGroupSpec
		 * groups} together.
		 * 
		 * @since 4.0
		 */
		public interface IOrdered {
			/**
			 * Returns the position in the options list in the usage help message at which
			 * this element should be shown. Elements with a lower number are shown before
			 * elements with a higher number. For options, this attribute is only honored if
			 * {@link UsageMessageSpec#sortOptions()} is {@code false} for this command. For
			 * argument groups, this attribute is only honored for groups that have a
			 * {@link ArgGroup#heading() heading} (or a {@link ArgGroup#headingKey()
			 * headingKey} with a non-{@code null} value).
			 */
			int order();
		}

		/**
		 * The scope of a getter/setter binding is the context where the current value
		 * should be gotten from or set to. Usually, this is an instance of the
		 * enclosing element. For a field, the scope is the object whose field value to
		 * get/set. For a method binding, it is the object on which the method should be
		 * invoked.
		 * <p>
		 * The getter and setter of the scope allow you to change the object onto which
		 * the option and positional parameter getters and setters should be applied.
		 * </p>
		 * 
		 * @since 4.0
		 */
		public interface IScope extends IGetter, ISetter {
		}

		/**
		 * This interface provides access to an {@link IScope} instance.
		 * 
		 * @since 4.7
		 */
		public interface IScoped {
			/**
			 * Get the {@link IScope} instance.
			 *
			 * @return {@link IScope} instance
			 */
			IScope getScope();
		}

		/**
		 * Customizable setter for modifying the value of an option or positional
		 * parameter. When an option or positional parameter is matched on the command
		 * line, its setter is invoked to capture the value. For example, an option can
		 * be bound to a field or a method, and when the option is matched on the
		 * command line, the field's value is set or the method is invoked with the
		 * option parameter value.
		 * 
		 * @since 3.0
		 */
		public interface ISetter {
			/**
			 * Sets the new value of the option or positional parameter.
			 *
			 * @param value the new value of the option or positional parameter
			 * @param <T>   type of the value
			 * @return the previous value of the binding (if supported by this binding)
			 * @throws PicocliException if a problem occurred while setting the new value
			 * @throws Exception        internally, picocli call sites will catch any
			 *                          exceptions thrown from here and rethrow them wrapped
			 *                          in a PicocliException
			 */
			<T> T set(T value) throws Exception;
		}

		/**
		 * Encapculates type information for an option or parameter to make this
		 * information available both at runtime and at compile time (when {@code Class}
		 * values are not available). Most of the methods in this interface (but not
		 * all!) are safe to use by annotation processors.
		 * 
		 * @since 4.0
		 */
		public interface ITypeInfo {
			/**
			 * Returns the names of the type arguments if this is a generic type. For
			 * example, returns {@code ["java.lang.String"]} if this type is
			 * {@code List<String>}.
			 */
			List<String> getActualGenericTypeArguments();

			/**
			 * Returns type information of components or elements of a
			 * {@link #isMultiValue() multivalue} type.
			 */
			List<ITypeInfo> getAuxiliaryTypeInfos();

			/**
			 * Returns the component class of an array, or the parameter type of a generic
			 * Collection, or the parameter types of the key and the value of a generic Map.
			 * This method is <em>not</em> safe for annotation processors to use.
			 * 
			 * @return the component type or types of an array, Collection or Map type
			 */
			Class<?>[] getAuxiliaryTypes();

			String getClassName();

			String getClassSimpleName();

			List<String> getEnumConstantNames();

			/**
			 * Returns the class that the option or parameter value should be converted to
			 * when matched on the command line. This method is <em>not</em> safe for
			 * annotation processors to use.
			 * 
			 * @return the class that the option or parameter value should be converted to
			 */
			Class<?> getType();

			/**
			 * Returns {@code true} if this type is an array multi-value type. Note that
			 * from picocli 4.7, {@code char[]} arrays are considered single values (similar
			 * to String) and are not treated as arrays.
			 */
			boolean isArray();

			/**
			 * Returns {@code true} if {@link #getType()} is {@code boolean} or
			 * {@code java.lang.Boolean}.
			 */
			boolean isBoolean();

			boolean isCollection();

			/** Returns {@code true} if {@link #getType()} is an enum. */
			boolean isEnum();

			boolean isMap();

			/**
			 * Returns {@code true} if {@link #getType()} is an array, map or collection.
			 * Note that from picocli 4.7, {@code char[]} arrays are considered single
			 * values (similar to String) and are not treated as arrays.
			 */
			boolean isMultiValue();

			/**
			 * Returns {@code true} if {@link #getType()} is {@code java.util.Optional}
			 * 
			 * @since 4.6
			 */
			boolean isOptional();
		}

		/**
		 * Utility class for getting resource bundle strings. Enhances the standard
		 * <a href=
		 * "https://docs.oracle.com/javase/8/docs/api/java/util/ResourceBundle.html">ResourceBundle</a>
		 * with support for String arrays and qualified keys: keys that may or may not
		 * be prefixed with the fully qualified command name.
		 * <p>
		 * Example properties resource bundle:
		 * </p>
		 * 
		 * <pre>
		 * # Usage Help Message Sections
		 * # ---------------------------
		 * # Numbered resource keys can be used to create multi-line sections.
		 * usage.headerHeading = This is my app. There are other apps like it but this one is mine.%n
		 * usage.header   = header first line
		 * usage.header.0 = header second line
		 * usage.descriptionHeading = Description:%n
		 * usage.description.0 = first line
		 * usage.description.1 = second line
		 * usage.description.2 = third line
		 * usage.synopsisHeading = Usage:&#92;u0020
		 * # Leading whitespace is removed by default. Start with &#92;u0020 to keep the leading whitespace.
		 * usage.customSynopsis.0 =      Usage: ln [OPTION]... [-T] TARGET LINK_NAME   (1st form)
		 * usage.customSynopsis.1 = &#92;u0020 or:  ln [OPTION]... TARGET                  (2nd form)
		 * usage.customSynopsis.2 = &#92;u0020 or:  ln [OPTION]... TARGET... DIRECTORY     (3rd form)
		 * # Headings can contain the %n character to create multi-line values.
		 * usage.parameterListHeading = %nPositional parameters:%n
		 * usage.optionListHeading = %nOptions:%n
		 * usage.commandListHeading = %nCommands:%n
		 * usage.footerHeading = Powered by picocli%n
		 * usage.footer = footer
		 *
		 * # Option Descriptions
		 * # -------------------
		 * # Use numbered keys to create multi-line descriptions.
		 * help = Show this help message and exit.
		 * version = Print version information and exit.
		 * </pre>
		 * <p>
		 * Resources for multiple commands can be specified in a single ResourceBundle.
		 * Keys and their value can be shared by multiple commands (so you don't need to
		 * repeat them for every command), but keys can be prefixed with
		 * {@code fully qualified command name + "."} to specify different values for
		 * different commands. The most specific key wins. For example:
		 * </p>
		 * 
		 * <pre>
		 * jfrog.rt.usage.header = Artifactory commands
		 * jfrog.rt.config.usage.header = Configure Artifactory details.
		 * jfrog.rt.upload.usage.header = Upload files.
		 *
		 * jfrog.bt.usage.header = Bintray commands
		 * jfrog.bt.config.usage.header = Configure Bintray details.
		 * jfrog.bt.upload.usage.header = Upload files.
		 *
		 * # shared between all commands
		 * usage.footerHeading = Environment Variables:
		 * usage.footer.0 = footer line 0
		 * usage.footer.1 = footer line 1
		 * </pre>
		 * 
		 * @see Command#resourceBundle()
		 * @see Option#descriptionKey()
		 * @see OptionSpec#description()
		 * @see PositionalParamSpec#description()
		 * @see CommandSpec#qualifiedName(String)
		 * @since 3.6
		 */
		public static class Messages {
			private static boolean loadBundles = true;

			private static List<String> addAllWithPrefix(ResourceBundle rb, String key, Set<String> keys,
					List<String> result) {
				if (keys.contains(key)) {
					result.add(rb.getString(key));
				}
				for (int i = 0; true; i++) {
					final String elementKey = key + "." + i;
					if (keys.contains(elementKey)) {
						result.add(rb.getString(elementKey));
					} else {
						return result;
					}
				}
			}

			/**
			 * Returns a copy of the specified Messages object with the CommandSpec replaced
			 * by the specified one.
			 * 
			 * @param spec     the CommandSpec of the returned Messages
			 * @param original the Messages object whose ResourceBundle to reference
			 * @return a Messages object with the specified CommandSpec and the
			 *         ResourceBundle of the specified Messages object
			 */
			public static Messages copy(CommandSpec spec, Messages original) {
				return original == null ? null : new Messages(spec, original.bundleBaseName, original.rb);
			}

			private static ResourceBundle createBundle(String baseName) {
				if (loadBundles) {
					tracer().debug("Messages: Loading ResourceBundle[base=%s]...", baseName);
					return ResourceBundle.getBundle(baseName);
				} else {
					tracer().debug("Messages: Returning new dummy ResourceBundle", loadBundles);
					return new ResourceBundle() {
						@Override
						public Enumeration<String> getKeys() {
							return Collections.enumeration(Collections.<String>emptyList());
						}

						@Override
						protected Object handleGetObject(String key) {
							return null;
						}
					};
				}
			}

			/**
			 * Returns {@code true} if the specified {@code Messages} is {@code null}, has a
			 * {@code null ResourceBundle}, or has a {@code null parent Messages}.
			 */
			public static boolean empty(Messages messages) {
				return messages == null || messages.isEmpty();
			}

			private static String extractName(ResourceBundle rb) {
				try { // ResourceBundle.getBaseBundleName was introduced in Java 8
					return (String) ResourceBundle.class.getDeclaredMethod("getBaseBundleName").invoke(rb);
				} catch (final Exception ignored) {
					return "?";
				}
			}

			private static Set<String> keys(ResourceBundle rb) {
				if (rb == null) {
					return Collections.emptySet();
				}
				final Set<String> keys = new LinkedHashSet<String>();
				for (final Enumeration<String> k = rb.getKeys(); k.hasMoreElements(); keys.add(k.nextElement())) {
					;
				}
				return keys;
			}

			/**
			 * Returns the ResourceBundle of the specified Messages object or {@code null}
			 * if the specified Messages object is {@code null}.
			 */
			public static ResourceBundle resourceBundle(Messages messages) {
				return messages == null ? null : messages.resourceBundle();
			}

			/**
			 * Returns the ResourceBundle of the specified Messages object or {@code null}
			 * if the specified Messages object is {@code null}.
			 * 
			 * @since 4.0
			 */
			public static String resourceBundleBaseName(Messages messages) {
				return messages == null ? null : messages.resourceBundleBaseName();
			}

			/**
			 * During annotation processing, resource bundles may not be available on the
			 * classpath and thereby cause failures. This method allows for disabling
			 * loading of resource bundles during annotation processing, preventing such
			 * errors.
			 * 
			 * @since 4.7.6-SNAPSHOT
			 * @param loadBundles true if bundles should be loaded (default), false if
			 *                    bundles should not be loaded
			 */
			public static final void setLoadBundles(boolean loadBundles) {
				tracer().debug("Messages: loadBundles was set to %s", loadBundles);
				Messages.loadBundles = loadBundles;
			}

			private final CommandSpec spec;
			private final String bundleBaseName;
			private final ResourceBundle rb;

			private final Set<String> keys;

			private Messages parent;

			public Messages(CommandSpec spec, ResourceBundle rb) {
				this(spec, extractName(rb), rb);
			}

			public Messages(CommandSpec spec, String baseName) {
				this(spec, baseName, createBundle(baseName));
			}

			public Messages(CommandSpec spec, String baseName, ResourceBundle rb) {
				this.spec = Assert.notNull(spec, "CommandSpec");
				this.bundleBaseName = baseName;
				this.rb = rb;
				this.keys = keys(rb);
				if (rb != null) {
					CommandLine.tracer().debug("Created Messages from resourceBundle[base=%s] for command '%s' (%s)",
							baseName, spec.name(), spec);
				}
			}

			/** Returns the CommandSpec of this object, never {@code null}. */
			public CommandSpec commandSpec() {
				return spec;
			}

			/**
			 * Returns the String value found in the resource bundle for the specified key,
			 * or the specified default value if not found.
			 * 
			 * @param key          unqualified resource bundle key. This method will first
			 *                     try to find a value by qualifying the key with the
			 *                     command's fully qualified name, and if not found, it will
			 *                     try with the unqualified key.
			 * @param defaultValue value to return if the resource bundle is null or empty,
			 *                     or if no value was found by the qualified or unqualified
			 *                     key
			 * @return the String value found in the resource bundle for the specified key,
			 *         or the specified default value
			 */
			public String getString(String key, String defaultValue) {
				if (isEmpty()) {
					return defaultValue;
				}
				final String cmd = spec.qualifiedName(".");
				final String qualifiedKey = cmd + "." + key;
				String result = getStringForExactKey(qualifiedKey);
				if (result == null) {
					result = getStringForExactKey(key);
				}
				return result != null ? result : defaultValue;
			}

			/**
			 * Returns the String array value found in the resource bundle for the specified
			 * key, or the specified default value if not found. Multi-line strings can be
			 * specified in the resource bundle with {@code key.0}, {@code key.1},
			 * {@code key.2}, etc.
			 * 
			 * @param key           unqualified resource bundle key. This method will first
			 *                      try to find a value by qualifying the key with the
			 *                      command's fully qualified name, and if not found, it
			 *                      will try with the unqualified key.
			 * @param defaultValues value to return if the resource bundle is null or empty,
			 *                      or if no value was found by the qualified or unqualified
			 *                      key
			 * @return the String array value found in the resource bundle for the specified
			 *         key, or the specified default value
			 */
			public String[] getStringArray(String key, String[] defaultValues) {
				if (isEmpty()) {
					return defaultValues;
				}
				final String cmd = spec.qualifiedName(".");
				final String qualifiedKey = cmd + "." + key;
				String[] result = getStringArrayForExactKey(qualifiedKey);
				if (result == null) {
					result = getStringArrayForExactKey(key);
				}
				return result != null ? result : defaultValues;
			}

			private String[] getStringArrayForExactKey(String key) {
				final List<String> result = addAllWithPrefix(rb, key, keys, new ArrayList<String>());
				if (!result.isEmpty()) {
					return result.toArray(new String[0]);
				}
				return parent() == null ? null : parent().getStringArrayForExactKey(key);
			}

			private String getStringForExactKey(String key) {
				if (keys.contains(key)) {
					return rb.getString(key);
				} else if (parent() != null) {
					return parent().getStringForExactKey(key);
				} else {
					return null;
				}
			}

			boolean isEmpty() {
				return (rb == null || keys.isEmpty()) && (parent() == null || parent().isEmpty());
			}

			public Messages parent() {
				CommandSpec parentSpec = this.spec.parent();
				if (parent == null || parent.spec != parentSpec) {
					parent = null; // Refresh if parentSpec doesn't match
					while (parent == null && parentSpec != null) {
						final String parentResourceBundleBaseName = parentSpec.resourceBundleBaseName();
						if (parentResourceBundleBaseName != null
								&& !parentResourceBundleBaseName.equals(this.bundleBaseName)) {
							parent = new Messages(parentSpec, parentResourceBundleBaseName);
						} else {
							parentSpec = parentSpec.parent();
						}
					}
				}
				return parent;
			}

			/** Returns the ResourceBundle of this object or {@code null}. */
			public ResourceBundle resourceBundle() {
				return rb;
			}

			/**
			 * Returns the base name of the ResourceBundle of this object or {@code null}.
			 * 
			 * @since 4.0
			 */
			public String resourceBundleBaseName() {
				return bundleBaseName;
			}
		}

		static class MethodBinding implements IGetter, ISetter, IScoped {
			private final IScope scope;
			private final Method method;
			private final CommandSpec spec;
			private Object currentValue;

			MethodBinding(IScope scope, Method method, CommandSpec spec) {
				this.scope = scope;
				this.method = method;
				this.spec = spec;
			}

			private ParameterException createParameterException(Object value, Throwable t) {
				final CommandLine cmd = spec.commandLine() == null ? new CommandLine(spec) : spec.commandLine();
				return new ParameterException(cmd, "Could not invoke " + method + " with " + value + " (" + t + ")", t);
			}

			@Override
			@SuppressWarnings("unchecked")
			public <T> T get() {
				return (T) currentValue;
			}

			@Override
			public IScope getScope() {
				return scope;
			}

			@Override
			public <T> T set(T value) throws PicocliException {
				Object obj;
				try {
					obj = scope.get();
				} catch (final Exception ex) {
					throw new PicocliException("Could not get scope for method " + method, ex);
				}
				try {
					@SuppressWarnings("unchecked")
					final T result = (T) currentValue;
					method.invoke(obj, value);
					currentValue = value;
					return result;
				} catch (final InvocationTargetException ex) {
					if (ex.getTargetException() instanceof PicocliException) {
						throw (PicocliException) ex.getTargetException();
					}
					throw createParameterException(value, ex.getTargetException());
				} catch (final Exception ex) {
					throw createParameterException(value, ex);
				}
			}

			@Override
			public String toString() {
				return String.format("%s(%s)", getClass().getSimpleName(), method);
			}
		}

		/**
		 * Command method parameter, similar to java.lang.reflect.Parameter (not
		 * available before Java 8).
		 * 
		 * @since 4.0
		 */
		public static class MethodParam extends AccessibleObject {
			final Method method;
			final int paramIndex;
			final String name;
			int position;

			public MethodParam(Method method, int paramIndex) {
				this.method = method;
				this.paramIndex = paramIndex;
				String tmp = "arg" + paramIndex;
				try {
					final Method getParameters = Method.class.getMethod("getParameters");
					final Object parameters = getParameters.invoke(method);
					final Object parameter = Array.get(parameters, paramIndex);
					tmp = (String) Class.forName("java.lang.reflect.Parameter").getDeclaredMethod("getName")
							.invoke(parameter);
				} catch (final Exception ignored) {
				}
				this.name = tmp;
			}

			@Override
			public <T extends Annotation> T getAnnotation(Class<T> annotationClass) {
				for (final Annotation annotation : getDeclaredAnnotations()) {
					if (annotationClass.isAssignableFrom(annotation.annotationType())) {
						return annotationClass.cast(annotation);
					}
				}
				return null;
			}

			@Override
			public Annotation[] getDeclaredAnnotations() {
				return method.getParameterAnnotations()[paramIndex];
			}

			public Method getDeclaringExecutable() {
				return method;
			}

			public String getName() {
				return name;
			}

			public Type getParameterizedType() {
				return method.getGenericParameterTypes()[paramIndex];
			}

			public Class<?> getType() {
				return method.getParameterTypes()[paramIndex];
			}

			@SuppressWarnings("deprecation")
			@Override
			public boolean isAccessible() throws SecurityException {
				return method.isAccessible();
			}

			@Override
			public void setAccessible(boolean flag) throws SecurityException {
				method.setAccessible(flag);
			}

			@Override
			public String toString() {
				return method.toString() + ":" + getName();
			}
		}

		private static class ObjectBinding implements IGetter, ISetter {
			private Object value;

			@Override
			@SuppressWarnings("unchecked")
			public <T> T get() {
				return (T) value;
			}

			@Override
			public <T> T set(T value) {
				final T result = value;
				this.value = value;
				return result;
			}

			@Override
			public String toString() {
				return String.format("%s(value=%s)", getClass().getSimpleName(), value);
			}
		}

		static class ObjectScope implements IScope {
			static IScope asScope(Object scope) {
				return scope instanceof IScope ? ((IScope) scope) : new ObjectScope(scope);
			}

			public static boolean hasInstance(IScope scope) {
				if (scope instanceof CommandUserObject) {
					return ((CommandUserObject) scope).instance != null;
				}
				return ObjectScope.tryGet(scope) != null;
			}

			public static boolean isProxyClass(IScope scope) {
				if (scope instanceof CommandUserObject) {
					return ((CommandUserObject) scope).isProxyClass();
				}
				final Object obj = ObjectScope.tryGet(scope);
				return obj != null && Proxy.isProxyClass(obj.getClass());
			}

			public static Object tryGet(IScope scope) {
				try {
					return scope.get();
				} catch (final Exception e) {
					throw new InitializationException("Could not get scope value", e);
				}
			}

			private Object value;

			public ObjectScope(Object value) {
				this.value = value;
			}

			@Override
			@SuppressWarnings("unchecked")
			public <T> T get() {
				return (T) value;
			}

			@Override
			@SuppressWarnings("unchecked")
			public <T> T set(T value) {
				final T old = (T) this.value;
				this.value = value;
				return old;
			}

			@Override
			public String toString() {
				return String.format("Scope(value=%s)", value);
			}
		}

		/**
		 * The {@code OptionSpec} class models aspects of a <em>named option</em> of a
		 * {@linkplain CommandSpec command}, including whether it is required or
		 * optional, the option parameters supported (or required) by the option, and
		 * attributes for the usage help message describing the option.
		 * <p>
		 * An option has one or more names. The option is matched when the parser
		 * encounters one of the option names in the command line arguments. Depending
		 * on the option's {@link #arity() arity}, the parser may expect it to have
		 * option parameters. The parser will call {@link #setValue(Object) setValue} on
		 * the matched option for each of the option parameters encountered.
		 * </p>
		 * <p>
		 * For multi-value options, the {@code type} may be an array, a
		 * {@code Collection} or a {@code Map}. In this case the parser will get the
		 * data structure by calling {@link #getValue() getValue} and modify the
		 * contents of this data structure. (In the case of arrays, the array is
		 * replaced with a new instance with additional elements.)
		 * </p>
		 * <p>
		 * Before calling the setter, picocli converts the option parameter value from a
		 * String to the option parameter's type.
		 * </p>
		 * <ul>
		 * <li>If a option-specific {@link #converters() converter} is configured, this
		 * will be used for type conversion. If the option's type is a {@code Map}, the
		 * map may have different types for its keys and its values, so
		 * {@link #converters() converters} should provide two converters: one for the
		 * map keys and one for the map values.</li>
		 * <li>Otherwise, the option's {@link #type() type} is used to look up a
		 * converter in the list of
		 * {@linkplain CommandLine#registerConverter(Class, ITypeConverter) registered
		 * converters}. For multi-value options, the {@code type} may be an array, or a
		 * {@code Collection} or a {@code Map}. In that case the elements are converted
		 * based on the option's {@link #auxiliaryTypes() auxiliaryTypes}. The
		 * auxiliaryType is used to look up the converter(s) to use to convert the
		 * individual parameter values. Maps may have different types for its keys and
		 * its values, so {@link #auxiliaryTypes() auxiliaryTypes} should provide two
		 * types: one for the map keys and one for the map values.</li>
		 * </ul>
		 * <p>
		 * {@code OptionSpec} objects are used by the picocli command line interpreter
		 * and help message generator. Picocli can construct an {@code OptionSpec}
		 * automatically from fields and methods with {@link Option @Option}
		 * annotations. Alternatively an {@code OptionSpec} can be constructed
		 * programmatically.
		 * </p>
		 * <p>
		 * When an {@code OptionSpec} is created from an {@link Option @Option}
		 * -annotated field or method, it is "bound" to that field or method: this field
		 * is set (or the method is invoked) when the option is matched and
		 * {@link #setValue(Object) setValue} is called. Programmatically constructed
		 * {@code OptionSpec} instances will remember the value passed to the
		 * {@link #setValue(Object) setValue} method so it can be retrieved with the
		 * {@link #getValue() getValue} method. This behaviour can be customized by
		 * installing a custom {@link IGetter} and {@link ISetter} on the
		 * {@code OptionSpec}.
		 * </p>
		 * 
		 * @since 3.0
		 */
		public static class OptionSpec extends ArgSpec implements IOrdered {
			/**
			 * Builder responsible for creating valid {@code OptionSpec} objects.
			 * 
			 * @since 3.0
			 */
			public static class Builder extends ArgSpec.Builder<Builder> {
				private String[] names;
				private boolean help;
				private boolean usageHelp;
				private boolean versionHelp;
				private boolean negatable;
				private String fallbackValue = DEFAULT_FALLBACK_VALUE;
				private String originalFallbackValue = ArgSpec.UNSPECIFIED;
				private int order = DEFAULT_ORDER;

				private Builder(IAnnotatedElement member, IFactory factory) {
					super(member.getAnnotation(Option.class), member, factory);
					final Option option = member.getAnnotation(Option.class);
					names = option.names();
					help = option.help();
					usageHelp = option.usageHelp();
					versionHelp = option.versionHelp();
					negatable = option.negatable();
					fallbackValue = NULL_VALUE.equals(option.fallbackValue()) ? null : option.fallbackValue();
					originalFallbackValue = option.fallbackValue();
					order = option.order();
				}

				private Builder(OptionSpec original) {
					super(original);
					names = original.names;
					help = original.help;
					usageHelp = original.usageHelp;
					versionHelp = original.versionHelp;
					negatable = original.negatable;
					fallbackValue = original.fallbackValue;
					originalFallbackValue = original.originalFallbackValue;
					order = original.order;
				}

				private Builder(String[] names) {
					this.names = names;
				}

				/** Returns a valid {@code OptionSpec} instance. */
				@Override
				public OptionSpec build() {
					return new OptionSpec(this);
				}

				/**
				 * Returns the fallback value for this option: the value that is assigned for
				 * options with an optional parameter if the option was specified on the command
				 * line without parameter.
				 * 
				 * @see Option#fallbackValue()
				 * @since 4.0
				 */
				public String fallbackValue() {
					return fallbackValue;
				}

				/**
				 * Sets the fallback value for this option: the value that is assigned for
				 * options with an optional parameter if the option was specified on the command
				 * line without parameter, and returns this builder.
				 * 
				 * @see Option#fallbackValue()
				 * @since 4.0
				 */
				public Builder fallbackValue(String fallbackValue) {
					this.fallbackValue = fallbackValue;
					return self();
				}

				/**
				 * Returns whether this option disables validation of the other arguments.
				 * 
				 * @see Option#help()
				 * @deprecated Use {@link #usageHelp()} and {@link #versionHelp()} instead.
				 */
				@Deprecated
				public boolean help() {
					return help;
				}

				/**
				 * Sets whether this option disables validation of the other arguments, and
				 * returns this builder.
				 */
				public Builder help(boolean help) {
					this.help = help;
					return self();
				}

				/**
				 * Returns one or more option names. At least one option name is required.
				 * 
				 * @see Option#names()
				 */
				public String[] names() {
					return names.clone();
				}

				/**
				 * Replaces the option names with the specified values. At least one option name
				 * is required, and returns this builder.
				 * 
				 * @return this builder instance to provide a fluent interface
				 */
				public Builder names(String... names) {
					this.names = Assert.notNull(names, "names").clone();
					return self();
				}

				/**
				 * Returns whether a negative version for this boolean option is automatically
				 * added. The form of the negative name is determined by the
				 * {@link INegatableOptionTransformer}.
				 * 
				 * @see Option#negatable()
				 * @since 4.0
				 */
				public boolean negatable() {
					return negatable;
				}

				/**
				 * Sets whether a negative version for this boolean option is automatically
				 * added, and returns this builder.
				 * 
				 * @since 4.0
				 */
				public Builder negatable(boolean negatable) {
					this.negatable = negatable;
					return self();
				}

				/**
				 * Returns the position in the options list in the usage help message at which
				 * this option should be shown. Options with a lower number are shown before
				 * options with a higher number. This attribute is only honored if
				 * {@link UsageMessageSpec#sortOptions()} is {@code false} for this command.
				 * 
				 * @see Option#order()
				 * @since 3.9
				 */
				public int order() {
					return order;
				}

				/**
				 * Sets the position in the options list in the usage help message at which this
				 * option should be shown, and returns this builder.
				 * 
				 * @since 3.9
				 */
				public Builder order(int order) {
					this.order = order;
					return self();
				}

				/** Returns this builder. */
				@Override
				protected Builder self() {
					return this;
				}

				/**
				 * Returns whether this option allows the user to request usage help.
				 * 
				 * @see Option#usageHelp()
				 */
				public boolean usageHelp() {
					return usageHelp;
				}

				/**
				 * Sets whether this option allows the user to request usage help, and returns
				 * this builder.
				 */
				public Builder usageHelp(boolean usageHelp) {
					this.usageHelp = usageHelp;
					return self();
				}

				/**
				 * Returns whether this option allows the user to request version information.
				 * 
				 * @see Option#versionHelp()
				 */
				public boolean versionHelp() {
					return versionHelp;
				}

				/**
				 * Sets whether this option allows the user to request version information, and
				 * returns this builder.
				 */
				public Builder versionHelp(boolean versionHelp) {
					this.versionHelp = versionHelp;
					return self();
				}
			}

			public static final String DEFAULT_FALLBACK_VALUE = "";
			static final int DEFAULT_ORDER = -1;

			public static OptionSpec.Builder builder(IAnnotatedElement source, IFactory factory) {
				return new Builder(source, factory);
			}

			/**
			 * Returns a Builder initialized from the specified {@code OptionSpec}.
			 * 
			 * @since 4.0
			 */
			public static OptionSpec.Builder builder(OptionSpec original) {
				return new Builder(original);
			}

			public static OptionSpec.Builder builder(String name, String... names) {
				final String[] copy = new String[Assert.notNull(names, "names").length + 1];
				copy[0] = Assert.notNull(name, "name");
				System.arraycopy(names, 0, copy, 1, names.length);
				return new Builder(copy);
			}

			public static OptionSpec.Builder builder(String[] names) {
				return new Builder(names);
			}

			private final String[] names;
			private final boolean help;
			private final boolean usageHelp;

			private final boolean versionHelp;
			private final boolean negatable;
			private final String fallbackValue;
			private final String originalFallbackValue;

			private final int order;

			/**
			 * Ensures all attributes of this {@code OptionSpec} have a valid value; throws
			 * an {@link InitializationException} if this cannot be achieved.
			 */
			private OptionSpec(Builder builder) {
				super(builder);
				if (builder.names == null) {
					throw new InitializationException(
							"OptionSpec names cannot be null. Specify at least one option name.");
				}
				names = builder.names.clone();
				help = builder.help;
				usageHelp = builder.usageHelp;
				versionHelp = builder.versionHelp;
				order = builder.order;
				negatable = builder.negatable;
				fallbackValue = builder.fallbackValue;
				originalFallbackValue = builder.originalFallbackValue;

				if (names.length == 0 || Arrays.asList(names).contains("")) {
					throw new InitializationException("Invalid names: " + Arrays.toString(names));
				}
				if (toString == null) {
					toString = "option " + longestName();
				}

				// https://github.com/remkop/picocli/issues/511
				// if (arity().max == 0 && !(isBoolean(type()) || (isMultiValue() &&
				// isBoolean(auxiliaryTypes()[0])))) {
				// throw new InitializationException("Option " + longestName() + " is not a
				// boolean so should not be defined with arity=" + arity());
				// }
			}

			@Override
			public boolean equals(Object obj) {
				if (obj == this) {
					return true;
				}
				if (!(obj instanceof OptionSpec)) {
					return false;
				}
				final OptionSpec other = (OptionSpec) obj;
				return super.equalsImpl(other) && help == other.help && usageHelp == other.usageHelp
						&& versionHelp == other.versionHelp && order == other.order && negatable == other.negatable
						&& Assert.equals(fallbackValue, other.fallbackValue)
						&& new HashSet<String>(Arrays.asList(names))
								.equals(new HashSet<String>(Arrays.asList(other.names)));
			}

			/**
			 * Returns the fallback value for this option: the value that is assigned for
			 * options with an optional parameter (for example, {@code arity = "0..1"}) if
			 * the option was specified on the command line without parameter.
			 * <p>
			 * If the special value {@link #NULL_VALUE} is set, this method returns
			 * {@code null}.
			 * </p>
			 * 
			 * @see Option#fallbackValue()
			 * @see #defaultValue()
			 * @since 4.0
			 */
			public String fallbackValue() {
				return interpolate(fallbackValue);
			}

			/**
			 * Returns the additional lookup keys for finding description lines in the
			 * resource bundle for this option.
			 * 
			 * @return option names (after variable interpolation), without leading hyphens,
			 *         slashes and other non-Java identifier characters.
			 * @since 4.0
			 */
			@Override
			protected Collection<String> getAdditionalDescriptionKeys() {
				final Set<String> result = new LinkedHashSet<String>();
				for (final String name : names()) {
					result.add(CommandSpec.stripPrefix(name));
				}
				return result;
			}

			@Override
			public int hashCode() {
				return super.hashCodeImpl() + 37 * Assert.hashCode(help) + 37 * Assert.hashCode(usageHelp)
						+ 37 * Assert.hashCode(versionHelp) + 37 * Arrays.hashCode(names)
						+ 37 * Assert.hashCode(negatable) + 37 * Assert.hashCode(fallbackValue) + 37 * order;
			}

			/**
			 * Returns whether this option disables validation of the other arguments.
			 * 
			 * @see Option#help()
			 * @deprecated Use {@link #usageHelp()} and {@link #versionHelp()} instead.
			 */
			@Deprecated
			public boolean help() {
				return help;
			}

			@Override
			protected boolean internalShowDefaultValue(boolean usageMessageShowDefaults) {
				return super.internalShowDefaultValue(usageMessageShowDefaults) && !help() && !versionHelp()
						&& !usageHelp();
			}

			@Override
			public boolean isOption() {
				return true;
			}

			@Override
			public boolean isPositional() {
				return false;
			}

			/** Returns the longest {@linkplain #names() option name}. */
			public String longestName() {
				return Help.ShortestFirst.longestFirst(names())[0];
			}

			/**
			 * Returns one or more option names. The returned array will contain at least
			 * one option name.
			 * 
			 * @see Option#names()
			 */
			public String[] names() {
				return interpolate(names.clone());
			}

			/**
			 * Returns whether a negative version for this boolean option is automatically
			 * added. The form of the negative name is determined by the
			 * {@link INegatableOptionTransformer}.
			 * 
			 * @see Option#negatable()
			 * @since 4.0
			 */
			public boolean negatable() {
				return negatable;
			}

			/**
			 * Returns the position in the options list in the usage help message at which
			 * this option should be shown. Options with a lower number are shown before
			 * options with a higher number. This attribute is only honored if
			 * {@link UsageMessageSpec#sortOptions()} is {@code false} for this command.
			 * 
			 * @see Option#order()
			 * @since 3.9
			 */
			@Override
			public int order() {
				return this.order;
			}

			/**
			 * Returns the shortest {@linkplain #names() option name}.
			 * 
			 * @since 3.8
			 */
			public String shortestName() {
				return Help.ShortestFirst.sort(names())[0];
			}

			/**
			 * Returns a new Builder initialized with the attributes from this
			 * {@code OptionSpec}. Calling {@code build} immediately will return a copy of
			 * this {@code OptionSpec}.
			 * 
			 * @return a builder that can create a copy of this spec
			 */
			public Builder toBuilder() {
				return new Builder(this);
			}

			/**
			 * Returns whether this option allows the user to request usage help.
			 * 
			 * @see Option#usageHelp()
			 */
			public boolean usageHelp() {
				return usageHelp;
			}

			/**
			 * Returns whether this option allows the user to request version information.
			 * 
			 * @see Option#versionHelp()
			 */
			public boolean versionHelp() {
				return versionHelp;
			}
		}

		/**
		 * Models parser configuration specification.
		 * 
		 * @since 3.0
		 */
		public static class ParserSpec {

			/**
			 * Constant String holding the default separator between options and option
			 * parameters: <code>{@value}</code>.
			 */
			static final String DEFAULT_SEPARATOR = "=";
			static final String DEFAULT_END_OF_OPTIONS_DELIMITER = "--";
			private boolean abbreviatedOptionsAllowed = false;
			private boolean abbreviatedSubcommandsAllowed = false;
			private boolean allowOptionsAsOptionParameters = false;
			private boolean allowSubcommandsAsOptionParameters = false;
			private boolean aritySatisfiedByAttachedOptionParam = false;
			private Character atFileCommentChar = '#';
			private boolean caseInsensitiveEnumValuesAllowed = false;
			private boolean collectErrors = false;
			private String endOfOptionsDelimiter = DEFAULT_END_OF_OPTIONS_DELIMITER;
			private boolean expandAtFiles = true;
			private boolean limitSplit = false;
			private boolean overwrittenOptionsAllowed = false;
			private boolean posixClusteredShortOptionsAllowed = true;
			private String separator;
			private boolean splitQuotedStrings = false;
			private boolean stopAtPositional = false;
			private boolean stopAtUnmatched = false;
			private boolean toggleBooleanFlags = false;
			private boolean trimQuotes = shouldTrimQuotes();
			private boolean unmatchedArgumentsAllowed = false;
			private boolean unmatchedOptionsAllowedAsOptionParameters = true;
			private boolean unmatchedOptionsArePositionalParams = false;
			private boolean useSimplifiedAtFiles = false;

			/** @see CommandLine#isAbbreviatedOptionsAllowed() */
			public boolean abbreviatedOptionsAllowed() {
				return abbreviatedOptionsAllowed;
			}

			/** @see CommandLine#setAbbreviatedOptionsAllowed(boolean) */
			public ParserSpec abbreviatedOptionsAllowed(boolean abbreviatedOptionsAllowed) {
				this.abbreviatedOptionsAllowed = abbreviatedOptionsAllowed;
				return this;
			}

			/** @see CommandLine#isAbbreviatedSubcommandsAllowed() */
			public boolean abbreviatedSubcommandsAllowed() {
				return abbreviatedSubcommandsAllowed;
			}

			/** @see CommandLine#setAbbreviatedSubcommandsAllowed(boolean) */
			public ParserSpec abbreviatedSubcommandsAllowed(boolean abbreviatedSubcommandsAllowed) {
				this.abbreviatedSubcommandsAllowed = abbreviatedSubcommandsAllowed;
				return this;
			}

			/**
			 * @see CommandLine#isAllowOptionsAsOptionParameters()
			 * @since 4.7.6-SNAPSHOT
			 */
			public boolean allowOptionsAsOptionParameters() {
				return allowOptionsAsOptionParameters;
			}

			/**
			 * @see CommandLine#setAllowOptionsAsOptionParameters(boolean)
			 * @since 4.7.6-SNAPSHOT
			 */
			public ParserSpec allowOptionsAsOptionParameters(boolean allowOptionsAsOptionParameters) {
				this.allowOptionsAsOptionParameters = allowOptionsAsOptionParameters;
				return this;
			}

			/**
			 * @see CommandLine#isAllowSubcommandsAsOptionParameters()
			 * @since 4.7.6-SNAPSHOT
			 */
			public boolean allowSubcommandsAsOptionParameters() {
				return allowSubcommandsAsOptionParameters;
			}

			/**
			 * @see CommandLine#setAllowSubcommandsAsOptionParameters(boolean)
			 * @since 4.7.6-SNAPSHOT
			 */
			public ParserSpec allowSubcommandsAsOptionParameters(boolean allowSubcommandsAsOptionParameters) {
				this.allowSubcommandsAsOptionParameters = allowSubcommandsAsOptionParameters;
				return this;
			}

			/**
			 * Returns true if options with attached arguments should not consume subsequent
			 * arguments and should not validate arity. The default is {@code false}.
			 */
			public boolean aritySatisfiedByAttachedOptionParam() {
				return aritySatisfiedByAttachedOptionParam;
			}

			/**
			 * Returns true if options with attached arguments should not consume subsequent
			 * arguments and should not validate arity. The default is {@code false}.
			 */
			public ParserSpec aritySatisfiedByAttachedOptionParam(boolean newValue) {
				aritySatisfiedByAttachedOptionParam = newValue;
				return this;
			}

			/**
			 * @see CommandLine#getAtFileCommentChar()
			 * @since 3.5
			 */
			public Character atFileCommentChar() {
				return atFileCommentChar;
			}

			/**
			 * @see CommandLine#setAtFileCommentChar(Character)
			 * @since 3.5
			 */
			public ParserSpec atFileCommentChar(Character atFileCommentChar) {
				this.atFileCommentChar = atFileCommentChar;
				return this;
			}

			/**
			 * @see CommandLine#isCaseInsensitiveEnumValuesAllowed()
			 * @since 3.4
			 */
			public boolean caseInsensitiveEnumValuesAllowed() {
				return caseInsensitiveEnumValuesAllowed;
			}

			/**
			 * @see CommandLine#setCaseInsensitiveEnumValuesAllowed(boolean)
			 * @since 3.4
			 */
			public ParserSpec caseInsensitiveEnumValuesAllowed(boolean caseInsensitiveEnumValuesAllowed) {
				this.caseInsensitiveEnumValuesAllowed = caseInsensitiveEnumValuesAllowed;
				return this;
			}

			/**
			 * Returns true if exceptions during parsing should be collected instead of
			 * thrown. Multiple errors may be encountered during parsing. These can be
			 * obtained from {@link ParseResult#errors()}.
			 * 
			 * @since 3.2
			 */
			public boolean collectErrors() {
				return collectErrors;
			}

			/**
			 * Sets whether exceptions during parsing should be collected instead of thrown.
			 * Multiple errors may be encountered during parsing. These can be obtained from
			 * {@link ParseResult#errors()}.
			 * 
			 * @since 3.2
			 */
			public ParserSpec collectErrors(boolean collectErrors) {
				this.collectErrors = collectErrors;
				return this;
			}

			/**
			 * @see CommandLine#getEndOfOptionsDelimiter()
			 * @since 3.5
			 */
			public String endOfOptionsDelimiter() {
				return endOfOptionsDelimiter;
			}

			/**
			 * @see CommandLine#setEndOfOptionsDelimiter(String)
			 * @since 3.5
			 */
			public ParserSpec endOfOptionsDelimiter(String delimiter) {
				this.endOfOptionsDelimiter = Assert.notNull(delimiter, "end-of-options delimiter");
				return this;
			}

			/** @see CommandLine#isExpandAtFiles() */
			public boolean expandAtFiles() {
				return expandAtFiles;
			}

			/** @see CommandLine#setExpandAtFiles(boolean) */
			public ParserSpec expandAtFiles(boolean expandAtFiles) {
				this.expandAtFiles = expandAtFiles;
				return this;
			}

			void initFrom(ParserSpec settings) {
				abbreviatedOptionsAllowed = settings.abbreviatedOptionsAllowed;
				abbreviatedSubcommandsAllowed = settings.abbreviatedSubcommandsAllowed;
				allowOptionsAsOptionParameters = settings.allowOptionsAsOptionParameters;
				allowSubcommandsAsOptionParameters = settings.allowSubcommandsAsOptionParameters;
				aritySatisfiedByAttachedOptionParam = settings.aritySatisfiedByAttachedOptionParam;
				atFileCommentChar = settings.atFileCommentChar;
				caseInsensitiveEnumValuesAllowed = settings.caseInsensitiveEnumValuesAllowed;
				collectErrors = settings.collectErrors;
				endOfOptionsDelimiter = settings.endOfOptionsDelimiter;
				expandAtFiles = settings.expandAtFiles;
				limitSplit = settings.limitSplit;
				overwrittenOptionsAllowed = settings.overwrittenOptionsAllowed;
				posixClusteredShortOptionsAllowed = settings.posixClusteredShortOptionsAllowed;
				separator = settings.separator;
				splitQuotedStrings = settings.splitQuotedStrings;
				stopAtPositional = settings.stopAtPositional;
				stopAtUnmatched = settings.stopAtUnmatched;
				toggleBooleanFlags = settings.toggleBooleanFlags;
				trimQuotes = settings.trimQuotes;
				unmatchedArgumentsAllowed = settings.unmatchedArgumentsAllowed;
				unmatchedOptionsAllowedAsOptionParameters = settings.unmatchedOptionsAllowedAsOptionParameters;
				unmatchedOptionsArePositionalParams = settings.unmatchedOptionsArePositionalParams;
				useSimplifiedAtFiles = settings.useSimplifiedAtFiles;
			}

			void initSeparator(String value) {
				if (initializable(separator, value, DEFAULT_SEPARATOR)) {
					separator = value;
				}
			}

			/**
			 * Returns true if arguments should be split first before any further processing
			 * and the number of parts resulting from the split is limited to the max arity
			 * of the argument.
			 */
			public boolean limitSplit() {
				return limitSplit;
			}

			/**
			 * Sets whether arguments should be {@linkplain ArgSpec#splitRegex() split}
			 * first before any further processing. If true, the original argument will only
			 * be split into as many parts as allowed by max arity.
			 */
			public ParserSpec limitSplit(boolean limitSplit) {
				this.limitSplit = limitSplit;
				return this;
			}

			/** @see CommandLine#isOverwrittenOptionsAllowed() */
			public boolean overwrittenOptionsAllowed() {
				return overwrittenOptionsAllowed;
			}

			/** @see CommandLine#setOverwrittenOptionsAllowed(boolean) */
			public ParserSpec overwrittenOptionsAllowed(boolean overwrittenOptionsAllowed) {
				this.overwrittenOptionsAllowed = overwrittenOptionsAllowed;
				return this;
			}

			/** @see CommandLine#isPosixClusteredShortOptionsAllowed() */
			public boolean posixClusteredShortOptionsAllowed() {
				return posixClusteredShortOptionsAllowed;
			}

			/** @see CommandLine#setPosixClusteredShortOptionsAllowed(boolean) */
			public ParserSpec posixClusteredShortOptionsAllowed(boolean posixClusteredShortOptionsAllowed) {
				this.posixClusteredShortOptionsAllowed = posixClusteredShortOptionsAllowed;
				return this;
			}

			/**
			 * Returns the String to use as the separator between options and option
			 * parameters. {@code "="} by default, initialized from
			 * {@link Command#separator()} if defined.
			 */
			public String separator() {
				return (separator == null) ? DEFAULT_SEPARATOR : separator;
			}

			/**
			 * Sets the String to use as the separator between options and option
			 * parameters.
			 * 
			 * @return this ParserSpec for method chaining
			 */
			public ParserSpec separator(String separator) {
				this.separator = separator;
				return this;
			}

			private boolean shouldTrimQuotes() {
				String value = System.getProperty("picocli.trimQuotes");
				if ("".equals(value)) {
					value = "true";
				}
				return Boolean.parseBoolean(value);
			}

			private boolean splitFirst() {
				return limitSplit();
			}

			/**
			 * @see CommandLine#isSplitQuotedStrings()
			 * @since 3.7
			 */
			public boolean splitQuotedStrings() {
				return splitQuotedStrings;
			}

			/**
			 * @see CommandLine#setSplitQuotedStrings(boolean)
			 * @since 3.7
			 */
			public ParserSpec splitQuotedStrings(boolean splitQuotedStrings) {
				this.splitQuotedStrings = splitQuotedStrings;
				return this;
			}

			/** @see CommandLine#isStopAtPositional() */
			public boolean stopAtPositional() {
				return stopAtPositional;
			}

			/** @see CommandLine#setStopAtPositional(boolean) */
			public ParserSpec stopAtPositional(boolean stopAtPositional) {
				this.stopAtPositional = stopAtPositional;
				return this;
			}

			/** @see CommandLine#isStopAtUnmatched() */
			public boolean stopAtUnmatched() {
				return stopAtUnmatched;
			}

			/** @see CommandLine#setStopAtUnmatched(boolean) */
			public ParserSpec stopAtUnmatched(boolean stopAtUnmatched) {
				this.stopAtUnmatched = stopAtUnmatched;
				return this;
			}

			/** @see CommandLine#isToggleBooleanFlags() */
			public boolean toggleBooleanFlags() {
				return toggleBooleanFlags;
			}

			/** @see CommandLine#setToggleBooleanFlags(boolean) */
			public ParserSpec toggleBooleanFlags(boolean toggleBooleanFlags) {
				this.toggleBooleanFlags = toggleBooleanFlags;
				return this;
			}

			@Override
			public String toString() {
				return String.format(
						"abbreviatedOptionsAllowed=%s, abbreviatedSubcommandsAllowed=%s, allowOptionsAsOptionParameters=%s, "
								+ "allowSubcommandsAsOptionParameters=%s, aritySatisfiedByAttachedOptionParam=%s, atFileCommentChar=%s, "
								+ "caseInsensitiveEnumValuesAllowed=%s, collectErrors=%s, endOfOptionsDelimiter=%s, expandAtFiles=%s, "
								+ "limitSplit=%s, overwrittenOptionsAllowed=%s, posixClusteredShortOptionsAllowed=%s, "
								+ "separator=%s, splitQuotedStrings=%s, stopAtPositional=%s, stopAtUnmatched=%s, "
								+ "toggleBooleanFlags=%s, trimQuotes=%s, "
								+ "unmatchedArgumentsAllowed=%s, unmatchedOptionsAllowedAsOptionParameters=%s, unmatchedOptionsArePositionalParams=%s, useSimplifiedAtFiles=%s",
						abbreviatedOptionsAllowed, abbreviatedSubcommandsAllowed, allowOptionsAsOptionParameters,
						allowSubcommandsAsOptionParameters, aritySatisfiedByAttachedOptionParam, atFileCommentChar,
						caseInsensitiveEnumValuesAllowed, collectErrors, endOfOptionsDelimiter, expandAtFiles,
						limitSplit, overwrittenOptionsAllowed, posixClusteredShortOptionsAllowed, separator,
						splitQuotedStrings, stopAtPositional, stopAtUnmatched, toggleBooleanFlags, trimQuotes,
						unmatchedArgumentsAllowed, unmatchedOptionsAllowedAsOptionParameters,
						unmatchedOptionsArePositionalParams, useSimplifiedAtFiles);
			}

			/**
			 * @see CommandLine#isTrimQuotes()
			 * @since 3.7
			 */
			public boolean trimQuotes() {
				return trimQuotes;
			}

			/**
			 * @see CommandLine#setTrimQuotes(boolean)
			 * @since 3.7
			 */
			public ParserSpec trimQuotes(boolean trimQuotes) {
				this.trimQuotes = trimQuotes;
				return this;
			}

			/** @see CommandLine#isUnmatchedArgumentsAllowed() */
			public boolean unmatchedArgumentsAllowed() {
				return unmatchedArgumentsAllowed;
			}

			/** @see CommandLine#setUnmatchedArgumentsAllowed(boolean) */
			public ParserSpec unmatchedArgumentsAllowed(boolean unmatchedArgumentsAllowed) {
				this.unmatchedArgumentsAllowed = unmatchedArgumentsAllowed;
				return this;
			}

			/**
			 * @see CommandLine#isUnmatchedOptionsAllowedAsOptionParameters()
			 * @since 4.4
			 */
			public boolean unmatchedOptionsAllowedAsOptionParameters() {
				return unmatchedOptionsAllowedAsOptionParameters;
			}

			/**
			 * @see CommandLine#setUnmatchedOptionsAllowedAsOptionParameters(boolean)
			 * @since 4.4
			 */
			public ParserSpec unmatchedOptionsAllowedAsOptionParameters(
					boolean unmatchedOptionsAllowedAsOptionParameters) {
				this.unmatchedOptionsAllowedAsOptionParameters = unmatchedOptionsAllowedAsOptionParameters;
				return this;
			}

			/** @see CommandLine#isUnmatchedOptionsArePositionalParams() */
			public boolean unmatchedOptionsArePositionalParams() {
				return unmatchedOptionsArePositionalParams;
			}

			/** @see CommandLine#setUnmatchedOptionsArePositionalParams(boolean) */
			public ParserSpec unmatchedOptionsArePositionalParams(boolean unmatchedOptionsArePositionalParams) {
				this.unmatchedOptionsArePositionalParams = unmatchedOptionsArePositionalParams;
				return this;
			}

			void updateSeparator(String value) {
				if (isNonDefault(value, DEFAULT_SEPARATOR)) {
					separator = value;
				}
			}

			/**
			 * @see CommandLine#isUseSimplifiedAtFiles()
			 * @since 3.9
			 */
			public boolean useSimplifiedAtFiles() {
				final String value = System.getProperty("picocli.useSimplifiedAtFiles");
				if (value != null) {
					return "".equals(value) || Boolean.parseBoolean(value);
				}
				return useSimplifiedAtFiles;
			}

			/**
			 * @see CommandLine#setUseSimplifiedAtFiles(boolean)
			 * @since 3.9
			 */
			public ParserSpec useSimplifiedAtFiles(boolean useSimplifiedAtFiles) {
				this.useSimplifiedAtFiles = useSimplifiedAtFiles;
				return this;
			}
		}

		private static class PicocliInvocationHandler implements InvocationHandler {
			class ProxyBinding implements IGetter, ISetter {
				private final Method method;

				ProxyBinding(Method method) {
					this.method = Assert.notNull(method, "method");
				}

				@Override
				@SuppressWarnings("unchecked")
				public <T> T get() {
					return (T) map.get(method.getName());
				}

				@Override
				public <T> T set(T value) {
					final T result = this.<T>get();
					map.put(method.getName(), value);
					return result;
				}
			}

			final Map<String, Object> map = new HashMap<String, Object>();

			@Override
			public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
				return map.get(method.getName());
			}
		}

		/**
		 * The {@code PositionalParamSpec} class models aspects of a <em>positional
		 * parameter</em> of a {@linkplain CommandSpec command}, including whether it is
		 * required or optional, and attributes for the usage help message describing
		 * the positional parameter.
		 * <p>
		 * Positional parameters have an {@link #index() index} (or a range of indices).
		 * A positional parameter is matched when the parser encounters a command line
		 * argument at that index. Named options and their parameters do not change the
		 * index counter, so the command line can contain a mixture of positional
		 * parameters and named options.
		 * </p>
		 * <p>
		 * Depending on the positional parameter's {@link #arity() arity}, the parser
		 * may consume multiple command line arguments starting from the current index.
		 * The parser will call {@link #setValue(Object) setValue} on the
		 * {@code PositionalParamSpec} for each of the parameters encountered. For
		 * multi-value positional parameters, the {@code type} may be an array, a
		 * {@code Collection} or a {@code Map}. In this case the parser will get the
		 * data structure by calling {@link #getValue() getValue} and modify the
		 * contents of this data structure. (In the case of arrays, the array is
		 * replaced with a new instance with additional elements.)
		 * </p>
		 * <p>
		 * Before calling the setter, picocli converts the positional parameter value
		 * from a String to the parameter's type.
		 * </p>
		 * <ul>
		 * <li>If a positional parameter-specific {@link #converters() converter} is
		 * configured, this will be used for type conversion. If the positional
		 * parameter's type is a {@code Map}, the map may have different types for its
		 * keys and its values, so {@link #converters() converters} should provide two
		 * converters: one for the map keys and one for the map values.</li>
		 * <li>Otherwise, the positional parameter's {@link #type() type} is used to
		 * look up a converter in the list of
		 * {@linkplain CommandLine#registerConverter(Class, ITypeConverter) registered
		 * converters}. For multi-value positional parameters, the {@code type} may be
		 * an array, or a {@code Collection} or a {@code Map}. In that case the elements
		 * are converted based on the positional parameter's {@link #auxiliaryTypes()
		 * auxiliaryTypes}. The auxiliaryType is used to look up the converter(s) to use
		 * to convert the individual parameter values. Maps may have different types for
		 * its keys and its values, so {@link #auxiliaryTypes() auxiliaryTypes} should
		 * provide two types: one for the map keys and one for the map values.</li>
		 * </ul>
		 * <p>
		 * {@code PositionalParamSpec} objects are used by the picocli command line
		 * interpreter and help message generator. Picocli can construct a
		 * {@code PositionalParamSpec} automatically from fields and methods with
		 * {@link Parameters @Parameters} annotations. Alternatively a
		 * {@code PositionalParamSpec} can be constructed programmatically.
		 * </p>
		 * <p>
		 * When a {@code PositionalParamSpec} is created from a
		 * {@link Parameters @Parameters} -annotated field or method, it is "bound" to
		 * that field or method: this field is set (or the method is invoked) when the
		 * position is matched and {@link #setValue(Object) setValue} is called.
		 * Programmatically constructed {@code PositionalParamSpec} instances will
		 * remember the value passed to the {@link #setValue(Object) setValue} method so
		 * it can be retrieved with the {@link #getValue() getValue} method. This
		 * behaviour can be customized by installing a custom {@link IGetter} and
		 * {@link ISetter} on the {@code PositionalParamSpec}.
		 * </p>
		 * 
		 * @since 3.0
		 */
		public static class PositionalParamSpec extends ArgSpec {
			/**
			 * Builder responsible for creating valid {@code PositionalParamSpec} objects.
			 * 
			 * @since 3.0
			 */
			public static class Builder extends ArgSpec.Builder<Builder> {
				private Range capacity;
				private Range index;

				private Builder() {
				}

				private Builder(IAnnotatedElement member, IFactory factory) {
					super(member.getAnnotation(Parameters.class), member, factory);
					index = Range.parameterIndex(member);
					capacity = Range.parameterCapacity(member);
				}

				private Builder(PositionalParamSpec original) {
					super(original);
					index = original.index;
					capacity = original.capacity;
				}

				/** Returns a valid {@code PositionalParamSpec} instance. */
				@Override
				public PositionalParamSpec build() {
					return new PositionalParamSpec(this);
				}

				Range capacity() {
					return capacity;
				}

				Builder capacity(Range capacity) {
					this.capacity = capacity;
					return self();
				}

				/**
				 * Returns an index or range specifying which of the command line arguments
				 * should be assigned to this positional parameter.
				 * 
				 * @see Parameters#index()
				 */
				public Range index() {
					return index;
				}

				/**
				 * Sets the index or range specifying which of the command line arguments should
				 * be assigned to this positional parameter, and returns this builder.
				 */
				public Builder index(Range index) {
					this.index = index;
					return self();
				}

				/**
				 * Sets the index or range specifying which of the command line arguments should
				 * be assigned to this positional parameter, and returns this builder.
				 */
				public Builder index(String range) {
					return index(Range.valueOf(range));
				}

				/** Returns this builder. */
				@Override
				protected Builder self() {
					return this;
				}
			}

			public static Builder builder() {
				return new Builder();
			}

			public static Builder builder(IAnnotatedElement source, IFactory factory) {
				return new Builder(source, factory);
			}

			/**
			 * Returns a Builder initialized from the specified {@code PositionalSpec}.
			 * 
			 * @since 4.0
			 */
			public static Builder builder(PositionalParamSpec original) {
				return new Builder(original);
			}

			private Range index;
			private Range capacity;
			private final Range builderCapacity;

			/**
			 * Ensures all attributes of this {@code PositionalParamSpec} have a valid
			 * value; throws an {@link InitializationException} if this cannot be achieved.
			 */
			private PositionalParamSpec(Builder builder) {
				super(builder);
				if (builder.index == null) {
					index = Range.defaultParameterIndex(typeInfo);
				} else {
					index = builder.index;
				}
				builderCapacity = builder.capacity;
				initCapacity();
				if (toString == null) {
					toString = "positional parameter[" + index() + "]";
				}
			}

			private Range capacity() {
				return capacity;
			}

			@Override
			public boolean equals(Object obj) {
				if (obj == this) {
					return true;
				}
				if (!(obj instanceof PositionalParamSpec)) {
					return false;
				}
				final PositionalParamSpec other = (PositionalParamSpec) obj;
				return super.equalsImpl(other) && Assert.equals(this.capacity, other.capacity)
						&& Assert.equals(this.index, other.index);
			}

			/**
			 * Returns the additional lookup keys for finding description lines in the
			 * resource bundle for this positional parameter.
			 * 
			 * @return a collection with the following single value:
			 *         {@code paramLabel() + "[" + index() + "]"}.
			 * @since 4.0
			 */
			@Override
			protected Collection<String> getAdditionalDescriptionKeys() {
				return Collections.singletonList(paramLabel() + "[" + index() + "]");
			}

			@Override
			public int hashCode() {
				return super.hashCodeImpl() + 37 * Assert.hashCode(capacity) + 37 * Assert.hashCode(index);
			}

			/**
			 * Returns an index or range specifying which of the command line arguments
			 * should be assigned to this positional parameter.
			 * 
			 * @see Parameters#index()
			 */
			public Range index() {
				return index;
			}

			private void initCapacity() {
				capacity = builderCapacity == null ? Range.parameterCapacity(arity(), index) : builderCapacity;
			}

			@Override
			public boolean isOption() {
				return false;
			}

			@Override
			public boolean isPositional() {
				return true;
			}

			/**
			 * Returns a new Builder initialized with the attributes from this
			 * {@code PositionalParamSpec}. Calling {@code build} immediately will return a
			 * copy of this {@code PositionalParamSpec}.
			 * 
			 * @return a builder that can create a copy of this spec
			 */
			public Builder toBuilder() {
				return new Builder(this);
			}
		}

		private interface Predicate<T> {
			boolean test(T t);
		}

		static class RuntimeTypeInfo implements ITypeInfo {
			final static String ERRORMSG = "Unsupported generic type %s. Only List<T>, Map<K,V>, Optional<T>, and Map<K, Optional<V>> are supported. Type parameters may be char[], a non-array type, or a wildcard type with an upper or lower bound.";

			public static ITypeInfo create(Class<?> type, Class<?>[] auxiliaryTypes,
					List<String> actualGenericTypeArguments, Range arity, Class<?> defaultType, boolean interactive) {
				if (type == null) {
					if (auxiliaryTypes == null || auxiliaryTypes.length == 0) {
						if (interactive) {
							type = char[].class;
						} else if (arity.isVariable || arity.max > 1) {
							type = String[].class;
						} else if (arity.max == 1) {
							type = String.class;
						} else {
							type = defaultType;
						}
					} else {
						type = auxiliaryTypes[0];
					}
				}
				if (auxiliaryTypes == null || auxiliaryTypes.length == 0) {
					if (type.isArray()) {
						if (type.equals(char[].class)) {
							auxiliaryTypes = new Class<?>[] { char[].class }; // TODO is this still needed?
						} else {
							auxiliaryTypes = new Class<?>[] { type.getComponentType() };
						}
					} else if (Collection.class.isAssignableFrom(type)) { // type is a collection but element type is
																			// unspecified
						auxiliaryTypes = new Class<?>[] { interactive ? char[].class : String.class }; // use String
																										// elements
					} else if (Map.class.isAssignableFrom(type)) { // type is a map but element type is unspecified
						auxiliaryTypes = new Class<?>[] { String.class, String.class }; // use String keys and String
																						// values
					} else {
						auxiliaryTypes = new Class<?>[] { type };
					}
				}
				return new RuntimeTypeInfo(type, auxiliaryTypes, actualGenericTypeArguments);
			}

			public static ITypeInfo create(Class<?> type, Class<?>[] annotationTypes, Type genericType, Range arity,
					Class<?> defaultType, boolean interactive) {
				final Class<?>[] auxiliaryTypes = RuntimeTypeInfo.inferTypes(type, annotationTypes, genericType);
				final List<String> actualGenericTypeArguments = new ArrayList<String>();
				if (genericType instanceof ParameterizedType) {
					final Class<?>[] declaredTypeParameters = extractTypeParameters((ParameterizedType) genericType);
					for (final Class<?> c : declaredTypeParameters) {
						actualGenericTypeArguments.add(c.getName());
					}
				}
				return create(type, auxiliaryTypes, actualGenericTypeArguments, arity, defaultType, interactive);
			}

			static ITypeInfo createForAuxType(Class<?> type) {
				return create(type, new Class<?>[0], (Type) null, Range.valueOf("1"), String.class, false);
			}

			@SuppressWarnings("rawtypes")
			static Class<?>[] extractTypeParameters(ParameterizedType genericType) {
				final Type[] paramTypes = genericType.getActualTypeArguments(); // e.g. ? extends Number
				final List<Class<?>> result = new ArrayList<Class<?>>();
				for (int i = 0; i < paramTypes.length; i++) {
					if (paramTypes[i] instanceof Class) {// e.g. Long
						result.add((Class<?>) paramTypes[i]);
						continue;
					} else if (paramTypes[i] instanceof ParameterizedType) { // e.g. Optional<Integer>
						final ParameterizedType parameterizedParamType = (ParameterizedType) paramTypes[i];
						final Type raw = parameterizedParamType.getRawType();
						if (raw instanceof Class) {
							result.add((Class) raw);
							if (i == 1 && CommandLine.isOptional((Class) raw)) { // #1108 and #1214
								final Class<?>[] aux = extractTypeParameters(parameterizedParamType);
								if (aux.length == 1) {
									result.add(aux[0]);
									continue;
								}
							}
							continue;
						}
					} else if (paramTypes[i] instanceof WildcardType) { // e.g. ? extends Number
						final WildcardType wildcardType = (WildcardType) paramTypes[i];
						final Type[] lower = wildcardType.getLowerBounds(); // e.g. []
						if (lower.length > 0 && lower[0] instanceof Class) {
							result.add((Class<?>) lower[0]);
							continue;
						}
						final Type[] upper = wildcardType.getUpperBounds(); // e.g. Number
						if (upper.length > 0 && upper[0] instanceof Class) {
							result.add((Class<?>) upper[0]);
							continue;
						}
					} else if (paramTypes[i] instanceof GenericArrayType) {
						final GenericArrayType gat = (GenericArrayType) paramTypes[i];
						if (char.class.equals(gat.getGenericComponentType())) {
							result.add(char[].class);
							continue;
						}
					}
					throw new InitializationException(String.format(ERRORMSG, genericType));
//                    // too convoluted generic type, giving up
//                    result.clear();
//                    result.addAll(Arrays.asList(String.class, String.class));
//                    break;
				}
				return result.toArray(new Class<?>[0]); // we inferred all types from ParameterizedType
			}

			static Class<?>[] inferTypes(Class<?> propertyType, Class<?>[] annotationTypes, Type genericType) {
				if (annotationTypes != null && annotationTypes.length > 0) {
					return annotationTypes;
				}
				if (propertyType.isArray()) {
					if (CommandLine.isOptional(propertyType.getComponentType())) { // #1108
						throw new InitializationException(String.format(ERRORMSG, genericType));
						// List<Class<?>> result = new ArrayList<Class<?>>();
						// result.add(propertyType.getComponentType());
						// GenericArrayType genericComponentType = (GenericArrayType) genericType;
						// result.addAll(Arrays.asList(extractTypeParameters((ParameterizedType)
						// genericComponentType.getGenericComponentType())));
						// return result.toArray(new Class[0]);
					}
					return new Class<?>[] { propertyType.getComponentType() };
				}
				if (CommandLine.isMultiValue(propertyType) || CommandLine.isOptional(propertyType)) { // #1108
					if (genericType instanceof ParameterizedType) {// e.g. Map<Long, ? extends Number>
						return extractTypeParameters((ParameterizedType) genericType);
					}
					return new Class<?>[] { String.class, String.class }; // field is multi-value but not
																			// ParameterizedType
				}
				return new Class<?>[] { propertyType }; // not a multi-value field
			}

			private final Class<?> type;

			private final Class<?>[] auxiliaryTypes;
			private final List<String> actualGenericTypeArguments;

			RuntimeTypeInfo(Class<?> type, Class<?>[] auxiliaryTypes, List<String> actualGenericTypeArguments) {
				this.type = Assert.notNull(type, "type");
				this.auxiliaryTypes = Assert.notNull(auxiliaryTypes, "auxiliaryTypes").clone();
				this.actualGenericTypeArguments = actualGenericTypeArguments == null ? Collections.<String>emptyList()
						: Collections.unmodifiableList(new ArrayList<String>(actualGenericTypeArguments));
			}

			@Override
			public boolean equals(Object obj) {
				if (obj == this) {
					return true;
				}
				if (!(obj instanceof RuntimeTypeInfo)) {
					return false;
				}
				final RuntimeTypeInfo other = (RuntimeTypeInfo) obj;
				return Arrays.equals(other.auxiliaryTypes, auxiliaryTypes) && type.equals(other.type);
			}

			@Override
			public List<String> getActualGenericTypeArguments() {
				return actualGenericTypeArguments;
			}

			@Override
			public List<ITypeInfo> getAuxiliaryTypeInfos() {
				final List<ITypeInfo> result = new ArrayList<ITypeInfo>();
				for (final Class<?> c : auxiliaryTypes) {
					result.add(createForAuxType(c));
				}
				return result;
			}

			@Override
			public Class<?>[] getAuxiliaryTypes() {
				return auxiliaryTypes;
			}

			@Override
			public String getClassName() {
				return type.getName();
			}

			@Override
			public String getClassSimpleName() {
				return type.getSimpleName();
			}

			@Override
			public List<String> getEnumConstantNames() {
				if (!isEnum()) {
					return Collections.emptyList();
				}
				final List<String> result = new ArrayList<String>();
				for (final Object c : auxiliaryTypes[0].getEnumConstants()) {
					result.add(c.toString());
				}
				return result;
			}

			@Override
			public Class<?> getType() {
				return type;
			}

			@Override
			public int hashCode() {
				return Arrays.hashCode(auxiliaryTypes) + 37 * Assert.hashCode(type);
			}

			@Override
			public boolean isArray() {
				return type.isArray() && type != char[].class;
			}

			@Override
			public boolean isBoolean() {
				return auxiliaryTypes[0] == boolean.class || auxiliaryTypes[0] == Boolean.class;
			}

			@Override
			public boolean isCollection() {
				return Collection.class.isAssignableFrom(type);
			}

			@Override
			public boolean isEnum() {
				return auxiliaryTypes[0].isEnum();
			}

			@Override
			public boolean isMap() {
				return Map.class.isAssignableFrom(type);
			}

			@Override
			public boolean isMultiValue() {
				return CommandLine.isMultiValue(type);
			}

			@Override
			public boolean isOptional() {
				return CommandLine.isOptional(type);
			}

			@Override
			public String toString() {
				return String.format("RuntimeTypeInfo(%s, aux=%s, collection=%s, map=%s)", type.getCanonicalName(),
						Arrays.toString(auxiliaryTypes), isCollection(), isMap());
			}
		}

		static class TypedMember implements IAnnotatedElement, IExtensible {
			static String abbreviate(String text) {
				return text.replace("private ", "").replace("protected ", "").replace("public ", "")
						.replace("java.lang.", "");
			}

			static TypedMember createIfAnnotated(Field field, IScope scope) {
				return isAnnotated(field) ? new TypedMember(field, scope) : null;
			}

			static TypedMember createIfAnnotated(Method method, IScope scope, CommandSpec spec) {
				return isAnnotated(method) ? new TypedMember(method, scope, spec) : null;
			}

			private static String decapitalize(String name) {
				if (name == null || name.length() == 0) {
					return name;
				}
				final char[] chars = name.toCharArray();
				chars[0] = Character.toLowerCase(chars[0]);
				return new String(chars);
			}

			static boolean isAnnotated(AnnotatedElement e) {
				return false || e.isAnnotationPresent(Option.class) || e.isAnnotationPresent(Parameters.class)
						|| e.isAnnotationPresent(ArgGroup.class) || e.isAnnotationPresent(Unmatched.class)
						|| e.isAnnotationPresent(Mixin.class) || e.isAnnotationPresent(Spec.class)
						|| e.isAnnotationPresent(ParentCommand.class);
			}

			static String propertyName(String methodName) {
				if (methodName.length() > 3 && (methodName.startsWith("get") || methodName.startsWith("set"))) {
					return decapitalize(methodName.substring(3));
				}
				return decapitalize(methodName);
			}

			final AccessibleObject accessible;
			final String name;
			final ITypeInfo typeInfo;
			private final InitialValueState initialValueState;
			private IScope scope;
			private IGetter getter;
			private ISetter setter;

			TypedMember(Field field) {
				accessible = Assert.notNull(field, "field");
				accessible.setAccessible(true);
				name = field.getName();
				typeInfo = createTypeInfo(field.getType(), field.getGenericType());
				initialValueState = InitialValueState.POSTPONED;
			}

			private TypedMember(Field field, IScope scope) {
				this(field);
				if (ObjectScope.isProxyClass(scope)) {
					throw new InitializationException("Invalid picocli annotation on interface field");
				}
				final FieldBinding binding = new FieldBinding(scope, field);
				getter = binding;
				setter = binding;
				this.scope = scope;
			}

			private TypedMember(Method method, IScope scope, CommandSpec spec) {
				this.scope = scope;
				accessible = Assert.notNull(method, "method");
				accessible.setAccessible(true);
				name = propertyName(method.getName());
				final Class<?>[] parameterTypes = method.getParameterTypes();
				if (parameterTypes.length > 0) {
					// accepts arguments, so must be a setter
					typeInfo = createTypeInfo(parameterTypes[0], method.getGenericParameterTypes()[0]);
					final MethodBinding binding = new MethodBinding(scope, method, spec);
					getter = binding;
					setter = binding;
					initialValueState = InitialValueState.UNAVAILABLE; // arg is setter method;
				} else if (method.getReturnType() == Void.TYPE || method.getReturnType() == Void.class) {
					// neither accepts arguments, nor returns non-void, so cannot be a setter or a
					// getter, respectively
					throw new InitializationException("Invalid method, must be either getter or setter: " + method);
				} else {
					// does not accept arguments, but returns non-void, so is a getter
					typeInfo = createTypeInfo(method.getReturnType(), method.getGenericReturnType());
					if (ObjectScope.isProxyClass(scope)) {
						final Object proxy = ObjectScope.tryGet(scope);
						final PicocliInvocationHandler handler = (PicocliInvocationHandler) Proxy
								.getInvocationHandler(proxy);
						final PicocliInvocationHandler.ProxyBinding binding = handler.new ProxyBinding(method);
						getter = binding;
						setter = binding;
						initializeInitialValue(method); // arg is an interface method; we have to create initial values
														// ourselves
					} else {
						// throw new IllegalArgumentException("Getter method but not a proxy: " + scope
						// + ": " + method);
						final MethodBinding binding = new MethodBinding(scope, method, spec);
						getter = binding;
						setter = binding;
					}
					initialValueState = InitialValueState.POSTPONED; // the initial value can be obtained from the
																		// getter
				}
			}

			TypedMember(MethodParam param, IScope scope) {
				this.scope = scope;
				accessible = Assert.notNull(param, "command method parameter");
				accessible.setAccessible(true);
				name = param.getName();
				typeInfo = createTypeInfo(param.getType(), param.getParameterizedType());
				// bind parameter
				final ObjectBinding binding = new ObjectBinding();
				getter = binding;
				setter = binding;
				initializeInitialValue(param); // arg is a method param; we have to create initial values ourselves
				initialValueState = InitialValueState.POSTPONED; // the initial value can be obtained from the getter
			}

			private Class<?>[] annotationTypes() {
				if (isOption()) {
					return getAnnotation(Option.class).type();
				}
				if (isParameter()) {
					return getAnnotation(Parameters.class).type();
				}
				return new Class<?>[0];
			}

			private ITypeInfo createTypeInfo(Class<?> type, Type genericType) {
				Range arity = null;
				if (isOption()) {
					arity = Range.valueOf(getAnnotation(Option.class).arity());
				}
				if (isParameter()) {
					arity = Range.valueOf(getAnnotation(Parameters.class).arity());
				}
				if (arity == null || arity.isUnspecified) {
					if (isOption()) {
						arity = (type == null || isBoolean(type)) ? Range.valueOf("0") : Range.valueOf("1");
					} else {
						arity = Range.valueOf("1");
					}
					arity = arity.unspecified(true);
				}
				return RuntimeTypeInfo.create(type, annotationTypes(), genericType, arity,
						(isOption() ? boolean.class : String.class), isInteractive());
			}

			@Override
			public <T extends Annotation> T getAnnotation(Class<T> annotationClass) {
				return accessible.getAnnotation(annotationClass);
			}

			public Class<?>[] getAuxiliaryTypes() {
				return typeInfo.getAuxiliaryTypes();
			}

			@Override
			public <T> T getExtension(Class<T> cls) {
				if (cls == InitialValueState.class) {
					return cls.cast(initialValueState);
				}
				return null;
			}

			@Override
			public int getMethodParamPosition() {
				return isMethodParameter() ? ((MethodParam) accessible).position : -1;
			}

			@Override
			public String getMixinName() {
				final String annotationName = getAnnotation(Mixin.class).name();
				return empty(annotationName) ? getName() : annotationName;
			}

			@Override
			public String getName() {
				return name;
			}

			@Override
			public IGetter getter() {
				return getter;
			}

			@Override
			public String getToString() {
				if (isMixin()) {
					return abbreviate("mixin from member " + toGenericString());
				}
				return (accessible instanceof Field ? "field "
						: accessible instanceof Method ? "method " : accessible.getClass().getSimpleName() + " ")
						+ abbreviate(toGenericString());
			}

			public Class<?> getType() {
				return typeInfo.getType();
			}

			@Override
			public ITypeInfo getTypeInfo() {
				return typeInfo;
			}

			@Override
			public boolean hasInitialValue() {
				return initialValueState == InitialValueState.CACHED
						|| initialValueState == InitialValueState.POSTPONED;
			}

			private void initializeInitialValue(Object arg) {
				final Class<?> type = typeInfo.getType();
				try {
					if (type == Boolean.TYPE) {
						setter.set(false);
					} else if (type == Byte.TYPE) {
						setter.set((byte) 0);
					} else if (type == Character.TYPE) {
						setter.set((char) 0);
					} else if (type == Short.TYPE) {
						setter.set((short) 0);
					} else if (type == Integer.TYPE) {
						setter.set(0);
					} else if (type == Long.TYPE) {
						setter.set(0L);
					} else if (type == Float.TYPE) {
						setter.set(0f);
					} else if (type == Double.TYPE) {
						setter.set(0d);
					} else {
						setter.set(null);
					}
				} catch (final Exception ex) {
					throw new InitializationException("Could not set initial value for " + arg + ": " + ex.toString(),
							ex);
				}
			}

			@Override
			public boolean isAnnotationPresent(Class<? extends Annotation> annotationClass) {
				return accessible.isAnnotationPresent(annotationClass);
			}

			@Override
			public boolean isArgGroup() {
				return isAnnotationPresent(ArgGroup.class);
			}

			@Override
			public boolean isArgSpec() {
				return isOption() || isParameter() || (isMethodParameter() && !isMixin());
			}

			@Override
			public boolean isInteractive() {
				return (isOption() && getAnnotation(Option.class).interactive())
						|| (isParameter() && getAnnotation(Parameters.class).interactive());
			}

			@Override
			public boolean isMethodParameter() {
				return accessible instanceof MethodParam;
			}

			@Override
			public boolean isMixin() {
				return isAnnotationPresent(Mixin.class);
			}

			@Override
			public boolean isMultiValue() {
				return CommandLine.isMultiValue(getType());
			}

			@Override
			public boolean isOption() {
				return isAnnotationPresent(Option.class);
			}

			@Override
			public boolean isParameter() {
				return isAnnotationPresent(Parameters.class);
			}

			@Override
			public boolean isParentCommand() {
				return isAnnotationPresent(ParentCommand.class);
			}

			@Override
			public boolean isSpec() {
				return isAnnotationPresent(Spec.class);
			}

			@Override
			public boolean isUnmatched() {
				return isAnnotationPresent(Unmatched.class);
			}

			@Override
			public IScope scope() {
				return scope;
			}

			@Override
			public ISetter setter() {
				return setter;
			}

			public String toGenericString() {
				return accessible instanceof Field ? ((Field) accessible).toGenericString()
						: accessible instanceof Method ? ((Method) accessible).toGenericString()
								: accessible.toString();
			}

			@Override
			public String toString() {
				return accessible.toString();
			}

			@Override
			public Object userObject() {
				return accessible;
			}
		}

		/**
		 * This class allows applications to specify a custom binding that will be
		 * invoked for unmatched arguments. A binding can be created with a
		 * {@code ISetter} that consumes the unmatched arguments {@code String[]}, or
		 * with a {@code IGetter} that produces a {@code Collection<String>} that the
		 * unmatched arguments can be added to.
		 * 
		 * @since 3.0
		 */
		public static class UnmatchedArgsBinding {
			/**
			 * Creates a {@code UnmatchedArgsBinding} for a setter that consumes
			 * {@code String[]} objects.
			 * 
			 * @param setter consumes the String[] array with unmatched arguments.
			 */
			public static UnmatchedArgsBinding forStringArrayConsumer(ISetter setter) {
				return new UnmatchedArgsBinding(null, setter);
			}

			/**
			 * Creates a {@code UnmatchedArgsBinding} for a getter that produces a
			 * {@code Collection<String>} that the unmatched arguments can be added to.
			 * 
			 * @param getter supplies a {@code Collection<String>} that the unmatched
			 *               arguments can be added to.
			 */
			public static UnmatchedArgsBinding forStringCollectionSupplier(IGetter getter) {
				return new UnmatchedArgsBinding(getter, null);
			}

			private final IGetter getter;

			private final ISetter setter;

			private Object initialValue;

			private UnmatchedArgsBinding(IGetter getter, ISetter setter) {
				if (getter == null && setter == null) {
					throw new IllegalArgumentException("Getter and setter cannot both be null");
				}
				this.setter = setter;
				this.getter = getter;
				final IGetter initialValueHolder = (setter instanceof IGetter) ? (IGetter) setter : getter;
				try {
					initialValue = initialValueHolder.get();
				} catch (final Exception ex) {
					CommandLine.tracer().debug("Could not obtain initial value for unmatched from %s",
							initialValueHolder);
				}
			}

			void addAll(String[] unmatched) {
				if (setter != null) {
					try {
						setter.set(unmatched);
					} catch (final Exception ex) {
						throw new PicocliException(
								String.format("Could not invoke setter (%s) with unmatched argument array '%s': %s",
										setter, Arrays.toString(unmatched), ex),
								ex);
					}
				} else {
					try {
						Collection<String> collection = getter.get();
						if (collection == null) {
							collection = new ArrayList<String>();
							((ISetter) getter).set(collection);
						}
						collection.addAll(Arrays.asList(unmatched));
					} catch (final Exception ex) {
						throw new PicocliException(String.format(
								"Could not add unmatched argument array '%s' to collection returned by getter (%s): %s",
								Arrays.toString(unmatched), getter, ex), ex);
					}
				}
			}

			void clear() {
				ISetter initialValueHolder = setter;
				if (initialValueHolder == null) {
					if (getter instanceof ISetter) {
						initialValueHolder = (ISetter) getter;
					} else {
						CommandLine.tracer().warn("Unable to clear %s: it does not implement ISetter", getter);
						return;
					}
				}
				try {
					initialValueHolder.set(initialValue);
				} catch (final Exception ex) {
					throw new PicocliException(String.format("Could not invoke setter (%s) with initial value: %s",
							initialValueHolder, ex), ex);
				}
			}

			/**
			 * Returns the getter responsible for producing a {@code Collection} that the
			 * unmatched arguments can be added to.
			 */
			public IGetter getter() {
				return getter;
			}

			/** Returns the setter responsible for consuming the unmatched arguments. */
			public ISetter setter() {
				return setter;
			}
		}

		/**
		 * Models the usage help message specification and can be used to customize the
		 * usage help message.
		 * <p>
		 * This class provides two ways to customize the usage help message:
		 * </p>
		 * <ul>
		 * <li>Change the text of the predefined sections (this may also be done
		 * declaratively using the annotations)</li>
		 * <li>Add custom sections, or remove or re-order predefined sections</li>
		 * </ul>
		 * <p>
		 * The pre-defined sections have getters and setters that return a String (or
		 * array of Strings). For example: {@link #description()} and
		 * {@link #description(String...)} or {@link #header()} and
		 * {@link #header(String...)}.
		 * </p>
		 * <p>
		 * Changing the section order, or adding custom sections can be accomplished
		 * with {@link #sectionKeys(List)} and {@link #sectionMap(Map)}. This gives
		 * complete freedom on how a usage help message section is rendered, but it also
		 * means that the {@linkplain IHelpSectionRenderer section renderer} is
		 * responsible for all aspects of rendering the section, including layout and
		 * emitting ANSI escape codes. The {@link Help.TextTable} and
		 * {@link Help.Ansi.Text} classes, and the
		 * {@link CommandLine.Help.Ansi#string(String)} and
		 * {@link CommandLine.Help.Ansi#text(String)} methods may be useful.
		 * </p>
		 * <p>
		 * The usage help message is created more or less like this:
		 * </p>
		 * 
		 * <pre>
		 * // CommandLine.usage(...) or CommandLine.getUsageMessage(...)
		 * Help.ColorScheme colorScheme = Help.defaultColorScheme(Help.Ansi.AUTO);
		 * Help help = getHelpFactory().create(getCommandSpec(), colorScheme)
		 * StringBuilder result = new StringBuilder();
		 * for (String key : getHelpSectionKeys()) {
		 *     IHelpSectionRenderer renderer = getHelpSectionMap().get(key);
		 *     if (renderer != null) { result.append(renderer.render(help)); }
		 * }
		 * // return or print result
		 * </pre>
		 * <p>
		 * Where the default {@linkplain #sectionMap() help section map} is constructed
		 * like this:
		 * </p>
		 * 
		 * <pre>{@code
		 * // The default section renderers delegate to methods in Help for their implementation
		 * // (using Java 8 lambda notation for brevity):
		 * Map<String, IHelpSectionRenderer> sectionMap = new HashMap<>();
		 * sectionMap.put(SECTION_KEY_HEADER_HEADING, help -> help.headerHeading());
		 * sectionMap.put(SECTION_KEY_HEADER, help -> help.header());
		 * sectionMap.put(SECTION_KEY_SYNOPSIS_HEADING, help -> help.synopsisHeading()); // e.g. Usage:
		 * sectionMap.put(SECTION_KEY_SYNOPSIS, help -> help.synopsis(help.synopsisHeadingLength())); // e.g. <cmd> [OPTIONS]
		 * 																							// <subcmd>
		 * 																							// [COMMAND-OPTIONS]
		 * 																							// [ARGUMENTS]
		 * sectionMap.put(SECTION_KEY_DESCRIPTION_HEADING, help -> help.descriptionHeading()); // e.g. %nDescription:%n%n
		 * sectionMap.put(SECTION_KEY_DESCRIPTION, help -> help.description()); // e.g. {"Converts foos to bars.", "Use options
		 * 																		// to control conversion mode."}
		 * sectionMap.put(SECTION_KEY_PARAMETER_LIST_HEADING, help -> help.parameterListHeading()); // e.g. %nPositional
		 * 																							// parameters:%n%n
		 * sectionMap.put(SECTION_KEY_PARAMETER_LIST, help -> help.parameterList()); // e.g. [FILE...] the files to convert
		 * sectionMap.put(SECTION_KEY_OPTION_LIST_HEADING, help -> help.optionListHeading()); // e.g. %nOptions:%n%n
		 * sectionMap.put(SECTION_KEY_OPTION_LIST, help -> help.optionList()); // e.g. -h, --help displays this help and exits
		 * sectionMap.put(SECTION_KEY_COMMAND_LIST_HEADING, help -> help.commandListHeading()); // e.g. %nCommands:%n%n
		 * sectionMap.put(SECTION_KEY_COMMAND_LIST, help -> help.commandList()); // e.g. add adds the frup to the frooble
		 * sectionMap.put(SECTION_KEY_EXIT_CODE_LIST_HEADING, help -> help.exitCodeListHeading());
		 * sectionMap.put(SECTION_KEY_EXIT_CODE_LIST, help -> help.exitCodeList());
		 * sectionMap.put(SECTION_KEY_FOOTER_HEADING, help -> help.footerHeading());
		 * sectionMap.put(SECTION_KEY_FOOTER, help -> help.footer());
		 * }</pre>
		 *
		 * @since 3.0
		 */
		public static class UsageMessageSpec {

			/**
			 * {@linkplain #sectionKeys() Section key} to {@linkplain #sectionMap() control}
			 * the {@linkplain IHelpSectionRenderer section renderer} for the Header Heading
			 * section. The default renderer for this section calls
			 * {@link Help#headerHeading(Object...)}.
			 * 
			 * @since 3.9
			 */
			public static final String SECTION_KEY_HEADER_HEADING = "headerHeading";

			/**
			 * {@linkplain #sectionKeys() Section key} to {@linkplain #sectionMap() control}
			 * the {@linkplain IHelpSectionRenderer section renderer} for the Header
			 * section. The default renderer for this section calls
			 * {@link Help#header(Object...)}.
			 * 
			 * @since 3.9
			 */
			public static final String SECTION_KEY_HEADER = "header";

			/**
			 * {@linkplain #sectionKeys() Section key} to {@linkplain #sectionMap() control}
			 * the {@linkplain IHelpSectionRenderer section renderer} for the Synopsis
			 * Heading section. The default renderer for this section calls
			 * {@link Help#synopsisHeading(Object...)}.
			 * 
			 * @since 3.9
			 */
			public static final String SECTION_KEY_SYNOPSIS_HEADING = "synopsisHeading";

			/**
			 * {@linkplain #sectionKeys() Section key} to {@linkplain #sectionMap() control}
			 * the {@linkplain IHelpSectionRenderer section renderer} for the Synopsis
			 * section. The default renderer for this section calls
			 * {@link Help#synopsis(int)}.
			 * 
			 * @since 3.9
			 */
			public static final String SECTION_KEY_SYNOPSIS = "synopsis";

			/**
			 * {@linkplain #sectionKeys() Section key} to {@linkplain #sectionMap() control}
			 * the {@linkplain IHelpSectionRenderer section renderer} for the Description
			 * Heading section. The default renderer for this section calls
			 * {@link Help#descriptionHeading(Object...)}.
			 * 
			 * @since 3.9
			 */
			public static final String SECTION_KEY_DESCRIPTION_HEADING = "descriptionHeading";

			/**
			 * {@linkplain #sectionKeys() Section key} to {@linkplain #sectionMap() control}
			 * the {@linkplain IHelpSectionRenderer section renderer} for the Description
			 * section. The default renderer for this section calls
			 * {@link Help#description(Object...)}.
			 * 
			 * @since 3.9
			 */
			public static final String SECTION_KEY_DESCRIPTION = "description";

			/**
			 * {@linkplain #sectionKeys() Section key} to {@linkplain #sectionMap() control}
			 * the {@linkplain IHelpSectionRenderer section renderer} for the Parameter List
			 * Heading section. The default renderer for this section calls
			 * {@link Help#parameterListHeading(Object...)}.
			 * 
			 * @since 3.9
			 */
			public static final String SECTION_KEY_PARAMETER_LIST_HEADING = "parameterListHeading";

			/**
			 * {@linkplain #sectionKeys() Section key} to {@linkplain #sectionMap() control}
			 * the {@linkplain IHelpSectionRenderer section renderer} for the @-file
			 * parameter list section. The default renderer for this section calls
			 * {@link Help#atFileParameterList()}.
			 * 
			 * @since 4.2
			 */
			public static final String SECTION_KEY_AT_FILE_PARAMETER = "atFileParameterList";

			/**
			 * {@linkplain #sectionKeys() Section key} to {@linkplain #sectionMap() control}
			 * the {@linkplain IHelpSectionRenderer section renderer} for the Parameter List
			 * section. The default renderer for this section calls
			 * {@link Help#parameterList()}.
			 * 
			 * @since 3.9
			 */
			public static final String SECTION_KEY_PARAMETER_LIST = "parameterList";

			/**
			 * {@linkplain #sectionKeys() Section key} to {@linkplain #sectionMap() control}
			 * the {@linkplain IHelpSectionRenderer section renderer} for the Option List
			 * Heading section. The default renderer for this section calls
			 * {@link Help#optionListHeading(Object...)}.
			 * 
			 * @since 3.9
			 */
			public static final String SECTION_KEY_OPTION_LIST_HEADING = "optionListHeading";

			/**
			 * {@linkplain #sectionKeys() Section key} to {@linkplain #sectionMap() control}
			 * the {@linkplain IHelpSectionRenderer section renderer} for the Option List
			 * section. The default renderer for this section calls
			 * {@link Help#optionList()}.
			 * 
			 * @since 3.9
			 */
			public static final String SECTION_KEY_OPTION_LIST = "optionList";

			/**
			 * {@linkplain #sectionKeys() Section key} to {@linkplain #sectionMap() control}
			 * the {@linkplain IHelpSectionRenderer section renderer} for the {@code --} End
			 * of Options list section. The default renderer for this section calls
			 * {@link Help#endOfOptionsList()}.
			 * 
			 * @since 4.3
			 */
			public static final String SECTION_KEY_END_OF_OPTIONS = "endOfOptionsList";

			/**
			 * {@linkplain #sectionKeys() Section key} to {@linkplain #sectionMap() control}
			 * the {@linkplain IHelpSectionRenderer section renderer} for the Subcommand
			 * List Heading section. The default renderer for this section calls
			 * {@link Help#commandListHeading(Object...)}.
			 * 
			 * @since 3.9
			 */
			public static final String SECTION_KEY_COMMAND_LIST_HEADING = "commandListHeading";

			/**
			 * {@linkplain #sectionKeys() Section key} to {@linkplain #sectionMap() control}
			 * the {@linkplain IHelpSectionRenderer section renderer} for the Subcommand
			 * List section. The default renderer for this section calls
			 * {@link Help#commandList()}.
			 * 
			 * @since 3.9
			 */
			public static final String SECTION_KEY_COMMAND_LIST = "commandList";

			/**
			 * {@linkplain #sectionKeys() Section key} to {@linkplain #sectionMap() control}
			 * the {@linkplain IHelpSectionRenderer section renderer} for the Exit Code List
			 * Heading section. The default renderer for this section calls
			 * {@link Help#exitCodeListHeading(Object...)}.
			 * 
			 * @since 4.0
			 */
			public static final String SECTION_KEY_EXIT_CODE_LIST_HEADING = "exitCodeListHeading";

			/**
			 * {@linkplain #sectionKeys() Section key} to {@linkplain #sectionMap() control}
			 * the {@linkplain IHelpSectionRenderer section renderer} for the Exit Code List
			 * section. The default renderer for this section calls
			 * {@link Help#exitCodeList()}.
			 * 
			 * @since 4.0
			 */
			public static final String SECTION_KEY_EXIT_CODE_LIST = "exitCodeList";

			/**
			 * {@linkplain #sectionKeys() Section key} to {@linkplain #sectionMap() control}
			 * the {@linkplain IHelpSectionRenderer section renderer} for the Footer Heading
			 * section. The default renderer for this section calls
			 * {@link Help#footerHeading(Object...)}.
			 * 
			 * @since 3.9
			 */
			public static final String SECTION_KEY_FOOTER_HEADING = "footerHeading";

			/**
			 * {@linkplain #sectionKeys() Section key} to {@linkplain #sectionMap() control}
			 * the {@linkplain IHelpSectionRenderer section renderer} for the Footer
			 * section. The default renderer for this section calls
			 * {@link Help#footer(Object...)}.
			 * 
			 * @since 3.9
			 */
			public static final String SECTION_KEY_FOOTER = "footer";

			/** Constant holding the default usage message width: <code>{@value}</code>. */
			public final static int DEFAULT_USAGE_WIDTH = 80;
			private final static int MINIMUM_USAGE_WIDTH = 55;
			final static int DEFAULT_USAGE_LONG_OPTIONS_WIDTH = 20;
			private final static int DEFAULT_SYNOPSIS_INDENT = -1; // by default, fall back to aligning to the synopsis
																	// heading
			private final static double DEFAULT_SYNOPSIS_AUTO_INDENT_THRESHOLD = 0.5;
			private final static double MAX_SYNOPSIS_AUTO_INDENT_THRESHOLD = 0.9;

			/**
			 * Constant Boolean holding the default setting for whether to attempt to adjust
			 * the width to the terminal width: <code>{@value}</code>.
			 */
			static final Boolean DEFAULT_USAGE_AUTO_WIDTH = Boolean.FALSE;

			/**
			 * Constant String holding the default synopsis heading: <code>{@value}</code>.
			 */
			static final String DEFAULT_SYNOPSIS_HEADING = "Usage: ";

			/**
			 * Constant String holding the default synopsis subcommands:
			 * <code>{@value}</code>.
			 */
			static final String DEFAULT_SYNOPSIS_SUBCOMMANDS = "[COMMAND]";

			/**
			 * Constant String holding the default command list heading:
			 * <code>{@value}</code>.
			 */
			static final String DEFAULT_COMMAND_LIST_HEADING = "Commands:%n";

			/**
			 * Constant String holding the default string that separates options from option
			 * parameters: {@code ' '} ({@value}).
			 */
			static final char DEFAULT_REQUIRED_OPTION_MARKER = ' ';

			/**
			 * Constant Boolean holding the default setting for whether to abbreviate the
			 * synopsis: <code>{@value}</code>.
			 */
			static final Boolean DEFAULT_ABBREVIATE_SYNOPSIS = Boolean.FALSE;

			/**
			 * Constant Boolean holding the default setting for whether to sort the options
			 * alphabetically: <code>{@value}</code>.
			 */
			static final Boolean DEFAULT_SORT_OPTIONS = Boolean.TRUE;

			/**
			 * Constant Boolean holding the default setting for whether to sort options in
			 * the synopsis alphabetically: <code>{@value}</code>.
			 */
			static final Boolean DEFAULT_SORT_SYNOPSIS = Boolean.TRUE;

			/**
			 * Constant Boolean holding the default setting for whether to show an entry
			 * for @-files in the usage help message.
			 */
			static final Boolean DEFAULT_SHOW_AT_FILE = Boolean.FALSE;

			/**
			 * Constant Boolean holding the default setting for whether to show an entry for
			 * the {@code --} End of Options delimiter in the usage help message.
			 */
			static final Boolean DEFAULT_SHOW_END_OF_OPTIONS = Boolean.FALSE;

			/**
			 * Constant Boolean holding the default setting for whether to show default
			 * values in the usage help message: <code>{@value}</code>.
			 */
			static final Boolean DEFAULT_SHOW_DEFAULT_VALUES = Boolean.FALSE;

			/**
			 * Constant Boolean holding the default setting for whether this command should
			 * be listed in the usage help of the parent command: <code>{@value}</code>.
			 */
			static final Boolean DEFAULT_HIDDEN = Boolean.FALSE;

			/**
			 * Constant Boolean holding the default setting for whether line breaks should
			 * take wide CJK characters into account: <code>{@value}</code>.
			 */
			static final Boolean DEFAULT_ADJUST_CJK = Boolean.TRUE;

			static final String DEFAULT_SINGLE_VALUE = "";
			static final String[] DEFAULT_MULTI_LINE = {};

			private static int detectTerminalWidth() {
				final long start = System.nanoTime();
				final Tracer tracer = CommandLine.tracer();
				final AtomicInteger size = new AtomicInteger(-1);
				final String[] cmd = (Help.Ansi.isWindows() && !Help.Ansi.isPseudoTTY())
						? new String[] { "cmd.exe", "/c", "mode con" }
						: (Help.Ansi.isMac() ? new String[] { "tput", "cols" }
								: new String[] { "stty", "-a", "-F", "/dev/tty" });
				final Thread t = new Thread(new Runnable() {
					@Override
					public void run() {
						Process proc = null;
						BufferedReader reader = null;
						try {
							final ProcessBuilder pb = new ProcessBuilder(cmd);
							tracer.debug("getTerminalWidth() executing command %s", pb.command());
							// proc = Runtime.getRuntime().exec(new String[] { "sh", "-c", "tput cols 2>
							// /dev/tty" });
							final Class<?> redirectClass = Class.forName("java.lang.ProcessBuilder$Redirect");
							final Object INHERIT = redirectClass.getField("INHERIT").get(null);
							final Method redirectError = ProcessBuilder.class.getDeclaredMethod("redirectError",
									redirectClass);
							redirectError.invoke(pb, INHERIT);
							proc = pb.start();
							reader = new BufferedReader(new InputStreamReader(proc.getInputStream()));
							String txt = "";
							String line;
							while ((line = reader.readLine()) != null) {
								txt += " " + line;
							}
							tracer.debug("getTerminalWidth() parsing output: %s", txt);
							final Pattern pattern = (Help.Ansi.isWindows() && !Help.Ansi.isPseudoTTY())
									? Pattern.compile(".*?:\\s*(\\d+)\\D.*?:\\s*(\\d+)\\D.*", Pattern.DOTALL)
									: (Help.Ansi.isMac() ? Pattern.compile("(\\s*)(\\d+)\\s*")
											: Pattern.compile(".*olumns(:)?\\s+(\\d+)\\D.*", Pattern.DOTALL));
							final Matcher matcher = pattern.matcher(txt);
							if (matcher.matches()) {
								size.set(Integer.parseInt(matcher.group(2)));
							}
						} catch (final Exception ignored) { // nothing to do...
							tracer.debug("getTerminalWidth() ERROR: %s", ignored);
						} finally {
							if (proc != null) {
								proc.destroy();
							}
							close(reader);
						}
					}
				});
				t.start();
				final long now = System.currentTimeMillis();
				do {
					if (size.intValue() >= 0) {
						break;
					}
					try {
						Thread.sleep(25);
					} catch (final InterruptedException ignored) {
					}
				} while (System.currentTimeMillis() < now + 2000 && t.isAlive());
				final double duration = (System.nanoTime() - start) / 1000000.0;
				tracer.debug("getTerminalWidth() returning: %s in %,.1fms", size, duration);
				return size.intValue();
			}

			private static int getTerminalWidth() {
				return (Help.Ansi.isTTY() || Help.Ansi.isPseudoTTY()) ? detectTerminalWidth() : -1;
			}

			/**
			 * Given a codePoint, is this codePoint considered to be a CJK character?
			 * Shamelessly stolen from <a href=
			 * "http://stackoverflow.com/questions/1499804/how-can-i-detect-japanese-text-in-a-java-string">StackOverflow</a>
			 * where it was contributed by user Rakesh N. (Upvote! :-) )
			 * 
			 * @param codePoint code point to test
			 * @return {@code true} if the character is a CJK character
			 */
			static boolean isCodePointCJK(int codePoint) {
				final Character.UnicodeBlock unicodeBlock = Character.UnicodeBlock.of(codePoint);
				return (codePoint == 0x00b1 || unicodeBlock == Character.UnicodeBlock.HIRAGANA)
						|| (unicodeBlock == Character.UnicodeBlock.KATAKANA)
						|| (unicodeBlock == Character.UnicodeBlock.KATAKANA_PHONETIC_EXTENSIONS)
						|| (unicodeBlock == Character.UnicodeBlock.HANGUL_COMPATIBILITY_JAMO)
						|| (unicodeBlock == Character.UnicodeBlock.HANGUL_JAMO)
						|| (unicodeBlock == Character.UnicodeBlock.HANGUL_SYLLABLES)
						|| (unicodeBlock == Character.UnicodeBlock.CJK_UNIFIED_IDEOGRAPHS)
						|| (unicodeBlock == Character.UnicodeBlock.CJK_UNIFIED_IDEOGRAPHS_EXTENSION_A)
						|| (unicodeBlock == Character.UnicodeBlock.CJK_UNIFIED_IDEOGRAPHS_EXTENSION_B)
						|| (unicodeBlock == Character.UnicodeBlock.CJK_COMPATIBILITY_FORMS)
						|| (unicodeBlock == Character.UnicodeBlock.CJK_COMPATIBILITY_IDEOGRAPHS)
						|| (unicodeBlock == Character.UnicodeBlock.CJK_RADICALS_SUPPLEMENT)
						|| (unicodeBlock == Character.UnicodeBlock.CJK_SYMBOLS_AND_PUNCTUATION)
						|| (unicodeBlock == Character.UnicodeBlock.ENCLOSED_CJK_LETTERS_AND_MONTHS)
						// The magic number here is the separating index between full-width and
						// half-width
						|| (unicodeBlock == Character.UnicodeBlock.HALFWIDTH_AND_FULLWIDTH_FORMS && codePoint < 0xFF61);
			}

			private static boolean isNumeric(String userValue) {
				try {
					Integer.parseInt(userValue);
					return true;
				} catch (final Exception any) {
					return false;
				}
			}

			/**
			 * Creates and returns a {@code Map} that contains an entry for each specified
			 * String that is in {@code "key:value"} format.
			 * 
			 * @param entries the strings to process; values that are not in
			 *                {@code "key:value"} format are ignored
			 * @return a {@code Map} with an entry for each line, preserving the input order
			 * @since 4.0
			 */
			public static Map<String, String> keyValuesMap(String... entries) {
				final Map<String, String> result = new LinkedHashMap<String, String>();
				if (entries == null) {
					return result;
				}
				for (int i = 0; i < entries.length; i++) {
					final int pos = entries[i].indexOf(':');
					if (pos >= 0) {
						result.put(entries[i].substring(0, pos), entries[i].substring(pos + 1));
					} else {
						CommandLine.tracer().info("Ignoring line at index %d: cannot split '%s' into 'key:value'", i,
								entries[i]);
					}
				}
				return result;
			}

			private static boolean shouldDetectTerminalSize(boolean autoWidthEnabledInApplication) {
				final String userValue = System.getProperty("picocli.usage.width");
				final boolean sysPropAutoWidth = Arrays.asList("AUTO", "TERM", "TERMINAL")
						.contains(String.valueOf(userValue).toUpperCase(Locale.ENGLISH));
				return sysPropAutoWidth || (autoWidthEnabledInApplication && !isNumeric(userValue));
			}

			private IHelpFactory helpFactory;
			private List<String> sectionKeys = Collections.unmodifiableList(Arrays.asList(SECTION_KEY_HEADER_HEADING,
					SECTION_KEY_HEADER, SECTION_KEY_SYNOPSIS_HEADING, SECTION_KEY_SYNOPSIS,
					SECTION_KEY_DESCRIPTION_HEADING, SECTION_KEY_DESCRIPTION, SECTION_KEY_PARAMETER_LIST_HEADING,
					SECTION_KEY_AT_FILE_PARAMETER, SECTION_KEY_PARAMETER_LIST, SECTION_KEY_OPTION_LIST_HEADING,
					SECTION_KEY_OPTION_LIST, SECTION_KEY_END_OF_OPTIONS, SECTION_KEY_COMMAND_LIST_HEADING,
					SECTION_KEY_COMMAND_LIST, SECTION_KEY_EXIT_CODE_LIST_HEADING, SECTION_KEY_EXIT_CODE_LIST,
					SECTION_KEY_FOOTER_HEADING, SECTION_KEY_FOOTER));
			private Map<String, IHelpSectionRenderer> helpSectionRendererMap = createHelpSectionRendererMap();
			private String[] description;
			private String[] customSynopsis;
			private String[] header;
			private String[] footer;
			private Boolean abbreviateSynopsis;
			private Boolean sortOptions;
			private Boolean sortSynopsis;
			private Boolean showDefaultValues;
			private Boolean showAtFileInUsageHelp;
			private Boolean showEndOfOptionsDelimiterInUsageHelp;
			private Boolean hidden;
			private Boolean autoWidth;
			private Character requiredOptionMarker;
			private String headerHeading;
			private String synopsisHeading;
			private String synopsisSubcommandLabel;
			private Double synopsisAutoIndentThreshold;
			private Integer synopsisIndent;
			private String descriptionHeading;
			private String parameterListHeading;
			private String optionListHeading;
			private String commandListHeading;
			private String footerHeading;

			private String exitCodeListHeading;
			private String[] exitCodeListStrings;
			private Map<String, String> exitCodeList;

			private Integer width;
			private Integer longOptionsMaxWidth;

			private Integer cachedTerminalWidth;
			private final Interpolator interpolator;

			private Messages messages;
			private Boolean adjustLineBreaksForWideCJKCharacters;

			public UsageMessageSpec() {
				this(null);
			}

			UsageMessageSpec(Interpolator interpolator) {
				this.interpolator = interpolator;
			}

			/**
			 * Returns whether the synopsis line(s) should show an abbreviated synopsis
			 * without detailed option names.
			 */
			public boolean abbreviateSynopsis() {
				return (abbreviateSynopsis == null) ? DEFAULT_ABBREVIATE_SYNOPSIS : abbreviateSynopsis;
			}

			/**
			 * Sets whether the synopsis line(s) should show an abbreviated synopsis without
			 * detailed option names.
			 * 
			 * @return this UsageMessageSpec for method chaining
			 */
			public UsageMessageSpec abbreviateSynopsis(boolean newValue) {
				abbreviateSynopsis = newValue;
				return this;
			}

			/**
			 * Returns whether line breaks should take wide Chinese, Japanese and Korean
			 * characters into account for line-breaking purposes.
			 * 
			 * @return true if wide Chinese, Japanese and Korean characters are counted as
			 *         double the size of other characters for line-breaking purposes
			 * @since 4.0
			 */
			public boolean adjustLineBreaksForWideCJKCharacters() {
				return adjustLineBreaksForWideCJKCharacters == null ? DEFAULT_ADJUST_CJK
						: adjustLineBreaksForWideCJKCharacters;
			}

			/**
			 * Sets whether line breaks should take wide Chinese, Japanese and Korean
			 * characters into account, and returns this UsageMessageSpec.
			 * 
			 * @param adjustForWideChars if true, wide Chinese, Japanese and Korean
			 *                           characters are counted as double the size of other
			 *                           characters for line-breaking purposes
			 * @since 4.0
			 */
			public UsageMessageSpec adjustLineBreaksForWideCJKCharacters(boolean adjustForWideChars) {
				adjustLineBreaksForWideCJKCharacters = adjustForWideChars;
				return this;
			}

			private String[] arr(String[] localized, String[] value, String[] defaultValue) {
				return interpolate(localized != null ? localized : (value != null ? value.clone() : defaultValue));
			}

			/**
			 * Returns whether picocli should attempt to detect the terminal size and adjust
			 * the usage help message width to take the full terminal width. End users may
			 * enable this by setting system property {@code "picocli.usage.width"} to
			 * {@code AUTO}, and may disable this by setting this system property to a
			 * {@linkplain #width() numeric value}. This feature requires Java 7 or greater.
			 * The default is {@code false}.
			 * 
			 * @see Command#usageHelpAutoWidth()
			 * @since 4.0
			 */
			public boolean autoWidth() {
				return shouldDetectTerminalSize((autoWidth == null) ? DEFAULT_USAGE_AUTO_WIDTH : autoWidth);
			}

			/**
			 * Sets whether picocli should attempt to detect the terminal size and adjust
			 * the usage help message width to take the full terminal width. The default is
			 * {@code false}.
			 * 
			 * @param detectTerminalSize whether picocli should attempt to detect the
			 *                           terminal size
			 * @see Command#usageHelpAutoWidth()
			 * @since 4.0
			 */
			public UsageMessageSpec autoWidth(boolean detectTerminalSize) {
				autoWidth = detectTerminalSize;
				return this;
			}

			/**
			 * Returns the optional heading preceding the subcommand list. Initialized from
			 * {@link Command#commandListHeading()}. {@code "Commands:%n"} by default.
			 */
			public String commandListHeading() {
				return str(resourceStr("usage.commandListHeading"), commandListHeading, DEFAULT_COMMAND_LIST_HEADING);
			}

			/**
			 * Sets the optional heading preceding the subcommand list.
			 * 
			 * @return this UsageMessageSpec for method chaining
			 */
			public UsageMessageSpec commandListHeading(String newValue) {
				commandListHeading = newValue;
				return this;
			}

			/**
			 * Returns the help section renderers for the predefined section keys. see:
			 * {@link #sectionKeys()}
			 */
			private Map<String, IHelpSectionRenderer> createHelpSectionRendererMap() {
				final Map<String, IHelpSectionRenderer> result = new HashMap<String, IHelpSectionRenderer>();

				result.put(SECTION_KEY_HEADER_HEADING, new IHelpSectionRenderer() {
					@Override
					public String render(Help help) {
						return help.headerHeading();
					}
				});
				result.put(SECTION_KEY_HEADER, new IHelpSectionRenderer() {
					@Override
					public String render(Help help) {
						return help.header();
					}
				});
				// e.g. Usage:
				result.put(SECTION_KEY_SYNOPSIS_HEADING, new IHelpSectionRenderer() {
					@Override
					public String render(Help help) {
						return help.synopsisHeading();
					}
				});
				// e.g. &lt;main class&gt; [OPTIONS] &lt;command&gt; [COMMAND-OPTIONS]
				// [ARGUMENTS]
				result.put(SECTION_KEY_SYNOPSIS, new IHelpSectionRenderer() {
					@Override
					public String render(Help help) {
						return help.synopsis(help.synopsisHeadingLength());
					}
				});
				// e.g. %nDescription:%n%n
				result.put(SECTION_KEY_DESCRIPTION_HEADING, new IHelpSectionRenderer() {
					@Override
					public String render(Help help) {
						return help.descriptionHeading();
					}
				});
				// e.g. {"Converts foos to bars.", "Use options to control conversion mode."}
				result.put(SECTION_KEY_DESCRIPTION, new IHelpSectionRenderer() {
					@Override
					public String render(Help help) {
						return help.description();
					}
				});
				// e.g. %nPositional parameters:%n%n
				result.put(SECTION_KEY_PARAMETER_LIST_HEADING, new IHelpSectionRenderer() {
					@Override
					public String render(Help help) {
						return help.parameterListHeading();
					}
				});
				// e.g. [@<filename>...] One or more argument files containing options.
				result.put(SECTION_KEY_AT_FILE_PARAMETER, new IHelpSectionRenderer() {
					@Override
					public String render(Help help) {
						return help.atFileParameterList();
					}
				});
				// e.g. [FILE...] the files to convert
				result.put(SECTION_KEY_PARAMETER_LIST, new IHelpSectionRenderer() {
					@Override
					public String render(Help help) {
						return help.parameterList();
					}
				});
				// e.g. %nOptions:%n%n
				result.put(SECTION_KEY_OPTION_LIST_HEADING, new IHelpSectionRenderer() {
					@Override
					public String render(Help help) {
						return help.optionListHeading();
					}
				});
				// e.g. -h, --help displays this help and exits
				result.put(SECTION_KEY_OPTION_LIST, new IHelpSectionRenderer() {
					@Override
					public String render(Help help) {
						return help.optionList();
					}
				});
				// e.g. [--] This option can be used to separate command-line options from the
				// list of positional parameters.
				result.put(SECTION_KEY_END_OF_OPTIONS, new IHelpSectionRenderer() {
					@Override
					public String render(Help help) {
						return help.endOfOptionsList();
					}
				});
				// e.g. %nCommands:%n%n
				result.put(SECTION_KEY_COMMAND_LIST_HEADING, new IHelpSectionRenderer() {
					@Override
					public String render(Help help) {
						return help.commandListHeading();
					}
				});
				// e.g. add adds the frup to the frooble
				result.put(SECTION_KEY_COMMAND_LIST, new IHelpSectionRenderer() {
					@Override
					public String render(Help help) {
						return help.commandList();
					}
				});
				result.put(SECTION_KEY_EXIT_CODE_LIST_HEADING, new IHelpSectionRenderer() {
					@Override
					public String render(Help help) {
						return help.exitCodeListHeading();
					}
				});
				result.put(SECTION_KEY_EXIT_CODE_LIST, new IHelpSectionRenderer() {
					@Override
					public String render(Help help) {
						return help.exitCodeList();
					}
				});
				result.put(SECTION_KEY_FOOTER_HEADING, new IHelpSectionRenderer() {
					@Override
					public String render(Help help) {
						return help.footerHeading();
					}
				});
				result.put(SECTION_KEY_FOOTER, new IHelpSectionRenderer() {
					@Override
					public String render(Help help) {
						return help.footer();
					}
				});
				return result;
			}

			/**
			 * Returns the optional custom synopsis lines to use instead of the
			 * auto-generated synopsis. Initialized from {@link Command#customSynopsis()} if
			 * the {@code Command} annotation is present, otherwise this is an empty array
			 * and the synopsis is generated. Applications may programmatically set this
			 * field to create a custom help message.
			 */
			public String[] customSynopsis() {
				return arr(resourceArr("usage.customSynopsis"), customSynopsis, DEFAULT_MULTI_LINE);
			}

			/**
			 * Sets the optional custom synopsis lines to use instead of the auto-generated
			 * synopsis.
			 * 
			 * @return this UsageMessageSpec for method chaining
			 */
			public UsageMessageSpec customSynopsis(String... customSynopsis) {
				this.customSynopsis = customSynopsis;
				return this;
			}

			/**
			 * Returns the optional text lines to use as the description of the help
			 * message, displayed between the synopsis and the options list. Initialized
			 * from {@link Command#description()} if the {@code Command} annotation is
			 * present, otherwise this is an empty array and the help message has no
			 * description. Applications may programmatically set this field to create a
			 * custom help message.
			 */
			public String[] description() {
				return arr(resourceArr("usage.description"), description, DEFAULT_MULTI_LINE);
			}

			/**
			 * Sets the optional text lines to use as the description of the help message,
			 * displayed between the synopsis and the options list.
			 * 
			 * @return this UsageMessageSpec for method chaining
			 */
			public UsageMessageSpec description(String... description) {
				this.description = description;
				return this;
			}

			/**
			 * Returns the optional heading preceding the description section. Initialized
			 * from {@link Command#descriptionHeading()}, or null.
			 */
			public String descriptionHeading() {
				return str(resourceStr("usage.descriptionHeading"), descriptionHeading, DEFAULT_SINGLE_VALUE);
			}

			/**
			 * Sets the heading preceding the description section.
			 * 
			 * @return this UsageMessageSpec for method chaining
			 */
			public UsageMessageSpec descriptionHeading(String newValue) {
				descriptionHeading = newValue;
				return this;
			}

			/**
			 * Returns an unmodifiable map with values to be displayed in the exit codes
			 * section: keys are exit codes, values are descriptions. Descriptions may
			 * contain {@code "%n"} line separators. Callers may be interested in the
			 * {@link UsageMessageSpec#keyValuesMap(String...) keyValuesMap} method for
			 * creating a map from a list of {@code "key:value"} Strings.
			 * <p>
			 * This may be configured in a resource bundle by listing up multiple
			 * {@code "key:value"} pairs. For example:
			 * </p>
			 * 
			 * <pre>
			 * usage.exitCodeList.0 = 0:Successful program execution.
			 * usage.exitCodeList.1 = 64:Invalid input: an unknown option or invalid parameter was specified.
			 * usage.exitCodeList.2 = 70:Execution exception: an exception occurred while executing the business logic.
			 * </pre>
			 * 
			 * @return an unmodifiable map with values to be displayed in the exit codes
			 *         section, or an empty map if no exit codes are
			 *         {@linkplain #exitCodeList(Map) registered}.
			 * @see #keyValuesMap(String...)
			 * @since 4.0
			 */
			public Map<String, String> exitCodeList() {
				final String[] bundleValues = resourceArr("usage.exitCodeList");
				if (bundleValues == null && exitCodeList != null) {
					return exitCodeList;
				}
				final Map<String, String> result = keyValuesMap(
						arr(bundleValues, exitCodeListStrings, DEFAULT_MULTI_LINE));
				return Collections.unmodifiableMap(result);
			}

			/**
			 * Sets the values to be displayed in the exit codes section: keys are exit
			 * codes, values are descriptions. Descriptions may contain {@code "%n"} line
			 * separators.
			 * <p>
			 * This may be configured in a resource bundle by listing up multiple
			 * {@code "key:value"} pairs. For example:
			 * </p>
			 * 
			 * <pre>
			 * usage.exitCodeList.0 = 0:Successful program execution.
			 * usage.exitCodeList.1 = 64:Invalid input: an unknown option or invalid parameter was specified.
			 * usage.exitCodeList.2 = 70:Execution exception: an exception occurred while executing the business logic.
			 * </pre>
			 * 
			 * @param newValue a map with values to be displayed in the exit codes section
			 * @see #keyValuesMap(String...)
			 * @since 4.0
			 */
			public UsageMessageSpec exitCodeList(Map<String, String> newValue) {
				exitCodeList = newValue == null ? null
						: Collections.unmodifiableMap(new LinkedHashMap<String, String>(newValue));
				return this;
			}

			/**
			 * Returns the optional heading preceding the exit codes section, may contain
			 * {@code "%n"} line separators. {@code ""} (empty string) by default.
			 */
			public String exitCodeListHeading() {
				return str(resourceStr("usage.exitCodeListHeading"), exitCodeListHeading, DEFAULT_SINGLE_VALUE);
			}

			/**
			 * Sets the optional heading preceding the exit codes section, may contain
			 * {@code "%n"} line separators. {@code ""} (empty string) by default.
			 * 
			 * @since 4.0
			 */
			public UsageMessageSpec exitCodeListHeading(String newValue) {
				exitCodeListHeading = newValue;
				return this;
			}

			/**
			 * Returns the optional footer text lines displayed at the bottom of the help
			 * message. Initialized from {@link Command#footer()} if the {@code Command}
			 * annotation is present, otherwise this is an empty array and the help message
			 * has no footer. Applications may programmatically set this field to create a
			 * custom help message.
			 */
			public String[] footer() {
				return arr(resourceArr("usage.footer"), footer, DEFAULT_MULTI_LINE);
			}

			/**
			 * Sets the optional footer text lines displayed at the bottom of the help
			 * message.
			 * 
			 * @return this UsageMessageSpec for method chaining
			 */
			public UsageMessageSpec footer(String... footer) {
				this.footer = footer;
				return this;
			}

			/**
			 * Returns the optional heading preceding the footer section. Initialized from
			 * {@link Command#footerHeading()}, or {@code ""} (empty string).
			 */
			public String footerHeading() {
				return str(resourceStr("usage.footerHeading"), footerHeading, DEFAULT_SINGLE_VALUE);
			}

			/**
			 * Sets the optional heading preceding the footer section.
			 * 
			 * @return this UsageMessageSpec for method chaining
			 */
			public UsageMessageSpec footerHeading(String newValue) {
				footerHeading = newValue;
				return this;
			}

			private int getSysPropertyWidthOrDefault(int defaultWidth, boolean detectTerminalSize) {
				if (detectTerminalSize) {
					if (cachedTerminalWidth == null) {
						cachedTerminalWidth = getTerminalWidth();
					}
					return cachedTerminalWidth < 0 ? defaultWidth : Math.max(cachedTerminalWidth, MINIMUM_USAGE_WIDTH);
				}
				final String userValue = System.getProperty("picocli.usage.width");
				if (userValue == null) {
					return defaultWidth;
				}
				try {
					final int width = Integer.parseInt(userValue);
					if (width < MINIMUM_USAGE_WIDTH) {
						CommandLine.tracer().warn("Invalid picocli.usage.width value %d. Using minimum usage width %d.",
								width, MINIMUM_USAGE_WIDTH);
						return MINIMUM_USAGE_WIDTH;
					}
					return width;
				} catch (final NumberFormatException ex) {
					CommandLine.tracer().warn("Invalid picocli.usage.width value '%s'. Using usage width %d.",
							userValue, defaultWidth);
					return defaultWidth;
				}
			}

			/**
			 * Returns the optional header lines displayed at the top of the help message.
			 * For subcommands, the first header line is displayed in the list of commands.
			 * Values are initialized from {@link Command#header()} if the {@code Command}
			 * annotation is present, otherwise this is an empty array and the help message
			 * has no header. Applications may programmatically set this field to create a
			 * custom help message.
			 */
			public String[] header() {
				return arr(resourceArr("usage.header"), header, DEFAULT_MULTI_LINE);
			}

			/**
			 * Sets the optional header lines displayed at the top of the help message. For
			 * subcommands, the first header line is displayed in the list of commands.
			 * 
			 * @return this UsageMessageSpec for method chaining
			 */
			public UsageMessageSpec header(String... header) {
				this.header = header;
				return this;
			}

			/**
			 * Returns the optional heading preceding the header section. Initialized from
			 * {@link Command#headerHeading()}, or {@code ""} (empty string).
			 */
			public String headerHeading() {
				return str(resourceStr("usage.headerHeading"), headerHeading, DEFAULT_SINGLE_VALUE);
			}

			/**
			 * Sets the heading preceding the header section. Initialized from
			 * {@link Command#headerHeading()}, or null.
			 * 
			 * @return this UsageMessageSpec for method chaining
			 */
			public UsageMessageSpec headerHeading(String headerHeading) {
				this.headerHeading = headerHeading;
				return this;
			}

			/**
			 * Returns the {@code IHelpFactory} that is used to construct the usage help
			 * message.
			 * 
			 * @see #setHelpFactory(IHelpFactory)
			 * @since 3.9
			 */
			public IHelpFactory helpFactory() {
				if (helpFactory == null) {
					helpFactory = new DefaultHelpFactory();
				}
				return helpFactory;
			}

			/**
			 * Sets a new {@code IHelpFactory} to customize the usage help message.
			 * 
			 * @param helpFactory the new help factory. Must be non-{@code null}.
			 * @return this {@code UsageMessageSpec} object, to allow method chaining
			 */
			public UsageMessageSpec helpFactory(IHelpFactory helpFactory) {
				this.helpFactory = Assert.notNull(helpFactory, "helpFactory");
				return this;
			}

			/**
			 * Returns whether this command should be hidden from the usage help message of
			 * the parent command.
			 * 
			 * @return {@code true} if this command should not appear in the usage help
			 *         message of the parent command
			 */
			public boolean hidden() {
				return (hidden == null) ? DEFAULT_HIDDEN : hidden;
			}

			/**
			 * Set the hidden flag on this command to control whether to show or hide it in
			 * the help usage text of the parent command.
			 * 
			 * @param value enable or disable the hidden flag
			 * @return this UsageMessageSpec for method chaining
			 * @see Command#hidden()
			 */
			public UsageMessageSpec hidden(boolean value) {
				hidden = value;
				return this;
			}

			void initFrom(UsageMessageSpec settings, CommandSpec commandSpec) {
				abbreviateSynopsis = settings.abbreviateSynopsis;
				adjustLineBreaksForWideCJKCharacters = settings.adjustLineBreaksForWideCJKCharacters;
				autoWidth = settings.autoWidth;
				// cachedTerminalWidth not copied
				commandListHeading = settings.commandListHeading;
				customSynopsis = settings.customSynopsis;
				description = settings.description;
				descriptionHeading = settings.descriptionHeading;
				exitCodeList = settings.exitCodeList;
				exitCodeListHeading = settings.exitCodeListHeading;
				exitCodeListStrings = settings.exitCodeListStrings;
				footer = settings.footer;
				footerHeading = settings.footerHeading;
				header = settings.header;
				headerHeading = settings.headerHeading;
				helpFactory = settings.helpFactory;
				helpSectionRendererMap = settings.helpSectionRendererMap;
				hidden = settings.hidden;
				longOptionsMaxWidth = settings.longOptionsMaxWidth;
				messages = Messages.copy(commandSpec, settings.messages());
				optionListHeading = settings.optionListHeading;
				parameterListHeading = settings.parameterListHeading;
				requiredOptionMarker = settings.requiredOptionMarker;
				sectionKeys = settings.sectionKeys;
				showAtFileInUsageHelp = settings.showAtFileInUsageHelp;
				showDefaultValues = settings.showDefaultValues;
				showEndOfOptionsDelimiterInUsageHelp = settings.showEndOfOptionsDelimiterInUsageHelp;
				sortOptions = settings.sortOptions;
				sortSynopsis = settings.sortSynopsis;
				synopsisAutoIndentThreshold = settings.synopsisAutoIndentThreshold;
				synopsisHeading = settings.synopsisHeading;
				synopsisIndent = settings.synopsisIndent;
				synopsisSubcommandLabel = settings.synopsisSubcommandLabel;
				width = settings.width;
			}

			void initFromMixin(UsageMessageSpec mixin, CommandSpec commandSpec) {
				if (initializable(abbreviateSynopsis, mixin.abbreviateSynopsis(), DEFAULT_ABBREVIATE_SYNOPSIS)) {
					abbreviateSynopsis = mixin.abbreviateSynopsis();
				}
				if (initializable(adjustLineBreaksForWideCJKCharacters, mixin.adjustLineBreaksForWideCJKCharacters(),
						DEFAULT_ADJUST_CJK)) {
					adjustLineBreaksForWideCJKCharacters = mixin.adjustLineBreaksForWideCJKCharacters();
				}
				if (initializable(autoWidth, mixin.autoWidth(), DEFAULT_USAGE_AUTO_WIDTH)) {
					autoWidth = mixin.autoWidth();
				}
				if (initializable(commandListHeading, mixin.commandListHeading(), DEFAULT_COMMAND_LIST_HEADING)) {
					commandListHeading = mixin.commandListHeading();
				}
				if (initializable(customSynopsis, mixin.customSynopsis(), DEFAULT_MULTI_LINE)) {
					customSynopsis = mixin.customSynopsis().clone();
				}
				if (initializable(description, mixin.description(), DEFAULT_MULTI_LINE)) {
					description = mixin.description().clone();
				}
				if (initializable(descriptionHeading, mixin.descriptionHeading(), DEFAULT_SINGLE_VALUE)) {
					descriptionHeading = mixin.descriptionHeading();
				}
				if (initializable(exitCodeList, mixin.exitCodeList(), Collections.emptyMap())
						&& exitCodeListStrings == null) {
					exitCodeList = Collections.unmodifiableMap(new LinkedHashMap<String, String>(mixin.exitCodeList()));
				}
				if (initializable(exitCodeListHeading, mixin.exitCodeListHeading(), DEFAULT_SINGLE_VALUE)) {
					exitCodeListHeading = mixin.exitCodeListHeading();
				}
				if (initializable(footer, mixin.footer(), DEFAULT_MULTI_LINE)) {
					footer = mixin.footer().clone();
				}
				if (initializable(footerHeading, mixin.footerHeading(), DEFAULT_SINGLE_VALUE)) {
					footerHeading = mixin.footerHeading();
				}
				if (initializable(header, mixin.header(), DEFAULT_MULTI_LINE)) {
					header = mixin.header().clone();
				}
				if (initializable(headerHeading, mixin.headerHeading(), DEFAULT_SINGLE_VALUE)) {
					headerHeading = mixin.headerHeading();
				}
				if (initializable(hidden, mixin.hidden(), DEFAULT_HIDDEN)) {
					hidden = mixin.hidden();
				}
				if (initializable(longOptionsMaxWidth, mixin.longOptionsMaxWidth(), DEFAULT_USAGE_LONG_OPTIONS_WIDTH)) {
					longOptionsMaxWidth = mixin.longOptionsMaxWidth();
				}
				if (Messages.empty(messages) && Messages.resourceBundleBaseName(messages) == null) {
					messages(Messages.copy(commandSpec, mixin.messages()));
				}
				if (initializable(optionListHeading, mixin.optionListHeading(), DEFAULT_SINGLE_VALUE)) {
					optionListHeading = mixin.optionListHeading();
				}
				if (initializable(parameterListHeading, mixin.parameterListHeading(), DEFAULT_SINGLE_VALUE)) {
					parameterListHeading = mixin.parameterListHeading();
				}
				if (initializable(requiredOptionMarker, mixin.requiredOptionMarker(), DEFAULT_REQUIRED_OPTION_MARKER)) {
					requiredOptionMarker = mixin.requiredOptionMarker();
				}
				if (initializable(showAtFileInUsageHelp, mixin.showAtFileInUsageHelp(), DEFAULT_SHOW_AT_FILE)) {
					showAtFileInUsageHelp = mixin.showAtFileInUsageHelp();
				}
				if (initializable(showDefaultValues, mixin.showDefaultValues(), DEFAULT_SHOW_DEFAULT_VALUES)) {
					showDefaultValues = mixin.showDefaultValues();
				}
				if (initializable(showEndOfOptionsDelimiterInUsageHelp, mixin.showEndOfOptionsDelimiterInUsageHelp(),
						DEFAULT_SHOW_END_OF_OPTIONS)) {
					showEndOfOptionsDelimiterInUsageHelp = mixin.showEndOfOptionsDelimiterInUsageHelp();
				}
				if (initializable(sortOptions, mixin.sortOptions(), DEFAULT_SORT_OPTIONS)) {
					sortOptions = mixin.sortOptions();
				}
				if (initializable(sortSynopsis, mixin.sortSynopsis(), DEFAULT_SORT_SYNOPSIS)) {
					sortSynopsis = mixin.sortSynopsis();
				}
				if (initializable(synopsisHeading, mixin.synopsisHeading(), DEFAULT_SYNOPSIS_HEADING)) {
					synopsisHeading = mixin.synopsisHeading();
				}
				if (initializable(synopsisSubcommandLabel, mixin.synopsisSubcommandLabel(),
						DEFAULT_SYNOPSIS_SUBCOMMANDS)) {
					synopsisSubcommandLabel = mixin.synopsisSubcommandLabel();
				}
				if (initializable(width, mixin.width(), DEFAULT_USAGE_WIDTH)) {
					width = mixin.width();
				}
			}

			private String interpolate(String value) {
				return interpolator == null ? value : interpolator.interpolate(value);
			}

			private String[] interpolate(String[] values) {
				return interpolator == null ? values : interpolator.interpolate(values);
			}

			/**
			 * Returns the maximum usage help long options column max width to the specified
			 * value. This value controls the maximum width of the long options column: any
			 * positional parameter labels or long options that are longer than the
			 * specified value will overflow into the description column, and cause the
			 * description to be displayed on the next line.
			 * 
			 * @return the new maximum usage help long options column max width. Always 20
			 *         or greater.
			 * @since 4.2
			 */
			public int longOptionsMaxWidth() {
				return longOptionsMaxWidth == null ? DEFAULT_USAGE_LONG_OPTIONS_WIDTH : longOptionsMaxWidth;
			}

			/**
			 * Sets the maximum usage help long options column max width to the specified
			 * value. This value controls the maximum width of the long options column: any
			 * positional parameter labels or long options that are longer than the
			 * specified value will overflow into the description column, and cause the
			 * description to be displayed on the next line.
			 * 
			 * @param newValue the new maximum usage help long options column max width.
			 *                 Must be 20 or greater, otherwise the new value will be
			 *                 ignored.
			 * @return this {@code UsageMessageSpec} for method chaining
			 * @since 4.2
			 */
			public UsageMessageSpec longOptionsMaxWidth(int newValue) {
				if (newValue < DEFAULT_USAGE_LONG_OPTIONS_WIDTH) {
					CommandLine.tracer().info("Invalid usage long options max width %d. Minimum value is %d", newValue,
							DEFAULT_USAGE_LONG_OPTIONS_WIDTH);
				} else if (newValue > width() - DEFAULT_USAGE_LONG_OPTIONS_WIDTH) {
					CommandLine.tracer().info(
							"Invalid usage long options max width %d. Value must not exceed width(%d) - %d", newValue,
							width(), DEFAULT_USAGE_LONG_OPTIONS_WIDTH);
				} else {
					longOptionsMaxWidth = newValue;
				}
				return this;
			}

			/**
			 * Returns the Messages for this usage help message specification, or
			 * {@code null}.
			 * 
			 * @return the Messages object that encapsulates this
			 *         {@linkplain CommandSpec#resourceBundle() command's resource bundle}
			 * @since 3.6
			 */
			public Messages messages() {
				return messages;
			}

			/**
			 * Sets the Messages for this usageMessage specification, and returns this
			 * UsageMessageSpec.
			 * 
			 * @param msgs the new Messages value that encapsulates this
			 *             {@linkplain CommandSpec#resourceBundle() command's resource
			 *             bundle}, may be {@code null}
			 * @since 3.6
			 */
			public UsageMessageSpec messages(Messages msgs) {
				messages = msgs;
				return this;
			}

			/**
			 * Returns the optional heading preceding the options list. Initialized from
			 * {@link Command#optionListHeading()}, or null.
			 */
			public String optionListHeading() {
				return str(resourceStr("usage.optionListHeading"), optionListHeading, DEFAULT_SINGLE_VALUE);
			}

			/**
			 * Sets the heading preceding the options list.
			 * 
			 * @return this UsageMessageSpec for method chaining
			 */
			public UsageMessageSpec optionListHeading(String newValue) {
				optionListHeading = newValue;
				return this;
			}

			/**
			 * Returns the optional heading preceding the parameter list. Initialized from
			 * {@link Command#parameterListHeading()}, or null.
			 */
			public String parameterListHeading() {
				return str(resourceStr("usage.parameterListHeading"), parameterListHeading, DEFAULT_SINGLE_VALUE);
			}

			/**
			 * Sets the optional heading preceding the parameter list.
			 * 
			 * @return this UsageMessageSpec for method chaining
			 */
			public UsageMessageSpec parameterListHeading(String newValue) {
				parameterListHeading = newValue;
				return this;
			}

			/**
			 * Returns the character used to prefix required options in the options list.
			 */
			public char requiredOptionMarker() {
				return (requiredOptionMarker == null) ? DEFAULT_REQUIRED_OPTION_MARKER : requiredOptionMarker;
			}

			/**
			 * Sets the character used to prefix required options in the options list.
			 * 
			 * @return this UsageMessageSpec for method chaining
			 */
			public UsageMessageSpec requiredOptionMarker(char newValue) {
				requiredOptionMarker = newValue;
				return this;
			}

			private String[] resourceArr(String key) {
				return messages == null ? null : messages.getStringArray(key, null);
			}

			private String resourceStr(String key) {
				return messages == null ? null : messages.getString(key, null);
			}

			/**
			 * Returns the section keys in the order that the usage help message should
			 * render the sections. This ordering may be modified with the
			 * {@link #sectionKeys(List) sectionKeys setter}. The default keys are (in
			 * order):
			 * <ol start="0">
			 * <li>{@link UsageMessageSpec#SECTION_KEY_HEADER_HEADING
			 * SECTION_KEY_HEADER_HEADING}</li>
			 * <li>{@link UsageMessageSpec#SECTION_KEY_HEADER SECTION_KEY_HEADER}</li>
			 * <li>{@link UsageMessageSpec#SECTION_KEY_SYNOPSIS_HEADING
			 * SECTION_KEY_SYNOPSIS_HEADING}</li>
			 * <li>{@link UsageMessageSpec#SECTION_KEY_SYNOPSIS SECTION_KEY_SYNOPSIS}</li>
			 * <li>{@link UsageMessageSpec#SECTION_KEY_DESCRIPTION_HEADING
			 * SECTION_KEY_DESCRIPTION_HEADING}</li>
			 * <li>{@link UsageMessageSpec#SECTION_KEY_DESCRIPTION
			 * SECTION_KEY_DESCRIPTION}</li>
			 * <li>{@link UsageMessageSpec#SECTION_KEY_PARAMETER_LIST_HEADING
			 * SECTION_KEY_PARAMETER_LIST_HEADING}</li>
			 * <li>{@link UsageMessageSpec#SECTION_KEY_AT_FILE_PARAMETER
			 * SECTION_KEY_AT_FILE_PARAMETER}</li>
			 * <li>{@link UsageMessageSpec#SECTION_KEY_PARAMETER_LIST
			 * SECTION_KEY_PARAMETER_LIST}</li>
			 * <li>{@link UsageMessageSpec#SECTION_KEY_OPTION_LIST_HEADING
			 * SECTION_KEY_OPTION_LIST_HEADING}</li>
			 * <li>{@link UsageMessageSpec#SECTION_KEY_OPTION_LIST
			 * SECTION_KEY_OPTION_LIST}</li>
			 * <li>{@link UsageMessageSpec#SECTION_KEY_END_OF_OPTIONS
			 * SECTION_KEY_END_OF_OPTIONS}</li>
			 * <li>{@link UsageMessageSpec#SECTION_KEY_COMMAND_LIST_HEADING
			 * SECTION_KEY_COMMAND_LIST_HEADING}</li>
			 * <li>{@link UsageMessageSpec#SECTION_KEY_COMMAND_LIST
			 * SECTION_KEY_COMMAND_LIST}</li>
			 * <li>{@link UsageMessageSpec#SECTION_KEY_EXIT_CODE_LIST_HEADING
			 * SECTION_KEY_EXIT_CODE_LIST_HEADING}</li>
			 * <li>{@link UsageMessageSpec#SECTION_KEY_EXIT_CODE_LIST
			 * SECTION_KEY_EXIT_CODE_LIST}</li>
			 * <li>{@link UsageMessageSpec#SECTION_KEY_FOOTER_HEADING
			 * SECTION_KEY_FOOTER_HEADING}</li>
			 * <li>{@link UsageMessageSpec#SECTION_KEY_FOOTER SECTION_KEY_FOOTER}</li>
			 * </ol>
			 * 
			 * @since 3.9
			 */
			public List<String> sectionKeys() {
				return sectionKeys;
			}

			/**
			 * Sets the section keys in the order that the usage help message should render
			 * the sections.
			 * 
			 * @see #sectionKeys
			 * @since 3.9
			 */
			public UsageMessageSpec sectionKeys(List<String> keys) {
				sectionKeys = Collections.unmodifiableList(new ArrayList<String>(keys));
				return this;
			}

			/**
			 * Returns the map of section keys and renderers used to construct the usage
			 * help message. The usage help message can be customized by adding, replacing
			 * and removing section renderers from this map. Sections can be reordered with
			 * the {@link #sectionKeys(List) sectionKeys setter}. Sections that are either
			 * not in this map or not in the list returned by {@link #sectionKeys()
			 * sectionKeys} are omitted.
			 * 
			 * @see #sectionKeys
			 * @since 3.9
			 */
			public Map<String, IHelpSectionRenderer> sectionMap() {
				return helpSectionRendererMap;
			}

			/**
			 * Sets the map of section keys and renderers used to construct the usage help
			 * message to a copy of the specified map.
			 * 
			 * @param map the mapping of section keys to their renderers, must be
			 *            non-{@code null}.
			 * @return this UsageMessageSpec for method chaining
			 * @see #sectionKeys
			 * @see #setHelpSectionMap(Map)
			 * @since 3.9
			 */
			public UsageMessageSpec sectionMap(Map<String, IHelpSectionRenderer> map) {
				this.helpSectionRendererMap = new LinkedHashMap<String, IHelpSectionRenderer>(map);
				return this;
			}

			/**
			 * Sets whether to show a {@code [@<filename>...]} entry in the synopsis and
			 * parameter list of the usage help message. (The entry is not shown if
			 * {@linkplain CommandLine#isExpandAtFiles() expanding parameter files} is
			 * disabled.)
			 * 
			 * @see Command#showAtFileInUsageHelp()
			 * @since 4.2
			 */
			public boolean showAtFileInUsageHelp() {
				return (showAtFileInUsageHelp == null) ? DEFAULT_SHOW_AT_FILE : showAtFileInUsageHelp;
			}

			/**
			 * Sets whether to show a {@code [@<filename>...]} entry in the synopsis and
			 * parameter list of the usage help message. (The entry is not shown if
			 * {@linkplain CommandLine#isExpandAtFiles() expanding parameter files} is
			 * disabled.)
			 * 
			 * @see Command#showAtFileInUsageHelp()
			 * @return this UsageMessageSpec for method chaining
			 * @since 4.2
			 */
			public UsageMessageSpec showAtFileInUsageHelp(boolean newValue) {
				showAtFileInUsageHelp = newValue;
				return this;
			}

			/**
			 * Returns whether the options list in the usage help message should show
			 * default values for all non-boolean options.
			 */
			public boolean showDefaultValues() {
				return (showDefaultValues == null) ? DEFAULT_SHOW_DEFAULT_VALUES : showDefaultValues;
			}

			/**
			 * Sets whether the options list in the usage help message should show default
			 * values for all non-boolean options.
			 * 
			 * @return this UsageMessageSpec for method chaining
			 */
			public UsageMessageSpec showDefaultValues(boolean newValue) {
				showDefaultValues = newValue;
				return this;
			}

			/**
			 * Sets whether to show a {@code [--]} (End of Options) entry in the synopsis
			 * and options list of the usage help message.
			 * 
			 * @see Command#showEndOfOptionsDelimiterInUsageHelp()
			 * @since 4.3
			 */
			public boolean showEndOfOptionsDelimiterInUsageHelp() {
				return (showEndOfOptionsDelimiterInUsageHelp == null) ? DEFAULT_SHOW_END_OF_OPTIONS
						: showEndOfOptionsDelimiterInUsageHelp;
			}

			/**
			 * Sets whether to show a {@code [--]} (End of Options) entry in the synopsis
			 * and options list of the usage help message.
			 * 
			 * @see Command#showEndOfOptionsDelimiterInUsageHelp()
			 * @return this UsageMessageSpec for method chaining
			 * @since 4.3
			 */
			public UsageMessageSpec showEndOfOptionsDelimiterInUsageHelp(boolean newValue) {
				showEndOfOptionsDelimiterInUsageHelp = newValue;
				return this;
			}

			/**
			 * Returns whether the options list in the usage help message should be sorted
			 * alphabetically.
			 */
			public boolean sortOptions() {
				return (sortOptions == null) ? DEFAULT_SORT_OPTIONS : sortOptions;
			}

			/**
			 * Sets whether the options list in the usage help message should be sorted
			 * alphabetically.
			 * 
			 * @return this UsageMessageSpec for method chaining
			 */
			public UsageMessageSpec sortOptions(boolean newValue) {
				sortOptions = newValue;
				return this;
			}

			/**
			 * Returns whether the options in the synopsis should be sorted alphabetically.
			 * 
			 * @since 4.7.6-SNAPSHOT
			 */
			public boolean sortSynopsis() {
				return (sortSynopsis == null) ? DEFAULT_SORT_SYNOPSIS : sortSynopsis;
			}

			/**
			 * Sets whether the options in the synopsis should be sorted alphabetically.
			 * 
			 * @return this UsageMessageSpec for method chaining
			 * @since 4.7.6-SNAPSHOT
			 */
			public UsageMessageSpec sortSynopsis(boolean newValue) {
				sortSynopsis = newValue;
				return this;
			}

			private String str(String localized, String value, String defaultValue) {
				return interpolate(localized != null ? localized : (value != null ? value : defaultValue));
			}

			/**
			 * Returns the fraction of the usage help {@link #width()} that is the threshold
			 * up to which the 2nd line and subsequent lines of a multi-line synopsis should
			 * be aligned to the end of the command name. The default value of this
			 * attribute is {@code 0.5}. If the length of the synopsis heading plus the
			 * length of the fully qualified command name exceeds this fraction of the
			 * width, the 2nd and subsequent rows of a multi-line synopsis will be aligned
			 * to the {@link #synopsisIndent()} instead of the end of the command name.
			 * 
			 * @since 4.0
			 */
			public double synopsisAutoIndentThreshold() {
				return synopsisAutoIndentThreshold == null ? DEFAULT_SYNOPSIS_AUTO_INDENT_THRESHOLD
						: synopsisAutoIndentThreshold;
			}

			/**
			 * Sets the fraction of the usage help {@link #width()} that is the threshold up
			 * to which the 2nd line and subsequent lines of a multi-line synopsis should be
			 * aligned to the end of the command name. The default value of this attribute
			 * is {@code 0.5}. If the length of the synopsis heading plus the length of the
			 * fully qualified command name exceeds this fraction of the width, the 2nd and
			 * subsequent rows of a multi-line synopsis will be aligned to the
			 * {@link #synopsisIndent()} instead of the end of the command name.
			 * 
			 * @param newValue the new threshold value. Must be a value between 0.0 and 0.9,
			 *                 inclusive
			 * @return this UsageMessageSpec for method chaining
			 * @throws IllegalArgumentException if the specified value is less than 0.0 or
			 *                                  greater than 0.9
			 * @since 4.0
			 */
			public UsageMessageSpec synopsisAutoIndentThreshold(double newValue) {
				if (newValue < 0 || newValue > MAX_SYNOPSIS_AUTO_INDENT_THRESHOLD) {
					throw new IllegalArgumentException(
							"synopsisAutoIndentThreshold must be between 0.0 and 0.9 (inclusive), but was " + newValue);
				}
				synopsisAutoIndentThreshold = newValue;
				return this;
			}

			/**
			 * Returns the optional heading preceding the synopsis. Initialized from
			 * {@link Command#synopsisHeading()}, {@code "Usage: "} by default.
			 */
			public String synopsisHeading() {
				return str(resourceStr("usage.synopsisHeading"), synopsisHeading, DEFAULT_SYNOPSIS_HEADING);
			}

			/**
			 * Sets the optional heading preceding the synopsis.
			 * 
			 * @return this UsageMessageSpec for method chaining
			 */
			public UsageMessageSpec synopsisHeading(String newValue) {
				synopsisHeading = newValue;
				return this;
			}

			/**
			 * Returns the indentation to use on the 2nd line and subsequent lines of a
			 * multi-line synopsis when the length of the synopsis heading and the fully
			 * qualified command name exceed the {@link #width()} times the
			 * {@link #synopsisAutoIndentThreshold()}, {@code -1} by default. A negative
			 * value for this option means that the 2nd line and subsequent lines are
			 * aligned to the synopsis heading length. A positive value means the exact
			 * number of spaces to indent for the 2nd line and subsequent lines of the
			 * synopsis.
			 * 
			 * @since 4.0
			 */
			public int synopsisIndent() {
				return synopsisIndent == null ? DEFAULT_SYNOPSIS_INDENT : synopsisIndent;
			}

			/**
			 * Sets the indentation to use on the 2nd line and subsequent lines of a
			 * multi-line synopsis when the length of the synopsis heading and the fully
			 * qualified command name exceed the {@link #synopsisAutoIndentThreshold()}
			 * fraction of the {@link #width()}, {@code -1} by default. A negative value for
			 * this option means that the 2nd line and subsequent lines are aligned to the
			 * synopsis heading length. A positive value means the exact number of spaces to
			 * indent for the 2nd line and subsequent lines of the synopsis.
			 * 
			 * @return this UsageMessageSpec for method chaining
			 * @since 4.0
			 */
			public UsageMessageSpec synopsisIndent(int newValue) {
				synopsisIndent = newValue;
				return this;
			}

			/**
			 * Returns the String representing the subcommands in the synopsis. Initialized
			 * from {@link Command#synopsisSubcommandLabel()}, {@code "[COMMANDS]"} by
			 * default.
			 * 
			 * @since 4.0
			 */
			public String synopsisSubcommandLabel() {
				return str(resourceStr("usage.synopsisSubcommandLabel"), synopsisSubcommandLabel,
						DEFAULT_SYNOPSIS_SUBCOMMANDS);
			}

			/**
			 * Sets the String representing the subcommands in the synopsis.
			 * 
			 * @return this UsageMessageSpec for method chaining
			 * @since 4.0
			 */
			public UsageMessageSpec synopsisSubcommandLabel(String newValue) {
				synopsisSubcommandLabel = newValue;
				return this;
			}

			void updateFromCommand(Command cmd, CommandSpec commandSpec, boolean loadResourceBundle) {
				if (!empty(cmd.resourceBundle())) { // else preserve superclass bundle
					if (loadResourceBundle) { // error if resource bundle does not exist
						messages(new Messages(commandSpec, cmd.resourceBundle()));
					} else { // from annotation processor, we may not be able to load the ResourceBundle
						try {
							messages(new Messages(commandSpec, cmd.resourceBundle()));
						} catch (final MissingResourceException ignored) {
							messages(new Messages(commandSpec, cmd.resourceBundle(), null));
						}
					}
				}
				if (isNonDefault(cmd.abbreviateSynopsis(), DEFAULT_ABBREVIATE_SYNOPSIS)) {
					abbreviateSynopsis = cmd.abbreviateSynopsis();
				}
				if (isNonDefault(cmd.usageHelpAutoWidth(), DEFAULT_USAGE_AUTO_WIDTH)) {
					autoWidth = cmd.usageHelpAutoWidth();
				}
				if (isNonDefault(cmd.commandListHeading(), DEFAULT_COMMAND_LIST_HEADING)) {
					commandListHeading = cmd.commandListHeading();
				}
				if (isNonDefault(cmd.customSynopsis(), DEFAULT_MULTI_LINE)) {
					customSynopsis = cmd.customSynopsis().clone();
				}
				if (isNonDefault(cmd.description(), DEFAULT_MULTI_LINE)) {
					description = cmd.description().clone();
				}
				if (isNonDefault(cmd.descriptionHeading(), DEFAULT_SINGLE_VALUE)) {
					descriptionHeading = cmd.descriptionHeading();
				}
				if (isNonDefault(cmd.exitCodeList(), DEFAULT_MULTI_LINE)) {
					exitCodeListStrings = cmd.exitCodeList().clone();
				}
				if (isNonDefault(cmd.exitCodeListHeading(), DEFAULT_SINGLE_VALUE)) {
					exitCodeListHeading = cmd.exitCodeListHeading();
				}
				if (isNonDefault(cmd.footer(), DEFAULT_MULTI_LINE)) {
					footer = cmd.footer().clone();
				}
				if (isNonDefault(cmd.footerHeading(), DEFAULT_SINGLE_VALUE)) {
					footerHeading = cmd.footerHeading();
				}
				if (isNonDefault(cmd.header(), DEFAULT_MULTI_LINE)) {
					header = cmd.header().clone();
				}
				if (isNonDefault(cmd.headerHeading(), DEFAULT_SINGLE_VALUE)) {
					headerHeading = cmd.headerHeading();
				}
				if (isNonDefault(cmd.hidden(), DEFAULT_HIDDEN)) {
					hidden = cmd.hidden();
				}
				// if (isNonDefault(cmd.longOptionsMaxWidth(),
				// DEFAULT_USAGE_LONG_OPTIONS_WIDTH)){longOptionsMaxWidth =
				// cmd.longOptionsMaxWidth();}
				if (isNonDefault(cmd.optionListHeading(), DEFAULT_SINGLE_VALUE)) {
					optionListHeading = cmd.optionListHeading();
				}
				if (isNonDefault(cmd.parameterListHeading(), DEFAULT_SINGLE_VALUE)) {
					parameterListHeading = cmd.parameterListHeading();
				}
				if (isNonDefault(cmd.requiredOptionMarker(), DEFAULT_REQUIRED_OPTION_MARKER)) {
					requiredOptionMarker = cmd.requiredOptionMarker();
				}
				if (isNonDefault(cmd.showAtFileInUsageHelp(), DEFAULT_SHOW_AT_FILE)) {
					showAtFileInUsageHelp = cmd.showAtFileInUsageHelp();
				}
				if (isNonDefault(cmd.showDefaultValues(), DEFAULT_SHOW_DEFAULT_VALUES)) {
					showDefaultValues = cmd.showDefaultValues();
				}
				if (isNonDefault(cmd.showEndOfOptionsDelimiterInUsageHelp(), DEFAULT_SHOW_END_OF_OPTIONS)) {
					showEndOfOptionsDelimiterInUsageHelp = cmd.showEndOfOptionsDelimiterInUsageHelp();
				}
				if (isNonDefault(cmd.sortOptions(), DEFAULT_SORT_OPTIONS)) {
					sortOptions = cmd.sortOptions();
				}
				if (isNonDefault(cmd.sortSynopsis(), DEFAULT_SORT_SYNOPSIS)) {
					sortSynopsis = cmd.sortSynopsis();
				}
				if (isNonDefault(cmd.synopsisHeading(), DEFAULT_SYNOPSIS_HEADING)) {
					synopsisHeading = cmd.synopsisHeading();
				}
				if (isNonDefault(cmd.synopsisSubcommandLabel(), DEFAULT_SYNOPSIS_SUBCOMMANDS)) {
					synopsisSubcommandLabel = cmd.synopsisSubcommandLabel();
				}
				if (isNonDefault(cmd.usageHelpWidth(), DEFAULT_USAGE_WIDTH)) {
					width(cmd.usageHelpWidth());
				} // validate
			}

			/**
			 * Returns the maximum usage help message width. Derived from system property
			 * {@code "picocli.usage.width"} if set, otherwise returns the value set via the
			 * {@link #width(int)} method, or if not set, the
			 * {@linkplain #DEFAULT_USAGE_WIDTH default width}.
			 * 
			 * @return the maximum usage help message width. Never returns less than 55.
			 */
			public int width() {
				return getSysPropertyWidthOrDefault(width == null ? DEFAULT_USAGE_WIDTH : width, autoWidth());
			}

			/**
			 * Sets the maximum usage help message width to the specified value. Longer
			 * values are wrapped.
			 * 
			 * @param newValue the new maximum usage help message width. Must be 55 or
			 *                 greater.
			 * @return this {@code UsageMessageSpec} for method chaining
			 * @throws IllegalArgumentException if the specified width is less than 55
			 */
			public UsageMessageSpec width(int newValue) {
				if (newValue < MINIMUM_USAGE_WIDTH) {
					throw new InitializationException(
							"Invalid usage message width " + newValue + ". Minimum value is " + MINIMUM_USAGE_WIDTH);
				}
				width = newValue;
				return this;
			}
		}

		private static boolean initializable(Object current, Object candidate, Object defaultValue) {
			return current == null && isNonDefault(candidate, defaultValue);
		}

		private static boolean initializable(Object current, Object[] candidate, Object[] defaultValue) {
			return current == null && isNonDefault(candidate, defaultValue);
		}

		private static boolean isNonDefault(Object candidate, Object defaultValue) {
			return !Assert.notNull(defaultValue, "defaultValue").equals(candidate);
		}

		private static boolean isNonDefault(Object[] candidate, Object[] defaultValue) {
			return !Arrays.equals(Assert.notNull(defaultValue, "defaultValue"), candidate);
		}

		private Model() {
		}
	}

	/**
	 * Exception indicating that the user input included multiple arguments from a
	 * mutually exclusive group.
	 * 
	 * @since 4.0
	 */
	public static class MutuallyExclusiveArgsException extends ParameterException {
		private static final long serialVersionUID = -5557715106221420986L;

		public MutuallyExclusiveArgsException(CommandLine commandLine, String msg) {
			super(commandLine, msg);
		}
	}

	private static class NoCompletionCandidates implements Iterable<String> {
		@Override
		public Iterator<String> iterator() {
			throw new UnsupportedOperationException();
		}
	}

	private static class NoDefaultProvider implements IDefaultValueProvider {
		@Override
		public String defaultValue(ArgSpec argSpec) {
			throw new UnsupportedOperationException();
		}
	}

	private static class NoOpModelTransformer implements IModelTransformer {
		@Override
		public CommandSpec transform(CommandSpec commandSpec) {
			return commandSpec;
		}
	}

	private static class NoOpParameterPreprocessor implements IParameterPreprocessor {
		@Override
		public boolean equals(Object obj) {
			return obj instanceof NoOpParameterPreprocessor;
		}

		@Override
		public int hashCode() {
			return NoOpParameterPreprocessor.class.hashCode() + 7;
		}

		@Override
		public boolean preprocess(Stack<String> args, CommandSpec commandSpec, ArgSpec argSpec,
				Map<String, Object> info) {
			return false;
		}
	}

	private static class NoVersionProvider implements IVersionProvider {
		@Override
		public String[] getVersion() throws Exception {
			throw new UnsupportedOperationException();
		}
	}

	private static class NullParameterConsumer implements IParameterConsumer {
		@Override
		public void consumeParameters(Stack<String> args, ArgSpec argSpec, CommandSpec commandSpec) {
			throw new UnsupportedOperationException();
		}
	}

	/**
	 * <p>
	 * Annotate fields in your class with {@code @Option} and picocli will
	 * initialize these fields when matching arguments are specified on the command
	 * line. In the case of command methods (annotated with {@code @Command}),
	 * command options can be defined by annotating method parameters with
	 * {@code @Option}.
	 * </p>
	 * <p>
	 * Command class example:
	 * </p>
	 * 
	 * <pre>
	 * import static picocli.CommandLine.*;
	 *
	 * public class MyClass {
	 * 	&#064;Parameters(description = "Any number of input files")
	 * 	private List&lt;File&gt; files = new ArrayList&lt;File&gt;();
	 *
	 * 	&#064;Option(names = { "-o", "--out" }, description = "Output file (default: print to console)")
	 * 	private File outputFile;
	 *
	 * 	&#064;Option(names = { "-v",
	 * 			"--verbose" }, description = "Verbose mode. Helpful for troubleshooting. Multiple -v options increase the verbosity.")
	 * 	private boolean[] verbose;
	 *
	 * 	&#064;Option(names = { "-h", "--help", "-?", "-help" }, usageHelp = true, description = "Display this help and exit")
	 * 	private boolean help;
	 * }
	 * </pre>
	 * <p>
	 * A field cannot be annotated with both {@code @Parameters} and {@code @Option}
	 * or a {@code ParameterException} is thrown.
	 * </p>
	 */
	@Retention(RetentionPolicy.RUNTIME)
	@Target({ ElementType.FIELD, ElementType.METHOD, ElementType.PARAMETER })
	public @interface Option {
		/**
		 * Special value that can be used in some annotation attributes to designate
		 * {@code null}.
		 * 
		 * @see Option#defaultValue()
		 * @see Option#fallbackValue()
		 * @see Option#mapFallbackValue()
		 * @since 4.6
		 */
		public static final String NULL_VALUE = ArgSpec.NULL_VALUE;

		/**
		 * Specifies the minimum number of required parameters and the maximum number of
		 * accepted parameters. If an option declares a positive arity, and the user
		 * specifies an insufficient number of parameters on the command line, a
		 * {@link MissingParameterException} is thrown by the {@link #parse(String...)}
		 * method.
		 * <p>
		 * In many cases picocli can deduce the number of required parameters from the
		 * field's type. By default, flags (boolean options) have arity "0..1", and
		 * single-valued type fields (String, int, Integer, double, Double, File, Date,
		 * etc) have arity one. Generally, fields with types that cannot hold multiple
		 * values can omit the {@code arity} attribute.
		 * </p>
		 * <p>
		 * Fields used to capture options with arity two or higher should have a type
		 * that can hold multiple values, like arrays or Collections. See
		 * {@link #type()} for strongly-typed Collection fields.
		 * </p>
		 * <p>
		 * For example, if an option has 2 required parameters and any number of
		 * optional parameters, specify
		 * {@code @Option(names = "-example", arity = "2..*")}.
		 * </p>
		 * <b>A note on boolean options</b>
		 * <p>
		 * By default picocli allows boolean options (also called "flags" or "switches")
		 * to have an optional parameter, which must be either "true" or "false"
		 * (lowercase, other values are rejected). You can make a boolean option take a
		 * required parameter by annotating your field with {@code arity="1"}. For
		 * example:
		 * </p>
		 * 
		 * <pre>
		 * &#064;Option(names = "-v", arity = "1")
		 * boolean verbose;
		 * </pre>
		 * <p>
		 * Because this boolean field is defined with arity 1, the user must specify
		 * either {@code <program> -v false} or {@code <program> -v true} on the command
		 * line, or a {@link MissingParameterException} is thrown by the
		 * {@link #parse(String...)} method.
		 * </p>
		 * <p>
		 * To remove the optional parameter, define the field with {@code arity = "0"}.
		 * For example:
		 * </p>
		 * 
		 * <pre>
		 * &#064;Option(names = "-v", arity = "0")
		 * boolean verbose;
		 * </pre>
		 * <p>
		 * This will reject any of the below:
		 * </p>
		 * 
		 * <pre>
		 * -v true
		 * -v false
		 * </pre>
		 * 
		 * @return how many arguments this option requires
		 */
		String arity() default "";

		/**
		 * Use this attribute to specify an {@code Iterable<String>} class that
		 * generates completion candidates for this option. For map fields, completion
		 * candidates should be in {@code key=value} form.
		 * <p>
		 * Completion candidates are used in bash completion scripts generated by the
		 * {@code picocli.AutoComplete} class. Bash has special completion options to
		 * generate file names and host names, and the bash completion scripts generated
		 * by {@code AutoComplete} delegate to these bash built-ins for {@code @Options}
		 * whose {@code type} is {@code java.io.File}, {@code java.nio.file.Path} or
		 * {@code java.net.InetAddress}.
		 * </p>
		 * <p>
		 * For {@code @Options} whose {@code type} is a Java {@code enum},
		 * {@code AutoComplete} can generate completion candidates from the type. For
		 * other types, use this attribute to specify completion candidates.
		 * </p>
		 *
		 * @return a class whose instances can iterate over the completion candidates
		 *         for this option
		 * @see org.rossonet.ext.picocli.CommandLine.IFactory
		 * @since 3.2
		 */
		Class<? extends Iterable<String>> completionCandidates() default NoCompletionCandidates.class;

		/**
		 * Optionally specify one or more {@link ITypeConverter} classes to use to
		 * convert the command line argument into a strongly typed value (or key-value
		 * pair for map fields). This is useful when a particular field should use a
		 * custom conversion that is different from the normal conversion for the
		 * field's type.
		 * <p>
		 * For example, for a specific field you may want to use a converter that maps
		 * the constant names defined in {@link java.sql.Types java.sql.Types} to the
		 * {@code int} value of these constants, but any other {@code int} fields should
		 * not be affected by this and should continue to use the standard int converter
		 * that parses numeric values.
		 * </p>
		 * 
		 * @return the type converter(s) to use to convert String values to strongly
		 *         typed values for this field
		 * @see CommandLine#registerConverter(Class, ITypeConverter)
		 */
		Class<? extends ITypeConverter<?>>[] converter() default {};

		/**
		 * Returns the default value of this option, before splitting and type
		 * conversion.
		 * <p>
		 * To get a {@code null} default value, omit specifying a default value or use
		 * the special value {@link Option#NULL_VALUE} - for options of type
		 * {@code Optional<T>} that will result in the {@code Optional.empty()} value
		 * being assigned when the option is not specified on the command line.
		 * </p>
		 * 
		 * @return a String that (after type conversion) will be used as the value for
		 *         this option if the option was not specified on the command line
		 * @see #fallbackValue()
		 * @since 3.2
		 */
		String defaultValue() default "__no_default_value__";

		/**
		 * Description of this option, used when generating the usage documentation.
		 * Each element of the array is rendered on a separate line.
		 * <p>
		 * May contain embedded {@linkplain java.util.Formatter format specifiers} like
		 * {@code %n} line separators. Literal percent {@code '%'} characters must be
		 * escaped with another {@code %}.
		 * </p>
		 * <p>
		 * The description may contain variables that are rendered when help is
		 * requested. The string {@code ${DEFAULT-VALUE}} is replaced with the default
		 * value of the option. This is regardless of the command's
		 * {@link Command#showDefaultValues() showDefaultValues} setting or the option's
		 * {@link #showDefaultValue() showDefaultValue} setting. The string
		 * {@code ${COMPLETION-CANDIDATES}} is replaced with the completion candidates
		 * generated by {@link #completionCandidates()} in the description for this
		 * option. Also, embedded {@code %n} newline markers are converted to actual
		 * newlines.
		 * </p>
		 * 
		 * @return the description of this option
		 * @see <a href="https://picocli.info/#_variable_interpolation">Variable
		 *      Interpolation section of the user manual</a>
		 */
		String[] description() default {};

		/**
		 * ResourceBundle key for this option. If not specified, (and a ResourceBundle
		 * {@linkplain Command#resourceBundle() exists for this command}) an attempt is
		 * made to find the option description using any of the option names (without
		 * leading hyphens) as key.
		 * 
		 * @see OptionSpec#description()
		 * @since 3.6
		 */
		String descriptionKey() default "";

		/**
		 * Use this attribute to control whether user input for an interactive option is
		 * echoed to the console or not. If {@code echo = true}, the user input is
		 * echoed to the console. This attribute is ignored when
		 * {@code interactive = false} (the default).
		 * 
		 * @return whether the user input for an interactive option should be echoed to
		 *         the console or not
		 * @see OptionSpec#echo()
		 * @since 4.6
		 */
		boolean echo() default false;

		/**
		 * For options with an optional parameter (for example, {@code arity = "0..1"}),
		 * this value is assigned to the annotated element if the option is specified on
		 * the command line without an option parameter.
		 * <p>
		 * This is different from the {@link #defaultValue()}, which is assigned if the
		 * option is not specified at all on the command line.
		 * </p>
		 * <p>
		 * Using a {@code fallbackValue} allows applications to distinguish between
		 * </p>
		 * <ul>
		 * <li>option was not specified on the command line (default value
		 * assigned)</li>
		 * <li>option was specified without parameter on the command line (fallback
		 * value assigned)</li>
		 * <li>option was specified with parameter on the command line (command line
		 * argument value assigned)</li>
		 * </ul>
		 * <p>
		 * This is useful to define options that can function as a boolean "switch" and
		 * optionally allow users to provide a (strongly typed) extra parameter value.
		 * </p>
		 * <p>
		 * Use the special value {@link Option#NULL_VALUE} to specify {@code null} - for
		 * options of type {@code Optional<T>} that will result in the
		 * {@code Optional.empty()} value being assigned when the option name is
		 * specified without a parameter on the command line.
		 * </p>
		 * 
		 * @see OptionSpec#fallbackValue()
		 * @since 4.0
		 */
		String fallbackValue() default "";

		/**
		 * <p>
		 * This should rarely be used: the recommended attributes are
		 * {@link #usageHelp() usageHelp} and {@link #versionHelp() versionHelp}.
		 * </p>
		 * <p>
		 * Only set {@code help=true} when this option should disable validation of the
		 * remaining arguments, and no error message should be generated for missing
		 * required options.
		 * </p>
		 * <p>
		 * This is useful for custom help options that are in addition to the standard
		 * help and version options. For example if your application has many hidden
		 * options or subcommands, and there is a custom help option like
		 * {@code --detailed-help} that prints the usage help message for these hidden
		 * options and subcommands.
		 * </p>
		 * <p>
		 * <b>Note:</b>
		 * </p>
		 * <p>
		 * Use the {@link #usageHelp() usageHelp} for "normal" help options (like
		 * {@code -h} and {@code --help} on unix, {@code -?} and {@code -Help} on
		 * Windows) and use {@link #versionHelp() versionHelp} for "normal" version help
		 * ({@code -V} and {@code --version} on unix, {@code -Version} on Windows):
		 * picocli has built-in logic so that options with {@code usageHelp=true} or
		 * {@code versionHelp=true} will automatically cause the requested help message
		 * to be printed in applications that use the {@link #execute(String...)}
		 * method, without any code in the application.
		 * </p>
		 * <p>
		 * Note that there is no such automatic help printing for options with
		 * {@code help=true}; applications need to check whether the end user specified
		 * this option and take appropriate action in the business logic of the
		 * application.
		 * </p>
		 * 
		 * @return whether this option disables validation of the other arguments
		 */
		boolean help() default false;

		/**
		 * Set {@code hidden=true} if this option should not be included in the usage
		 * help message.
		 * 
		 * @return whether this option should be excluded from the usage documentation
		 */
		boolean hidden() default false;

		/**
		 * Returns whether usage syntax decorations around the {@linkplain #paramLabel()
		 * paramLabel} should be suppressed. The default is {@code false}: by default,
		 * the paramLabel is surrounded with {@code '['} and {@code ']'} characters if
		 * the value is optional and followed by ellipses ("...") when multiple values
		 * can be specified.
		 * 
		 * @since 3.6.0
		 */
		boolean hideParamSyntax() default false;

		/**
		 * Set {@code interactive=true} to make this option prompt the end user for a
		 * value (like a password). Only supported for single-value options and
		 * {@code char[]} arrays (no collections, maps or other array types). When
		 * running on Java 6 or greater and {@link Option#echo() echo = false} (the
		 * default), this will use the {@link Console#readPassword()} API to get a value
		 * without echoing input to the console, otherwise it will simply read a value
		 * from {@code System.in}.
		 * <p>
		 * For passwords, best security practice is to use type {@code char[]} instead
		 * of {@code String}, and to to null out the array after use.
		 * </p>
		 * <p>
		 * When defined with {@code arity = "0..1"}, the option can also take a value
		 * from the command line. (The user will still be prompted if no option
		 * parameter was specified on the command line.) This is useful for commands
		 * that need to be run interactively as well as in batch mode.
		 * </p>
		 * 
		 * @return whether this option prompts the end user for a value to be entered on
		 *         the command line
		 * @since 3.5
		 */
		boolean interactive() default false;

		/**
		 * For options of type Map, setting the {@code mapFallbackValue} to any value
		 * allows end user to specify key-only parameters for this option. For example,
		 * {@code -Dkey} instead of {@code -Dkey=value}.
		 * <p>
		 * The value specified in this annotation is the value that is put into the Map
		 * for the user-specified key. Use the special value {@link Option#NULL_VALUE}
		 * to specify {@code null} - for maps of type {@code Map<K, Optional<V>>} that
		 * will result in {@code Optional.empty()} values in the map when only the key
		 * is specified.
		 * </p>
		 * <p>
		 * If no {@code mapFallbackValue} is set, key-only Map parameters like
		 * {@code -Dkey} are considered invalid user input and cause a
		 * {@link ParameterException} to be thrown.
		 * </p>
		 * 
		 * @see ArgSpec#mapFallbackValue()
		 * @since 4.6
		 */
		String mapFallbackValue() default ArgSpec.UNSPECIFIED;

		/**
		 * One or more option names. At least one option name is required.
		 * <p>
		 * Different environments have different conventions for naming options, but
		 * usually options have a prefix that sets them apart from parameters. Picocli
		 * supports all of the below styles. The default separator is {@code '='}, but
		 * this can be configured.
		 * </p>
		 * <p>
		 * <b>*nix</b>
		 * </p>
		 * <p>
		 * In Unix and Linux, options have a short (single-character) name, a long name
		 * or both. Short options (<a href=
		 * "http://pubs.opengroup.org/onlinepubs/9699919799/basedefs/V1_chap12.html#tag_12_02">POSIX
		 * style</a> are single-character and are preceded by the {@code '-'} character,
		 * e.g., {@code `-v'}. <a href=
		 * "https://www.gnu.org/software/tar/manual/html_node/Long-Options.html">GNU-style</a>
		 * long (or <em>mnemonic</em>) options start with two dashes in a row, e.g.,
		 * {@code `--file'}.
		 * </p>
		 * <p>
		 * Picocli supports the POSIX convention that short options can be grouped, with
		 * the last option optionally taking a parameter, which may be attached to the
		 * option name or separated by a space or a {@code '='} character. The below
		 * examples are all equivalent:
		 * </p>
		 * 
		 * <pre>
		 * -xvfFILE
		 * -xvf FILE
		 * -xvf=FILE
		 * -xv --file FILE
		 * -xv --file=FILE
		 * -x -v --file FILE
		 * -x -v --file=FILE
		 * </pre>
		 * <p>
		 * <b>DOS</b>
		 * </p>
		 * <p>
		 * DOS options mostly have upper case single-character names and start with a
		 * single slash {@code '/'} character. Option parameters are separated by a
		 * {@code ':'} character. Options cannot be grouped together but must be
		 * specified separately. For example:
		 * </p>
		 * 
		 * <pre>
		 * DIR /S /A:D /T:C
		 * </pre>
		 * <p>
		 * <b>PowerShell</b>
		 * </p>
		 * <p>
		 * Windows PowerShell options generally are a word preceded by a single
		 * {@code '-'} character, e.g., {@code `-Help'}. Option parameters are separated
		 * by a space or by a {@code ':'} character.
		 * </p>
		 * 
		 * @return one or more option names
		 */
		String[] names();

		/**
		 * (Only for boolean options): set this to automatically add a negative version
		 * for this boolean option. For example, for a {@code --force} option the
		 * negative version would be {@code --no-force}, and for a
		 * {@code -XX:+PrintGCDetails} option, the negative version would be
		 * {@code -XX:-PrintGCDetails}. The synopsis would show {@code --[no-]force} and
		 * {@code -XX:(+|-)PrintGCDetails}, respectively.
		 * <p>
		 * The form of the negative name can be customized by modifying the regular
		 * expressions used by {@linkplain RegexTransformer#createDefault() default}, or
		 * by replacing the default {@link INegatableOptionTransformer} with a custom
		 * implementation entirely.
		 * </p>
		 * <p>
		 * Negative option names used to parse the command line are collected when the
		 * command is constructed (so any variables in the option names will be resolved
		 * at that time). Documentation strings for negatable options are generated on
		 * demand when the usage help message is shown.
		 * </p>
		 * 
		 * @see CommandLine#getNegatableOptionTransformer()
		 * @see CommandLine#setNegatableOptionTransformer(INegatableOptionTransformer)
		 * @since 4.0
		 */
		boolean negatable() default false;

		/**
		 * When {@link Command#sortOptions() @Command(sortOptions = false)} is
		 * specified, this attribute can be used to control the order in which options
		 * are listed in the usage help message. When
		 * {@link Command#sortSynopsis() @Command(sortSynopsis = false)} is specified,
		 * this attribute controls the order in which options appear in the synopsis of
		 * the usage help message.
		 * 
		 * @return the position in the options list at which this option should be
		 *         shown. Options with a lower number are shown before options with a
		 *         higher number. Gaps are allowed.
		 * @since 3.9
		 */
		int order() default -1;

		/**
		 * Optionally specify a custom {@code IParameterConsumer} to temporarily suspend
		 * picocli's parsing logic and process one or more command line arguments in a
		 * custom manner. This may be useful when passing arguments through to another
		 * program.
		 * 
		 * @since 4.0
		 */
		Class<? extends IParameterConsumer> parameterConsumer() default NullParameterConsumer.class;

		/**
		 * Specify a {@code paramLabel} for the option parameter to be used in the usage
		 * help message. If omitted, picocli uses the field name in fish brackets
		 * ({@code '<'} and {@code '>'}) by default. Example:
		 * 
		 * <pre>
		 * class Example {
		 * 	&#064;Option(names = { "-o", "--output" }, paramLabel = "FILE", description = "path of the output file")
		 * 	private File out;
		 * 	&#064;Option(names = { "-j",
		 * 			"--jobs" }, arity = "0..1", description = "Allow N jobs at once; infinite jobs with no arg.")
		 * 	private int maxJobs = -1;
		 * }
		 * </pre>
		 * <p>
		 * By default, the above gives a usage help message like the following:
		 * </p>
		 * 
		 * <pre>
		 * Usage: &lt;main class&gt; [OPTIONS]
		 * -o, --output FILE       path of the output file
		 * -j, --jobs [&lt;maxJobs&gt;]  Allow N jobs at once; infinite jobs with no arg.
		 * </pre>
		 * 
		 * @return name of the option parameter used in the usage help message
		 */
		String paramLabel() default "";

		/**
		 * Returns the preprocessor for this option.
		 * 
		 * @see IParameterPreprocessor
		 * @since 4.6
		 */
		Class<? extends IParameterPreprocessor> preprocessor() default NoOpParameterPreprocessor.class;

		/**
		 * Use this attribute to customize the text displayed to the end user for an
		 * interactive option when asking for user input. When omitted, the displayed
		 * text is derived from the option name and the first description line. This
		 * attribute is ignored when {@code interactive = false} (the default).
		 * 
		 * @return the text to display to the end user for an interactive option when
		 *         asking for user input
		 * @see OptionSpec#prompt()
		 * @since 4.6
		 */
		String prompt() default "";

		/**
		 * Indicates whether this option is required. By default this is false.
		 * <p>
		 * If an option is required, but a user invokes the program without specifying
		 * the required option, a {@link MissingParameterException} is thrown from the
		 * {@link #parse(String...)} method.
		 * </p>
		 * <p>
		 * Required options that are part of a {@linkplain ArgGroup group} are required
		 * <em>within the group</em>, not required within the command: the group's
		 * {@linkplain ArgGroup#multiplicity() multiplicity} determines whether the
		 * group itself is required or optional.
		 * </p>
		 * 
		 * @return whether this option is required
		 */
		boolean required() default false;

		/**
		 * Determines on which command(s) this option exists: on this command only (the
		 * default), or whether this is a "global" option that is applied to this
		 * command and all subcommands, sub-subcommands, etc.
		 * 
		 * @since 4.3
		 */
		ScopeType scope() default ScopeType.LOCAL;

		/**
		 * Use this attribute to control for a specific option whether its default value
		 * should be shown in the usage help message. If not specified, the default
		 * value is only shown when the {@link Command#showDefaultValues()} is set
		 * {@code true} on the command. Use this attribute to specify whether the
		 * default value for this specific option should always be shown or never be
		 * shown, regardless of the command setting.
		 * <p>
		 * Note that picocli 3.2 allows {@linkplain #description() embedding default
		 * values} by specifying the variable {@code ${DEFAULT-VALUE}} anywhere in the
		 * description that ignores this setting.
		 * </p>
		 * 
		 * @return whether this option's default value should be shown in the usage help
		 *         message
		 */
		Help.Visibility showDefaultValue() default Help.Visibility.ON_DEMAND;

		/**
		 * Specify a regular expression to use to split option parameter values before
		 * applying them to the field. All elements resulting from the split are added
		 * to the array or Collection. Previously ignored for single-value fields, from
		 * picocli 4.0 a {@code split} regex can only be specified on multi-value
		 * options and positional parameters.
		 * 
		 * @return a regular expression to split option parameter values or {@code ""}
		 *         if the value should not be split
		 * @see String#split(String)
		 */
		String split() default "";

		/**
		 * Specify the string to display for the {@link #split split} regular expression
		 * in the usage help synopsis.
		 * 
		 * @since 4.3
		 */
		String splitSynopsisLabel() default "";

		/**
		 * <p>
		 * Optionally specify a {@code type} to control exactly what Class the option
		 * parameter should be converted to. This may be useful when the field type is
		 * an interface or an abstract class. For example, a field can be declared to
		 * have type {@code java.lang.Number}, and annotating
		 * {@code @Option(type=Short.class)} ensures that the option parameter value is
		 * converted to a {@code Short} before setting the field value.
		 * </p>
		 * <p>
		 * For array fields whose <em>component</em> type is an interface or abstract
		 * class, specify the concrete <em>component</em> type. For example, a field
		 * with type {@code Number[]} may be annotated with
		 * {@code @Option(type=Short.class)} to ensure that option parameter values are
		 * converted to {@code Short} before adding an element to the array.
		 * </p>
		 * <p>
		 * Picocli will use the {@link ITypeConverter} that is
		 * {@linkplain #registerConverter(Class, ITypeConverter) registered} for the
		 * specified type to convert the raw String values before modifying the field
		 * value.
		 * </p>
		 * <p>
		 * Prior to 2.0, the {@code type} attribute was necessary for {@code Collection}
		 * and {@code Map} fields, but starting from 2.0 picocli will infer the
		 * component type from the generic type's type arguments. For example, for a
		 * field of type {@code Map<TimeUnit, Long>} picocli will know the option
		 * parameter should be split up in key=value pairs, where the key should be
		 * converted to a {@code java.util.concurrent.TimeUnit} enum value, and the
		 * value should be converted to a {@code Long}. No {@code @Option(type=...)}
		 * type attribute is required for this. For generic types with wildcards,
		 * picocli will take the specified upper or lower bound as the Class to convert
		 * to, unless the {@code @Option} annotation specifies an explicit {@code type}
		 * attribute.
		 * </p>
		 * <p>
		 * If the field type is a raw collection or a raw map, and you want it to
		 * contain other values than Strings, or if the generic type's type arguments
		 * are interfaces or abstract classes, you may specify a {@code type} attribute
		 * to control the Class that the option parameter should be converted to.
		 * 
		 * @return the type(s) to convert the raw String values
		 */
		Class<?>[] type() default {};

		/**
		 * Set {@code usageHelp=true} for the {@code --help} option that triggers
		 * display of the usage help message. The
		 * <a href="https://picocli.info/#_printing_help_automatically">convenience
		 * methods</a> {@code Commandline.call}, {@code Commandline.run}, and
		 * {@code Commandline.parseWithHandler(s)} will automatically print usage help
		 * when an option with {@code usageHelp=true} was specified on the command line.
		 * <p>
		 * By default, <em>all</em> options and positional parameters are included in
		 * the usage help message <em>except when explicitly marked
		 * {@linkplain #hidden() hidden}.</em>
		 * </p>
		 * <p>
		 * If this option is specified on the command line, picocli will not validate
		 * the remaining arguments (so no "missing required option" errors) and the
		 * {@link CommandLine#isUsageHelpRequested()} method will return {@code true}.
		 * </p>
		 * <p>
		 * Alternatively, consider annotating your command with
		 * {@linkplain Command#mixinStandardHelpOptions() @Command(mixinStandardHelpOptions
		 * = true)}.
		 * </p>
		 * 
		 * @return whether this option allows the user to request usage help
		 * @since 0.9.8
		 * @see #hidden()
		 * @see #run(Runnable, String...)
		 * @see #call(Callable, String...)
		 * @see #parseWithHandler(IParseResultHandler2, String[])
		 * @see #printHelpIfRequested(List, PrintStream, PrintStream, Help.Ansi)
		 */
		boolean usageHelp() default false;

		/**
		 * Set {@code versionHelp=true} for the {@code --version} option that triggers
		 * display of the version information. The
		 * <a href="https://picocli.info/#_printing_help_automatically">convenience
		 * methods</a> {@code Commandline.call}, {@code Commandline.run}, and
		 * {@code Commandline.parseWithHandler(s)} will automatically print version
		 * information when an option with {@code versionHelp=true} was specified on the
		 * command line.
		 * <p>
		 * The version information string is obtained from the command's
		 * {@linkplain Command#version() version} annotation or from the
		 * {@linkplain Command#versionProvider() version provider}.
		 * </p>
		 * <p>
		 * If this option is specified on the command line, picocli will not validate
		 * the remaining arguments (so no "missing required option" errors) and the
		 * {@link CommandLine#isUsageHelpRequested()} method will return {@code true}.
		 * </p>
		 * <p>
		 * Alternatively, consider annotating your command with
		 * {@linkplain Command#mixinStandardHelpOptions() @Command(mixinStandardHelpOptions
		 * = true)}.
		 * </p>
		 * 
		 * @return whether this option allows the user to request version information
		 * @since 0.9.8
		 * @see #hidden()
		 * @see #run(Runnable, String...)
		 * @see #call(Callable, String...)
		 * @see #parseWithHandler(IParseResultHandler2, String[])
		 * @see #printHelpIfRequested(List, PrintStream, PrintStream, Help.Ansi)
		 */
		boolean versionHelp() default false;
	}

	/**
	 * Exception indicating that an option for a single-value option field has been
	 * specified multiple times on the command line.
	 */
	public static class OverwrittenOptionException extends ParameterException {
		private static final long serialVersionUID = 1338029208271055776L;
		private final ArgSpec overwrittenArg;

		public OverwrittenOptionException(CommandLine commandLine, ArgSpec overwritten, String msg) {
			super(commandLine, msg);
			overwrittenArg = overwritten;
		}

		/**
		 * Returns the {@link ArgSpec} for the option which was being overwritten.
		 * 
		 * @since 3.8
		 */
		public ArgSpec getOverwritten() {
			return overwrittenArg;
		}
	}

	/**
	 * Exception indicating something went wrong while parsing command line options.
	 */
	public static class ParameterException extends PicocliException {
		private static final long serialVersionUID = 1477112829129763139L;

		private static ParameterException create(CommandLine cmd, Exception ex, String arg, int i, String[] args) {
			final String msg = ex.getClass().getSimpleName() + ": " + ex.getLocalizedMessage()
					+ " while processing argument at or before arg[" + i + "] '" + arg + "' in " + Arrays.toString(args)
					+ ": " + ex.toString();
			return new ParameterException(cmd, msg, ex, null, arg);
		}

		protected final CommandLine commandLine;
		private ArgSpec argSpec = null;

		private String value = null;

		/**
		 * Constructs a new ParameterException with the specified CommandLine and error
		 * message.
		 * 
		 * @param commandLine the command or subcommand whose input was invalid
		 * @param msg         describes the problem
		 * @since 2.0
		 */
		public ParameterException(CommandLine commandLine, String msg) {
			super(msg);
			this.commandLine = Assert.notNull(commandLine, "commandLine");
		}

		/**
		 * Constructs a new ParameterException with the specified CommandLine and error
		 * message.
		 * 
		 * @param commandLine the command or subcommand whose input was invalid
		 * @param msg         describes the problem
		 * @param argSpec     the argSpec that caused this ParameterException
		 * @param value       the value that caused this ParameterException
		 * @since 3.2
		 */
		public ParameterException(CommandLine commandLine, String msg, ArgSpec argSpec, String value) {
			super(msg);
			this.commandLine = Assert.notNull(commandLine, "commandLine");
			if (argSpec == null && value == null) {
				throw new IllegalArgumentException("ArgSpec and value cannot both be null");
			}
			this.argSpec = argSpec;
			this.value = value;
		}

		/**
		 * Constructs a new ParameterException with the specified CommandLine and error
		 * message.
		 * 
		 * @param commandLine the command or subcommand whose input was invalid
		 * @param msg         describes the problem
		 * @param t           the throwable that caused this ParameterException
		 * @since 2.0
		 */
		public ParameterException(CommandLine commandLine, String msg, Throwable t) {
			super(msg, t);
			this.commandLine = Assert.notNull(commandLine, "commandLine");
		}

		/**
		 * Constructs a new ParameterException with the specified CommandLine and error
		 * message.
		 * 
		 * @param commandLine the command or subcommand whose input was invalid
		 * @param msg         describes the problem
		 * @param t           the throwable that caused this ParameterException
		 * @param argSpec     the argSpec that caused this ParameterException
		 * @param value       the value that caused this ParameterException
		 * @since 3.2
		 */
		public ParameterException(CommandLine commandLine, String msg, Throwable t, ArgSpec argSpec, String value) {
			super(msg, t);
			this.commandLine = Assert.notNull(commandLine, "commandLine");
			if (argSpec == null && value == null) {
				throw new IllegalArgumentException("ArgSpec and value cannot both be null");
			}
			this.argSpec = argSpec;
			this.value = value;
		}

		/**
		 * Returns the {@code ArgSpec} object for the (sub)command whose input could not
		 * be parsed.
		 * 
		 * @return the {@code ArgSpec} object for the (sub)command where parsing failed.
		 * @since 3.2
		 */
		public ArgSpec getArgSpec() {
			return argSpec;
		}

		/**
		 * Returns the {@code CommandLine} object for the (sub)command whose input could
		 * not be parsed.
		 * 
		 * @return the {@code CommandLine} object for the (sub)command where parsing
		 *         failed.
		 * @since 2.0
		 */
		public CommandLine getCommandLine() {
			return commandLine;
		}

		/**
		 * Returns the {@code String} value for the (sub)command whose input could not
		 * be parsed.
		 * 
		 * @return the {@code String} value for the (sub)command where parsing failed.
		 * @since 3.2
		 */
		public String getValue() {
			return value;
		}
	}

	/**
	 * Exception indicating that there was a gap in the indices of the fields
	 * annotated with {@link Parameters}.
	 */
	public static class ParameterIndexGapException extends InitializationException {
		private static final long serialVersionUID = -1520981133257618319L;

		public ParameterIndexGapException(String msg) {
			super(msg);
		}
	}

	/**
	 * <p>
	 * Fields annotated with {@code @Parameters} will be initialized with positional
	 * parameters. By specifying the {@link #index()} attribute you can pick the
	 * exact position or a range of positional parameters to apply. If no index is
	 * specified, the field will get all positional parameters (and so it should be
	 * an array or a collection).
	 * </p>
	 * <p>
	 * In the case of command methods (annotated with {@code @Command}), method
	 * parameters may be annotated with {@code @Parameters}, but are are considered
	 * positional parameters by default, unless they are annotated with
	 * {@code @Option}.
	 * </p>
	 * <p>
	 * Command class example:
	 * </p>
	 * 
	 * <pre>
	 * import static picocli.CommandLine.*;
	 *
	 * public class MyCalcParameters {
	 * 	&#064;Parameters(description = "Any number of input numbers")
	 * 	private List&lt;BigDecimal&gt; files = new ArrayList&lt;BigDecimal&gt;();
	 *
	 * 	&#064;Option(names = { "-h", "--help" }, usageHelp = true, description = "Display this help and exit")
	 * 	private boolean help;
	 * }
	 * </pre>
	 * <p>
	 * A field cannot be annotated with both {@code @Parameters} and {@code @Option}
	 * or a {@code ParameterException} is thrown.
	 * </p>
	 */
	@Retention(RetentionPolicy.RUNTIME)
	@Target({ ElementType.FIELD, ElementType.METHOD, ElementType.PARAMETER })
	public @interface Parameters {
		/**
		 * Special value that can be used in some annotation attributes to designate
		 * {@code null}.
		 * 
		 * @see Parameters#defaultValue()
		 * @see Parameters#mapFallbackValue()
		 * @since 4.6
		 */
		public static final String NULL_VALUE = ArgSpec.NULL_VALUE;

		/**
		 * Specifies the minimum number of required parameters and the maximum number of
		 * accepted parameters. If a positive arity is declared, and the user specifies
		 * an insufficient number of parameters on the command line,
		 * {@link MissingParameterException} is thrown by the {@link #parse(String...)}
		 * method.
		 * <p>
		 * The default depends on the type of the parameter: booleans require no
		 * parameters, arrays and Collections accept zero to any number of parameters,
		 * and any other type accepts one parameter.
		 * </p>
		 * <p>
		 * For single-value parameters, setting {@code arity = "0..1"} makes a
		 * positional parameter optional, while setting {@code arity = "1"} makes it
		 * required.
		 * </p>
		 * <p>
		 * Required parameters that are part of a {@linkplain ArgGroup group} are
		 * required <em>within the group</em>, not required within the command: the
		 * group's {@linkplain ArgGroup#multiplicity() multiplicity} determines whether
		 * the group itself is required or optional.
		 * </p>
		 * 
		 * @return the range of minimum and maximum parameters accepted by this command
		 */
		String arity() default "";

		/**
		 * Use this attribute to specify an {@code Iterable<String>} class that
		 * generates completion candidates for this positional parameter. For map
		 * fields, completion candidates should be in {@code key=value} form.
		 * <p>
		 * Completion candidates are used in bash completion scripts generated by the
		 * {@code picocli.AutoComplete} class. Unfortunately,
		 * {@code picocli.AutoComplete} is not very good yet at generating completions
		 * for positional parameters.
		 * </p>
		 *
		 * @return a class whose instances can iterate over the completion candidates
		 *         for this positional parameter
		 * @see org.rossonet.ext.picocli.CommandLine.IFactory
		 * @since 3.2
		 */
		Class<? extends Iterable<String>> completionCandidates() default NoCompletionCandidates.class;

		/**
		 * Optionally specify one or more {@link ITypeConverter} classes to use to
		 * convert the command line argument into a strongly typed value (or key-value
		 * pair for map fields). This is useful when a particular field should use a
		 * custom conversion that is different from the normal conversion for the
		 * field's type.
		 * <p>
		 * For example, for a specific field you may want to use a converter that maps
		 * the constant names defined in {@link java.sql.Types java.sql.Types} to the
		 * {@code int} value of these constants, but any other {@code int} fields should
		 * not be affected by this and should continue to use the standard int converter
		 * that parses numeric values.
		 * </p>
		 * 
		 * @return the type converter(s) to use to convert String values to strongly
		 *         typed values for this field
		 * @see CommandLine#registerConverter(Class, ITypeConverter)
		 */
		Class<? extends ITypeConverter<?>>[] converter() default {};

		/**
		 * Returns the default value of this positional parameter, before splitting and
		 * type conversion.
		 * <p>
		 * To get a {@code null} default value, omit specifying a default value or use
		 * the special value {@link Parameters#NULL_VALUE} - for positional parameters
		 * of type {@code Optional<T>} that will result in the {@code Optional.empty()}
		 * value being assigned when the positional parameters is not specified on the
		 * command line.
		 * </p>
		 * 
		 * @return a String that (after type conversion) will be used as the value for
		 *         this positional parameter if no value was specified on the command
		 *         line
		 * @since 3.2
		 */
		String defaultValue() default "__no_default_value__";

		/**
		 * Description of the parameter(s), used when generating the usage
		 * documentation. Each element of the array is rendered on a separate line.
		 * <p>
		 * May contain embedded {@linkplain java.util.Formatter format specifiers} like
		 * {@code %n} line separators. Literal percent {@code '%'} characters must be
		 * escaped with another {@code %}.
		 * </p>
		 * <p>
		 * The description may contain variables that are rendered when help is
		 * requested. The string {@code ${DEFAULT-VALUE}} is replaced with the default
		 * value of the positional parameter. This is regardless of the command's
		 * {@link Command#showDefaultValues() showDefaultValues} setting or the
		 * positional parameter's {@link #showDefaultValue() showDefaultValue} setting.
		 * The string {@code ${COMPLETION-CANDIDATES}} is replaced with the completion
		 * candidates generated by {@link #completionCandidates()} in the description
		 * for this positional parameter. Also, embedded {@code %n} newline markers are
		 * converted to actual newlines.
		 * </p>
		 * 
		 * @return the description of the parameter(s)
		 * @see <a href="https://picocli.info/#_variable_interpolation">Variable
		 *      Interpolation section of the user manual</a>
		 */
		String[] description() default {};

		/**
		 * ResourceBundle key for this option. If not specified, (and a ResourceBundle
		 * {@linkplain Command#resourceBundle() exists for this command}) an attempt is
		 * made to find the positional parameter description using
		 * {@code paramLabel() + "[" + index() + "]"} as key.
		 *
		 * @see PositionalParamSpec#description()
		 * @since 3.6
		 */
		String descriptionKey() default "";

		/**
		 * Use this attribute to control whether user input for an interactive
		 * positional parameter is echoed to the console or not. If {@code echo = true},
		 * the user input is echoed to the console. This attribute is ignored when
		 * {@code interactive = false} (the default).
		 * 
		 * @return whether the user input for an interactive positional parameter should
		 *         be echoed to the console or not
		 * @see PositionalParamSpec#echo()
		 * @since 4.6
		 */
		boolean echo() default false;

		/**
		 * Set {@code hidden=true} if this parameter should not be included in the usage
		 * message.
		 * 
		 * @return whether this parameter should be excluded from the usage message
		 */
		boolean hidden() default false;

		/**
		 * Returns whether usage syntax decorations around the {@linkplain #paramLabel()
		 * paramLabel} should be suppressed. The default is {@code false}: by default,
		 * the paramLabel is surrounded with {@code '['} and {@code ']'} characters if
		 * the value is optional and followed by ellipses ("...") when multiple values
		 * can be specified.
		 * 
		 * @since 3.6.0
		 */
		boolean hideParamSyntax() default false;

		/**
		 * Specify an index ("0", or "1", etc.) to pick which of the command line
		 * arguments should be assigned to this field. For array or Collection fields,
		 * you can also specify an index range ("0..3", or "2..*", etc.) to assign a
		 * subset of the command line arguments to this field. The default is "*",
		 * meaning all command line arguments.
		 * 
		 * @return an index or range specifying which of the command line arguments
		 *         should be assigned to this field
		 */
		String index() default "";

		/**
		 * Set {@code interactive=true} if this positional parameter will prompt the end
		 * user for a value (like a password). Only supported for single-value
		 * positional parameters (not arrays, collections or maps). When running on Java
		 * 6 or greater and {@link Option#echo() echo = false} (the default), this will
		 * use the {@link Console#readPassword()} API to get a value without echoing
		 * input to the console, otherwise it will simply read a value from
		 * {@code System.in}.
		 * 
		 * @return whether this positional parameter prompts the end user for a value to
		 *         be entered on the command line
		 * @since 3.5
		 */
		boolean interactive() default false;

		/**
		 * For positional parameters of type Map, setting the {@code mapFallbackValue}
		 * to any value allows end user to specify key-only parameters for this
		 * parameter. For example, {@code key} instead of {@code key=value}.
		 * <p>
		 * The value specified in this annotation is the value that is put into the Map
		 * for the user-specified key. Use the special value
		 * {@link Parameters#NULL_VALUE} to specify {@code null} - for maps of type
		 * {@code Map<K, Optional<V>>} that will result in {@code Optional.empty()}
		 * values in the map when only the key is specified.
		 * </p>
		 * <p>
		 * If no {@code mapFallbackValue} is set, key-only Map parameters like
		 * {@code -Dkey} are considered invalid user input and cause a
		 * {@link ParameterException} to be thrown.
		 * </p>
		 * 
		 * @see ArgSpec#mapFallbackValue()
		 * @since 4.6
		 */
		String mapFallbackValue() default ArgSpec.UNSPECIFIED;

		/**
		 * Optionally specify a custom {@code IParameterConsumer} to temporarily suspend
		 * picocli's parsing logic and process one or more command line arguments in a
		 * custom manner.
		 * 
		 * @since 4.0
		 */
		Class<? extends IParameterConsumer> parameterConsumer() default NullParameterConsumer.class;

		/**
		 * Specify a {@code paramLabel} for the parameter to be used in the usage help
		 * message. If omitted, picocli uses the field name in fish brackets
		 * ({@code '<'} and {@code '>'}) by default. Example:
		 * 
		 * <pre>
		 * class Example {
		 * 	&#064;Parameters(paramLabel = "FILE", description = "path of the input FILE(s)")
		 * 	private File[] inputFiles;
		 * }
		 * </pre>
		 * <p>
		 * By default, the above gives a usage help message like the following:
		 * </p>
		 * 
		 * <pre>
		 * Usage: &lt;main class&gt; [FILE...]
		 * [FILE...]       path of the input FILE(s)
		 * </pre>
		 * 
		 * @return name of the positional parameter used in the usage help message
		 */
		String paramLabel() default "";

		/**
		 * Returns the preprocessor for this positional parameter.
		 * 
		 * @see IParameterPreprocessor
		 * @since 4.6
		 */
		Class<? extends IParameterPreprocessor> preprocessor() default NoOpParameterPreprocessor.class;

		/**
		 * Use this attribute to customize the text displayed to the end user for an
		 * interactive positional parameter when asking for user input. When omitted,
		 * the displayed text is derived from the positional parameter's position
		 * (index) and the first description line. This attribute is ignored when
		 * {@code interactive = false} (the default).
		 * 
		 * @return the text to display to the end user for an interactive positional
		 *         parameter when asking for user input
		 * @see PositionalParamSpec#prompt()
		 * @since 4.6
		 */
		String prompt() default "";

		/**
		 * Determines on which command(s) this positional parameter exists: on this
		 * command only (the default), or whether this is a "global" parameter that is
		 * applied to this command and all subcommands, sub-subcommands, etc.
		 * 
		 * @since 4.3
		 */
		ScopeType scope() default ScopeType.LOCAL;

		/**
		 * Use this attribute to control for a specific positional parameter whether its
		 * default value should be shown in the usage help message. If not specified,
		 * the default value is only shown when the {@link Command#showDefaultValues()}
		 * is set {@code true} on the command. Use this attribute to specify whether the
		 * default value for this specific positional parameter should always be shown
		 * or never be shown, regardless of the command setting.
		 * <p>
		 * Note that picocli 3.2 allows {@linkplain #description() embedding default
		 * values} by specifying the variable {@code ${DEFAULT-VALUE}} anywhere in the
		 * description that ignores this setting.
		 * </p>
		 * 
		 * @return whether this positional parameter's default value should be shown in
		 *         the usage help message
		 */
		Help.Visibility showDefaultValue() default Help.Visibility.ON_DEMAND;

		/**
		 * Specify a regular expression to use to split positional parameter values
		 * before applying them to the field. All elements resulting from the split are
		 * added to the array or Collection. Previously ignored for single-value fields,
		 * from picocli 4.0 a {@code split} regex can only be specified on multi-value
		 * options and positional parameters.
		 * 
		 * @return a regular expression to split operand values or {@code ""} if the
		 *         value should not be split
		 * @see String#split(String)
		 */
		String split() default "";

		/**
		 * Specify a string to show the split option parameter values in usage
		 * 
		 * @since 4.3
		 */
		String splitSynopsisLabel() default "";

		/**
		 * <p>
		 * Optionally specify a {@code type} to control exactly what Class the
		 * positional parameter should be converted to. This may be useful when the
		 * field type is an interface or an abstract class. For example, a field can be
		 * declared to have type {@code java.lang.Number}, and annotating
		 * {@code @Parameters(type=Short.class)} ensures that the positional parameter
		 * value is converted to a {@code Short} before setting the field value.
		 * </p>
		 * <p>
		 * For array fields whose <em>component</em> type is an interface or abstract
		 * class, specify the concrete <em>component</em> type. For example, a field
		 * with type {@code Number[]} may be annotated with
		 * {@code @Parameters(type=Short.class)} to ensure that positional parameter
		 * values are converted to {@code Short} before adding an element to the array.
		 * </p>
		 * <p>
		 * Picocli will use the {@link ITypeConverter} that is
		 * {@linkplain #registerConverter(Class, ITypeConverter) registered} for the
		 * specified type to convert the raw String values before modifying the field
		 * value.
		 * </p>
		 * <p>
		 * Prior to 2.0, the {@code type} attribute was necessary for {@code Collection}
		 * and {@code Map} fields, but starting from 2.0 picocli will infer the
		 * component type from the generic type's type arguments. For example, for a
		 * field of type {@code Map<TimeUnit, Long>} picocli will know the positional
		 * parameter should be split up in key=value pairs, where the key should be
		 * converted to a {@code java.util.concurrent.TimeUnit} enum value, and the
		 * value should be converted to a {@code Long}. No {@code @Parameters(type=...)}
		 * type attribute is required for this. For generic types with wildcards,
		 * picocli will take the specified upper or lower bound as the Class to convert
		 * to, unless the {@code @Parameters} annotation specifies an explicit
		 * {@code type} attribute.
		 * </p>
		 * <p>
		 * If the field type is a raw collection or a raw map, and you want it to
		 * contain other values than Strings, or if the generic type's type arguments
		 * are interfaces or abstract classes, you may specify a {@code type} attribute
		 * to control the Class that the positional parameter should be converted to.
		 * 
		 * @return the type(s) to convert the raw String values
		 */
		Class<?>[] type() default {};
	}

	/**
	 * <p>
	 * Fields annotated with {@code @ParentCommand} will be initialized with the
	 * parent command of the current subcommand. If the current command does not
	 * have a parent command, this annotation has no effect.
	 * </p>
	 * <p>
	 * Parent commands often define options that apply to all the subcommands. This
	 * annotation offers a convenient way to inject a reference to the parent
	 * command into a subcommand, so the subcommand can access its parent options.
	 * For example:
	 * </p>
	 * 
	 * <pre>
	 * &#064;Command(name = "top", subcommands = Sub.class)
	 * class Top implements Runnable {
	 *
	 * 	&#064;Option(names = { "-d", "--directory" }, description = "this option applies to all subcommands")
	 * 	File baseDirectory;
	 *
	 * 	public void run() {
	 * 		System.out.println("Hello from top");
	 * 	}
	 * }
	 *
	 * &#064;Command(name = "sub")
	 * class Sub implements Runnable {
	 *
	 * 	&#064;ParentCommand
	 * 	private Top parent;
	 *
	 * 	public void run() {
	 * 		System.out.println("Subcommand: parent command 'directory' is " + parent.baseDirectory);
	 * 	}
	 * }
	 * </pre>
	 * 
	 * @since 2.2
	 */
	@Retention(RetentionPolicy.RUNTIME)
	@Target(ElementType.FIELD)
	public @interface ParentCommand {
	}

	/**
	 * Encapsulates the result of parsing an array of command line arguments.
	 * 
	 * @since 3.0
	 */
	public static class ParseResult {
		/** Builds immutable {@code ParseResult} instances. */
		public static class Builder {
			private final CommandSpec commandSpec;
			private final List<ArgSpec> matchedArgsList = new ArrayList<ArgSpec>();
			private final List<OptionSpec> matchedOptionsList = new ArrayList<OptionSpec>();
			private final List<PositionalParamSpec> matchedPositionalsList = new ArrayList<PositionalParamSpec>();
			private final Set<OptionSpec> options = new LinkedHashSet<OptionSpec>();
			private final Set<PositionalParamSpec> positionals = new LinkedHashSet<PositionalParamSpec>();
			private final List<String> unmatched = new ArrayList<String>();
			private int firstUnmatchedPosition = Integer.MAX_VALUE;
			private final List<String> originalArgList = new ArrayList<String>();
			private final List<String> expandedArgList = new ArrayList<String>();
			private final List<List<PositionalParamSpec>> positionalParams = new ArrayList<List<PositionalParamSpec>>();
			private final List<ParseResult> subcommands = new ArrayList<ParseResult>();
			private boolean usageHelpRequested;
			private boolean versionHelpRequested;
			boolean isInitializingDefaultValues;
			private final List<Exception> errors = new ArrayList<Exception>(1);
			private List<Object> nowProcessing;
			private final GroupMatchContainer groupMatchContainer = new GroupMatchContainer(null, null);

			private Builder(CommandSpec spec) {
				commandSpec = Assert.notNull(spec, "commandSpec");
			}

			/**
			 * Adds the specified {@code OptionSpec} or {@code PositionalParamSpec} to the
			 * list of options and parameters that were matched on the command line.
			 * 
			 * @param arg      the matched {@code OptionSpec} or {@code PositionalParamSpec}
			 * @param position the command line position at which the
			 *                 {@code PositionalParamSpec} was matched. Ignored for
			 *                 {@code OptionSpec}s.
			 * @return this builder for method chaining
			 */
			public Builder add(ArgSpec arg, int position) {
				if (arg.isOption()) {
					addOption((OptionSpec) arg);
				} else {
					addPositionalParam((PositionalParamSpec) arg, position);
				}
				return this;
			}

			public void addError(PicocliException ex) {
				errors.add(Assert.notNull(ex, "exception"));
			}

			/**
			 * Adds the specified {@code OptionSpec} to the list of options that were
			 * matched on the command line.
			 */
			public Builder addOption(OptionSpec option) {
				if (!isInitializingDefaultValues) {
					options.add(option);
					matchedOptionsList.add(option);
					matchedArgsList.add(option);
				}
				return this;
			}

			void addOriginalStringValue(ArgSpec argSpec, String value) {
				if (!isInitializingDefaultValues) {
					argSpec.originalStringValues.add(value);
					if (argSpec.group() != null) {
						final GroupMatchContainer groupMatchContainer = this.groupMatchContainer
								.findLastMatchContainer(argSpec.group());
						groupMatchContainer.lastMatch().addOriginalStringValue(argSpec, value);
					}
				}
			}

			/**
			 * Adds the specified {@code PositionalParamSpec} to the list of parameters that
			 * were matched on the command line.
			 * 
			 * @param positionalParam the matched {@code PositionalParamSpec}
			 * @param position        the command line position at which the
			 *                        {@code PositionalParamSpec} was matched.
			 * @return this builder for method chaining
			 */
			public Builder addPositionalParam(PositionalParamSpec positionalParam, int position) {
				if (isInitializingDefaultValues) {
					return this;
				}
				positionals.add(positionalParam);
				matchedPositionalsList.add(positionalParam);
				matchedArgsList.add(positionalParam);
				while (positionalParams.size() <= position) {
					positionalParams.add(new ArrayList<PositionalParamSpec>());
				}
				positionalParams.get(position).add(positionalParam);
				return this;
			}

			void addStringValue(ArgSpec argSpec, String value) {
				if (!isInitializingDefaultValues) {
					argSpec.stringValues.add(value);
				}
			}

			void addTypedValues(ArgSpec argSpec, int position, Object typedValue) {
				if (!isInitializingDefaultValues) {
					argSpec.typedValues.add(typedValue);
					if (argSpec.group() == null) {
						argSpec.typedValueAtPosition.put(position, typedValue);
					} else {
						final GroupMatchContainer groupMatchContainer = this.groupMatchContainer
								.findLastMatchContainer(argSpec.group());
						groupMatchContainer.lastMatch().addMatchedValue(argSpec, position, typedValue,
								CommandLine.tracer());
					}
				}
			}

			/**
			 * Adds the specified command line argument to the list of unmatched command
			 * line arguments, and remembers the position of this argument. This position is
			 * used in the UnmatchedArgumentException message.
			 * 
			 * @position ignored if negative, remembered in firstUnmatchedPosition if it
			 *           precedes a previously set value
			 * @since 4.6
			 */
			private Builder addUnmatched(int position, String arg) {
				if (position >= 0) {
					firstUnmatchedPosition = Math.min(position, firstUnmatchedPosition);
				}
				unmatched.add(arg);
				return this;
			}

			/**
			 * Adds all elements of the specified command line arguments stack to the list
			 * of unmatched command line arguments.
			 */
			public Builder addUnmatched(Stack<String> args) {
				while (!args.isEmpty()) {
					addUnmatched(totalArgCount() - args.size(), args.pop());
				}
				return this;
			}

			/**
			 * Adds the specified command line argument to the list of unmatched command
			 * line arguments.
			 */
			public Builder addUnmatched(String arg) {
				return addUnmatched(-1, arg);
			}

			void beforeMatchingGroupElement(ArgSpec argSpec) throws Exception {
				final ArgGroupSpec group = argSpec.group();
				if (group == null || isInitializingDefaultValues) {
					return;
				}
				final Tracer tracer = CommandLine.tracer();
				final GroupMatchContainer foundGroupMatchContainer = this.groupMatchContainer
						.findOrCreateMatchingGroup(argSpec, commandSpec.commandLine);
				final GroupMatch match = foundGroupMatchContainer.lastMatch();
				final boolean greedy = true; // commandSpec.parser().greedyMatchMultiValueArgsInGroup(); // or
												// @Option(multiplicity=0..*) to control min/max matches
				final boolean allowMultipleMatchesInGroup = greedy && argSpec.isMultiValue(); // https://github.com/remkop/picocli/issues/815
				final String elementDescription = ArgSpec.describe(argSpec, "=");
				if (match.matchedMinElements() && (argSpec.required() || match.matchCount(argSpec) > 0)
						&& !allowMultipleMatchesInGroup) {
					// we need to create a new match; if maxMultiplicity has been reached, we need
					// to add a new GroupMatchContainer.
					final String previousMatch = argSpec.required() ? "is required" : "has already been matched";
					tracer.info(
							"GroupMatch %s is complete: its mandatory elements are all matched. (User object: %s.) %s %s in the group, so it starts a new GroupMatch.",
							foundGroupMatchContainer.lastMatch(), foundGroupMatchContainer.group.userObject(),
							elementDescription, previousMatch);
					foundGroupMatchContainer.addMatch(commandSpec.commandLine);
					this.groupMatchContainer.findOrCreateMatchingGroup(argSpec, commandSpec.commandLine);
				} else if (match.matchCount(argSpec) > 0 && !allowMultipleMatchesInGroup) {
					tracer.info(
							"GroupMatch %s is incomplete: its mandatory elements are not all matched. (User object: %s.) However, %s has already been matched in the group, so it starts a new GroupMatch.",
							foundGroupMatchContainer.lastMatch(), foundGroupMatchContainer.group.userObject(),
							elementDescription);
					foundGroupMatchContainer.addMatch(commandSpec.commandLine);
					this.groupMatchContainer.findOrCreateMatchingGroup(argSpec, commandSpec.commandLine);
				}
			}

			/**
			 * Creates and returns a new {@code ParseResult} instance for this builder's
			 * configuration.
			 */
			public ParseResult build() {
				return new ParseResult(this);
			}

			/**
			 * Sets the specified command line arguments after
			 * {@linkplain CommandLine#isExpandAtFiles() @-files were expanded}; these are
			 * the arguments that were actually parsed.
			 * 
			 * @see #originalArgs(String[])
			 * @see ParseResult#expandedArgs()
			 * @since 4.4
			 */
			public Builder expandedArgs(Collection<String> expandedArgs) {
				expandedArgList.addAll(expandedArgs);
				return this;
			}

			private void nowProcessing(ArgSpec spec, Object value) {
				if (nowProcessing != null && !isInitializingDefaultValues) {
					nowProcessing.add(spec.isPositional() ? spec : value);
				}
			}

			/**
			 * Sets the specified original command line arguments that were passed to the
			 * {@link CommandLine#parseArgs(String...)} method, before
			 * {@linkplain CommandLine#isExpandAtFiles() @-file expansion}.
			 * 
			 * @see #expandedArgs(Collection)
			 * @see ParseResult#originalArgs()
			 */
			public Builder originalArgs(String[] originalArgs) {
				originalArgList.addAll(Arrays.asList(originalArgs));
				return this;
			}

			/**
			 * Sets the specified {@code ParseResult} for a subcommand that was matched on
			 * the command line.
			 */
			// implementation note: this method gets called with the most recently matched
			// subcommand first, then the subcommand matched before that, etc
			public Builder subcommand(ParseResult subcommand) {
				subcommands.add(0, subcommand);
				return this;
			}

			private int totalArgCount() {
				final CommandLine c = commandSpec.root().commandLine;
				final Builder b = (c == null || c.interpreter.parseResultBuilder == null) ? this
						: c.interpreter.parseResultBuilder;
				return b.expandedArgList.size();
			}
		}

		/**
		 * A group's {@linkplain ArgGroup#multiplicity() multiplicity} specifies how
		 * many matches of a group may appear on the command line. This class models a
		 * single "match". For example, this group: {@code (-a -b) (-a -b)} requires two
		 * matches of its arguments to fully match.
		 * 
		 * @since 4.0
		 */
		public static class GroupMatch {
			int position;
			final int startPosition;
			final GroupMatchContainer container;

			Map<ArgGroupSpec, GroupMatchContainer> matchedSubgroups = new LinkedHashMap<ArgGroupSpec, GroupMatchContainer>(
					2); // preserve order: used in toString()
			Map<ArgSpec, List<Object>> matchedValues = new IdentityHashMap<ArgSpec, List<Object>>(); // identity map for
																										// performance
			Map<ArgSpec, List<String>> originalStringValues = new LinkedHashMap<ArgSpec, List<String>>(); // preserve
																											// order:
																											// used in
																											// toString()
			Map<ArgSpec, Map<Integer, List<Object>>> matchedValuesAtPosition = new IdentityHashMap<ArgSpec, Map<Integer, List<Object>>>();
			private GroupValidationResult validationResult;

			GroupMatch(GroupMatchContainer container) {
				this.container = container;
				if (!container.matches().isEmpty()) {
					position = container.matches().get(container.matches().size() - 1).position;
				}
				startPosition = position;
			}

			void addMatchedValue(ArgSpec argSpec, int matchPosition, Object stronglyTypedValue, Tracer tracer) {
				addValueToListInMap(matchedValues, argSpec, stronglyTypedValue);

				Map<Integer, List<Object>> positionalValues = matchedValuesAtPosition.get(argSpec);
				if (positionalValues == null) {
					positionalValues = new TreeMap<Integer, List<Object>>();
					matchedValuesAtPosition.put(argSpec, positionalValues);
				}
				addValueToListInMap(positionalValues, matchPosition, stronglyTypedValue);
			}

			void addOriginalStringValue(ArgSpec argSpec, String value) {
				addValueToListInMap(originalStringValues, argSpec, value);
			}

			/** Returns the container {@code GroupMatchContainer} of this match. */
			public GroupMatchContainer container() {
				return container;
			}

			/**
			 * Returns the {@code ArgGroupSpec} of the container {@code GroupMatchContainer}
			 * of this match.
			 */
			public ArgGroupSpec group() {
				return container.group;
			}

			private boolean hasFullyMatchedSubgroup(boolean allRequired) {
				for (final GroupMatchContainer sub : matchedSubgroups.values()) {
					if (sub.matchedFully(allRequired)) {
						return true;
					}
				}
				return false;
			}

			boolean hasMatchedValueAtPosition(ArgSpec arg, int position) {
				final Map<Integer, List<Object>> atPos = matchedValuesAtPosition.get(arg);
				return atPos != null && atPos.containsKey(position);
			}

			/**
			 * Returns {@code true} if this match has no matched arguments and no matched
			 * subgroups.
			 */
			public boolean isEmpty() {
				return originalStringValues.isEmpty() && matchedSubgroups.isEmpty();
			}

			int matchCount(ArgSpec argSpec) {
				return matchedValues.get(argSpec) == null ? 0 : matchedValues.get(argSpec).size();
			}

			private boolean matchedFully(boolean allRequired) {
				if (group().exclusive()) {
					return !matchedValues.isEmpty() || hasFullyMatchedSubgroup(allRequired);
				}
				for (final ArgSpec arg : group().args()) {
					if (matchedValues.get(arg) == null && (arg.required() || allRequired)) {
						return false;
					}
				}
				for (final ArgGroupSpec subgroup : group().subgroups()) {
					final GroupMatchContainer groupMatchContainer = matchedSubgroups.get(subgroup);
					if (groupMatchContainer != null) {
						if (!groupMatchContainer.matchedFully(allRequired)) {
							return false;
						}
					} else {
						if (allRequired || subgroup.multiplicity().min > 0) {
							return false;
						}
					}
				}
				return true;
			}

			/**
			 * Returns {@code true} if the maximum number of matches has been reached for
			 * this match: all arguments (required or not) have been matched, and for each
			 * subgroup, the {@linkplain GroupMatchContainer#matchedMaxElements() maximum
			 * number of elements have been matched}.
			 */
			boolean matchedMaxElements() {
				return matchedFully(true);
			}

			/**
			 * Returns {@code true} if the minimum number of elements have been reached for
			 * this match: all required arguments have been matched, and for each subgroup,
			 * the {@linkplain GroupMatchContainer#matchedMinElements() minimum number of
			 * elements have been matched}.
			 */
			boolean matchedMinElements() {
				return matchedFully(false);
			}

			/** Returns matches for the subgroups, if any. */
			public Map<ArgGroupSpec, GroupMatchContainer> matchedSubgroups() {
				return Collections.unmodifiableMap(matchedSubgroups);
			}

			/**
			 * Returns the values matched for the specified argument, converted to the type
			 * of the argument.
			 */
			public List<Object> matchedValues(ArgSpec argSpec) {
				return matchedValues.get(argSpec) == null ? Collections.emptyList()
						: Collections.unmodifiableList(matchedValues.get(argSpec));
			}

			@Override
			public String toString() {
				return toString(new StringBuilder()).toString();
			}

			private StringBuilder toString(StringBuilder result) {
				final int originalLength = result.length();
				for (final ArgSpec arg : originalStringValues.keySet()) {
					final List<String> values = originalStringValues.get(arg);
					for (final String value : values) {
						if (result.length() != originalLength) {
							result.append(" ");
						}
						result.append(ArgSpec.describe(arg, "=", value));
					}
				}
				for (final GroupMatchContainer sub : matchedSubgroups.values()) {
					if (result.length() != originalLength) {
						result.append(" ");
					}
					if (originalLength == 0) {
						result.append(sub.toString()); // include synopsis
					} else {
						sub.toString(result); // without synopsis
					}
				}
				return result;
			}

			void validate(CommandLine commandLine) {
				validationResult = GroupValidationResult.SUCCESS_PRESENT; // we matched _something_ or this object would
																			// not exist...
				if (group() != null && !group().validate()) {
					return;
				}
				for (final GroupMatchContainer sub : matchedSubgroups.values()) {
					sub.validate(commandLine);
					if (sub.validationResult.blockingFailure()) {
						this.validationResult = sub.validationResult;
						return;
					}
				}
				// finally, validate that the combination of matched args and matched subgroups
				// is valid
				if (group() != null) {

					// first validate the options and positional params in the group in isolation
					validationResult = group().validateArgs(commandLine, matchedValues.keySet());
					if (validationResult.blockingFailure()) {
						return;
					}

					// now validate the args combined with subgroups:
					// some groups may consist of a required option + a required subgroup:
					// the presence of either means that the other must occur
					final Set<ArgSpec> intersection = new LinkedHashSet<ArgSpec>(group().args());
					final Set<ArgSpec> missing = new LinkedHashSet<ArgSpec>(group().requiredArgs());
					final Set<ArgSpec> found = new LinkedHashSet<ArgSpec>(matchedValues.keySet());
					missing.removeAll(matchedValues.keySet());
					intersection.retainAll(found);
					String exclusiveElements = ArgSpec.describe(intersection);
					String requiredElements = ArgSpec.describe(group().requiredArgs());
					String missingElements = ArgSpec.describe(missing);

					final Set<ArgGroupSpec> missingSubgroups = new LinkedHashSet<ArgGroupSpec>(group().subgroups());
					missingSubgroups.removeAll(matchedSubgroups.keySet());
					int missingRequiredSubgroupCount = 0;
					for (final ArgGroupSpec missingSubgroup : missingSubgroups) {
						if (missingSubgroup.multiplicity().min() > 0) {
							missingRequiredSubgroupCount++;
							if (missingElements.length() > 0) {
								missingElements += " and ";
							}
							missingElements += missingSubgroup.synopsisUnit();
						}
					}

					for (final ArgGroupSpec subgroup : group().subgroups()) {
						if (exclusiveElements.length() > 0) {
							exclusiveElements += " and ";
						}
						exclusiveElements += subgroup.synopsisUnit();
						if (subgroup.multiplicity().min > 0) {
							if (requiredElements.length() > 0) {
								requiredElements += " and ";
							}
							requiredElements += subgroup.synopsisUnit();
						}
					}
					final int presentCount = matchedValues.size() + matchedSubgroups.size();
					final boolean haveMissing = !missing.isEmpty() || missingRequiredSubgroupCount > 0;
					validationResult = group().validate(commandLine, presentCount, haveMissing,
							presentCount > 0 && haveMissing, exclusiveElements, requiredElements, missingElements);
				}
			}
		}

		/**
		 * Provides information about an {@link ArgGroup} that was matched on the
		 * command line.
		 * <p>
		 * The {@code ParseResult} may have more than one {@code GroupMatchContainer}
		 * for an {@code ArgGroupSpec}, when the group was matched more often than its
		 * maximum {@linkplain ArgGroup#multiplicity() multiplicity}. This is not
		 * necessarily a problem: the parser will add a match to the parent matched
		 * group until the maximum multiplicity of the parent group is exceeded, in
		 * which case parser will add a match to the parent's parent group, etc.
		 * </p>
		 * <p>
		 * Ultimately, as long as the {@link ParseResult#getGroupMatches()} method does
		 * not return more than one match, the maximum number of elements is not
		 * exceeded.
		 * </p>
		 * 
		 * @since 4.0
		 */
		public static class GroupMatchContainer {
			private final ArgGroupSpec group;
			private GroupMatchContainer parentContainer;
			private final List<ArgGroupSpec> unmatchedSubgroups = new ArrayList<ArgGroupSpec>();
			private final List<GroupMatch> matches = new ArrayList<GroupMatch>();
			private GroupValidationResult validationResult;

			GroupMatchContainer(ArgGroupSpec group, CommandLine cmd) {
				this.group = group;
				addMatch(cmd);
			}

			void addMatch(CommandLine commandLine) {
				final Tracer tracer = CommandLine.tracer();
				if (group != null && isMaxMultiplicityReached()) {
					tracer.info("Completing GroupMatchContainer %s: max multiplicity is reached.", this);
					complete(commandLine);
				} else {
					if (group != null) {
						tracer.info("Adding match to GroupMatchContainer %s (group=%s %s).", this, group.id(),
								group.synopsisUnit());
					}
					matches.add(new GroupMatch(this));
					if (group == null) {
						return;
					}
				}
				group.initUserObject(commandLine);
			}

			boolean canMatchPositionalParam(PositionalParamSpec positionalParam) {
				final boolean mayCreateNewMatch = !matches.isEmpty() && lastMatch().matchedMinElements();
				final boolean mustCreateNewMatch = !matches.isEmpty() && lastMatch().matchedMaxElements();
				if (mustCreateNewMatch && isMaxMultiplicityReached()) {
					return false;
				}
				final int startPosition = matches.isEmpty() ? 0 : lastMatch().startPosition;
				final int accumulatedPosition = matches.isEmpty() ? 0 : lastMatch().position;
				int localPosition = accumulatedPosition - startPosition;
				if (mayCreateNewMatch) {
					final int positionalParamCount = positionalParam.group().localPositionalParamCount();
					if (positionalParamCount != 0) {
						localPosition %= positionalParamCount;
					} // #1213 prevent ArithmeticException: / by zero
				}
				return positionalParam.index().contains(localPosition)
						&& !lastMatch().hasMatchedValueAtPosition(positionalParam, accumulatedPosition);
			}

			void complete(CommandLine commandLine) {
				if (parentContainer == null) {
					addMatch(commandLine); // we have no choice but to potentially exceed the max multiplicity of this
											// group...
				} else {
					parentContainer.addMatch(commandLine);
				}
			}

			private boolean containsRequiredSubgroups(ArgGroupSpec argGroupSpec) {
				for (final ArgGroupSpec subgroup : argGroupSpec.subgroups()) {
					if (subgroup.exclusive()) {
						// Only return true if all of the subgroups contain required options or
						// subgroups
						boolean result = true;
						for (final ArgGroupSpec subsubgroup : subgroup.subgroups()) {
							result &= !isGroupEffectivelyOptional(subsubgroup);
						}
						return result && !requiredOptionsExistAndAllHaveDefaultValues(subgroup);
					} else {
						return !isGroupEffectivelyOptional(subgroup);
					}
				}
				return false;
			}

			private GroupMatchContainer createGroupMatchContainer(ArgGroupSpec group, GroupMatchContainer parent,
					CommandLine commandLine) {
				final GroupMatchContainer result = new GroupMatchContainer(group, commandLine);
				result.parentContainer = parent;
				parent.lastMatch().matchedSubgroups.put(group, result);
				return result;
			}

			private void failGroupMultiplicityExceeded(List<ParseResult.GroupMatch> groupMatches,
					CommandLine commandLine) {
				final Map<ArgGroupSpec, List<List<ParseResult.GroupMatch>>> matchesPerGroup = new LinkedHashMap<ArgGroupSpec, List<List<GroupMatch>>>();
				String msg = "";
				for (final ParseResult.GroupMatch match : groupMatches) {
					if (msg.length() > 0) {
						msg += " and ";
					}
					msg += match;
					final Map<ArgGroupSpec, GroupMatchContainer> subgroups = match.matchedSubgroups();
					for (final ArgGroupSpec group : subgroups.keySet()) {
						if (group.validate()) { // don't raise errors for non-validating groups:
												// https://github.com/remkop/picocli/issues/810
							addValueToListInMap(matchesPerGroup, group, subgroups.get(group).matches());
						}
					}
				}
				if (!matchesPerGroup.isEmpty()) {
					if (!simplifyErrorMessageForSingleGroup(matchesPerGroup, commandLine)) {
						commandLine.interpreter.maybeThrow(new MaxValuesExceededException(commandLine,
								"Error: expected only one match but got " + msg));
					}
				}
			}

			GroupMatchContainer findLastMatchContainer(ArgGroupSpec group) {
				final List<GroupMatchContainer> all = findMatchContainers(group, new ArrayList<GroupMatchContainer>());
				return all.isEmpty() ? null : all.get(all.size() - 1);
			}

			List<GroupMatchContainer> findMatchContainers(ArgGroupSpec group, List<GroupMatchContainer> result) {
				if (this.group == group) {
					result.add(this);
					return result;
				}
				for (final GroupMatch multiple : matches()) {
					for (final GroupMatchContainer mg : multiple.matchedSubgroups.values()) {
						mg.findMatchContainers(group, result);
					}
				}
				return result;
			}

			private GroupMatchContainer findOrCreateMatchingGroup(ArgSpec argSpec, CommandLine commandLine) {
				ArgGroupSpec searchGroup = Assert.notNull(argSpec.group(), "group for " + argSpec);
				GroupMatchContainer container = this;
				if (searchGroup == container.group()) {
					return container;
				}
				final List<ArgGroupSpec> keys = new ArrayList<ArgGroupSpec>();
				while (searchGroup != null) {
					keys.add(searchGroup);
					searchGroup = searchGroup.parentGroup();
				}
				Collections.reverse(keys);
				for (final ArgGroupSpec key : keys) {
					GroupMatchContainer sub = container.lastMatch().matchedSubgroups().get(key);
					if (sub == null) {
						sub = createGroupMatchContainer(key, container, commandLine);
					}
					container = sub;
				}
				return container;
			}

			/**
			 * Returns the {@code ArgGroupSpec} whose matches are captured in this
			 * {@code GroupMatchContainer}.
			 */
			public ArgGroupSpec group() {
				return group;
			}

			/**
			 * A group is optional if
			 * <ul>
			 * <li>its minimum <code>multiplicity</code> is zero</li>
			 * <li>otherwise, if the group has at least one required option and all required
			 * options have default values</li>
			 * </ul>
			 *
			 * Conversely, a group is required if
			 * <ul>
			 * <li>if its minimum <code>multiplicity</code> is <code>1</code> or more
			 * <em>AND</em></li>
			 * <li>
			 * <ul>
			 * <li>either the group has no required options</li>
			 * <li>or the group has at least one required option without a default
			 * value</li>
			 * </ul>
			 * </li>
			 * </ul>
			 * 
			 * @see <a href=
			 *      "https://github.com/remkop/picocli/issues/1848">https://github.com/remkop/picocli/issues/1848</a>
			 *      and <a href=
			 *      "https://github.com/remkop/picocli/issues/2059">https://github.com/remkop/picocli/issues/2059</a>
			 */
			private boolean isGroupEffectivelyOptional(ArgGroupSpec argGroupSpec) {
				if (argGroupSpec.multiplicity().min == 0) {
					return true;
				}
				return requiredOptionsExistAndAllHaveDefaultValues(argGroupSpec)
						&& !containsRequiredSubgroups(argGroupSpec);
			}

			/**
			 * Returns {@code true} if no more {@code MatchedGroupMultiples} can be added to
			 * this {@code GroupMatchContainer}. Each multiple may be a complete or an
			 * incomplete match.
			 */
			boolean isMaxMultiplicityReached() {
				return matches.size() >= group.multiplicity.max;
			}

			/**
			 * Returns {@code true} if this {@code GroupMatchContainer} has at least the
			 * minimum number of {@code MatchedGroupMultiples}. Each multiple may be a
			 * complete or an incomplete match.
			 */
			boolean isMinMultiplicityReached() {
				return matches.size() >= group.multiplicity.min;
			}

			/** Returns the "active" multiple of this GroupMatchContainer. */
			GroupMatch lastMatch() {
				return matches.get(matches.size() - 1);
			}

			private boolean matchedFully(boolean allRequired) {
				for (final GroupMatch multiple : matches) {
					final boolean actuallyAllRequired = allRequired && multiple == lastMatch();
					if (!multiple.matchedFully(actuallyAllRequired)) {
						return false;
					}
				}
				return allRequired ? isMaxMultiplicityReached() : isMinMultiplicityReached();
			}

			/**
			 * Returns {@code true} if the maximum number of multiples has been matched for
			 * the multiplicity of this group, and the last multiple has
			 * {@linkplain GroupMatch#matchedMaxElements() matched the maximum number of
			 * elements}, while all other multiples have matched at least the
			 * {@linkplain GroupMatch#matchedMinElements() minimum number of elements}.
			 */
			boolean matchedMaxElements() {
				return matchedFully(true);
			}

			/**
			 * Returns {@code true} if the minimum number of multiples has been matched for
			 * the multiplicity of this group, and each multiple has matched at least the
			 * {@linkplain GroupMatch#matchedMinElements() minimum number of elements}.
			 */
			boolean matchedMinElements() {
				return matchedFully(false);
			}

			// /** Returns the {@code GroupMatchContainer} of the parent {@code
			// ArgGroupSpec}, or {@code null} if this group has no parent. */
//            public GroupMatchContainer parentContainer() { return parentContainer; }
			/**
			 * Returns the list of {@code GroupMatch} instances: {@code ArgGroupSpec}s with
			 * a multiplicity greater than one may be matched multiple times.
			 */
			public List<GroupMatch> matches() {
				return Collections.unmodifiableList(matches);
			}

			private boolean requiredOptionsExistAndAllHaveDefaultValues(ArgGroupSpec argGroupSpec) {
				if (argGroupSpec.requiredArgs().size() == 0) {
					return false;
				}
				for (final OptionSpec option : argGroupSpec.options()) {
					if (option.required() && !option.valueIsDefaultValue) {
						return false;
					}
				}
				for (final PositionalParamSpec param : argGroupSpec.positionalParameters()) {
					if (param.required() && !param.valueIsDefaultValue) {
						return false;
					}
				}
				return true;
			}

			private boolean simplifyErrorMessageForSingleGroup(
					Map<ArgGroupSpec, List<List<ParseResult.GroupMatch>>> matchesPerGroup, CommandLine commandLine) {
				for (final ArgGroupSpec group : matchesPerGroup.keySet()) {
					final List<ParseResult.GroupMatch> flat = flatList(matchesPerGroup.get(group));
					final Set<ArgSpec> matchedArgs = new LinkedHashSet<ArgSpec>();
					for (final ParseResult.GroupMatch match : flat) {
						if (!match.matchedSubgroups().isEmpty()) {
							return false;
						}
						matchedArgs.addAll(match.matchedValues.keySet());
					}
					final ParseResult.GroupValidationResult validationResult = group.validateArgs(commandLine,
							matchedArgs);
					if (validationResult.exception != null) {
						commandLine.interpreter.maybeThrow(validationResult.exception); // there may be multiple
																						// failures, just throw on the
																						// first one for now
						return true;
					}
				}
				return false;
			}

			@Override
			public String toString() {
				return toString(new StringBuilder()).toString();
			}

			private StringBuilder toString(StringBuilder result) {
				final String prefix = result.length() == 0 ? "={" : "";
				final String suffix = result.length() == 0 ? "}" : "";
				if (group != null && result.length() == 0) {
					result.append(group.synopsis());
				}
				result.append(prefix);
				String infix = "";
				for (final GroupMatch occurrence : matches) {
					result.append(infix);
					occurrence.toString(result);
					infix = " ";
				}
				return result.append(suffix);

			}

			GroupMatchContainer trim() {
				for (final Iterator<GroupMatch> iter = matches.iterator(); iter.hasNext();) {
					final GroupMatch multiple = iter.next();
					if (multiple.isEmpty()) {
						iter.remove();
					}
					for (final GroupMatchContainer sub : multiple.matchedSubgroups.values()) {
						sub.trim();
					}
				}
				return this;
			}

			void updateUnmatchedGroups(final ArgGroupSpec group) {
				Assert.assertTrue(Assert.equals(group(), group.parentGroup()), new IHelpSectionRenderer() {
					@Override
					public String render(Help h) {
						return "Internal error: expected " + group.parentGroup() + " (the parent of " + group
								+ "), but was " + group();
					}
				});

				final List<GroupMatchContainer> groupMatchContainers = findMatchContainers(group,
						new ArrayList<GroupMatchContainer>());
				if (groupMatchContainers.isEmpty()) {
					this.unmatchedSubgroups.add(group);
				}
				for (final GroupMatchContainer groupMatchContainer : groupMatchContainers) {
					for (final ArgGroupSpec subGroup : group.subgroups()) {
						groupMatchContainer.updateUnmatchedGroups(subGroup);
					}
				}
			}

			void validate(CommandLine commandLine) {
				// first, validate the top-level GroupMatchContainer:
				// Even if cmd has more than one group that each have matches,
				// we should have a *single* top-level GroupMatch, with a subgroup for each
				// GroupMatchContainer.
				// If we have more than one top-level GroupMatch, it means that the parser
				// was forced to "spill over" matches into additional GroupMatches because max
				// multiplicity was exceeded.
				if (group() == null && matches.size() > 1) {
					failGroupMultiplicityExceeded(matches, commandLine);
				}

				validationResult = matches.isEmpty() ? GroupValidationResult.SUCCESS_ABSENT
						: GroupValidationResult.SUCCESS_PRESENT;
				for (final ArgGroupSpec missing : unmatchedSubgroups) {
					// error is 1+ occurrence is required, unless all required options have a
					// default
					if (missing.validate() && !isGroupEffectivelyOptional(missing)) {
						final int presentCount = 0;
						final boolean haveMissing = true;
						final boolean someButNotAllSpecified = false;
						final String exclusiveElements = missing.synopsisUnit();
						final String missingElements = missing.synopsisUnit(); // ArgSpec.describe(missing.requiredArgs());
						validationResult = missing.validate(commandLine, presentCount, haveMissing,
								someButNotAllSpecified, exclusiveElements, missingElements, missingElements);
					}
				}
				validateGroupMultiplicity(commandLine);
				if (validationResult.blockingFailure()) {
					commandLine.interpreter.maybeThrow(validationResult.exception); // composite parent validations
																					// cannot succeed anyway
				}
				for (final GroupMatch match : matches()) {
					match.validate(commandLine);
					if (match.validationResult.blockingFailure()) {
						validationResult = match.validationResult; // potentially overwrites existing blocking failure
																	// with subgroup's!
						break;
					}
				}
				if (validationResult.blockingFailure()) {
					commandLine.interpreter.maybeThrow(validationResult.exception); // composite parent validations
																					// cannot succeed anyway
				}
				if (group() == null) {
					if (!validationResult.success()) {
						commandLine.interpreter.maybeThrow(validationResult.exception);
					}
				}
			}

			private void validateGroupMultiplicity(CommandLine commandLine) {
				if (group == null || !group.validate()) {
					return;
				}
				final int matchCount = matches().size();
				// note: matchCount == 0 if only subgroup(s) are matched for a group without
				// args (subgroups-only) // TODO really?
				final boolean checkMinimum = matchCount > 0 || !group.args().isEmpty();
				if (checkMinimum && matchCount < group.multiplicity().min) {
					if (validationResult.success()) {
						validationResult = new ParseResult.GroupValidationResult(
								matchCount == 0 ? ParseResult.GroupValidationResult.Type.FAILURE_ABSENT
										: ParseResult.GroupValidationResult.Type.FAILURE_PARTIAL,
								new MissingParameterException(commandLine, group.args(),
										"Error: Group: " + group.synopsisUnit() + " must be specified "
												+ group.multiplicity().min + " times but was matched " + matchCount
												+ " times"));
					}
				} else if (matchCount > group.multiplicity().max) {
					if (!validationResult.blockingFailure()) {
						validationResult = new ParseResult.GroupValidationResult(
								ParseResult.GroupValidationResult.Type.FAILURE_PRESENT,
								new MaxValuesExceededException(commandLine,
										"Error: Group: " + group.synopsisUnit() + " can only be specified "
												+ group.multiplicity().max + " times but was matched " + matchCount
												+ " times."));
					}
				}
			}
		}

		static class GroupValidationResult {
			enum Type {
				SUCCESS_PRESENT, SUCCESS_ABSENT, FAILURE_PRESENT, FAILURE_ABSENT, FAILURE_PARTIAL
			}

			static final GroupValidationResult SUCCESS_PRESENT = new GroupValidationResult(Type.SUCCESS_PRESENT);
			static final GroupValidationResult SUCCESS_ABSENT = new GroupValidationResult(Type.SUCCESS_ABSENT);

			static GroupValidationResult extractBlockingFailure(List<GroupValidationResult> set) {
				for (final GroupValidationResult element : set) {
					if (element.blockingFailure()) {
						return element;
					}
				}
				return null;
			}

			Type type;

			ParameterException exception;

			GroupValidationResult(Type type) {
				this.type = type;
			}

			GroupValidationResult(Type type, ParameterException exception) {
				this.type = type;
				this.exception = exception;
			}

			/** FAILURE_PRESENT or FAILURE_PARTIAL */
			boolean blockingFailure() {
				return type == Type.FAILURE_PRESENT || type == Type.FAILURE_PARTIAL;
			}

			boolean present() {
				return type == Type.SUCCESS_PRESENT /* || this == Type.FAILURE_PRESENT */;
			}

			boolean success() {
				return type == Type.SUCCESS_ABSENT || type == Type.SUCCESS_PRESENT;
			}

			@Override
			public String toString() {
				return type + (exception == null ? "" : ": " + exception.getMessage());
			}
		}

		/**
		 * Creates and returns a new {@code ParseResult.Builder} for the specified
		 * command spec.
		 */
		public static Builder builder(CommandSpec commandSpec) {
			return new Builder(commandSpec);
		}

		private final CommandSpec commandSpec;
		private final Set<OptionSpec> matchedUniqueOptions;
		private final Set<PositionalParamSpec> matchedUniquePositionals;
		private final List<ArgSpec> matchedArgs;
		private final List<OptionSpec> matchedOptions;
		private final List<PositionalParamSpec> matchedPositionals;
		private final List<String> originalArgs;
		private final List<String> expandedArgs;
		private final List<String> unmatched;

		private final List<List<PositionalParamSpec>> matchedPositionalParams;
		private final List<Exception> errors;

		private final GroupMatchContainer groupMatchContainer;
		private final List<ParseResult> subcommands;

		final List<Object> tentativeMatch;

		private final boolean usageHelpRequested;
		private final boolean versionHelpRequested;

		private ParseResult(ParseResult.Builder builder) {
			commandSpec = builder.commandSpec;
			subcommands = builder.subcommands;
			matchedOptions = new ArrayList<OptionSpec>(builder.matchedOptionsList);
			matchedUniqueOptions = new LinkedHashSet<OptionSpec>(builder.options);
			unmatched = new ArrayList<String>(builder.unmatched);
			originalArgs = new ArrayList<String>(builder.originalArgList);
			expandedArgs = new ArrayList<String>(builder.expandedArgList);
			matchedArgs = new ArrayList<ArgSpec>(builder.matchedArgsList);
			matchedUniquePositionals = new LinkedHashSet<PositionalParamSpec>(builder.positionals);
			matchedPositionals = new ArrayList<PositionalParamSpec>(builder.matchedPositionalsList);
			matchedPositionalParams = new ArrayList<List<PositionalParamSpec>>(builder.positionalParams);
			errors = new ArrayList<Exception>(builder.errors);
			usageHelpRequested = builder.usageHelpRequested;
			versionHelpRequested = builder.versionHelpRequested;
			tentativeMatch = builder.nowProcessing;
			groupMatchContainer = builder.groupMatchContainer.trim();
		}

		/**
		 * Returns this {@code ParseResult} as a list of {@code CommandLine} objects,
		 * one for each matched command/subcommand. Note that for repeatable
		 * subcommands, there may be multiple commands at each level of the hierarchy in
		 * the returned list.
		 * 
		 * @since 3.0
		 */
		public List<CommandLine> asCommandLineList() {
			return recursivelyAddCommandLineTo(new ArrayList<CommandLine>());
		}

		/** Returns the {@code CommandSpec} for the matched command. */
		public CommandSpec commandSpec() {
			return commandSpec;
		}

		/**
		 * If {@link ParserSpec#collectErrors} is {@code true}, returns the list of
		 * exceptions that were encountered during parsing, otherwise, returns an empty
		 * list.
		 * 
		 * @since 3.2
		 */
		public List<Exception> errors() {
			return Collections.unmodifiableList(errors);
		}

		/**
		 * Returns the command line arguments after
		 * {@linkplain CommandLine#isExpandAtFiles() @-files were expanded}; these are
		 * the arguments that were actually parsed.
		 * 
		 * @see #originalArgs()
		 * @since 4.4
		 */
		public List<String> expandedArgs() {
			return Collections.unmodifiableList(expandedArgs);
		}

		/**
		 * Returns the matches for the specified argument group.
		 * 
		 * @since 4.0
		 */
		public List<GroupMatchContainer> findMatches(ArgGroupSpec group) {
			return groupMatchContainer.findMatchContainers(group, new ArrayList<GroupMatchContainer>());
		}

		/**
		 * Returns the top-level container for the {@link ArgGroupSpec ArgGroupSpec}
		 * match or matches found.
		 * <p>
		 * If the user input was a valid combination of group arguments, the returned
		 * list should contain a single {@linkplain GroupMatch match}. Details of the
		 * {@linkplain GroupMatchContainer matched groups} encountered on the command
		 * line can be obtained via its {@link GroupMatch#matchedSubgroups()
		 * matchedSubgroups()} method. The top-level match returned by this method
		 * contains no {@linkplain GroupMatch#matchedValues(CommandLine.Model.ArgSpec)
		 * matched arguments}.
		 * </p>
		 * <p>
		 * If the returned list contains more than one {@linkplain GroupMatch match},
		 * the user input was invalid: the maximum {@linkplain ArgGroup#multiplicity()
		 * multiplicity} of a group was exceeded, and the parser created an extra
		 * {@code match} to capture the values. Usually this results in a
		 * {@link ParameterException ParameterException} being thrown by the
		 * {@code parse} method, unless the parser is configured to
		 * {@linkplain ParserSpec#collectErrors() collect errors}.
		 * </p>
		 * 
		 * @since 4.0
		 */
		public List<GroupMatch> getGroupMatches() {
			return groupMatchContainer.matches();
		}

		/**
		 * Returns whether an option whose aliases include the specified short name was
		 * matched on the command line.
		 * 
		 * @param shortName used to search the matched options. May be an alias of the
		 *                  option name that was actually specified on the command line.
		 */
		public boolean hasMatchedOption(char shortName) {
			return matchedOption(shortName) != null;
		}

		/** Returns whether the specified option was matched on the command line. */
		public boolean hasMatchedOption(OptionSpec option) {
			return matchedOptions.contains(option);
		}

		/**
		 * Returns whether an option whose aliases include the specified name was
		 * matched on the command line.
		 * 
		 * @param name used to search the matched options. May be an alias of the option
		 *             name that was actually specified on the command line. The
		 *             specified name may include option name prefix characters or not.
		 */
		public boolean hasMatchedOption(String name) {
			return matchedOption(name) != null;
		}

		/**
		 * Returns whether a positional parameter was matched at the specified position.
		 */
		public boolean hasMatchedPositional(int position) {
			return matchedPositional(position) != null;
		}

		/**
		 * Returns whether the specified positional parameter was matched on the command
		 * line.
		 */
		public boolean hasMatchedPositional(PositionalParamSpec positional) {
			return matchedUniquePositionals.contains(positional);
		}

		/**
		 * Returns {@code true} if a subcommand was matched on the command line,
		 * {@code false} otherwise.
		 */
		public boolean hasSubcommand() {
			return !subcommands.isEmpty();
		}

		/**
		 * Returns {@code true} if one of the options that was matched on the command
		 * line is a {@link OptionSpec#usageHelp() usageHelp} option.
		 */
		public boolean isUsageHelpRequested() {
			return usageHelpRequested;
		}

		/**
		 * Returns {@code true} if one of the options that was matched on the command
		 * line is a {@link OptionSpec#versionHelp() versionHelp} option.
		 */
		public boolean isVersionHelpRequested() {
			return versionHelpRequested;
		}

		/**
		 * Returns a list of matched options and positional parameters, in order they
		 * were matched on the command line. The returned list may contain an
		 * {@code OptionSpec} or {@code PositionalParamSpec} multiple times, if the
		 * option or parameter was matched multiple times on the command line.
		 * 
		 * @since 4.0
		 */
		public List<ArgSpec> matchedArgs() {
			return Collections.unmodifiableList(matchedArgs);
		}

		/**
		 * Returns the option with the specified short name, or {@code null} if no
		 * option with that name was matched on the command line.
		 * <p>
		 * Use {@link OptionSpec#getValue() getValue} on the returned {@code OptionSpec}
		 * to get the matched value (or values), converted to the type of the option.
		 * Alternatively, use {@link OptionSpec#stringValues() stringValues} to get the
		 * matched String values after they were {@linkplain OptionSpec#splitRegex()
		 * split} into parts, or {@link OptionSpec#originalStringValues()
		 * originalStringValues} to get the original String values that were matched on
		 * the command line, before any processing.
		 * </p>
		 * <p>
		 * To get the {@linkplain OptionSpec#defaultValue() default value} of an option
		 * that was {@linkplain #hasMatchedOption(char) <em>not</em> matched} on the
		 * command line, use
		 * {@code parseResult.commandSpec().findOption(shortName).getValue()}.
		 * </p>
		 * 
		 * @see CommandSpec#findOption(char)
		 */
		public OptionSpec matchedOption(char shortName) {
			return CommandSpec.findOption(shortName, matchedOptions);
		}

		/**
		 * Returns the option with the specified name, or {@code null} if no option with
		 * that name was matched on the command line.
		 * <p>
		 * Use {@link OptionSpec#getValue() getValue} on the returned {@code OptionSpec}
		 * to get the matched value (or values), converted to the type of the option.
		 * Alternatively, use {@link OptionSpec#stringValues() stringValues} to get the
		 * matched String values after they were {@linkplain OptionSpec#splitRegex()
		 * split} into parts, or {@link OptionSpec#originalStringValues()
		 * originalStringValues} to get the original String values that were matched on
		 * the command line, before any processing.
		 * </p>
		 * <p>
		 * To get the {@linkplain OptionSpec#defaultValue() default value} of an option
		 * that was {@linkplain #hasMatchedOption(String) <em>not</em> matched} on the
		 * command line, use
		 * {@code parseResult.commandSpec().findOption(String).getValue()}.
		 * </p>
		 * 
		 * @see CommandSpec#findOption(String)
		 * @param name used to search the matched options. May be an alias of the option
		 *             name that was actually specified on the command line. The
		 *             specified name may include option name prefix characters or not.
		 */
		public OptionSpec matchedOption(String name) {
			return CommandSpec.findOption(name, matchedOptions);
		}

		/**
		 * Returns a list of matched options, in order they were matched on the command
		 * line. The returned list may contain the same {@code OptionSpec} multiple
		 * times, if the option was matched multiple times on the command line.
		 */
		public List<OptionSpec> matchedOptions() {
			return Collections.unmodifiableList(matchedOptions);
		}

		/**
		 * Returns a set of matched options.
		 * 
		 * @since 4.0
		 */
		public Set<OptionSpec> matchedOptionsSet() {
			return Collections.unmodifiableSet(matchedUniqueOptions);
		}

		/**
		 * Returns the command line argument value of the option with the specified
		 * name, converted to the {@linkplain OptionSpec#type() type} of the option, or
		 * the specified default value if no option with the specified name was matched.
		 */
		public <T> T matchedOptionValue(char shortName, T defaultValue) {
			return matchedOptionValue(matchedOption(shortName), defaultValue);
		}

		/**
		 * Returns the command line argument value of the specified option, converted to
		 * the {@linkplain OptionSpec#type() type} of the option, or the specified
		 * default value if the specified option is {@code null}.
		 */
		@SuppressWarnings("unchecked")
		private <T> T matchedOptionValue(OptionSpec option, T defaultValue) {
			return option == null ? defaultValue : (T) option.getValue();
		}

		/**
		 * Returns the command line argument value of the option with the specified
		 * name, converted to the {@linkplain OptionSpec#type() type} of the option, or
		 * the specified default value if no option with the specified name was matched.
		 */
		public <T> T matchedOptionValue(String name, T defaultValue) {
			return matchedOptionValue(matchedOption(name), defaultValue);
		}

		/**
		 * Returns the first {@code PositionalParamSpec} that matched an argument at the
		 * specified position, or {@code null} if no positional parameters were matched
		 * at that position.
		 */
		public PositionalParamSpec matchedPositional(int position) {
			if (matchedPositionalParams.size() <= position || matchedPositionalParams.get(position).isEmpty()) {
				return null;
			}
			return matchedPositionalParams.get(position).get(0);
		}

		/**
		 * Returns a list of matched positional parameters, in order they were matched
		 * on the command line. The returned list may contain the same
		 * {@code PositionalParamSpec} multiple times, if the parameter was matched
		 * multiple times on the command line.
		 */
		public List<PositionalParamSpec> matchedPositionals() {
			return Collections.unmodifiableList(matchedPositionals);
		}

		/**
		 * Returns all {@code PositionalParamSpec} objects that matched an argument at
		 * the specified position, or an empty list if no positional parameters were
		 * matched at that position.
		 */
		public List<PositionalParamSpec> matchedPositionals(int position) {
			if (matchedPositionalParams.size() <= position) {
				return Collections.emptyList();
			}
			return matchedPositionalParams.get(position) == null ? Collections.<PositionalParamSpec>emptyList()
					: matchedPositionalParams.get(position);
		}

		/**
		 * Returns a set of matched positional parameters.
		 * 
		 * @since 4.0
		 */
		public Set<PositionalParamSpec> matchedPositionalsSet() {
			return Collections.unmodifiableSet(matchedUniquePositionals);
		}

		/**
		 * Returns the command line argument value of the positional parameter at the
		 * specified position, converted to the {@linkplain PositionalParamSpec#type()
		 * type} of the positional parameter, or the specified default value if no
		 * positional parameter was matched at that position.
		 */
		public <T> T matchedPositionalValue(int position, T defaultValue) {
			return matchedPositionalValue(matchedPositional(position), defaultValue);
		}

		/**
		 * Returns the command line argument value of the specified positional
		 * parameter, converted to the {@linkplain PositionalParamSpec#type() type} of
		 * the positional parameter, or the specified default value if the specified
		 * positional parameter is {@code null}.
		 */
		@SuppressWarnings("unchecked")
		private <T> T matchedPositionalValue(PositionalParamSpec positional, T defaultValue) {
			return positional == null ? defaultValue : (T) positional.getValue();
		}

		/**
		 * Returns the original command line arguments that were passed to the
		 * {@link CommandLine#parseArgs(String...)} method, before
		 * {@linkplain CommandLine#isExpandAtFiles() @-file expansion}.
		 * 
		 * @see #expandedArgs()
		 */
		public List<String> originalArgs() {
			return Collections.unmodifiableList(originalArgs);
		}

		private List<CommandLine> recursivelyAddCommandLineTo(List<CommandLine> result) {
			result.add(this.commandSpec().commandLine());
			for (final ParseResult sub : subcommands()) {
				sub.recursivelyAddCommandLineTo(result);
			}
			return result;
		}

		/**
		 * Returns the {@code ParseResult} for the last subcommand of this command that
		 * was matched on the command line, or {@code null} if no subcommand was
		 * matched.
		 */
		public ParseResult subcommand() {
			return !hasSubcommand() ? null : subcommands.get(subcommands.size() - 1);
		}

		/**
		 * Returns a list with the {@code ParseResult} objects for each subcommand of
		 * this command that was matched on the command line or an empty list if no
		 * subcommands were matched. The returned list can only contain multiple values
		 * if this command's {@link CommandSpec#subcommandsRepeatable()
		 * subcommandsRepeatable} attribute is {@code true}.
		 * 
		 * @since 4.2
		 */
		public List<ParseResult> subcommands() {
			return Collections.unmodifiableList(subcommands);
		}

		/**
		 * Returns a list of command line arguments that did not match any options or
		 * positional parameters.
		 */
		public List<String> unmatched() {
			return Collections.unmodifiableList(unmatched);
		}

		void validateGroups() {
			for (final ArgGroupSpec group : commandSpec.argGroups()) {
				groupMatchContainer.updateUnmatchedGroups(group);
			}
			groupMatchContainer.validate(commandSpec.commandLine());
		}
	}

	/**
	 * Base class of all exceptions thrown by {@code picocli.CommandLine}.
	 * <h2>Class Diagram of the Picocli Exceptions</h2>
	 * <p>
	 * <img src="doc-files/class-diagram-exceptions.png" alt="Class Diagram of the
	 * Picocli Exceptions">
	 * </p>
	 * 
	 * @since 2.0
	 */
	public static class PicocliException extends RuntimeException {
		private static final long serialVersionUID = -2574128880125050818L;

		public PicocliException(String msg) {
			super(msg);
		}

		public PicocliException(String msg, Throwable t) {
			super(msg, t);
		}
	}

	static class PositionalParametersSorter implements Comparator<ArgSpec> {
		private static final Range OPTION_INDEX = new Range(0, 0, false, true, "0");

		@Override
		public int compare(ArgSpec p1, ArgSpec p2) {
			final int result = index(p1).compareTo(index(p2));
			return (result == 0) ? p1.arity().compareTo(p2.arity()) : result;
		}

		private Range index(ArgSpec arg) {
			return arg.isOption() ? OPTION_INDEX : ((PositionalParamSpec) arg).index();
		}
	}

	/**
	 * {@link IDefaultValueProvider IDefaultValueProvider} implementation that loads
	 * default values for command line options and positional parameters from a
	 * properties file or {@code Properties} object.
	 * <h2>Location</h2> By default, this implementation tries to find a properties
	 * file named {@code ".<YOURCOMMAND>.properties"} in the user home directory,
	 * where {@code "<YOURCOMMAND>"} is the {@linkplain CommandLine.Command#name()
	 * name} of the command. If a command has
	 * {@linkplain CommandLine.Command#aliases() aliases} in addition to its
	 * {@linkplain CommandLine.Command#name() name}, these aliases are also used to
	 * try to find the properties file. For example:
	 * 
	 * <pre>
	 * &#64;Command(name = "git", defaultValueProvider = PropertiesDefaultProvider.class)
	 * class Git {
	 * }
	 * </pre>
	 * <p>
	 * The above will try to load default values from
	 * {@code new File(System.getProperty("user.home"), ".git.properties")}.
	 * </p>
	 * <p>
	 * The location of the properties file can also be controlled with system
	 * property {@code "picocli.defaults.<YOURCOMMAND>.path"}, in which case the
	 * value of the property must be the path to the file containing the default
	 * values.
	 * </p>
	 * <p>
	 * The location of the properties file may also be specified programmatically.
	 * For example:
	 * </p>
	 * 
	 * <pre>
	 * CommandLine cmd = new CommandLine(new MyCommand());
	 * File defaultsFile = new File("path/to/config/mycommand.properties");
	 * cmd.setDefaultValueProvider(new PropertiesDefaultProvider(defaultsFile));
	 * cmd.execute(args);
	 * </pre>
	 * 
	 * <h2>Format</h2>
	 * <p>
	 * For options, the key is either the
	 * {@linkplain CommandLine.Option#descriptionKey() descriptionKey}, or the
	 * option's {@linkplain OptionSpec#longestName() longest name}.
	 * </p>
	 * <p>
	 * For positional parameters, the key is either the
	 * {@linkplain CommandLine.Parameters#descriptionKey() descriptionKey}, or the
	 * positional parameter's {@linkplain PositionalParamSpec#paramLabel() param
	 * label}.
	 * </p>
	 * <p>
	 * End users may not know what the {@code descriptionKey} of your options and
	 * positional parameters are, so be sure to document that with your application.
	 * </p>
	 * <h2>Subcommands</h2>
	 * <p>
	 * The default values for options and positional parameters of subcommands can
	 * be included in the properties file for the top-level command, so that end
	 * users need to maintain only a single file. This can be achieved by prefixing
	 * the key with the command's qualified name. For example, to give the
	 * {@code `git commit`} command's {@code --cleanup} option a default value of
	 * {@code strip}, define a key of {@code git.commit.cleanup} and assign it a
	 * default value.
	 * </p>
	 * 
	 * <pre>
	 * # /home/remko/.git.properties
	 * git.commit.cleanup = strip
	 * </pre>
	 * 
	 * @since 4.1
	 */
	public static class PropertiesDefaultProvider implements IDefaultValueProvider {

		private static Properties createProperties(File file, CommandSpec commandSpec) {
			if (file == null) {
				throw new NullPointerException("file is null");
			}
			if (file.exists() && file.canRead()) {
				try {
					return createProperties(new FileInputStream(file), file.getAbsolutePath(), commandSpec);
				} catch (final Exception ex) {
					tracer().warn("PropertiesDefaultProvider could not read defaults from %s: %s",
							file.getAbsolutePath(), ex);
				}
			} else {
				tracer().warn(
						"PropertiesDefaultProvider: defaults configuration file %s does not exist or is not readable",
						file.getAbsolutePath());
			}
			return new Properties();
		}

		private static Properties createProperties(InputStream in, String locationDescription,
				CommandSpec commandSpec) {
			if (in == null) {
				throw new NullPointerException("InputStream is null");
			}
			final Properties result = new Properties();
			try {
				final String command = commandSpec == null ? "unknown command" : commandSpec.qualifiedName();
				tracer().debug("PropertiesDefaultProvider reading defaults from %s for %s", locationDescription,
						command);
				result.load(in);
				result.put("__picocli_internal_location", locationDescription);
			} catch (final IOException ioe) {
				tracer().warn("PropertiesDefaultProvider could not read defaults from %s: %s", locationDescription,
						ioe);
			} finally {
				close(in);
			}
			return result;
		}

		private static Properties loadProperties(CommandSpec commandSpec) {
			if (commandSpec == null) {
				return null;
			}
			final Properties p = System.getProperties();
			for (final String name : commandSpec.names()) {
				final String propertiesFileName = "." + name + ".properties";
				final String path = p.getProperty("picocli.defaults." + name + ".path");
				final File defaultPath = new File(p.getProperty("user.home"), propertiesFileName);
				final File file = path == null ? defaultPath : new File(path);
				if (path != null) {
					tracer().debug("PropertiesDefaultProvider using path from system property %s: %s",
							"picocli.defaults." + name + ".path", path);
				}
				if (file.canRead()) {
					return createProperties(file, commandSpec);
				} else {
					Object userObject = commandSpec.userObject();
					if (userObject == null) {
						userObject = commandSpec.commandLine;
					}
					final URL resource = userObject.getClass().getClassLoader().getResource(propertiesFileName);
					if (resource != null) {
						try {
							return createProperties(resource.openStream(), resource.toString(), commandSpec);
						} catch (final Exception ex) {
							tracer().warn("PropertiesDefaultProvider could not read defaults from %s: %s", resource,
									ex);
						}
					} else {
						tracer().debug(
								"PropertiesDefaultProvider defaults configuration file %s does not exist on classpath or user home or specified location",
								propertiesFileName);
					}
				}
			}
			return loadProperties(commandSpec.parent());
		}

		private static String stripPrefix(String prefixed) {
			for (int i = 0; i < prefixed.length(); i++) {
				if (Character.isJavaIdentifierPart(prefixed.charAt(i))) {
					return prefixed.substring(i);
				}
			}
			return prefixed;
		}

		private Properties properties;

		private String location;

		/**
		 * Default constructor, used when this default value provider is specified in
		 * the annotations:
		 * 
		 * <pre>
		 * &#64;Command(name = "mycmd",
		 *     defaultValueProvider = PropertiesDefaultProvider.class)
		 * class MyCommand // ...
		 * </pre>
		 * <p>
		 * This loads default values from a properties file named
		 * {@code ".mycmd.properties"} in the user home directory.
		 * </p>
		 * <p>
		 * The location of the properties file can also be controlled with system
		 * property {@code "picocli.defaults.<YOURCOMMAND>.path"}, in which case the
		 * value of the property must be the path to the file containing the default
		 * values.
		 * </p>
		 * 
		 * @see PropertiesDefaultProvider the PropertiesDefaultProvider class
		 *      description
		 */
		public PropertiesDefaultProvider() {
		}

		/**
		 * This constructor loads default values from the specified properties file.
		 * This may be used programmatically. For example:
		 * 
		 * <pre>
		 * CommandLine cmd = new CommandLine(new MyCommand());
		 * File defaultsFile = new File("path/to/config/file.properties");
		 * cmd.setDefaultValueProvider(new PropertiesDefaultProvider(defaultsFile));
		 * cmd.execute(args);
		 * </pre>
		 * 
		 * @param file the file to load default values from. Must be non-{@code null}
		 *             and must contain default values in the standard java
		 *             {@link Properties} format.
		 * @see PropertiesDefaultProvider the PropertiesDefaultProvider class
		 *      description
		 */
		public PropertiesDefaultProvider(File file) {
			this(createProperties(file, null));
			properties.remove("__picocli_internal_location");
			location = file.getAbsolutePath();
		}

		/**
		 * This constructor loads default values from the specified properties object.
		 * This may be used programmatically. For example:
		 * 
		 * <pre>
		 * CommandLine cmd = new CommandLine(new MyCommand());
		 * Properties defaults = getProperties();
		 * cmd.setDefaultValueProvider(new PropertiesDefaultProvider(defaults));
		 * cmd.execute(args);
		 * </pre>
		 * 
		 * @param properties the properties containing the default values
		 * @see PropertiesDefaultProvider the PropertiesDefaultProvider class
		 *      description
		 */
		public PropertiesDefaultProvider(Properties properties) {
			this.properties = properties;
		}

		@Override
		public String defaultValue(ArgSpec argSpec) throws Exception {
			if (properties == null) {
				properties = loadProperties(argSpec.command());
				location = properties == null ? null : (String) properties.remove("__picocli_internal_location");
			}
			if (properties == null || properties.isEmpty()) {
				return null;
			}
			return argSpec.isOption() ? optionDefaultValue((OptionSpec) argSpec)
					: positionalDefaultValue((PositionalParamSpec) argSpec);
		}

		private String getValue(String key, CommandSpec spec) {
			String result = null;
			if (spec != null) {
				final String cmd = spec.qualifiedName(".");
				result = properties.getProperty(cmd + "." + key);
			}
			if (result != null) {
				return result;
			}
			return key == null ? null : properties.getProperty(key);
		}

		private String optionDefaultValue(OptionSpec option) {
			String result = getValue(option.descriptionKey(), option.command());
			result = result != null ? result : getValue(stripPrefix(option.longestName()), option.command());
			return result;
		}

		private String positionalDefaultValue(PositionalParamSpec positional) {
			String result = getValue(positional.descriptionKey(), positional.command());
			result = result != null ? result : getValue(positional.paramLabel(), positional.command());
			return result;
		}

		@Override
		public String toString() {
			return getClass().getSimpleName() + "[" + location + "]";
		}
	}

	/**
	 * Describes the number of parameters required and accepted by an option or a
	 * positional parameter.
	 * 
	 * @since 0.9.7
	 */
	public static class Range implements Comparable<Range> {
		static Range adjustForType(Range result, IAnnotatedElement member) {
			return result.isUnspecified ? defaultArity(member) : result;
		}

		/**
		 * Returns the default arity {@code Range} for {@link Option options}: booleans
		 * have arity 0, other types have arity 1.
		 * 
		 * @param type the type whose default arity to return
		 * @return a new {@code Range} indicating the default arity of the specified
		 *         type
		 * @deprecated use {@link #defaultArity(Field)} instead
		 */
		@Deprecated
		public static Range defaultArity(Class<?> type) {
			return isBoolean(type) ? Range.valueOf("0").unspecified(true) : Range.valueOf("1").unspecified(true);
		}

		/**
		 * Returns the default arity {@code Range}: for interactive options/positional
		 * parameters, this is 0; for {@link Option options} this is effectively "0..1"
		 * for booleans and 1 for other types, for {@link Parameters parameters}
		 * booleans have arity 1, arrays or Collections have arity "0..*", and other
		 * types have arity 1.
		 * <p>
		 * <b><em>Implementation Notes</em></b>
		 * </p>
		 * <p>
		 * The returned {@code Range} for boolean options has an <em>effective</em>
		 * arity of "0..1". This is implemented by returning a {@code Range} with arity
		 * "0", and its {@code unspecified} property set to {@code true}. This
		 * implementation may change in the future.
		 * </p>
		 * 
		 * @param field the field whose default arity to return
		 * @return a new {@code Range} indicating the default arity of the specified
		 *         field
		 * @since 2.0
		 */
		public static Range defaultArity(Field field) {
			return defaultArity(new TypedMember(field));
		}

		private static Range defaultArity(IAnnotatedElement member) {
			if (member.isInteractive()) {
				return Range.valueOf("0").unspecified(true);
			}
			final ITypeInfo info = member.getTypeInfo();
			if (member.isAnnotationPresent(Option.class)) {
				final boolean zeroArgs = info.isBoolean()
						|| (info.isMultiValue() && info.getAuxiliaryTypeInfos().get(0).isBoolean());
				return zeroArgs ? Range.valueOf("0").unspecified(true) : Range.valueOf("1").unspecified(true);
			}
			if (info.isMultiValue()) {
				return Range.valueOf("0..1").unspecified(true);
			}
			return Range.valueOf("1").unspecified(true);// for single-valued fields (incl. boolean positional
														// parameters)
		}

		private static Range defaultParameterIndex(ITypeInfo typeInfo) {
			return Range.valueOf(typeInfo.isMultiValue() ? "*" : "0+"); // the default
		}

		/**
		 * Returns a new {@code Range} based on the {@link Option#arity()} annotation on
		 * the specified field, or the field type's default arity if no arity was
		 * specified.
		 * 
		 * @param field the field whose Option annotation to inspect
		 * @return a new {@code Range} based on the Option arity annotation on the
		 *         specified field
		 */
		public static Range optionArity(Field field) {
			return optionArity(new TypedMember(field));
		}

		private static Range optionArity(IAnnotatedElement member) {
			return member.isAnnotationPresent(Option.class)
					? adjustForType(Range.valueOf(member.getAnnotation(Option.class).arity()), member)
					: new Range(0, 0, false, true, "0");
		}

		/**
		 * Returns a new {@code Range} based on the {@link Parameters#arity()}
		 * annotation on the specified field, or the field type's default arity if no
		 * arity was specified.
		 * 
		 * @param field the field whose Parameters annotation to inspect
		 * @return a new {@code Range} based on the Parameters arity annotation on the
		 *         specified field
		 */
		public static Range parameterArity(Field field) {
			return parameterArity(new TypedMember(field));
		}

		private static Range parameterArity(IAnnotatedElement member) {
			if (member.isAnnotationPresent(Parameters.class)) {
				return adjustForType(Range.valueOf(member.getAnnotation(Parameters.class).arity()), member);
			} else {
				return member.isMethodParameter() ? adjustForType(Range.valueOf(""), member)
						: new Range(0, 0, false, true, "0");
			}
		}

		static Range parameterCapacity(IAnnotatedElement member) {
			final Range arity = parameterArity(member);
			if (!member.isMultiValue()) {
				return arity;
			}
			final Range index = parameterIndex(member);
			return parameterCapacity(arity, index);
		}

		private static Range parameterCapacity(Range arity, Range index) {
			if (arity.max == 0) {
				return arity;
			}
			if (index.size() == 1) {
				return arity;
			}
			if (index.isVariable) {
				return Range.valueOf(arity.min + "..*");
			}
			if (arity.size() == 1) {
				return Range.valueOf(arity.min * index.size() + "");
			}
			if (arity.isVariable) {
				return Range.valueOf(arity.min * index.size() + "..*");
			}
			return Range.valueOf(arity.min * index.size() + ".." + arity.max * index.size());
		}

		/**
		 * Returns a new {@code Range} based on the {@link Parameters#index()}
		 * annotation on the specified field.
		 * 
		 * @param field the field whose Parameters annotation to inspect
		 * @return a new {@code Range} based on the Parameters index annotation on the
		 *         specified field
		 */
		public static Range parameterIndex(Field field) {
			return parameterIndex(new TypedMember(field));
		}

		private static Range parameterIndex(IAnnotatedElement member) {
			if (member.isAnnotationPresent(Parameters.class)) {
				final Range result = Range.valueOf(member.getAnnotation(Parameters.class).index());
				if (!result.isUnspecified) {
					return result;
				}
			}
			if (member.isMethodParameter()) {
				final int min = member.getMethodParamPosition();
				final int max = member.isMultiValue() ? Integer.MAX_VALUE : min;
				return new Range(min, max, member.isMultiValue(), false, null);
			}
			return defaultParameterIndex(member.getTypeInfo());
		}

		private static int parseInt(String str, int defaultValue) {
			try {
				final int pos = str.indexOf('+');
				return Integer.parseInt(pos < 0 ? str : str.substring(0, str.indexOf('+')));
			} catch (final Exception ex) {
				return defaultValue;
			}
		}

		/**
		 * Leniently parses the specified String as a {@code Range} value and return the
		 * result. A range string can be a fixed integer value or a range of the form
		 * {@code MIN_VALUE + ".." + MAX_VALUE}. If the {@code MIN_VALUE} string is not
		 * numeric, the minimum is zero. If the {@code MAX_VALUE} is not numeric, the
		 * range is taken to be variable and the maximum is {@code Integer.MAX_VALUE}.
		 * 
		 * @param range the value range string to parse
		 * @return a new {@code Range} value
		 */
		public static Range valueOf(String range) {
			if (range.contains("${")) {
				return new Range(0, 0, false, false, range); // unresolved
			}
			range = range.trim();
			final boolean unspecified = range.length() == 0 || range.startsWith(".."); // || range.endsWith("..");
			int min, max;
			boolean variable;
			int dots;
			if ((dots = range.indexOf("..")) >= 0) {
				min = parseInt(range.substring(0, dots), 0);
				max = parseInt(range.substring(dots + 2), Integer.MAX_VALUE);
				variable = max == Integer.MAX_VALUE;
			} else {
				max = parseInt(range, Integer.MAX_VALUE);
				variable = !range.contains("+") && max == Integer.MAX_VALUE;
				min = variable ? 0 : max;
			}
			return new Range(min, max, variable, unspecified, unspecified ? null : range);
		}

		/** @deprecated use {@link #min()} instead */
		@Deprecated
		public final int min;
		/** @deprecated use {@link #max()} instead */
		@Deprecated
		public final int max;
		/** @deprecated use {@link #isVariable()} instead */
		@Deprecated
		public final boolean isVariable;
		private final boolean isUnspecified;
		private final String originalValue;
		private final boolean relative;
		private final int anchor;

		/**
		 * Constructs a new Range object with the specified parameters.
		 * 
		 * @param min           minimum number of required parameters
		 * @param max           maximum number of allowed parameters (or
		 *                      Integer.MAX_VALUE if variable)
		 * @param variable      {@code true} if any number or parameters is allowed,
		 *                      {@code false} otherwise
		 * @param unspecified   {@code true} if no arity was specified on the
		 *                      option/parameter (value is based on type)
		 * @param originalValue the original value that was specified on the option or
		 *                      parameter
		 */
		public Range(int min, int max, boolean variable, boolean unspecified, String originalValue) {
			if (min < 0 || max < 0) {
				throw new InitializationException("Invalid negative range (min=" + min + ", max=" + max + ")");
			}
			if (min > max) {
				throw new InitializationException("Invalid range (min=" + min + ", max=" + max + ")");
			}
			this.min = min;
			this.max = max;
			this.isVariable = variable;
			this.isUnspecified = unspecified;
			this.originalValue = originalValue;
			this.relative = originalValue != null && originalValue.contains("+");
			// relative indices have an anchorPoint that is used for sorting
			if (relative) {
				anchor = "+".equals(originalValue) ? Integer.MAX_VALUE : parseInt(originalValue, Integer.MAX_VALUE);
			} else {
				anchor = min;
			}
		}

		/**
		 * Returns the anchor position that this Range is {@linkplain #isRelative()
		 * relative} to, or {@link #min()} if this Range is absolute.
		 * 
		 * @return {@code 1} for a relative index like {@code "1+"}, or
		 *         {@code Integer.MAX_VALUE} for a relative index without an anchor,
		 *         like {@code "+"}
		 * @since 4.3
		 */
		int anchor() {
			return anchor;
		} // not public (yet): TBD if we need a minAnchor and maxAnchor in the future

		@Override
		public int compareTo(Range other) {
			if (originalValue != null && other.originalValue != null && originalValue.equals(other.originalValue)) {
				return 0; // try to keep stable sort for relative indexes
			}
			int result = (anchor() < other.anchor()) ? -1 : ((anchor() == other.anchor()) ? 0 : 1); // don't subtract;
																									// prevent overflow
			if (result == 0) {
				result = (max < other.max) ? -1 : ((max == other.max) ? 0 : 1);
			}
			if (result == 0) {
				if (this.isRelative() != other.isRelative()) {
					result = isRelative() ? 1 : -1; // relative follows absolute
				}
			}
			return result;
		}

		/**
		 * Returns {@code true} if this Range includes the specified value,
		 * {@code false} otherwise.
		 * 
		 * @param value the value to check
		 * @return {@code true} if the specified value is not less than the minimum and
		 *         not greater than the maximum of this Range
		 */
		public boolean contains(int value) {
			return min <= value && max >= value;
		}

		@Override
		public boolean equals(Object object) {
			if (!(object instanceof Range)) {
				return false;
			}
			final Range other = (Range) object;
			return other.max == this.max && other.min == this.min && other.isVariable == this.isVariable;
		}

		@Override
		public int hashCode() {
			return ((17 * 37 + max) * 37 + min) * 37 + (isVariable ? 1 : 0);
		}

		/**
		 * Returns equivalent of {@code format("%s (%s)", originalValue, toString())}.
		 */
		String internalToString() {
			if (isUnresolved()) {
				return originalValue;
			}
			return isRelative() ? originalValue + " (" + toString() + ")" : toString();
		}

		/**
		 * Returns {@code true} if this Range contains a relative index like
		 * {@code "1+"}, or {@code false} if this Range does not contain any relative
		 * indices.
		 * 
		 * @since 4.3
		 */
		public boolean isRelative() {
			return relative;
		}

		boolean isRelativeToAnchor() {
			return anchor != Integer.MAX_VALUE && isRelative();
		}

		/**
		 * Returns {@code true} if this range contains a relative index like
		 * {@code "1+"}, or variables that have not been expanded yet, {@code false} if
		 * this Range does not contain any variables or relative indices.
		 * 
		 * @since 4.0
		 */
		public boolean isUnresolved() {
			return originalValue != null && originalValue.contains("${");
		}

		/**
		 * Returns {@code true} if this Range is a default value, {@code false} if the
		 * user specified this value.
		 * 
		 * @since 4.0
		 */
		public boolean isUnspecified() {
			return isUnspecified;
		}

		/** Returns true for these ranges: 0 and 0..1. */
		boolean isValidForInteractiveArgs() {
			return (min == 0 && (max == 0 || max == 1));
		}

		/**
		 * Returns {@code true} if this range has no fixed upper bound.
		 * 
		 * @since 4.0
		 */
		public boolean isVariable() {
			return isVariable;
		}

		/**
		 * Returns the upper bound of this range (inclusive), or
		 * {@code Integer.MAX_VALUE} if this range has {@linkplain #isVariable() no
		 * upper bound}.
		 * 
		 * @since 4.0
		 */
		public int max() {
			return max;
		}

		/**
		 * Returns a new Range object with the {@code max} value replaced by the
		 * specified value. The {@code min} of the returned Range is guaranteed not to
		 * be greater than the new {@code max} value.
		 * 
		 * @param newMax the {@code max} value of the returned Range object
		 * @return a new Range object with the specified {@code max} value
		 */
		public Range max(int newMax) {
			return new Range(Math.min(min, newMax), newMax, isVariable, isUnspecified, originalValue);
		}

		/**
		 * Returns the lower bound of this range (inclusive).
		 * 
		 * @since 4.0
		 */
		public int min() {
			return min;
		}

		/**
		 * Returns a new Range object with the {@code min} value replaced by the
		 * specified value. The {@code max} of the returned Range is guaranteed not to
		 * be less than the new {@code min} value.
		 * 
		 * @param newMin the {@code min} value of the returned Range object
		 * @return a new Range object with the specified {@code min} value
		 */
		public Range min(int newMin) {
			return new Range(newMin, Math.max(newMin, max), isVariable, isUnspecified, originalValue);
		}

		/**
		 * Returns the original String value that this range was constructed with.
		 * 
		 * @since 4.3
		 */
		public String originalValue() {
			return originalValue;
		}

		boolean overlaps(Range index) {
			return contains(index.min) || contains(index.max) || index.contains(min) || index.contains(max);
		}

		private int size() {
			return isRelative() ? 1 : 1 + max - min;
		}

		@Override
		public String toString() {
			if (isUnresolved()) {
				return originalValue;
			}
			if (min == max) {
				return (relative && min == Integer.MAX_VALUE) ? "+" : String.valueOf(min);
			} else {
				return min + ".." + (isVariable ? "*" : max);
			}
		}

		/**
		 * Returns a new Range object with the {@code isUnspecified} value replaced by
		 * the specified value.
		 * 
		 * @param unspecified the {@code unspecified} value of the returned Range object
		 * @return a new Range object with the specified {@code unspecified} value
		 */
		public Range unspecified(boolean unspecified) {
			return new Range(min, max, isVariable, unspecified, originalValue);
		}
	}

	/**
	 * A regular expression-based option name transformation for
	 * {@linkplain Option#negatable() negatable} options.
	 * <p>
	 * A common way to negate GNU *nix long options is to prefix them with
	 * {@code "no-"}, so for a {@code --force} option the negative version would be
	 * {@code --no-force}. Java has the {@code -XX:[+|-]} JVM options, where
	 * "Boolean options are turned on with {@code -XX:+<option>} and turned off with
	 * {@code -XX:-<option>}". These are the negative forms
	 * {@linkplain #createDefault() supported by default} by this class.
	 * </p>
	 * <p>
	 * See the {@link org.rossonet.ext.picocli.CommandLine.RegexTransformer.Builder}
	 * for an example of customizing this to create negative forms for short
	 * options.
	 * </p>
	 * 
	 * @since 4.0
	 */
	public static class RegexTransformer implements INegatableOptionTransformer {
		/**
		 * Builder for creating {@code RegexTransformer} objects.
		 * 
		 * @since 4.0
		 */
		public static class Builder {
			Map<Pattern, String> replacements = new LinkedHashMap<Pattern, String>();
			Map<Pattern, String> synopsis = new LinkedHashMap<Pattern, String>();

			/** Constructs an empty builder. */
			public Builder() {
			}

			/**
			 * Constructs a builder populated with the values from the specified
			 * RegexTransformer.
			 */
			public Builder(RegexTransformer old) {
				replacements.putAll(old.replacements);
				synopsis.putAll(old.synopsis);
			}

			/**
			 * Adds the specified negative replacement and synopsis replacement for the
			 * specified regular expression. For example, to add negative forms for short
			 * options:
			 * <table border="1">
			 * <caption>Regular expressions for adding negative forms for short
			 * options</caption>
			 * <tr>
			 * <th>Regex</th>
			 * <th>Negative Replacement</th>
			 * <th>Synopsis Replacement</th>
			 * <th>Comment</th>
			 * </tr>
			 * <tr>
			 * <td>^-(\w)$</td>
			 * <td>+$1</td>
			 * <td>(+|-)$1</td>
			 * <td>Converts <code>-v</code> to <code>+v</code></td>
			 * </tr>
			 * <tr>
			 * <td>^\+(\w)$</td>
			 * <td>-$1</td>
			 * <td>(+|-)$1</td>
			 * <td>Converts <code>-v</code> to <code>+v</code></td>
			 * </tr>
			 * </table>
			 *
			 * @param regex               regular expression to match an option name
			 * @param negativeReplacement the replacement to use to generate a
			 *                            {@linkplain RegexTransformer#makeNegative(String, CommandLine.Model.CommandSpec)
			 *                            negative name} when the option name matches
			 * @param synopsisReplacement the replacement to use to generate a
			 *                            {@linkplain RegexTransformer#makeSynopsis(String, CommandLine.Model.CommandSpec)
			 *                            documentation string} when the option name matches
			 * @return this {@code RegexTransformer} for method chaining
			 */
			public RegexTransformer.Builder addPattern(String regex, String negativeReplacement,
					String synopsisReplacement) {
				final Pattern pattern = Pattern.compile(regex);
				replacements.put(pattern, negativeReplacement);
				synopsis.put(pattern, synopsisReplacement);
				return this;
			}

			public RegexTransformer build() {
				return new RegexTransformer(this);
			}

			/**
			 * Removes the negative replacement and synopsis replacement for the specified
			 * regular expression.
			 * 
			 * @param regex regular expression to remove
			 * @return this {@code RegexTransformer} for method chaining
			 */
			public RegexTransformer.Builder removePattern(String regex) {
				for (final Iterator<Pattern> iter = replacements.keySet().iterator(); iter.hasNext();) {
					final Pattern pattern = iter.next();
					if (pattern.toString().equals(regex)) {
						iter.remove();
						synopsis.remove(pattern);
					}
				}
				return this;
			}
		}

		/**
		 * Returns the {@code RegexTransformer} for case-insensitive negatable options.
		 * <table border="1">
		 * <caption>The regular expressions for case-insensitive negatable
		 * options</caption>
		 * <tr>
		 * <th>Regex</th>
		 * <th>Negative Replacement</th>
		 * <th>Synopsis Replacement</th>
		 * <th>Comment</th>
		 * </tr>
		 * <tr>
		 * <td>^--((?i)no)-(\w(-|\w)*)$</td>
		 * <td>--$2</td>
		 * <td>--[$1-]$2</td>
		 * <td>Converts <code>--no-force</code> to <code>--force</code></td>
		 * <td>and <code>--NO-force</code> to <code>--force</code></td>
		 * </tr>
		 * <tr>
		 * <td>^--(\w(-|\w)*)$</td>
		 * <td>--no-$1</td>
		 * <td>--[no-]$1</td>
		 * <td>Converts <code>--force</code> to <code>--no-force</code></td>
		 * </tr>
		 * <tr>
		 * <td>^(-|--)(\w*:)\+(\w(-|\w)*)$</td>
		 * <td>$1$2-$3</td>
		 * <td>$1$2(+|-)$3</td>
		 * <td>Converts <code>-XX:+Inline</code> to <code>-XX:-Inline</code></td>
		 * </tr>
		 * <tr>
		 * <td>^(-|--)(\w*:)\-(\w(-|\w)*)$</td>
		 * <td>$1$2+$3</td>
		 * <td>$1$2(+|-)$3</td>
		 * <td>Converts <code>-XX:-Inline</code> to <code>-XX:+Inline</code></td>
		 * </tr>
		 * </table>
		 */
		public static RegexTransformer createCaseInsensitive() {
			final CommandLine.RegexTransformer transformer = new CommandLine.RegexTransformer.Builder()
					.addPattern("^--((?i)no)-(\\w(-|\\w)*)$", "--$2", "--[$1-]$2")
					.addPattern("^--(\\w(-|\\w)*)$", "--no-$1", "--[no-]$1")
					.addPattern("^(-|--)(\\w*:)\\+(\\w(-|\\w)*)$", "$1$2-$3", "$1$2(+|-)$3")
					.addPattern("^(-|--)(\\w*:)\\-(\\w(-|\\w)*)$", "$1$2+$3", "$1$2(+|-)$3").build();
			return transformer;
		}

		/**
		 * Returns the {@code RegexTransformer} used by default for negatable options.
		 * <table border="1">
		 * <caption>The regular expressions used by default for negatable
		 * options</caption>
		 * <tr>
		 * <th>Regex</th>
		 * <th>Negative Replacement</th>
		 * <th>Synopsis Replacement</th>
		 * <th>Comment</th>
		 * </tr>
		 * <tr>
		 * <td>^--no-(\w(-|\w)*)$</td>
		 * <td>--$1</td>
		 * <td>--[no-]$1</td>
		 * <td>Converts <code>--no-force</code> to <code>--force</code></td>
		 * </tr>
		 * <tr>
		 * <td>^--(\w(-|\w)*)$</td>
		 * <td>--no-$1</td>
		 * <td>--[no-]$1</td>
		 * <td>Converts <code>--force</code> to <code>--no-force</code></td>
		 * </tr>
		 * <tr>
		 * <td>^(-|--)(\w*:)\+(\w(-|\w)*)$</td>
		 * <td>$1$2-$3</td>
		 * <td>$1$2(+|-)$3</td>
		 * <td>Converts <code>-XX:+Inline</code> to <code>-XX:-Inline</code></td>
		 * </tr>
		 * <tr>
		 * <td>^(-|--)(\w*:)\-(\w(-|\w)*)$</td>
		 * <td>$1$2+$3</td>
		 * <td>$1$2(+|-)$3</td>
		 * <td>Converts <code>-XX:-Inline</code> to <code>-XX:+Inline</code></td>
		 * </tr>
		 * </table>
		 */
		public static RegexTransformer createDefault() {
			final CommandLine.RegexTransformer transformer = new CommandLine.RegexTransformer.Builder()
					.addPattern("^--no-(\\w(-|\\w)*)$", "--$1", "--[no-]$1")
					.addPattern("^--(\\w(-|\\w)*)$", "--no-$1", "--[no-]$1")
					.addPattern("^(-|--)(\\w*:)\\+(\\w(-|\\w)*)$", "$1$2-$3", "$1$2(+|-)$3")
					.addPattern("^(-|--)(\\w*:)\\-(\\w(-|\\w)*)$", "$1$2+$3", "$1$2(+|-)$3").build();
			return transformer;
		}

		final Map<Pattern, String> replacements;

		final Map<Pattern, String> synopsis;

		RegexTransformer(RegexTransformer.Builder builder) {
			replacements = Collections.unmodifiableMap(new LinkedHashMap<Pattern, String>(builder.replacements));
			synopsis = Collections.unmodifiableMap(new LinkedHashMap<Pattern, String>(builder.synopsis));
		}

		/** {@inheritDoc} */
		@Override
		public String makeNegative(String optionName, CommandSpec cmd) {
			for (final Map.Entry<Pattern, String> entry : replacements.entrySet()) {
				final Matcher matcher = entry.getKey().matcher(optionName);
				if (matcher.find()) {
					return matcher.replaceAll(entry.getValue());
				}
			}
			return optionName;
		}

		/** {@inheritDoc} */
		@Override
		public String makeSynopsis(String optionName, CommandSpec cmd) {
			for (final Map.Entry<Pattern, String> entry : synopsis.entrySet()) {
				final Matcher matcher = entry.getKey().matcher(optionName);
				if (matcher.find()) {
					return matcher.replaceAll(entry.getValue());
				}
			}
			return optionName;
		}

		@Override
		public String toString() {
			return getClass().getName() + "[replacements=" + replacements + ", synopsis=" + synopsis + "]@"
					+ System.identityHashCode(this);
		}
	}

	/**
	 * Command line {@linkplain IExecutionStrategy execution strategy} that prints
	 * help if requested, and otherwise executes the top-level command and all
	 * subcommands as {@code Runnable}, {@code Callable} or {@code Method}. For use
	 * by the {@link #execute(String...) execute} method.
	 * 
	 * @since 2.0
	 */
	public static class RunAll extends AbstractParseResultHandler<List<Object>> implements IParseResultHandler {
		/** {@inheritDoc} */
		@Override
		public int execute(ParseResult parseResult) throws ExecutionException {
			return super.execute(parseResult);
		}

		@Override
		protected List<IExitCodeGenerator> extractExitCodeGenerators(ParseResult parseResult) {
			return recursivelyExtractExitCodeGenerators(parseResult, new ArrayList<IExitCodeGenerator>());
		}

		/**
		 * Executes the top-level command and all subcommands as {@code Runnable} or
		 * {@code Callable}. If any of the {@code CommandLine} commands does not
		 * implement either {@code Runnable} or {@code Callable} and is not a
		 * {@code Method}, an {@code ExecutionException} is thrown detailing the problem
		 * and capturing the offending {@code CommandLine} object.
		 *
		 * @param parseResult the {@code ParseResult} that resulted from successfully
		 *                    parsing the command line arguments
		 * @return an empty list if help was requested, or a list containing the result
		 *         of executing all commands: the return values from calling the
		 *         {@code Callable} commands, {@code null} elements for commands that
		 *         implement {@code Runnable}
		 * @throws ExecutionException if a problem occurred while processing the parse
		 *                            results; use
		 *                            {@link ExecutionException#getCommandLine()} to get
		 *                            the command or subcommand where processing failed
		 * @since 3.0
		 */
		@Override
		protected List<Object> handle(ParseResult parseResult) throws ExecutionException {
			final Tracer t = CommandLine.tracer();
			return returnResultOrExit(recursivelyExecuteUserObject(parseResult, new ArrayList<Object>(), t));
		}

		/**
		 * Prints help if requested, and otherwise executes the top-level command and
		 * all subcommands as {@code Runnable}, {@code Callable} or {@code Method}.
		 * Finally, either a list of result objects is returned, or the JVM is
		 * terminated if an exit code {@linkplain #andExit(int) was set}. If any of the
		 * {@code CommandLine} commands does not implement either {@code Runnable} or
		 * {@code Callable}, an {@code ExecutionException} is thrown detailing the
		 * problem and capturing the offending {@code CommandLine} object.
		 *
		 * @param parsedCommands the {@code CommandLine} objects that resulted from
		 *                       successfully parsing the command line arguments
		 * @param out            the {@code PrintStream} to print help to if requested
		 * @param ansi           for printing help messages using ANSI styles and colors
		 * @return an empty list if help was requested, or a list containing the result
		 *         of executing all commands: the return values from calling the
		 *         {@code Callable} commands, {@code null} elements for commands that
		 *         implement {@code Runnable}
		 * @throws ParameterException if the {@link HelpCommand HelpCommand} was invoked
		 *                            for an unknown subcommand. Any
		 *                            {@code ParameterExceptions} thrown from this
		 *                            method are treated as if this exception was thrown
		 *                            during parsing and passed to the
		 *                            {@link IExceptionHandler}
		 * @throws ExecutionException if a problem occurred while processing the parse
		 *                            results; use
		 *                            {@link ExecutionException#getCommandLine()} to get
		 *                            the command or subcommand where processing failed
		 */
		@Override
		public List<Object> handleParseResult(List<CommandLine> parsedCommands, PrintStream out, Help.Ansi ansi) {
			if (printHelpIfRequested(parsedCommands, out, err(), ansi)) {
				return returnResultOrExit(Collections.emptyList());
			}
			final List<Object> result = new ArrayList<Object>();
			for (final CommandLine parsed : parsedCommands) {
				executeUserObject(parsed, result);
			}
			return returnResultOrExit(result);
		}

		private List<Object> recursivelyExecuteUserObject(ParseResult parseResult, List<Object> result, Tracer t)
				throws ExecutionException {
			t.debug("%s: executing user object for '%s'...", getClass().getSimpleName(),
					parseResult.commandSpec.qualifiedName());
			executeUserObject(parseResult.commandSpec().commandLine(), result);
			for (final ParseResult pr : parseResult.subcommands()) {
				recursivelyExecuteUserObject(pr, result, t);
			}
			return result;
		}

		private List<IExitCodeGenerator> recursivelyExtractExitCodeGenerators(ParseResult parseResult,
				List<IExitCodeGenerator> result) throws ExecutionException {
			if (parseResult.commandSpec().userObject() instanceof IExitCodeGenerator) {
				result.add((IExitCodeGenerator) parseResult.commandSpec().userObject());
			}
			for (final ParseResult pr : parseResult.subcommands()) {
				recursivelyExtractExitCodeGenerators(pr, result);
			}
			return result;
		}

		@Override
		protected RunAll self() {
			return this;
		}
	}

	/**
	 * Command line {@linkplain IExecutionStrategy execution strategy} that prints
	 * help if requested, and otherwise executes the top-level {@code Runnable} or
	 * {@code Callable} command. For use by the {@link #execute(String...) execute}
	 * method.
	 * 
	 * @since 2.0
	 */
	public static class RunFirst extends AbstractParseResultHandler<List<Object>> implements IParseResultHandler {
		/** {@inheritDoc} */
		@Override
		public int execute(ParseResult parseResult) throws ExecutionException {
			return super.execute(parseResult);
		}

		@Override
		protected List<IExitCodeGenerator> extractExitCodeGenerators(ParseResult parseResult) {
			if (parseResult.commandSpec().userObject() instanceof IExitCodeGenerator) {
				return Collections.singletonList((IExitCodeGenerator) parseResult.commandSpec().userObject());
			}
			return Collections.emptyList();
		}

		/**
		 * Executes the top-level {@code Runnable} or {@code Callable} subcommand. If
		 * the top-level command does not implement either {@code Runnable} or
		 * {@code Callable} and is not a {@code Method}, an {@code ExecutionException}
		 * is thrown detailing the problem and capturing the offending
		 * {@code CommandLine} object.
		 *
		 * @param parseResult the {@code ParseResult} that resulted from successfully
		 *                    parsing the command line arguments
		 * @return an empty list if help was requested, or a list containing a single
		 *         element: the result of calling the {@code Callable}, or a
		 *         {@code null} element if the last (sub)command was a {@code Runnable}
		 * @throws ExecutionException if a problem occurred while processing the parse
		 *                            results; use
		 *                            {@link ExecutionException#getCommandLine()} to get
		 *                            the command or subcommand where processing failed
		 * @since 3.0
		 */
		@Override
		protected List<Object> handle(ParseResult parseResult) throws ExecutionException {
			final Tracer t = CommandLine.tracer();
			t.debug("RunFirst: executing user object for '%s'...", parseResult.commandSpec().qualifiedName());
			return executeUserObject(parseResult.commandSpec().commandLine(), new ArrayList<Object>()); // first
		}

		/**
		 * Prints help if requested, and otherwise executes the top-level
		 * {@code Runnable} or {@code Callable} command. Finally, either a list of
		 * result objects is returned, or the JVM is terminated if an exit code
		 * {@linkplain #andExit(int) was set}. If the top-level command does not
		 * implement either {@code Runnable} or {@code Callable}, an
		 * {@code ExecutionException} is thrown detailing the problem and capturing the
		 * offending {@code CommandLine} object.
		 *
		 * @param parsedCommands the {@code CommandLine} objects that resulted from
		 *                       successfully parsing the command line arguments
		 * @param out            the {@code PrintStream} to print help to if requested
		 * @param ansi           for printing help messages using ANSI styles and colors
		 * @return an empty list if help was requested, or a list containing a single
		 *         element: the result of calling the {@code Callable}, or a
		 *         {@code null} element if the top-level command was a {@code Runnable}
		 * @throws ParameterException if the {@link HelpCommand HelpCommand} was invoked
		 *                            for an unknown subcommand. Any
		 *                            {@code ParameterExceptions} thrown from this
		 *                            method are treated as if this exception was thrown
		 *                            during parsing and passed to the
		 *                            {@link IExceptionHandler}
		 * @throws ExecutionException if a problem occurred while processing the parse
		 *                            results; use
		 *                            {@link ExecutionException#getCommandLine()} to get
		 *                            the command or subcommand where processing failed
		 */
		@Override
		public List<Object> handleParseResult(List<CommandLine> parsedCommands, PrintStream out, Help.Ansi ansi) {
			if (printHelpIfRequested(parsedCommands, out, err(), ansi)) {
				return returnResultOrExit(Collections.emptyList());
			}
			return returnResultOrExit(executeUserObject(parsedCommands.get(0), new ArrayList<Object>()));
		}

		@Override
		protected RunFirst self() {
			return this;
		}
	}

	/**
	 * Command line {@linkplain IExecutionStrategy execution strategy} that prints
	 * help if requested, and otherwise executes the most specific {@code Runnable}
	 * or {@code Callable} subcommand. For use by the {@link #execute(String...)
	 * execute} method.
	 * <p>
	 * Something like this:
	 * </p>
	 * 
	 * <pre>{@code
	 * // RunLast implementation: print help if requested, otherwise execute the
	 * // most specific subcommand
	 * List<CommandLine> parsedCommands = parseResult.asCommandLineList();
	 * if (CommandLine.printHelpIfRequested(parsedCommands, out(), err(), ansi())) {
	 * 	return emptyList();
	 * }
	 * CommandLine last = parsedCommands.get(parsedCommands.size() - 1);
	 * Object command = last.getCommand();
	 * Object result = null;
	 * if (command instanceof Runnable) {
	 * 	try {
	 * 		((Runnable) command).run();
	 * 	} catch (Exception ex) {
	 * 		throw new ExecutionException(last, "Error in runnable " + command, ex);
	 * 	}
	 * } else if (command instanceof Callable) {
	 * 	try {
	 * 		result = ((Callable) command).call();
	 * 	} catch (Exception ex) {
	 * 		throw new ExecutionException(last, "Error in callable " + command, ex);
	 * 	}
	 * } else {
	 * 	throw new ExecutionException(last, "Parsed command (" + command + ") is not Runnable or Callable");
	 * }
	 * last.setExecutionResult(result);
	 * return Arrays.asList(result);
	 * }</pre>
	 * <p>
	 * From picocli v2.0, {@code RunLast} is used to implement the
	 * {@link #run(Runnable, PrintStream, PrintStream, Help.Ansi, String...) run}
	 * and {@link #call(Callable, PrintStream, PrintStream, Help.Ansi, String...)
	 * call} convenience methods.
	 * </p>
	 * 
	 * @since 2.0
	 */
	public static class RunLast extends AbstractParseResultHandler<List<Object>> implements IParseResultHandler {
		private static List<Object> executeUserObjectOfLastSubcommandWithSameParent(List<CommandLine> parsedCommands) {
			final Tracer t = CommandLine.tracer();
			final int start = indexOfLastSubcommandWithSameParent(parsedCommands);
			final List<Object> result = new ArrayList<Object>();
			for (int i = start; i < parsedCommands.size(); i++) {
				t.debug("RunLast: executing user object for '%s'...",
						parsedCommands.get(i).commandSpec.qualifiedName());
				executeUserObject(parsedCommands.get(i), result);
			}
			return result;
		}

		// find list of most deeply nested sub-(sub*)-commands
		private static int indexOfLastSubcommandWithSameParent(List<CommandLine> parsedCommands) {
			int start = parsedCommands.size() - 1;
			for (int i = parsedCommands.size() - 2; i >= 0; i--) {
				if (parsedCommands.get(i).getParent() != parsedCommands.get(i + 1).getParent()) {
					break;
				}
				start = i;
			}
			return start;
		}

		/** {@inheritDoc} */
		@Override
		public int execute(ParseResult parseResult) throws ExecutionException {
			return super.execute(parseResult);
		}

		@Override
		protected List<IExitCodeGenerator> extractExitCodeGenerators(ParseResult parseResult) {
			final List<CommandLine> parsedCommands = parseResult.asCommandLineList();
			final int start = indexOfLastSubcommandWithSameParent(parsedCommands);
			final List<IExitCodeGenerator> result = new ArrayList<IExitCodeGenerator>();
			for (int i = start; i < parsedCommands.size(); i++) {
				final Object userObject = parsedCommands.get(i).getCommandSpec().userObject();
				if (userObject instanceof IExitCodeGenerator) {
					result.add((IExitCodeGenerator) userObject);
				}
			}
			return result;
		}

		/**
		 * Executes the most specific {@code Runnable} or {@code Callable} subcommand.
		 * <p>
		 * For {@linkplain Command#subcommandsRepeatable() repeatable subcommands}, this
		 * method may execute multiple subcommands: the most deeply nested subcommands
		 * that have the same parent command.
		 * </p>
		 * <p>
		 * If the user object of the executed (sub)command does not implement either
		 * {@code Runnable} or {@code Callable} and is not a {@code Method}, an
		 * {@code ExecutionException} is thrown detailing the problem and capturing the
		 * offending {@code CommandLine} object.
		 * </p>
		 *
		 * @param parseResult the {@code ParseResult} that resulted from successfully
		 *                    parsing the command line arguments
		 * @return an empty list if help was requested, or a list containing a single
		 *         element: the result of calling the {@code Callable}, or a
		 *         {@code null} element if the last (sub)command was a {@code Runnable}
		 * @throws ExecutionException if a problem occurred while processing the parse
		 *                            results; use
		 *                            {@link ExecutionException#getCommandLine()} to get
		 *                            the command or subcommand where processing failed
		 * @since 3.0
		 */
		@Override
		protected List<Object> handle(ParseResult parseResult) throws ExecutionException {
			return executeUserObjectOfLastSubcommandWithSameParent(parseResult.asCommandLineList());
		}

		/**
		 * Prints help if requested, and otherwise executes the most specific
		 * {@code Runnable} or {@code Callable} subcommand.
		 * <p>
		 * For {@linkplain Command#subcommandsRepeatable() repeatable subcommands}, this
		 * method may execute multiple subcommands: the most deeply nested subcommands
		 * that have the same parent command.
		 * </p>
		 * <p>
		 * Finally, either a list of result objects is returned, or the JVM is
		 * terminated if an exit code {@linkplain #andExit(int) was set}.
		 * </p>
		 * <p>
		 * If the last (sub)command does not implement either {@code Runnable} or
		 * {@code Callable}, an {@code ExecutionException} is thrown detailing the
		 * problem and capturing the offending {@code CommandLine} object.
		 * </p>
		 *
		 * @param parsedCommands the {@code CommandLine} objects that resulted from
		 *                       successfully parsing the command line arguments
		 * @param out            the {@code PrintStream} to print help to if requested
		 * @param ansi           for printing help messages using ANSI styles and colors
		 * @return an empty list if help was requested, or a list containing a single
		 *         element: the result of calling the {@code Callable}, or a
		 *         {@code null} element if the last (sub)command was a {@code Runnable}
		 * @throws ParameterException if the {@link HelpCommand HelpCommand} was invoked
		 *                            for an unknown subcommand. Any
		 *                            {@code ParameterExceptions} thrown from this
		 *                            method are treated as if this exception was thrown
		 *                            during parsing and passed to the
		 *                            {@link IExceptionHandler}
		 * @throws ExecutionException if a problem occurred while processing the parse
		 *                            results; use
		 *                            {@link ExecutionException#getCommandLine()} to get
		 *                            the command or subcommand where processing failed
		 */
		@Override
		public List<Object> handleParseResult(List<CommandLine> parsedCommands, PrintStream out, Help.Ansi ansi) {
			if (printHelpIfRequested(parsedCommands, out, err(), ansi)) {
				return returnResultOrExit(Collections.emptyList());
			}
			return returnResultOrExit(executeUserObjectOfLastSubcommandWithSameParent(parsedCommands));
		}

		@Override
		protected RunLast self() {
			return this;
		}
	}

	/**
	 * Specifies the scope of the element.
	 * 
	 * @since 4.3
	 */
	public enum ScopeType {
		/** The element only exists in the current command. */
		LOCAL,
		/**
		 * The element exists in the command where the element is defined and all
		 * descendents (subcommands, sub-subcommands, etc.).
		 */
		INHERIT,
	}

	/**
	 * Fields annotated with {@code @Spec} will be initialized with the
	 * {@code CommandSpec} for the command the field is part of. Example usage:
	 * 
	 * <pre>
	 * class InjectSpecExample implements Runnable {
	 * 	&#064;Spec
	 * 	CommandSpec commandSpec;
	 * 
	 * 	// ...
	 * 	public void run() {
	 * 		// do something with the injected objects
	 * 	}
	 * }
	 * </pre>
	 * 
	 * @since 3.2
	 */
	@Retention(RetentionPolicy.RUNTIME)
	@Target({ ElementType.FIELD, ElementType.METHOD })
	public @interface Spec {
		/**
		 * Identifies what kind of {@code CommandSpec} should be injected.
		 * 
		 * @since 4.3.0
		 */
		enum Target {
			/**
			 * Injects the {@code CommandSpec} of the command where this
			 * {@code @Spec}-annotated program element is declared.
			 */
			SELF,
			/**
			 * Injects the {@code CommandSpec} of the "mixee" command that receives the
			 * options and other command elements defined here, or {@code null} if this
			 * commands is not {@linkplain Mixin mixed into} another command. The "mixee"
			 * command has a {@code @Mixin}-annotated program element with the type of the
			 * class where this {@code @Spec}-annotated program element is declared.
			 */
			MIXEE
		}

		/**
		 * Whether to inject the {@code CommandSpec} of this command (the default) or
		 * the {@code CommandSpec} of the "mixee" command that receives the options and
		 * other command elements defined here.
		 * 
		 * @see Mixin
		 * @since 4.3.0
		 */
		Target value() default Target.SELF;
	}

	/**
	 * Enumerates over the trace level values for filtering which internal debug
	 * statements should be printed.
	 * 
	 * @since 4.7.6-SNAPSHOT
	 */
	public enum TraceLevel {
		OFF, WARN, INFO, DEBUG;

		static TraceLevel lookup(String key) {
			return key == null ? WARN
					: empty(key) || "true".equalsIgnoreCase(key) ? INFO : valueOf(key.toUpperCase(ENGLISH));
		}

		/**
		 * Returns whether messages at the specified {@code other} trace level would be
		 * printed for the current trace level.
		 */
		public boolean isEnabled(TraceLevel other) {
			return ordinal() >= other.ordinal();
		}

		private String prefix(String msg) {
			return "[picocli " + this + "] " + msg;
		}

		private void print(Tracer tracer, String msg, Object... params) {
			if (tracer.level.isEnabled(this)) {
				tracer.stream.printf(prefix(msg) + "%n", params);
			}
		}
	}

	/**
	 * Utility class for printing internal debug statements.
	 * 
	 * @see CommandLine#tracer()
	 * @since 4.7.6-SNAPSHOT
	 */
	public static final class Tracer {
		private PrintStream stream = System.err;
		private TraceLevel level = TraceLevel.lookup(System.getProperty("picocli.trace"));
		boolean modified;

		private Tracer() {
		}

		/**
		 * Prints the specified message if the current trace level is DEBUG or higher.
		 * 
		 * @param msg    the message to print; may use
		 *               {@link String#format(String, Object...)} syntax
		 * @param params Arguments referenced by the format specifiers in the format
		 *               string. If there are more arguments than format specifiers, the
		 *               extra arguments are ignored. The number of arguments is
		 *               variable and may be zero.
		 * @see Formatter
		 */
		public void debug(String msg, Object... params) {
			TraceLevel.DEBUG.print(this, msg, params);
		}

		/**
		 * Returns the trace level that needs to be matched or exceeded for internal
		 * tracing statements to be printed. The initial trace level is WARN, unless
		 * system property {@code "picocli.trace"} is set to another valid
		 * {@code TraceLevel} value.
		 * 
		 * @return the trace level that needs to be matched or exceeded for tracing
		 *         statements to be printed
		 */
		public TraceLevel getLevel() {
			return level;
		}

		/**
		 * Prints the specified message if the current trace level is INFO or higher.
		 * 
		 * @param msg    the message to print; may use
		 *               {@link String#format(String, Object...)} syntax
		 * @param params Arguments referenced by the format specifiers in the format
		 *               string. If there are more arguments than format specifiers, the
		 *               extra arguments are ignored. The number of arguments is
		 *               variable and may be zero.
		 * @see Formatter
		 */
		public void info(String msg, Object... params) {
			TraceLevel.INFO.print(this, msg, params);
		}

		/** Returns whether the current trace level is DEBUG (the highest). */
		public boolean isDebug() {
			return level.isEnabled(TraceLevel.DEBUG);
		}

		/** Returns whether the current trace level is INFO or higher. */
		public boolean isInfo() {
			return level.isEnabled(TraceLevel.INFO);
		}

		/** Returns whether the current trace level is OFF (the lowest). */
		public boolean isOff() {
			return level == TraceLevel.OFF;
		}

		/** Returns whether the current trace level is WARN or higher. */
		public boolean isWarn() {
			return level.isEnabled(TraceLevel.WARN);
		}

		/**
		 * Sets the trace level that needs to be matched or exceeded for internal
		 * tracing statements to be printed. The initial trace level is WARN, unless
		 * system property {@code "picocli.trace"} is set to another valid
		 * {@code TraceLevel} value.
		 * 
		 * @param level the trace level that needs to be matched or exceeded for tracing
		 *              statements to be printed
		 */
		public void setLevel(TraceLevel level) {
			this.level = Assert.notNull(level, "level");
			modified = true;
		}

		@Override
		public String toString() {
			return "Tracer[" + level + "]";
		}

		/**
		 * Prints the specified message if the current trace level is WARN or higher.
		 * 
		 * @param msg    the message to print; may use
		 *               {@link String#format(String, Object...)} syntax
		 * @param params Arguments referenced by the format specifiers in the format
		 *               string. If there are more arguments than format specifiers, the
		 *               extra arguments are ignored. The number of arguments is
		 *               variable and may be zero.
		 * @see Formatter
		 */
		public void warn(String msg, Object... params) {
			TraceLevel.WARN.print(this, msg, params);
		}
	}

	/**
	 * Exception thrown by {@link ITypeConverter} implementations to indicate a
	 * String could not be converted.
	 */
	public static class TypeConversionException extends PicocliException {
		private static final long serialVersionUID = 4251973913816346114L;

		/**
		 * Constructs a TypeConversionException.
		 * 
		 * @see ITypeConverter#convert(String)
		 */
		public TypeConversionException(String msg) {
			super(msg);
		}
	}

	/**
	 * Fields annotated with {@code @Unmatched} will be initialized with the list of
	 * unmatched command line arguments, if any. If this annotation is found,
	 * picocli automatically sets
	 * {@linkplain CommandLine#setUnmatchedArgumentsAllowed(boolean)
	 * unmatchedArgumentsAllowed} to {@code true}.
	 * 
	 * @see CommandLine#isUnmatchedArgumentsAllowed()
	 * @since 3.0
	 */
	@Retention(RetentionPolicy.RUNTIME)
	@Target(ElementType.FIELD)
	public @interface Unmatched {
	}

	/**
	 * Exception indicating that a command line argument could not be mapped to any
	 * of the fields annotated with {@link Option} or {@link Parameters}.
	 */
	public static class UnmatchedArgumentException extends ParameterException {
		private static final long serialVersionUID = -8700426380701452440L;

		private static String describe(List<String> unmatch, CommandLine cmd) {
			final String plural = unmatch.size() == 1 ? "" : "s";
			if (isUnknownOption(unmatch, cmd)) {
				return "Unknown option" + plural;
			}
			final String at = unmatch.size() == 1 ? " at" : " from";
			final int index = cmd.interpreter.parseResultBuilder == null ? 0
					: cmd.interpreter.parseResultBuilder.firstUnmatchedPosition;
			return "Unmatched argument" + plural + at + " index " + index;
		}

		private static boolean isUnknownOption(List<String> unmatch, CommandLine cmd) {
			return unmatch != null && !unmatch.isEmpty() && cmd.getCommandSpec().resemblesOption(unmatch.get(0));
		}

		/**
		 * Returns {@code true} and prints suggested solutions to the specified stream
		 * if such solutions exist, otherwise returns {@code false}.
		 * 
		 * @since 3.3.0
		 */
		public static boolean printSuggestions(ParameterException ex, PrintStream out) {
			return ex instanceof UnmatchedArgumentException && ((UnmatchedArgumentException) ex).printSuggestions(out);
		}

		/**
		 * Returns {@code true} and prints suggested solutions to the specified writer
		 * if such solutions exist, otherwise returns {@code false}.
		 * 
		 * @since 4.0
		 */
		public static boolean printSuggestions(ParameterException ex, PrintWriter writer) {
			return ex instanceof UnmatchedArgumentException
					&& ((UnmatchedArgumentException) ex).printSuggestions(writer);
		}

		static String quoteElements(List<String> list) {
			String result = "", suffix = "";
			int pos;
			for (String element : list) {
				if (result.length() > 0) {
					result += ", ";
				}
				if (element != null && (pos = element.indexOf(" (while processing option:")) >= 0) {
					suffix = element.substring(pos);
					element = element.substring(0, pos);
				}
				result += "'" + element + "'" + suffix;
				suffix = "";
			}
			return result;
		}

		private static String str(List<String> list) {
			final String s = list.toString();
			return s.substring(0, s.length() - 1).substring(1);
		}

		static List<String> stripErrorMessage(List<String> unmatched) {
			final List<String> result = new ArrayList<String>();
			for (final String s : unmatched) {
				if (s == null) {
					result.add(null);
					continue;
				}
				final int pos = s.indexOf(" (while processing option:");
				result.add(pos < 0 ? s : s.substring(0, pos));
			}
			return Collections.unmodifiableList(result);
		}

		private List<String> unmatched = Collections.emptyList();

		public UnmatchedArgumentException(CommandLine commandLine, List<String> args) {
			this(commandLine, args, "");
		}

		public UnmatchedArgumentException(CommandLine commandLine, List<String> args, String extraMsg) {
			this(commandLine, describe(Assert.notNull(args, "unmatched list"), commandLine) + ": " + quoteElements(args)
					+ extraMsg);
			unmatched = new ArrayList<String>(args);
		}

		public UnmatchedArgumentException(CommandLine commandLine, Stack<String> args) {
			this(commandLine, new ArrayList<String>(reverse(args)));
		}

		public UnmatchedArgumentException(CommandLine commandLine, String msg) {
			super(commandLine, msg);
		}

		/**
		 * Returns suggested solutions if such solutions exist, otherwise returns an
		 * empty list.
		 * 
		 * @since 3.3.0
		 */
		public List<String> getSuggestions() {
			if (unmatched.isEmpty()) {
				return Collections.emptyList();
			}
			final String arg = unmatched.get(0);
			final String stripped = CommandSpec.stripPrefix(arg);
			final CommandSpec spec = getCommandLine().getCommandSpec();
			if (spec.resemblesOption(arg)) {
				return spec.findVisibleOptionNamesWithPrefix(stripped.substring(0, Math.min(2, stripped.length())));
			} else if (!spec.subcommands().isEmpty()) {
				final List<String> visibleSubs = new ArrayList<String>();
				for (final Map.Entry<String, CommandLine> entry : spec.subcommands().entrySet()) {
					if (!entry.getValue().getCommandSpec().usageMessage().hidden()) {
						visibleSubs.add(entry.getKey());
					}
				}
				final List<String> mostSimilar = CosineSimilarity.mostSimilar(arg, visibleSubs);
				return mostSimilar.subList(0, Math.min(3, mostSimilar.size()));
			}
			return Collections.emptyList();
		}

		/**
		 * Returns the unmatched command line arguments.
		 * 
		 * @since 3.3.0
		 */
		public List<String> getUnmatched() {
			return stripErrorMessage(unmatched);
		}

		/**
		 * Returns {@code true} if the first unmatched command line arguments resembles
		 * an option, {@code false} otherwise.
		 * 
		 * @since 3.3.0
		 */
		public boolean isUnknownOption() {
			return isUnknownOption(unmatched, getCommandLine());
		}

		private List<String> prefixCommandName(List<String> suggestions) {
			final String commandName = this.commandLine.commandSpec.name;
			if (commandName == null || commandName.trim().length() == 0) {
				return suggestions;
			}
			final List<String> prefixedSuggestions = new ArrayList<String>();
			for (final String s : suggestions) {
				prefixedSuggestions.add(commandName + " " + s);
			}
			return prefixedSuggestions;
		}

		/**
		 * Returns {@code true} and prints suggested solutions to the specified stream
		 * if such solutions exist, otherwise returns {@code false}.
		 * 
		 * @since 3.3.0
		 */
		public boolean printSuggestions(PrintStream out) {
			return printSuggestions(newPrintWriter(out, getStdoutEncoding()));
		}

		/**
		 * Returns {@code true} and prints suggested solutions to the specified stream
		 * if such solutions exist, otherwise returns {@code false}.
		 * 
		 * @since 4.0
		 */
		public boolean printSuggestions(PrintWriter writer) {
			final List<String> suggestions = getSuggestions();
			if (!suggestions.isEmpty()) {
				writer.println(isUnknownOption() ? "Possible solutions: " + str(suggestions)
						: "Did you mean: " + str(prefixCommandName(suggestions)).replace(", ", " or ") + "?");
				writer.flush();
			}
			return !suggestions.isEmpty();
		}
	}

	/**
	 * Converter that can be used to signal to picocli that it should use the
	 * default converter. This can be useful with maps:
	 * 
	 * <pre>
	 * class App {
	 * 	&#064;Option(names = "-D", converter = { UseDefaultConverter.class, GenericValueConverter.class })
	 * 	Map&lt;String, GenericValue&lt;?&gt;&gt; values;
	 * }
	 * </pre>
	 *
	 * The {@link #convert(String)} method of this class always throws an
	 * UnsupportedOperationException.
	 * 
	 * @since 4.7.6-SNAPSHOT
	 */
	public static final class UseDefaultConverter implements ITypeConverter<Object> {
		/**
		 * Always throws UnsupportedOperationException.
		 * 
		 * @throws UnsupportedOperationException always
		 */
		@Override
		public Object convert(String value) throws Exception {
			throw new UnsupportedOperationException("This method should never be called.");
		}
	}

	/** This is picocli version {@value}. */
	public static final String VERSION = "4.7.6-SNAPSHOT";

	private static final Tracer TRACER = new Tracer();

	static <K, T> void addValueToListInMap(Map<K, List<T>> map, K key, T value) {
		List<T> values = map.get(key);
		if (values == null) {
			values = new ArrayList<T>();
			map.put(key, values);
		}
		values.add(value);
	}

	/**
	 * Delegates to
	 * {@link #call(Callable, PrintStream, PrintStream, Help.Ansi, String...)} with
	 * {@code System.err} for diagnostic error messages.
	 * 
	 * @param callable the command to call when {@linkplain #parseArgs(String...)
	 *                 parsing} succeeds.
	 * @param out      the printStream to print the usage help message to when the
	 *                 user requested help
	 * @param ansi     the ANSI style to use
	 * @param args     the command line arguments to parse
	 * @param <C>      the annotated object must implement Callable
	 * @param <T>      the return type of the most specific command (must implement
	 *                 {@code Callable})
	 * @throws InitializationException if the specified command object does not have
	 *                                 a {@link Command}, {@link Option} or
	 *                                 {@link Parameters} annotation
	 * @throws ExecutionException      if the Callable throws an exception
	 * @return {@code null} if an error occurred while parsing the command line
	 *         options, or if help was requested and printed. Otherwise returns the
	 *         result of calling the Callable
	 * @deprecated use {@link #execute(String...)} and {@link #getExecutionResult()}
	 *             instead
	 * @see RunLast
	 */
	@Deprecated
	public static <C extends Callable<T>, T> T call(C callable, PrintStream out, Help.Ansi ansi, String... args) {
		return call(callable, out, System.err, ansi, args);
	}

	/**
	 * Convenience method to allow command line application authors to avoid some
	 * boilerplate code in their application. The annotated object needs to
	 * implement {@link Callable}.
	 * <p>
	 * Consider using the {@link #execute(String...)} method instead:
	 * </p>
	 * 
	 * <pre>{@code
	 * CommandLine cmd = new CommandLine(callable).setOut(myOutWriter()) // System.out by default
	 * 		.setErr(myErrWriter()) // System.err by default
	 * 		.setColorScheme(myColorScheme()); // default color scheme, Ansi.AUTO by default
	 * int exitCode = cmd.execute(args);
	 * //System.exit(exitCode);
	 * }</pre>
	 * <p>
	 * If the specified Callable command has subcommands, the {@linkplain RunLast
	 * last} subcommand specified on the command line is executed.
	 * </p>
	 * 
	 * @param callable the command to call when {@linkplain #parse(String...)
	 *                 parsing} succeeds.
	 * @param out      the printStream to print the usage help message to when the
	 *                 user requested help
	 * @param err      the printStream to print diagnostic messages to
	 * @param ansi     including whether the usage message should include ANSI
	 *                 escape codes or not
	 * @param args     the command line arguments to parse
	 * @param <C>      the annotated object must implement Callable
	 * @param <T>      the return type of the specified {@code Callable}
	 * @throws InitializationException if the specified command object does not have
	 *                                 a {@link Command}, {@link Option} or
	 *                                 {@link Parameters} annotation
	 * @throws ExecutionException      if the Callable throws an exception
	 * @return {@code null} if an error occurred while parsing the command line
	 *         options, or if help was requested and printed. Otherwise returns the
	 *         result of calling the Callable
	 * @deprecated use {@link #execute(String...)} and {@link #getExecutionResult()}
	 *             instead
	 * @since 3.0
	 */
	@Deprecated
	public static <C extends Callable<T>, T> T call(C callable, PrintStream out, PrintStream err, Help.Ansi ansi,
			String... args) {
		final CommandLine cmd = new CommandLine(callable);
		final List<Object> results = cmd.parseWithHandlers(new RunLast().useOut(out).useAnsi(ansi),
				new DefaultExceptionHandler<List<Object>>().useErr(err).useAnsi(ansi), args);
		return CommandLine.<T>firstElement(results);
	}

	/**
	 * Delegates to
	 * {@link #call(Callable, PrintStream, PrintStream, Help.Ansi, String...)} with
	 * {@code System.err} for diagnostic error messages and {@link Help.Ansi#AUTO}.
	 * 
	 * @param callable the command to call when {@linkplain #parseArgs(String...)
	 *                 parsing} succeeds.
	 * @param out      the printStream to print the usage help message to when the
	 *                 user requested help
	 * @param args     the command line arguments to parse
	 * @param <C>      the annotated object must implement Callable
	 * @param <T>      the return type of the most specific command (must implement
	 *                 {@code Callable})
	 * @throws InitializationException if the specified command object does not have
	 *                                 a {@link Command}, {@link Option} or
	 *                                 {@link Parameters} annotation
	 * @throws ExecutionException      if the Callable throws an exception
	 * @return {@code null} if an error occurred while parsing the command line
	 *         options, or if help was requested and printed. Otherwise returns the
	 *         result of calling the Callable
	 * @deprecated use {@link #execute(String...)} and {@link #getExecutionResult()}
	 *             instead
	 * @see RunLast
	 */
	@Deprecated
	public static <C extends Callable<T>, T> T call(C callable, PrintStream out, String... args) {
		return call(callable, out, System.err, Help.Ansi.AUTO, args);
	}

	/**
	 * Equivalent to {@code new CommandLine(callable).execute(args)}, except for the
	 * return value.
	 * 
	 * @param callable the command to call when {@linkplain #parseArgs(String...)
	 *                 parsing} succeeds.
	 * @param args     the command line arguments to parse
	 * @param <C>      the annotated object must implement Callable
	 * @param <T>      the return type of the most specific command (must implement
	 *                 {@code Callable})
	 * @throws InitializationException if the specified command object does not have
	 *                                 a {@link Command}, {@link Option} or
	 *                                 {@link Parameters} annotation
	 * @throws ExecutionException      if the Callable throws an exception
	 * @return {@code null} if an error occurred while parsing the command line
	 *         options, or if help was requested and printed. Otherwise returns the
	 *         result of calling the Callable
	 * @see #execute(String...)
	 * @since 3.0
	 * @deprecated use {@link #execute(String...)} and {@link #getExecutionResult()}
	 *             instead
	 */
	@Deprecated
	public static <C extends Callable<T>, T> T call(C callable, String... args) {
		final CommandLine cmd = new CommandLine(callable);
		final List<Object> results = cmd.parseWithHandler(new RunLast(), args);
		return CommandLine.<T>firstElement(results);
	}

	/**
	 * Delegates to
	 * {@link #call(Class, IFactory, PrintStream, PrintStream, Help.Ansi, String...)}
	 * with {@code System.err} for diagnostic error messages.
	 * 
	 * @param callableClass class of the command to call when
	 *                      {@linkplain #parseArgs(String...) parsing} succeeds.
	 * @param factory       the factory responsible for instantiating the specified
	 *                      callable class and potentially injecting other
	 *                      components
	 * @param out           the printStream to print the usage help message to when
	 *                      the user requested help
	 * @param ansi          the ANSI style to use
	 * @param args          the command line arguments to parse
	 * @param <C>           the annotated class must implement Callable
	 * @param <T>           the return type of the most specific command (must
	 *                      implement {@code Callable})
	 * @throws InitializationException if the specified class cannot be instantiated
	 *                                 by the factory, or does not have a
	 *                                 {@link Command}, {@link Option} or
	 *                                 {@link Parameters} annotation
	 * @throws ExecutionException      if the Callable throws an exception
	 * @return {@code null} if an error occurred while parsing the command line
	 *         options, or if help was requested and printed. Otherwise returns the
	 *         result of calling the Callable
	 * @deprecated use {@link #execute(String...)} and {@link #getExecutionResult()}
	 *             instead
	 * @since 3.2
	 */
	@Deprecated
	public static <C extends Callable<T>, T> T call(Class<C> callableClass, IFactory factory, PrintStream out,
			Help.Ansi ansi, String... args) {
		return call(callableClass, factory, out, System.err, ansi, args);
	}

	/**
	 * Convenience method to allow command line application authors to avoid some
	 * boilerplate code in their application. The specified {@linkplain IFactory
	 * factory} will create an instance of the specified {@code callableClass}; use
	 * this method instead of
	 * {@link #call(Callable, PrintStream, PrintStream, Help.Ansi, String...)
	 * call(Callable, ...)} if you want to use a factory that performs Dependency
	 * Injection. The annotated class needs to implement {@link Callable}.
	 * <p>
	 * Consider using the {@link #execute(String...)} method instead:
	 * </p>
	 * 
	 * <pre>{@code
	 * CommandLine cmd = new CommandLine(callableClass, factory).setOut(myOutWriter()) // System.out by default
	 * 		.setErr(myErrWriter()) // System.err by default
	 * 		.setColorScheme(myColorScheme()); // default color scheme, Ansi.AUTO by default
	 * int exitCode = cmd.execute(args);
	 * //System.exit(exitCode);
	 * }</pre>
	 * <p>
	 * If the specified Callable command has subcommands, the {@linkplain RunLast
	 * last} subcommand specified on the command line is executed.
	 * </p>
	 * 
	 * @param callableClass class of the command to call when
	 *                      {@linkplain #parseArgs(String...) parsing} succeeds.
	 * @param factory       the factory responsible for instantiating the specified
	 *                      callable class and potentially injecting other
	 *                      components
	 * @param out           the printStream to print the usage help message to when
	 *                      the user requested help
	 * @param err           the printStream to print diagnostic messages to
	 * @param ansi          the ANSI style to use
	 * @param args          the command line arguments to parse
	 * @param <C>           the annotated class must implement Callable
	 * @param <T>           the return type of the most specific command (must
	 *                      implement {@code Callable})
	 * @throws InitializationException if the specified class cannot be instantiated
	 *                                 by the factory, or does not have a
	 *                                 {@link Command}, {@link Option} or
	 *                                 {@link Parameters} annotation
	 * @throws ExecutionException      if the Callable throws an exception
	 * @return {@code null} if an error occurred while parsing the command line
	 *         options, or if help was requested and printed. Otherwise returns the
	 *         result of calling the Callable
	 * @deprecated use {@link #execute(String...)} and {@link #getExecutionResult()}
	 *             instead
	 * @since 3.2
	 */
	@Deprecated
	public static <C extends Callable<T>, T> T call(Class<C> callableClass, IFactory factory, PrintStream out,
			PrintStream err, Help.Ansi ansi, String... args) {
		final CommandLine cmd = new CommandLine(callableClass, factory);
		final List<Object> results = cmd.parseWithHandlers(new RunLast().useOut(out).useAnsi(ansi),
				new DefaultExceptionHandler<List<Object>>().useErr(err).useAnsi(ansi), args);
		return CommandLine.<T>firstElement(results);
	}

	/**
	 * Delegates to
	 * {@link #call(Class, IFactory, PrintStream, PrintStream, Help.Ansi, String...)}
	 * with {@code System.err} for diagnostic error messages, and
	 * {@link Help.Ansi#AUTO}.
	 * 
	 * @param callableClass class of the command to call when
	 *                      {@linkplain #parseArgs(String...) parsing} succeeds.
	 * @param factory       the factory responsible for instantiating the specified
	 *                      callable class and potentially injecting other
	 *                      components
	 * @param out           the printStream to print the usage help message to when
	 *                      the user requested help
	 * @param args          the command line arguments to parse
	 * @param <C>           the annotated class must implement Callable
	 * @param <T>           the return type of the most specific command (must
	 *                      implement {@code Callable})
	 * @throws InitializationException if the specified class cannot be instantiated
	 *                                 by the factory, or does not have a
	 *                                 {@link Command}, {@link Option} or
	 *                                 {@link Parameters} annotation
	 * @throws ExecutionException      if the Callable throws an exception
	 * @return {@code null} if an error occurred while parsing the command line
	 *         options, or if help was requested and printed. Otherwise returns the
	 *         result of calling the Callable
	 * @deprecated use {@link #execute(String...)} and {@link #getExecutionResult()}
	 *             instead
	 * @since 3.2
	 */
	@Deprecated
	public static <C extends Callable<T>, T> T call(Class<C> callableClass, IFactory factory, PrintStream out,
			String... args) {
		return call(callableClass, factory, out, System.err, Help.Ansi.AUTO, args);
	}

	/**
	 * Equivalent to {@code new CommandLine(callableClass, factory).execute(args)},
	 * except for the return value.
	 * 
	 * @param callableClass class of the command to call when
	 *                      {@linkplain #parseArgs(String...) parsing} succeeds.
	 * @param factory       the factory responsible for instantiating the specified
	 *                      callable class and potentially inject other components
	 * @param args          the command line arguments to parse
	 * @param <C>           the annotated class must implement Callable
	 * @param <T>           the return type of the most specific command (must
	 *                      implement {@code Callable})
	 * @throws InitializationException if the specified class cannot be instantiated
	 *                                 by the factory, or does not have a
	 *                                 {@link Command}, {@link Option} or
	 *                                 {@link Parameters} annotation
	 * @throws ExecutionException      if the Callable throws an exception
	 * @return {@code null} if an error occurred while parsing the command line
	 *         options, or if help was requested and printed. Otherwise returns the
	 *         result of calling the Callable
	 * @see #execute(String...)
	 * @since 3.2
	 * @deprecated use {@link #execute(String...)} and {@link #getExecutionResult()}
	 *             instead
	 */
	@Deprecated
	public static <C extends Callable<T>, T> T call(Class<C> callableClass, IFactory factory, String... args) {
		final CommandLine cmd = new CommandLine(callableClass, factory);
		final List<Object> results = cmd.parseWithHandler(new RunLast(), args);
		return CommandLine.<T>firstElement(results);
	}

	static Charset charsetForName(String encoding) {
		if (encoding != null) {
			if ("cp65001".equalsIgnoreCase(encoding)) {
				encoding = "UTF-8";
			} // #1474 MS Windows uses code page 65001 for UTF8
			try {
				return Charset.forName(encoding);
			} catch (final Exception e) {
				// fallback to default charset if the requested encoding is not available
				final Charset defaultCharset = Charset.defaultCharset();
				CommandLine.tracer().info("The %s encoding in not available, falling back to %s", encoding,
						defaultCharset.name());
				return defaultCharset;
			}
		}
		return Charset.defaultCharset();
	}

	static void close(Closeable closeable) {
		if (closeable == null) {
			return;
		}
		try {
			closeable.close();
		} catch (final Exception ex) {
			CommandLine.tracer().warn("Could not close " + closeable + ": " + ex.toString());
		}
	}

	@SuppressWarnings("unchecked")
	private static Stack<String> copy(Stack<String> stack) {
		return (Stack<String>) stack.clone();
	}

	private static String createMissingParameterMessage(ArgSpec argSpec, Range arity,
			List<PositionalParamSpec> missingList, Stack<String> args, int available) {
		if (arity.min == 1) {
			if (argSpec.isOption()) {
				return "Missing required parameter for " + optionDescription("", argSpec, 0);
			}
			String sep = "";
			String names = ": ";
			String indices = "";
			String infix = " at index ";
			int count = 0;
			for (final PositionalParamSpec missing : missingList) {
				if (missing.arity().min > 0) {
					names += sep + "'" + missing.paramLabel() + "'";
					indices += sep + missing.index();
					sep = ", ";
					count++;
				}
			}
			String msg = "Missing required parameter";
			if (count > 1 || arity.min - available > 1) {
				msg += "s";
			}
			if (count > 1) {
				infix = " at indices ";
			}
			return System.getProperty("picocli.verbose.errors") != null ? msg + names + infix + indices : msg + names;
		} else if (args.isEmpty()) {
			return optionDescription("", argSpec, 0) + " requires at least " + arity.min
					+ " values, but none were specified.";
		} else {
			return optionDescription("", argSpec, 0) + " requires at least " + arity.min + " values, but only "
					+ available + " were specified: " + reverse(args);
		}
	}

	/**
	 * Convenience method that returns
	 * {@code new DefaultExceptionHandler<List<Object>>()}.
	 */
	public static DefaultExceptionHandler<List<Object>> defaultExceptionHandler() {
		return new DefaultExceptionHandler<List<Object>>();
	}

	/**
	 * Returns the default {@link IFactory} implementation used if no factory was
	 * specified in the {@link #CommandLine(Object) CommandLine constructor}.
	 * <p>
	 * This implementation has special logic for instantiating {@code Collections}
	 * and {@code Maps}, and otherwise tries to create an instance by invoking the
	 * default constructor of the specified class.
	 * </p>
	 * <p>
	 * Special logic for instantiating Collections and Maps:
	 * </p>
	 * 
	 * <pre>{@code
	 * // if class is an interface that extends java.util.Collection, return a new instance of:
	 * 1. List       -> ArrayList
	 * 2. SortedSet  -> TreeSet
	 * 3. Set        -> LinkedHashSet
	 * 4. Queue      -> LinkedList
	 * 5. Collection -> ArrayList
	 *
	 * // if extending or implementing java.util.Map:
	 * 1. try invoking the default constructor; return this on success.
	 * 2. if this fails, return a LinkedHashMap
	 * }</pre>
	 * 
	 * @since 4.0
	 */
	public static IFactory defaultFactory() {
		return new DefaultFactory();
	}

	private static boolean empty(Object[] array) {
		return array == null || array.length == 0;
	}

	private static boolean empty(String str) {
		return str == null || str.trim().length() == 0;
	}

	/** @since 4.0 */
	static Integer executeHelpRequest(List<CommandLine> parsedCommands) {
		final Tracer t = CommandLine.tracer();
		for (final CommandLine parsed : parsedCommands) {
			final Help.ColorScheme colorScheme = parsed.getColorScheme();
			final PrintWriter out = parsed.getOut();
			if (parsed.isUsageHelpRequested()) {
				t.debug("Printing usage help for '%s' as requested.", parsed.commandSpec.qualifiedName());
				parsed.usage(out, colorScheme);
				return parsed.getCommandSpec().exitCodeOnUsageHelp();
			} else if (parsed.isVersionHelpRequested()) {
				t.debug("Printing version info for '%s' as requested.", parsed.commandSpec.qualifiedName());
				parsed.printVersionHelp(out, colorScheme.ansi);
				return parsed.getCommandSpec().exitCodeOnVersionHelp();
			} else if (parsed.getCommandSpec().helpCommand()) {
				final String fullName = parsed.commandSpec.qualifiedName();
				final PrintWriter err = parsed.getErr();
				if ((parsed.getCommand()) instanceof IHelpCommandInitializable2) {
					t.debug("Initializing helpCommand '%s' (IHelpCommandInitializable2::init)...", fullName);
					((IHelpCommandInitializable2) parsed.getCommand()).init(parsed, colorScheme, out, err);
				} else if ((parsed.getCommand()) instanceof IHelpCommandInitializable) {
					t.debug("Initializing helpCommand '%s' (IHelpCommandInitializable::init)...", fullName);
					((IHelpCommandInitializable) parsed.getCommand()).init(parsed, colorScheme.ansi, System.out,
							System.err);
				} else {
					t.debug("helpCommand '%s' does not implement IHelpCommandInitializable2 or IHelpCommandInitializable...",
							fullName);
				}
				t.debug("Executing helpCommand '%s'...", fullName);
				executeUserObject(parsed, new ArrayList<Object>());
				return parsed.getCommandSpec().exitCodeOnUsageHelp();
			}
		}
		t.debug("Help was not requested. Continuing to process ParseResult...");
		return null;
	}

	/**
	 * Helper method that may be useful when processing the {@code ParseResult} that
	 * results from successfully {@linkplain #parseArgs(String...) parsing} command
	 * line arguments. This method prints out
	 * {@linkplain #usage(PrintWriter, Help.ColorScheme) usage help} to the
	 * {@linkplain CommandLine#getOut() configured output writer} if
	 * {@linkplain #isUsageHelpRequested() requested} or
	 * {@linkplain #printVersionHelp(PrintWriter, Help.Ansi, Object...) version
	 * help} to the {@linkplain CommandLine#getOut() configured output writer} if
	 * {@linkplain #isVersionHelpRequested() requested} and returns
	 * {@link CommandSpec#exitCodeOnUsageHelp()} or
	 * {@link CommandSpec#exitCodeOnVersionHelp()}, respectively. If the command is
	 * a {@link Command#helpCommand()} and {@code runnable} or {@code callable},
	 * that command is executed and this method returns
	 * {@link CommandSpec#exitCodeOnUsageHelp()}. Otherwise, if none of the
	 * specified {@code CommandLine} objects have help requested, this method
	 * returns {@code null}.
	 * <p>
	 * Note that this method <em>only</em> looks at the {@link Option#usageHelp()
	 * usageHelp} and {@link Option#versionHelp() versionHelp} attributes. The
	 * {@link Option#help() help} attribute is ignored.
	 * </p>
	 * <p>
	 * <b>Implementation note:</b>
	 * </p>
	 * <p>
	 * When an error occurs while processing the help request, it is recommended
	 * custom Help commands throw a {@link ParameterException} with a reference to
	 * the parent command. This will print the error message and the usage for the
	 * parent command, and will use the exit code of the exception handler if one
	 * was set.
	 * </p>
	 * 
	 * @param parseResult contains the {@code CommandLine} objects found during
	 *                    parsing; check these to see if help was requested
	 * @return {@link CommandSpec#exitCodeOnUsageHelp()} if usage help was
	 *         requested, {@link CommandSpec#exitCodeOnVersionHelp()} if version
	 *         help was requested, and {@code null} otherwise
	 * @see IHelpCommandInitializable2
	 * @since 4.0
	 */
	public static Integer executeHelpRequest(ParseResult parseResult) {
		return executeHelpRequest(parseResult.asCommandLineList());
	}

	private static List<Object> executeUserObject(CommandLine parsed, List<Object> executionResultList) {
		final Tracer tracer = CommandLine.tracer();

		final Object command = parsed.getCommand();
		if (command instanceof Runnable) {
			try {
				tracer.debug("Invoking Runnable::run on user object %s@%s...", command.getClass().getName(),
						Integer.toHexString(command.hashCode()));
				((Runnable) command).run();
				parsed.setExecutionResult(null); // 4.0
				executionResultList.add(null); // for compatibility with picocli 2.x
				return executionResultList;
			} catch (final ParameterException ex) {
				throw ex;
			} catch (final ExecutionException ex) {
				throw ex;
			} catch (final Exception ex) {
				throw new ExecutionException(parsed, "Error while running command (" + command + "): " + ex, ex);
			}
		} else if (command instanceof Callable) {
			try {
				tracer.debug("Invoking Callable::call on user object %s@%s...", command.getClass().getName(),
						Integer.toHexString(command.hashCode()));
				@SuppressWarnings("unchecked")
				final Callable<Object> callable = (Callable<Object>) command;
				final Object executionResult = callable.call();
				parsed.setExecutionResult(executionResult);
				executionResultList.add(executionResult);
				return executionResultList;
			} catch (final ParameterException ex) {
				throw ex;
			} catch (final ExecutionException ex) {
				throw ex;
			} catch (final Exception ex) {
				throw new ExecutionException(parsed, "Error while calling command (" + command + "): " + ex, ex);
			}
		} else if (command instanceof Method) {
			try {
				final Method method = (Method) command;
				final Object[] parsedArgs = parsed.getCommandSpec().commandMethodParamValues();
				Object executionResult;
				if (Modifier.isStatic(method.getModifiers())) {
					tracer.debug("Invoking static method %s with parameters %s", method, Arrays.toString(parsedArgs));
					executionResult = method.invoke(null, parsedArgs); // invoke static method
				} else {
					final Object instance = (parsed.getCommandSpec().parent() != null)
							? parsed.getCommandSpec().parent().userObject()
							: parsed.factory.create(method.getDeclaringClass());
					tracer.debug("Invoking method %s on %s@%s with parameters %s", method,
							instance.getClass().getName(), Integer.toHexString(instance.hashCode()),
							Arrays.toString(parsedArgs));
					executionResult = method.invoke(instance, parsedArgs);
				}
				parsed.setExecutionResult(executionResult);
				executionResultList.add(executionResult);
				return executionResultList;
			} catch (final InvocationTargetException ex) {
				final Throwable t = ex.getTargetException();
				if (t instanceof ParameterException) {
					throw (ParameterException) t;
				} else if (t instanceof ExecutionException) {
					throw (ExecutionException) t;
				} else {
					throw new ExecutionException(parsed, "Error while calling command (" + command + "): " + t, t);
				}
			} catch (final Exception ex) {
				throw new ExecutionException(parsed, "Unhandled error while calling command (" + command + "): " + ex,
						ex);
			}
		}
		if (parsed.getSubcommands().isEmpty()) {
			throw new ExecutionException(parsed,
					"Parsed command (" + command + ") is not a Method, Runnable or Callable");
		} else {
			throw new ParameterException(parsed, "Missing required subcommand");
		}
	}

	@SuppressWarnings("unchecked")
	private static <T> T firstElement(List<Object> results) {
		return (results == null || results.isEmpty()) ? null : (T) results.get(0);
	}

	static <T> List<T> flatList(Collection<? extends Collection<T>> collection) {
		final List<T> result = new ArrayList<T>();
		for (final Collection<T> sub : collection) {
			result.addAll(sub);
		}
		return result;
	}

	private static String format(String formatString, Object... params) {
		try {
			return formatString == null ? "" : String.format(formatString, params);
		} catch (final IllegalFormatException ex) {
			CommandLine.tracer()
					.warn("Could not format '%s' (Underlying error: %s). "
							+ "Using raw String: '%%n' format strings have not been replaced with newlines. "
							+ "Please ensure to escape '%%' characters with another '%%'.", formatString,
							ex.getMessage());
			return formatString;
		}
	}

	/**
	 * Helper to get methods of a class annotated with {@link Command @Command} via
	 * reflection, optionally filtered by method name (not
	 * {@link Command#name() @Command.name}). Methods have to be either public
	 * (inherited) members or be declared by {@code cls}, that is "inherited" static
	 * or protected methods will not be picked up.
	 *
	 * @param cls        the class to search for methods annotated with
	 *                   {@code @Command}
	 * @param methodName if not {@code null}, return only methods whose method name
	 *                   (not {@link Command#name() @Command.name}) equals this
	 *                   string. Ignored if {@code null}.
	 * @return the matching command methods, or an empty list
	 * @see #invoke(String, Class, String...)
	 * @since 3.6.0
	 */
	public static List<Method> getCommandMethods(Class<?> cls, String methodName) {
		return getCommandMethods(cls, methodName, true);
	}

	private static List<Method> getCommandMethods(Class<?> cls, String methodName, boolean includeInherited) {
		final Set<Method> candidates = new HashSet<Method>();
		if (includeInherited) {
			// traverse public member methods (excludes static/non-public, includes
			// inherited)
			candidates.addAll(Arrays.asList(Assert.notNull(cls, "class").getMethods()));
		}
		// traverse directly declared methods (includes static/non-public, excludes
		// inherited)
		candidates.addAll(Arrays.asList(Assert.notNull(cls, "class").getDeclaredMethods()));

		final List<Method> result = new ArrayList<Method>();
		for (final Method method : candidates) {
			if (method.isAnnotationPresent(Command.class)) {
				if (methodName == null || methodName.equals(method.getName())) {
					result.add(method);
				}
			}
		}
		Collections.sort(result, new Comparator<Method>() {
			@Override
			public int compare(Method o1, Method o2) {
				return o1.getName().compareTo(o2.getName());
			}
		});
		return result;
	}

	private static Object getOptionalEmpty() throws Exception {
		return Class.forName("java.util.Optional").getMethod("empty").invoke(null);
	}

	private static Object getOptionalOfNullable(Object newValue) throws Exception {
		return Class.forName("java.util.Optional").getMethod("ofNullable", Object.class).invoke(null, newValue);
	}

	static Charset getStderrEncoding() {
		return charsetForName(System.getProperty("sun.stderr.encoding"));
	}

	static Charset getStdoutEncoding() {
		return charsetForName(System.getProperty("sun.stdout.encoding"));
	}

	private static int handleUnhandled(Exception ex, CommandLine cmd, int defaultExitCode) {
		cmd.getErr().print(throwableToColorString(ex, cmd.getColorScheme()));
		cmd.getErr().flush();
		return mappedExitCode(ex, cmd.getExitCodeExceptionMapper(), defaultExitCode);
	}

	/**
	 * Delegates to
	 * {@link #invoke(String, Class, PrintStream, PrintStream, Help.Ansi, String...)}
	 * with the specified stream for requested usage help messages,
	 * {@code System.err} for diagnostic error messages, and the specified Ansi
	 * mode.
	 * 
	 * @param methodName the {@code @Command}-annotated method to build a
	 *                   {@link CommandSpec} model from, and run when
	 *                   {@linkplain #parseArgs(String...) parsing} succeeds.
	 * @param cls        the class where the {@code @Command}-annotated method is
	 *                   declared, or a subclass
	 * @param out        the printstream to print requested help message to
	 * @param ansi       whether the usage message should include ANSI escape codes
	 *                   or not
	 * @param args       the command line arguments to parse
	 * @see #invoke(String, Class, PrintStream, PrintStream, Help.Ansi, String...)
	 * @throws InitializationException if the specified method does not have a
	 *                                 {@link Command} annotation, or if the
	 *                                 specified class contains multiple
	 *                                 {@code @Command}-annotated methods with the
	 *                                 specified name
	 * @throws ExecutionException      if the Runnable throws an exception
	 * @see #parseWithHandlers(IParseResultHandler2, IExceptionHandler2, String...)
	 * @deprecated use {@link #execute(String...)} and {@link #getExecutionResult()}
	 *             instead
	 * @since 3.6
	 */
	@Deprecated
	public static Object invoke(String methodName, Class<?> cls, PrintStream out, Help.Ansi ansi, String... args) {
		return invoke(methodName, cls, out, System.err, ansi, args);
	}

	/**
	 * Convenience method to allow command line application authors to avoid some
	 * boilerplate code in their application. Constructs a {@link CommandSpec} model
	 * from the {@code @Option} and {@code @Parameters}-annotated method parameters
	 * of the {@code @Command}-annotated method, parses the specified command line
	 * arguments and invokes the specified method.
	 * <p>
	 * Consider using the {@link #execute(String...)} method instead:
	 * </p>
	 * 
	 * <pre>{@code
	 * Method commandMethod = getCommandMethods(cls, methodName).get(0);
	 * CommandLine cmd = new CommandLine(commandMethod).setOut(myOutWriter()) // System.out by default
	 * 		.setErr(myErrWriter()) // System.err by default
	 * 		.setColorScheme(myColorScheme()); // default color scheme, Ansi.AUTO by default
	 * int exitCode = cmd.execute(args);
	 * //System.exit(exitCode);
	 * }</pre>
	 * 
	 * @param methodName the {@code @Command}-annotated method to build a
	 *                   {@link CommandSpec} model from, and run when
	 *                   {@linkplain #parseArgs(String...) parsing} succeeds.
	 * @param cls        the class where the {@code @Command}-annotated method is
	 *                   declared, or a subclass
	 * @param out        the printStream to print the usage help message to when the
	 *                   user requested help
	 * @param err        the printStream to print diagnostic messages to
	 * @param ansi       whether the usage message should include ANSI escape codes
	 *                   or not
	 * @param args       the command line arguments to parse
	 * @throws InitializationException if the specified method does not have a
	 *                                 {@link Command} annotation, or if the
	 *                                 specified class contains multiple
	 *                                 {@code @Command}-annotated methods with the
	 *                                 specified name
	 * @throws ExecutionException      if the method throws an exception
	 * @deprecated use {@link #execute(String...)} and {@link #getExecutionResult()}
	 *             instead
	 * @since 3.6
	 */
	@Deprecated
	public static Object invoke(String methodName, Class<?> cls, PrintStream out, PrintStream err, Help.Ansi ansi,
			String... args) {
		final List<Method> candidates = getCommandMethods(cls, methodName);
		if (candidates.size() != 1) {
			throw new InitializationException("Expected exactly one @Command-annotated method for " + cls.getName()
					+ "::" + methodName + "(...), but got: " + candidates);
		}
		final Method method = candidates.get(0);
		final CommandLine cmd = new CommandLine(method);
		final List<Object> list = cmd.parseWithHandlers(new RunLast().useOut(out).useAnsi(ansi),
				new DefaultExceptionHandler<List<Object>>().useErr(err).useAnsi(ansi), args);
		return list == null ? null : list.get(0);
	}

	/**
	 * Delegates to
	 * {@link #invoke(String, Class, PrintStream, PrintStream, Help.Ansi, String...)}
	 * with the specified stream for requested usage help messages,
	 * {@code System.err} for diagnostic error messages, and {@link Help.Ansi#AUTO}.
	 * 
	 * @param methodName the {@code @Command}-annotated method to build a
	 *                   {@link CommandSpec} model from, and run when
	 *                   {@linkplain #parseArgs(String...) parsing} succeeds.
	 * @param cls        the class where the {@code @Command}-annotated method is
	 *                   declared, or a subclass
	 * @param out        the printstream to print requested help message to
	 * @param args       the command line arguments to parse
	 * @see #invoke(String, Class, PrintStream, PrintStream, Help.Ansi, String...)
	 * @throws InitializationException if the specified method does not have a
	 *                                 {@link Command} annotation, or if the
	 *                                 specified class contains multiple
	 *                                 {@code @Command}-annotated methods with the
	 *                                 specified name
	 * @throws ExecutionException      if the Runnable throws an exception
	 * @see #parseWithHandlers(IParseResultHandler2, IExceptionHandler2, String...)
	 * @deprecated use {@link #execute(String...)} and {@link #getExecutionResult()}
	 *             instead
	 * @since 3.6
	 */
	@Deprecated
	public static Object invoke(String methodName, Class<?> cls, PrintStream out, String... args) {
		return invoke(methodName, cls, out, System.err, Help.Ansi.AUTO, args);
	}

	/**
	 * Delegates to
	 * {@link #invoke(String, Class, PrintStream, PrintStream, Help.Ansi, String...)}
	 * with {@code System.out} for requested usage help messages, {@code System.err}
	 * for diagnostic error messages, and {@link Help.Ansi#AUTO}.
	 * 
	 * @param methodName the {@code @Command}-annotated method to build a
	 *                   {@link CommandSpec} model from, and run when
	 *                   {@linkplain #parseArgs(String...) parsing} succeeds.
	 * @param cls        the class where the {@code @Command}-annotated method is
	 *                   declared, or a subclass
	 * @param args       the command line arguments to parse
	 * @see #invoke(String, Class, PrintStream, PrintStream, Help.Ansi, String...)
	 * @throws InitializationException if the specified method does not have a
	 *                                 {@link Command} annotation, or if the
	 *                                 specified class contains multiple
	 *                                 {@code @Command}-annotated methods with the
	 *                                 specified name
	 * @throws ExecutionException      if the Runnable throws an exception
	 * @see #parseWithHandlers(IParseResultHandler2, IExceptionHandler2, String...)
	 * @since 3.6
	 * @deprecated use {@link #execute(String...)} and {@link #getExecutionResult()}
	 *             instead
	 */
	@Deprecated
	public static Object invoke(String methodName, Class<?> cls, String... args) {
		return invoke(methodName, cls, System.out, System.err, Help.Ansi.AUTO, args);
	}

	private static boolean isBoolean(Class<?> type) {
		return type == Boolean.class || type == Boolean.TYPE;
	}

	private static boolean isBoolean(Class<?>[] types) {
		return isBoolean(types[0]) || (isOptional(types[0]) && isBoolean(types[1]));
	}

	private static boolean isMultiValue(Class<?> cls) {
		return (cls.isArray() && cls != char[].class) || Collection.class.isAssignableFrom(cls)
				|| Map.class.isAssignableFrom(cls);
	}

	private static boolean isOptional(Class<?> cls) {
		return cls != null && "java.util.Optional".equals(cls.getName());
	} // #1108

	private static Map<String, Object> mapOf(String key, Object value, Object... other) {
		final LinkedHashMap<String, Object> result = new LinkedHashMap<String, Object>();
		result.put(key, value);
		for (int i = 0; i < other.length - 1; i += 2) {
			result.put(String.valueOf(other[i]), other[i + 1]);
		}
		return result;
	}

	private static int mappedExitCode(Throwable t, IExitCodeExceptionMapper mapper, int defaultExitCode) {
		try {
			return (mapper != null) ? mapper.getExitCode(t) : defaultExitCode;
		} catch (final Exception ex) {
			ex.printStackTrace();
			return defaultExitCode;
		}
	}

	static PrintWriter newPrintWriter(OutputStream stream, Charset charset) {
		return new PrintWriter(new BufferedWriter(new OutputStreamWriter(stream, charset)), true);
	}

	private static String optionDescription(String prefix, ArgSpec argSpec, int optionParamIndex) {
		String desc;
		if (argSpec.isOption()) {
			desc = prefix + "option '" + ((OptionSpec) argSpec).longestName() + "'";
			if (optionParamIndex >= 0) { // we are describing an option parameter
				if (argSpec.arity().max > 1) {
					desc += " at index " + optionParamIndex;
				}
				if (argSpec.arity().max > 0) {
					desc += " (" + argSpec.paramLabel() + ")";
				}
			}
		} else {
			desc = prefix + "positional parameter at index " + ((PositionalParamSpec) argSpec).index() + " ("
					+ argSpec.paramLabel() + ")";
		}
		return desc;
	}

	/**
	 * <p>
	 * Convenience method that initializes the specified annotated object from the
	 * specified command line arguments.
	 * </p>
	 * <p>
	 * This is equivalent to
	 * </p>
	 * 
	 * <pre>
	 * new CommandLine(command).parseArgs(args);
	 * return command;
	 * </pre>
	 * <p>
	 * All this method does is parse the arguments and populate the annotated fields
	 * and methods. The caller is responsible for catching any exceptions, handling
	 * requests for usage help or version information, and invoking the business
	 * logic. Applications may be interested in using the
	 * {@link #execute(String...)} method instead.
	 * </p>
	 *
	 * @param command the object to initialize. This object contains fields
	 *                annotated with {@code @Option} or {@code @Parameters}.
	 * @param args    the command line arguments to parse
	 * @param <T>     the type of the annotated object
	 * @return the specified annotated object
	 * @throws InitializationException if the specified command object does not have
	 *                                 a {@link Command}, {@link Option} or
	 *                                