/*
 * The MIT License
 *
 *  Copyright (c) 2020, Mahmoud Ben Hassine (mahmoud.benhassine@icloud.com)
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a copy
 *  of this software and associated documentation files (the "Software"), to deal
 *  in the Software without restriction, including without limitation the rights
 *  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 *  copies of the Software, and to permit persons to whom the Software is
 *  furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 *  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 *  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 *  THE SOFTWARE.
 */
package org.rossonet.ext.rules.mvel;

import java.io.Serializable;

import org.mvel2.MVEL;
import org.mvel2.ParserContext;
import org.rossonet.ext.rules.api.Condition;
import org.rossonet.ext.rules.api.Facts;

/**
 * This class is an implementation of {@link Condition} that uses
 * <a href="https://github.com/mvel/mvel">MVEL</a> to evaluate the condition.
 *
 * @author Mahmoud Ben Hassine (mahmoud.benhassine@icloud.com)
 */
public class MVELCondition implements Condition {

	private final Serializable compiledExpression;

	/**
	 * Create a new {@link MVELCondition}.
	 *
	 * @param expression the condition written in expression language
	 */
	public MVELCondition(String expression) {
		compiledExpression = MVEL.compileExpression(expression);
	}

	/**
	 * Create a new {@link MVELCondition}.
	 *
	 * @param expression    the condition written in expression language
	 * @param parserContext the MVEL parser context
	 */
	public MVELCondition(String expression, ParserContext parserContext) {
		compiledExpression = MVEL.compileExpression(expression, parserContext);
	}

	@Override
	public boolean evaluate(Facts facts) {
		// MVEL.evalToBoolean does not accept compiled expressions..
		return (boolean) MVEL.executeExpression(compiledExpression, facts.asMap());
	}
}
