/*
 * The MIT License
 *
 *  Copyright (c) 2020, Mahmoud Ben Hassine (mahmoud.benhassine@icloud.com)
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a copy
 *  of this software and associated documentation files (the "Software"), to deal
 *  in the Software without restriction, including without limitation the rights
 *  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 *  copies of the Software, and to permit persons to whom the Software is
 *  furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 *  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 *  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 *  THE SOFTWARE.
 */
package org.rossonet.ext.rules.api;

/**
 * A listener for rule execution events.
 *
 * @author Mahmoud Ben Hassine (mahmoud.benhassine@icloud.com)
 */
public interface RuleListener {

	/**
	 * Triggered after the evaluation of a rule.
	 *
	 * @param rule             that has been evaluated
	 * @param facts            known after evaluating the rule
	 * @param evaluationResult true if the rule evaluated to true, false otherwise
	 */
	default void afterEvaluate(Rule rule, Facts facts, boolean evaluationResult) {
	}

	/**
	 * Triggered before the evaluation of a rule.
	 *
	 * @param rule  being evaluated
	 * @param facts known before evaluating the rule
	 * @return true if the rule should be evaluated, false otherwise
	 */
	default boolean beforeEvaluate(Rule rule, Facts facts) {
		return true;
	}

	/**
	 * Triggered before the execution of a rule.
	 *
	 * @param rule  the current rule
	 * @param facts known facts before executing the rule
	 */
	default void beforeExecute(Rule rule, Facts facts) {
	}

	/**
	 * Triggered on condition evaluation error due to any runtime exception.
	 *
	 * @param rule      that has been evaluated
	 * @param facts     known while evaluating the rule
	 * @param exception that happened while attempting to evaluate the condition.
	 */
	default void onEvaluationError(Rule rule, Facts facts, Exception exception) {
	}

	/**
	 * Triggered after a rule has failed.
	 *
	 * @param rule      the current rule
	 * @param facts     known facts after executing the rule
	 * @param exception the exception thrown when attempting to execute the rule
	 */
	default void onFailure(Rule rule, Facts facts, Exception exception) {
	}

	/**
	 * Triggered after a rule has been executed successfully.
	 *
	 * @param rule  the current rule
	 * @param facts known facts after executing the rule
	 */
	default void onSuccess(Rule rule, Facts facts) {
	}

}
