/**
 * The BSD License
 *
 * Copyright (c) 2010-2012 RIPE NCC
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *   - Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 *   - Redistributions in binary form must reproduce the above copyright notice,
 *     this list of conditions and the following disclaimer in the documentation
 *     and/or other materials provided with the distribution.
 *   - Neither the name of the RIPE NCC nor the names of its contributors may be
 *     used to endorse or promote products derived from this software without
 *     specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
package net.ripe.rpki.validator
package config

import java.io.File
import java.util.EnumSet
import javax.servlet.DispatcherType

import com.codahale.metrics.servlets.HealthCheckServlet
import grizzled.slf4j.Logging
import net.ripe.rpki.validator.api.RestApi
import net.ripe.rpki.validator.bgp.preview._
import net.ripe.rpki.validator.config.health.HealthChecks
import net.ripe.rpki.validator.lib.{UserPreferences, _}
import net.ripe.rpki.validator.models.{Idle, IgnoreFilter, TrustAnchorData, _}
import net.ripe.rpki.validator.rtr.{Pdu, RTRServer}
import net.ripe.rpki.validator.util.TrustAnchorLocator
import org.apache.commons.io.FileUtils
import org.apache.http.client.methods.HttpGet
import org.eclipse.jetty.server.Server
import org.joda.time.DateTime

import scala.Predef._
import scala.collection.JavaConverters._
import scala.concurrent.Future
import scala.concurrent.stm._
import scala.math.Ordering.Implicits._
import scalaz.{Failure, Success}

object Main {
  private val sessionId: Pdu.SessionId = Pdu.randomSessionid

  def main(args: Array[String]): Unit = {
    System.setProperty("VALIDATOR_LOG_FILE", ApplicationOptions.applicationLogFileName)
    System.setProperty("RTR_LOG_FILE", ApplicationOptions.rtrLogFileName)
    new Main()
  }
}

class Main extends Http with Logging { main =>
  import scala.concurrent.duration._

  implicit val actorSystem = akka.actor.ActorSystem()
  import actorSystem.dispatcher

  val startedAt = System.currentTimeMillis

  val bgpRisDumps = Ref(Seq(
    BgpRisDump("http://www.ris.ripe.net/dumps/riswhoisdump.IPv4.gz"),
    BgpRisDump("http://www.ris.ripe.net/dumps/riswhoisdump.IPv6.gz")))

  val bgpAnnouncementValidator = new BgpAnnouncementValidator

  val dataFile = ApplicationOptions.dataFileLocation
  val data = PersistentDataSerialiser.read(dataFile).getOrElse(PersistentData())

  val trustAnchors = loadTrustAnchors().all.map { ta => ta.copy(enabled = data.trustAnchorData.get(ta.name).map(_.enabled).getOrElse(true)) }
  val roas = ValidatedObjects(new TrustAnchors(trustAnchors.filter(ta => ta.enabled)))

  override def trustedCertsLocation = ApplicationOptions.trustedSslCertsLocation

  val userPreferences = Ref(data.userPreferences)

  val bgpRisDumpDownloader = new BgpRisDumpDownloader(http)

  val memoryImage = Ref(
    MemoryImage(data.filters, data.whitelist, new TrustAnchors(trustAnchors), roas))

  def updateMemoryImage(f: MemoryImage => MemoryImage)(implicit transaction: MaybeTxn) {
    atomic { implicit transaction =>
      val oldVersion = memoryImage().version

      memoryImage.transform(f)

      if (oldVersion != memoryImage().version) {
        bgpAnnouncementValidator.startUpdate(main.bgpRisDumps().flatMap(_.announcedRoutes), memoryImage().getDistinctRtrPrefixes.toSeq)
        rtrServer.notify(memoryImage().version)
      }
    }
  }

  wipeRsyncDiskCache()

  val rtrServer = runRtrServer()

  runWebServer()

  actorSystem.scheduler.schedule(initialDelay = 0.seconds, interval = 10.seconds) { runValidator() }
  actorSystem.scheduler.schedule(initialDelay = 0.seconds, interval = 2.hours) { refreshRisDumps() }

  private def loadTrustAnchors(): TrustAnchors = {
    val tals = FileUtils.listFiles(ApplicationOptions.talDirLocation, Array("tal"), false)
    TrustAnchors.load(tals.asScala.toSeq)
  }

  private def refreshRisDumps() {
    Future.traverse(bgpRisDumps.single.get)(bgpRisDumpDownloader.download) foreach { dumps =>
      atomic { implicit transaction =>
        bgpRisDumps() = dumps
        bgpAnnouncementValidator.startUpdate(dumps.flatMap(_.announcedRoutes), memoryImage().getDistinctRtrPrefixes.toSeq)
      }
    }
  }

  private def runValidator() {
    import lib.DateAndTime._

    val now = new DateTime
    val needUpdating = for {
      ta <- memoryImage.single.get.trustAnchors.all if ta.status.isIdle
      Idle(nextUpdate, _) = ta.status
      if nextUpdate <= now
    } yield ta.name

    runValidator(needUpdating)
  }

  private def runValidator(trustAnchorNames: Seq[String]) {
    val maxStaleDays = userPreferences.single.get.maxStaleDays
    val trustAnchors = memoryImage.single.get.trustAnchors.all

    val taLocators = trustAnchorNames.flatMap { name => trustAnchors.find(_.name == name) }

    for (trustAnchorLocator <- taLocators) {
      Future {
        val process = new TrustAnchorValidationProcess(trustAnchorLocator.locator, maxStaleDays,
          ApplicationOptions.workDirLocation,
          ApplicationOptions.rsyncDirLocation,
          trustAnchorLocator.name,
          ApplicationOptions.enableLooseValidation
        ) with TrackValidationProcess with ValidationProcessLogger {
          override val memoryImage = main.memoryImage
        }
        try {
          process.runProcess() match {
            case Success(validatedObjectsByUri) =>
              val validatedObjects = validatedObjectsByUri.values.toSeq
              updateMemoryImage(_.updateValidatedObjects(trustAnchorLocator.locator, validatedObjects))
            case Failure(_) =>
          }
        } finally {
          process.shutdown()
        }
      }
    }
  }

  private def runWebServer() {
    val server = setup(new Server(ApplicationOptions.httpPort))

    sys.addShutdownHook({
      server.stop()
      logger.info("Terminating...")
    })
    server.start()
    logger.info("Welcome to the RIPE NCC RPKI Validator, now available on port " + ApplicationOptions.httpPort + ". Hit CTRL+C to terminate.")
  }

  private def runRtrServer(): RTRServer = {
    val rtrServer = new RTRServer(
      port = ApplicationOptions.rtrPort,
      closeOnError = ApplicationOptions.rtrCloseOnError,
      sendNotify = ApplicationOptions.rtrSendNotify,
      getCurrentCacheSerial = {
        () => memoryImage.single.get.version
      },
      getCurrentRtrPrefixes = {
        () => memoryImage.single.get.getDistinctRtrPrefixes
      },
      getCurrentSessionId = {
        () => Main.sessionId
      },
      hasTrustAnchorsEnabled = {
        () => memoryImage.single.get.trustAnchors.hasEnabledAnchors
      })
    rtrServer.startServer()
    rtrServer
  }

  private def setup(server: Server): Server = {
    import org.eclipse.jetty.server.NCSARequestLog
    import org.eclipse.jetty.server.handler.{HandlerCollection, RequestLogHandler}
    import org.eclipse.jetty.servlet._
    import org.scalatra._

    val webFilter = new WebFilter {
      private val dataFileLock = new Object()
      private def updateAndPersist(f: InTxn => Unit) {
        dataFileLock synchronized {
          val (image, userPreferences) = atomic { implicit transaction =>
            f(transaction)
            (memoryImage.get, main.userPreferences.get)
          }
          PersistentDataSerialiser.write(
            PersistentData(filters = image.filters, whitelist = image.whitelist, userPreferences = userPreferences,
              trustAnchorData = image.trustAnchors.all.map(ta => ta.name -> TrustAnchorData(ta.enabled))(collection.breakOut)), dataFile)
        }
      }

      override protected def startTrustAnchorValidation(trustAnchors: Seq[String]) = main.runValidator(trustAnchors)

      override protected def trustAnchors = memoryImage.single.get.trustAnchors
      override protected def validatedObjects = memoryImage.single.get.validatedObjects

      override protected def filters = memoryImage.single.get.filters
      override protected def addFilter(filter: IgnoreFilter) = updateAndPersist { implicit transaction => updateMemoryImage(_.addFilter(filter)) }
      override protected def removeFilter(filter: IgnoreFilter) = updateAndPersist { implicit transaction => updateMemoryImage(_.removeFilter(filter)) }

      override protected def whitelist = memoryImage.single.get.whitelist
      override protected def addWhitelistEntry(entry: RtrPrefix) = updateAndPersist { implicit transaction => updateMemoryImage(_.addWhitelistEntry(entry)) }
      override protected def removeWhitelistEntry(entry: RtrPrefix) = updateAndPersist { implicit transaction => updateMemoryImage(_.removeWhitelistEntry(entry)) }

      override protected def bgpRisDumps = main.bgpRisDumps.single.get
      override protected def validatedAnnouncements = bgpAnnouncementValidator.validatedAnnouncements

      override protected def getRtrPrefixes = memoryImage.single.get.getDistinctRtrPrefixes

      protected def sessionData = rtrServer.rtrSessions.allClientData

      // Software Update checker
      override def newVersionDetailFetcher = new OnlineNewVersionDetailFetcher(ReleaseInfo.version,
        () => {
          val get = new HttpGet("https://lirportal.ripe.net/certification/content/static/validator/latest-version.properties")
          val response = http.execute(get)
          scala.io.Source.fromInputStream(response.getEntity.getContent).mkString
        })

      // UserPreferences
      override def userPreferences = main.userPreferences.single.get
      override def updateUserPreferences(userPreferences: UserPreferences) = updateAndPersist { implicit transaction => main.userPreferences.set(userPreferences) }

      override protected def updateTrustAnchorState(locator: TrustAnchorLocator, enabled: Boolean) = updateAndPersist { implicit transaction =>
        memoryImage.transform(_.updateTrustAnchorState(locator, enabled))
      }
    }

    val restApiServlet = new RestApi() {
      protected def getVrpObjects = memoryImage.single.get.getDistinctRtrPrefixes
    }

    val root = new ServletContextHandler(server, "/", ServletContextHandler.SESSIONS)
    root.setResourceBase(getClass.getResource("/public").toString)
    val defaultServletHolder = new ServletHolder(new DefaultServlet())
    defaultServletHolder.setName("default")
    defaultServletHolder.setInitParameter("dirAllowed", "false")
    root.addServlet(defaultServletHolder, "/*")
    root.addServlet(new ServletHolder(restApiServlet), "/api/*")
    root.addServlet(new ServletHolder(new HealthCheckServlet(HealthChecks.registry)), "/health")
    root.addFilter(new FilterHolder(webFilter), "/*", EnumSet.allOf(classOf[DispatcherType]))

    val requestLogHandler = {
      val handler = new RequestLogHandler()
      val requestLog = new NCSARequestLog(ApplicationOptions.accessLogFileName)
      requestLog.setRetainDays(90)
      requestLog.setAppend(true)
      requestLog.setExtended(false)
      requestLog.setLogLatency(true)
      handler.setRequestLog(requestLog)
      handler
    }

    val handlers = new HandlerCollection()
    handlers.addHandler(root)
    handlers.addHandler(requestLogHandler)
    server.setHandler(handlers)
    server
  }

  private def wipeRsyncDiskCache() {
    val diskCache = new File(ApplicationOptions.rsyncDirLocation)
    if (diskCache.isDirectory) {
      FileUtils.cleanDirectory(diskCache)
    }
  }

}
