/*
 * Decompiled with CFR 0.152.
 */
package net.ripe.rpki.commons.ta.serializers;

import com.google.common.base.Charsets;
import com.google.common.io.Files;
import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.nio.charset.Charset;
import java.util.Base64;
import java.util.List;
import java.util.UUID;
import javax.security.auth.x500.X500Principal;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;
import net.ripe.rpki.commons.crypto.x509cert.X509CertificateInformationAccessDescriptor;
import net.ripe.rpki.commons.ta.domain.request.ResourceCertificateRequestData;
import net.ripe.rpki.commons.ta.domain.request.RevocationRequest;
import net.ripe.rpki.commons.ta.domain.request.SigningRequest;
import net.ripe.rpki.commons.ta.domain.request.TaRequest;
import net.ripe.rpki.commons.ta.domain.request.TrustAnchorRequest;
import net.ripe.rpki.commons.ta.serializers.TrustAnchorRequestSerializer;
import net.ripe.rpki.commons.ta.serializers.Utils;
import net.ripe.rpki.commons.util.XML;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

public class TrustAnchorRequestSerializerTest {
    private static final String TA_REQUEST_PATH = "src/test/resources/ta/ta-request.xml";
    private static final String LEGACY_TA_REQUEST_PATH = "src/test/resources/ta/legacy-ta-request.xml";
    private static final String TA_REQUEST_NO_TA_URI_PATH = "src/test/resources/ta/ta-request-without-ta-publication-uri.xml";
    public static final Base64.Encoder BASE64_ENCODER = Base64.getMimeEncoder(10000, "\n".getBytes());
    private Document document;
    private final XPath xpath = XPathFactory.newInstance().newXPath();
    private TrustAnchorRequest request;
    private final String signingRequest = "<requests.TrustAnchorRequest>\n  <creationTimestamp>1558700883582</creationTimestamp>\n  <taCertificatePublicationUri>rsync://localhost:10873/ta/</taCertificatePublicationUri>\n  <taRequests>\n    <requests.SigningRequest>\n      <requestId>4ee2e78c-f746-426b-bf8b-c37e0155ca3e</requestId>\n      <resourceCertificateRequest>\n        <resourceClassName>DEFAULT</resourceClassName>\n        <subjectDN>CN=8ecc2cdf3247ef43295ebafca8c711ffd51de071</subjectDN>\n        <subjectInformationAccess>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.5</method>\n            <location>rsync://localhost/online/aca/</location>\n          </X509CertificateInformationAccessDescriptor>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.10</method>\n            <location>rsync://localhost/online/aca/jsws3zJH70MpXrr8qMcR_9Ud4HE.mft</location>\n          </X509CertificateInformationAccessDescriptor>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.13</method>\n            <location>http://localhost:7788/notification.xml</location>\n          </X509CertificateInformationAccessDescriptor>\n        </subjectInformationAccess>\n        <ipResourceSet>10.0.0.0/8, 11.0.0.0/8</ipResourceSet>\n        <encodedSubjectPublicKey>MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAtZC7nbyxIqHdncRCXV6wBtBfXtMjuz0TQLd20Hunnr/982wFMqRfsBqEI4+Q/KnPV+N1rsKGhTrAzOCnISDFO5d111qOrWWd/X0T3AjoBLu2yFwtsc+2PYXxM7aAwPl1YfBsmvDjc+BlZEmPgIVLTbkYW2dXaOKVWi5CHpcbHuzox3stStSF9C2CT49N7URwL5qQ7f55BA4kQ1U1grnQR9nbFWT0HjiVIeZow+9ofRD6Io/T6+sMS2LWb3E+YMK6DCdStlYwmZEu+2HpqBjRqB7/3nfO74djpnUXLMzSFIv4x95ZFAeV0GTvLbflfTRd9G9Wa5CF5hd9zrj5OMNwAwIDAQAB</encodedSubjectPublicKey>\n      </resourceCertificateRequest>\n    </requests.SigningRequest>\n  </taRequests>\n  <siaDescriptors>\n    <X509CertificateInformationAccessDescriptor>\n      <method>1.3.6.1.5.5.7.48.13</method>\n      <location>http://localhost:7788/notification.xml</location>\n    </X509CertificateInformationAccessDescriptor>\n    <X509CertificateInformationAccessDescriptor>\n      <method>1.3.6.1.5.5.7.48.5</method>\n      <location>rsync://localhost/online/</location>\n    </X509CertificateInformationAccessDescriptor>\n  </siaDescriptors>\n</requests.TrustAnchorRequest>";
    private final String revocationRequest = "<requests.TrustAnchorRequest>\n  <creationTimestamp>1610359575105</creationTimestamp>\n  <taCertificatePublicationUri>rsync://localhost:10873/ta/</taCertificatePublicationUri>\n  <taRequests>\n    <requests.RevocationRequest>\n      <requestId>3ced3f70-a2b4-42d4-9e46-2fe4cac6b4bf</requestId>\n      <resourceClassName>DEFAULT</resourceClassName>\n      <encodedPublicKey>MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAtZC7nbyxIqHdncRCXV6wBtBfXtMjuz0TQLd20Hunnr/982wFMqRfsBqEI4+Q/KnPV+N1rsKGhTrAzOCnISDFO5d111qOrWWd/X0T3AjoBLu2yFwtsc+2PYXxM7aAwPl1YfBsmvDjc+BlZEmPgIVLTbkYW2dXaOKVWi5CHpcbHuzox3stStSF9C2CT49N7URwL5qQ7f55BA4kQ1U1grnQR9nbFWT0HjiVIeZow+9ofRD6Io/T6+sMS2LWb3E+YMK6DCdStlYwmZEu+2HpqBjRqB7/3nfO74djpnUXLMzSFIv4x95ZFAeV0GTvLbflfTRd9G9Wa5CF5hd9zrj5OMNwAwIDAQAB</encodedPublicKey>\n    </requests.RevocationRequest>\n  </taRequests>\n  <siaDescriptors/>\n</requests.TrustAnchorRequest>";

    @Before
    public void loadState() throws IOException, SAXException, ParserConfigurationException {
        String stateXML = Files.asCharSource((File)new File(TA_REQUEST_PATH), (Charset)Charsets.UTF_8).read();
        TrustAnchorRequestSerializer trustAnchorRequestSerializer = new TrustAnchorRequestSerializer();
        this.request = trustAnchorRequestSerializer.deserialize(stateXML);
        DocumentBuilder builder = XML.newNamespaceAwareDocumentBuilder();
        this.document = builder.parse(new File(TA_REQUEST_PATH));
    }

    private String xpathQuery(String query) throws XPathExpressionException {
        return this.xpath.evaluate(query, this.document);
    }

    @Test
    public void shouldReadBasicFields() throws XPathExpressionException {
        Assert.assertEquals((Object)Long.valueOf(this.xpathQuery("/requests.TrustAnchorRequest/creationTimestamp")), (Object)this.request.getCreationTimestamp());
        Assert.assertEquals((Object)URI.create(this.xpathQuery("/requests.TrustAnchorRequest/taCertificatePublicationUri")), (Object)this.request.getTaCertificatePublicationUri());
    }

    private void validateX509CertificateInformationAccessDescriptor(X509CertificateInformationAccessDescriptor ciad, Node node) throws XPathExpressionException {
        Assert.assertEquals((Object)this.xpath.evaluate("method", node), (Object)ciad.getMethod().toString());
        Assert.assertEquals((Object)URI.create(this.xpath.evaluate("location", node)), (Object)ciad.getLocation());
    }

    private void shouldReadSigningRequest(SigningRequest sr, Node cur) throws XPathExpressionException {
        Assert.assertEquals((Object)UUID.fromString(this.xpath.evaluate("requestId", cur)), (Object)sr.getRequestId());
        ResourceCertificateRequestData rcrd = sr.getResourceCertificateRequest();
        Assert.assertEquals((Object)Utils.cleanupBase64(this.xpath.evaluate("resourceCertificateRequest/encodedSubjectPublicKey", cur)), (Object)org.bouncycastle.util.encoders.Base64.toBase64String((byte[])rcrd.getEncodedSubjectPublicKey()));
        Assert.assertEquals((Object)this.xpath.evaluate("resourceCertificateRequest/resourceClassName", cur), (Object)rcrd.getResourceClassName());
        Assert.assertEquals((Object)this.xpath.evaluate("resourceCertificateRequest/subjectDN", cur), (Object)rcrd.getSubjectDN().getName());
        X509CertificateInformationAccessDescriptor[] sia = rcrd.getSubjectInformationAccess();
        NodeList siaList = (NodeList)this.xpath.evaluate("resourceCertificateRequest/subjectInformationAccess/X509CertificateInformationAccessDescriptor", cur, XPathConstants.NODESET);
        Assert.assertEquals((long)siaList.getLength(), (long)sia.length);
        for (int j = 0; j < siaList.getLength(); ++j) {
            this.validateX509CertificateInformationAccessDescriptor(sia[j], siaList.item(j));
        }
    }

    @Test
    public void shouldReadTARequests() throws XPathExpressionException {
        List signingRequests = this.request.getTaRequests();
        XPath xpath = XPathFactory.newInstance().newXPath();
        NodeList list = (NodeList)xpath.evaluate("/requests.TrustAnchorRequest/taRequests/requests.SigningRequest", this.document, XPathConstants.NODESET);
        Assert.assertEquals((long)list.getLength(), (long)signingRequests.size());
        for (int i = 0; i < list.getLength(); ++i) {
            this.shouldReadSigningRequest((SigningRequest)signingRequests.get(i), list.item(i));
        }
    }

    @Test
    public void shouldReadSiaDescriptors() throws XPathExpressionException {
        X509CertificateInformationAccessDescriptor[] siaDescriptors = this.request.getSiaDescriptors();
        XPath xpath = XPathFactory.newInstance().newXPath();
        NodeList list = (NodeList)xpath.evaluate("/requests.TrustAnchorRequest/siaDescriptors/X509CertificateInformationAccessDescriptor", this.document, XPathConstants.NODESET);
        Assert.assertEquals((long)list.getLength(), (long)siaDescriptors.length);
        for (int i = 0; i < list.getLength(); ++i) {
            this.validateX509CertificateInformationAccessDescriptor(siaDescriptors[i], list.item(i));
        }
    }

    @Test
    public void itShouldDeserializeSigningRequestTaCertificatePublicationUri() {
        TrustAnchorRequest taRequest = new TrustAnchorRequestSerializer().deserialize("<requests.TrustAnchorRequest>\n  <creationTimestamp>1558700883582</creationTimestamp>\n  <taCertificatePublicationUri>rsync://localhost:10873/ta/</taCertificatePublicationUri>\n  <taRequests>\n    <requests.SigningRequest>\n      <requestId>4ee2e78c-f746-426b-bf8b-c37e0155ca3e</requestId>\n      <resourceCertificateRequest>\n        <resourceClassName>DEFAULT</resourceClassName>\n        <subjectDN>CN=8ecc2cdf3247ef43295ebafca8c711ffd51de071</subjectDN>\n        <subjectInformationAccess>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.5</method>\n            <location>rsync://localhost/online/aca/</location>\n          </X509CertificateInformationAccessDescriptor>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.10</method>\n            <location>rsync://localhost/online/aca/jsws3zJH70MpXrr8qMcR_9Ud4HE.mft</location>\n          </X509CertificateInformationAccessDescriptor>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.13</method>\n            <location>http://localhost:7788/notification.xml</location>\n          </X509CertificateInformationAccessDescriptor>\n        </subjectInformationAccess>\n        <ipResourceSet>10.0.0.0/8, 11.0.0.0/8</ipResourceSet>\n        <encodedSubjectPublicKey>MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAtZC7nbyxIqHdncRCXV6wBtBfXtMjuz0TQLd20Hunnr/982wFMqRfsBqEI4+Q/KnPV+N1rsKGhTrAzOCnISDFO5d111qOrWWd/X0T3AjoBLu2yFwtsc+2PYXxM7aAwPl1YfBsmvDjc+BlZEmPgIVLTbkYW2dXaOKVWi5CHpcbHuzox3stStSF9C2CT49N7URwL5qQ7f55BA4kQ1U1grnQR9nbFWT0HjiVIeZow+9ofRD6Io/T6+sMS2LWb3E+YMK6DCdStlYwmZEu+2HpqBjRqB7/3nfO74djpnUXLMzSFIv4x95ZFAeV0GTvLbflfTRd9G9Wa5CF5hd9zrj5OMNwAwIDAQAB</encodedSubjectPublicKey>\n      </resourceCertificateRequest>\n    </requests.SigningRequest>\n  </taRequests>\n  <siaDescriptors>\n    <X509CertificateInformationAccessDescriptor>\n      <method>1.3.6.1.5.5.7.48.13</method>\n      <location>http://localhost:7788/notification.xml</location>\n    </X509CertificateInformationAccessDescriptor>\n    <X509CertificateInformationAccessDescriptor>\n      <method>1.3.6.1.5.5.7.48.5</method>\n      <location>rsync://localhost/online/</location>\n    </X509CertificateInformationAccessDescriptor>\n  </siaDescriptors>\n</requests.TrustAnchorRequest>");
        Assert.assertEquals((Object)"rsync://localhost:10873/ta/", (Object)taRequest.getTaCertificatePublicationUri().toString());
    }

    @Test
    public void itShouldDeserializeSigningRequestCreationTimestamp() {
        TrustAnchorRequest taRequest = new TrustAnchorRequestSerializer().deserialize("<requests.TrustAnchorRequest>\n  <creationTimestamp>1558700883582</creationTimestamp>\n  <taCertificatePublicationUri>rsync://localhost:10873/ta/</taCertificatePublicationUri>\n  <taRequests>\n    <requests.SigningRequest>\n      <requestId>4ee2e78c-f746-426b-bf8b-c37e0155ca3e</requestId>\n      <resourceCertificateRequest>\n        <resourceClassName>DEFAULT</resourceClassName>\n        <subjectDN>CN=8ecc2cdf3247ef43295ebafca8c711ffd51de071</subjectDN>\n        <subjectInformationAccess>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.5</method>\n            <location>rsync://localhost/online/aca/</location>\n          </X509CertificateInformationAccessDescriptor>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.10</method>\n            <location>rsync://localhost/online/aca/jsws3zJH70MpXrr8qMcR_9Ud4HE.mft</location>\n          </X509CertificateInformationAccessDescriptor>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.13</method>\n            <location>http://localhost:7788/notification.xml</location>\n          </X509CertificateInformationAccessDescriptor>\n        </subjectInformationAccess>\n        <ipResourceSet>10.0.0.0/8, 11.0.0.0/8</ipResourceSet>\n        <encodedSubjectPublicKey>MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAtZC7nbyxIqHdncRCXV6wBtBfXtMjuz0TQLd20Hunnr/982wFMqRfsBqEI4+Q/KnPV+N1rsKGhTrAzOCnISDFO5d111qOrWWd/X0T3AjoBLu2yFwtsc+2PYXxM7aAwPl1YfBsmvDjc+BlZEmPgIVLTbkYW2dXaOKVWi5CHpcbHuzox3stStSF9C2CT49N7URwL5qQ7f55BA4kQ1U1grnQR9nbFWT0HjiVIeZow+9ofRD6Io/T6+sMS2LWb3E+YMK6DCdStlYwmZEu+2HpqBjRqB7/3nfO74djpnUXLMzSFIv4x95ZFAeV0GTvLbflfTRd9G9Wa5CF5hd9zrj5OMNwAwIDAQAB</encodedSubjectPublicKey>\n      </resourceCertificateRequest>\n    </requests.SigningRequest>\n  </taRequests>\n  <siaDescriptors>\n    <X509CertificateInformationAccessDescriptor>\n      <method>1.3.6.1.5.5.7.48.13</method>\n      <location>http://localhost:7788/notification.xml</location>\n    </X509CertificateInformationAccessDescriptor>\n    <X509CertificateInformationAccessDescriptor>\n      <method>1.3.6.1.5.5.7.48.5</method>\n      <location>rsync://localhost/online/</location>\n    </X509CertificateInformationAccessDescriptor>\n  </siaDescriptors>\n</requests.TrustAnchorRequest>");
        Assert.assertEquals((long)Long.parseLong("1558700883582"), (long)taRequest.getCreationTimestamp());
    }

    @Test
    public void itShouldDeserializeSigningRequestTaSigningRequests() {
        TrustAnchorRequest taRequest = new TrustAnchorRequestSerializer().deserialize("<requests.TrustAnchorRequest>\n  <creationTimestamp>1558700883582</creationTimestamp>\n  <taCertificatePublicationUri>rsync://localhost:10873/ta/</taCertificatePublicationUri>\n  <taRequests>\n    <requests.SigningRequest>\n      <requestId>4ee2e78c-f746-426b-bf8b-c37e0155ca3e</requestId>\n      <resourceCertificateRequest>\n        <resourceClassName>DEFAULT</resourceClassName>\n        <subjectDN>CN=8ecc2cdf3247ef43295ebafca8c711ffd51de071</subjectDN>\n        <subjectInformationAccess>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.5</method>\n            <location>rsync://localhost/online/aca/</location>\n          </X509CertificateInformationAccessDescriptor>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.10</method>\n            <location>rsync://localhost/online/aca/jsws3zJH70MpXrr8qMcR_9Ud4HE.mft</location>\n          </X509CertificateInformationAccessDescriptor>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.13</method>\n            <location>http://localhost:7788/notification.xml</location>\n          </X509CertificateInformationAccessDescriptor>\n        </subjectInformationAccess>\n        <ipResourceSet>10.0.0.0/8, 11.0.0.0/8</ipResourceSet>\n        <encodedSubjectPublicKey>MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAtZC7nbyxIqHdncRCXV6wBtBfXtMjuz0TQLd20Hunnr/982wFMqRfsBqEI4+Q/KnPV+N1rsKGhTrAzOCnISDFO5d111qOrWWd/X0T3AjoBLu2yFwtsc+2PYXxM7aAwPl1YfBsmvDjc+BlZEmPgIVLTbkYW2dXaOKVWi5CHpcbHuzox3stStSF9C2CT49N7URwL5qQ7f55BA4kQ1U1grnQR9nbFWT0HjiVIeZow+9ofRD6Io/T6+sMS2LWb3E+YMK6DCdStlYwmZEu+2HpqBjRqB7/3nfO74djpnUXLMzSFIv4x95ZFAeV0GTvLbflfTRd9G9Wa5CF5hd9zrj5OMNwAwIDAQAB</encodedSubjectPublicKey>\n      </resourceCertificateRequest>\n    </requests.SigningRequest>\n  </taRequests>\n  <siaDescriptors>\n    <X509CertificateInformationAccessDescriptor>\n      <method>1.3.6.1.5.5.7.48.13</method>\n      <location>http://localhost:7788/notification.xml</location>\n    </X509CertificateInformationAccessDescriptor>\n    <X509CertificateInformationAccessDescriptor>\n      <method>1.3.6.1.5.5.7.48.5</method>\n      <location>rsync://localhost/online/</location>\n    </X509CertificateInformationAccessDescriptor>\n  </siaDescriptors>\n</requests.TrustAnchorRequest>");
        Assert.assertEquals((long)1L, (long)taRequest.getTaRequests().size());
        TaRequest signingRequest = (TaRequest)taRequest.getTaRequests().get(0);
        Assert.assertEquals((Object)UUID.fromString("4ee2e78c-f746-426b-bf8b-c37e0155ca3e"), (Object)signingRequest.getRequestId());
    }

    @Test
    public void itShouldDeserializeSigningRequestResourceCertificateRequestInSigningRequest() {
        TrustAnchorRequest taRequest = new TrustAnchorRequestSerializer().deserialize("<requests.TrustAnchorRequest>\n  <creationTimestamp>1558700883582</creationTimestamp>\n  <taCertificatePublicationUri>rsync://localhost:10873/ta/</taCertificatePublicationUri>\n  <taRequests>\n    <requests.SigningRequest>\n      <requestId>4ee2e78c-f746-426b-bf8b-c37e0155ca3e</requestId>\n      <resourceCertificateRequest>\n        <resourceClassName>DEFAULT</resourceClassName>\n        <subjectDN>CN=8ecc2cdf3247ef43295ebafca8c711ffd51de071</subjectDN>\n        <subjectInformationAccess>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.5</method>\n            <location>rsync://localhost/online/aca/</location>\n          </X509CertificateInformationAccessDescriptor>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.10</method>\n            <location>rsync://localhost/online/aca/jsws3zJH70MpXrr8qMcR_9Ud4HE.mft</location>\n          </X509CertificateInformationAccessDescriptor>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.13</method>\n            <location>http://localhost:7788/notification.xml</location>\n          </X509CertificateInformationAccessDescriptor>\n        </subjectInformationAccess>\n        <ipResourceSet>10.0.0.0/8, 11.0.0.0/8</ipResourceSet>\n        <encodedSubjectPublicKey>MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAtZC7nbyxIqHdncRCXV6wBtBfXtMjuz0TQLd20Hunnr/982wFMqRfsBqEI4+Q/KnPV+N1rsKGhTrAzOCnISDFO5d111qOrWWd/X0T3AjoBLu2yFwtsc+2PYXxM7aAwPl1YfBsmvDjc+BlZEmPgIVLTbkYW2dXaOKVWi5CHpcbHuzox3stStSF9C2CT49N7URwL5qQ7f55BA4kQ1U1grnQR9nbFWT0HjiVIeZow+9ofRD6Io/T6+sMS2LWb3E+YMK6DCdStlYwmZEu+2HpqBjRqB7/3nfO74djpnUXLMzSFIv4x95ZFAeV0GTvLbflfTRd9G9Wa5CF5hd9zrj5OMNwAwIDAQAB</encodedSubjectPublicKey>\n      </resourceCertificateRequest>\n    </requests.SigningRequest>\n  </taRequests>\n  <siaDescriptors>\n    <X509CertificateInformationAccessDescriptor>\n      <method>1.3.6.1.5.5.7.48.13</method>\n      <location>http://localhost:7788/notification.xml</location>\n    </X509CertificateInformationAccessDescriptor>\n    <X509CertificateInformationAccessDescriptor>\n      <method>1.3.6.1.5.5.7.48.5</method>\n      <location>rsync://localhost/online/</location>\n    </X509CertificateInformationAccessDescriptor>\n  </siaDescriptors>\n</requests.TrustAnchorRequest>");
        SigningRequest signingRequest = (SigningRequest)taRequest.getTaRequests().get(0);
        ResourceCertificateRequestData resourceCertificateRequest = signingRequest.getResourceCertificateRequest();
        Assert.assertEquals((Object)"DEFAULT", (Object)resourceCertificateRequest.getResourceClassName());
        Assert.assertEquals((Object)new X500Principal("CN=8ecc2cdf3247ef43295ebafca8c711ffd51de071"), (Object)resourceCertificateRequest.getSubjectDN());
        String subjectPublicKey = "MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAtZC7nbyxIqHdncRCXV6wBtBfXtMjuz0TQLd20Hunnr/982wFMqRfsBqEI4+Q/KnPV+N1rsKGhTrAzOCnISDFO5d111qOrWWd/X0T3AjoBLu2yFwtsc+2PYXxM7aAwPl1YfBsmvDjc+BlZEmPgIVLTbkYW2dXaOKVWi5CHpcbHuzox3stStSF9C2CT49N7URwL5qQ7f55BA4kQ1U1grnQR9nbFWT0HjiVIeZow+9ofRD6Io/T6+sMS2LWb3E+YMK6DCdStlYwmZEu+2HpqBjRqB7/3nfO74djpnUXLMzSFIv4x95ZFAeV0GTvLbflfTRd9G9Wa5CF5hd9zrj5OMNwAwIDAQAB";
        Assert.assertEquals((Object)"MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAtZC7nbyxIqHdncRCXV6wBtBfXtMjuz0TQLd20Hunnr/982wFMqRfsBqEI4+Q/KnPV+N1rsKGhTrAzOCnISDFO5d111qOrWWd/X0T3AjoBLu2yFwtsc+2PYXxM7aAwPl1YfBsmvDjc+BlZEmPgIVLTbkYW2dXaOKVWi5CHpcbHuzox3stStSF9C2CT49N7URwL5qQ7f55BA4kQ1U1grnQR9nbFWT0HjiVIeZow+9ofRD6Io/T6+sMS2LWb3E+YMK6DCdStlYwmZEu+2HpqBjRqB7/3nfO74djpnUXLMzSFIv4x95ZFAeV0GTvLbflfTRd9G9Wa5CF5hd9zrj5OMNwAwIDAQAB", (Object)BASE64_ENCODER.encodeToString(resourceCertificateRequest.getEncodedSubjectPublicKey()));
    }

    @Test
    public void itShouldDeserializeSigningRequestSubjectInformationAccessDescriptorsInResourceCertificateRequest() {
        TrustAnchorRequest taRequest = new TrustAnchorRequestSerializer().deserialize("<requests.TrustAnchorRequest>\n  <creationTimestamp>1558700883582</creationTimestamp>\n  <taCertificatePublicationUri>rsync://localhost:10873/ta/</taCertificatePublicationUri>\n  <taRequests>\n    <requests.SigningRequest>\n      <requestId>4ee2e78c-f746-426b-bf8b-c37e0155ca3e</requestId>\n      <resourceCertificateRequest>\n        <resourceClassName>DEFAULT</resourceClassName>\n        <subjectDN>CN=8ecc2cdf3247ef43295ebafca8c711ffd51de071</subjectDN>\n        <subjectInformationAccess>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.5</method>\n            <location>rsync://localhost/online/aca/</location>\n          </X509CertificateInformationAccessDescriptor>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.10</method>\n            <location>rsync://localhost/online/aca/jsws3zJH70MpXrr8qMcR_9Ud4HE.mft</location>\n          </X509CertificateInformationAccessDescriptor>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.13</method>\n            <location>http://localhost:7788/notification.xml</location>\n          </X509CertificateInformationAccessDescriptor>\n        </subjectInformationAccess>\n        <ipResourceSet>10.0.0.0/8, 11.0.0.0/8</ipResourceSet>\n        <encodedSubjectPublicKey>MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAtZC7nbyxIqHdncRCXV6wBtBfXtMjuz0TQLd20Hunnr/982wFMqRfsBqEI4+Q/KnPV+N1rsKGhTrAzOCnISDFO5d111qOrWWd/X0T3AjoBLu2yFwtsc+2PYXxM7aAwPl1YfBsmvDjc+BlZEmPgIVLTbkYW2dXaOKVWi5CHpcbHuzox3stStSF9C2CT49N7URwL5qQ7f55BA4kQ1U1grnQR9nbFWT0HjiVIeZow+9ofRD6Io/T6+sMS2LWb3E+YMK6DCdStlYwmZEu+2HpqBjRqB7/3nfO74djpnUXLMzSFIv4x95ZFAeV0GTvLbflfTRd9G9Wa5CF5hd9zrj5OMNwAwIDAQAB</encodedSubjectPublicKey>\n      </resourceCertificateRequest>\n    </requests.SigningRequest>\n  </taRequests>\n  <siaDescriptors>\n    <X509CertificateInformationAccessDescriptor>\n      <method>1.3.6.1.5.5.7.48.13</method>\n      <location>http://localhost:7788/notification.xml</location>\n    </X509CertificateInformationAccessDescriptor>\n    <X509CertificateInformationAccessDescriptor>\n      <method>1.3.6.1.5.5.7.48.5</method>\n      <location>rsync://localhost/online/</location>\n    </X509CertificateInformationAccessDescriptor>\n  </siaDescriptors>\n</requests.TrustAnchorRequest>");
        SigningRequest signingRequest = (SigningRequest)taRequest.getTaRequests().get(0);
        ResourceCertificateRequestData resourceCertificateRequest = signingRequest.getResourceCertificateRequest();
        X509CertificateInformationAccessDescriptor[] subjectInformationAccessDescriptors = resourceCertificateRequest.getSubjectInformationAccess();
        Assert.assertEquals((long)3L, (long)subjectInformationAccessDescriptors.length);
        X509CertificateInformationAccessDescriptor subjectInformationAccess = subjectInformationAccessDescriptors[0];
        Assert.assertEquals((Object)new ASN1ObjectIdentifier("1.3.6.1.5.5.7.48.5"), (Object)subjectInformationAccess.getMethod());
        Assert.assertEquals((Object)URI.create("rsync://localhost/online/aca/"), (Object)subjectInformationAccess.getLocation());
    }

    @Test
    public void itShouldDeserializeSigningRequestSubjectInformationAccessDescriptorsInTrustAnchorRequest() {
        TrustAnchorRequest taRequest = new TrustAnchorRequestSerializer().deserialize("<requests.TrustAnchorRequest>\n  <creationTimestamp>1558700883582</creationTimestamp>\n  <taCertificatePublicationUri>rsync://localhost:10873/ta/</taCertificatePublicationUri>\n  <taRequests>\n    <requests.SigningRequest>\n      <requestId>4ee2e78c-f746-426b-bf8b-c37e0155ca3e</requestId>\n      <resourceCertificateRequest>\n        <resourceClassName>DEFAULT</resourceClassName>\n        <subjectDN>CN=8ecc2cdf3247ef43295ebafca8c711ffd51de071</subjectDN>\n        <subjectInformationAccess>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.5</method>\n            <location>rsync://localhost/online/aca/</location>\n          </X509CertificateInformationAccessDescriptor>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.10</method>\n            <location>rsync://localhost/online/aca/jsws3zJH70MpXrr8qMcR_9Ud4HE.mft</location>\n          </X509CertificateInformationAccessDescriptor>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.13</method>\n            <location>http://localhost:7788/notification.xml</location>\n          </X509CertificateInformationAccessDescriptor>\n        </subjectInformationAccess>\n        <ipResourceSet>10.0.0.0/8, 11.0.0.0/8</ipResourceSet>\n        <encodedSubjectPublicKey>MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAtZC7nbyxIqHdncRCXV6wBtBfXtMjuz0TQLd20Hunnr/982wFMqRfsBqEI4+Q/KnPV+N1rsKGhTrAzOCnISDFO5d111qOrWWd/X0T3AjoBLu2yFwtsc+2PYXxM7aAwPl1YfBsmvDjc+BlZEmPgIVLTbkYW2dXaOKVWi5CHpcbHuzox3stStSF9C2CT49N7URwL5qQ7f55BA4kQ1U1grnQR9nbFWT0HjiVIeZow+9ofRD6Io/T6+sMS2LWb3E+YMK6DCdStlYwmZEu+2HpqBjRqB7/3nfO74djpnUXLMzSFIv4x95ZFAeV0GTvLbflfTRd9G9Wa5CF5hd9zrj5OMNwAwIDAQAB</encodedSubjectPublicKey>\n      </resourceCertificateRequest>\n    </requests.SigningRequest>\n  </taRequests>\n  <siaDescriptors>\n    <X509CertificateInformationAccessDescriptor>\n      <method>1.3.6.1.5.5.7.48.13</method>\n      <location>http://localhost:7788/notification.xml</location>\n    </X509CertificateInformationAccessDescriptor>\n    <X509CertificateInformationAccessDescriptor>\n      <method>1.3.6.1.5.5.7.48.5</method>\n      <location>rsync://localhost/online/</location>\n    </X509CertificateInformationAccessDescriptor>\n  </siaDescriptors>\n</requests.TrustAnchorRequest>");
        X509CertificateInformationAccessDescriptor[] subjectInformationAccessDescriptors = taRequest.getSiaDescriptors();
        Assert.assertEquals((long)2L, (long)subjectInformationAccessDescriptors.length);
        X509CertificateInformationAccessDescriptor subjectInformationAccess = subjectInformationAccessDescriptors[0];
        Assert.assertEquals((Object)new ASN1ObjectIdentifier("1.3.6.1.5.5.7.48.13"), (Object)subjectInformationAccess.getMethod());
        Assert.assertEquals((Object)URI.create("http://localhost:7788/notification.xml"), (Object)subjectInformationAccess.getLocation());
    }

    @Test
    public void itShouldDeserializeRevocationRequestTaCertificatePublicationUri() {
        TrustAnchorRequest taRequest = new TrustAnchorRequestSerializer().deserialize("<requests.TrustAnchorRequest>\n  <creationTimestamp>1610359575105</creationTimestamp>\n  <taCertificatePublicationUri>rsync://localhost:10873/ta/</taCertificatePublicationUri>\n  <taRequests>\n    <requests.RevocationRequest>\n      <requestId>3ced3f70-a2b4-42d4-9e46-2fe4cac6b4bf</requestId>\n      <resourceClassName>DEFAULT</resourceClassName>\n      <encodedPublicKey>MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAtZC7nbyxIqHdncRCXV6wBtBfXtMjuz0TQLd20Hunnr/982wFMqRfsBqEI4+Q/KnPV+N1rsKGhTrAzOCnISDFO5d111qOrWWd/X0T3AjoBLu2yFwtsc+2PYXxM7aAwPl1YfBsmvDjc+BlZEmPgIVLTbkYW2dXaOKVWi5CHpcbHuzox3stStSF9C2CT49N7URwL5qQ7f55BA4kQ1U1grnQR9nbFWT0HjiVIeZow+9ofRD6Io/T6+sMS2LWb3E+YMK6DCdStlYwmZEu+2HpqBjRqB7/3nfO74djpnUXLMzSFIv4x95ZFAeV0GTvLbflfTRd9G9Wa5CF5hd9zrj5OMNwAwIDAQAB</encodedPublicKey>\n    </requests.RevocationRequest>\n  </taRequests>\n  <siaDescriptors/>\n</requests.TrustAnchorRequest>");
        Assert.assertEquals((Object)"rsync://localhost:10873/ta/", (Object)taRequest.getTaCertificatePublicationUri().toString());
    }

    @Test
    public void itShouldDeserializeRevocationRequestCreationTimestamp() {
        TrustAnchorRequest taRequest = new TrustAnchorRequestSerializer().deserialize("<requests.TrustAnchorRequest>\n  <creationTimestamp>1610359575105</creationTimestamp>\n  <taCertificatePublicationUri>rsync://localhost:10873/ta/</taCertificatePublicationUri>\n  <taRequests>\n    <requests.RevocationRequest>\n      <requestId>3ced3f70-a2b4-42d4-9e46-2fe4cac6b4bf</requestId>\n      <resourceClassName>DEFAULT</resourceClassName>\n      <encodedPublicKey>MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAtZC7nbyxIqHdncRCXV6wBtBfXtMjuz0TQLd20Hunnr/982wFMqRfsBqEI4+Q/KnPV+N1rsKGhTrAzOCnISDFO5d111qOrWWd/X0T3AjoBLu2yFwtsc+2PYXxM7aAwPl1YfBsmvDjc+BlZEmPgIVLTbkYW2dXaOKVWi5CHpcbHuzox3stStSF9C2CT49N7URwL5qQ7f55BA4kQ1U1grnQR9nbFWT0HjiVIeZow+9ofRD6Io/T6+sMS2LWb3E+YMK6DCdStlYwmZEu+2HpqBjRqB7/3nfO74djpnUXLMzSFIv4x95ZFAeV0GTvLbflfTRd9G9Wa5CF5hd9zrj5OMNwAwIDAQAB</encodedPublicKey>\n    </requests.RevocationRequest>\n  </taRequests>\n  <siaDescriptors/>\n</requests.TrustAnchorRequest>");
        Assert.assertEquals((long)Long.parseLong("1610359575105"), (long)taRequest.getCreationTimestamp());
    }

    @Test
    public void itShouldDeserializeRevocationRequestTaSigningRequests() {
        TrustAnchorRequest taRequest = new TrustAnchorRequestSerializer().deserialize("<requests.TrustAnchorRequest>\n  <creationTimestamp>1610359575105</creationTimestamp>\n  <taCertificatePublicationUri>rsync://localhost:10873/ta/</taCertificatePublicationUri>\n  <taRequests>\n    <requests.RevocationRequest>\n      <requestId>3ced3f70-a2b4-42d4-9e46-2fe4cac6b4bf</requestId>\n      <resourceClassName>DEFAULT</resourceClassName>\n      <encodedPublicKey>MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAtZC7nbyxIqHdncRCXV6wBtBfXtMjuz0TQLd20Hunnr/982wFMqRfsBqEI4+Q/KnPV+N1rsKGhTrAzOCnISDFO5d111qOrWWd/X0T3AjoBLu2yFwtsc+2PYXxM7aAwPl1YfBsmvDjc+BlZEmPgIVLTbkYW2dXaOKVWi5CHpcbHuzox3stStSF9C2CT49N7URwL5qQ7f55BA4kQ1U1grnQR9nbFWT0HjiVIeZow+9ofRD6Io/T6+sMS2LWb3E+YMK6DCdStlYwmZEu+2HpqBjRqB7/3nfO74djpnUXLMzSFIv4x95ZFAeV0GTvLbflfTRd9G9Wa5CF5hd9zrj5OMNwAwIDAQAB</encodedPublicKey>\n    </requests.RevocationRequest>\n  </taRequests>\n  <siaDescriptors/>\n</requests.TrustAnchorRequest>");
        Assert.assertEquals((long)1L, (long)taRequest.getTaRequests().size());
        TaRequest signingRequest = (TaRequest)taRequest.getTaRequests().get(0);
        Assert.assertEquals((Object)UUID.fromString("3ced3f70-a2b4-42d4-9e46-2fe4cac6b4bf"), (Object)signingRequest.getRequestId());
    }

    @Test
    public void itShouldDeserializeRevocationRequestResourceCertificateRequestInSigningRequest() {
        TrustAnchorRequest taRequest = new TrustAnchorRequestSerializer().deserialize("<requests.TrustAnchorRequest>\n  <creationTimestamp>1610359575105</creationTimestamp>\n  <taCertificatePublicationUri>rsync://localhost:10873/ta/</taCertificatePublicationUri>\n  <taRequests>\n    <requests.RevocationRequest>\n      <requestId>3ced3f70-a2b4-42d4-9e46-2fe4cac6b4bf</requestId>\n      <resourceClassName>DEFAULT</resourceClassName>\n      <encodedPublicKey>MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAtZC7nbyxIqHdncRCXV6wBtBfXtMjuz0TQLd20Hunnr/982wFMqRfsBqEI4+Q/KnPV+N1rsKGhTrAzOCnISDFO5d111qOrWWd/X0T3AjoBLu2yFwtsc+2PYXxM7aAwPl1YfBsmvDjc+BlZEmPgIVLTbkYW2dXaOKVWi5CHpcbHuzox3stStSF9C2CT49N7URwL5qQ7f55BA4kQ1U1grnQR9nbFWT0HjiVIeZow+9ofRD6Io/T6+sMS2LWb3E+YMK6DCdStlYwmZEu+2HpqBjRqB7/3nfO74djpnUXLMzSFIv4x95ZFAeV0GTvLbflfTRd9G9Wa5CF5hd9zrj5OMNwAwIDAQAB</encodedPublicKey>\n    </requests.RevocationRequest>\n  </taRequests>\n  <siaDescriptors/>\n</requests.TrustAnchorRequest>");
        RevocationRequest revocationRequest = (RevocationRequest)taRequest.getTaRequests().get(0);
        Assert.assertEquals((Object)"DEFAULT", (Object)revocationRequest.getResourceClassName());
        String subjectPublicKey = "MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAtZC7nbyxIqHdncRCXV6wBtBfXtMjuz0TQLd20Hunnr/982wFMqRfsBqEI4+Q/KnPV+N1rsKGhTrAzOCnISDFO5d111qOrWWd/X0T3AjoBLu2yFwtsc+2PYXxM7aAwPl1YfBsmvDjc+BlZEmPgIVLTbkYW2dXaOKVWi5CHpcbHuzox3stStSF9C2CT49N7URwL5qQ7f55BA4kQ1U1grnQR9nbFWT0HjiVIeZow+9ofRD6Io/T6+sMS2LWb3E+YMK6DCdStlYwmZEu+2HpqBjRqB7/3nfO74djpnUXLMzSFIv4x95ZFAeV0GTvLbflfTRd9G9Wa5CF5hd9zrj5OMNwAwIDAQAB";
        Assert.assertEquals((Object)"MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAtZC7nbyxIqHdncRCXV6wBtBfXtMjuz0TQLd20Hunnr/982wFMqRfsBqEI4+Q/KnPV+N1rsKGhTrAzOCnISDFO5d111qOrWWd/X0T3AjoBLu2yFwtsc+2PYXxM7aAwPl1YfBsmvDjc+BlZEmPgIVLTbkYW2dXaOKVWi5CHpcbHuzox3stStSF9C2CT49N7URwL5qQ7f55BA4kQ1U1grnQR9nbFWT0HjiVIeZow+9ofRD6Io/T6+sMS2LWb3E+YMK6DCdStlYwmZEu+2HpqBjRqB7/3nfO74djpnUXLMzSFIv4x95ZFAeV0GTvLbflfTRd9G9Wa5CF5hd9zrj5OMNwAwIDAQAB", (Object)revocationRequest.getEncodedPublicKey());
    }

    @Test
    public void itShouldSerializeSigningRequestTaCertificatePublicationUri() {
        TrustAnchorRequest taRequest0 = new TrustAnchorRequestSerializer().deserialize("<requests.TrustAnchorRequest>\n  <creationTimestamp>1558700883582</creationTimestamp>\n  <taCertificatePublicationUri>rsync://localhost:10873/ta/</taCertificatePublicationUri>\n  <taRequests>\n    <requests.SigningRequest>\n      <requestId>4ee2e78c-f746-426b-bf8b-c37e0155ca3e</requestId>\n      <resourceCertificateRequest>\n        <resourceClassName>DEFAULT</resourceClassName>\n        <subjectDN>CN=8ecc2cdf3247ef43295ebafca8c711ffd51de071</subjectDN>\n        <subjectInformationAccess>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.5</method>\n            <location>rsync://localhost/online/aca/</location>\n          </X509CertificateInformationAccessDescriptor>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.10</method>\n            <location>rsync://localhost/online/aca/jsws3zJH70MpXrr8qMcR_9Ud4HE.mft</location>\n          </X509CertificateInformationAccessDescriptor>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.13</method>\n            <location>http://localhost:7788/notification.xml</location>\n          </X509CertificateInformationAccessDescriptor>\n        </subjectInformationAccess>\n        <ipResourceSet>10.0.0.0/8, 11.0.0.0/8</ipResourceSet>\n        <encodedSubjectPublicKey>MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAtZC7nbyxIqHdncRCXV6wBtBfXtMjuz0TQLd20Hunnr/982wFMqRfsBqEI4+Q/KnPV+N1rsKGhTrAzOCnISDFO5d111qOrWWd/X0T3AjoBLu2yFwtsc+2PYXxM7aAwPl1YfBsmvDjc+BlZEmPgIVLTbkYW2dXaOKVWi5CHpcbHuzox3stStSF9C2CT49N7URwL5qQ7f55BA4kQ1U1grnQR9nbFWT0HjiVIeZow+9ofRD6Io/T6+sMS2LWb3E+YMK6DCdStlYwmZEu+2HpqBjRqB7/3nfO74djpnUXLMzSFIv4x95ZFAeV0GTvLbflfTRd9G9Wa5CF5hd9zrj5OMNwAwIDAQAB</encodedSubjectPublicKey>\n      </resourceCertificateRequest>\n    </requests.SigningRequest>\n  </taRequests>\n  <siaDescriptors>\n    <X509CertificateInformationAccessDescriptor>\n      <method>1.3.6.1.5.5.7.48.13</method>\n      <location>http://localhost:7788/notification.xml</location>\n    </X509CertificateInformationAccessDescriptor>\n    <X509CertificateInformationAccessDescriptor>\n      <method>1.3.6.1.5.5.7.48.5</method>\n      <location>rsync://localhost/online/</location>\n    </X509CertificateInformationAccessDescriptor>\n  </siaDescriptors>\n</requests.TrustAnchorRequest>");
        String request = new TrustAnchorRequestSerializer().serialize(taRequest0);
        TrustAnchorRequest taRequest1 = new TrustAnchorRequestSerializer().deserialize(request);
        Assert.assertEquals((Object)taRequest0.getTaCertificatePublicationUri(), (Object)taRequest1.getTaCertificatePublicationUri());
    }

    @Test
    public void itShouldSerializeSigningRequestCreationTimestamp() {
        TrustAnchorRequest taRequest0 = new TrustAnchorRequestSerializer().deserialize("<requests.TrustAnchorRequest>\n  <creationTimestamp>1558700883582</creationTimestamp>\n  <taCertificatePublicationUri>rsync://localhost:10873/ta/</taCertificatePublicationUri>\n  <taRequests>\n    <requests.SigningRequest>\n      <requestId>4ee2e78c-f746-426b-bf8b-c37e0155ca3e</requestId>\n      <resourceCertificateRequest>\n        <resourceClassName>DEFAULT</resourceClassName>\n        <subjectDN>CN=8ecc2cdf3247ef43295ebafca8c711ffd51de071</subjectDN>\n        <subjectInformationAccess>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.5</method>\n            <location>rsync://localhost/online/aca/</location>\n          </X509CertificateInformationAccessDescriptor>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.10</method>\n            <location>rsync://localhost/online/aca/jsws3zJH70MpXrr8qMcR_9Ud4HE.mft</location>\n          </X509CertificateInformationAccessDescriptor>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.13</method>\n            <location>http://localhost:7788/notification.xml</location>\n          </X509CertificateInformationAccessDescriptor>\n        </subjectInformationAccess>\n        <ipResourceSet>10.0.0.0/8, 11.0.0.0/8</ipResourceSet>\n        <encodedSubjectPublicKey>MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAtZC7nbyxIqHdncRCXV6wBtBfXtMjuz0TQLd20Hunnr/982wFMqRfsBqEI4+Q/KnPV+N1rsKGhTrAzOCnISDFO5d111qOrWWd/X0T3AjoBLu2yFwtsc+2PYXxM7aAwPl1YfBsmvDjc+BlZEmPgIVLTbkYW2dXaOKVWi5CHpcbHuzox3stStSF9C2CT49N7URwL5qQ7f55BA4kQ1U1grnQR9nbFWT0HjiVIeZow+9ofRD6Io/T6+sMS2LWb3E+YMK6DCdStlYwmZEu+2HpqBjRqB7/3nfO74djpnUXLMzSFIv4x95ZFAeV0GTvLbflfTRd9G9Wa5CF5hd9zrj5OMNwAwIDAQAB</encodedSubjectPublicKey>\n      </resourceCertificateRequest>\n    </requests.SigningRequest>\n  </taRequests>\n  <siaDescriptors>\n    <X509CertificateInformationAccessDescriptor>\n      <method>1.3.6.1.5.5.7.48.13</method>\n      <location>http://localhost:7788/notification.xml</location>\n    </X509CertificateInformationAccessDescriptor>\n    <X509CertificateInformationAccessDescriptor>\n      <method>1.3.6.1.5.5.7.48.5</method>\n      <location>rsync://localhost/online/</location>\n    </X509CertificateInformationAccessDescriptor>\n  </siaDescriptors>\n</requests.TrustAnchorRequest>");
        String request = new TrustAnchorRequestSerializer().serialize(taRequest0);
        TrustAnchorRequest taRequest1 = new TrustAnchorRequestSerializer().deserialize(request);
        Assert.assertEquals((Object)taRequest0.getCreationTimestamp(), (Object)taRequest1.getCreationTimestamp());
    }

    @Test
    public void itShouldSerializeSigningRequestTaSigningRequests() {
        TrustAnchorRequest taRequest0 = new TrustAnchorRequestSerializer().deserialize("<requests.TrustAnchorRequest>\n  <creationTimestamp>1558700883582</creationTimestamp>\n  <taCertificatePublicationUri>rsync://localhost:10873/ta/</taCertificatePublicationUri>\n  <taRequests>\n    <requests.SigningRequest>\n      <requestId>4ee2e78c-f746-426b-bf8b-c37e0155ca3e</requestId>\n      <resourceCertificateRequest>\n        <resourceClassName>DEFAULT</resourceClassName>\n        <subjectDN>CN=8ecc2cdf3247ef43295ebafca8c711ffd51de071</subjectDN>\n        <subjectInformationAccess>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.5</method>\n            <location>rsync://localhost/online/aca/</location>\n          </X509CertificateInformationAccessDescriptor>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.10</method>\n            <location>rsync://localhost/online/aca/jsws3zJH70MpXrr8qMcR_9Ud4HE.mft</location>\n          </X509CertificateInformationAccessDescriptor>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.13</method>\n            <location>http://localhost:7788/notification.xml</location>\n          </X509CertificateInformationAccessDescriptor>\n        </subjectInformationAccess>\n        <ipResourceSet>10.0.0.0/8, 11.0.0.0/8</ipResourceSet>\n        <encodedSubjectPublicKey>MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAtZC7nbyxIqHdncRCXV6wBtBfXtMjuz0TQLd20Hunnr/982wFMqRfsBqEI4+Q/KnPV+N1rsKGhTrAzOCnISDFO5d111qOrWWd/X0T3AjoBLu2yFwtsc+2PYXxM7aAwPl1YfBsmvDjc+BlZEmPgIVLTbkYW2dXaOKVWi5CHpcbHuzox3stStSF9C2CT49N7URwL5qQ7f55BA4kQ1U1grnQR9nbFWT0HjiVIeZow+9ofRD6Io/T6+sMS2LWb3E+YMK6DCdStlYwmZEu+2HpqBjRqB7/3nfO74djpnUXLMzSFIv4x95ZFAeV0GTvLbflfTRd9G9Wa5CF5hd9zrj5OMNwAwIDAQAB</encodedSubjectPublicKey>\n      </resourceCertificateRequest>\n    </requests.SigningRequest>\n  </taRequests>\n  <siaDescriptors>\n    <X509CertificateInformationAccessDescriptor>\n      <method>1.3.6.1.5.5.7.48.13</method>\n      <location>http://localhost:7788/notification.xml</location>\n    </X509CertificateInformationAccessDescriptor>\n    <X509CertificateInformationAccessDescriptor>\n      <method>1.3.6.1.5.5.7.48.5</method>\n      <location>rsync://localhost/online/</location>\n    </X509CertificateInformationAccessDescriptor>\n  </siaDescriptors>\n</requests.TrustAnchorRequest>");
        String request = new TrustAnchorRequestSerializer().serialize(taRequest0);
        TrustAnchorRequest taRequest1 = new TrustAnchorRequestSerializer().deserialize(request);
        Assert.assertEquals((long)taRequest0.getTaRequests().size(), (long)taRequest1.getTaRequests().size());
        Assert.assertEquals((Object)((TaRequest)taRequest0.getTaRequests().get(0)).getRequestId(), (Object)((TaRequest)taRequest1.getTaRequests().get(0)).getRequestId());
    }

    @Test
    public void itShouldSerializeSigningRequestResourceCertificateRequestInSigningRequest() {
        TrustAnchorRequest taRequest0 = new TrustAnchorRequestSerializer().deserialize("<requests.TrustAnchorRequest>\n  <creationTimestamp>1558700883582</creationTimestamp>\n  <taCertificatePublicationUri>rsync://localhost:10873/ta/</taCertificatePublicationUri>\n  <taRequests>\n    <requests.SigningRequest>\n      <requestId>4ee2e78c-f746-426b-bf8b-c37e0155ca3e</requestId>\n      <resourceCertificateRequest>\n        <resourceClassName>DEFAULT</resourceClassName>\n        <subjectDN>CN=8ecc2cdf3247ef43295ebafca8c711ffd51de071</subjectDN>\n        <subjectInformationAccess>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.5</method>\n            <location>rsync://localhost/online/aca/</location>\n          </X509CertificateInformationAccessDescriptor>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.10</method>\n            <location>rsync://localhost/online/aca/jsws3zJH70MpXrr8qMcR_9Ud4HE.mft</location>\n          </X509CertificateInformationAccessDescriptor>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.13</method>\n            <location>http://localhost:7788/notification.xml</location>\n          </X509CertificateInformationAccessDescriptor>\n        </subjectInformationAccess>\n        <ipResourceSet>10.0.0.0/8, 11.0.0.0/8</ipResourceSet>\n        <encodedSubjectPublicKey>MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAtZC7nbyxIqHdncRCXV6wBtBfXtMjuz0TQLd20Hunnr/982wFMqRfsBqEI4+Q/KnPV+N1rsKGhTrAzOCnISDFO5d111qOrWWd/X0T3AjoBLu2yFwtsc+2PYXxM7aAwPl1YfBsmvDjc+BlZEmPgIVLTbkYW2dXaOKVWi5CHpcbHuzox3stStSF9C2CT49N7URwL5qQ7f55BA4kQ1U1grnQR9nbFWT0HjiVIeZow+9ofRD6Io/T6+sMS2LWb3E+YMK6DCdStlYwmZEu+2HpqBjRqB7/3nfO74djpnUXLMzSFIv4x95ZFAeV0GTvLbflfTRd9G9Wa5CF5hd9zrj5OMNwAwIDAQAB</encodedSubjectPublicKey>\n      </resourceCertificateRequest>\n    </requests.SigningRequest>\n  </taRequests>\n  <siaDescriptors>\n    <X509CertificateInformationAccessDescriptor>\n      <method>1.3.6.1.5.5.7.48.13</method>\n      <location>http://localhost:7788/notification.xml</location>\n    </X509CertificateInformationAccessDescriptor>\n    <X509CertificateInformationAccessDescriptor>\n      <method>1.3.6.1.5.5.7.48.5</method>\n      <location>rsync://localhost/online/</location>\n    </X509CertificateInformationAccessDescriptor>\n  </siaDescriptors>\n</requests.TrustAnchorRequest>");
        String request = new TrustAnchorRequestSerializer().serialize(taRequest0);
        TrustAnchorRequest taRequest1 = new TrustAnchorRequestSerializer().deserialize(request);
        ResourceCertificateRequestData resourceCertificateRequest0 = ((SigningRequest)taRequest0.getTaRequests().get(0)).getResourceCertificateRequest();
        ResourceCertificateRequestData resourceCertificateRequest1 = ((SigningRequest)taRequest1.getTaRequests().get(0)).getResourceCertificateRequest();
        Assert.assertEquals((Object)resourceCertificateRequest0.getResourceClassName(), (Object)resourceCertificateRequest1.getResourceClassName());
        Assert.assertEquals((Object)resourceCertificateRequest0.getSubjectDN(), (Object)resourceCertificateRequest1.getSubjectDN());
        Assert.assertEquals((Object)BASE64_ENCODER.encodeToString(resourceCertificateRequest0.getEncodedSubjectPublicKey()), (Object)BASE64_ENCODER.encodeToString(resourceCertificateRequest1.getEncodedSubjectPublicKey()));
    }

    @Test
    public void itShouldSerializeSigningRequestSubjectInformationAccessDescriptorsInResourceCertificateRequest() {
        TrustAnchorRequest taRequest0 = new TrustAnchorRequestSerializer().deserialize("<requests.TrustAnchorRequest>\n  <creationTimestamp>1558700883582</creationTimestamp>\n  <taCertificatePublicationUri>rsync://localhost:10873/ta/</taCertificatePublicationUri>\n  <taRequests>\n    <requests.SigningRequest>\n      <requestId>4ee2e78c-f746-426b-bf8b-c37e0155ca3e</requestId>\n      <resourceCertificateRequest>\n        <resourceClassName>DEFAULT</resourceClassName>\n        <subjectDN>CN=8ecc2cdf3247ef43295ebafca8c711ffd51de071</subjectDN>\n        <subjectInformationAccess>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.5</method>\n            <location>rsync://localhost/online/aca/</location>\n          </X509CertificateInformationAccessDescriptor>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.10</method>\n            <location>rsync://localhost/online/aca/jsws3zJH70MpXrr8qMcR_9Ud4HE.mft</location>\n          </X509CertificateInformationAccessDescriptor>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.13</method>\n            <location>http://localhost:7788/notification.xml</location>\n          </X509CertificateInformationAccessDescriptor>\n        </subjectInformationAccess>\n        <ipResourceSet>10.0.0.0/8, 11.0.0.0/8</ipResourceSet>\n        <encodedSubjectPublicKey>MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAtZC7nbyxIqHdncRCXV6wBtBfXtMjuz0TQLd20Hunnr/982wFMqRfsBqEI4+Q/KnPV+N1rsKGhTrAzOCnISDFO5d111qOrWWd/X0T3AjoBLu2yFwtsc+2PYXxM7aAwPl1YfBsmvDjc+BlZEmPgIVLTbkYW2dXaOKVWi5CHpcbHuzox3stStSF9C2CT49N7URwL5qQ7f55BA4kQ1U1grnQR9nbFWT0HjiVIeZow+9ofRD6Io/T6+sMS2LWb3E+YMK6DCdStlYwmZEu+2HpqBjRqB7/3nfO74djpnUXLMzSFIv4x95ZFAeV0GTvLbflfTRd9G9Wa5CF5hd9zrj5OMNwAwIDAQAB</encodedSubjectPublicKey>\n      </resourceCertificateRequest>\n    </requests.SigningRequest>\n  </taRequests>\n  <siaDescriptors>\n    <X509CertificateInformationAccessDescriptor>\n      <method>1.3.6.1.5.5.7.48.13</method>\n      <location>http://localhost:7788/notification.xml</location>\n    </X509CertificateInformationAccessDescriptor>\n    <X509CertificateInformationAccessDescriptor>\n      <method>1.3.6.1.5.5.7.48.5</method>\n      <location>rsync://localhost/online/</location>\n    </X509CertificateInformationAccessDescriptor>\n  </siaDescriptors>\n</requests.TrustAnchorRequest>");
        String request = new TrustAnchorRequestSerializer().serialize(taRequest0);
        TrustAnchorRequest taRequest1 = new TrustAnchorRequestSerializer().deserialize(request);
        ResourceCertificateRequestData resourceCertificateRequest0 = ((SigningRequest)taRequest0.getTaRequests().get(0)).getResourceCertificateRequest();
        ResourceCertificateRequestData resourceCertificateRequest1 = ((SigningRequest)taRequest1.getTaRequests().get(0)).getResourceCertificateRequest();
        X509CertificateInformationAccessDescriptor[] subjectInformationAccess0 = resourceCertificateRequest0.getSubjectInformationAccess();
        X509CertificateInformationAccessDescriptor[] subjectInformationAccess1 = resourceCertificateRequest1.getSubjectInformationAccess();
        Assert.assertEquals((long)subjectInformationAccess0.length, (long)subjectInformationAccess1.length);
        X509CertificateInformationAccessDescriptor x509CertificateInformationAccessDescriptor0 = subjectInformationAccess0[0];
        X509CertificateInformationAccessDescriptor x509CertificateInformationAccessDescriptor1 = subjectInformationAccess1[0];
        Assert.assertEquals((Object)x509CertificateInformationAccessDescriptor0.getMethod(), (Object)x509CertificateInformationAccessDescriptor1.getMethod());
        Assert.assertEquals((Object)x509CertificateInformationAccessDescriptor0.getLocation(), (Object)x509CertificateInformationAccessDescriptor1.getLocation());
    }

    @Test
    public void itShouldSerializeSigningRequestSubjectInformationAccessDescriptorsInTrustAnchorRequest() {
        TrustAnchorRequest taRequest0 = new TrustAnchorRequestSerializer().deserialize("<requests.TrustAnchorRequest>\n  <creationTimestamp>1558700883582</creationTimestamp>\n  <taCertificatePublicationUri>rsync://localhost:10873/ta/</taCertificatePublicationUri>\n  <taRequests>\n    <requests.SigningRequest>\n      <requestId>4ee2e78c-f746-426b-bf8b-c37e0155ca3e</requestId>\n      <resourceCertificateRequest>\n        <resourceClassName>DEFAULT</resourceClassName>\n        <subjectDN>CN=8ecc2cdf3247ef43295ebafca8c711ffd51de071</subjectDN>\n        <subjectInformationAccess>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.5</method>\n            <location>rsync://localhost/online/aca/</location>\n          </X509CertificateInformationAccessDescriptor>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.10</method>\n            <location>rsync://localhost/online/aca/jsws3zJH70MpXrr8qMcR_9Ud4HE.mft</location>\n          </X509CertificateInformationAccessDescriptor>\n          <X509CertificateInformationAccessDescriptor>\n            <method>1.3.6.1.5.5.7.48.13</method>\n            <location>http://localhost:7788/notification.xml</location>\n          </X509CertificateInformationAccessDescriptor>\n        </subjectInformationAccess>\n        <ipResourceSet>10.0.0.0/8, 11.0.0.0/8</ipResourceSet>\n        <encodedSubjectPublicKey>MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAtZC7nbyxIqHdncRCXV6wBtBfXtMjuz0TQLd20Hunnr/982wFMqRfsBqEI4+Q/KnPV+N1rsKGhTrAzOCnISDFO5d111qOrWWd/X0T3AjoBLu2yFwtsc+2PYXxM7aAwPl1YfBsmvDjc+BlZEmPgIVLTbkYW2dXaOKVWi5CHpcbHuzox3stStSF9C2CT49N7URwL5qQ7f55BA4kQ1U1grnQR9nbFWT0HjiVIeZow+9ofRD6Io/T6+sMS2LWb3E+YMK6DCdStlYwmZEu+2HpqBjRqB7/3nfO74djpnUXLMzSFIv4x95ZFAeV0GTvLbflfTRd9G9Wa5CF5hd9zrj5OMNwAwIDAQAB</encodedSubjectPublicKey>\n      </resourceCertificateRequest>\n    </requests.SigningRequest>\n  </taRequests>\n  <siaDescriptors>\n    <X509CertificateInformationAccessDescriptor>\n      <method>1.3.6.1.5.5.7.48.13</method>\n      <location>http://localhost:7788/notification.xml</location>\n    </X509CertificateInformationAccessDescriptor>\n    <X509CertificateInformationAccessDescriptor>\n      <method>1.3.6.1.5.5.7.48.5</method>\n      <location>rsync://localhost/online/</location>\n    </X509CertificateInformationAccessDescriptor>\n  </siaDescriptors>\n</requests.TrustAnchorRequest>");
        String request = new TrustAnchorRequestSerializer().serialize(taRequest0);
        TrustAnchorRequest taRequest1 = new TrustAnchorRequestSerializer().deserialize(request);
        X509CertificateInformationAccessDescriptor[] subjectInformationAccess0 = taRequest0.getSiaDescriptors();
        X509CertificateInformationAccessDescriptor[] subjectInformationAccess1 = taRequest1.getSiaDescriptors();
        Assert.assertEquals((long)subjectInformationAccess0.length, (long)subjectInformationAccess1.length);
        X509CertificateInformationAccessDescriptor x509CertificateInformationAccessDescriptor0 = subjectInformationAccess0[0];
        X509CertificateInformationAccessDescriptor x509CertificateInformationAccessDescriptor1 = subjectInformationAccess1[0];
        Assert.assertEquals((Object)x509CertificateInformationAccessDescriptor0.getMethod(), (Object)x509CertificateInformationAccessDescriptor1.getMethod());
        Assert.assertEquals((Object)x509CertificateInformationAccessDescriptor0.getLocation(), (Object)x509CertificateInformationAccessDescriptor1.getLocation());
    }

    @Test
    public void itShouldSerializeRevocationRequestTaCertificatePublicationUri() {
        TrustAnchorRequest taRequest0 = new TrustAnchorRequestSerializer().deserialize("<requests.TrustAnchorRequest>\n  <creationTimestamp>1610359575105</creationTimestamp>\n  <taCertificatePublicationUri>rsync://localhost:10873/ta/</taCertificatePublicationUri>\n  <taRequests>\n    <requests.RevocationRequest>\n      <requestId>3ced3f70-a2b4-42d4-9e46-2fe4cac6b4bf</requestId>\n      <resourceClassName>DEFAULT</resourceClassName>\n      <encodedPublicKey>MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAtZC7nbyxIqHdncRCXV6wBtBfXtMjuz0TQLd20Hunnr/982wFMqRfsBqEI4+Q/KnPV+N1rsKGhTrAzOCnISDFO5d111qOrWWd/X0T3AjoBLu2yFwtsc+2PYXxM7aAwPl1YfBsmvDjc+BlZEmPgIVLTbkYW2dXaOKVWi5CHpcbHuzox3stStSF9C2CT49N7URwL5qQ7f55BA4kQ1U1grnQR9nbFWT0HjiVIeZow+9ofRD6Io/T6+sMS2LWb3E+YMK6DCdStlYwmZEu+2HpqBjRqB7/3nfO74djpnUXLMzSFIv4x95ZFAeV0GTvLbflfTRd9G9Wa5CF5hd9zrj5OMNwAwIDAQAB</encodedPublicKey>\n    </requests.RevocationRequest>\n  </taRequests>\n  <siaDescriptors/>\n</requests.TrustAnchorRequest>");
        String request = new TrustAnchorRequestSerializer().serialize(taRequest0);
        TrustAnchorRequest taRequest1 = new TrustAnchorRequestSerializer().deserialize(request);
        Assert.assertEquals((Object)taRequest0.getTaCertificatePublicationUri(), (Object)taRequest1.getTaCertificatePublicationUri());
    }

    @Test
    public void itShouldSerializeRevocationRequestCreationTimestamp() {
        TrustAnchorRequest taRequest0 = new TrustAnchorRequestSerializer().deserialize("<requests.TrustAnchorRequest>\n  <creationTimestamp>1610359575105</creationTimestamp>\n  <taCertificatePublicationUri>rsync://localhost:10873/ta/</taCertificatePublicationUri>\n  <taRequests>\n    <requests.RevocationRequest>\n      <requestId>3ced3f70-a2b4-42d4-9e46-2fe4cac6b4bf</requestId>\n      <resourceClassName>DEFAULT</resourceClassName>\n      <encodedPublicKey>MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAtZC7nbyxIqHdncRCXV6wBtBfXtMjuz0TQLd20Hunnr/982wFMqRfsBqEI4+Q/KnPV+N1rsKGhTrAzOCnISDFO5d111qOrWWd/X0T3AjoBLu2yFwtsc+2PYXxM7aAwPl1YfBsmvDjc+BlZEmPgIVLTbkYW2dXaOKVWi5CHpcbHuzox3stStSF9C2CT49N7URwL5qQ7f55BA4kQ1U1grnQR9nbFWT0HjiVIeZow+9ofRD6Io/T6+sMS2LWb3E+YMK6DCdStlYwmZEu+2HpqBjRqB7/3nfO74djpnUXLMzSFIv4x95ZFAeV0GTvLbflfTRd9G9Wa5CF5hd9zrj5OMNwAwIDAQAB</encodedPublicKey>\n    </requests.RevocationRequest>\n  </taRequests>\n  <siaDescriptors/>\n</requests.TrustAnchorRequest>");
        String request = new TrustAnchorRequestSerializer().serialize(taRequest0);
        TrustAnchorRequest taRequest1 = new TrustAnchorRequestSerializer().deserialize(request);
        Assert.assertEquals((Object)taRequest0.getCreationTimestamp(), (Object)taRequest1.getCreationTimestamp());
    }

    @Test
    public void itShouldSerializeRevocationRequestTaSigningRequests() {
        TrustAnchorRequest taRequest0 = new TrustAnchorRequestSerializer().deserialize("<requests.TrustAnchorRequest>\n  <creationTimestamp>1610359575105</creationTimestamp>\n  <taCertificatePublicationUri>rsync://localhost:10873/ta/</taCertificatePublicationUri>\n  <taRequests>\n    <requests.RevocationRequest>\n      <requestId>3ced3f70-a2b4-42d4-9e46-2fe4cac6b4bf</requestId>\n      <resourceClassName>DEFAULT</resourceClassName>\n      <encodedPublicKey>MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAtZC7nbyxIqHdncRCXV6wBtBfXtMjuz0TQLd20Hunnr/982wFMqRfsBqEI4+Q/KnPV+N1rsKGhTrAzOCnISDFO5d111qOrWWd/X0T3AjoBLu2yFwtsc+2PYXxM7aAwPl1YfBsmvDjc+BlZEmPgIVLTbkYW2dXaOKVWi5CHpcbHuzox3stStSF9C2CT49N7URwL5qQ7f55BA4kQ1U1grnQR9nbFWT0HjiVIeZow+9ofRD6Io/T6+sMS2LWb3E+YMK6DCdStlYwmZEu+2HpqBjRqB7/3nfO74djpnUXLMzSFIv4x95ZFAeV0GTvLbflfTRd9G9Wa5CF5hd9zrj5OMNwAwIDAQAB</encodedPublicKey>\n    </requests.RevocationRequest>\n  </taRequests>\n  <siaDescriptors/>\n</requests.TrustAnchorRequest>");
        String request = new TrustAnchorRequestSerializer().serialize(taRequest0);
        TrustAnchorRequest taRequest1 = new TrustAnchorRequestSerializer().deserialize(request);
        Assert.assertEquals((long)taRequest0.getTaRequests().size(), (long)taRequest1.getTaRequests().size());
        Assert.assertEquals((Object)((TaRequest)taRequest0.getTaRequests().get(0)).getRequestId(), (Object)((TaRequest)taRequest1.getTaRequests().get(0)).getRequestId());
    }

    @Test
    public void itShouldSerializeRevocationRequestResourceCertificateRequestInSigningRequest() {
        TrustAnchorRequest taRequest0 = new TrustAnchorRequestSerializer().deserialize("<requests.TrustAnchorRequest>\n  <creationTimestamp>1610359575105</creationTimestamp>\n  <taCertificatePublicationUri>rsync://localhost:10873/ta/</taCertificatePublicationUri>\n  <taRequests>\n    <requests.RevocationRequest>\n      <requestId>3ced3f70-a2b4-42d4-9e46-2fe4cac6b4bf</requestId>\n      <resourceClassName>DEFAULT</resourceClassName>\n      <encodedPublicKey>MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAtZC7nbyxIqHdncRCXV6wBtBfXtMjuz0TQLd20Hunnr/982wFMqRfsBqEI4+Q/KnPV+N1rsKGhTrAzOCnISDFO5d111qOrWWd/X0T3AjoBLu2yFwtsc+2PYXxM7aAwPl1YfBsmvDjc+BlZEmPgIVLTbkYW2dXaOKVWi5CHpcbHuzox3stStSF9C2CT49N7URwL5qQ7f55BA4kQ1U1grnQR9nbFWT0HjiVIeZow+9ofRD6Io/T6+sMS2LWb3E+YMK6DCdStlYwmZEu+2HpqBjRqB7/3nfO74djpnUXLMzSFIv4x95ZFAeV0GTvLbflfTRd9G9Wa5CF5hd9zrj5OMNwAwIDAQAB</encodedPublicKey>\n    </requests.RevocationRequest>\n  </taRequests>\n  <siaDescriptors/>\n</requests.TrustAnchorRequest>");
        String request = new TrustAnchorRequestSerializer().serialize(taRequest0);
        TrustAnchorRequest taRequest1 = new TrustAnchorRequestSerializer().deserialize(request);
        RevocationRequest revocationRequest0 = (RevocationRequest)taRequest0.getTaRequests().get(0);
        RevocationRequest revocationRequest1 = (RevocationRequest)taRequest1.getTaRequests().get(0);
        Assert.assertEquals((Object)revocationRequest0.getResourceClassName(), (Object)revocationRequest1.getResourceClassName());
        Assert.assertEquals((Object)revocationRequest0.getEncodedPublicKey(), (Object)revocationRequest1.getEncodedPublicKey());
    }

    @Test
    public void itShouldDeserializeLegacyXmlRequestElements() throws IOException {
        String stateXML = Files.asCharSource((File)new File(LEGACY_TA_REQUEST_PATH), (Charset)Charsets.UTF_8).read();
        TrustAnchorRequestSerializer trustAnchorRequestSerializer = new TrustAnchorRequestSerializer();
        TrustAnchorRequest trustAnchorRequest = trustAnchorRequestSerializer.deserialize(stateXML);
        Assert.assertFalse((boolean)trustAnchorRequest.getTaRequests().isEmpty());
        Assert.assertEquals((long)2L, (long)trustAnchorRequest.getTaRequests().size());
    }

    @Test
    public void itShouldDeserializeXmlWithoutTaCertificatePublicationUriElement() throws IOException {
        String stateXML = Files.asCharSource((File)new File(TA_REQUEST_NO_TA_URI_PATH), (Charset)Charsets.UTF_8).read();
        TrustAnchorRequestSerializer trustAnchorRequestSerializer = new TrustAnchorRequestSerializer();
        TrustAnchorRequest trustAnchorRequest = trustAnchorRequestSerializer.deserialize(stateXML);
        Assert.assertNotNull((Object)trustAnchorRequest);
    }
}

