package net.peachjean.commons.base.service;

import java.io.IOException;
import java.io.Writer;
import java.util.Set;

import javax.annotation.processing.AbstractProcessor;
import javax.annotation.processing.ProcessingEnvironment;
import javax.annotation.processing.RoundEnvironment;
import javax.annotation.processing.SupportedAnnotationTypes;
import javax.annotation.processing.SupportedSourceVersion;
import javax.lang.model.SourceVersion;
import javax.lang.model.element.Element;
import javax.lang.model.element.PackageElement;
import javax.lang.model.element.TypeElement;
import javax.tools.Diagnostic;
import javax.tools.JavaFileObject;

import net.peachjean.commons.base.constructor.ConstructorSignatureProcessor;

import com.google.common.collect.Iterables;
import com.google.common.io.OutputSupplier;


@SupportedAnnotationTypes("net.peachjean.commons.base.service.Service")
public class ServiceProcessor extends AbstractProcessor
{
	private ConstructorSignatureProcessor signatureProcessor;

	@Override
	public void init(final ProcessingEnvironment processingEnv)
	{
		super.init(processingEnv);
		this.signatureProcessor = new ConstructorSignatureProcessor();
		this.signatureProcessor.init(processingEnv);
	}

    @Override
    public SourceVersion getSupportedSourceVersion() {
        return SourceVersion.latest();
    }

    @Override
	public boolean process(final Set<? extends TypeElement> annotations, final RoundEnvironment roundEnv)
	{
		for(TypeElement typeElement: annotations)
		{
			for(Element element: roundEnv.getElementsAnnotatedWith(typeElement))
			{
				if(element instanceof TypeElement)
				{
					TypeElement serviceElement = (TypeElement) element;
					PackageElement packageElement = getPackage(serviceElement);
					final String packageName =
							packageElement == null ? "" : packageElement.getQualifiedName().toString();
					final String simpleName = serviceElement.getSimpleName().toString();
					ServiceDescriptor serviceDescriptor = new ServiceDescriptor(packageName, simpleName, buildConstructorSignature(serviceElement));
					createFactorySourceFile(serviceDescriptor, serviceElement);
				}
			}
		}
		return true;
	}

	private Iterable<ConstructorSignatureProcessor.ConstructorRequirement> buildConstructorSignature(
			final TypeElement serviceElement)
	{
		return this.signatureProcessor.determineRequirements(serviceElement);
	}

	private void createFactorySourceFile(final ServiceDescriptor serviceDescriptor, final TypeElement serviceElement)
	{
		String className = processingEnv.getElementUtils().getBinaryName(serviceElement) + "Factory";
		try
		{
			final JavaFileObject sourceFile = processingEnv.getFiler().createSourceFile(className, serviceElement);
			serviceDescriptor.generateSource(new OutputSupplier<Writer>()
			{
				@Override
				public Writer getOutput() throws IOException
				{
					return sourceFile.openWriter();
				}
			});
		}
		catch (IOException e)
		{
			processingEnv.getMessager().printMessage(Diagnostic.Kind.ERROR, "Could not create source file: " + className + "[" + e.getMessage() + "]");
		}
	}

	private PackageElement getPackage(final Element serviceElement)
	{
		if(serviceElement == null)
		{
			return null;
		}
		Element enclosing = serviceElement.getEnclosingElement();
		if(enclosing instanceof PackageElement)
		{
			return (PackageElement) enclosing;
		}
		else
		{
			return getPackage(enclosing);
		}
	}
}
