package net.peachjean.commons.base.service;

import java.io.IOException;
import java.net.URL;
import java.util.Enumeration;
import java.util.Map;
import java.util.concurrent.ConcurrentMap;

import com.google.common.base.Charsets;
import com.google.common.base.Throwables;
import com.google.common.collect.Maps;
import com.google.common.io.Resources;

/**
 * Utility used by generated {service}Factory classes.
 * @param <T>
 */
public class ServiceLoader<T>
{
	private Class<T> serviceType;
	private Map<String, Class<? extends T>> nameMap;

	public ServiceLoader(final Class<T> serviceType)
	{
		this.serviceType = serviceType;
		this.nameMap = loadNameMap(serviceType);
	}

	private static <T> Map<String, Class<? extends T>> loadNameMap(final Class<T> serviceType)
	{
		final ConcurrentMap<String,Class<? extends T>> nameMap = Maps.newConcurrentMap();

		try
		{
			final ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
			Enumeration<URL> serviceDescriptors = classLoader.getResources(serviceType.getName());
			while(serviceDescriptors.hasMoreElements())
			{
				for(String line: Resources.readLines(serviceDescriptors.nextElement(), Charsets.UTF_8))
				{
					Class<? extends T> implementationType = (Class<? extends T>) classLoader.loadClass(line);
					if(!serviceType.isAssignableFrom(implementationType))
					{
						throw new ServiceLookupException(serviceType, implementationType);
					}
					if(implementationType.isAnnotationPresent(ServiceName.class))
					{
						nameMap.put(implementationType.getAnnotation(ServiceName.class).value(), implementationType);
					}
					nameMap.put(implementationType.getName(), implementationType);
				}
			}
		}
		catch (IOException e)
		{
			throw Throwables.propagate(e);
		}
		catch (ClassNotFoundException e)
		{
			throw Throwables.propagate(e);
		}

		return nameMap;
	}

	public ServiceImplementation<T> lookup(String name) throws ServiceLookupException
	{
		if(nameMap.containsKey(name))
		{
			return new ServiceImplementation<T>(serviceType, nameMap.get(name));
		}
		else
		{
			try
			{
				Class<? extends T> fqcn = loadAsFQCN(name);
				if(!serviceType.isAssignableFrom(fqcn))
				{
					throw new ServiceLookupException(serviceType, fqcn);
				}
				nameMap.put(name, fqcn);
				return new ServiceImplementation<T>(serviceType, fqcn);
			}
			catch (ClassNotFoundException e)
			{
				throw new ServiceLookupException(serviceType, name);
			}
		}
	}

	@SuppressWarnings({"unchecked"})
	private Class<? extends T> loadAsFQCN(final String name) throws ClassNotFoundException
	{
		return (Class<? extends T>) Thread.currentThread().getContextClassLoader().loadClass(name);
	}
}
