package net.peachjean.commons.base.constructor;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Sets;

import javax.annotation.processing.AbstractProcessor;
import javax.annotation.processing.ProcessingEnvironment;
import javax.annotation.processing.RoundEnvironment;
import javax.annotation.processing.SupportedAnnotationTypes;
import javax.annotation.processing.SupportedSourceVersion;
import javax.lang.model.SourceVersion;
import javax.lang.model.element.AnnotationMirror;
import javax.lang.model.element.AnnotationValue;
import javax.lang.model.element.Element;
import javax.lang.model.element.ElementKind;
import javax.lang.model.element.ExecutableElement;
import javax.lang.model.element.Modifier;
import javax.lang.model.element.TypeElement;
import javax.lang.model.type.DeclaredType;
import javax.lang.model.type.ExecutableType;
import javax.lang.model.type.NoType;
import javax.lang.model.type.TypeMirror;
import javax.lang.model.util.ElementFilter;
import javax.lang.model.util.SimpleAnnotationValueVisitor6;
import javax.lang.model.util.SimpleElementVisitor6;
import javax.lang.model.util.SimpleTypeVisitor6;
import javax.tools.Diagnostic;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * @author jbunting
 */
@SupportedAnnotationTypes("*")
public class ConstructorSignatureProcessor extends AbstractProcessor
{
	private TypeMirror constructorArgsType;
	private ExecutableElement valueElement;

	@Override
	public void init(final ProcessingEnvironment processingEnv)
	{
		super.init(processingEnv);
		constructorArgsType = processingEnv.getElementUtils().getTypeElement(ConstructorSignature.class.getName()).asType();
		valueElement = getValueElement((DeclaredType) constructorArgsType);
	}

    @Override
    public SourceVersion getSupportedSourceVersion() {
        return SourceVersion.latest();
    }

    public boolean process(Set<? extends TypeElement> annotations,
	                       RoundEnvironment env)
	{
//        processingEnv.getMessager().printMessage(Diagnostic.Kind.ERROR, "DAMMIT (Constructor)");
		findAndValidate(env.getRootElements());
		return false;
	}

	private void findAndValidate(final Collection<? extends Element> elements)
	{
		for (TypeElement type : ElementFilter.typesIn(elements))
		{
			findAndValidate(type.getEnclosedElements());
			findAndValidate(type);
		}
	}

	private void findAndValidate(final TypeElement type)
	{
		for (ConstructorRequirement constructorRequirement : determineConcreteRequirements(type))
		{
			validateRequirement(type, constructorRequirement);
		}
	}

	private Iterable<ConstructorRequirement> determineConcreteRequirements(final TypeElement type)
	{
		if (type.getKind().isInterface())
		{
			return Collections.emptySet();
		}
		return determineRequirements(type);
	}

	public Iterable<ConstructorRequirement> determineRequirements(
			final TypeElement type)
	{
		Set<ConstructorRequirement> requirements = Sets.newHashSet();
		populateRequirementsSet(requirements, type);
		return requirements;
	}

	private void populateRequirementsSet(final Set<ConstructorRequirement> requirements, final TypeElement type)
	{
		this.populateRequirementsSet(requirements, type, Sets.<TypeElement>newHashSet());
	}

	private void populateRequirementsSet(final Set<ConstructorRequirement> requirements, final TypeElement type,
	                                     Set<TypeElement> previouslyConsidered)
	{
		if (hasAlreadyBeenConsidered(type, previouslyConsidered))
		{
			return;
		}
		populateRequirementsFromAnnotations(requirements, previouslyConsidered, buildAnnotationMirrorList(type));
		populateRequirementsFromInterfaces(requirements, previouslyConsidered, type.getInterfaces());
	}

    private List<? extends AnnotationMirror> buildAnnotationMirrorList(final TypeElement type) {
        ImmutableList.Builder<AnnotationMirror> mirrorList = ImmutableList.builder();

        TypeElement target = type;
        while(target != null) {
            mirrorList.addAll(target.getAnnotationMirrors());
            final TypeMirror superclass = target.getSuperclass();
            if(superclass instanceof NoType) {
                break;
            }
            target = (TypeElement) processingEnv.getTypeUtils().asElement(superclass);
        }
        return mirrorList.build();
    }

    private boolean hasAlreadyBeenConsidered(final TypeElement type, final Set<TypeElement> previouslyConsidered)
	{
		if (previouslyConsidered.contains(type))
		{
			return true;
		}
		else
		{
			previouslyConsidered.add(type);
		}
		return false;
	}

	private void populateRequirementsFromAnnotations(
			final Set<ConstructorRequirement> requirements,
			final Set<TypeElement> previouslyConsidered, final List<? extends AnnotationMirror> annotationMirrors)
	{
		for (AnnotationMirror mirror : annotationMirrors)
		{
			TypeElement annotationType = mirror.getAnnotationType().asElement().accept(typeConversionVisitor, null);
			if (isConstructorArg(mirror))
			{
				requirements.add(new ConstructorRequirement(mirror));
			}
			populateRequirementsSet(requirements, annotationType, previouslyConsidered);
		}
	}

	private void populateRequirementsFromInterfaces(
			final Set<ConstructorRequirement> requirements,
			final Set<TypeElement> previouslyConsidered, final List<? extends TypeMirror> interfaces)
	{
		for (TypeMirror iface : interfaces)
		{
			TypeElement interfaceType =
					processingEnv.getTypeUtils().asElement(iface).accept(typeConversionVisitor, null);
			populateRequirementsSet(requirements, interfaceType, previouslyConsidered);
		}
	}

	private boolean isConstructorArg(final AnnotationMirror mirror)
	{
		boolean isType = mirror.getAnnotationType().accept(new SimpleTypeVisitor6<Boolean, Object>() {
			@Override
			protected Boolean defaultAction(final TypeMirror e, final Object o)
			{
				return false;
			}

			@Override
			public Boolean visitDeclared(final DeclaredType t, final Object o)
			{
				return true;
			}
		}, null);
		return isType && processingEnv.getTypeUtils().isSameType(constructorArgsType, mirror.getAnnotationType());
	}

	private void validateRequirement(final TypeElement type, final ConstructorRequirement constructorRequirement)
	{
		if (!doesClassMeetRequirement(type, constructorRequirement))
		{
			processingEnv.getMessager().printMessage(
					Diagnostic.Kind.ERROR,
					"Class " + type + " needs a public Constructor with parameters " + constructorRequirement.getParams());
		}
	}

	private boolean doesClassMeetRequirement(final TypeElement type, final ConstructorRequirement constructorRequirement)
	{
		for (Element subelement : type.getEnclosedElements())
		{
			if (subelement.getKind() == ElementKind.CONSTRUCTOR &&
			    subelement.getModifiers().contains(Modifier.PUBLIC))
			{
				TypeMirror mirror = subelement.asType();
				if (mirror.accept(requirementVisitor, constructorRequirement))
				{
					return true;
				}
			}
		}
		return false;
	}

	public class ConstructorRequirement
	{
		private final List<TypeMirror> params;

		private ConstructorRequirement(AnnotationMirror annotationMirror)
		{
			Preconditions.checkArgument(isConstructorArg(annotationMirror), "Requirement must be built from @" +
			                                                                ConstructorSignature.class.getName());
			Map<? extends ExecutableElement,? extends AnnotationValue> elementValuesWithDefaults =
					processingEnv.getElementUtils().getElementValuesWithDefaults(annotationMirror);
			this.params = Collections.unmodifiableList(elementValuesWithDefaults.get(valueElement)
			                                                           .accept(valueVisitor, null));
		}


		public List<TypeMirror> getParams()
		{
			return params;
		}
	}

	private static ExecutableElement getValueElement(final DeclaredType type)
	{
		for(Element enclosed : type.asElement().getEnclosedElements())
		{
			if( enclosed instanceof ExecutableElement && enclosed.getSimpleName().contentEquals("value")) {
				return (ExecutableElement) enclosed;
			}
		}
		throw new IllegalStateException("Could not locate value element on " + type);
	}

	private static final SimpleAnnotationValueVisitor6<TypeMirror,Object> elementVisitor =
			new SimpleAnnotationValueVisitor6<TypeMirror, Object>()
			{
				@Override
				public TypeMirror visitType(final TypeMirror t, final Object o)
				{
					return t;
				}
			};

	private static final SimpleAnnotationValueVisitor6<List<TypeMirror>, Object> valueVisitor =
			new SimpleAnnotationValueVisitor6<List<TypeMirror>, Object>()
			{


				@Override
				public List<TypeMirror> visitArray(final List<? extends AnnotationValue> vals, final Object o)
				{
					List<TypeMirror> returnValues = new ArrayList<TypeMirror>(vals.size());
					for (AnnotationValue val : vals)
					{
						returnValues.add(val.accept(elementVisitor, null));
					}
					return returnValues;
				}
			};

	private static final SimpleElementVisitor6<TypeElement, Object> typeConversionVisitor =
			new SimpleElementVisitor6<TypeElement, Object>()
			{
				@Override
				public TypeElement visitType(final TypeElement e, final Object o)
				{
					return e;
				}
			};

	private static final SimpleTypeVisitor6<Boolean, ConstructorRequirement> requirementVisitor =
			new SimpleTypeVisitor6<Boolean, ConstructorRequirement>()
			{
				@Override
				public Boolean visitExecutable(final ExecutableType t, final ConstructorRequirement requirement)
				{
					List<? extends TypeMirror> parameterTypes = t.getParameterTypes();
					return parameterTypes.equals(requirement.getParams());
				}
			};
}
