package net.overburn.redfort.keygen;

import net.overburn.redfort.util.Toolbox;
import org.bouncycastle.crypto.generators.Argon2BytesGenerator;
import org.bouncycastle.crypto.params.Argon2Parameters;

import javax.crypto.spec.SecretKeySpec;
import java.nio.charset.StandardCharsets;

/**
 * Derives a suitable key from a secret/password, salt, extra secret for keyed hashing. Additional info can be used to tag. Implements  {@link SecretBasedKeyGenerator}
 * WARNING: Stick to {@link Argon2KeyGenerator} and avoid modifying/implementing this or any implementations without understanding the ramifications.
 * For more info see:
 * @see <a href="https://datatracker.ietf.org/doc/html/rfc9106">rfc9106</a>
 */
public class Argon2KeyGenerator implements SecretBasedKeyGenerator
{

    private static final String KEY_DERIVATION_FUNCTION = ARGON2;
    private static final int KEY_LENGTH = 256;
    private static final int ITERATIONS = 3;

    public SecretKeySpec generate(String secret, String salt)
    {
        return generate(secret, salt,null);
    }

    public SecretKeySpec generate(String secret, String salt, String pepper)
    {
        return generate(secret, salt, pepper, null);
    }

    public SecretKeySpec generate(String secret, String salt, String pepper, byte[] ad)
    {
        return generate(secret.toCharArray(), ((salt==null)?null:salt.getBytes(StandardCharsets.UTF_8)), ((pepper ==null)?null: pepper.getBytes(StandardCharsets.UTF_8)), ad);
    }

    public SecretKeySpec generate(char[] secret, byte[] salt, byte[] pepper, byte[] ad)
    {
        return wrap(derive(secret, salt, pepper, ad));
    }


    public byte[] derive(char[] secret, byte[] salt, byte[] pepper, byte[] additional)
    {
        Argon2BytesGenerator bytesGenerator = builder(salt, pepper, additional);

        byte[] rawKey = new byte[KEY_LENGTH/8];
        bytesGenerator.generateBytes(secret, rawKey);

        return rawKey;
    }

    private Argon2BytesGenerator builder(byte[] salt, byte[] pepper, byte[] ad)
    {
        Argon2BytesGenerator bytesGenerator = new Argon2BytesGenerator();
        Argon2Parameters.Builder builder = new Argon2Parameters.Builder(Argon2Parameters.ARGON2_id)
                .withVersion(Argon2Parameters.ARGON2_VERSION_13) // 19
                .withIterations(ITERATIONS)
                .withMemoryAsKB(32)
                .withParallelism(4)
                .withSalt(salt);

        if(pepper!=null)
            builder.withSecret(pepper);

        if(ad != null)
            builder.withAdditional(ad);


        bytesGenerator.init(builder.build());

        return bytesGenerator;
    }

    public SecretKeySpec wrap(byte[] rawKey)
    {
        return wrap(rawKey, KEY_DERIVATION_FUNCTION);
    }

    public SecretKeySpec wrap(String rawKey)
    {
        return wrap(Toolbox.decode(rawKey));
    }

}
