package net.overburn.redfort.devices;

import net.overburn.redfort.exceptions.CryptoDeviceException;
import net.overburn.redfort.keygen.*;

import javax.crypto.spec.SecretKeySpec;

import java.nio.charset.StandardCharsets;

import static net.overburn.redfort.keygen.BasicKeyGenerator.*;

/**
 * Driver with opinionated defaults to Encrypt and Decrypt data on demand.
 * @since 0.0.1
 */
public class CryptoDriver {

    private final Scrambler scrambler;

    private final BasicKeyGenerator keyGenerator;

    public CryptoDriver(String cipherSuite, String kdf)
    {
        this(newScrambler(cipherSuite), newKeyGenerator(kdf));
    }

    public CryptoDriver(Scrambler scrambler, BasicKeyGenerator keyGenerator)
    {
        this.scrambler = scrambler;
        this.keyGenerator = keyGenerator;
    }

    public Scrambler getScrambler() {
        return scrambler;
    }

    public BasicKeyGenerator getKeyGenerator() {
        return keyGenerator;
    }

    private static Scrambler newScrambler(String cipherSuite)
    {
        if(Scrambler.CHACHA20_POLY1305.equalsIgnoreCase(cipherSuite) || Scrambler.AES_GCM_256.equalsIgnoreCase(cipherSuite))
        {
            return new Scrambler(cipherSuite);
        }
        else
        {
            throw new CryptoDeviceException("Unsupported choice of Cipher Suite. Use "+Scrambler.CHACHA20_POLY1305+" for good security margin");
        }
    }

    private static BasicKeyGenerator newKeyGenerator(String kdf)
    {
        switch (kdf)
        {
            case ARGON2: return new Argon2KeyGenerator();
            case PBKDF2_WITH_HMAC_SHA_512: return new PBKDF2KeyGenerator();
            case AES: return new AESKeyGenerator();
            case CHACHA20: return new ChaCha20KeyGenerator();
            default: throw new CryptoDeviceException("Unsupported choice of KDF. Use "+ ARGON2+" for good security margin");
        }
    }

    public static CryptoDriver newDriver(String cipherSuite, String kdf)
    {
        return new CryptoDriver(cipherSuite, kdf);
    }
    public static CryptoDriver newSymKeyDriver()
    {
        return newDriver(Scrambler.CHACHA20_POLY1305, CHACHA20);
    }
    public static CryptoDriver newSecretDriver()
    {
        return newDriver(Scrambler.CHACHA20_POLY1305, ARGON2);
    }

    public String encrypt(String clearText, SecretKeySpec key)
    {
        return encrypt(clearText.getBytes(StandardCharsets.UTF_8), key);
    }
    public String encrypt(byte[] clearBytes, SecretKeySpec key)
    {
        return scrambler.encrypt(clearBytes, key);
    }

    public byte[] decrypt(String cipherText, SecretKeySpec key)
    {
        return scrambler.decrypt(cipherText, key);
    }

    public String encrypt(byte[] clearBytes, String secret, String salt)
    {
        SecretBasedKeyGenerator keyGenerator = (SecretBasedKeyGenerator) this.keyGenerator;
        return encrypt(clearBytes, keyGenerator.generate(secret, salt));
    }

    public byte[] decrypt(String cipherText, String secret, String salt)
    {
        SecretBasedKeyGenerator keyGenerator = (SecretBasedKeyGenerator) this.keyGenerator;
        return decrypt(cipherText, keyGenerator.generate(secret, salt));
    }

    public String encrypt(byte[] clearBytes, String secret)
    {
        return encrypt(clearBytes, secret, "");
    }

    public byte[] decrypt(String cipherText, String secret)
    {
        return decrypt(cipherText, secret, "");
    }

}
