package net.overburn.redfort.keygen;

import net.overburn.redfort.util.Toolbox;
import net.overburn.redfort.exceptions.KeyDerivationException;

import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.PBEKeySpec;
import javax.crypto.spec.SecretKeySpec;
import java.nio.charset.StandardCharsets;
import java.security.NoSuchAlgorithmException;
import java.security.spec.InvalidKeySpecException;


/**
 * Derives a suitable key from a secret/password, salt, extra secret for keyed hashing. Additional info can be used to tag. Implements {@link SecretBasedKeyGenerator}.
 * Use this where mandated by law/guidelines.
 * WARNING: Stick to {@link Argon2KeyGenerator} and avoid modifying/implementing this or any implementations without understanding the ramifications.
 * For more info look:
 * @see <a href="https://nvlpubs.nist.gov/nistpubs/Legacy/SP/nistspecialpublication800-132.pdf">NIST800</a>
 */
public class PBKDF2KeyGenerator implements SecretBasedKeyGenerator
{

    private static final String KEY_DERIVATION_FUNCTION = PBKDF2_WITH_HMAC_SHA_512;
    private static final int KEY_LENGTH = 256;
    private static final int ITERATIONS = 3;

    public SecretKeySpec generate(String secret, String salt)
    {
        return generate(secret, salt,null);
    }

    public SecretKeySpec generate(String secret, String salt, String pepper)
    {
        return generate(secret, salt, pepper, null);
    }

    public SecretKeySpec generate(String secret, String salt, String pepper, byte[] ad)
    {
        return generate(secret.toCharArray(), ((salt==null)?null:salt.getBytes(StandardCharsets.UTF_8)), ((pepper ==null)?null: pepper.getBytes(StandardCharsets.UTF_8)), ad);
    }

    public SecretKeySpec generate(char[] secret, byte[] salt, byte[] pepper, byte[] ad)
    {
        return wrap(derive(secret, salt, pepper, ad));
    }

    public byte[] derive(char[] secret, byte[] salt, byte[] pepper, byte[] additional)
    {
        try {
            PBEKeySpec keySpec = new PBEKeySpec(secret ,salt, ITERATIONS, KEY_LENGTH);
            SecretKeyFactory pbkdfKeyFactory = SecretKeyFactory.getInstance(KEY_DERIVATION_FUNCTION);

            SecretKey rawKey = pbkdfKeyFactory.generateSecret(keySpec);

            return rawKey.getEncoded();

        } catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
            throw new KeyDerivationException(e.getMessage());
        }

    }

    public SecretKeySpec wrap(byte[] rawKey)
    {
        return wrap(rawKey, KEY_DERIVATION_FUNCTION);
    }

    public SecretKeySpec wrap(String rawKey)
    {
        return wrap(Toolbox.decode(rawKey));
    }

}
