package cicada.mq.receive.config;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.springframework.stereotype.Component;

import cicada.core.PropertyResolverCustom;

@Component
public class ReceiverConfigImpl implements ReceiverConfig
{
	private static final Map<String, FillData> fillDataStrategies;

	static
	{
		Map<String, FillData> map = new HashMap<String, FillData>();
		map.put("contract", new FillDataImplContract());
		map.put("channel", new FillDataImplChannelName());
		map.put("type", new FillDataImplType());
		map.put("server", new FillDataImplServer());
		map.put("server.port", new FillDataImplPort());
		map.put("server.username", new FillDataImplServerUserName());
		map.put("server.password", new FillDataImplServerPassword());
		fillDataStrategies = map;
	}

	private final List<ReceiverInfo> _receivers = new ArrayList<ReceiverInfo>();

	private final Map<String, String> _configurationDataRespository;

	private boolean _loaded;

	public ReceiverConfigImpl() throws IOException
	{
		_configurationDataRespository = PropertyResolverCustom.getConfigProperties("conf/cicada.properties");
	}

	@Override
	public List<ReceiverInfo> getReceivers() throws Exception
	{
		if (!_loaded)
		{
			synchronized (this)
			{
				if (!_loaded)
				{
					List<ReceiverInfo> list = loadByConfig();
					for (ReceiverInfo current : list)
					{
						_receivers.add(current);
					}
				}
				_loaded = true;
			}
		}
		return _receivers;
	}

	private List<ReceiverInfo> loadByConfig() throws Exception
	{
		Map<String, ReceiverInfo> receiverInfos = new HashMap<String, ReceiverInfo>();
		for (Map.Entry<String, String> entry : _configurationDataRespository.entrySet())
		{
			String text = entry.getKey();
			String regEx = "^Cicada.Mq.Receivers.(\\w+).([\\w.]+)$";
			Pattern pattern = Pattern.compile(regEx, Pattern.CASE_INSENSITIVE);
			Matcher matcher = pattern.matcher(text);
			if (matcher.matches() && matcher.groupCount() == 2)
			{
				ReceiverInfo receiverInfo = null;
				String group1 = matcher.group(1);
				if (receiverInfos.containsKey(group1))
				{
					receiverInfo = receiverInfos.get(group1);
				}
				else
				{
					receiverInfo = new ReceiverInfo();
					receiverInfos.put(group1, receiverInfo);
				}
				String key = matcher.group(2).toLowerCase();
				if (!ReceiverConfigImpl.fillDataStrategies.containsKey(key))
				{
					throw new Exception(String.format("未知的配置，请检查节点%s是否配置正确", text));
				}

				FillData temp = ReceiverConfigImpl.fillDataStrategies.get(key);
				boolean result = temp.fill(text, key, entry.getValue(), receiverInfo);
				if (!result)
				{
					throw new Exception(String.format("错误的配置，请检查节点%s是否配置正确", text));
				}
			}
		}

		ArrayList<ReceiverInfo> result = new ArrayList<ReceiverInfo>(receiverInfos.values());
		return result;
	}
}
