package cicada.filesystem.local;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URI;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.text.SimpleDateFormat;
import java.nio.file.WatchEvent.Kind;
import java.nio.file.WatchEvent.Modifier;
import java.util.Date;
import java.util.Iterator;
import java.util.Map;
import java.util.UUID;

import org.apache.log4j.Logger;

import cicada.core.Guard;
import cicada.filesystem.FileSystem;
import cicada.filesystem.distributed.FileSystemImplDistributed;

public class FileSystemImplLocal implements FileSystem
{

	private static final Logger log = Logger.getLogger(FileSystemImplLocal.class);

	private final String rootConfigName = "Cicada.FileSystem.Local.Root";

	private String _root;

	public FileSystemImplLocal(Map<String, String> configurationDataRespository) throws Exception
	{
		this._root = configurationDataRespository.get(rootConfigName);
		if (_root == null || _root.isEmpty())
		{
			throw new Exception(String.format("请为本地文件系统设置根目录，请配置%s节点", rootConfigName));
		}

		String projectRoot = System.getProperty("user.dir");
		StringBuffer sb = new StringBuffer();
		_root = sb.append(projectRoot).append(File.separator).append(_root).toString();
	}

	@Override
	public String upload(byte[] data, String fileExt) throws Exception
	{
		String result = this.upload(data, fileExt, 0);
		return result;
	}

	@Override
	public String upload(byte[] data, String fileExt, int resultPathType) throws Exception
	{
		Guard.ThrowIfArgumentIsNull(data, "data");
		Guard.ThrowIfArgumentIsNullOrEmpty(fileExt, "fileExt");
		SimpleDateFormat sdf = new SimpleDateFormat(String.format("yyyy%sMM%sdd", File.separator, File.separator));
		String path = sdf.format(new Date());
		path = _root + File.separator + path;
		File filePath = new File(path);
		if (!filePath.exists())
		{
			if (!filePath.mkdirs()) return null;
		}

		String uuid = UUID.randomUUID().toString().replace("-", "");
		String fileName = uuid + "." + fileExt;
		fileName = path + File.separator + fileName;
		File temp = new File(fileName);
		boolean result = temp.createNewFile();
		if (!result) return null;

		BufferedOutputStream stream = null;
		try
		{
			stream = new BufferedOutputStream(new FileOutputStream(temp));
			stream.write(data);
		}
		catch (Exception e)
		{
			log.error(e.getStackTrace());
			e.printStackTrace();
		}
		finally
		{
			stream.close();
		}
		if (resultPathType == 0) return fileName.replace(_root, "");
		return fileName;
	}

	@Override
	public byte[] download(String fileName) throws Exception
	{
		Guard.ThrowIfArgumentIsNull(fileName, "fileName");
		String physicsPath = this._root + File.separator + fileName;
		File file = new File(physicsPath);
		if (file.exists())
		{
			ByteArrayOutputStream bos = new ByteArrayOutputStream((int) file.length());
			BufferedInputStream in = null;
			try
			{
				in = new BufferedInputStream(new FileInputStream(file));
				int buf_size = 1024;
				byte[] buffer = new byte[buf_size];
				int len = 0;
				while (-1 != (len = in.read(buffer, 0, buf_size)))
				{
					bos.write(buffer, 0, len);
				}
				return bos.toByteArray();
			}
			catch (IOException e)
			{
				e.printStackTrace();
			}
			finally
			{
				if (in != null) in.close();
				if (bos != null) bos.close();
			}
		}
		return null;
	}

	@Override
	public boolean remove(String fileName) throws Exception
	{
		Guard.ThrowIfArgumentIsNull(fileName, "fileName");
		String physicsPath = this._root + File.separator + fileName;
		File file = new File(physicsPath);
		if (!file.exists())
		{
			return true;
		}
		return file.delete();
	}

}
