package net.oschina.suyeer.fastwechat.util;

import com.alibaba.fastjson.JSONObject;
import net.oschina.suyeer.basic.util.BHttpResUtil;
import net.oschina.suyeer.basic.util.BJsonUtil;
import net.oschina.suyeer.cache.MemCachedUtil;
import net.oschina.suyeer.fastwechat.bean.base.CacheBean;
import net.oschina.suyeer.fastwechat.bean.base.JsSdkConfig;
import net.oschina.suyeer.fastwechat.bean.base.TicketBean;
import net.oschina.suyeer.fastwechat.module.WeChatDataCache;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.TreeMap;

/**
 * Created by jun on 2017/8/11.
 */
public class FwTokenUtil {
    private static Logger logger = LoggerFactory.getLogger(FwBaseUtil.class);

    private static TicketBean jsApiTicket = new TicketBean(ConstUtil.IF_MEMCACHED);

    public static String getAccessToken() {
        if (ConstUtil.IF_MEMCACHED) {
            return getAccessTokenFromMC();
        }
        if (ConstUtil.IF_LOCALCACHE) {
            return getAccessTokenFromLocal();
        }
        return getAccessTokenFromWx();
    }

    public static String getJsApiTicket() {
        if (ConstUtil.IF_MEMCACHED) {
            return getJsApiTicketFromMC();
        }
        if (ConstUtil.IF_LOCALCACHE) {
            return getJsApiTicketFromLocal();
        }
        return getJsApiTicketFromWx();
    }

    private static String getAccessTokenFromWx() {
        try {
            return getFromWx(ConstUtil.WX_URL_CGI_BIN_ACCESS_TOKEN, ConstUtil.WX_PARAMETER_ACCESS_TOKEN);
        } catch (Exception e) {
            logger.error("{} 获取AccessToken失败, {}", e.getMessage());
            return null;
        }
    }

    private static String getJsApiTicketFromWx() {
        try {
            return getFromWx(ConstUtil.WX_URL_GET_JS_API_TICKET + getAccessToken(), ConstUtil.WX_PARAMETER_TICKET);
        } catch (Exception e) {
            logger.error("{} 获取JsApiTicket失败, {}", ConstUtil.ERROR, e.getMessage());
            return null;
        }
    }

    /**
     * User: jun
     * Date: 2017/8/12
     * Desc: 本地AccessToken缓存时间设为7000秒, 超过这个时间后, 则请求微信生成新的AccessToken数据;
     */
    private static String getAccessTokenFromLocal() {
        try {
            CacheBean cache = WeChatDataCache.getInstance().get(ConstUtil.WX_KEY_ACCESS_TOKEN);
            if (cache != null && cache.isNotExpired()) {
                return cache.getValue();
            }
            String value = getAccessTokenFromWx();
            if (value == null) {
                throw new Exception("与微信交互异常!");
            }
            WeChatDataCache.getInstance().add(ConstUtil.WX_KEY_ACCESS_TOKEN, value, 7000);
            return value;
        } catch (Exception e) {
            logger.error("{} 获取AccessToken失败, {}", ConstUtil.ERROR, e.getMessage());
            return null;
        }
    }

    /**
     * User: jun
     * Date: 2017/8/12
     * Desc: 考虑到两个小时内从微信获取的 JsApiTicket 数据是同一个,
     * 假设我们在某个时刻从微信获取了 JsApiTicket , 然后重启服务器, 此时我们
     * 只好再次从微信获取 JsApiTicket 并把数据放到了本地缓存, 但这个 JsApiTicket
     * 和之前的是同一个, 我们并不知道这个数据有效期还有多少时间.
     * <p>
     * 解决方案是:
     * 项目启动后获取的 第一个 JsApiTicket 数据不确定还有多少时间, 所以我们不将它放到
     * 缓存, 需要数据时直接请求微信服务器, 直到微信返回一个新的 JsApiTicket, 此时的
     * 数据肯定是 7200 有效时长的数据, 将其放到缓存这样保证存在缓存的数据是可控数据.
     * <p>
     * CacheBean 的缓存时间设为 7180秒 ,超过这个时间后,需要数据时直接请求微信服务器, 直到
     * 微信返回一个新的 JsApiTicket;
     */
    private static String getJsApiTicketFromLocal() {
        try {
            String value = getValue();
            if (jsApiTicket.isFirstValue()) {
                return value;
            }
            CacheBean cache = WeChatDataCache.getInstance().get(ConstUtil.WX_KEY_JS_API_TICKET);
            if (cache != null && cache.isNotExpired()) {
                return cache.getValue();
            }
            if (value == null) {
                value = getJsApiTicketFromWx();
            }
            if (value == null) {
                throw new Exception("与微信交互异常!");
            }
            if (cache != null && value.equals(cache.getValue())) {
                return value;
            }
            WeChatDataCache.getInstance().add(ConstUtil.WX_KEY_JS_API_TICKET, value, 7180);
            return value;
        } catch (Exception e) {
            logger.error("{} 获取JsApiTicket失败, {}", ConstUtil.ERROR, e.getMessage());
            return null;
        }
    }

    /**
     * User: jun
     * Date: 2017/8/12
     * Desc: memcached AccessToken缓存时间设为7000秒, 超过这个时间后, 则请求微信生成新的AccessToken数据;
     * 无需调用 CacheBean 的 isNotExpired 检查;
     */
    private static String getAccessTokenFromMC() {
        try {
            CacheBean cache = MemCachedUtil.get(ConstUtil.MC_WX_KEY_ACCESS_TOKEN);
            if (cache != null) {
                return cache.getValue();
            }
            String value = getAccessTokenFromWx();
            if (value == null) {
                throw new Exception("与微信交互异常!");
            }
            CacheBean newBean = new CacheBean(ConstUtil.MC_WX_KEY_ACCESS_TOKEN, value);
            MemCachedUtil.set(ConstUtil.MC_WX_KEY_ACCESS_TOKEN, newBean, 7000);
            return value;
        } catch (Exception e) {
            logger.error("{} 获取AccessToken失败, {}", ConstUtil.ERROR, e.getMessage());
            return null;
        }
    }

    /**
     * User: jun
     * Date: 2017/8/12
     * Desc: 考虑到两个小时内从微信获取的 JsApiTicket 数据是同一个,
     * 假设我们在某个时刻从微信获取了 JsApiTicket 并把数据放到了memcached,
     * 然后 memcached 在两小时内突然挂掉并重启, 此时我们只好再次从微信获取
     * JsApiTicket 并把数据放到了memcached, 但这个 JsApiTicket 和之前的
     * 是同一个, 我们并不知道这个数据有效期还有多少时间.
     * <p>
     * 解决方案是:
     * 若 memcached 不存在缓存, 此时获取的 JsApiTicket 数据是相对于 memcached
     * 的 key 为 MC_WX_KEY_JS_API_TICKET 的第一个数据, 但此数据因为不确定还有多少
     * 时间, 所以我们不将它放到memcached, 需要数据时直接请求微信服务器, 直到微信返回
     * 一个新的 JsApiTicket, 此时的数据肯定是 7200 有效时长的数据, 将其放到 memcached
     * 这样保证存在memcached的数据是可控数据.
     * <p>
     * memcached JsApiTicket缓存时间尽量长, 这里设置为 10 天;
     * CacheBean 的缓存时间设为 7000 ,超过这个时间后,需要数据时直接请求微信服务器, 直到
     * 微信返回一个新的 JsApiTicket直到微信生成新的JsApiTicket;
     */
    private static String getJsApiTicketFromMC() {
        try {
            String value = getValue();
            if (jsApiTicket.isFirstValue()) {
                return value;
            }
            CacheBean cache = MemCachedUtil.get(ConstUtil.MC_WX_KEY_JS_API_TICKET);
            if (cache != null && cache.isNotExpired()) {
                return cache.getValue();
            }
            if (value == null) {
                value = getJsApiTicketFromWx();
            }
            if (value == null) {
                throw new Exception("与微信交互异常!");
            }
            if (cache != null && value.equals(cache.getValue())) {
                return value;
            }
            if (cache == null) {
                jsApiTicket = new TicketBean();
                return value;
            }
            CacheBean newBean = new CacheBean(ConstUtil.MC_WX_KEY_ACCESS_TOKEN, value, 7000);
            MemCachedUtil.set(ConstUtil.MC_WX_KEY_JS_API_TICKET, newBean, 60 * 60 * 24 * 10);
            return value;
        } catch (Exception e) {
            logger.error("{} 获取JsApiTicket失败, {}", ConstUtil.ERROR, e.getMessage());
            return null;
        }
    }

    private static String getFromWx(String url, String key) throws Exception {
        JSONObject obj = BHttpResUtil.sendHttpGetRequest(url);
        if (obj == null || !obj.containsKey(key)) {
            throw new Exception(String.format("微信返回值错误: %s, 请求URL: %s", BJsonUtil.toString(obj), url));
        }
        return obj.getString(key);
    }

    public static JsSdkConfig getJsSdkConfig(String url) {
        JsSdkConfig jsSdkConfig = new JsSdkConfig();
        try {
            TreeMap<String, String> treeMap = new TreeMap<>();
            treeMap.put("noncestr", jsSdkConfig.getNonceStr());
            treeMap.put("jsapi_ticket", getJsApiTicket());
            treeMap.put("timestamp", jsSdkConfig.getTimestamp());
            treeMap.put("url", url);
            jsSdkConfig.setSignature(FwBaseUtil.createJsSdKSign(treeMap));
        } catch (Exception e) {
            logger.error("{} 获取JsSdk的配置信息JsSdkConfig失败：{}", ConstUtil.ERROR, e.getMessage());
        } finally {
            return jsSdkConfig;
        }
    }

    private static String getValue() throws Exception {
        String value = null;
        if (jsApiTicket.isFirstValue()) {
            value = getJsApiTicketFromWx();
            if (value == null) {
                throw new Exception("与微信交互异常!");
            }
            if (jsApiTicket.getValue() == null) {
                jsApiTicket.setValue(value);
            }
            jsApiTicket.setIsFirstValue(value.equals(jsApiTicket.getValue()));
            if (jsApiTicket.isFirstValue()) {
                jsApiTicket.setIsFirstValue(jsApiTicket.isTwoHours());
            }
        }
        return value;
    }
}
