package net.oschina.suyeer.fastwechat.util;

import com.alibaba.fastjson.JSONObject;
import net.oschina.suyeer.basic.util.BHttpResUtil;
import net.oschina.suyeer.basic.util.BJsonUtil;
import net.oschina.suyeer.cache.MemCachedUtil;
import net.oschina.suyeer.fastwechat.bean.fwcommon.CacheBean;
import net.oschina.suyeer.fastwechat.bean.fwcommon.JsSdkConfig;
import net.oschina.suyeer.fastwechat.module.WeChatDataCache;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.TreeMap;

/**
 * Created by jun on 2017/8/11.
 */
public class FwTokenUtil {
    private static Logger logger = LoggerFactory.getLogger(FwBaseUtil.class);
    private final static int CACHE_TIME = ConstUtil.ACCESS_TOKEN_ACTIVE_TIME;
    private final static int TWO_HOURS_SECOND = 7200;
    private final static int ONE_MIN = 1000 * 60;

    public static String getAccessToken() {
        if (ConstUtil.IF_MEMCACHED) {
            return getAccessTokenFromMC();
        }
        if (ConstUtil.IF_LOCALCACHE) {
            return getAccessTokenFromLocal();
        }
        return getAccessTokenFromWx();
    }

    public static String getJsApiTicket() {
        if (ConstUtil.IF_MEMCACHED || ConstUtil.IF_LOCALCACHE) {
            return getJsApiTicketFromLocal();
        }
        return getJsApiTicketFromWx();
    }

    private static String getAccessTokenFromWx() {
        try {
            return getFromWx(ConstUtil.WX_URL_CGI_BIN_ACCESS_TOKEN, ConstUtil.WX_PARAMETER_ACCESS_TOKEN);
        } catch (Exception e) {
            logger.error("{} 获取AccessToken失败, {}", ConstUtil.ERROR, e.getMessage());
            return null;
        }
    }

    private static String getJsApiTicketFromWx() {
        try {
            return getFromWx(ConstUtil.WX_URL_GET_JS_API_TICKET + getAccessToken(), ConstUtil.WX_PARAMETER_TICKET);
        } catch (Exception e) {
            logger.error("{} 获取JsApiTicket失败, {}", ConstUtil.ERROR, e.getMessage());
            return null;
        }
    }

    /**
     * User: jun
     * Date: 2017/8/12
     * Desc: 本地AccessToken缓存时间默认设为6000秒, 超过这个时间后, 则请求微信生成新的AccessToken数据;
     * 缓存时间可自主修改参数 accessTokenActiveTime
     */
    private static String getAccessTokenFromLocal() {
        try {
            CacheBean cache = WeChatDataCache.getInstance().get(ConstUtil.WX_KEY_ACCESS_TOKEN);
            if (cache != null && cache.isNotExpired()) {
                return cache.getValue();
            }
            String value = getAccessTokenFromWx();
            if (value == null) {
                throw new Exception("与微信服务器交互异常!");
            }
            WeChatDataCache.getInstance().add(ConstUtil.WX_KEY_ACCESS_TOKEN, value, CACHE_TIME);
            return value;
        } catch (Exception e) {
            logger.error("{} 获取AccessToken失败, {}", ConstUtil.ERROR, e.getMessage());
            return null;
        }
    }

    /**
     * User: jun
     * Date: 2017/8/12
     * Desc: JsApiTicket 在微信生成规则是, 从 00:00 开始, 每隔 2 小时更新一次.
     * 所以此数据只保存本地即可. 无需同步到 memcached.
     */
    private static String getJsApiTicketFromLocal() {
        try {
            CacheBean cache = WeChatDataCache.getInstance().get(ConstUtil.WX_KEY_JS_API_TICKET);
            if (cache != null && cache.isNotExpired()) {
                return cache.getValue();
            }
            String value = getJsApiTicketFromWx();
            if (value == null) {
                throw new Exception("与微信服务器交互异常!");
            }
            newThread4UpdateJsApiTicketCache(value);
            return value;
        } catch (Exception e) {
            logger.error("{} 获取JsApiTicket失败, {}", ConstUtil.ERROR, e.getMessage());
            return null;
        }
    }

    private static void newThread4UpdateJsApiTicketCache(String value) {
        new Thread(new Runnable() {
            @Override
            public void run() {
                refreshJsApiTicketFromWx(value, 0);
            }
        }).start();
    }

    private static void refreshJsApiTicketFromWx(String value, int errorTime) {
        try {
            if (errorTime < 3) {
                Long nowSec = FwBaseUtil.getTimestamp();
                int overSec = nowSec.intValue() % TWO_HOURS_SECOND;
                int remainderSec = TWO_HOURS_SECOND - overSec;
                boolean flag = (remainderSec < ConstUtil.TICKET_CACHE_REFRESH_TIME || overSec < ConstUtil.TICKET_CACHE_REFRESH_TIME);
                int cacheTime = flag ? ConstUtil.TICKET_CACHE_REFRESH_TIME : remainderSec - ConstUtil.TICKET_CACHE_REFRESH_TIME;
                if (value != null) {
                    WeChatDataCache.getInstance().add(ConstUtil.WX_KEY_JS_API_TICKET, value, cacheTime);
                }
                if (flag) {
                    Thread.sleep(ONE_MIN);
                    value = getJsApiTicketFromWx();
                    if (value == null) {
                        throw new Exception("与微信服务器交互异常!");
                    }
                    refreshJsApiTicketFromWx(value, errorTime);
                }
            }
        } catch (Exception e) {
            errorTime++;
            logger.error("{} 第 {} 次执行 refreshJsApiTicketFromWx 失败, {}", ConstUtil.ERROR, errorTime, e.getMessage());
            refreshJsApiTicketFromWx(value, errorTime);
        }
    }

    /**
     * User: jun
     * Date: 2017/8/12
     * Desc: memcached AccessToken缓存时间默认设为6000秒, 超过这个时间后, 则请求微信生成新的AccessToken数据;
     * 缓存时间可自主修改参数 accessTokenActiveTime
     * 无需调用 CacheBean 的 isNotExpired 检查;
     */
    private static String getAccessTokenFromMC() {
        try {
            CacheBean cache = MemCachedUtil.get(ConstUtil.MC_WX_KEY_ACCESS_TOKEN);
            if (cache != null) {
                return cache.getValue();
            }
            String value = getAccessTokenFromWx();
            if (value == null) {
                throw new Exception("与微信服务器交互异常!");
            }
            CacheBean newCache = new CacheBean(ConstUtil.MC_WX_KEY_ACCESS_TOKEN, value, CACHE_TIME);
            MemCachedUtil.set(ConstUtil.MC_WX_KEY_ACCESS_TOKEN, newCache, CACHE_TIME);
            return value;
        } catch (Exception e) {
            logger.error("{} 获取AccessToken失败, {}", ConstUtil.ERROR, e.getMessage());
            return null;
        }
    }

    private static String getFromWx(String url, String key) throws Exception {
        JSONObject obj = BHttpResUtil.sendHttpGetRequest(url);
        if (obj == null || !obj.containsKey(key)) {
            throw new Exception(String.format("微信返回值错误: %s, 请求URL: %s", BJsonUtil.toString(obj), url));
        }
        return obj.getString(key);
    }

    public static JSONObject getJsSdkConfig(String url) {
        JSONObject retObj = new JSONObject();
        try {
            JsSdkConfig jsSdkConfig = new JsSdkConfig();
            TreeMap<String, String> treeMap = new TreeMap<>();
            treeMap.put("noncestr", jsSdkConfig.getNonceStr());
            treeMap.put("jsapi_ticket", getJsApiTicket());
            treeMap.put("timestamp", jsSdkConfig.getTimestamp());
            treeMap.put("url", url);
            jsSdkConfig.setSignature(FwBaseUtil.createJsSdKSign(treeMap));
            retObj = BJsonUtil.changeType(jsSdkConfig, JSONObject.class);
        } catch (Exception e) {
            logger.error("{} 获取JsSdk的配置信息JsSdkConfig失败：{}", ConstUtil.ERROR, e.getMessage());
        } finally {
            return retObj;
        }
    }

}
