/**
 * 
 */
package jmind.core.cache.xmemcached;

import java.net.InetSocketAddress;
import java.util.Collection;
import java.util.Map;

import jmind.base.cache.ICache;
import jmind.base.lang.ExpireRecord;
import net.rubyeye.xmemcached.CASOperation;
import net.rubyeye.xmemcached.Counter;
import net.rubyeye.xmemcached.GetsResponse;
import net.rubyeye.xmemcached.exception.MemcachedException;
import net.rubyeye.xmemcached.utils.Protocol;

/**
 * memcached
 * @author wbxie
 * 2013-7-18
 */
public interface Memcached  extends ICache{
    public ExpireRecord getException();

    public <T> GetsResponse<T> gets(final String key);

    public <T> GetsResponse<T> gets(final String key, final long timeout);

    public <T> Map<String, GetsResponse<T>> gets(final Collection<String> keyCollections);

    public <T> Map<String, GetsResponse<T>> gets(final Collection<String> keyCollections, final long timeout);

    /**
     * Get value by key
     * 
     * @param <T>
     * @param key
     *            Key
     * @param timeout
     *            Operation timeout,if the method is not returned in this
     *            time,throw TimeoutException
     * @param transcoder
     *            The value's transcoder
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    <T> T get(String key, long timeout, Transcoder<T> transcoder);

    <T> T get(String key, long timeout);

    <T> T get(String key, Transcoder<T> transcoder);

    <T> T get(String key);

    /**
     * Bulk get items
     * 
     * @param <T>
     * @param keyCollections
     *            key collection
     * @param opTimeout
     *            opTimeout
     * @param transcoder
     *            Value transcoder
     * @return Exists items map
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    <T> Map<String, T> get(Collection<String> keyCollections, long opTimeout, Transcoder<T> transcoder);

    /**
     * @see #get(Collection, long, Transcoder)
     * @param <T>
     * @param keyCollections
     * @param transcoder
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    <T> Map<String, T> get(Collection<String> keyCollections, Transcoder<T> transcoder);

    /**
     * @see #get(Collection, long, Transcoder)
     * @param <T>
     * @param keyCollections
     * @return
     * @throws InterruptedException
     * @throws MemcachedException
     */
    <T> Map<String, T> get(Collection<String> keyCollections);

    /**
     * @see #get(Collection, long, Transcoder)
     * @param <T>
     * @param keyCollections
     * @param timeout
     * @return
     * @throws InterruptedException
     * @throws MemcachedException
     */
    <T> Map<String, T> get(Collection<String> keyCollections, long timeout);

    /**
     * Store key-value item to memcached
     * 
     * @param <T>
     * @param key
     *            stored key
     * @param exp
     *            An expiration time, in seconds. Can be up to 30 days. After 30
     *            days, is treated as a unix timestamp of an exact date.
     * @param value
     *            stored data
     * @param transcoder
     *            transocder
     * @param timeout
     *            operation timeout,in milliseconds
     * @return boolean result
     * @throws InterruptedException
     * @throws MemcachedException
     */
    <T> boolean set(String key, int exp, T value, Transcoder<T> transcoder, long timeout);

    /**
     * @see #set(String, int, Object, Transcoder, long)
     */
    boolean set(String key, int exp, Object value);

    /**
     * 默认一个月
     * 2013-7-22 
     * @param key
     * @param value
     * @return
     * @throws InterruptedException
     * @throws MemcachedException
     */
    boolean set(String key, Object value);

    /**
     * @see #set(String, int, Object, Transcoder, long)
     */
    boolean set(String key, int exp, Object value, long timeout);

    /**
     * @see #set(String, int, Object, Transcoder, long)
     */
    <T> boolean set(String key, int exp, T value, Transcoder<T> transcoder);

    /**
     * Store key-value item to memcached,doesn't wait for reply
     * 
     * @param <T>
     * @param key
     *            stored key
     * @param exp
     *            An expiration time, in seconds. Can be up to 30 days. After 30
     *            days, is treated as a unix timestamp of an exact date.
     * @param value
     *            stored data
     * @param transcoder
     *            transocder
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    void setWithNoReply(String key, int exp, Object value);

    /**
     * @see #setWithNoReply(String, int, Object, Transcoder)
     * @param <T>
     * @param key
     * @param exp
     * @param value
     * @param transcoder
     * @throws InterruptedException
     * @throws MemcachedException
     */
    <T> void setWithNoReply(String key, int exp, T value, Transcoder<T> transcoder);

    /**
     * Add key-value item to memcached, success only when the key is not exists
     * in memcached.
     * 
     * @param <T>
     * @param key
     * @param exp
     *            An expiration time, in seconds. Can be up to 30 days. After 30
     *            days, is treated as a unix timestamp of an exact date.
     * @param value
     * @param transcoder
     * @param timeout
     * @return boolean result
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    <T> boolean add(String key, int exp, T value, Transcoder<T> transcoder, long timeout);

    /**
     * @see #add(String, int, Object, Transcoder, long)
     * @param key
     * @param exp
     * @param value
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    boolean add(String key, int exp, Object value);

    /**
     * @see #add(String, int, Object, Transcoder, long)
     * @param key
     * @param exp
     * @param value
     * @param timeout
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    boolean add(String key, int exp, Object value, long timeout);

    /**
     * @see #add(String, int, Object, Transcoder, long)
     * 
     * @param <T>
     * @param key
     * @param exp
     * @param value
     * @param transcoder
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    <T> boolean add(String key, int exp, T value, Transcoder<T> transcoder);

    /**
     * Add key-value item to memcached, success only when the key is not exists
     * in memcached.This method doesn't wait for reply.
     * 
     * @param <T>
     * @param key
     * @param exp
     *            An expiration time, in seconds. Can be up to 30 days. After 30
     *            days, is treated as a unix timestamp of an exact date.
     * @param value
     * @param transcoder
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */

    void addWithNoReply(String key, int exp, Object value);

    /**
     * @see #addWithNoReply(String, int, Object, Transcoder)
     * @param <T>
     * @param key
     * @param exp
     * @param value
     * @param transcoder
     * @throws InterruptedException
     * @throws MemcachedException
     */
    <T> void addWithNoReply(String key, int exp, T value, Transcoder<T> transcoder);

    /**
     * Replace the key's data item in memcached,success only when the key's data
     * item is exists in memcached.This method will wait for reply from server.
     * 
     * @param <T>
     * @param key
     * @param exp
     *            An expiration time, in seconds. Can be up to 30 days. After 30
     *            days, is treated as a unix timestamp of an exact date.
     * @param value
     * @param transcoder
     * @param timeout
     * @return boolean result
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    <T> boolean replace(String key, int exp, T value, Transcoder<T> transcoder, long timeout);

    /**
     * @see #replace(String, int, Object, Transcoder, long)
     * @param key
     * @param exp
     * @param value
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    boolean replace(String key, int exp, Object value);

    /**
     * @see #replace(String, int, Object, Transcoder, long)
     * @param key
     * @param exp
     * @param value
     * @param timeout
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    boolean replace(String key, int exp, Object value, long timeout);

    /**
     * @see #replace(String, int, Object, Transcoder, long)
     * @param <T>
     * @param key
     * @param exp
     * @param value
     * @param transcoder
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    <T> boolean replace(String key, int exp, T value, Transcoder<T> transcoder);

    /**
     * Replace the key's data item in memcached,success only when the key's data
     * item is exists in memcached.This method doesn't wait for reply from
     * server.
     * 
     * @param <T>
     * @param key
     * @param exp
     *            An expiration time, in seconds. Can be up to 30 days. After 30
     *            days, is treated as a unix timestamp of an exact date.
     * @param value
     * @param transcoder
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    void replaceWithNoReply(String key, int exp, Object value);

    /**
     * @see #replaceWithNoReply(String, int, Object, Transcoder)
     * @param <T>
     * @param key
     * @param exp
     * @param value
     * @param transcoder
     * @throws InterruptedException
     * @throws MemcachedException
     */
    <T> void replaceWithNoReply(String key, int exp, T value, Transcoder<T> transcoder);

    /**
     * @see #append(String, Object, long)
     * @param key
     * @param value
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    boolean append(String key, Object value);

    /**
     * Append value to key's data item,this method will wait for reply
     * 
     * @param key
     * @param value
     * @param timeout
     * @return boolean result
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    boolean append(String key, Object value, long timeout);

    /**
     * Append value to key's data item,this method doesn't wait for reply.
     * 
     * @param key
     * @param value
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    void appendWithNoReply(String key, Object value);

    /**
     * @see #prepend(String, Object, long)
     * @param key
     * @param value
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    boolean prepend(String key, Object value);

    /**
     * Prepend value to key's data item in memcached.This method doesn't wait
     * for reply.
     * 
     * @param key
     * @param value
     * @return boolean result
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    boolean prepend(String key, Object value, long timeout);

    /**
     * Prepend value to key's data item in memcached.This method doesn't wait
     * for reply.
     * 
     * @param key
     * @param value
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    void prependWithNoReply(String key, Object value);

    /**
     * @see #cas(String, int, Object, Transcoder, long, long)
     * @param key
     * @param exp
     * @param value
     * @param cas
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    boolean cas(String key, int exp, Object value, long cas);

    /**
     *Cas is a check and set operation which means "store this data but only if
     * no one else has updated since I last fetched it."
     * 
     * @param <T>
     * @param key
     * @param exp
     *            An expiration time, in seconds. Can be up to 30 days. After 30
     *            days, is treated as a unix timestamp of an exact date.
     * @param value
     * @param transcoder
     * @param timeout
     * @param cas
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    <T> boolean cas(String key, int exp, T value, Transcoder<T> transcoder, long timeout, long cas);

    /**
     * @see #cas(String, int, Object, Transcoder, long, long)
     * @param key
     * @param exp
     * @param value
     * @param timeout
     * @param cas
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    boolean cas(String key, int exp, Object value, long timeout, long cas);

    /**
     * @see #cas(String, int, Object, Transcoder, long, long)
     * @param <T>
     * @param key
     * @param exp
     * @param value
     * @param transcoder
     * @param cas
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    <T> boolean cas(String key, int exp, T value, Transcoder<T> transcoder, long cas);

    /**
     *Cas is a check and set operation which means "store this data but only if
     * no one else has updated since I last fetched it."
     * 
     * @param <T>
     * @param key
     * @param exp
     *            An expiration time, in seconds. Can be up to 30 days. After 30
     *            days, is treated as a unix timestamp of an exact date.
     * @param operation
     *            CASOperation
     * @param transcoder
     *            object transcoder
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    <T> boolean cas(String key, int exp, CASOperation<T> operation, Transcoder<T> transcoder);

    /**
     *cas is a check and set operation which means "store this data but only if
     * no one else has updated since I last fetched it."
     * 
     * @param <T>
     * @param key
     * @param exp
     *            An expiration time, in seconds. Can be up to 30 days. After 30
     *            days, is treated as a unix timestamp of an exact date.
     * @param getsResponse
     *            gets method's result
     * @param operation
     *            CASOperation
     * @param transcoder
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    <T> boolean cas(String key, int exp, GetsResponse<T> getsResponse, CASOperation<T> operation,
            Transcoder<T> transcoder);

    /**
     * @see #cas(String, int, GetsResponse, CASOperation, Transcoder)
     * @param <T>
     * @param key
     * @param exp
     * @param getsResponse
     * @param operation
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    <T> boolean cas(String key, int exp, GetsResponse<T> getsResponse, CASOperation<T> operation);

    /**
     * @see #cas(String, int, GetsResponse, CASOperation, Transcoder)
     * @param <T>
     * @param key
     * @param getsResponse
     * @param operation
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    <T> boolean cas(String key, GetsResponse<T> getsResponse, CASOperation<T> operation);

    /**
     * @see #cas(String, int, GetsResponse, CASOperation, Transcoder)
     * @param <T>
     * @param key
     * @param exp
     * @param operation
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    <T> boolean cas(String key, int exp, CASOperation<T> operation);

    /**
     * @see #cas(String, int, GetsResponse, CASOperation, Transcoder)
     * @param <T>
     * @param key
     * @param operation
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    <T> boolean cas(String key, CASOperation<T> operation);

    /**
     * 
     * @param <T>
     * @param key
     * @param getsResponse
     * @param operation
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    <T> void casWithNoReply(String key, GetsResponse<T> getsResponse, CASOperation<T> operation);

    /**
     * cas noreply
     * 
     * @param <T>
     * @param key
     * @param exp
     * @param getsResponse
     * @param operation
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    <T> void casWithNoReply(String key, int exp, GetsResponse<T> getsResponse, CASOperation<T> operation);

    /**
     * @see #casWithNoReply(String, int, GetsResponse, CASOperation)
     * @param <T>
     * @param key
     * @param exp
     * @param operation
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    <T> void casWithNoReply(String key, int exp, CASOperation<T> operation);

    /**
     * @see #casWithNoReply(String, int, GetsResponse, CASOperation)
     * @param <T>
     * @param key
     * @param operation
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    <T> void casWithNoReply(String key, CASOperation<T> operation);

    /**
     * Set a new expiration time for an existing item
     * 
     * @param key
     *            item's key
     * @param exp
     *            New expiration time, in seconds. Can be up to 30 days. After
     *            30 days, is treated as a unix timestamp of an exact date.
     * @param opTimeout
     *            operation timeout
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    boolean touch(String key, int exp, long opTimeout);

    /**
     * Set a new expiration time for an existing item,using default opTimeout
     * second.
     * 
     * @param key
     *            item's key
     * @param exp
     *            New expiration time, in seconds. Can be up to 30 days. After
     *            30 days, is treated as a unix timestamp of an exact date.
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    boolean touch(String key, int exp);

    /**
     * Get item and set a new expiration time for it
     * 
     * @param <T>
     * @param key
     *            item's key
     * @param newExp
     *            New expiration time, in seconds. Can be up to 30 days. After
     *            30 days, is treated as a unix timestamp of an exact date.
     * @param opTimeout
     *            operation timeout
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    <T> T getAndTouch(String key, int newExp, long opTimeout);

    /**
     * Get item and set a new expiration time for it,using default opTimeout
     * 
     * @param <T>
     * @param key
     * @param newExp
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    <T> T getAndTouch(String key, int newExp);

    /**
     * "incr" are used to change data for some item in-place, incrementing it.
        * The data for the item is treated as decimal representation of a 64-bit
        * unsigned integer. If the current data value does not conform to such a
        * representation, the commands behave as if the value were 0. Also, the
        * item must already exist for incr to work; these commands won't pretend
        * that a non-existent key exists with value 0; instead, it will fail.This
        * method doesn't wait for reply.
        * 
        * @return the new value of the item's data, after the increment operation
        *         was carried out.
     * 2014-2-12 
     * @param key  
     * @param delta 增加值
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    long incr(String key, long delta);

    /**
     * 
     * 2014-2-12 
     * @param key 
     * @param delta 增加值
     * @param initValue 默认值
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */

    long incr(String key, long delta, long initValue);

    /**
     * "incr" are used to change data for some item in-place, incrementing it.
     * The data for the item is treated as decimal representation of a 64-bit
     * unsigned integer. If the current data value does not conform to such a
     * representation, the commands behave as if the value were 0. Also, the
     * item must already exist for incr to work; these commands won't pretend
     * that a non-existent key exists with value 0; instead, it will fail.This
     * method doesn't wait for reply.
     * 
     * @param key
     *            key
     * @param num
     *            increment
     * @param initValue
     *            initValue if the data is not exists.
     * @param timeout
     *            operation timeout
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    long incr(String key, long delta, long initValue, long timeout);

    /**
     * "incr" are used to change data for some item in-place, incrementing it.
     * The data for the item is treated as decimal representation of a 64-bit
     * unsigned integer. If the current data value does not conform to such a
     * representation, the commands behave as if the value were 0. Also, the
     * item must already exist for incr to work; these commands won't pretend
     * that a non-existent key exists with value 0; instead, it will fail.This
     * method doesn't wait for reply.
     * 
     * @param key
     *            key
     * @param delta
     *            increment delta
     * @param initValue
     *            the initial value to be added when value is not found
     * @param timeout
     *            operation timeout
     * @param exp
     *            the initial vlaue expire time, in seconds. Can be up to 30
     *            days. After 30 days, is treated as a unix timestamp of an
     *            exact date.
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    long incr(String key, long delta, long initValue, long timeout, int exp);

    /**
     * "decr" are used to change data for some item in-place, decrementing it.
     * The data for the item is treated as decimal representation of a 64-bit
     * unsigned integer. If the current data value does not conform to such a
     * representation, the commands behave as if the value were 0. Also, the
     * item must already exist for decr to work; these commands won't pretend
     * that a non-existent key exists with value 0; instead, it will fail.This
     * method doesn't wait for reply.
     * 
     * @return the new value of the item's data, after the decrement operation
     *         was carried out.
     * @param key
     * @param num
     * @throws InterruptedException
     * @throws MemcachedException
     */
    long decr(String key, long delta);

    /**
     * @see decr
     * @param key
     * @param num
     * @param initValue
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    long decr(String key, long delta, long initValue);

    /**
     * "decr" are used to change data for some item in-place, decrementing it.
     * The data for the item is treated as decimal representation of a 64-bit
     * unsigned integer. If the current data value does not conform to such a
     * representation, the commands behave as if the value were 0. Also, the
     * item must already exist for decr to work; these commands won't pretend
     * that a non-existent key exists with value 0; instead, it will fail.This
     * method doesn't wait for reply.
     * 
     * @param key
     *            The key
     * @param num
     *            The increment
     * @param initValue
     *            The initial value if the data is not exists.
     * @param timeout
     *            Operation timeout
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    long decr(String key, long delta, long initValue, long timeout);

    /**
     * "incr" are used to change data for some item in-place, incrementing it.
     * The data for the item is treated as decimal representation of a 64-bit
     * unsigned integer. If the current data value does not conform to such a
     * representation, the commands behave as if the value were 0. Also, the
     * item must already exist for incr to work; these commands won't pretend
     * that a non-existent key exists with value 0; instead, it will fail.This
     * method doesn't wait for reply.
     * 
     * @param key
     * @param delta
     * @param initValue
     *            the initial value to be added when value is not found
     * @param timeout
     * @param exp
     *            the initial vlaue expire time, in seconds. Can be up to 30
     *            days. After 30 days, is treated as a unix timestamp of an
     *            exact date.
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    long decr(String key, long delta, long initValue, long timeout, int exp);

    /**
     * "incr" are used to change data for some item in-place, incrementing it.
     * The data for the item is treated as decimal representation of a 64-bit
     * unsigned integer. If the current data value does not conform to such a
     * representation, the commands behave as if the value were 0. Also, the
     * item must already exist for incr to work; these commands won't pretend
     * that a non-existent key exists with value 0; instead, it will fail.This
     * method doesn't wait for reply.
     * 
     * @param key
     * @param num
     * @throws InterruptedException
     * @throws MemcachedException
     */
    void incrWithNoReply(String key, long delta);

    /**
     * "decr" are used to change data for some item in-place, decrementing it.
     * The data for the item is treated as decimal representation of a 64-bit
     * unsigned integer. If the current data value does not conform to such a
     * representation, the commands behave as if the value were 0. Also, the
     * item must already exist for decr to work; these commands won't pretend
     * that a non-existent key exists with value 0; instead, it will fail.This
     * method doesn't wait for reply.
     * 
     * @param key
     * @param num
     * @throws InterruptedException
     * @throws MemcachedException
     */
    void decrWithNoReply(String key, long delta);

    /**
     * Delete key's date item from memcached
     * 
     * @param key
     * @param opTimeout
     *            Operation timeout
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     * @since 1.3.2
     */
    boolean delete(String key, long opTimeout);

    boolean delete(String key);

    void deleteWithNoReply(String key);

    /**
     * Get counter for key,and if the key's value is not set,then set it with 0.
     * 
     * @param key
     * @return
     */
    Counter getCounter(String key);

    /**
     * Get counter for key,and if the key's value is not set,then set it with
     * initial value.
     * 
     * @param key
     * @param initialValue
     * @return
     */
    Counter getCounter(String key, long initialValue);

    Map<InetSocketAddress, String> getVersions(long timeout);

    /**
     * Get all connected memcached servers's version.
     * 
     * @return
     * @throws TimeoutException
     * @throws InterruptedException
     * @throws MemcachedException
     */
    Map<InetSocketAddress, String> getVersions();

    /**
     * Get stats from all memcached servers
     * 
     * @param timeout
     * @return server->item->value map
     * @throws MemcachedException
     * @throws InterruptedException
     * @throws TimeoutException
     */
    Map<InetSocketAddress, Map<String, String>> getStats(long timeout);

    Map<InetSocketAddress, Map<String, String>> getStats();

    /**
     * Get special item stats. "stats items" for example
     * 
     * @param item
     * @return
     */
    Map<InetSocketAddress, Map<String, String>> getStatsByItem(String itemName);

    Map<InetSocketAddress, Map<String, String>> getStatsByItem(String itemName, long timeout);

    <T> Transcoder<T> getTranscoder();

    /**
     * get operation timeout setting
     * 
     * @return
     */
    long getOpTimeout();

    /**
     * get avaliable memcached servers's socket address.
     * 
     * @return
     */
    Collection<InetSocketAddress> getAvaliableServers();

    void flushAllWithNoReply(int exptime);

    void flushAll(int exptime, long timeout);

    void flushAllWithNoReply(InetSocketAddress address, int exptime);

    void flushAll(InetSocketAddress address, long timeout, int exptime);

    /**
     * Return the default heal session interval in milliseconds
     * 
     * @return
     */
    long getHealSessionInterval();

    /**
     * Get the connect timeout
     * 
     * @param connectTimeout
     */
    long getConnectTimeout();

    Protocol getProtocol();

    boolean isSanitizeKeys();

    /**
     * @return
     */
    boolean isShutdown();

    /**
     * Return the cache instance name
     * 
     * @return
     */
    String getName();

}
