package jmind.core.cache.support;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;

/**
 * 自动过期内存计数器，线程安全,允许负数
 * @author wbxie
 * 2013-7-24
 */
public class LocalCounter implements Counter {

    private final Cache<String, AtomicInteger> cache;

    public LocalCounter(int maxSize, int duration, boolean access) {
        CacheBuilder<Object, Object> builder = CacheBuilder.newBuilder().maximumSize(maxSize).softValues();
        if (access)
            cache = builder.expireAfterAccess(duration, TimeUnit.MINUTES).build();
        else
            cache = builder.expireAfterWrite(duration, TimeUnit.MINUTES).build();

    }

    public AtomicInteger mustGet(String key) {
        AtomicInteger value = cache.getIfPresent(key);
        if (value == null) {
            synchronized (cache) {
                value = cache.getIfPresent(key);
                if (value == null) {
                    value = new AtomicInteger();
                    cache.put(key, value);
                }
            }

        }
        return value;
    }

    public boolean set(String key, Integer value) {
        cache.put(key, new AtomicInteger(value));
        return true;
    }

    public boolean delete(String key) {
        if (cache.getIfPresent(key) != null) {
            cache.invalidate(key);
            return true;
        } else
            return false;
    }

    @Override
    public Map<String, ?> getMulti(Collection<String> keys) {
        return null;
    }


    public Integer get(String key) {
        AtomicInteger integer = cache.getIfPresent(key);
        return integer == null ? 0 : integer.get();

    }



    public Object getCache() {
        return cache;
    }

    public void clear() {
        cache.invalidateAll();

    }

    @Override
    public int getAndInc(String key) {
        return mustGet(key).getAndIncrement();
    }

    @Override
    public int incAndGet(String key) {
        return mustGet(key).incrementAndGet();
    }

    @Override
    public int decrAndGet(String key) {

        return mustGet(key).decrementAndGet();
    }

    @Override
    public int getAndDecr(String key) {
        return mustGet(key).getAndDecrement();
    }

    @Override
    public int decr(String key, int val) {
        return mustGet(key).addAndGet(-val);
    }

    @Override
    public int inc(String key, int val) {
        return mustGet(key).addAndGet(val);
    }


    public boolean set(String key, int seconds, Integer value) {
        set(key, value);
        return true;
    }

    @Override
    public boolean exists(String key) {
        return cache.getIfPresent(key) != null;
    }

    @Override
    public boolean set(String key, Object value) {
        return false;
    }

    @Override
    public boolean set(String key, int seconds, Object value) {
        return false;
    }

}
