package jmind.core.cache.support;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.ImmutableMap;
import jmind.base.cache.ICache;

import java.util.Collection;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * google guava 缓存， 不允许null 建 和null 值
 * @author wbxie
 * 2013-8-15

 */
public class GuavaCache implements ICache {

    final Cache<String, Object> cache;

    public GuavaCache() {
        this(3000);
    }

    public GuavaCache(int maxSize) {
        cache = CacheBuilder.newBuilder().maximumSize(maxSize).softValues().build();
    }

    /**
     * 
     * @param maxSize 最大条数
     * @param duration 过期时间，单位分钟
     */
    public GuavaCache(int maxSize, int duration) {
        this(maxSize, duration, false);
    }

    public GuavaCache(int maxSize, int duration, boolean access) {
        CacheBuilder<Object, Object> builder = CacheBuilder.newBuilder().maximumSize(maxSize).softValues();
        if (access)
            cache = builder.expireAfterAccess(duration, TimeUnit.MINUTES).build();
        else
            cache = builder.expireAfterWrite(duration, TimeUnit.MINUTES).build();

    }

    @Override
    public boolean set(String key, Object value) {
        if (key != null && value != null)
            cache.put(key, value);
        return true;
    }

    @Override
    public boolean set(String key, int seconds, Object value) {
        set(key, value);
        return true;
    }

    @Override
    public boolean delete(String key) {
        if (cache.getIfPresent(key) != null)
            cache.invalidate(key);
        return true;
    }

    @Override
    public <V> V get(String key) {
        return (V)cache.getIfPresent(key);
    }

    @Override
    public Map<String, ?> getMulti(Collection<String> keys) {
        ImmutableMap<String, Object> allPresent = cache.getAllPresent(keys);
        return allPresent;

    }

    public String toString() {
        return cache.asMap().toString();
    }

    @Override
    public Object getCache() {

        return cache;
    }

    @Override
    public void clear() {
        cache.invalidateAll();

    }

    @Override
    public boolean exists(String key) {
        return cache.getIfPresent(key) != null;
    }

}
