package jmind.core.jdbc;

/*
 * Copyright 2002-2008 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;

import javax.sql.DataSource;

import org.springframework.dao.DataAccessException;
import org.springframework.dao.DataAccessResourceFailureException;
import org.springframework.jdbc.datasource.DataSourceUtils;
import org.springframework.jdbc.support.JdbcUtils;
import org.springframework.jdbc.support.incrementer.AbstractColumnMaxValueIncrementer;
import org.springframework.jdbc.support.incrementer.DataFieldMaxValueIncrementer;

/**
 * {@link DataFieldMaxValueIncrementer} that increments the maximum value of a given MySQL table
 * with the equivalent of an auto-increment column. Note: If you use this class, your MySQL
 * key column should <i>NOT</i> be auto-increment, as the sequence table does the job.
 *
 * <p>The sequence is kept in a table; there should be one sequence table per
 * table that needs an auto-generated key. The table type of the sequence table
 * should be MyISAM so the sequences are allocated without regard to any
 * transactions that might be in progress.
 *
 * <p>Example:
 *
 * <pre class="code">create table tab (id int unsigned not null primary key, text varchar(100));
 * create table tab_sequence (value int not null) type=MYISAM;
 * insert into tab_sequence values(0);</pre>
 *
 * If "cacheSize" is set, the intermediate values are served without querying the
 * database. If the server or your application is stopped or crashes or a transaction
 * is rolled back, the unused values will never be served. The maximum hole size in
 * numbering is consequently the value of cacheSize.
 * @author wbxie mysql 序列实现
 * 2014-3-12
 */
public class MySQLSeq extends AbstractColumnMaxValueIncrementer {

    private int increment = 1;

    private String query = "";

    /**
     * Convenience constructor.
     * @param dataSource the DataSource to use
     * @param incrementerName the name of the sequence/table to use
     * @param columnName the name of the column in the sequence table to use
     */
    public MySQLSeq(DataSource dataSource, String incrementerName, String columnName) {
        super(dataSource, incrementerName, columnName);
    }

    public MySQLSeq(DataSource dataSource, String incrementerName, String columnName, int increment, String query) {
        super(dataSource, incrementerName, columnName);
        this.increment = increment;
        this.query = query;
    }

    @Override
    protected synchronized long getNextKey() throws DataAccessException {
        long nextId = 0;
        /*
        * Need to use straight JDBC code because we need to make sure that the insert and select
        * are performed on the same connection (otherwise we can't be sure that last_insert_id()
        * returned the correct value)
        */
        Connection con = DataSourceUtils.getConnection(getDataSource());
        Statement stmt = null;
        try {
            stmt = con.createStatement();
            DataSourceUtils.applyTransactionTimeout(stmt, getDataSource());
            // Increment the sequence column...
            String sql = "update " + getIncrementerName() + " set " + getColumnName() + " = last_insert_id("
                    + getColumnName() + " + " + increment + ") " + query;

            stmt.executeUpdate(sql);

            // Retrieve the new max of the sequence column...
            ResultSet rs = stmt.executeQuery(BaseJdbc.LAST_ID_SQL);
            try {
                if (!rs.next()) {
                    throw new DataAccessResourceFailureException("last_insert_id() failed after executing an update");
                }
                nextId = rs.getLong(1);
            } finally {
                JdbcUtils.closeResultSet(rs);
            }

        } catch (SQLException ex) {
            throw new DataAccessResourceFailureException("Could not obtain last_insert_id()", ex);
        } finally {
            JdbcUtils.closeStatement(stmt);
            DataSourceUtils.releaseConnection(con, getDataSource());
        }
        return nextId;

    }

    public long getNextKey(String query) {
        return getNextKey(getIncrementerName(), query);
    }

    /**
     *  适用于分表分库
     * 2014-3-12 
     * @param tableName
     * @param query
     * @return
     * @throws DataAccessException
     */
    public synchronized long getNextKey(String tableName, String query) throws DataAccessException {
        long nextId = 0;
        Connection con = DataSourceUtils.getConnection(getDataSource());
        Statement stmt = null;
        try {
            stmt = con.createStatement();
            DataSourceUtils.applyTransactionTimeout(stmt, getDataSource());
            // Increment the sequence column...
            String sql = "update " + tableName + " set " + getColumnName() + " = last_insert_id(" + getColumnName()
                    + " + " + increment + ") " + query;

            stmt.executeUpdate(sql);

            // Retrieve the new max of the sequence column...
            ResultSet rs = stmt.executeQuery(BaseJdbc.LAST_ID_SQL);
            try {
                if (!rs.next()) {
                    throw new DataAccessResourceFailureException("last_insert_id() failed after executing an update");
                }
                nextId = rs.getLong(1);
            } finally {
                JdbcUtils.closeResultSet(rs);
            }

        } catch (SQLException ex) {
            throw new DataAccessResourceFailureException("Could not obtain last_insert_id()", ex);
        } finally {
            JdbcUtils.closeStatement(stmt);
            DataSourceUtils.releaseConnection(con, getDataSource());
        }
        return nextId;

    }

}
