package jmind.core.cache.xmemcached;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import jmind.base.lang.IProperties;
import jmind.base.util.DataUtil;
import net.rubyeye.xmemcached.CommandFactory;
import net.rubyeye.xmemcached.MemcachedClient;
import net.rubyeye.xmemcached.MemcachedClientBuilder;
import net.rubyeye.xmemcached.MemcachedSessionLocator;
import net.rubyeye.xmemcached.XMemcachedClientBuilder;
import net.rubyeye.xmemcached.auth.AuthInfo;
import net.rubyeye.xmemcached.command.BinaryCommandFactory;
import net.rubyeye.xmemcached.command.KestrelCommandFactory;
import net.rubyeye.xmemcached.command.TextCommandFactory;
import net.rubyeye.xmemcached.impl.ArrayMemcachedSessionLocator;
import net.rubyeye.xmemcached.impl.ElectionMemcachedSessionLocator;
import net.rubyeye.xmemcached.impl.KetamaMemcachedSessionLocator;
import net.rubyeye.xmemcached.impl.LibmemcachedMemcachedSessionLocator;
import net.rubyeye.xmemcached.transcoders.SerializingTranscoder;
import net.rubyeye.xmemcached.transcoders.TokyoTyrantTranscoder;
import net.rubyeye.xmemcached.transcoders.WhalinTranscoder;
import net.rubyeye.xmemcached.transcoders.WhalinV1Transcoder;
import net.rubyeye.xmemcached.utils.AddrUtil;

/**
 * http://www.blogjava.net/killme2008/archive/2014/02/13/325564.html
 * 构造一个MemcachedClient
 * @author wbxie
 * https://code.google.com/p/xmemcached/wiki/User_Guide_zh
 * http://www.blogjava.net/killme2008/archive/2012/09/11/325564.html
 * http://www.blogjava.net/killme2008/archive/2009/03/10/258838.html
 *     结果如下表格，命中率一行表示增加节点后的命中率情况（增加前为100%），后续的行表示各个节点存储的单词数,CRC32_HASH表示采用CRC32散列函数，KETAMA_HASH是基于md5的散列函数也是默认情况下一致性哈希的推荐算法，FNV1_32_HASH就是FNV 32位散列函数，NATIVE_HASH就是java.lang.String.hashCode()方法返回的long取32位的结果，MYSQL_HASH是xmemcached添加的传说来自于mysql源码中的哈希函数。
 * 1、命中率最高看起来是NATIVE_HASH，然而NATIVE_HASH情况下数据集中存储在第一个节点，显然没有实际使用价值。为什么会集中存储在第一个节点呢？这是由于在查找存储的节点的过程中，会比较hash(key)和hash(节点IP地址)，而在采用了NATIVE_HASH的情况下，所有连接的hash值会呈现一个递增状况（因为String.hashCode是乘法散列函数），如：
192.168.0.100:12000 736402923
192.168.0.100:12001 736402924
192.168.0.100:12002 736402925
192.168.0.100:12003 736402926
如果这些值很大的会，那么单词的hashCode()会通常小于这些值的第一个，那么查找就经常只找到第一个节点并存储数据，当然，这里有测试的局限性，因为memcached都跑在一个台机器上只是端口不同造成了hash(节点IP地址)的连续递增，将分布不均匀的问题放大了。

2、从结果上看，KETAMA_HASH维持了一个最佳平衡，在增加两个节点后还能访问到83.3%的单词，并且数据分布在各个节点上的数目也相对平均，难怪作为默认散列算法。

3、最后，单纯比较下散列函数的计算效率：

CRC32_HASH:3266
KETAMA_HASH:7500
FNV1_32_HASH:375
NATIVE_HASH:187
MYSQL_HASH:500

   NATIVE_HASH > FNV1_32_HASH > MYSQL_HASH > CRC32_HASH > KETAMA_HASH
 *
 */
public class MemcachedBuilder {

    private static final String MEMCACHED_PROFIX = "memcached.";
    private static final String HOSTS = "hosts";// 10.22.32.143:65000 10.22.32.152:65000 10.22.32.144:65000 10.22.32.145:65000 10.22.32.165:65000 10.22.32.167:65000 10.22.32.168:65000 10.22.32.146:65000 10.22.32.147:65000 10.22.32.154:65000
    private static final String USERNAME = "username";//
    private static final String PASSWORD = "password";//
    private static final String SESSION_LOCATOR = "shard.hashingAlgorithm";// KETAMA

    private static final String COMMAND_FACTORY = "useBinaryCommand"; // true
    //  private static final String BUFFER_ALLOCATOR = "bufferAllocator"; // SIMPLE
    private static final String POOL_SIZE = "pool.size";//5
    private static final String POOL_FAILURE_MODE = "pool.failureMode";//false
    private static final String TRANSCODER = "transcoder";// SERIALIZING
    private static final String TRANSCODER_COMPRESSION_THRESHOLD = "transcoder.compressionThreshold";// 16384
    private static final String TRANSCODER_PACK_ZEROS = "transcoder.packZeros";// true
    private static final String TRANSCODER_PRIMITIVE_AS_STRING = "primitiveAsString";// false
    private static final String SANITIZE_KEYS = "sanitizeKeys";// false
    private static final String JMX_ENABLE = "jmx.enable"; // false
    private static final String JMX_RMI_PORT = "jmx.rmiPort";//10001
    private static final String JMX_RMI_NAME = "jmx.rmiName";//memcached

    private MemcachedClient clientDelegate;
    private final IProperties prop;
    private final String name;

    public MemcachedBuilder(IProperties properties, String name) {
        this.prop = properties;
        this.name = name + ".";
        if (Boolean.parseBoolean(getProperty("simple", "true")))
            initDefault();
        else
            initClient();
    }

    public final MemcachedClient client() {
        return clientDelegate;
    }

    private void initDefault() {
        final String hosts = this.getProperty(HOSTS, "127.0.0.1:11211");
        MemcachedClientBuilder builder = new XMemcachedClientBuilder(AddrUtil.getAddresses(hosts));
        builder.setConnectionPoolSize(DataUtil.toInt(getProperty(POOL_SIZE, "1")));
        builder.setOpTimeout(Long.parseLong(this.getProperty("timeout", MemcachedClient.DEFAULT_OP_TIMEOUT + "")));
        System.out.println(name + " init mem hosts=" + hosts + ",poolsize=" + getProperty(POOL_SIZE, "1"));
        try {
            clientDelegate = builder.build();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    private void initClient() {
        //  MemcachedConnector 连接管理，心跳检查 MemcachedHandler
        // nio  com.google.code.yanf4j.util.SystemUtils  sun.nio.ch.EPollSelectorProvider
        this.initJmx();
        MemcachedClientBuilder builder = this.initBuilder();

        //   builder.setBufferAllocator(this.parseBufferAllocator());

        /**
         *  协议工厂，net.rubyeye.xmemcached.command.BinaryCommandFactory,TextCommandFactory(默认),KestrelCommandFactory
         */
        builder.setCommandFactory(this.parseCommandFactory()); // use text protocol 设置为文本数据传输

        // nio 1个就可以，高并发可是适当增加 1~30 就行，不要太大
        builder.setConnectionPoolSize(Integer.parseInt(this.getProperty(POOL_SIZE, "5")));
        //http://blog.csdn.net/mrliu20082009/article/details/7072427
        // 这是由于xmemcached的通讯层是基于非阻塞IO的，那么在请求发送给memcached之后，需要等待应答的到来，这个等待时间默认是5秒，如果 超过1秒就抛出java.util.TimeoutExpcetion给用户。
        builder.setOpTimeout(Long.parseLong(this.getProperty("timeout", MemcachedClient.DEFAULT_OP_TIMEOUT + "")));
        /**
         * 从1.3版本开始，xmemcached支持failure模式。所谓failure模式是指，
         * 当一个memcached节点down掉的时候，发往这个节点的请求将直接失败，
         * 而不是发送给下一个有效的memcached节点。具体可以看memcached的文档。默认不启用failure模式，
         */
        builder.setFailureMode(Boolean.parseBoolean(this.getProperty(POOL_FAILURE_MODE, "false")));
        /**
         * Memcached的分布是通过客户端实现的，客户端根据key的哈希值得到将要存储的memcached节点，并将对应的value存储到相应的节点。

        XMemcached同样支持客户端的分布策略，默认分布的策略是按照key的哈希值模以连接数得到的余数，
        对应的连接就是将要存储的节点。如果使用默认的分布策略，你不需要做任何配置或者编程。
         */
        builder.setSessionLocator(this.parseSessionLoactor());
        builder.getConfiguration().setSoTimeout(5000);
        // builder.getConfiguration().setSessionIdleTimeout(sessionIdleTimeout);
        // builder.getConfiguration().setSessionReadBufferSize(tcpHandlerReadBufferSize);
        // builder.getConfiguration().setCheckSessionTimeoutInterval(checkSessionTimeoutInterval);
        //        builder.setSocketOption(StandardSocketOption.TCP_NODELAY, false); // 启用nagle算法，提高吞吐量，默认关闭
        //        builder.getConfiguration().setStatisticsServer(false);
        builder.setTranscoder(this.parseTranscoder());
        /**
         * memcached存储大数据的效率是比较低的，当数据比较大的时候xmemcached会帮你压缩在存储，
         * 取出来的时候自动解压并反序列化，这个大小阀值默认是16K，
         * 可以通过Transcoder接口的setCompressionThreshold(1.2.1引入)方法修改阀值，比如设置为1K：
         */
        builder.getTranscoder().setCompressionThreshold(
                Integer.parseInt(this.getProperty(TRANSCODER_COMPRESSION_THRESHOLD, "16384")));
        /**
         * XMemcached的序列化转换器在序列化数值类型的时候有个特殊处理，如果前面N个字节都是0，
         * 那么将会去除这些0，缩减后的数据将更小，例如数字3序列化是0x0003，那么前面3个0将去除掉成一个字节0x3。
         * 反序列化的时候将自动在前面根据数值类型补0。这一特性默认是开启的，如果考虑到与其他client兼容的话需要关闭此特性可以通过：
         */
        builder.getTranscoder().setPackZeros(Boolean.parseBoolean(this.getProperty(TRANSCODER_PACK_ZEROS, "true")));
        /**
         * 关于最后一点需要补充说明，由于kestrel不支持flag，因此xmemcached在存储的数据之前加了4个字节的flag，
         * 如果你的全部应用都使用xmemcached，那么没有问题，如果使用其他clients，会有兼容性的问题，因此Xmemcached还允许关闭这个功能，通过

        client.setPrimitiveAsString(true);
        设置为true后，原生类型都将存储为字符串，而序列化类型将无法存储了。
         */
        builder.getTranscoder().setPrimitiveAsString(
                Boolean.parseBoolean(this.getProperty(TRANSCODER_PRIMITIVE_AS_STRING, "false")));

        try {
            this.clientDelegate = builder.build();
            /**
             * 在官方客户端有提供一个sanitizeKeys选项，当选择用URL当key的时候，MemcachedClient会自动将URL encode再存储。默认是关闭的
             */
            this.clientDelegate.setSanitizeKeys(Boolean.parseBoolean(this.getProperty(SANITIZE_KEYS, "false")));

        } catch (final IOException e) {
            e.printStackTrace();
        }
    }

    private String getProperty(String key, String defaultVal) {
        return prop.getProperty(MEMCACHED_PROFIX + name + key, defaultVal);
    }

    private void initJmx() {
        final Boolean enable = Boolean.parseBoolean(this.getProperty(JMX_ENABLE, "false"));
        final Integer rmiPort = Integer.parseInt(this.getProperty(JMX_RMI_PORT, "10001"));
        final String rmiName = this.getProperty(JMX_RMI_NAME, "memcached");

        if (enable)
            System.setProperty("xmemcached.jmx.enable", enable.toString());

        System.setProperty("xmemcached.rmi.port", rmiPort.toString());
        System.setProperty("xmemcached.rmi.name", rmiName);
    }

    private MemcachedClientBuilder initBuilder() {

        final String hosts = this.getProperty(HOSTS, "127.0.0.1:11211");
        System.out.println(name + " init mem hosts=" + hosts);
        final Map<InetSocketAddress, InetSocketAddress> addressMap = this.parseAddress(hosts);
        final int[] weights = this.parseHostWeights(hosts, addressMap.size());
        final XMemcachedClientBuilder builder = new XMemcachedClientBuilder(addressMap, weights);

        // AuthInfo
        final String username = this.getProperty(USERNAME, null);
        final String password = this.getProperty(PASSWORD, null);
        if (username != null && username.trim().length() > 0 && password != null && password.trim().length() > 0) {
            for (final InetSocketAddress address : addressMap.keySet()) {
                final AuthInfo authInfo = AuthInfo.typical(username, password);
                builder.addAuthInfo(address, authInfo);
            }
        }

        return builder;
    }

    /**
     * @return
     */
    private MemcachedSessionLocator parseSessionLoactor() {
        final String sessionLocatorName = this.getProperty(SESSION_LOCATOR, "KETAMA").toUpperCase();
        if ("KETAMA".equals(sessionLocatorName)) { // 一致性hash
            return new KetamaMemcachedSessionLocator();
        } else if ("MEMCACHED".equals(sessionLocatorName)) {
            return new LibmemcachedMemcachedSessionLocator();
        } else if ("ELECTION".equals(sessionLocatorName)) {
            return new ElectionMemcachedSessionLocator();
        }
        return new ArrayMemcachedSessionLocator();
    }

    //    private BufferAllocator parseBufferAllocator() {
    //        final String bufferAllocatorName = this.getProperty(BUFFER_ALLOCATOR, "SIMPLE").toUpperCase();
    //        if ("SIMPLE".equals(bufferAllocatorName)) {
    //            return new SimpleBufferAllocator();
    //        } else if ("CACHED".equals(bufferAllocatorName)) {
    //            return new CachedBufferAllocator();
    //        }
    //        return null;
    //    }

    private CommandFactory parseCommandFactory() {
        final Boolean useBinaryCommand = Boolean.parseBoolean(this.getProperty(COMMAND_FACTORY, "false"));

        final String commandFactoryName = useBinaryCommand ? "BINARY" : "TEXT";
        if ("BINARY".equals(commandFactoryName)) {
            return new BinaryCommandFactory();
        } else if ("TEXT".equals(commandFactoryName)) {
            return new TextCommandFactory();
        } else if ("KESTREL".equals(commandFactoryName)) {
            return new KestrelCommandFactory();
        }
        return null;
    }

    /**
     * @return
     */
    private net.rubyeye.xmemcached.transcoders.Transcoder<?> parseTranscoder() {
        final String sessionLocatorName = this.getProperty(TRANSCODER, "SERIALIZING").toUpperCase();
        if ("SERIALIZING".equals(sessionLocatorName)) {
            return new SerializingTranscoder();
        } else if ("TT".equals(sessionLocatorName)) {
            return new TokyoTyrantTranscoder();
        } else if ("WHALIN".equals(sessionLocatorName)) {
            return new WhalinTranscoder();
        } else if ("WHALIN_V1".equals(sessionLocatorName)) {
            return new WhalinV1Transcoder();
        }
        return null;
    }

    /**
     * destory memcached clientDelegate
     */
    public final void destory() {
        if (this.clientDelegate != null && !this.clientDelegate.isShutdown()) {
            try {
                this.client().shutdown();
            } catch (final IOException e) {
                e.printStackTrace();
            }
            this.clientDelegate = null;
        }
    }

    private Map<InetSocketAddress, InetSocketAddress> parseAddress(final String hosts) {
        final String hostsWithoutWeights = hosts.replaceAll("\\*\\d+", "");
        // hosts
        final List<InetSocketAddress> addressList = AddrUtil.getAddresses(hostsWithoutWeights);
        final Map<InetSocketAddress, InetSocketAddress> addressMap = new LinkedHashMap<InetSocketAddress, InetSocketAddress>();
        if (addressList != null) {
            for (final InetSocketAddress addr : addressList) {
                addressMap.put(addr, null);
            }
        }
        return addressMap;
    }

    /**
     * @param hosts
     * @param expectedSize
     * @return
     */
    private int[] parseHostWeights(final String hosts, final int expectedSize) {
        // host weights
        final List<Integer> weighList = new ArrayList<Integer>();
        final String regex = "\\*(\\d+)";
        final Pattern pattern = Pattern.compile(regex);
        final Matcher matcher = pattern.matcher(hosts);
        while (matcher.find()) {
            weighList.add(Integer.parseInt(matcher.group(1)));
        }

        final Integer[] weights;
        if (expectedSize == weighList.size()) {
            weights = weighList.toArray(new Integer[weighList.size()]);
        } else {
            weights = null;
        }
        return toPrimitive(weights);
    }

    // Int array converters
    // ----------------------------------------------------------------------
    /**
     * <p>Converts an array of object Integers to primitives.</p>
     *
     * <p>This method returns <code>null</code> for a <code>null</code> input array.</p>
     *
     * @param array  a <code>Integer</code> array, may be <code>null</code>
     * @return an <code>int</code> array, <code>null</code> if null array input
     * @throws NullPointerException if array content is <code>null</code>
     */
    private static int[] toPrimitive(Integer[] array) {
        if (array == null) {
            return null;
        } else if (array.length == 0) {
            return new int[0];
        }
        final int[] result = new int[array.length];
        for (int i = 0; i < array.length; i++) {
            result[i] = array[i].intValue();
        }
        return result;
    }

}
