package jmind.core.cache.support;

import java.util.Collection;
import java.util.Map;
import java.util.Map.Entry;

import jmind.base.lang.ExpireRecord;
import jmind.base.util.GlobalConstants;
import net.rubyeye.xmemcached.GetsResponse;
import net.rubyeye.xmemcached.MemcachedClient;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Maps;

/**
 * memcache 缓存，计数实现
 * memcache计数不允许负数。 最小值0
 * @author wbxie
 * 2014-2-12
 * @param <K>
 * @param <V>
 */
public class XMemCache implements Counter<String, Object> {
    private final MemcachedClient cache;
    private final int expSecond;
    private final Logger logger = LoggerFactory.getLogger(getClass());

    public XMemCache(MemcachedClient cache) {
        this(cache, GlobalConstants.WEEK);

    }

    public XMemCache(MemcachedClient cache, int expSecond) {
        this.cache = cache;
        this.expSecond = expSecond;

    }

    public final ExpireRecord RECORD = new ExpireRecord(GlobalConstants.MINUTE * 5);

    private final void err(String key, Exception e) {
        logger.error("get key=" + key, e);
        RECORD.ExceptionRecord(e);
    }

    @Override
    public boolean set(String key, Object value) {
        try {
            if (value != null)
                return cache.set(key, expSecond, value);
        } catch (Exception e) {
            logger.error("set key=" + key, e);

        }
        return false;
    }

    @Override
    public boolean set(String key, int seconds, Object value) {
        try {
            if (value != null)
                return cache.set(key, seconds, value);
        } catch (Exception e) {
            logger.error("set key=" + key, e);
        }
        return false;
    }

    @Override
    public boolean delete(String key) {
        try {
            return cache.delete(key);
        } catch (Exception e) {
            logger.error("delete key=" + key, e);
        }
        return false;
    }

    @Override
    public Object get(String key) {
        try {
            return cache.get(key);
        } catch (Exception e) {
            err(key, e);
        }
        return null;
    }

    @Override
    public Map<String, Object> getMulti(Collection<String> keys) {
        Map<String, Object> maps = Maps.newLinkedHashMap();
        try {
            Map<String, GetsResponse<Object>> gets = cache.gets(keys);
            for (Entry<String, GetsResponse<Object>> entry : gets.entrySet()) {
                maps.put(entry.getKey(), entry.getValue().getValue());
            }
        } catch (Exception e) {
            logger.error("getMulti", e);
        }
        return maps;
    }

    @Override
    public Object getCache() {
        return cache;
    }

    @Override
    public void clear() {
    }

    @Override
    public int getAndInc(String key) {
        return inc(key, 1) - 1;
    }

    @Override
    public int incAndGet(String key) {
        return inc(key, 1);
    }

    @Override
    public int inc(String key, int vaule) {
        try {
            return (int) cache.incr(key, vaule, vaule);
        } catch (Exception e) {
            logger.error("inc key=" + key, e);
        }
        return 0;
    }

    @Override
    public int decrAndGet(String key) {
        return decr(key, 1);
    }

    @Override
    public int getAndDecr(String key) {
        return decr(key, 1) + 1;
    }

    @Override
    public int decr(String key, int value) {
        try {
            return (int) cache.decr(key, value, value);
        } catch (Exception e) {
            logger.error("decr key=" + key, e);
        }
        return 0;
    }

    @Override
    public boolean exists(String key) {
        return get(key) != null;
    }

}
