package jmind.core.cache.support;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;

/**
 * 自动过期内存计数器，线程安全,允许负数
 * @author wbxie
 * 2013-7-24
 * @param <K>
 */
public class LocalCounter<K> implements Counter<K, Integer> {

    private final Cache<K, AtomicInteger> cache;

    public LocalCounter(int maxSize, int duration, boolean access) {
        CacheBuilder<Object, Object> builder = CacheBuilder.newBuilder().maximumSize(maxSize).softValues();
        if (access)
            cache = builder.expireAfterAccess(duration, TimeUnit.MINUTES).build();
        else
            cache = builder.expireAfterWrite(duration, TimeUnit.MINUTES).build();

    }

    public AtomicInteger mustGet(K key) {
        AtomicInteger value = cache.getIfPresent(key);
        if (value == null) {
            synchronized (cache) {
                value = cache.getIfPresent(key);
                if (value == null) {
                    value = new AtomicInteger();
                    cache.put(key, value);
                }
            }

        }
        return value;
    }

    public boolean set(K key, Integer value) {
        cache.put(key, new AtomicInteger(value));
        return true;
    }

    public boolean delete(K key) {
        if (cache.getIfPresent(key) != null) {
            cache.invalidate(key);
            return true;
        } else
            return false;
    }

    public Integer get(K key) {
        AtomicInteger integer = cache.getIfPresent(key);
        return integer == null ? 0 : integer.get();

    }

    public Map<K, Integer> getMulti(Collection<K> keys) {
        Map<K, Integer> map = new HashMap<K, Integer>();
        for (K key : keys) {
            Integer value = get(key);
            if (value != null) {
                map.put(key, value);
            }
        }
        return map;
    }

    public Object getCache() {
        return cache;
    }

    public void clear() {
        cache.invalidateAll();

    }

    @Override
    public int getAndInc(K key) {
        return mustGet(key).getAndIncrement();
    }

    @Override
    public int incAndGet(K key) {
        return mustGet(key).incrementAndGet();
    }

    @Override
    public int decrAndGet(K key) {

        return mustGet(key).decrementAndGet();
    }

    @Override
    public int getAndDecr(K key) {
        return mustGet(key).getAndDecrement();
    }

    @Override
    public int decr(K key, int val) {
        return mustGet(key).addAndGet(-val);
    }

    @Override
    public int inc(K key, int val) {
        return mustGet(key).addAndGet(val);
    }

    @Override
    public boolean set(K key, int seconds, Integer value) {
        set(key, value);
        return true;
    }

    @Override
    public boolean exists(K key) {
        return cache.getIfPresent(key) != null;
    }

}
