package jmind.core.cache.support;

import java.util.Collection;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import jmind.base.cache.MemCache;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;

/**
 * google guava 缓存， 不允许null 建 和null 值
 * @author wbxie
 * 2013-8-15
 * @param <K>
 * @param <V>
 */
public class GuavaCache<K, V> implements MemCache<K, V> {

    final Cache<K, V> cache;

    public GuavaCache() {
        this(3000);
    }

    public GuavaCache(int maxSize) {
        cache = CacheBuilder.newBuilder().maximumSize(maxSize).softValues().build();
    }

    /**
     * 
     * @param maxSize 最大条数
     * @param duration 过期时间，单位分钟
     */
    public GuavaCache(int maxSize, int duration) {
        this(maxSize, duration, false);
    }

    public GuavaCache(int maxSize, int duration, boolean access) {
        CacheBuilder<Object, Object> builder = CacheBuilder.newBuilder().maximumSize(maxSize).softValues();
        if (access)
            cache = builder.expireAfterAccess(duration, TimeUnit.MINUTES).build();
        else
            cache = builder.expireAfterWrite(duration, TimeUnit.MINUTES).build();

    }

    @Override
    public boolean set(K key, V value) {
        if (key != null && value != null)
            cache.put(key, value);
        return true;
    }

    @Override
    public boolean set(K key, int seconds, V value) {
        set(key, value);
        return true;
    }

    @Override
    public boolean delete(K key) {
        if (cache.getIfPresent(key) != null)
            cache.invalidate(key);
        return true;
    }

    @Override
    public V get(K key) {
        return cache.getIfPresent(key);
    }

    @Override
    public Map<K, V> getMulti(Collection<K> keys) {
        return cache.getAllPresent(keys);

    }

    public String toString() {
        return cache.asMap().toString();
    }

    @Override
    public Object getCache() {

        return cache;
    }

    @Override
    public void clear() {
        cache.invalidateAll();

    }

    @Override
    public boolean exists(K key) {

        return cache.getIfPresent(key) != null;
    }

}
