package net.orandja.ktm.adapters

import net.orandja.ktm.Ktm
import kotlin.reflect.KClass
import kotlin.reflect.KType

/**
 * BaseKtmAdapter is an open class that implements the [KtmAdapter.Provider] interface.
 * It provides a set of default adapters for primitives most primitive types.
 *
 * You can extend it to create your own or use the [make] method to construct another provider on top of the current one.
 *
 * @property enableAnyKtmAdapter set to `false` to disable the transformation of unknown value as String.
 *
 * @see KtmAdapterProviderBuilder
 */
open class BaseKtmAdapterProvider : KtmAdapter.Provider {

    /**
     * Create a new set of adapters on top of the current one [backing] with the given [configuration]
     *
     * Usage:
     * ```kotlin
     * @KtmContext
     * class Foo(val foo: String)
     *
     * @KtmContext
     * enum class Status { LOADING, VISIBLE, HIDDEN }
     *
     * val adapters = Ktm.adapter.make {
     *    + FooKtmAdapter // Auto generated by KSP
     *    + StatusKtmAdapter // Auto generated by KSP
     * }
     *
     * val context = Foo("value").toMustacheContext(adapters)
     * val context = adapters.contextOf(Status.LOADING)
     * ```
     *
     * @see KtmAdapterProviderBuilder
     */
    fun make(
        backing: KtmAdapter.Provider? = Ktm.adapters,
        configuration: KtmAdapterProviderBuilder.() -> Unit
    ): BaseKtmAdapterProvider = KtmAdapterProviderBuilder(backing).apply(configuration).build()

    /**
     * Create a new set of adapters on top of the current one [backing]
     */
    fun make(
        backing: KtmAdapter.Provider? = Ktm.adapters,
        adapters: Map<KType, KtmAdapter<*>>,
    ): BaseKtmAdapterProvider = KtmAdapterProvider(backing, adapters)


    // Instead of creating an adapter on every type, we store them.
    private val iterators = mutableMapOf<KType, KtmAdapter<*>>()
    private val iterables = mutableMapOf<KType, KtmAdapter<*>>()
    private val sequences = mutableMapOf<KType, KtmAdapter<*>>()
    private val maps = mutableMapOf<KType, KtmAdapter<*>>()
    private val mapEntries = mutableMapOf<KType, KtmAdapter<*>>()
    private val arrays = mutableMapOf<KType, KtmAdapter<*>>()

    override fun get(kType: KType): KtmAdapter<*>? = when (val kClass = kType.asKClass()) {

        // Adapter for primitive
        // Others falls under the AnyKtmAdapter which is used when nothing is found.
        String::class -> StringKtmAdapter
        Boolean::class -> BooleanKtmAdapter

        // Adapter for primitive arrays

        ByteArray::class -> ByteArrayKtmAdapter
        CharArray::class -> CharArrayKtmAdapter
        ShortArray::class -> ShortArrayKtmAdapter
        IntArray::class -> IntArrayKtmAdapter
        LongArray::class -> LongArrayKtmAdapter
        FloatArray::class -> FloatArrayKtmAdapter
        DoubleArray::class -> DoubleArrayKtmAdapter
        BooleanArray::class -> BooleanArrayKtmAdapter

        // Adapter for kotlin.collections package

        MutableListIterator::class,
        MutableIterator::class,
        ListIterator::class,
        Iterator::class -> iterators.getOrPut(kType.requireTypeProjection(0)) {
            IteratorKtmAdapter(kType.requireTypeProjection(0))
        }

        Iterable::class,
        Collection::class,
        Set::class,
        List::class,
        MutableIterable::class,
        MutableCollection::class,
        MutableSet::class,
        MutableList::class -> iterables.getOrPut(kType.requireTypeProjection(0)) {
            IterableKtmAdapter(kType.requireTypeProjection(0))
        }

        Sequence::class -> sequences.getOrPut(kType.requireTypeProjection(0)) {
            SequenceKtmAdapter(kType.requireTypeProjection(0))
        }

        Map::class,
        MutableMap::class -> maps.getOrPut(kType.requireTypeProjection(1)) {
            MapKtmAdapter(kType.requireTypeProjection(1))
        }

        Map.Entry::class,
        MutableMap.MutableEntry::class -> mapEntries.getOrPut(kType.requireTypeProjection(1)) {
            MapEntryKtmAdapter(kType.requireTypeProjection(1))
        }

        else -> {
            // Arrays are a pain. All `Array` class falls here even if the Array is not `kotlin.Array`
            // JS do not allow qualified name on kClass, so it is impossible to check for `kotlin.Array` string.
            when (kClass.simpleName) {
                Array::class.simpleName -> arrays.getOrPut(kType.requireTypeProjection(0)) {
                    ArrayKtmAdapter(kType.requireTypeProjection(0))
                }

                else -> AnyKtmAdapter
            }
        }
    }

    /** Return `Type` in `Class<Type>` */
    private fun KType.requireTypeProjection(position: Int): KType {
        val projection = arguments.getOrNull(position)
            ?: error("There is no projection at positional argument '$position' on $this")
        return projection.type ?: error("Cannot create adapter for 'Nothing' type ($this)")
    }

    @Suppress("UNCHECKED_CAST")
    private fun KType.asKClass() = when (val classifier = classifier) {
        is KClass<*> -> classifier
        else -> error("Only KClass supported as type classifier, got $classifier")
    } as KClass<Any>
}