package net.orandja.ktm.adapters

import kotlin.reflect.KClass
import kotlin.reflect.KType

/**
 * BaseKtmAdapter is an open class that implements the [KtmAdapter.Provider] interface.
 * It provides a set of default adapters for primitives most primitive types.
 *
 * You can extend it to create your own or use the [make] method to construct another provider on top of the current one.
 *
 * @property enableAnyKtmAdapter set to `false` to disable the transformation of unknown value as String.
 *
 * @see KtmAdapterProviderBuilder
 */
open class BaseKtmAdapterProvider : KtmAdapter.Provider {

    /**
     * Create a new set of adapters on top of the current one [backing] with the given [configuration]
     *
     * Usage:
     * ```kotlin
     * @KtmContext
     * class Foo(val foo: String)
     *
     * @KtmContext
     * enum class Status { LOADING, VISIBLE, HIDDEN }
     *
     * val adapters = Ktm.adapter.make {
     *    + FooKtmAdapter // Auto generated by KSP
     *    + StatusKtmAdapter // Auto generated by KSP
     * }
     *
     * val context = Foo("value").toMustacheContext(adapters)
     * val context = adapters.contextOf(Status.LOADING)
     * ```
     *
     * @see KtmAdapterProviderBuilder
     */
    fun make(
        backing: KtmAdapter.Provider? = this,
        configuration: KtmAdapterProviderBuilder.() -> Unit
    ): BaseKtmAdapterProvider = KtmAdapterProviderBuilder(backing).apply(configuration).build()

    /**
     * Create a new set of adapters on top of the current one [backing]
     */
    fun make(
        backing: KtmAdapter.Provider? = this,
        adapters: Map<TypeKey, KtmAdapter<*>>,
    ): BaseKtmAdapterProvider = KtmAdapterProvider(backing, adapters)

    // From testing, the order of kClass check impact performances in benchmark
    override fun get(kType: TypeKey): KtmAdapter<*>? = when (val kClass = kType.type.asKClass()) {
        // Adapter for primitive
        // Others falls under the AnyKtmAdapter which is used when nothing is found.
        String::class -> StringKtmAdapter
        Boolean::class -> BooleanKtmAdapter
        Int::class -> IntKtmAdapter
        Long::class -> LongKtmAdapter
        Short::class -> ShortKtmAdapter
        Float::class -> FloatKtmAdapter
        Double::class -> DoubleKtmAdapter

        // Adapter for kotlin.collections package

        List::class,
        MutableList::class,
        Set::class,
        MutableSet::class,
        Collection::class,
        MutableCollection::class,
        Iterable::class,
        MutableIterable::class -> IterableKtmAdapter(kType.type.arguments[0].type!!)

        Map::class,
        MutableMap::class -> MapKtmAdapter(kType.type.arguments[1].type!!)

        ListIterator::class,
        Iterator::class,
        MutableIterator::class,
        MutableListIterator::class -> IteratorKtmAdapter(kType.type.arguments[0].type!!)

        Sequence::class -> SequenceKtmAdapter(kType.type.arguments[0].type!!)

        Map.Entry::class,
        MutableMap.MutableEntry::class -> MapEntryKtmAdapter(kType.type.arguments[1].type!!)

        // Adapter for primitive arrays

        ByteArray::class -> ByteArrayKtmAdapter
        CharArray::class -> CharArrayKtmAdapter
        ShortArray::class -> ShortArrayKtmAdapter
        IntArray::class -> IntArrayKtmAdapter
        LongArray::class -> LongArrayKtmAdapter
        FloatArray::class -> FloatArrayKtmAdapter
        DoubleArray::class -> DoubleArrayKtmAdapter
        BooleanArray::class -> BooleanArrayKtmAdapter

        else -> {
            // Arrays are a pain. All `Array` class falls here even if the Array is not `kotlin.Array`
            // JS do not allow qualified name on kClass, so it is impossible to check for `kotlin.Array` string.
            when (kClass.simpleName) {
                Array::class.simpleName -> ArrayKtmAdapter(kType.type.arguments[0].type!!)

                else -> AnyKtmAdapter
            }
        }
    }


    @Suppress("UNCHECKED_CAST")
    private fun KType.asKClass() = when (val classifier = classifier) {
        is KClass<*> -> classifier
        else -> error("Only KClass supported as type classifier, got $classifier")
    } as KClass<Any>
}