/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.utils;

import java.lang.reflect.Constructor;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.function.LongSupplier;
import net.nmoncho.shaded.com.google.common.annotations.VisibleForTesting;
import org.apache.cassandra.concurrent.ScheduledExecutors;
import org.apache.cassandra.config.CassandraRelevantProperties;
import org.apache.cassandra.utils.Clock;
import org.apache.cassandra.utils.MonotonicClockTranslation;
import org.apache.cassandra.utils.Shared;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Shared(scope={Shared.Scope.SIMULATION})
public interface MonotonicClock {
    public long now();

    public long error();

    public MonotonicClockTranslation translate();

    public boolean isAfter(long var1);

    public boolean isAfter(long var1, long var3);

    public static class SampledClock
    implements MonotonicClock {
        private static final Logger logger = LoggerFactory.getLogger(SampledClock.class);
        private static final int UPDATE_INTERVAL_MS = Math.max(1, Integer.parseInt(System.getProperty("cassandra.approximate_time_precision_ms", "2")));
        private static final long ERROR_NANOS = TimeUnit.MILLISECONDS.toNanos(UPDATE_INTERVAL_MS);
        private final MonotonicClock precise;
        private volatile long almostNow;
        private Future<?> almostNowUpdater;

        public SampledClock(MonotonicClock precise) {
            this.precise = precise;
            this.resumeNowSampling();
        }

        @Override
        public long now() {
            return this.almostNow;
        }

        @Override
        public long error() {
            return ERROR_NANOS;
        }

        @Override
        public MonotonicClockTranslation translate() {
            return this.precise.translate();
        }

        @Override
        public boolean isAfter(long instant) {
            return this.isAfter(this.almostNow, instant);
        }

        @Override
        public boolean isAfter(long now, long instant) {
            return now - ERROR_NANOS > instant;
        }

        public synchronized void pauseNowSampling() {
            if (this.almostNowUpdater == null) {
                return;
            }
            this.almostNowUpdater.cancel(true);
            try {
                this.almostNowUpdater.get();
            }
            catch (Throwable throwable) {
                // empty catch block
            }
            this.almostNowUpdater = null;
        }

        public synchronized void resumeNowSampling() {
            if (this.almostNowUpdater != null) {
                throw new IllegalStateException("Already running");
            }
            this.almostNow = this.precise.now();
            logger.info("Scheduling approximate time-check task with a precision of {} milliseconds", (Object)UPDATE_INTERVAL_MS);
            this.almostNowUpdater = ScheduledExecutors.scheduledFastTasks.scheduleWithFixedDelay(() -> {
                this.almostNow = this.precise.now();
            }, UPDATE_INTERVAL_MS, UPDATE_INTERVAL_MS, TimeUnit.MILLISECONDS);
        }

        public synchronized void refreshNow() {
            this.pauseNowSampling();
            this.resumeNowSampling();
        }
    }

    public static class SystemClock
    extends AbstractEpochSamplingClock {
        private SystemClock() {
            super(Clock.Global::currentTimeMillis);
        }

        @Override
        public long now() {
            return Clock.Global.nanoTime();
        }

        @Override
        public long error() {
            return 1L;
        }

        @Override
        public boolean isAfter(long instant) {
            return this.now() > instant;
        }

        @Override
        public boolean isAfter(long now, long instant) {
            return now > instant;
        }
    }

    public static abstract class AbstractEpochSamplingClock
    implements MonotonicClock {
        private static final Logger logger = LoggerFactory.getLogger(AbstractEpochSamplingClock.class);
        private static final String UPDATE_INTERVAL_PROPERTY = "cassandra.NANOTIMETOMILLIS_TIMESTAMP_UPDATE_INTERVAL";
        private static final long UPDATE_INTERVAL_MS = Long.getLong("cassandra.NANOTIMETOMILLIS_TIMESTAMP_UPDATE_INTERVAL", 10000L);
        final LongSupplier millisSinceEpoch;
        private volatile AlmostSameTime almostSameTime = new AlmostSameTime(0L, 0L, Long.MAX_VALUE);
        private Future<?> almostSameTimeUpdater;
        private static double failedAlmostSameTimeUpdateModifier = 1.0;

        AbstractEpochSamplingClock(LongSupplier millisSinceEpoch) {
            this.millisSinceEpoch = millisSinceEpoch;
            this.resumeEpochSampling();
        }

        @Override
        public MonotonicClockTranslation translate() {
            return this.almostSameTime;
        }

        public synchronized void pauseEpochSampling() {
            if (this.almostSameTimeUpdater == null) {
                return;
            }
            this.almostSameTimeUpdater.cancel(true);
            try {
                this.almostSameTimeUpdater.get();
            }
            catch (Throwable throwable) {
                // empty catch block
            }
            this.almostSameTimeUpdater = null;
        }

        public synchronized void resumeEpochSampling() {
            if (this.almostSameTimeUpdater != null) {
                throw new IllegalStateException("Already running");
            }
            this.updateAlmostSameTime();
            logger.info("Scheduling approximate time conversion task with an interval of {} milliseconds", (Object)UPDATE_INTERVAL_MS);
            this.almostSameTimeUpdater = ScheduledExecutors.scheduledFastTasks.scheduleWithFixedDelay(this::updateAlmostSameTime, UPDATE_INTERVAL_MS, UPDATE_INTERVAL_MS, TimeUnit.MILLISECONDS);
        }

        private void updateAlmostSameTime() {
            int tries = 3;
            long[] samples = new long[7];
            samples[0] = Clock.Global.nanoTime();
            for (int i = 1; i < samples.length; i += 2) {
                samples[i] = this.millisSinceEpoch.getAsLong();
                samples[i + 1] = this.now();
            }
            int best = 1;
            for (int i = 3; i < samples.length - 1; i += 2) {
                if (samples[i + 1] - samples[i - 1] >= samples[best + 1] - samples[best - 1]) continue;
                best = i;
            }
            long millis = samples[best];
            long nanos = samples[best + 1] / 2L + samples[best - 1] / 2L;
            long error = samples[best + 1] / 2L - samples[best - 1] / 2L;
            AlmostSameTime prev = this.almostSameTime;
            AlmostSameTime next = new AlmostSameTime(millis, nanos, error);
            if (next.error > prev.error && (double)next.error > (double)prev.error * failedAlmostSameTimeUpdateModifier) {
                failedAlmostSameTimeUpdateModifier *= 1.1;
                return;
            }
            failedAlmostSameTimeUpdateModifier = 1.0;
            this.almostSameTime = next;
        }

        @VisibleForTesting
        public static class AlmostSameTime
        implements MonotonicClockTranslation {
            final long millisSinceEpoch;
            final long monotonicNanos;
            final long error;

            @VisibleForTesting
            public AlmostSameTime(long millisSinceEpoch, long monotonicNanos, long errorNanos) {
                this.millisSinceEpoch = millisSinceEpoch;
                this.monotonicNanos = monotonicNanos;
                this.error = errorNanos;
            }

            @Override
            public long fromMillisSinceEpoch(long currentTimeMillis) {
                return this.monotonicNanos + TimeUnit.MILLISECONDS.toNanos(currentTimeMillis - this.millisSinceEpoch);
            }

            @Override
            public long toMillisSinceEpoch(long nanoTime) {
                return this.millisSinceEpoch + TimeUnit.NANOSECONDS.toMillis(nanoTime - this.monotonicNanos);
            }

            @Override
            public long error() {
                return this.error;
            }
        }
    }

    public static class Global {
        private static final Logger logger = LoggerFactory.getLogger(MonotonicClock.class);
        public static final MonotonicClock preciseTime = Global.precise();
        public static final MonotonicClock approxTime = Global.approx(preciseTime);

        private static MonotonicClock precise() {
            String sclock = CassandraRelevantProperties.CLOCK_MONOTONIC_PRECISE.getString();
            if (sclock != null) {
                try {
                    logger.debug("Using custom clock implementation: {}", (Object)sclock);
                    return (MonotonicClock)Class.forName(sclock).newInstance();
                }
                catch (Exception e) {
                    logger.error(e.getMessage(), (Throwable)e);
                }
            }
            return new SystemClock();
        }

        private static MonotonicClock approx(MonotonicClock precise) {
            String sclock = CassandraRelevantProperties.CLOCK_MONOTONIC_APPROX.getString();
            if (sclock != null) {
                try {
                    logger.debug("Using custom clock implementation: {}", (Object)sclock);
                    Class<?> clazz = Class.forName(sclock);
                    if (SystemClock.class.equals(clazz) && SystemClock.class.equals(precise.getClass())) {
                        return precise;
                    }
                    try {
                        Constructor<?> withPrecise = clazz.getConstructor(MonotonicClock.class);
                        return (MonotonicClock)withPrecise.newInstance(precise);
                    }
                    catch (NoSuchMethodException noSuchMethodException) {
                        return (MonotonicClock)clazz.newInstance();
                    }
                }
                catch (Exception e) {
                    logger.error(e.getMessage(), (Throwable)e);
                }
            }
            return new SampledClock(precise);
        }
    }
}

